<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}
if (!class_exists('Felan_Admin_freelancer_withdraw')) {
    /**
     * Class Felan_Admin_freelancer_withdraw
     */
    class Felan_Admin_freelancer_withdraw
    {
        /**
         * Register custom columns
         * @param $columns
         * @return array
         */
        public function register_custom_column_titles($columns)
        {
            $columns['cb'] = "<input type=\"checkbox\" />";
            $columns['thumb'] = esc_html__('Avatar', 'felan-framework');
            $columns['title'] = esc_html__('Title', 'felan-framework');
            $columns['price'] = esc_html__('Price', 'felan-framework');
            $columns['payment_method'] = esc_html__('Payment', 'felan-framework');
            $columns['status'] = esc_html__('Status', 'felan-framework');
            $columns['request_date'] = esc_html__('Request Date', 'felan-framework');
            $columns['process_date'] = esc_html__('Process Date', 'felan-framework');
            $new_columns = array();
            $custom_order = array('cb', 'thumb', 'title', 'price', 'payment_method', 'status', 'request_date', 'process_date');
            foreach ($custom_order as $colname) {
                $new_columns[$colname] = $columns[$colname];
            }
            return $new_columns;
        }

        /**
         * sortable_columns
         * @param $columns
         * @return mixed
         */
        public function sortable_columns($columns)
        {
            $columns['status'] = 'status';
            $columns['payment_method'] = 'payment_method';
            $columns['title'] = 'title';
            return $columns;
        }

        /**
         * @param $vars
         * @return array
         */
        public function column_orderby($vars)
        {
            if (!is_admin())
                return $vars;

            if (isset($vars['orderby']) && 'status' == $vars['orderby']) {
                $vars = array_merge($vars, array(
                    'meta_key' => FELAN_METABOX_PREFIX . 'freelancer_withdraw_status',
                    'orderby' => 'meta_value_num',
                ));
            }

            return $vars;
        }

        /**
         * @param $actions
         * @param $post
         * @return mixed
         */
        public function modify_list_row_actions($actions, $post)
        {
            // Check for your post type.
            $post_status = get_post_meta($post->ID, FELAN_METABOX_PREFIX . 'freelancer_withdraw_status', true);
            $post_price = get_post_meta($post->ID, FELAN_METABOX_PREFIX . 'freelancer_withdraw_price', true);
            $total_price = get_post_meta($post->ID, FELAN_METABOX_PREFIX . 'freelancer_withdraw_total_price', true);
            if ($post->post_type == 'freelancer_withdraw') {
                // Remove the View action
                unset($actions['view']);

                if ($post_price > $total_price) {
                    $actions['ex-withdraw'] = '<span>' . esc_html__('Not enough balance', 'felan-framework') . '</span>';
                } else {
                    if ($post_status === 'pending') {
                        $actions['completed-withdraw'] = '<a href="' . wp_nonce_url(add_query_arg('completed_withdraw', $post->ID), 'completed_withdraw') . '">' . esc_html__('Completed', 'felan-framework') . '</a>';
                        $actions['canceled-withdraw'] = '<a href="' . wp_nonce_url(add_query_arg('canceled_withdraw', $post->ID), 'canceled_withdraw') . '">' . esc_html__('Canceled', 'felan-framework') . '</a>';
                    } elseif ($post_status === 'completed') {
                        $actions['pending-withdraw'] = '<a href="' . wp_nonce_url(add_query_arg('pending_withdraw', $post->ID), 'pending_withdraw') . '">' . esc_html__('Pending', 'felan-framework') . '</a>';
                    }
                }
            }
            return $actions;
        }

        public function freelancer_withdraw_active()
        {
            if (!empty($_GET['completed_withdraw']) && wp_verify_nonce($_REQUEST['_wpnonce'], 'completed_withdraw')) {
                $post_id = absint(felan_clean(wp_unslash($_GET['completed_withdraw'])));
                $author_id = get_post_meta($post_id, FELAN_METABOX_PREFIX . 'freelancer_withdraw_user_id', true);
                $total_price = get_user_meta($author_id, FELAN_METABOX_PREFIX . 'freelancer_withdraw_total_price', true);
                $post_price = get_post_meta($post_id, FELAN_METABOX_PREFIX . 'freelancer_withdraw_price', true);
                $current_date = date('Y-m-d');

                update_post_meta($post_id, FELAN_METABOX_PREFIX . 'freelancer_withdraw_status', 'completed');
                update_post_meta($post_id, FELAN_METABOX_PREFIX . 'freelancer_withdraw_process_date', $current_date);
                if ($total_price >= $post_price) {
                    // Get the original requested amount (before fee deduction)
                    $withdrawal_fee = get_post_meta($post_id, FELAN_METABOX_PREFIX . 'freelancer_withdrawal_fee', true);
                    $original_request = floatval($post_price) + floatval($withdrawal_fee);

                    // Deduct the original requested amount from balance
                    $price = $total_price - $original_request;
                    update_user_meta($author_id, FELAN_METABOX_PREFIX . 'freelancer_withdraw_total_price', $price);
                }

                // Send email notification to user
                $user = get_user_by('id', $author_id);
                if ($user) {
                    $payment_method = get_post_meta($post_id, FELAN_METABOX_PREFIX . 'freelancer_withdraw_payment_method', true);
                    $withdrawal_fee = get_post_meta($post_id, FELAN_METABOX_PREFIX . 'freelancer_withdrawal_fee', true);
                    $currency_sign_default = felan_get_option('currency_sign_default');
                    $currency_position = felan_get_option('currency_position');

                    if ($currency_position == 'before') {
                        $amount_formatted = $currency_sign_default . number_format($post_price, 2);
                        $fee_formatted = $currency_sign_default . number_format($withdrawal_fee, 2);
                        $receive_formatted = $currency_sign_default . number_format($post_price, 2);
                    } else {
                        $amount_formatted = number_format($post_price, 2) . $currency_sign_default;
                        $fee_formatted = number_format($withdrawal_fee, 2) . $currency_sign_default;
                        $receive_formatted = number_format($post_price, 2) . $currency_sign_default;
                    }

                    $args_mail = array(
                        'user_name' => $user->display_name,
                        'withdraw_amount' => $amount_formatted,
                        'withdraw_fee' => $fee_formatted,
                        'receive_amount' => $receive_formatted,
                        'payment_method' => $payment_method,
                        'process_date' => $current_date,
                        'withdraw_id' => $post_id,
                    );

                    felan_send_email($user->user_email, 'mail_withdrawal_completed', $args_mail);

                    // Send push notification
                    felan_get_data_ajax_notification($post_id, 'withdrawal-completed');
                }

                wp_redirect(remove_query_arg('completed_withdraw', add_query_arg('completed_withdraw', $post_id, admin_url('edit.php?post_type=freelancer_withdraw'))));
                exit;
            }
        }

        public function freelancer_withdraw_pending()
        {
            if (!empty($_GET['pending_withdraw']) && wp_verify_nonce($_REQUEST['_wpnonce'], 'pending_withdraw')) {
                $post_id = absint(felan_clean(wp_unslash($_GET['pending_withdraw'])));
                update_post_meta($post_id, FELAN_METABOX_PREFIX . 'freelancer_withdraw_status', 'pending');

                wp_redirect(remove_query_arg('pending_withdraw', add_query_arg('pending_withdraw', $post_id, admin_url('edit.php?post_type=freelancer_withdraw'))));
                exit;
            }
        }

        public function freelancer_withdraw_canceled()
        {
            if (!empty($_GET['canceled_withdraw']) && wp_verify_nonce($_REQUEST['_wpnonce'], 'canceled_withdraw')) {
                $post_id = absint(felan_clean(wp_unslash($_GET['canceled_withdraw'])));
                $current_date = date('Y-m-d');
                update_post_meta($post_id, FELAN_METABOX_PREFIX . 'freelancer_withdraw_status', 'canceled');
                update_post_meta($post_id, FELAN_METABOX_PREFIX . 'freelancer_withdraw_process_date', $current_date);

                // Send email notification to user
                $author_id = get_post_meta($post_id, FELAN_METABOX_PREFIX . 'freelancer_withdraw_user_id', true);
                $user = get_user_by('id', $author_id);
                if ($user) {
                    $post_price = get_post_meta($post_id, FELAN_METABOX_PREFIX . 'freelancer_withdraw_price', true);
                    $withdrawal_fee = get_post_meta($post_id, FELAN_METABOX_PREFIX . 'freelancer_withdrawal_fee', true);
                    $payment_method = get_post_meta($post_id, FELAN_METABOX_PREFIX . 'freelancer_withdraw_payment_method', true);
                    $currency_sign_default = felan_get_option('currency_sign_default');
                    $currency_position = felan_get_option('currency_position');

                    // Calculate original request and refund amount
                    $original_request = floatval($post_price) + floatval($withdrawal_fee);

                    if ($currency_position == 'before') {
                        $amount_formatted = $currency_sign_default . number_format($original_request, 2);
                        $fee_formatted = $currency_sign_default . number_format($withdrawal_fee, 2);
                        $receive_formatted = $currency_sign_default . number_format($original_request, 2);
                    } else {
                        $amount_formatted = number_format($original_request, 2) . $currency_sign_default;
                        $fee_formatted = number_format($withdrawal_fee, 2) . $currency_sign_default;
                        $receive_formatted = number_format($original_request, 2) . $currency_sign_default;
                    }

                    $args_mail = array(
                        'user_name' => $user->display_name,
                        'withdraw_amount' => $amount_formatted,
                        'withdraw_fee' => $fee_formatted,
                        'receive_amount' => $receive_formatted,
                        'payment_method' => $payment_method,
                        'process_date' => $current_date,
                        'withdraw_id' => $post_id,
                    );

                    felan_send_email($user->user_email, 'mail_withdrawal_canceled', $args_mail);

                    // Send push notification
                    felan_get_data_ajax_notification($post_id, 'withdrawal-canceled');
                }

                wp_redirect(remove_query_arg('canceled_withdraw', add_query_arg('canceled_withdraw', $post_id, admin_url('edit.php?post_type=freelancer_withdraw'))));
                exit;
            }
        }

        /**
         * Display custom column for freelancer_withdraw
         * @param $column
         */
        public function display_custom_column($column)
        {
            global $post;
            $author_id = get_post_meta($post->ID, FELAN_METABOX_PREFIX . 'freelancer_withdraw_user_id', true);
            $payment_method = get_post_meta($post->ID, FELAN_METABOX_PREFIX . 'freelancer_withdraw_payment_method', true);
            $payment_method = str_replace(['-', '_'], ' ', $payment_method);
            $price = get_post_meta($post->ID, FELAN_METABOX_PREFIX . 'freelancer_withdraw_price', true);
            $currency_position = felan_get_option('currency_position');
            $currency_sign_default = felan_get_option('currency_sign_default');
            if ($currency_position == 'before') {
                $price = $currency_sign_default . $price;
            } else {
                $price = $price . $currency_sign_default;
            }
            $service_avatar = get_the_author_meta('author_avatar_image_url', $author_id);
            $request_date = get_the_date('Y-m-d');
            $process_date = get_post_meta($post->ID, FELAN_METABOX_PREFIX . 'freelancer_withdraw_process_date', true);
            if (empty($process_date)) {
                $process_date = '...';
            }
            switch ($column) {
                case 'thumb':
                    if (!empty($service_avatar)) {
                        echo '<img src = " ' . $service_avatar . '" alt=""/>';
                    } else {
                        echo '&ndash;';
                    }
                    break;
                case 'price':
                    echo $price;
                    break;
                case 'payment_method':
                    echo $payment_method;
                    echo '<a href="#" class="view-payment-info" data-user-id="' . $author_id . '" data-withdraw-id="' . $post->ID . '" style="margin-left: 5px;">' . esc_html__('(View)', 'felan-framework') . '</a>';
                    break;
                case 'status':
                    $freelancer_withdraw_status = get_post_meta($post->ID, FELAN_METABOX_PREFIX . 'freelancer_withdraw_status', true);
                    if ($freelancer_withdraw_status == 'completed') {
                        echo '<span class="label felan-label-blue">' . esc_html__('Completed', 'felan-framework') . '</span>';
                    } elseif ($freelancer_withdraw_status == 'canceled') {
                        echo '<span class="label felan-label-gray">' . esc_html__('Canceled', 'felan-framework') . '</span>';
                    } else {
                        echo '<span class="label felan-label-yellow">' . esc_html__('Pending', 'felan-framework') . '</span>';
                    }
                    break;
                case 'request_date':
                    echo $request_date;
                    break;
                case 'process_date':
                    echo $process_date;
                    break;
            }
        }

        /**
         * Modify freelancer_withdraw slug
         * @param $existing_slug
         * @return string
         */
        public function modify_freelancer_withdraw_slug($existing_slug)
        {
            $freelancer_withdraw_url_slug = felan_get_option('freelancer_withdraw_url_slug');
            if ($freelancer_withdraw_url_slug) {
                return $freelancer_withdraw_url_slug;
            }
            return $existing_slug;
        }

        /**
         * Filter Restrict
         */
        public function filter_restrict_manage_freelancer_withdraw()
        {
            global $typenow;
            $post_type = 'freelancer_withdraw';
            if ($typenow == $post_type) {
                //Status
                $values = array(
                    'completed' => esc_html__('Completed', 'felan-framework'),
                    'pending' => esc_html__('Pending', 'felan-framework'),
                    'canceled' => esc_html__('Canceled', 'felan-framework'),
                );
?>
                <select name="freelancer_withdraw_status">
                    <option value=""><?php esc_html_e('All Status', 'felan-framework'); ?></option>
                    <?php $current_v = isset($_GET['freelancer_withdraw_status']) ? felan_clean(wp_unslash($_GET['freelancer_withdraw_status'])) : '';
                    foreach ($values as $value => $label) {
                        printf(
                            '<option value="%s"%s>%s</option>',
                            $value,
                            $value == $current_v ? ' selected="selected"' : '',
                            $label
                        );
                    }
                    ?>
                </select>
                <?php
                //Payment method
                $values = array(
                    'paypal' => esc_html__('Paypal', 'felan-framework'),
                    'stripe' => esc_html__('Stripe', 'felan-framework'),
                    'wire_transfer' => esc_html__('Wire Transfer', 'felan-framework'),
                );
                ?>
                <select name="freelancer_withdraw_payment_method">
                    <option value=""><?php esc_html_e('All Payment', 'felan-framework'); ?></option>
                    <?php $current_v = isset($_GET['freelancer_withdraw_payment_method']) ? wp_unslash(felan_clean($_GET['freelancer_withdraw_payment_method'])) : '';
                    foreach ($values as $value => $label) {
                        printf(
                            '<option value="%s"%s>%s</option>',
                            $value,
                            $value == $current_v ? ' selected="selected"' : '',
                            $label
                        );
                    }
                    ?>
                </select>
                <?php $freelancer_withdraw_user = isset($_GET['freelancer_withdraw_user']) ? felan_clean(wp_unslash($_GET['freelancer_withdraw_user'])) : ''; ?>
                <input type="text" placeholder="<?php esc_attr_e('Search user id', 'felan-framework'); ?>" name="freelancer_withdraw_user" value="<?php echo esc_attr($freelancer_withdraw_user); ?>">
            <?php }
        }

        /**
         * freelancer_withdraw_filter
         * @param $query
         */
        public function freelancer_withdraw_filter($query)
        {
            global $pagenow;
            $post_type = 'freelancer_withdraw';
            $q_vars    = &$query->query_vars;
            $filter_arr = array();
            if ($pagenow == 'edit.php' && isset($q_vars['post_type']) && $q_vars['post_type'] == $post_type) {
                $freelancer_withdraw_user = isset($_GET['freelancer_withdraw_user']) ? felan_clean(wp_unslash($_GET['freelancer_withdraw_user'])) : '';
                if ($freelancer_withdraw_user !== '') {
                    $filter_arr[] = array(
                        'key' => FELAN_METABOX_PREFIX . 'freelancer_withdraw_user_id',
                        'value' => $freelancer_withdraw_user,
                        'compare' => '=',
                    );
                }

                $freelancer_withdraw_status = isset($_GET['freelancer_withdraw_status']) ? felan_clean(wp_unslash($_GET['freelancer_withdraw_status'])) : '';
                if ($freelancer_withdraw_status !== '') {
                    $filter_arr[] = array(
                        'key' => FELAN_METABOX_PREFIX . 'freelancer_withdraw_status',
                        'value' => $freelancer_withdraw_status,
                        'compare' => '=',
                    );
                }

                $freelancer_withdraw_payment_method = isset($_GET['freelancer_withdraw_payment_method']) ? felan_clean(wp_unslash($_GET['freelancer_withdraw_payment_method'])) : '';
                if ($freelancer_withdraw_payment_method !== '') {
                    $filter_arr[] = array(
                        'key' => FELAN_METABOX_PREFIX . 'freelancer_withdraw_payment_method',
                        'value' => $freelancer_withdraw_payment_method,
                        'compare' => '=',
                    );
                }

                if (!empty($filter_arr)) {
                    $q_vars['meta_query'] = $filter_arr;
                }
            }
        }

        /**
         * Enqueue admin scripts for payment info popup
         */
        public function enqueue_admin_scripts($hook)
        {
            global $post_type;
            if ('edit.php' !== $hook || 'freelancer_withdraw' !== $post_type) {
                return;
            }

            // Enqueue merged admin CSS
            wp_enqueue_style(
                'felan-admin-style',
                FELAN_PLUGIN_URL . 'assets/css/admin/admin.min.css',
                array(),
                FELAN_PLUGIN_VER
            );

            // Enqueue JS
            wp_enqueue_script(
                'felan-freelancer-withdraw-admin',
                FELAN_PLUGIN_URL . 'assets/js/admin/js/freelancer-withdraw.js',
                array('jquery'),
                FELAN_PLUGIN_VER,
                true
            );

            // Localize script
            wp_localize_script(
                'felan-freelancer-withdraw-admin',
                'felanWithdrawAdmin',
                array(
                    'modalTitle' => __('Payment Information', 'felan-framework'),
                    'loadingText' => __('Loading payment information...', 'felan-framework'),
                    'errorText' => __('Error loading payment information.', 'felan-framework'),
                    'nonce' => wp_create_nonce('get_payment_info')
                )
            );
        }

        /**
         * AJAX handler to get user payment info
         */
        public function ajax_get_user_payment_info()
        {
            // Verify nonce
            check_ajax_referer('get_payment_info', 'nonce');

            // Check permissions
            if (!current_user_can('manage_options')) {
                wp_send_json_error(array('message' => __('Permission denied.', 'felan-framework')));
            }

            // Validate input
            $user_id = isset($_POST['user_id']) ? absint($_POST['user_id']) : 0;
            $withdraw_id = isset($_POST['withdraw_id']) ? absint($_POST['withdraw_id']) : 0;

            if (!$user_id || !$withdraw_id) {
                wp_send_json_error(array('message' => __('Invalid request. Missing required parameters.', 'felan-framework')));
            }

            // Verify user exists
            $user = get_user_by('id', $user_id);
            if (!$user) {
                wp_send_json_error(array('message' => __('User not found. The user may have been deleted.', 'felan-framework')));
            }

            // Verify withdrawal post exists
            $withdraw_post = get_post($withdraw_id);
            if (!$withdraw_post || $withdraw_post->post_type !== 'freelancer_withdraw') {
                wp_send_json_error(array('message' => __('Withdrawal request not found.', 'felan-framework')));
            }

            // Get payment method and validate
            $payment_method = get_post_meta($withdraw_id, FELAN_METABOX_PREFIX . 'freelancer_withdraw_payment_method', true);
            if (empty($payment_method)) {
                wp_send_json_error(array('message' => __('Payment method not specified for this withdrawal.', 'felan-framework')));
            }

            // Normalize payment method for comparison
            $payment_method_normalized = strtolower(str_replace([' ', '-', '_'], '', $payment_method));
            $payment_method_display = ucwords(str_replace(['-', '_'], ' ', $payment_method));

            // Get withdrawal amounts
            $withdraw_price = floatval(get_post_meta($withdraw_id, FELAN_METABOX_PREFIX . 'freelancer_withdraw_price', true));
            $withdraw_fee = floatval(get_post_meta($withdraw_id, FELAN_METABOX_PREFIX . 'freelancer_withdrawal_fee', true));

            // Calculate original request (before fee deduction)
            $original_request = $withdraw_price + $withdraw_fee;

            // Format amounts
            $currency_sign = felan_get_option('currency_sign_default', '$');
            $currency_position = felan_get_option('currency_position', 'before');

            $original_formatted = $this->format_currency($original_request, $currency_sign, $currency_position);
            $fee_formatted = $this->format_currency($withdraw_fee, $currency_sign, $currency_position);
            $received_formatted = $this->format_currency($withdraw_price, $currency_sign, $currency_position);

            // Get user payment information
            $payment_info = $this->get_user_payment_info($user_id);

            // Generate HTML
            $html = $this->generate_payment_info_html($user, $payment_method_normalized, $payment_method_display, $original_formatted, $fee_formatted, $received_formatted, $payment_info);

            wp_send_json_success(array('html' => $html));
        }

        /**
         * Format currency amount
         */
        private function format_currency($amount, $currency_sign, $position)
        {
            $formatted = number_format($amount, 2);
            return ($position === 'before') ? $currency_sign . $formatted : $formatted . $currency_sign;
        }

        /**
         * Get user payment information
         */
        private function get_user_payment_info($user_id)
        {
            return array(
                'paypal_email' => get_user_meta($user_id, FELAN_METABOX_PREFIX . 'author_payout_paypal', true),
                'stripe_account' => get_user_meta($user_id, FELAN_METABOX_PREFIX . 'author_payout_stripe', true),
                'bank_account_number' => get_user_meta($user_id, FELAN_METABOX_PREFIX . 'author_payout_bank_account_number', true),
                'bank_account_holder' => get_user_meta($user_id, FELAN_METABOX_PREFIX . 'author_payout_bank_account_holder', true),
                'bank_name' => get_user_meta($user_id, FELAN_METABOX_PREFIX . 'author_payout_bank_name', true),
                'bank_country' => get_user_meta($user_id, FELAN_METABOX_PREFIX . 'author_payout_bank_country', true),
                'bank_routing' => get_user_meta($user_id, FELAN_METABOX_PREFIX . 'author_payout_bank_routing_number', true),
                'bank_swift_bic' => get_user_meta($user_id, FELAN_METABOX_PREFIX . 'author_payout_bank_swift_bic', true),
                'bank_iban' => get_user_meta($user_id, FELAN_METABOX_PREFIX . 'author_payout_bank_iban', true),
                'bank_address' => get_user_meta($user_id, FELAN_METABOX_PREFIX . 'author_payout_bank_address', true),
            );
        }

        /**
         * Render a payment info item with copy button
         */
        private function render_info_item($label, $value, $show_copy = true)
        {
            if (empty($value)) {
                return;
            }
            ?>
            <div class="payment-info-item">
                <strong><?php echo esc_html($label); ?></strong>
                <span class="copyable-text"><?php echo $label === __('Bank Address:', 'felan-framework') ? nl2br(esc_html($value)) : esc_html($value); ?></span>
                <?php if ($show_copy): ?>
                    <button class="copy-btn" data-copy="<?php echo esc_attr($value); ?>" title="<?php esc_attr_e('Copy to clipboard', 'felan-framework'); ?>">
                        <span class="dashicons dashicons-admin-page"></span>
                    </button>
                <?php endif; ?>
            </div>
        <?php
        }

        /**
         * Generate payment info HTML
         */
        private function generate_payment_info_html($user, $payment_method_normalized, $payment_method_display, $original_formatted, $fee_formatted, $received_formatted, $payment_info)
        {
            ob_start();
        ?>
            <!-- User Information Section -->
            <div class="payment-info-section-header">
                <h3><?php _e('👤 User Information', 'felan-framework'); ?></h3>
            </div>


            <div class="payment-info-item">
                <strong><?php _e('User:', 'felan-framework'); ?></strong>
                <span class="copyable-text">
                    <?php echo esc_html($user->display_name); ?> <span class="user-id-label">(ID: <?php echo esc_html($user->ID); ?>)</span>
                </span>
                <?php
                // Get freelancer profile link
                $args_freelancer = array(
                    'post_type' => 'freelancer',
                    'posts_per_page' => 1,
                    'author' => $user->ID,
                    'post_status' => 'any',
                );
                $freelancer_posts = get_posts($args_freelancer);
                if (!empty($freelancer_posts)):
                    $profile_link = get_permalink($freelancer_posts[0]->ID);
                ?>
                    <a href="<?php echo esc_url($profile_link); ?>" class="view-profile-link" target="_blank" title="<?php esc_attr_e('View Freelancer Profile', 'felan-framework'); ?>">
                        <span class="dashicons dashicons-external"></span>
                    </a>
                <?php endif; ?>
            </div>

            <?php $this->render_info_item(__('Email:', 'felan-framework'), $user->user_email); ?>

            <!-- Withdrawal Information Section -->
            <div class="payment-info-section-header payment-info-section-spacing">
                <h3><?php _e('💰 Withdrawal Details', 'felan-framework'); ?></h3>
            </div>

            <div class="payment-info-item">
                <strong><?php _e('Payment Method:', 'felan-framework'); ?></strong>
                <span class="payment-method-badge"><?php echo esc_html($payment_method_display); ?></span>
            </div>
            <div class="payment-info-item">
                <strong><?php _e('Withdrawal Amount:', 'felan-framework'); ?></strong>
                <span><?php echo esc_html($original_formatted); ?></span>
            </div>
            <div class="payment-info-item">
                <strong><?php _e('Service Fee:', 'felan-framework'); ?></strong>
                <span class="amount-negative"><?php echo esc_html($fee_formatted); ?></span>
            </div>
            <div class="payment-info-item">
                <strong><?php _e('Amount Received:', 'felan-framework'); ?></strong>
                <span class="amount-positive"><?php echo esc_html($received_formatted); ?></span>
            </div>

            <!-- Payment Account Details Section -->
            <div class="payment-info-section-header payment-info-section-spacing">
                <h3><?php _e('💳 Payment Account Details', 'felan-framework'); ?></h3>
            </div>

            <?php
            // Render payment method specific information
            $has_payment_info = false;

            if ($payment_method_normalized === 'paypal') {
                $has_payment_info = $this->render_paypal_info($payment_info);
            } elseif ($payment_method_normalized === 'stripe') {
                $has_payment_info = $this->render_stripe_info($payment_info);
            } elseif ($payment_method_normalized === 'wiretransfer') {
                $has_payment_info = $this->render_bank_info($payment_info);
            } else {
                // Unknown payment method
                echo '<div class="payment-info-error">';
                printf(
                    __('⚠️ Unknown payment method: %s. Please contact support.', 'felan-framework'),
                    '<strong>' . esc_html($payment_method_display) . '</strong>'
                );
                echo '</div>';
            }

            // Show warning if no payment info found
            if (!$has_payment_info) {
                echo '<div class="payment-info-error">';
                _e('⚠️ No payment information configured. Please ask the user to update their payout settings.', 'felan-framework');
                echo '</div>';
            }
            ?>
            <?php
            return ob_get_clean();
        }

        /**
         * Render PayPal payment information
         */
        private function render_paypal_info($payment_info)
        {
            if (empty($payment_info['paypal_email'])) {
                return false;
            }

            echo '<div class="payment-info-subsection-title paypal-subsection">';
            _e('PayPal Account', 'felan-framework');
            echo '</div>';

            $this->render_info_item(__('PayPal Email:', 'felan-framework'), $payment_info['paypal_email']);

            return true;
        }

        /**
         * Render Stripe payment information
         */
        private function render_stripe_info($payment_info)
        {
            if (empty($payment_info['stripe_account'])) {
                return false;
            }

            echo '<div class="payment-info-subsection-title stripe-subsection">';
            _e('Stripe Account', 'felan-framework');
            echo '</div>';

            $this->render_info_item(__('Stripe Account ID:', 'felan-framework'), $payment_info['stripe_account']);

            return true;
        }

        /**
         * Render Bank transfer payment information
         */
        private function render_bank_info($payment_info)
        {
            // Check if any bank info exists
            $has_info = !empty($payment_info['bank_account_number'])
                     || !empty($payment_info['bank_account_holder'])
                     || !empty($payment_info['bank_name'])
                     || !empty($payment_info['bank_swift_bic'])
                     || !empty($payment_info['bank_address']);

            if (!$has_info) {
                return false;
            }

            echo '<div class="payment-info-subsection-title bank-subsection">';
            _e('Bank Account Details', 'felan-framework');
            echo '</div>';

            // Render bank information fields
            $this->render_info_item(__('Country:', 'felan-framework'), $payment_info['bank_country']);
            $this->render_info_item(__('Account Holder:', 'felan-framework'), $payment_info['bank_account_holder']);
            $this->render_info_item(__('Account Number:', 'felan-framework'), $payment_info['bank_account_number']);
            $this->render_info_item(__('Bank Name:', 'felan-framework'), $payment_info['bank_name']);
            $this->render_info_item(__('SWIFT/BIC Code:', 'felan-framework'), $payment_info['bank_swift_bic']);
            $this->render_info_item(__('Bank Address:', 'felan-framework'), $payment_info['bank_address']);
            $this->render_info_item(__('IBAN:', 'felan-framework'), $payment_info['bank_iban']);
            $this->render_info_item(__('Routing Number / ABA:', 'felan-framework'), $payment_info['bank_routing']);

            return true;
        }
    }
}
