<?php
if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

if (!class_exists('Felan_Admin_Project_Alerts')) {
    /**
     * Class Felan_Admin_Project_Alerts
     * Admin management for Project Alerts
     */
    class Felan_Admin_Project_Alerts
    {
        /**
         * Post type name
         */
        const POST_TYPE = 'project_alerts';

        /**
         * User role for freelancers
         */
        const FREELANCER_ROLE = 'felan_user_freelancer';

        /**
         * Match types
         */
        const MATCH_TYPE_OR = 'OR';
        const MATCH_TYPE_AND = 'AND';

        /**
         * Frequency options
         */
        const FREQUENCY_DAILY = 'daily';
        const FREQUENCY_WEEKLY = 'weekly';
        const FREQUENCY_MONTHLY = 'monthly';

        /**
         * AJAX actions
         */
        const AJAX_GET_USER_EMAIL = 'felan_get_user_email';
        const AJAX_SEARCH_TAXONOMY_TERMS = 'felan_search_taxonomy_terms';
        const AJAX_SEARCH_USERS = 'felan_search_users';

        /**
         * Meta keys
         */
        const META_USER_TYPE = 'project_alerts_user_type';
        const META_USER_ID = 'project_alerts_user_id';
        const META_EMAIL = 'project_alerts_email';
        const META_MATCH_TYPE = 'project_alerts_match_type';
        const META_FREQUENCY = 'project_alerts_frequency';
        const META_CATEGORIES = 'project_alerts_categories';
        const META_SKILLS = 'project_alerts_skills';
        const META_LANGUAGE = 'project_alerts_language';
        const META_CAREER = 'project_alerts_career';
        const META_LOCATION = 'project_alerts_location';
        const META_STATE = 'project_alerts_state';

        /**
         * Constructor
         */
        public function __construct()
        {
            // Only run in admin
            if (!is_admin()) {
                return;
            }

            // Custom columns
            add_filter('manage_edit-' . self::POST_TYPE . '_columns', array($this, 'register_custom_column_titles'));
            add_action('manage_' . self::POST_TYPE . '_posts_custom_column', array($this, 'display_custom_column'), 10, 2);
            add_filter('manage_edit-' . self::POST_TYPE . '_sortable_columns', array($this, 'sortable_columns'));

            // Filters
            add_action('restrict_manage_posts', array($this, 'add_admin_filters'));
            add_filter('parse_query', array($this, 'filter_admin_query'));

            // Bulk actions
            add_filter('bulk_actions-edit-' . self::POST_TYPE, array($this, 'add_bulk_actions'));
            add_filter('handle_bulk_actions-edit-' . self::POST_TYPE, array($this, 'handle_bulk_actions'), 10, 3);

            // Quick edit
            add_action('quick_edit_custom_box', array($this, 'add_quick_edit_fields'), 10, 2);
            add_action('save_post', array($this, 'save_quick_edit'));

            // Extended search
            add_filter('posts_search', array($this, 'extend_admin_search'), 10, 2);

            // Export functionality
            add_action('manage_posts_extra_tablenav', array($this, 'add_export_button'), 10, 1);
            add_action('admin_init', array($this, 'handle_export'));

            // Admin notices
            add_action('admin_notices', array($this, 'admin_notices'));

            // Admin scripts and styles
            add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));

            // Add custom metabox validation
            add_action('save_post', array($this, 'validate_project_alert_metabox'), 10, 2);

            // AJAX handlers
            add_action('wp_ajax_' . self::AJAX_GET_USER_EMAIL, array($this, 'ajax_get_user_email'));
            add_action('wp_ajax_' . self::AJAX_SEARCH_TAXONOMY_TERMS, array($this, 'ajax_search_taxonomy_terms'));
            add_action('wp_ajax_' . self::AJAX_SEARCH_USERS, array($this, 'ajax_search_users'));

            // Ensure trash functionality works
            add_filter('post_row_actions', array($this, 'ensure_trash_row_actions'), 10, 2);

            // Map meta capabilities for admin
            add_filter('map_meta_cap', array($this, 'map_project_alerts_meta_cap'), 10, 4);

            // Grant capabilities directly to admin for project_alerts
            add_filter('user_has_cap', array($this, 'grant_project_alerts_caps_to_admin'), 10, 4);
        }

        /**
         * Register custom column titles
         * @param array $columns
         * @return array
         */
        public function register_custom_column_titles($columns)
        {
            // Remove default columns we don't need
            unset($columns['date']);

            // Add custom columns
            $columns['cb'] = '<input type="checkbox" />';
            $columns['title'] = esc_html__('Alert Name', 'felan-framework');
            $columns['user'] = esc_html__('User', 'felan-framework');
            $columns['taxonomies'] = esc_html__('Taxonomies', 'felan-framework');
            $columns['match_type'] = esc_html__('Match Type', 'felan-framework');
            $columns['frequency'] = esc_html__('Frequency', 'felan-framework');
            $columns['date'] = esc_html__('Date', 'felan-framework');

            // Reorder columns
            $new_columns = array();
            $custom_order = array('cb', 'title', 'user', 'taxonomies', 'match_type', 'frequency', 'date');
            foreach ($custom_order as $colname) {
                if (isset($columns[$colname])) {
                    $new_columns[$colname] = $columns[$colname];
                }
            }

            return $new_columns;
        }

        /**
         * Display custom column content
         * @param string $column
         * @param int $post_id
         */
        /**
         * Display custom column content
         * @param string $column
         * @param int $post_id
         */
        public function display_custom_column($column, $post_id)
        {
            switch ($column) {
                case 'user':
                    $this->display_user_column($post_id);
                    break;

                case 'taxonomies':
                    $this->display_taxonomies_summary($post_id);
                    break;

                case 'match_type':
                    $this->display_match_type_column($post_id);
                    break;

                case 'frequency':
                    $this->display_frequency_column($post_id);
                    break;
            }
        }

        /**
         * Display user column
         * @param int $post_id
         */
        private function display_user_column($post_id)
        {
            $user_id = $this->get_meta($post_id, self::META_USER_ID);
            $email = $this->get_meta($post_id, self::META_EMAIL);
            $user_type = $this->get_meta($post_id, FELAN_METABOX_PREFIX . 'project_alerts_user_type');

            // Check if non-registered user
            if (!$user_id || $user_id == 0 || $user_type === 'non_registered') {
                // Non-registered user - display email
                if (!empty($email)) {
                    echo '<span class="non-registered-user">' . esc_html__('Non-registered', 'felan-framework') . '</span>';
                    echo '<br><small><a href="mailto:' . esc_attr($email) . '">' . esc_html($email) . '</a></small>';
                } else {
                    echo '-';
                }
                return;
            }

            // Registered user
            $user = get_user_by('ID', $user_id);
            if (!$user) {
                // User not found, but try to show email if available
                if (!empty($email)) {
                    echo '<span class="user-not-found">' . esc_html__('User not found', 'felan-framework') . '</span>';
                    echo '<br><small><a href="mailto:' . esc_attr($email) . '">' . esc_html($email) . '</a></small>';
                } else {
                    echo '-';
                }
                return;
            }

            // Display registered user info
            $edit_link = get_edit_user_link($user_id);
            echo '<a href="' . esc_url($edit_link) . '">' . esc_html($user->display_name) . '</a>';

            // Always show email (use from user object or meta)
            $display_email = !empty($user->user_email) ? $user->user_email : $email;
            if (!empty($display_email)) {
                echo '<br><small><a href="mailto:' . esc_attr($display_email) . '">' . esc_html($display_email) . '</a></small>';
            }
        }

        /**
         * Display match type column
         * @param int $post_id
         */
        private function display_match_type_column($post_id)
        {
            $match_type = $this->get_meta($post_id, self::META_MATCH_TYPE);
            $match_type = $match_type ?: self::MATCH_TYPE_OR;
            $label = $match_type === self::MATCH_TYPE_AND
                ? esc_html__('AND', 'felan-framework')
                : esc_html__('OR', 'felan-framework');
            echo '<span class="match-type-badge match-type-' . esc_attr(strtolower($match_type)) . '">' . $label . '</span>';
        }

        /**
         * Display frequency column
         * @param int $post_id
         */
        private function display_frequency_column($post_id)
        {
            $frequency = $this->get_meta($post_id, self::META_FREQUENCY);
            $frequency = $frequency ?: self::FREQUENCY_DAILY;
            $labels = array(
                self::FREQUENCY_DAILY => esc_html__('Daily', 'felan-framework'),
                self::FREQUENCY_WEEKLY => esc_html__('Weekly', 'felan-framework'),
                self::FREQUENCY_MONTHLY => esc_html__('Monthly', 'felan-framework'),
            );
            $label = isset($labels[$frequency]) ? $labels[$frequency] : $frequency;
            echo '<span class="frequency-badge frequency-' . esc_attr($frequency) . '">' . $label . '</span>';
        }

        /**
         * Get meta value with prefix
         * @param int $post_id
         * @param string $meta_key
         * @return mixed
         */
        private function get_meta($post_id, $meta_key)
        {
            return get_post_meta($post_id, FELAN_METABOX_PREFIX . $meta_key, true);
        }

        /**
         * Display taxonomies summary
         * @param int $post_id
         */
        private function display_taxonomies_summary($post_id)
        {
            $taxonomies = $this->get_taxonomy_config();
            $summary = array();

            foreach ($taxonomies as $tax_data) {
                $term_ids = $this->get_meta($post_id, $tax_data['meta_key']);
                if (!empty($term_ids) && is_array($term_ids)) {
                    $summary[] = sprintf('%s: %d', $tax_data['label'], count($term_ids));
                }
            }

            if (!empty($summary)) {
                echo '<div class="taxonomies-summary">';
                echo '<span class="taxonomies-count">' . esc_html(implode(', ', $summary)) . '</span>';
                echo '</div>';
            } else {
                echo '-';
            }
        }

        /**
         * Get taxonomy configuration
         * @return array
         */
        private function get_taxonomy_config()
        {
            return array(
                array(
                    'meta_key' => self::META_CATEGORIES,
                    'taxonomy' => 'project-categories',
                    'label' => esc_html__('Categories', 'felan-framework'),
                ),
                array(
                    'meta_key' => self::META_SKILLS,
                    'taxonomy' => 'project-skills',
                    'label' => esc_html__('Skills', 'felan-framework'),
                ),
                array(
                    'meta_key' => self::META_LANGUAGE,
                    'taxonomy' => 'project-language',
                    'label' => esc_html__('Languages', 'felan-framework'),
                ),
                array(
                    'meta_key' => self::META_CAREER,
                    'taxonomy' => 'project-career',
                    'label' => esc_html__('Careers', 'felan-framework'),
                ),
                array(
                    'meta_key' => self::META_LOCATION,
                    'taxonomy' => 'felan_location',
                    'label' => esc_html__('Location', 'felan-framework'),
                ),
                array(
                    'meta_key' => self::META_STATE,
                    'taxonomy' => 'felan_state',
                    'label' => esc_html__('State', 'felan-framework'),
                ),
            );
        }

        /**
         * Make columns sortable
         * @param array $columns
         * @return array
         */
        public function sortable_columns($columns)
        {
            $columns['user'] = 'user_id';
            $columns['match_type'] = 'match_type';
            $columns['frequency'] = 'frequency';
            $columns['date'] = 'date';
            return $columns;
        }

        /**
         * Add admin filters
         */
        /**
         * Add admin filters
         */
        public function add_admin_filters()
        {
            global $typenow;

            if ($typenow !== self::POST_TYPE) {
                return;
            }

            // User filter
            $selected_user = isset($_GET['filter_user']) ? intval($_GET['filter_user']) : 0;
            $users = get_users(array(
                'role' => self::FREELANCER_ROLE,
                'number' => 100,
                'orderby' => 'display_name',
            ));
            ?>
            <select name="filter_user" id="filter_user">
                <option value=""><?php esc_html_e('All Users', 'felan-framework'); ?></option>
                <?php foreach ($users as $user) : ?>
                    <option value="<?php echo esc_attr($user->ID); ?>" <?php selected($selected_user, $user->ID); ?>>
                        <?php echo esc_html($user->display_name . ' (' . $user->user_email . ')'); ?>
                    </option>
                <?php endforeach; ?>
            </select>

            <?php
            // Match Type filter
            $selected_match_type = isset($_GET['filter_match_type']) ? sanitize_text_field($_GET['filter_match_type']) : '';
            ?>
            <select name="filter_match_type" id="filter_match_type">
                <option value=""><?php esc_html_e('All Match Types', 'felan-framework'); ?></option>
                <option value="<?php echo esc_attr(self::MATCH_TYPE_OR); ?>" <?php selected($selected_match_type, self::MATCH_TYPE_OR); ?>><?php esc_html_e('OR', 'felan-framework'); ?></option>
                <option value="<?php echo esc_attr(self::MATCH_TYPE_AND); ?>" <?php selected($selected_match_type, self::MATCH_TYPE_AND); ?>><?php esc_html_e('AND', 'felan-framework'); ?></option>
            </select>

            <?php
            // Frequency filter
            $selected_frequency = isset($_GET['filter_frequency']) ? sanitize_text_field($_GET['filter_frequency']) : '';
            ?>
            <select name="filter_frequency" id="filter_frequency">
                <option value=""><?php esc_html_e('All Frequencies', 'felan-framework'); ?></option>
                <option value="<?php echo esc_attr(self::FREQUENCY_DAILY); ?>" <?php selected($selected_frequency, self::FREQUENCY_DAILY); ?>><?php esc_html_e('Daily', 'felan-framework'); ?></option>
                <option value="<?php echo esc_attr(self::FREQUENCY_WEEKLY); ?>" <?php selected($selected_frequency, self::FREQUENCY_WEEKLY); ?>><?php esc_html_e('Weekly', 'felan-framework'); ?></option>
                <option value="<?php echo esc_attr(self::FREQUENCY_MONTHLY); ?>" <?php selected($selected_frequency, self::FREQUENCY_MONTHLY); ?>><?php esc_html_e('Monthly', 'felan-framework'); ?></option>
            </select>

            <?php
            // Date range filter
            $date_from = isset($_GET['date_from']) ? sanitize_text_field($_GET['date_from']) : '';
            $date_to = isset($_GET['date_to']) ? sanitize_text_field($_GET['date_to']) : '';
            ?>
            <input type="date" name="date_from" id="date_from" value="<?php echo esc_attr($date_from); ?>" placeholder="<?php esc_attr_e('From Date', 'felan-framework'); ?>">
            <input type="date" name="date_to" id="date_to" value="<?php echo esc_attr($date_to); ?>" placeholder="<?php esc_attr_e('To Date', 'felan-framework'); ?>">
            <?php
        }

        /**
         * Handle filter queries
         * @param WP_Query $query
         */
        /**
         * Handle filter queries
         * @param WP_Query $query
         */
        public function filter_admin_query($query)
        {
            global $pagenow, $typenow;

            if ($pagenow !== 'edit.php' || $typenow !== self::POST_TYPE || !$query->is_main_query()) {
                return;
            }

            $meta_query = array();

            // User filter
            if (isset($_GET['filter_user']) && !empty($_GET['filter_user'])) {
                $meta_query[] = array(
                    'key' => FELAN_METABOX_PREFIX . self::META_USER_ID,
                    'value' => intval($_GET['filter_user']),
                    'compare' => '=',
                );
            }

            // Match Type filter
            if (isset($_GET['filter_match_type']) && !empty($_GET['filter_match_type'])) {
                $meta_query[] = array(
                    'key' => FELAN_METABOX_PREFIX . self::META_MATCH_TYPE,
                    'value' => sanitize_text_field($_GET['filter_match_type']),
                    'compare' => '=',
                );
            }

            // Frequency filter
            if (isset($_GET['filter_frequency']) && !empty($_GET['filter_frequency'])) {
                $meta_query[] = array(
                    'key' => FELAN_METABOX_PREFIX . self::META_FREQUENCY,
                    'value' => sanitize_text_field($_GET['filter_frequency']),
                    'compare' => '=',
                );
            }

            // Date range filter
            if (!empty($_GET['date_from']) || !empty($_GET['date_to'])) {
                $date_query = array();
                if (!empty($_GET['date_from'])) {
                    $date_query['after'] = sanitize_text_field($_GET['date_from']);
                }
                if (!empty($_GET['date_to'])) {
                    $date_query['before'] = sanitize_text_field($_GET['date_to']);
                    $date_query['inclusive'] = true;
                }
                $query->set('date_query', array($date_query));
            }

            if (!empty($meta_query)) {
                $query->set('meta_query', $meta_query);
            }

            // Handle sorting
            $orderby = $query->get('orderby');
            $sort_map = array(
                'user_id' => array(
                    'meta_key' => FELAN_METABOX_PREFIX . self::META_USER_ID,
                    'orderby' => 'meta_value_num',
                ),
                'match_type' => array(
                    'meta_key' => FELAN_METABOX_PREFIX . self::META_MATCH_TYPE,
                    'orderby' => 'meta_value',
                ),
                'frequency' => array(
                    'meta_key' => FELAN_METABOX_PREFIX . self::META_FREQUENCY,
                    'orderby' => 'meta_value',
                ),
            );

            if (isset($sort_map[$orderby])) {
                $query->set('meta_key', $sort_map[$orderby]['meta_key']);
                $query->set('orderby', $sort_map[$orderby]['orderby']);
            }
        }

        /**
         * Add bulk actions
         * @param array $actions
         * @return array
         */
        public function add_bulk_actions($actions)
        {
            // Keep default trash action, add custom delete permanently
            $actions['delete_project_alerts'] = esc_html__('Delete Permanently', 'felan-framework');
            return $actions;
        }

        /**
         * Handle bulk actions
         * @param string $redirect_to
         * @param string $action
         * @param array $post_ids
         * @return string
         */
        public function handle_bulk_actions($redirect_to, $action, $post_ids)
        {
            if ($action !== 'delete_project_alerts') {
                return $redirect_to;
            }

            if (empty($post_ids) || !is_array($post_ids)) {
                return $redirect_to;
            }

            $deleted = 0;
            foreach ($post_ids as $post_id) {
                if (current_user_can('delete_post', $post_id)) {
                    // Force delete (bypass trash)
                    if (wp_delete_post($post_id, true)) {
                        $deleted++;
                    }
                }
            }

            $redirect_to = add_query_arg(
                array(
                    'bulk_deleted' => $deleted,
                    'post_type' => 'project_alerts',
                ),
                $redirect_to
            );

            return $redirect_to;
        }

        /**
         * Add quick edit fields
         * @param string $column_name
         * @param string $post_type
         */
        /**
         * Add quick edit fields
         * @param string $column_name
         * @param string $post_type
         */
        public function add_quick_edit_fields($column_name, $post_type)
        {
            if ($post_type !== self::POST_TYPE) {
                return;
            }

            static $printed = false;
            if ($printed) {
                return;
            }
            $printed = true;
            ?>
            <fieldset class="inline-edit-col-right">
                <div class="inline-edit-col">
                    <div class="inline-edit-group wp-clearfix">
                        <label class="inline-edit-status alignleft">
                            <span class="title"><?php esc_html_e('Match Type', 'felan-framework'); ?></span>
                            <select name="<?php echo esc_attr(FELAN_METABOX_PREFIX . self::META_MATCH_TYPE); ?>">
                                <option value="<?php echo esc_attr(self::MATCH_TYPE_OR); ?>"><?php esc_html_e('OR', 'felan-framework'); ?></option>
                                <option value="<?php echo esc_attr(self::MATCH_TYPE_AND); ?>"><?php esc_html_e('AND', 'felan-framework'); ?></option>
                            </select>
                        </label>
                    </div>
                    <div class="inline-edit-group wp-clearfix">
                        <label class="inline-edit-status alignleft">
                            <span class="title"><?php esc_html_e('Frequency', 'felan-framework'); ?></span>
                            <select name="<?php echo esc_attr(FELAN_METABOX_PREFIX . self::META_FREQUENCY); ?>">
                                <option value="<?php echo esc_attr(self::FREQUENCY_DAILY); ?>"><?php esc_html_e('Daily', 'felan-framework'); ?></option>
                                <option value="<?php echo esc_attr(self::FREQUENCY_WEEKLY); ?>"><?php esc_html_e('Weekly', 'felan-framework'); ?></option>
                                <option value="<?php echo esc_attr(self::FREQUENCY_MONTHLY); ?>"><?php esc_html_e('Monthly', 'felan-framework'); ?></option>
                            </select>
                        </label>
                    </div>
                </div>
            </fieldset>
            <?php
        }

        /**
         * Save quick edit
         * @param int $post_id
         */
        public function save_quick_edit($post_id)
        {
            if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
                return;
            }

            if (!current_user_can('edit_post', $post_id)) {
                return;
            }

            if (get_post_type($post_id) !== self::POST_TYPE) {
                return;
            }

            // Check if this is a quick edit save
            if (!isset($_POST['_inline_edit'])) {
                return;
            }

            check_admin_referer('inlineeditnonce', '_inline_edit');

            // Update match type
            $match_type_key = FELAN_METABOX_PREFIX . self::META_MATCH_TYPE;
            if (isset($_POST[$match_type_key])) {
                $match_type = sanitize_text_field($_POST[$match_type_key]);
                if (in_array($match_type, array(self::MATCH_TYPE_OR, self::MATCH_TYPE_AND), true)) {
                    update_post_meta($post_id, $match_type_key, $match_type);
                }
            }

            // Update frequency
            $frequency_key = FELAN_METABOX_PREFIX . self::META_FREQUENCY;
            if (isset($_POST[$frequency_key])) {
                $frequency = sanitize_text_field($_POST[$frequency_key]);
                if (in_array($frequency, array(self::FREQUENCY_DAILY, self::FREQUENCY_WEEKLY, self::FREQUENCY_MONTHLY), true)) {
                    update_post_meta($post_id, $frequency_key, $frequency);
                }
            }
        }

        /**
         * Extend admin search
         * @param string $search
         * @param WP_Query $query
         * @return string
         */
        public function extend_admin_search($search, $query)
        {
            global $wpdb;

            if (!is_admin() || !$query->is_main_query()) {
                return $search;
            }

            if ($query->get('post_type') !== self::POST_TYPE) {
                return $search;
            }

            $search_term = $query->get('s');
            if (empty($search_term)) {
                return $search;
            }

            // Search in email meta
            $search = preg_replace(
                '/\(\(\(.*?\)\)\)/',
                '',
                $search
            );

            $search_term = '%' . $wpdb->esc_like($search_term) . '%';

            $search .= $wpdb->prepare(
                " OR EXISTS (
                    SELECT 1 FROM {$wpdb->postmeta}
                    WHERE {$wpdb->postmeta}.post_id = {$wpdb->posts}.ID
                    AND (
                        ({$wpdb->postmeta}.meta_key = %s AND {$wpdb->postmeta}.meta_value LIKE %s)
                        OR ({$wpdb->postmeta}.meta_key = %s AND {$wpdb->postmeta}.meta_value LIKE %s)
                    )
                )",
                FELAN_METABOX_PREFIX . self::META_EMAIL,
                $search_term,
                FELAN_METABOX_PREFIX . self::META_USER_ID,
                $search_term
            );

            return $search;
        }

        /**
         * Add export button
         * @param string $which 'top' or 'bottom'
         */
        /**
         * Add export button
         * @param string $which 'top' or 'bottom'
         */
        public function add_export_button($which)
        {
            global $typenow;

            if ($typenow !== self::POST_TYPE || $which !== 'top') {
                return;
            }

            // Build export URL with current filters
            $export_url = add_query_arg(array(
                'action' => 'felan_export_project_alerts',
                'post_type' => self::POST_TYPE,
            ), admin_url('admin.php'));

            // Preserve filters
            if (isset($_GET['filter_user']) && !empty($_GET['filter_user'])) {
                $export_url = add_query_arg('filter_user', intval($_GET['filter_user']), $export_url);
            }
            if (isset($_GET['filter_match_type']) && !empty($_GET['filter_match_type'])) {
                $export_url = add_query_arg('filter_match_type', sanitize_text_field($_GET['filter_match_type']), $export_url);
            }
            if (isset($_GET['filter_frequency']) && !empty($_GET['filter_frequency'])) {
                $export_url = add_query_arg('filter_frequency', sanitize_text_field($_GET['filter_frequency']), $export_url);
            }
            if (isset($_GET['date_from']) && !empty($_GET['date_from'])) {
                $export_url = add_query_arg('date_from', sanitize_text_field($_GET['date_from']), $export_url);
            }
            if (isset($_GET['date_to']) && !empty($_GET['date_to'])) {
                $export_url = add_query_arg('date_to', sanitize_text_field($_GET['date_to']), $export_url);
            }

            // Add nonce
            $export_url = wp_nonce_url($export_url, 'felan_export_project_alerts');

            echo '<a href="' . esc_url($export_url) . '" class="button" style="margin-left: 10px;">' . esc_html__('Export to CSV', 'felan-framework') . '</a>';
        }

        /**
         * Handle export
         */
        public function handle_export()
        {
            if (!isset($_GET['action']) || $_GET['action'] !== 'felan_export_project_alerts') {
                return;
            }

            if (!current_user_can('manage_options')) {
                wp_die(esc_html__('You do not have permission to export.', 'felan-framework'));
            }

            check_admin_referer('felan_export_project_alerts');

            // Get all alerts
            $args = array(
                'post_type' => self::POST_TYPE,
                'post_status' => 'publish',
                'posts_per_page' => -1,
            );

            // Apply filters if set
            if (isset($_GET['filter_user']) && !empty($_GET['filter_user'])) {
                $args['meta_query'][] = array(
                    'key' => FELAN_METABOX_PREFIX . self::META_USER_ID,
                    'value' => intval($_GET['filter_user']),
                    'compare' => '=',
                );
            }

            $alerts = get_posts($args);

            // Set headers for CSV download
            header('Content-Type: text/csv; charset=utf-8');
            header('Content-Disposition: attachment; filename=project-alerts-' . date('Y-m-d') . '.csv');
            header('Pragma: no-cache');
            header('Expires: 0');

            $output = fopen('php://output', 'w');

            // Add BOM for UTF-8
            fprintf($output, chr(0xEF) . chr(0xBB) . chr(0xBF));

            // CSV headers
            fputcsv($output, array(
                esc_html__('ID', 'felan-framework'),
                esc_html__('Title', 'felan-framework'),
                esc_html__('User ID', 'felan-framework'),
                esc_html__('User Name', 'felan-framework'),
                esc_html__('Email', 'felan-framework'),
                esc_html__('Categories', 'felan-framework'),
                esc_html__('Skills', 'felan-framework'),
                esc_html__('Languages', 'felan-framework'),
                esc_html__('Careers', 'felan-framework'),
                esc_html__('Location', 'felan-framework'),
                esc_html__('State', 'felan-framework'),
                esc_html__('Match Type', 'felan-framework'),
                esc_html__('Frequency', 'felan-framework'),
                esc_html__('Created Date', 'felan-framework'),
            ));

            // Export data
            foreach ($alerts as $alert) {
                $user_id = $this->get_meta($alert->ID, self::META_USER_ID);
                $user = $user_id ? get_user_by('ID', $user_id) : null;
                $email = $this->get_meta($alert->ID, self::META_EMAIL);
                if (!$email && $user) {
                    $email = $user->user_email;
                }

                // Get taxonomy terms
                $taxonomies = $this->get_taxonomy_config();
                $taxonomy_data = array();
                foreach ($taxonomies as $tax_data) {
                    $taxonomy_data[] = $this->get_term_names($alert->ID, $tax_data['meta_key'], $tax_data['taxonomy']);
                }

                fputcsv($output, array(
                    $alert->ID,
                    $alert->post_title,
                    $user_id ?: '',
                    $user ? $user->display_name : '',
                    $email ?: '',
                    $taxonomy_data[0], // Categories
                    $taxonomy_data[1], // Skills
                    $taxonomy_data[2], // Languages
                    $taxonomy_data[3], // Careers
                    $taxonomy_data[4], // Locations
                    $taxonomy_data[5], // States
                    $this->get_meta($alert->ID, self::META_MATCH_TYPE) ?: self::MATCH_TYPE_OR,
                    $this->get_meta($alert->ID, self::META_FREQUENCY) ?: self::FREQUENCY_DAILY,
                    $alert->post_date,
                ));
            }

            fclose($output);
            exit;
        }

        /**
         * Get term names for export
         * @param int $post_id
         * @param string $meta_key
         * @param string $taxonomy
         * @return string
         */
        private function get_term_names($post_id, $meta_key, $taxonomy)
        {
            $term_ids = $this->get_meta($post_id, $meta_key);
            if (empty($term_ids) || !is_array($term_ids)) {
                return '';
            }

            $names = array();
            foreach ($term_ids as $term_id) {
                $term = get_term($term_id, $taxonomy);
                if ($term && !is_wp_error($term)) {
                    $names[] = $term->name;
                }
            }

            return implode(', ', $names);
        }

        /**
         * Admin notices
         */
        /**
         * Admin notices
         */
        public function admin_notices()
        {
            global $typenow, $pagenow;

            if ($pagenow !== 'edit.php' || $typenow !== self::POST_TYPE) {
                return;
            }

            // Bulk delete notice
            if (isset($_GET['bulk_deleted']) && intval($_GET['bulk_deleted']) > 0) {
                $count = intval($_GET['bulk_deleted']);
                ?>
                <div class="notice notice-success is-dismissible">
                    <p><?php printf(esc_html(_n('%d alert deleted permanently.', '%d alerts deleted permanently.', $count, 'felan-framework')), $count); ?></p>
                </div>
                <?php
            }
        }

        /**
         * Enqueue admin assets
         */
        public function enqueue_admin_assets($hook)
        {
            global $typenow, $post;

            // For list view
            if ($hook === 'edit.php' && $typenow === self::POST_TYPE) {
                $this->enqueue_list_assets();
                return;
            }

            // For edit/add post screen
            if (($hook === 'post.php' || $hook === 'post-new.php') && $typenow === self::POST_TYPE) {
                $this->enqueue_edit_assets();
            }
        }

        /**
         * Enqueue assets for list view
         */
        private function enqueue_list_assets()
        {

            // Add inline styles - Minimal design
            $css = '
                .match-type-badge, .frequency-badge {
                    display: inline-block;
                    padding: 2px 6px;
                    border-radius: 2px;
                    font-size: 11px;
                    font-weight: 500;
                    text-transform: none;
                    border: 1px solid;
                }
                .match-type-or {
                    background: transparent;
                    color: #666;
                    border-color: #ddd;
                }
                .match-type-and {
                    background: transparent;
                    color: #666;
                    border-color: #ddd;
                }
                .frequency-daily {
                    background: transparent;
                    color: #666;
                    border-color: #ddd;
                }
                .frequency-weekly {
                    background: transparent;
                    color: #666;
                    border-color: #ddd;
                }
                .frequency-monthly {
                    background: transparent;
                    color: #666;
                    border-color: #ddd;
                }
                .taxonomies-summary {
                    font-size: 12px;
                    color: #666;
                }
                .taxonomies-count {
                    color: #666;
                }
            ';

            wp_add_inline_style('wp-admin', $css);

            // Add JavaScript for quick edit
            $js = "
            jQuery(document).ready(function($) {
                // Populate quick edit fields
                var \$wpInlineEdit = inlineEditPost.edit;
                inlineEditPost.edit = function(id) {
                    \$wpInlineEdit.apply(this, arguments);
                    var post_id = 0;
                    if (typeof(id) == 'object') {
                        post_id = parseInt(this.getId(id));
                    } else {
                        post_id = id;
                    }

                    if (post_id > 0) {
                        var \$row = $('#post-' + post_id);
                        var \$matchTypeCell = \$row.find('td.column-match_type');
                        var \$frequencyCell = \$row.find('td.column-frequency');

                        // Get match type from cell text or badge
                        var matchTypeText = \$matchTypeCell.text().trim();
                        var matchTypeValue = (matchTypeText.indexOf('AND') !== -1) ? '" . esc_js(self::MATCH_TYPE_AND) . "' : '" . esc_js(self::MATCH_TYPE_OR) . "';
                        $('#edit-' + post_id + ' select[name=\"" . esc_js(FELAN_METABOX_PREFIX . self::META_MATCH_TYPE) . "\"]').val(matchTypeValue);

                        // Get frequency from cell text or badge
                        var frequencyText = \$frequencyCell.text().trim().toLowerCase();
                        var frequencyValue = '" . esc_js(self::FREQUENCY_DAILY) . "';
                        if (frequencyText.indexOf('weekly') !== -1) {
                            frequencyValue = '" . esc_js(self::FREQUENCY_WEEKLY) . "';
                        } else if (frequencyText.indexOf('monthly') !== -1) {
                            frequencyValue = '" . esc_js(self::FREQUENCY_MONTHLY) . "';
                        }
                        $('#edit-' + post_id + ' select[name=\"" . esc_js(FELAN_METABOX_PREFIX . self::META_FREQUENCY) . "\"]').val(frequencyValue);
                    }
                };
            });
            ";

            wp_add_inline_script('jquery', $js);
        }

        /**
         * Enqueue assets for edit/add screen
         */
        private function enqueue_edit_assets()
        {
            // Add CSS for better UX
            $css = '
                .glf-field-project_alerts_email .glf-label label .required {
                    color: #d63638;
                    font-weight: bold;
                }
                .glf-field-project_alerts_email input[type="text"] {
                    border-left: 3px solid #d63638;
                }
                .glf-field-project_alerts_email input[type="text"]:valid {
                    border-left-color: #00a32a;
                }
                .glf-field-project_alerts_email input[type="text"][readonly] {
                    background-color: #f5f5f5;
                    cursor: not-allowed;
                }
            ';

            wp_add_inline_style('wp-admin', $css);

            // Add JavaScript to handle user type toggle and auto-fill email
            $user_type_field_name = FELAN_METABOX_PREFIX . self::META_USER_TYPE;
            $user_field_name = FELAN_METABOX_PREFIX . self::META_USER_ID;
            $email_field_name = FELAN_METABOX_PREFIX . self::META_EMAIL;

            $js = "
            jQuery(document).ready(function($) {
                function findUserTypeField() {
                    var selectors = [
                        'input[name=\"" . esc_js($user_type_field_name) . "\"]:checked',
                        '.glf-field-project_alerts_user_type input:checked',
                        '.glf-field[data-input-name*=\"project_alerts_user_type\"] input:checked'
                    ];
                    for (var i = 0; i < selectors.length; i++) {
                        var \$field = $(selectors[i]);
                        if (\$field.length) {
                            return \$field;
                        }
                    }
                    return $();
                }

                function findUserField() {
                    var selectors = [
                        'select[name=\"" . esc_js($user_field_name) . "\"]',
                        'select[data-input-name*=\"project_alerts_user_id\"]',
                        '.glf-field-project_alerts_user_id select',
                        '.glf-field[data-input-name*=\"project_alerts_user_id\"] select'
                    ];
                    for (var i = 0; i < selectors.length; i++) {
                        var \$field = $(selectors[i]);
                        if (\$field.length) {
                            return \$field;
                        }
                    }
                    return $();
                }

                function findEmailField() {
                    var selectors = [
                        'input[name=\"" . esc_js($email_field_name) . "\"]',
                        'input[data-input-name*=\"project_alerts_email\"]',
                        '.glf-field-project_alerts_email input[type=\"text\"]',
                        '.glf-field[data-input-name*=\"project_alerts_email\"] input[type=\"text\"]'
                    ];
                    for (var i = 0; i < selectors.length; i++) {
                        var \$field = $(selectors[i]);
                        if (\$field.length) {
                            return \$field;
                        }
                    }
                    return $();
                }

                function findUserFieldContainer() {
                    var selectors = [
                        '.glf-field-project_alerts_user_id',
                        '.glf-field[data-input-name*=\"project_alerts_user_id\"]',
                        '.felan-user-field-registered'
                    ];
                    for (var i = 0; i < selectors.length; i++) {
                        var \$field = $(selectors[i]);
                        if (\$field.length) {
                            return \$field.closest('.glf-field');
                        }
                    }
                    return $();
                }

                function attachUserChangeHandler() {
                    var \$userField = findUserField();
                    var \$emailField = findEmailField();
                    var \$userTypeField = findUserTypeField();

                    if (!\$userField.length || !\$emailField.length) {
                        return;
                    }

                    // Remove existing handlers to avoid duplicates
                    \$userField.off('change.felanUserEmail');

                    // Attach change handler for user field
                    \$userField.on('change.felanUserEmail', function() {
                        var \$userTypeField = findUserTypeField();
                        var userType = \$userTypeField.length ? \$userTypeField.val() : 'registered';

                        // Only auto-fill if registered user mode
                        if (userType === 'registered') {
                            var userId = $(this).val();
                            if (userId && userId !== '' && userId !== '0') {
                                $.ajax({
                                    url: ajaxurl,
                                    type: 'POST',
                                    data: {
                                        action: '" . esc_js(self::AJAX_GET_USER_EMAIL) . "',
                                        user_id: userId,
                                        nonce: '" . wp_create_nonce(self::AJAX_GET_USER_EMAIL) . "'
                                    },
                                    success: function(response) {
                                        if (response.success && response.data && response.data.email) {
                                            \$emailField.val(response.data.email);
                                            \$emailField.trigger('change');
                                        }
                                    },
                                    error: function() {
                                        // Silently fail - user can enter email manually
                                    }
                                });
                            } else {
                                \$emailField.val('');
                            }
                        }
                    });
                }

                function toggleUserType() {
                    var \$userTypeField = findUserTypeField();
                    var \$userField = findUserField();
                    var \$emailField = findEmailField();
                    var \$userContainer = findUserFieldContainer();

                    if (!\$userTypeField.length || !\$emailField.length) {
                        setTimeout(toggleUserType, 500);
                        return;
                    }

                    var userType = \$userTypeField.val();
                    // Store current email value before any changes
                    var currentEmail = \$emailField.val() || '';

                    if (userType === 'registered') {
                        // Show user field, make email readonly
                        \$userContainer.show();
                        \$emailField.prop('readonly', true);
                        \$emailField.css('background-color', '#f5f5f5');

                        // Attach user change handler
                        attachUserChangeHandler();

                        // Auto-fill email if user is already selected
                        if (\$userField.length) {
                            var userId = \$userField.val();
                            if (userId && userId !== '' && userId !== '0') {
                                $.ajax({
                                    url: ajaxurl,
                                    type: 'POST',
                                    data: {
                                        action: '" . esc_js(self::AJAX_GET_USER_EMAIL) . "',
                                        user_id: userId,
                                        nonce: '" . wp_create_nonce(self::AJAX_GET_USER_EMAIL) . "'
                                    },
                                    success: function(response) {
                                        if (response.success && response.data && response.data.email) {
                                            \$emailField.val(response.data.email);
                                            \$emailField.trigger('change');
                                        }
                                    }
                                });
                            } else if (!currentEmail) {
                                // No user selected and no email, clear field
                                \$emailField.val('');
                            }
                        }
                    } else {
                        // Hide user field, show email field as editable
                        \$userContainer.hide();
                        \$emailField.prop('readonly', false);
                        \$emailField.css('background-color', '');
                        // Preserve existing email value - don't clear it
                        // The email value should already be loaded from post meta by the framework
                        // Only ensure it's not cleared when toggling
                    }
                }

                function initUserTypeToggle() {
                    var \$userTypeField = findUserTypeField();

                    if (!\$userTypeField.length) {
                        setTimeout(initUserTypeToggle, 500);
                        return;
                    }

                    // Remove existing handlers
                    $(document).off('change.felanUserType', 'input[name=\"" . esc_js($user_type_field_name) . "\"]');

                    // Handle user type change - use event delegation
                    $(document).on('change.felanUserType', 'input[name=\"" . esc_js($user_type_field_name) . "\"]', function() {
                        toggleUserType();
                    });

                    // Also listen for button_set field changes
                    $(document).on('change.felanUserType', '.glf-field-button_set-inner input[type=\"radio\"]', function() {
                        var \$field = $(this);
                        if (\$field.attr('name') === '" . esc_js($user_type_field_name) . "') {
                            setTimeout(toggleUserType, 100);
                        }
                    });

                    // Initial toggle
                    toggleUserType();

                    // Attach user change handler initially
                    attachUserChangeHandler();
                }

                // Initialize
                initUserTypeToggle();

                // Re-initialize after framework loads
                $(document).on('glf_make_template_done', function() {
                    setTimeout(function() {
                        initUserTypeToggle();
                        attachUserChangeHandler();
                    }, 200);
                });
            });
            ";

            wp_add_inline_script('jquery', $js);

            // Add JavaScript for selectize AJAX loading with pagination
            $this->enqueue_selectize_ajax_script();
        }

        /**
         * Enqueue JavaScript for selectize AJAX loading with pagination
         */
        private function enqueue_selectize_ajax_script()
        {
            // Taxonomy mapping
            $taxonomy_map = array();
            foreach ($this->get_taxonomy_config() as $tax_data) {
                $taxonomy_map[FELAN_METABOX_PREFIX . $tax_data['meta_key']] = $tax_data['taxonomy'];
            }

            $user_field_id = FELAN_METABOX_PREFIX . self::META_USER_ID;

            $js = "
            jQuery(document).ready(function($) {
                var taxonomyMap = " . json_encode($taxonomy_map) . ";
                var userFieldId = '" . esc_js($user_field_id) . "';

                // Initialize user field with AJAX search
                function initUserFieldSelectize() {
                    var \$userField = $('#' + userFieldId).closest('.glf-field').find('.glf-field-selectize-inner select.glf-selectize');
                    if (!\$userField.length) {
                        \$userField = $('.glf-field[data-input-name*=\"project_alerts_user_id\"] .glf-field-selectize-inner select.glf-selectize');
                    }
                    if (!\$userField.length) {
                        \$userField = $('.glf-field-project_alerts_user_id .glf-field-selectize-inner select.glf-selectize');
                    }

                    if (\$userField.length) {
                        setTimeout(function() {
                            var selectize = \$userField[0].selectize;
                            if (selectize) {
                                // Get current value
                                var currentValue = selectize.getValue();

                                // Configure AJAX loading
                                selectize.settings.load = function(query, callback) {
                                    if (!query.length && selectize.options && Object.keys(selectize.options).length > 0) {
                                        callback(); // Don't reload if already has options
                                        return;
                                    }

                                    $.ajax({
                                        url: ajaxurl,
                                        type: 'POST',
                                        data: {
                                            action: '" . esc_js(self::AJAX_SEARCH_USERS) . "',
                                            search: query,
                                            page: 1,
                                            per_page: 20,
                                            include: currentValue ? [currentValue] : [],
                                            nonce: '" . wp_create_nonce(self::AJAX_SEARCH_USERS) . "'
                                        },
                                        success: function(response) {
                                            if (response.success && response.data && response.data.options) {
                                                callback(response.data.options);
                                            } else {
                                                callback();
                                            }
                                        },
                                        error: function() {
                                            callback();
                                        }
                                    });
                                };

                                // Load current value if exists
                                if (currentValue) {
                                    $.ajax({
                                        url: ajaxurl,
                                        type: 'POST',
                                        data: {
                                            action: '" . esc_js(self::AJAX_SEARCH_USERS) . "',
                                            include: [currentValue],
                                            nonce: '" . wp_create_nonce(self::AJAX_SEARCH_USERS) . "'
                                        },
                                        success: function(response) {
                                            if (response.success && response.data && response.data.options) {
                                                response.data.options.forEach(function(option) {
                                                    if (!selectize.options[option.id]) {
                                                        selectize.addOption(option);
                                                    }
                                                });
                                                selectize.setValue(currentValue);
                                            }
                                        }
                                    });
                                }

                                // Handle change event for email auto-fill
                                selectize.on('change', function(value) {
                                    if (value && value !== '') {
                                        // Trigger email auto-fill
                                        var \$emailField = $('input[name*=\"project_alerts_email\"]');
                                        if (\$emailField.length) {
                                            $.ajax({
                                                url: ajaxurl,
                                                type: 'POST',
                                                data: {
                                                    action: '" . esc_js(self::AJAX_GET_USER_EMAIL) . "',
                                                    user_id: value,
                                                    nonce: '" . wp_create_nonce(self::AJAX_GET_USER_EMAIL) . "'
                                                },
                                                success: function(response) {
                                                    if (response.success && response.data && response.data.email) {
                                                        \$emailField.val(response.data.email);
                                                        \$emailField.trigger('change');
                                                    }
                                                }
                                            });
                                        }
                                    }
                                });
                            }
                        }, 500);
                    }
                }

                function initSelectizeAjax() {
                    $('.glf-field-selectize-inner').each(function() {
                        var \$container = $(this);
                        var \$field = \$container.closest('.glf-field');
                        var fieldId = \$field.attr('id');

                        // Skip user field - handled separately
                        if (fieldId === userFieldId || \$field.find('[data-ajax-user-search]').length) {
                            return;
                        }

                        if (!fieldId || !taxonomyMap[fieldId]) {
                            return;
                        }

                        var taxonomy = taxonomyMap[fieldId];
                        var \$select = \$container.find('select.glf-selectize');

                        if (\$select.length) {
                            // Add data attribute for identification
                            \$select.attr('data-taxonomy', taxonomy);

                            // Wait for selectize to initialize
                            setTimeout(function() {
                                var selectize = \$select[0].selectize;

                                if (selectize) {
                                    // Get current options count
                                    var optionsCount = Object.keys(selectize.options).length;

                                    // If too many options (>50), enable AJAX search
                                    if (optionsCount > 50) {
                                        // Store selected values
                                        var selectedValues = selectize.getValue() || [];

                                        // Configure AJAX loading for search
                                        selectize.settings.load = function(query, callback) {
                                            if (!query.length) {
                                                // For empty query, load first 50
                                                loadInitialOptions(callback);
                                                return;
                                            }

                                            // Search via AJAX
                                            $.ajax({
                                                url: ajaxurl,
                                                type: 'POST',
                                                data: {
                                                    action: '" . esc_js(self::AJAX_SEARCH_TAXONOMY_TERMS) . "',
                                                    taxonomy: taxonomy,
                                                    search: query,
                                                    page: 1,
                                                    per_page: 50,
                                                    selected: selectedValues,
                                                    nonce: '" . wp_create_nonce(self::AJAX_SEARCH_TAXONOMY_TERMS) . "'
                                                },
                                                error: function() {
                                                    callback();
                                                },
                                                success: function(response) {
                                                    if (response.success && response.data.options) {
                                                        var options = [];
                                                        $.each(response.data.options, function(id, name) {
                                                            options.push({value: id, text: name});
                                                        });
                                                        callback(options);
                                                    } else {
                                                        callback();
                                                    }
                                                }
                                            });
                                        };

                                        // Function to load initial options
                                        function loadInitialOptions(callback) {
                                            $.ajax({
                                                url: ajaxurl,
                                                type: 'POST',
                                                data: {
                                                    action: '" . esc_js(self::AJAX_SEARCH_TAXONOMY_TERMS) . "',
                                                    taxonomy: taxonomy,
                                                    search: '',
                                                    page: 1,
                                                    per_page: 50,
                                                    selected: selectedValues,
                                                    nonce: '" . wp_create_nonce(self::AJAX_SEARCH_TAXONOMY_TERMS) . "'
                                                },
                                                success: function(response) {
                                                    if (response.success && response.data.options) {
                                                        var options = [];
                                                        $.each(response.data.options, function(id, name) {
                                                            options.push({value: id, text: name});
                                                        });
                                                        callback(options);

                                                        // Restore selected values after loading
                                                        if (selectedValues && selectedValues.length) {
                                                            setTimeout(function() {
                                                                selectize.setValue(selectedValues);
                                                            }, 100);
                                                        }
                                                    } else {
                                                        callback();
                                                    }
                                                },
                                                error: function() {
                                                    callback();
                                                }
                                            });
                                        }

                                        // Clear and reload with limited options
                                        selectize.clearOptions();
                                        loadInitialOptions(function(options) {
                                            if (options && options.length) {
                                                selectize.addOption(options);
                                                if (selectedValues && selectedValues.length) {
                                                    selectize.setValue(selectedValues);
                                                }
                                            }
                                        });
                                    }
                                }
                            }, 500);
                        }
                    });
                }

                // Initialize user field
                initUserFieldSelectize();

                // Wait for selectize to initialize
                $(document).on('glf_make_template_done', function() {
                    setTimeout(function() {
                        initSelectizeAjax();
                        initUserFieldSelectize();
                    }, 800);
                });

                // Also init on page load with delay
                setTimeout(function() {
                    initSelectizeAjax();
                    initUserFieldSelectize();
                }, 1500);
            });
            ";

            wp_add_inline_script('jquery', $js);
        }

        /**
         * Validate project alert metabox on save
         * @param int $post_id
         * @param WP_Post $post
         */
        /**
         * Validate project alert metabox on save
         * @param int $post_id
         * @param WP_Post $post
         */
        public function validate_project_alert_metabox($post_id, $post)
        {
            // Only for project_alerts post type
            if (!isset($post->post_type) || $post->post_type !== self::POST_TYPE) {
                return;
            }

            // Skip autosave
            if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
                return;
            }

            // Skip if post is being trashed or deleted
            if (isset($_POST['action']) && ($_POST['action'] === 'trash' || $_POST['action'] === 'delete')) {
                return;
            }

            // Skip if post status is trash
            if (isset($post->post_status) && $post->post_status === 'trash') {
                return;
            }

            // Check permissions
            if (!current_user_can('edit_post', $post_id)) {
                return;
            }

            // Validate email (required)
            $user_type_key = FELAN_METABOX_PREFIX . self::META_USER_TYPE;
            $email_key = FELAN_METABOX_PREFIX . self::META_EMAIL;
            $user_id_key = FELAN_METABOX_PREFIX . self::META_USER_ID;

            // Get POST data safely
            $post_data = isset($_POST) && is_array($_POST) ? $_POST : array();

            $user_type = isset($post_data[$user_type_key])
                ? sanitize_text_field($post_data[$user_type_key])
                : 'registered';

            $email = isset($post_data[$email_key])
                ? sanitize_email($post_data[$email_key])
                : '';

            $user_id = isset($post_data[$user_id_key])
                ? intval($post_data[$user_id_key])
                : 0;

            // If registered user is selected, get email from user
            if ($user_type === 'registered' && $user_id > 0) {
                $user = get_user_by('ID', $user_id);
                if ($user && !empty($user->user_email)) {
                    $email = $user->user_email;
                    // Auto-fill email from user
                    update_post_meta($post_id, $email_key, $email);
                }
            }

            // Save user_id and user_type
            if ($user_type === 'registered' && $user_id > 0) {
                update_post_meta($post_id, $user_id_key, $user_id);
                update_post_meta($post_id, $user_type_key, 'registered');
            } else {
                // Non-registered user - clear user_id
                update_post_meta($post_id, $user_id_key, 0);
                update_post_meta($post_id, $user_type_key, 'non_registered');
            }

            // Save email (always required)
            if (!empty($email)) {
                update_post_meta($post_id, $email_key, $email);
            }

            // Email is always required

            // Validate email format
            if (!empty($email) && !is_email($email)) {
                // Remove action to prevent infinite loop
                remove_action('save_post', array($this, 'validate_project_alert_metabox'), 10);

                // Set post to draft
                wp_update_post(array(
                    'ID' => $post_id,
                    'post_status' => 'draft',
                ));

                // Add admin notice
                add_action('admin_notices', function() {
                    echo '<div class="notice notice-error"><p>' . esc_html__('Project Alert could not be saved. Please provide a valid email address.', 'felan-framework') . '</p></div>';
                });

                // Re-add action
                add_action('save_post', array($this, 'validate_project_alert_metabox'), 10, 2);
                return;
            }

            // If still no email, show error
            if (empty($email)) {
                remove_action('save_post', array($this, 'validate_project_alert_metabox'), 10);

                wp_update_post(array(
                    'ID' => $post_id,
                    'post_status' => 'draft',
                ));

                if ($user_type === 'registered') {
                    $error_message = $user_id > 0
                        ? esc_html__('Project Alert could not be saved. Selected user does not have a valid email address. Please select a different user or switch to Non-Registered User mode.', 'felan-framework')
                        : esc_html__('Project Alert could not be saved. Please select a registered user.', 'felan-framework');
                } else {
                    $error_message = esc_html__('Project Alert could not be saved. Email address is required for non-registered users. Please enter an email address.', 'felan-framework');
                }

                add_action('admin_notices', function() use ($error_message) {
                    echo '<div class="notice notice-error"><p>' . $error_message . '</p></div>';
                });

                add_action('save_post', array($this, 'validate_project_alert_metabox'), 10, 2);
            }
        }

        /**
         * AJAX handler to get user email
         */
        /**
         * AJAX handler to get user email
         */
        public function ajax_get_user_email()
        {
            check_ajax_referer(self::AJAX_GET_USER_EMAIL, 'nonce');

            $user_id = isset($_POST['user_id']) ? intval($_POST['user_id']) : 0;

            if ($user_id <= 0) {
                wp_send_json_error(array('message' => esc_html__('Invalid user ID', 'felan-framework')));
            }

            $user = get_user_by('ID', $user_id);
            if (!$user) {
                wp_send_json_error(array('message' => esc_html__('User not found', 'felan-framework')));
            }

            wp_send_json_success(array('email' => $user->user_email));
        }

        /**
         * AJAX handler to search users
         */
        /**
         * AJAX handler to search users
         */
        public function ajax_search_users()
        {
            check_ajax_referer(self::AJAX_SEARCH_USERS, 'nonce');

            // Get POST data
            $post_data = isset($_POST) ? $_POST : array();

            $search = isset($post_data['search']) ? sanitize_text_field($post_data['search']) : '';
            $page = isset($post_data['page']) ? intval($post_data['page']) : 1;
            $per_page = isset($post_data['per_page']) ? intval($post_data['per_page']) : 20;
            $include = isset($post_data['include']) && is_array($post_data['include'])
                ? array_map('intval', $post_data['include'])
                : array();

            // Build query args
            $args = array(
                'role' => self::FREELANCER_ROLE,
                'number' => $per_page,
                'offset' => ($page - 1) * $per_page,
                'orderby' => 'display_name',
                'order' => 'ASC',
            );

            // Add search
            if (!empty($search)) {
                $args['search'] = '*' . esc_attr($search) . '*';
                $args['search_columns'] = array('user_login', 'user_nicename', 'user_email', 'display_name');
            }

            // If include is specified, get those users first
            $include_users = array();
            if (!empty($include)) {
                $include_users = get_users(array(
                    'include' => $include,
                    'role' => self::FREELANCER_ROLE,
                ));
            }

            // Get users
            $users = get_users($args);

            // Merge included users if not already in results
            if (!empty($include_users)) {
                $user_ids = wp_list_pluck($users, 'ID');
                foreach ($include_users as $user) {
                    if (!in_array($user->ID, $user_ids)) {
                        $users[] = $user;
                    }
                }
            }

            // Get total count for pagination
            $total = count_users();
            $total_freelancers = isset($total['avail_roles'][self::FREELANCER_ROLE])
                ? $total['avail_roles'][self::FREELANCER_ROLE]
                : 0;

            // Format response
            $options = array();
            foreach ($users as $user) {
                $display_name = $user->display_name;
                $email = $user->user_email;
                $label = $display_name . ' (' . $email . ')';
                $options[] = array(
                    'id' => $user->ID,
                    'name' => $label,
                    'value' => $user->ID,
                    'text' => $label,
                );
            }

            wp_send_json_success(array(
                'options' => $options,
                'total' => $total_freelancers,
                'page' => $page,
                'per_page' => $per_page,
                'has_more' => ($page * $per_page) < $total_freelancers,
            ));
        }

        /**
         * AJAX handler to search taxonomy terms with pagination
         */
        /**
         * AJAX handler to search taxonomy terms with pagination
         */
        public function ajax_search_taxonomy_terms()
        {
            check_ajax_referer(self::AJAX_SEARCH_TAXONOMY_TERMS, 'nonce');

            // Get POST data
            $post_data = isset($_POST) ? $_POST : array();

            $taxonomy = isset($post_data['taxonomy']) ? sanitize_text_field($post_data['taxonomy']) : '';
            $search = isset($post_data['search']) ? sanitize_text_field($post_data['search']) : '';
            $page = isset($post_data['page']) ? intval($post_data['page']) : 1;
            $per_page = isset($post_data['per_page']) ? intval($post_data['per_page']) : 50;
            $selected = isset($post_data['selected']) && is_array($post_data['selected'])
                ? array_map('intval', $post_data['selected'])
                : array();

            if (empty($taxonomy)) {
                wp_send_json_error(array('message' => esc_html__('Taxonomy is required', 'felan-framework')));
            }

            // Validate taxonomy exists
            if (!taxonomy_exists($taxonomy)) {
                wp_send_json_error(array('message' => esc_html__('Invalid taxonomy', 'felan-framework')));
            }

            // Build query args
            $args = array(
                'taxonomy' => $taxonomy,
                'hide_empty' => false,
                'number' => $per_page,
                'offset' => ($page - 1) * $per_page,
                'orderby' => 'name',
                'order' => 'ASC',
            );

            // Add search
            if (!empty($search)) {
                $args['search'] = $search;
            }

            // Always include selected terms
            if (!empty($selected)) {
                $selected_terms = get_terms(array(
                    'taxonomy' => $taxonomy,
                    'include' => $selected,
                    'hide_empty' => false,
                ));

                $all_terms = get_terms($args);
                $all_term_ids = wp_list_pluck($all_terms, 'term_id');

                // Merge selected terms that are not in current page
                foreach ($selected_terms as $term) {
                    if (!in_array($term->term_id, $all_term_ids)) {
                        $all_terms[] = $term;
                    }
                }
            } else {
                $all_terms = get_terms($args);
            }

            // Get total count for pagination
            $count_args = $args;
            unset($count_args['number'], $count_args['offset']);
            $total = wp_count_terms($count_args);

            // Format response
            $options = array();
            foreach ($all_terms as $term) {
                if (!is_wp_error($term)) {
                    $options[$term->term_id] = $term->name;
                }
            }

            wp_send_json_success(array(
                'options' => $options,
                'total' => $total,
                'page' => $page,
                'per_page' => $per_page,
                'has_more' => ($page * $per_page) < $total,
            ));
        }

        /**
         * Ensure trash row actions are available
         * @param array $actions
         * @param WP_Post $post
         * @return array
         */
        public function ensure_trash_row_actions($actions, $post)
        {
            if ($post->post_type !== self::POST_TYPE) {
                return $actions;
            }

            // Ensure trash link is available if user can delete
            if (isset($post->post_status) && $post->post_status !== 'trash') {
                if (current_user_can('delete_post', $post->ID)) {
                    // WordPress should automatically add trash link, but ensure it exists
                    if (!isset($actions['trash'])) {
                        $trash_url = get_delete_post_link($post->ID);
                        if ($trash_url) {
                            $actions['trash'] = sprintf(
                                '<a href="%s" class="submitdelete" aria-label="%s">%s</a>',
                                esc_url($trash_url),
                                esc_attr(sprintf(__('Move &#8220;%s&#8221; to the Trash', 'felan-framework'), $post->post_title)),
                                __('Trash', 'felan-framework')
                            );
                        }
                    }
                }
            }

            return $actions;
        }

        /**
         * Map meta capabilities for Project Alerts
         * @param array $caps Required capabilities
         * @param string $cap Capability being checked
         * @param int $user_id User ID
         * @param array $args Additional arguments
         * @return array
         */
        public function map_project_alerts_meta_cap($caps, $cap, $user_id, $args)
        {
            // Check if we're in project_alerts context
            $post_type = isset($_GET['post_type']) ? $_GET['post_type'] : '';
            if (empty($post_type) && isset($args[0])) {
                $post = get_post($args[0]);
                if ($post) {
                    $post_type = $post->post_type;
                }
            }

            // Only handle project_alerts post type
            if ($post_type !== self::POST_TYPE && !in_array($cap, array('edit_posts', 'edit_others_posts', 'publish_posts', 'delete_posts'))) {
                return $caps;
            }

            // Handle edit_posts capability when in project_alerts context
            if (($cap === 'edit_posts' || $cap === 'edit_post') && ($post_type === self::POST_TYPE || empty($post_type))) {
                // Check if we're actually in project_alerts context
                if (empty($post_type)) {
                    $post_type = isset($_GET['post_type']) ? $_GET['post_type'] : '';
                }
                if ($post_type === self::POST_TYPE) {
                    $user = get_userdata($user_id);
                    if ($user && in_array('administrator', $user->roles)) {
                        // Admin has edit_posts capability, allow it
                        return array('edit_posts');
                    }
                }
            }

            // Only handle project_alerts capabilities
            $project_alerts_caps = array(
                'edit_project_alerts',
                'read_project_alert',
                'delete_project_alert',
                'edit_others_project_alerts',
                'publish_project_alerts',
                'read_private_project_alerts',
                'delete_project_alerts',
                'delete_private_project_alerts',
                'delete_published_project_alerts',
                'delete_others_project_alerts',
                'edit_private_project_alerts',
                'edit_published_project_alerts',
            );

            if (!in_array($cap, $project_alerts_caps)) {
                return $caps;
            }

            // Map to standard post capabilities for admin
            $user = get_userdata($user_id);
            if ($user && in_array('administrator', $user->roles)) {
                // Map project_alerts capabilities to standard post capabilities
                $cap_mapping = array(
                    'edit_project_alerts' => 'edit_posts',
                    'read_project_alert' => 'read',
                    'delete_project_alert' => 'delete_posts',
                    'edit_others_project_alerts' => 'edit_others_posts',
                    'publish_project_alerts' => 'publish_posts',
                    'read_private_project_alerts' => 'read_private_posts',
                    'delete_project_alerts' => 'delete_posts',
                    'delete_private_project_alerts' => 'delete_private_posts',
                    'delete_published_project_alerts' => 'delete_published_posts',
                    'delete_others_project_alerts' => 'delete_others_posts',
                    'edit_private_project_alerts' => 'edit_private_posts',
                    'edit_published_project_alerts' => 'edit_published_posts',
                );

                if (isset($cap_mapping[$cap])) {
                    return array($cap_mapping[$cap]);
                }
            }

            return $caps;
        }

        /**
         * Grant project_alerts capabilities to admin
         * @param array $allcaps All capabilities
         * @param array $caps Required capabilities
         * @param array $args Additional arguments
         * @param WP_User $user User object
         * @return array
         */
        public function grant_project_alerts_caps_to_admin($allcaps, $caps, $args, $user)
        {
            // Only for admin
            if (!isset($user->roles) || !in_array('administrator', $user->roles)) {
                return $allcaps;
            }

            // Check if we're in project_alerts context
            $post_type = isset($_GET['post_type']) ? sanitize_text_field($_GET['post_type']) : '';
            if (empty($post_type) && isset($args[0]) && is_numeric($args[0])) {
                $post = get_post($args[0]);
                if ($post) {
                    $post_type = $post->post_type;
                }
            }

            // Always grant project_alerts capabilities to admin
            $project_alerts_caps = array(
                'edit_project_alerts',
                'read_project_alert',
                'delete_project_alert',
                'edit_others_project_alerts',
                'publish_project_alerts',
                'read_private_project_alerts',
                'delete_project_alerts',
                'delete_private_project_alerts',
                'delete_published_project_alerts',
                'delete_others_project_alerts',
                'edit_private_project_alerts',
                'edit_published_project_alerts',
            );

            foreach ($project_alerts_caps as $cap) {
                $allcaps[$cap] = true;
            }

            // Also grant edit_posts if in project_alerts context or if checking edit_posts capability
            if ($post_type === self::POST_TYPE || in_array('edit_posts', $caps)) {
                // Check if we're actually checking for project_alerts
                if ($post_type === self::POST_TYPE || (isset($_GET['post_type']) && $_GET['post_type'] === self::POST_TYPE)) {
                    $allcaps['edit_posts'] = true;
                    $allcaps['edit_others_posts'] = true;
                    $allcaps['publish_posts'] = true;
                    $allcaps['delete_posts'] = true;
                }
            }

            return $allcaps;
        }
    }
}

