<?php
if (!defined('ABSPATH')) {
	exit; // Exit if accessed directly
}

// Load traits
require_once FELAN_PLUGIN_DIR . 'includes/admin/traits/trait-capabilities.php';
require_once FELAN_PLUGIN_DIR . 'includes/admin/traits/trait-post-types.php';
require_once FELAN_PLUGIN_DIR . 'includes/admin/traits/trait-taxonomies.php';
require_once FELAN_PLUGIN_DIR . 'includes/admin/traits/trait-term-meta.php';
require_once FELAN_PLUGIN_DIR . 'includes/admin/metaboxes/trait-metaboxes.php';

// Load option traits
require_once FELAN_PLUGIN_DIR . 'includes/admin/options/trait-general-options.php';
require_once FELAN_PLUGIN_DIR . 'includes/admin/options/trait-settings-options.php';
require_once FELAN_PLUGIN_DIR . 'includes/admin/options/trait-payment-options.php';
require_once FELAN_PLUGIN_DIR . 'includes/admin/options/trait-user-options.php';
require_once FELAN_PLUGIN_DIR . 'includes/admin/options/trait-post-type-options.php';
require_once FELAN_PLUGIN_DIR . 'includes/admin/options/trait-email-options.php';

if (!class_exists('Felan_Admin')) {
	/**
	 * Class Felan_Admin
	 */
	class Felan_Admin
	{
		// Use traits for modular code
		use Felan_Capabilities_Trait;
		use Felan_Post_Types_Trait;
		use Felan_Taxonomies_Trait;
		use Felan_Term_Meta_Trait;
		use Felan_Metaboxes_Trait;

		// Option traits
		use Felan_General_Options_Trait;
		use Felan_Settings_Options_Trait;
		use Felan_Payment_Options_Trait;
		use Felan_User_Options_Trait;
		use Felan_Post_Type_Options_Trait;
		use Felan_Email_Options_Trait;

		public function __construct()
		{
			add_action('init', array($this, 'location_rewrite_rules'));
			add_filter('term_link', array($this, 'location_term_link'), 10, 3);

			// Auto-generate ID for custom fields before saving
			add_filter('pre_update_option_felan-framework', array($this, 'auto_generate_custom_field_ids'), 10, 2);
			// For multilingual sites
			if (function_exists('pll_the_languages')) {
				add_filter('pre_update_option', array($this, 'auto_generate_custom_field_ids_multilang'), 10, 3);
			}
			if (defined('ICL_SITEPRESS_VERSION')) {
				add_filter('pre_update_option', array($this, 'auto_generate_custom_field_ids_multilang'), 10, 3);
			}

			// Enqueue admin custom field scripts
			add_action('admin_enqueue_scripts', array($this, 'enqueue_custom_field_scripts'));
		}

		/**
		 * Auto-generate ID for custom fields if empty
		 */
		public function auto_generate_custom_field_ids($value, $old_value)
		{
			$custom_field_keys = array(
				'custom_field_jobs',
				'custom_field_company',
				'custom_field_freelancer',
				'custom_field_project',
				'custom_field_service'
			);

			foreach ($custom_field_keys as $key) {
				if (isset($value[$key]) && is_array($value[$key])) {
					foreach ($value[$key] as $index => $field) {
						if (!empty($field['label']) && empty($field['id'])) {
							$value[$key][$index]['id'] = sanitize_title($field['label']);
						}
					}
				}
			}

			return $value;
		}

		/**
		 * Auto-generate ID for multilingual sites
		 */
		public function auto_generate_custom_field_ids_multilang($value, $option, $old_value)
		{
			// Check if this is a felan-framework option (with language prefix)
			if (preg_match('/^[a-z]{2}_felan-framework$/', $option) || $option === 'felan-framework') {
				return $this->auto_generate_custom_field_ids($value, $old_value);
			}
			return $value;
		}

		/**
		 * Validate wire transfer settings for multilingual sites
		 */
		public function validate_wire_transfer_settings_multilang($value, $option, $old_value)
		{
			// Check if this is a felan-framework option (with language prefix)
			if (preg_match('/^[a-z]{2}_felan-framework$/', $option) || $option === 'felan-framework') {
				return $this->validate_wire_transfer_settings($value);
			}
			return $value;
		}

		/**
		 * Enqueue admin custom field scripts and styles
		 */
		public function enqueue_custom_field_scripts($hook)
		{
			// Load on any admin page that might have theme options
			// Hook examples: toplevel_page_theme-options, appearance_page_theme-options, etc.
			wp_enqueue_script(
				'felan-admin-custom-field',
				FELAN_PLUGIN_URL . 'assets/js/admin-custom-field.js',
				array('jquery'),
				FELAN_PLUGIN_VER,
				true
			);

			wp_localize_script('felan-admin-custom-field', 'felanAdminCustomField', array(
				'idLockedMessage' => __('ID cannot be changed after being set', 'felan-framework')
			));
		}

		public function location_rewrite_rules()
		{
			$post_types = ['jobs', 'company', 'service', 'project', 'freelancer'];

			foreach ($post_types as $post_type) {
				add_rewrite_rule(
					"{$post_type}-location/([^/]+)/?$",
					'index.php?felan_location=$matches[1]&post_type=' . $post_type,
					'top'
				);
				add_rewrite_rule(
					"{$post_type}-state/([^/]+)/?$",
					'index.php?felan_state=$matches[1]&post_type=' . $post_type,
					'top'
				);
			}
		}

		public function location_term_link($termlink, $term, $taxonomy)
		{
			if ($taxonomy === 'felan_location') {
				$post_type = get_query_var('post_type');
				if (in_array($post_type, ['jobs', 'company', 'service', 'project', 'freelancer'])) {
					return home_url("/{$post_type}-location/{$term->slug}/");
				}
			}
			if ($taxonomy === 'felan_state') {
				$post_type = get_query_var('post_type');
				if (in_array($post_type, ['jobs', 'company', 'service', 'project', 'freelancer'])) {
					return home_url("/{$post_type}-state/{$term->slug}/");
				}
			}
			return $termlink;
		}

		/**
		 * Remove admin bar
		 * @return bool
		 */
		function remove_admin_bar()
		{
			if (!current_user_can('administrator') && !is_admin()) {
				show_admin_bar(false);
			}
		}

		/**
		 * Check if it is a jobs edit page.
		 * @return bool
		 */
		public function is_felan_admin()
		{
			if (is_admin()) {
				global $pagenow;
				if (in_array($pagenow, array('edit.php', 'post.php', 'post-new.php', 'edit-tags.php'))) {
					global $post_type;
					if ('jobs' == $post_type) {
						return true;
					}
				}
			}
			return false;
		}

		/**
		 * Register admin_menu
		 */
		public function admin_menu()
		{
			// Temporarily disabled - Claim Listing feature
			if (false) {
				$enable_claim_listing = felan_get_option('enable_claim_listing', '0');
				if ($enable_claim_listing) :
					add_menu_page(
						esc_html__('Claim Listing', 'felan-framework'),
						esc_html__('Claim Listing', 'felan-framework'),
						'manage_options',
						'claim_listing',
						array($this, 'menu_claim_listing_callback'),
						'dashicons-list-view',
						12
					);
				endif;
			}
		}

		// Temporarily disabled - Claim Listing feature
		public function menu_claim_listing_callback()
		{
			$claim_email = $claim_name = $claim_username = $claim_status = '';

			$meta_query = array(
				'relative' => 'AND',
				array(
					'key' => 'felan-claim_request',
					'value' => 1,
					'compare' => '=',
				),
			);
			if (isset($_GET['claim_name']) && $_GET['claim_name'] != '') {
				$claim_name = $_GET['claim_name'];
				$meta_query[] = array(
					'key' => 'felan-cd_your_name',
					'value' => $_GET['claim_name'],
					'compare' => 'LIKE',
				);
			}
			if (isset($_GET['claim_email']) && $_GET['claim_email'] != '') {
				$claim_email = $_GET['claim_email'];
				$meta_query[] = array(
					'key' => 'felan-cd_your_email',
					'value' => $_GET['claim_email'],
					'compare' => 'LIKE',
				);
			}
			if (isset($_GET['claim_username']) && $_GET['claim_username'] != '') {
				$claim_username = $_GET['claim_username'];
				$meta_query[] = array(
					'key' => 'felan-cd_your_username',
					'value' => $_GET['claim_username'],
					'compare' => 'LIKE',
				);
			}
			if (isset($_GET['claim_status']) && $_GET['claim_status'] != '') {
				$claim_status = $_GET['claim_status'];
				$meta_query[] = array(
					'key' => 'felan-cd_status',
					'value' => $_GET['claim_status'],
					'compare' => '=',
				);
			}
			$paged = isset($_REQUEST['paged']) ? max(1, (int)$_REQUEST['paged']) : 1;
			$args = array(
				'post_type' => 'jobs',
				'posts_per_page' => 20,
				'paged' => $paged,
				'post_status' => 'publish',
				'meta_query' => $meta_query,
			);
			// The Query
			$the_query = new WP_Query($args);
			$count = $the_query->found_posts;
?>
			<div class="felan-wrap wrap about-wrap claim-wrap">
				<div class="entry-search">
					<div class="claim-action">
						<a href="#" class="button button-delete"><?php esc_html_e('Delete', 'felan-framework'); ?></a>
					</div>
					<form action="" method="GET" class="claimFilter">
						<div class="field-group">
							<input type="text" name="claim_name" value="<?php echo $claim_name; ?>" placeholder="<?php esc_html_e('Name', 'felan-framework'); ?>">
							<input type="email" name="claim_email" value="<?php echo $claim_email; ?>" placeholder="<?php esc_html_e('Email', 'felan-framework'); ?>">
							<input type="text" name="claim_username" value="<?php echo $claim_username; ?>" placeholder="<?php esc_html_e('Username', 'felan-framework'); ?>">
							<select name="claim_status" id="claim_status">
								<option value=""><?php esc_html_e('All Status', 'felan-framework'); ?></option>
								<option value="pending" <?php if ($claim_status == 'pending') {
																					echo 'selected';
																				} ?>><?php esc_html_e('Pending', 'felan-framework'); ?></option>
								<option value="accept" <?php if ($claim_status == 'accept') {
																					echo 'selected';
																				} ?>><?php esc_html_e('Accept', 'felan-framework'); ?></option>
								<option value="refuse" <?php if ($claim_status == 'refuse') {
																					echo 'selected';
																				} ?>><?php esc_html_e('Refuse', 'felan-framework'); ?></option>
							</select>
							<input type="hidden" name="page" value="claim_listing">
							<input type="submit" name="submit" value="<?php esc_html_e('Filter', 'felan-framework'); ?>">
						</div>
					</form>
					<div class="total"><?php printf(_n('%s item', '%s items', $count, 'felan-framework'), '<span class="count">' . esc_html($count) . '</span>'); ?></div>
				</div>

				<div class="wrap-content">
					<form action="" method="POST">
						<table class="table-changelogs">
							<thead>
								<tr>
									<th><input type="checkbox" id="checkall" name="claim_item"></th>
									<th><?php esc_html_e('Name', 'felan-framework'); ?></th>
									<th><?php esc_html_e('Email', 'felan-framework'); ?></th>
									<th><?php esc_html_e('Username', 'felan-framework'); ?></th>
									<th><?php esc_html_e('Listing Url', 'felan-framework'); ?></th>
									<th><?php esc_html_e('Message', 'felan-framework'); ?></th>
									<th><?php esc_html_e('Status', 'felan-framework'); ?></th>
									<th><?php esc_html_e('Action', 'felan-framework'); ?></th>
								</tr>
							</thead>
							<tbody>
								<?php
								// The Loop
								if ($the_query->have_posts()) {

									$i = 0;
									while ($the_query->have_posts()) {
										$the_query->the_post();
										$i++;
										$id = get_the_ID();
										$cd_your_name = get_post_meta($id, FELAN_METABOX_PREFIX . 'cd_your_name', true);
										$cd_your_email = get_post_meta($id, FELAN_METABOX_PREFIX . 'cd_your_email', true);
										$cd_your_listing = get_post_meta($id, FELAN_METABOX_PREFIX . 'cd_your_listing', true);
										$cd_your_username = get_post_meta($id, FELAN_METABOX_PREFIX . 'cd_your_username', true);
										$cd_messager = get_post_meta($id, FELAN_METABOX_PREFIX . 'cd_messager', true);
										$cd_status = get_post_meta($id, FELAN_METABOX_PREFIX . 'cd_status', true);
										$verified_listing = get_post_meta($id, FELAN_METABOX_PREFIX . 'verified_listing', true);
										echo '<tr>';
										echo '<td><input type="checkbox" name="claim_item"></td>';
										echo '<td>' . $cd_your_name . '</td>';
										echo '<td>' . $cd_your_email . '</td>';
										echo '<td>' . $cd_your_username . '</td>';
										echo '<td><a href="' . $cd_your_listing . '" target="_Blank">' . $cd_your_listing . '</a></td>';
										echo '<td>' . $cd_messager . '</td>';
										if ($cd_status == 'pending') {
											$value = 'pending';
										} else if ($verified_listing == 1) {
											$value = 'accept';
										} else {
											$value = 'refuse';
										}
										$data = 'data-status="' . $value . '"';
										echo '<td class="status"' . $data . '>' . $value . '</td>';
										echo '<td>
                                                <input type="submit" data-status="accept" data-jobs_id="' . $id . '" class="button button-primary" value="' . esc_attr('Accept', 'felan-framework') . '">
                                                <input type="submit" data-status="refuse" data-jobs_id="' . $id . '" class="button button-secondary" value="' . esc_attr('Refuse', 'felan-framework') . '">
                                                <input type="submit" data-status="delete" data-jobs_id="' . $id . '" class="button button-delete" value="' . esc_attr('Delete', 'felan-framework') . '"></td>';
										echo '</tr>';
									}
								} else {
									echo '<tr class="align-center">';
									echo '<td colspan="7">' . esc_attr('No result', 'felan-framework') . '</td>';
									echo '</tr>';
								}
								/* Restore original Post Data */
								wp_reset_postdata();
								?>
							</tbody>
						</table>
						<div class="pagination">
							<?php
							$big = 999999999; // need an unlikely integer

							echo paginate_links(array(
								'base' => admin_url('admin.php?page=claim_listing&paged=%#%'),
								'format' => '?paged=%#%',
								'current' => max(1, $paged),
								'total' => $the_query->max_num_pages,
								'prev_text' => '<i class="far fa-angle-left"></i>',
								'next_text' => '<i class="far fa-angle-right"></i>',
							));
							?>
						</div>
						<div class="felan-loading-effect"><span class="felan-dual-ring small"></span></div>
					</form>
				</div><!-- end .wrap-content -->
			</div>
<?php
		}

		// =====================================================
		// POST TYPES METHODS - Moved to traits/trait-post-types.php
		// Use: Felan_Post_Types_Trait
		// Includes: register_post_type(), register_post_status()
		// =====================================================

		// =====================================================
		// CAPABILITIES METHODS - Moved to traits/trait-capabilities.php
		// Use: Felan_Capabilities_Trait
		// =====================================================

		// =====================================================
		// TAXONOMIES METHODS - Moved to traits/trait-taxonomies.php
		// Use: Felan_Taxonomies_Trait
		// Includes: register_taxonomy()
		// =====================================================

		// =====================================================
		// TERM META METHODS - Moved to traits/trait-term-meta.php
		// Use: Felan_Term_Meta_Trait
		// Includes: register_term_meta()
		// =====================================================

		// =====================================================
		// METABOXES METHODS - Moved to metaboxes/trait-metaboxes.php
		// Use: Felan_Metaboxes_Trait
		// Includes: register_meta_boxes()
		// =====================================================

		/**
		 * Register options config
		 * @param $configs
		 * @return mixed
		 */
		public function register_options_config($configs)
		{
			if (function_exists('pll_the_languages')) {
				$enable_post_type_jobs = felan_get_option('enable_post_type_jobs', '1');
				$enable_post_type_service = felan_get_option('enable_post_type_service', '1');
				$enable_post_type_project = felan_get_option('enable_post_type_project', '1');
				$enable_wallet_mode = felan_get_option('enable_wallet_mode', '0');

				if (pll_current_language() !== null) {
					$option_name = pll_current_language() . '_felan-framework';
					if ($option_name == '_felan-framework') {
						$option_name = 'felan-framework';
					}
				} else {
					$option_name = 'felan-framework';
				}

				$configs['felan-framework'] = array(
					'layout' => 'horizontal',
					'page_title' => esc_html__('Theme Options', 'felan-framework'),
					'menu_title' => esc_html__('Theme Options', 'felan-framework'),
					'option_name' => $option_name,
					'permission' => 'edit_theme_options',
					'section' => array_merge(
						apply_filters('felan_register_options_config_top', array()),
						array_filter(array(
							$this->general_option(),
							($enable_post_type_jobs == '1') ? $this->jobs_option() : null,
							$this->company_option(),
							$this->freelancer_option(),
							($enable_post_type_service == '1') ? $this->service_option() : null,
							($enable_post_type_project == '1') ? $this->project_option() : null,
							($enable_wallet_mode == '1' && $enable_post_type_jobs !== '1') ? $this->wallet_option() : null,
							($enable_wallet_mode == '0') ? $this->payment_option() : null,
							$this->payout_option(),
							$this->social_network(),
							$this->login_option(),
							$this->locations_option(),
							$this->google_map_option(),
							$this->price_format_option(),
							$this->user_option(),
							$this->url_slugs_option(),
							$this->setup_page(),
							$this->ai_helper(),
							$this->email_management_option(),
							($enable_post_type_jobs == '1') ? $this->custom_field_jobs_option() : null,
							$this->custom_field_company_option(),
							$this->custom_field_freelancer_option(),
							($enable_post_type_project == '1') ? $this->custom_field_project_option() : null,
							($enable_post_type_service == '1') ? $this->custom_field_service_option() : null,
						)),
						apply_filters('felan_register_options_config_bottom', array())
					)
				);
			} else if (defined('ICL_SITEPRESS_VERSION')) {
				$current_language = apply_filters('wpml_current_language', NULL);

				if ($current_language) {
					$option_name = $current_language . '_felan-framework';
				} else {
					$option_name = 'felan-framework';
				}
				$enable_post_type_jobs = felan_get_option('enable_post_type_jobs', '1');
				$enable_post_type_service = felan_get_option('enable_post_type_service', '1');
				$enable_post_type_project = felan_get_option('enable_post_type_project', '1');
				$enable_wallet_mode = felan_get_option('enable_wallet_mode', '0');

				$configs['felan-framework'] = array(
					'layout' => 'horizontal',
					'page_title' => esc_html__('Theme Options', 'felan-framework'),
					'menu_title' => esc_html__('Theme Options', 'felan-framework'),
					'option_name' => $option_name,
					'permission' => 'edit_theme_options',
					'section' => array_merge(
						apply_filters('felan_register_options_config_top', array()),
						array_filter(array(
							$this->general_option(),
							($enable_post_type_jobs == '1') ? $this->jobs_option() : null,
							$this->company_option(),
							$this->freelancer_option(),
							($enable_post_type_service == '1') ? $this->service_option() : null,
							($enable_post_type_project == '1') ? $this->project_option() : null,
							($enable_wallet_mode == '1' && $enable_post_type_jobs !== '1') ? $this->wallet_option() : null,
							($enable_wallet_mode == '0') ? $this->payment_option() : null,
							$this->payout_option(),
							$this->social_network(),
							$this->login_option(),
							$this->locations_option(),
							$this->google_map_option(),
							$this->price_format_option(),
							$this->user_option(),
							$this->url_slugs_option(),
							$this->setup_page(),
							$this->ai_helper(),
							$this->email_management_option(),
							($enable_post_type_jobs == '1') ? $this->custom_field_jobs_option() : null,
							$this->custom_field_company_option(),
							$this->custom_field_freelancer_option(),
							($enable_post_type_project == '1') ? $this->custom_field_project_option() : null,
							($enable_post_type_service == '1') ? $this->custom_field_service_option() : null,
						)),
						apply_filters('felan_register_options_config_bottom', array())
					)
				);
			} else {
				$enable_post_type_jobs = felan_get_option('enable_post_type_jobs', '1');
				$enable_post_type_service = felan_get_option('enable_post_type_service', '1');
				$enable_post_type_project = felan_get_option('enable_post_type_project', '1');
				$enable_wallet_mode = felan_get_option('enable_wallet_mode', '0');

				$configs['felan-framework'] = array(
					'layout' => 'horizontal',
					'page_title' => esc_html__('Theme Options', 'felan-framework'),
					'menu_title' => esc_html__('Theme Options', 'felan-framework'),
					'option_name' => 'felan-framework',
					'permission' => 'edit_theme_options',
					'section' => array_merge(
						apply_filters('felan_register_options_config_top', array()),
						array_filter(array(
							$this->general_option(),
							($enable_post_type_jobs == '1') ? $this->jobs_option() : null,
							$this->company_option(),
							$this->freelancer_option(),
							($enable_post_type_service == '1') ? $this->service_option() : null,
							($enable_post_type_project == '1') ? $this->project_option() : null,
							($enable_wallet_mode == '1' && $enable_post_type_jobs !== '1') ? $this->wallet_option() : null,
							($enable_wallet_mode == '0') ? $this->payment_option() : null,
							$this->payout_option(),
							$this->social_network(),
							$this->login_option(),
							$this->locations_option(),
							$this->google_map_option(),
							$this->price_format_option(),
							$this->user_option(),
							$this->url_slugs_option(),
							$this->setup_page(),
							$this->ai_helper(),
							$this->email_management_option(),
							($enable_post_type_jobs == '1') ? $this->custom_field_jobs_option() : null,
							$this->custom_field_company_option(),
							$this->custom_field_freelancer_option(),
							($enable_post_type_project == '1') ? $this->custom_field_project_option() : null,
							($enable_post_type_service == '1') ? $this->custom_field_service_option() : null,
						)),
						apply_filters('felan_register_options_config_bottom', array())
					)
				);
			}
			return apply_filters('felan_register_options_config', $configs);
		}

		// =====================================================
		// OPTIONS METHODS - Moved to options/ folder
		// =====================================================
	}
}
