<?php
/**
 * Felan Framework Auto Update
 *
 * Automatically checks for and updates Felan Framework plugin
 *
 * @package Felan Framework
 * @since 1.1.6
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

if (!class_exists('Felan_Framework_Updater')) {

    class Felan_Framework_Updater {

        /**
         * API endpoint for checking updates (using theme API)
         */
        private $api_url;

        /**
         * Plugin slug
         */
        private $plugin_slug = 'felan-framework';

        /**
         * Plugin file path
         */
        private $plugin_file = 'felan-framework/felan-framework.php';

        /**
         * Constructor
         */
        public function __construct() {
            // Set API URL after WordPress is loaded
            add_action('init', array($this, 'set_api_url'), 1);

            // Hook into WordPress update system
            add_filter('pre_set_site_transient_update_plugins', array($this, 'check_for_updates'));
            add_filter('plugins_api', array($this, 'plugin_info'), 10, 3);
            add_filter('upgrader_package_options', array($this, 'maybe_adjust_source_dir'));

            // Disable default WordPress update notices for this plugin
            add_filter('in_plugin_update_message-' . $this->plugin_file, '__return_empty_string', 10, 2);
            add_action('after_plugin_row_' . $this->plugin_file, array($this, 'remove_default_update_notice'), 10, 3);

            // Admin notice for updates
            add_action('admin_notices', array($this, 'update_notice'));
            add_action('admin_init', array($this, 'dismiss_notice'));



            // Handle force refresh
            add_action('admin_init', array($this, 'handle_force_refresh'));

            // Add update link to plugin row actions
            add_filter('plugin_action_links_' . $this->plugin_file, array($this, 'add_plugin_update_link'), 10, 4);

            // Add CSS for update messages
            add_action('admin_enqueue_scripts', array($this, 'add_update_styles'));

            // Highlight plugin row when update is available
            add_filter('plugin_row_meta', array($this, 'highlight_plugin_row'), 10, 4);
        }

        /**
         * Set API URL after WordPress init - use theme changelogs API
         */
        public function set_api_url() {
            // Use theme's changelogs API instead of separate file
            if (class_exists('Felan_Updater')) {
                $theme_info = Felan_Updater::get_info();
                $this->api_url = $theme_info['api'] . '/changelogs.json';
            }
        }

        /**
         * Get download URL for the plugin
         */
        private function get_download_url() {
            // Use the correct download URL provided by user
            return 'https://felan.ricetheme.com/wp-content/plugins/felan-framework.zip';
        }

        /**
         * Add update link to plugin row actions (only if purchase code is valid)
         */
        public function add_plugin_update_link($actions, $plugin_file, $plugin_data, $context) {
            // Check if update is available
            $remote_info = $this->get_remote_info();
            if (!$remote_info || !version_compare(FELAN_PLUGIN_VER, $remote_info['version'], '<')) {
                return $actions;
            }

            // Check if purchase code is valid (similar to theme updater)
            if (!$this->is_purchase_code_valid()) {
                // Add notice about purchase code requirement
                $actions['update_required'] = sprintf(
                    '<span class="update-message">%s <a href="%s">%s</a></span>',
                    __('Purchase code required for updates.', 'felan-framework'),
                    admin_url('admin.php?page=felan_welcome'),
                    __('Enter code', 'felan-framework')
                );
                return $actions;
            }

            // Add update link
            $update_url = wp_nonce_url(
                admin_url('update.php?action=upgrade-plugin&plugin=' . urlencode($this->plugin_file)),
                'upgrade-plugin_' . $this->plugin_file
            );

            $actions['update'] = sprintf(
                '<a href="%s" class="update-link" aria-label="%s">%s</a>',
                esc_url($update_url),
                esc_attr(sprintf(__('Update %s now', 'felan-framework'), $plugin_data['Name'])),
                __('Update now', 'felan-framework')
            );

            return $actions;
        }

        /**
         * Check if purchase code is valid (similar to theme updater)
         * License check disabled – always treated as valid.
         */
        private function is_purchase_code_valid() {
            return true;

            // Use same logic as theme updater
            $purchase_code = get_option('ricetheme_purchase_code');
            if (empty($purchase_code)) {
                return false;
            }

            // Check purchase code validity using theme updater method
            if (class_exists('Felan_Updater')) {
                return Felan_Updater::check_valid_update();
            }

            return false;
        }

        /**
         * Remove default WordPress update notice for this plugin
         */
        public function remove_default_update_notice($plugin_file, $plugin_data, $status) {
            if ($plugin_file !== $this->plugin_file) {
                return;
            }

            // Remove the default update notice that WordPress adds
            remove_action('after_plugin_row_' . $plugin_file, 'wp_plugin_update_row', 10, 3);
        }

        /**
         * Add CSS styles for update messages
         */
        public function add_update_styles($hook) {
            if ('plugins.php' !== $hook) {
                return;
            }

            ?>
            <style type="text/css">
                .update-message {
                    color: #856404;
                    font-style: italic;
                }
                .update-message a {
                    color: #856404;
                    text-decoration: underline;
                }
                .update-message a:hover {
                    color: #533f00;
                }
                .plugin-action-links .update-link {
                    color: #2271b1;
                    font-weight: 600;
                }
                .plugin-action-links .update-link:hover {
                    color: #135e96;
                }
            </style>
            <?php
        }

        /**
         * Highlight plugin row when update is available
         */
        public function highlight_plugin_row($plugin_meta, $plugin_file, $plugin_data, $status) {
            if ($plugin_file !== $this->plugin_file) {
                return $plugin_meta;
            }

            // Check if update is available
            $remote_info = $this->get_remote_info();
            if ($remote_info && version_compare(FELAN_PLUGIN_VER, $remote_info['version'], '<')) {
                // Add update available notice to plugin description
                $update_info = sprintf(
                    '<strong style="color: #d63638;">%s</strong> %s',
                    __('Update Available:', 'felan-framework'),
                    sprintf(__('Version %s is available.', 'felan-framework'), $remote_info['version'])
                );

                // Add to plugin meta (this will appear below description)
                $plugin_meta[] = $update_info;
            }

            return $plugin_meta;
        }

        /**
         * Check for plugin updates
         */
        public function check_for_updates($transient) {
            if (empty($transient->checked)) {
                return $transient;
            }

            // Always check purchase code first before allowing any update data
            if (!$this->is_purchase_code_valid()) {
                return $transient;
            }

            $remote_info = $this->get_remote_info();

            if ($remote_info && version_compare(FELAN_PLUGIN_VER, $remote_info['version'], '<')) {
                $plugin_data = array(
                    'slug'        => $this->plugin_slug,
                    'plugin'      => $this->plugin_file,
                    'new_version' => $remote_info['version'],
                    'url'         => $remote_info['homepage'] ?? 'https://ricetheme.com/felan-framework',
                    'package'     => $this->get_download_url(),
                    'tested'      => $remote_info['tested'] ?? '',
                    'requires'    => $remote_info['requires'] ?? '',
                    'requires_php' => $remote_info['requires_php'] ?? '',
                );

                $transient->response[$this->plugin_file] = (object) $plugin_data;
            }

            return $transient;
        }

        /**
         * Get plugin information from remote API
         */
        public function plugin_info($false, $action, $args) {
            if ($action !== 'plugin_information' || $args->slug !== $this->plugin_slug) {
                return $false;
            }

            // Check purchase code before providing plugin info
            if (!$this->is_purchase_code_valid()) {
                return $false;
            }

            $remote_info = $this->get_remote_info();

            if (!$remote_info) {
                return $false;
            }

            return (object) array(
                'name'          => $remote_info['name'] ?? 'Felan Framework',
                'slug'          => $this->plugin_slug,
                'version'       => $remote_info['version'],
                'author'        => $remote_info['author'] ?? 'RiceTheme',
                'author_profile' => $remote_info['author_profile'] ?? 'https://ricetheme.com',
                'contributors'  => $remote_info['contributors'] ?? array(),
                'requires'      => $remote_info['requires'] ?? '',
                'tested'        => $remote_info['tested'] ?? '',
                'requires_php'  => $remote_info['requires_php'] ?? '',
                'rating'        => $remote_info['rating'] ?? 100,
                'ratings'       => $remote_info['ratings'] ?? array(),
                'num_ratings'   => $remote_info['num_ratings'] ?? 0,
                'downloaded'    => $remote_info['downloaded'] ?? 0,
                'last_updated'  => $remote_info['last_updated'] ?? '',
                'added'         => $remote_info['added'] ?? '',
                'homepage'      => $remote_info['homepage'] ?? 'https://ricetheme.com/felan-framework',
                'sections'      => $remote_info['sections'] ?? array(),
                'download_link' => $remote_info['download_url'],
                'screenshots'   => $remote_info['screenshots'] ?? array(),
                'tags'          => $remote_info['tags'] ?? array(),
                'donate_link'   => $remote_info['donate_link'] ?? '',
                'banners'       => $remote_info['banners'] ?? array(),
            );
        }

        /**
         * Get remote plugin information from changelogs
         */
        public function get_remote_info() {
            // Check cache first
            $cache_key = 'felan_framework_remote_info';
            $cached_info = get_transient($cache_key);

            if ($cached_info !== false) {
                return $cached_info;
            }

            // Get changelogs using theme updater method
            $changelogs = $this->get_framework_changelogs();

            if (!$changelogs) {
                return false;
            }

            // Find latest version from changelogs
            $latest_version = $this->parse_latest_version_from_changelogs($changelogs);

            if (!$latest_version) {
                return false;
            }

            // Build plugin info array
            $data = array(
                'name' => 'Felan Framework',
                'slug' => 'felan-framework',
                'version' => $latest_version['version'],
                'download_url' => $this->get_download_url(),
                'author' => 'RiceTheme',
                'author_profile' => 'https://ricetheme.com',
                'requires' => '5.0',
                'tested' => '6.4',
                'requires_php' => '7.4',
                'last_updated' => $latest_version['time'] ?? current_time('Y-m-d H:i:s'),
                'homepage' => 'https://ricetheme.com/felan-framework',
                'sections' => array(
                    'description' => 'Felan Framework is a powerful WordPress plugin that provides essential functionality for the Felan theme.',
                    'changelog' => $this->format_changelog_for_display($changelogs)
                )
            );

            // Cache for 6 hours
            set_transient($cache_key, $data, 6 * HOUR_IN_SECONDS);

            return $data;
        }

        /**
         * Get framework changelogs using theme API
         * Server returns only Framework changelogs, no filtering needed
         */
        public function get_framework_changelogs() {
            $response = wp_remote_get($this->api_url, array(
                'timeout' => 15,
                'headers' => array(
                    'Accept' => 'application/json',
                ),
            ));

            if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
                return false;
            }

            $body = wp_remote_retrieve_body($response);
            $changelogs = json_decode($body, true);

            return is_array($changelogs) ? $changelogs : false;
        }

        /**
         * Parse latest version from changelogs
         */
        private function parse_latest_version_from_changelogs($changelogs) {
            if (!is_array($changelogs) || empty($changelogs)) {
                return false;
            }

            // Find the highest version number
            $versions = array_keys($changelogs);
            usort($versions, 'version_compare');

            $latest_version = end($versions);
            $latest_data = $changelogs[$latest_version];

            return array(
                'version' => $latest_version,
                'time' => $latest_data['time'] ?? '',
                'desc' => $latest_data['desc'] ?? ''
            );
        }

        /**
         * Format changelog for display
         */
        private function format_changelog_for_display($changelogs) {
            $changelog_text = '';

            foreach ($changelogs as $version => $data) {
                $changelog_text .= "Version {$version}\n";
                if (is_array($data['desc'])) {
                    $changelog_text .= implode("\n", $data['desc']);
                } else {
                    $changelog_text .= $data['desc'];
                }
                $changelog_text .= "\n\n";
            }

            return trim($changelog_text);
        }

        /**
         * Adjust source directory for proper extraction
         */
        public function maybe_adjust_source_dir($options) {
            if (isset($options['hook_extra']['plugin']) && $options['hook_extra']['plugin'] === $this->plugin_file) {
                // Double-check purchase code before allowing extraction
                if (!$this->is_purchase_code_valid()) {
                    // Block the update process if purchase code is invalid
                    return new WP_Error('purchase_code_invalid', __('Purchase code validation failed. Update blocked.'));
                }
                $options['source_dir_name'] = $this->plugin_slug;
            }
            return $options;
        }

        /**
         * Show update notice
         */
        public function update_notice() {
            if (!current_user_can('update_plugins')) {
                return;
            }

            // Check if purchase code is valid (same as theme updater)
            if (!$this->is_purchase_code_valid()) {
                return;
            }

            $remote_info = $this->get_remote_info();
            if (!$remote_info || !version_compare(FELAN_PLUGIN_VER, $remote_info['version'], '<')) {
                return;
            }

            // Check if notice was dismissed (temporary - 1 hour)
            $dismiss_transient = 'felan_framework_update_dismissed_' . get_current_user_id();
            $dismissed_version = get_transient($dismiss_transient);
            if ($dismissed_version && version_compare($dismissed_version, $remote_info['version'], '>=')) {
                return;
            }

            $update_url = wp_nonce_url(
                admin_url('update.php?action=upgrade-plugin&plugin=' . urlencode($this->plugin_file)),
                'upgrade-plugin_' . $this->plugin_file
            );

            echo '<div class="notice notice-warning is-dismissible felan-framework-update-notice">
                <p><strong>' . esc_html__('Felan Framework Update Required!', 'felan-framework') . '</strong></p>
                <p>' . sprintf(
                    esc_html__('A new version of Felan Framework (%s) is available. You are currently using version %s.', 'felan-framework'),
                    '<strong>' . esc_html($remote_info['version']) . '</strong>',
                    '<strong>' . esc_html(FELAN_PLUGIN_VER) . '</strong>'
                ) . '</p>
                <p><strong>' . esc_html__('Important:', 'felan-framework') . '</strong> ' .
                    esc_html__('Felan Theme and Framework versions must be synchronized for optimal performance.', 'felan-framework') . '</p>
                <p>
                    <a href="' . esc_url($update_url) . '" class="button button-primary">' .
                        esc_html__('Update Framework', 'felan-framework') . '</a>
                    <a href="' . esc_url(wp_nonce_url(
                        add_query_arg('felan_dismiss_framework_update', '1'),
                        'felan_dismiss_notice'
                    )) . '" class="button">' . esc_html__('Remind Me Later', 'felan-framework') . '</a>
                </p>
            </div>';

            // Add JavaScript for dismiss functionality
            ?>
            <script type="text/javascript">
            jQuery(document).ready(function($) {
                $('.felan-framework-update-notice .notice-dismiss').on('click', function() {
                    $.post(ajaxurl, {
                        action: 'felan_dismiss_framework_update',
                        _wpnonce: '<?php echo wp_create_nonce('felan_dismiss_notice'); ?>'
                    });
                });
            });
            </script>
            <?php
        }

        /**
         * Handle force refresh
         */
        public function handle_force_refresh() {
            if (isset($_GET['force_framework_update_check']) && current_user_can('update_plugins')) {
                // Clear caches
                delete_transient('felan_framework_remote_info');
                delete_site_transient('update_plugins');

                // Clear all dismiss transients for current user
                global $wpdb;
                $user_id = get_current_user_id();
                $transient_pattern = '_transient_felan_framework_update_dismissed_' . $user_id;
                $wpdb->query($wpdb->prepare(
                    "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
                    $transient_pattern . '%'
                ));

                // Force check updates
                wp_update_plugins();

                // Add success message
                add_action('admin_notices', function() {
                    echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Framework update check completed. Cache cleared.', 'felan-framework') . '</p></div>';
                });

                // Redirect back
                wp_redirect(remove_query_arg('force_framework_update_check'));
                exit;
            }
        }

        /**
         * Handle notice dismissal
         */
        public function dismiss_notice() {
            if (isset($_GET['felan_dismiss_framework_update']) && wp_verify_nonce($_GET['_wpnonce'], 'felan_dismiss_notice')) {
                $remote_info = $this->get_remote_info();
                if ($remote_info) {
                    // Dismiss for 1 hour only (temporary)
                    $dismiss_transient = 'felan_framework_update_dismissed_' . get_current_user_id();
                    set_transient($dismiss_transient, $remote_info['version'], HOUR_IN_SECONDS);
                }
                wp_redirect(remove_query_arg(array('felan_dismiss_framework_update', '_wpnonce')));
                exit;
            }
        }


    }

    // Initialize the updater
    new Felan_Framework_Updater();

}

// AJAX handler for dismissing update notice
add_action('wp_ajax_felan_dismiss_framework_update', 'felan_dismiss_framework_update_ajax');
function felan_dismiss_framework_update_ajax() {
    check_ajax_referer('felan_dismiss_notice', '_wpnonce');

    $updater = new Felan_Framework_Updater();
    $remote_info = $updater->get_remote_info();

    if ($remote_info) {
        // Dismiss for 1 hour only (temporary)
        $dismiss_transient = 'felan_framework_update_dismissed_' . get_current_user_id();
        set_transient($dismiss_transient, $remote_info['version'], HOUR_IN_SECONDS);
    }

    wp_die();
}
