<?php
if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('Felan_Project_Alert_Cron')) {
    /**
     * Class Felan_Project_Alert_Cron
     * Handles cron jobs for sending email alerts
     */
    class Felan_Project_Alert_Cron
    {
        /**
         * Meta keys
         */
        const META_SETTINGS = 'project_alert_settings';
        const META_QUEUED = 'queued_project_alerts';
        const META_LAST_SENT = 'last_alert_sent_date';

        /**
         * Freelancer role
         */
        const FREELANCER_ROLE = 'felan_user_freelancer';

        /**
         * Constructor
         */
        public function __construct()
        {
            // Register custom cron schedules
            add_filter('cron_schedules', array($this, 'add_custom_cron_schedules'));

            // Schedule cron events
            add_action('init', array($this, 'schedule_events'));

            // Hook cron actions - Only use hourly job (handles all frequencies)
            add_action('felan_send_hourly_project_alerts', array($this, 'send_hourly_alerts'));

            // Legacy hooks for backward compatibility (deprecated - will be removed)
            add_action('felan_send_daily_project_alerts', array($this, 'send_daily_alerts'));
            add_action('felan_send_weekly_project_alerts', array($this, 'send_weekly_alerts'));
            add_action('felan_send_monthly_project_alerts', array($this, 'send_monthly_alerts'));

            // Hook to trigger alerts when new project is published
            add_action('transition_post_status', array($this, 'on_project_published'), 10, 3);

            // Cleanup invalid scheduled events on admin init
            add_action('admin_init', array($this, 'cleanup_invalid_events'));
        }

        /**
         * Add custom cron schedules
         * @param array $schedules
         * @return array
         */
        public function add_custom_cron_schedules($schedules)
        {
            if (!isset($schedules['weekly'])) {
                $schedules['weekly'] = array(
                    'interval' => WEEK_IN_SECONDS,
                    'display' => __('Once Weekly', 'felan-framework')
                );
            }

            if (!isset($schedules['monthly'])) {
                $schedules['monthly'] = array(
                    'interval' => MONTH_IN_SECONDS,
                    'display' => __('Once Monthly', 'felan-framework')
                );
            }

            return $schedules;
        }

        /**
         * Schedule cron events
         * OPTIMIZED: Only schedule hourly job (handles all frequencies)
         */
        public function schedule_events()
        {
            // Only schedule hourly job - it handles all frequencies based on user preferences
            if (!wp_next_scheduled('felan_send_hourly_project_alerts')) {
                wp_schedule_event(time(), 'hourly', 'felan_send_hourly_project_alerts');
            }

            // Cleanup legacy jobs (deprecated - kept for backward compatibility during migration)
            $this->cleanup_legacy_events();
        }

        /**
         * Cleanup legacy scheduled events (deprecated)
         * These are no longer needed as hourly job handles everything
         */
        private function cleanup_legacy_events()
        {
            $legacy_hooks = array(
                'felan_send_daily_project_alerts',
                'felan_send_weekly_project_alerts',
                'felan_send_monthly_project_alerts',
            );

            foreach ($legacy_hooks as $hook) {
                $timestamp = wp_next_scheduled($hook);
                if ($timestamp) {
                    wp_unschedule_event($timestamp, $hook);
                }
            }
        }

        /**
         * Cleanup invalid scheduled events
         * Removes events that reference non-existent users or invalid data
         */
        public function cleanup_invalid_events()
        {
            // Only run on admin pages to avoid performance impact
            if (!is_admin()) {
                return;
            }

            // Run cleanup once per day
            $last_cleanup = get_transient('felan_project_alert_cron_cleanup');
            if ($last_cleanup) {
                return;
            }

            set_transient('felan_project_alert_cron_cleanup', time(), DAY_IN_SECONDS);

            global $wpdb;
            $queued_key = FELAN_METABOX_PREFIX . self::META_QUEUED;

            // Find users with queued projects but invalid user IDs
            $invalid_users = $wpdb->get_col($wpdb->prepare("
                SELECT DISTINCT um.user_id
                FROM {$wpdb->usermeta} um
                LEFT JOIN {$wpdb->users} u ON um.user_id = u.ID
                WHERE um.meta_key = %s
                AND u.ID IS NULL
            ", $queued_key));

            // Cleanup invalid user meta
            if (!empty($invalid_users)) {
                foreach ($invalid_users as $user_id) {
                    delete_user_meta($user_id, $queued_key);
                    delete_user_meta($user_id, FELAN_METABOX_PREFIX . self::META_SETTINGS);
                    delete_user_meta($user_id, FELAN_METABOX_PREFIX . self::META_LAST_SENT);
                }
            }

            // Cleanup queued projects for users without valid email
            $users_without_email = $wpdb->get_col("
                SELECT DISTINCT um.user_id
                FROM {$wpdb->usermeta} um
                INNER JOIN {$wpdb->users} u ON um.user_id = u.ID
                WHERE um.meta_key = " . $wpdb->prepare('%s', $queued_key) . "
                AND (u.user_email = '' OR u.user_email IS NULL)
            ");

            if (!empty($users_without_email)) {
                foreach ($users_without_email as $user_id) {
                    delete_user_meta($user_id, $queued_key);
                    $this->log_email_error($user_id, 'User has no email address - queue cleared');
                }
            }

            // Cleanup queued projects that reference non-existent posts
            $all_queued = $wpdb->get_results($wpdb->prepare("
                SELECT user_id, meta_value
                FROM {$wpdb->usermeta}
                WHERE meta_key = %s
                AND meta_value != ''
            ", $queued_key));

            foreach ($all_queued as $row) {
                $user_id = intval($row->user_id);
                $queued_projects = maybe_unserialize($row->meta_value);

                if (!is_array($queued_projects) || empty($queued_projects)) {
                    continue;
                }

                // Validate project IDs
                $valid_projects = array();
                foreach ($queued_projects as $project_id) {
                    $project_id = intval($project_id);
                    if ($project_id > 0 && get_post_type($project_id) === 'project' && get_post_status($project_id) === 'publish') {
                        $valid_projects[] = $project_id;
                    }
                }

                // Update or delete if empty
                if (empty($valid_projects)) {
                    delete_user_meta($user_id, $queued_key);
                } elseif (count($valid_projects) !== count($queued_projects)) {
                    update_user_meta($user_id, $queued_key, $valid_projects);
                }
            }
        }

        /**
         * Send daily alerts (DEPRECATED - use send_hourly_alerts instead)
         * @deprecated Kept for backward compatibility
         */
        public function send_daily_alerts()
        {
            // Redirect to hourly alerts with daily frequency filter
            $this->send_alerts_by_frequency('daily');
        }

        /**
         * Send weekly alerts (DEPRECATED - use send_hourly_alerts instead)
         * @deprecated Kept for backward compatibility
         */
        public function send_weekly_alerts()
        {
            // Redirect to hourly alerts with weekly frequency filter
            $this->send_alerts_by_frequency('weekly');
        }

        /**
         * Send monthly alerts (DEPRECATED - use send_hourly_alerts instead)
         * @deprecated Kept for backward compatibility
         */
        public function send_monthly_alerts()
        {
            // Redirect to hourly alerts with monthly frequency filter
            $this->send_alerts_by_frequency('monthly');
        }

        /**
         * Send hourly alerts based on user-specific time preferences
         * NEW: Checks current hour and sends to users who have that hour as preference
         *
         * @param int|null $test_timestamp Optional timestamp for testing. If provided, uses this time instead of current time.
         * @param bool $dry_run If true, only simulate sending without actually sending emails.
         * @param array $test_options Optional test options: ['test_user_id' => int, 'test_preferred_time' => string, 'test_frequency' => string, 'skip_frequency_check' => bool]
         * @return array Statistics: ['emails_count' => int, 'users_count' => int]
         */
        public function send_hourly_alerts($test_timestamp = null, $dry_run = false, $test_options = array())
        {
            global $wpdb;

            require_once FELAN_PLUGIN_DIR . 'includes/partials/project-alert/class-felan-project-alert.php';
            require_once FELAN_PLUGIN_DIR . 'includes/partials/project-alert/class-felan-project-alert-email.php';

            $project_alert = new Felan_Project_Alert();
            $email_sender = new Felan_Project_Alert_Email();

            // Get time in site timezone (use test timestamp if provided)
            if ($test_timestamp !== null) {
                // Convert timestamp to site timezone
                $timezone = get_option('timezone_string');
                if (empty($timezone)) {
                    $gmt_offset = get_option('gmt_offset');
                    $timezone = timezone_name_from_abbr('', $gmt_offset * 3600, false);
                }

                $date = new DateTime('@' . $test_timestamp);
                if ($timezone) {
                    $date->setTimezone(new DateTimeZone($timezone));
                }

                $current_time = $date->format('H:i');
                $current_hour = $date->format('H');
                $current_date = $date->format('Y-m-d');
            } else {
                $current_time = current_time('H:i');
                $current_hour = current_time('H');
                $current_date = current_time('Y-m-d');
            }

            // Extract test options
            $test_user_id = isset($test_options['test_user_id']) ? intval($test_options['test_user_id']) : null;
            $test_preferred_time = isset($test_options['test_preferred_time']) ? $test_options['test_preferred_time'] : '';
            $test_frequency = isset($test_options['test_frequency']) ? $test_options['test_frequency'] : '';
            $skip_frequency_check = isset($test_options['skip_frequency_check']) ? (bool) $test_options['skip_frequency_check'] : false;

            // Get all freelancer user IDs (or specific user if testing)
            if ($test_user_id !== null) {
                $user = get_user_by('ID', $test_user_id);
                if (!$user || !in_array('felan_user_freelancer', $user->roles)) {
                    return array('emails_count' => 0, 'users_count' => 0);
                }
                $user_ids = array($test_user_id);
            } else {
                $users = get_users(array(
                    'role' => 'felan_user_freelancer',
                    'fields' => 'ID'
                ));

                if (empty($users)) {
                    return array('emails_count' => 0, 'users_count' => 0);
                }

                $user_ids = wp_list_pluck($users, 'ID');
            }

            // Batch load user meta
            $placeholders = implode(',', array_fill(0, count($user_ids), '%d'));

            $settings_key = FELAN_METABOX_PREFIX . self::META_SETTINGS;
            $queued_key = FELAN_METABOX_PREFIX . self::META_QUEUED;
            $last_sent_key = FELAN_METABOX_PREFIX . self::META_LAST_SENT;

            // OPTIMIZATION: Only load users who have queued projects
            $users_with_queues = $wpdb->get_col($wpdb->prepare("
                SELECT DISTINCT user_id
                FROM {$wpdb->usermeta}
                WHERE meta_key = %s
                AND meta_value != ''
                AND meta_value != 'a:0:{}'
                AND user_id IN ($placeholders)
            ", array_merge(array($queued_key), $user_ids)));

            if (empty($users_with_queues)) {
                return array('emails_count' => 0, 'users_count' => 0);
            }

            // Only process users with queued projects
            $user_ids = array_map('intval', $users_with_queues);
            $placeholders = implode(',', array_fill(0, count($user_ids), '%d'));

            $meta_results = $wpdb->get_results($wpdb->prepare("
                SELECT user_id, meta_key, meta_value
                FROM {$wpdb->usermeta}
                WHERE user_id IN ($placeholders)
                AND meta_key IN (%s, %s, %s)
            ", array_merge($user_ids, array($settings_key, $queued_key, $last_sent_key))));

            // Group meta by user_id
            $user_data = array();
            foreach ($meta_results as $row) {
                if (!isset($user_data[$row->user_id])) {
                    $user_data[$row->user_id] = array();
                }
                $user_data[$row->user_id][$row->meta_key] = maybe_unserialize($row->meta_value);
            }

            // Statistics
            $emails_count = 0;
            $users_count = 0;

            // Process users
            foreach ($user_ids as $user_id) {
                $settings = isset($user_data[$user_id][$settings_key])
                    ? $user_data[$user_id][$settings_key]
                    : $project_alert->get_notification_settings($user_id);

                // Check if email is enabled
                if (!$settings['email_enabled']) {
                    continue;
                }

                // Get user's preferred email time (override with test option if provided)
                if (!empty($test_preferred_time)) {
                    $user_email_time = $test_preferred_time;
                } else {
                    $user_email_time = isset($settings['email_time']) ? $settings['email_time'] : '09:00';
                }
                $user_hour = substr($user_email_time, 0, 2);

                // Check if current hour matches user's preference (or test preferred time)
                if ($user_hour !== $current_hour) {
                    continue;
                }

                // Get frequency (override with test option if provided)
                if (!empty($test_frequency)) {
                    $frequency = $test_frequency;
                } else {
                    $frequency = isset($settings['frequency']) ? $settings['frequency'] : 'daily';
                }

                // Get last sent date
                $last_sent = isset($user_data[$user_id][$last_sent_key])
                    ? $user_data[$user_id][$last_sent_key]
                    : '';

                // Check if we should send based on frequency (skip if test option is set)
                if (!$skip_frequency_check && !$this->should_send_alert($frequency, $last_sent)) {
                    continue;
                }

                // Get queued projects for this user
                $queued_projects = isset($user_data[$user_id][$queued_key])
                    ? $user_data[$user_id][$queued_key]
                    : array();

                if (!is_array($queued_projects) || empty($queued_projects)) {
                    continue;
                }

                // Filter out old projects based on frequency (dynamic time range)
                $queued_projects = $this->filter_recent_projects($queued_projects, $frequency);

                if (empty($queued_projects)) {
                    // No recent projects, clear the queue and skip sending email
                    delete_user_meta($user_id, $queued_key);
                    continue; // Skip sending email when no projects found
                }

                // Validate user email before sending
                $user = get_user_by('ID', $user_id);
                if (!$user || empty($user->user_email) || !is_email($user->user_email)) {
                    // Invalid or missing email - clear queue and log error
                    delete_user_meta($user_id, $queued_key);
                    $this->log_email_error($user_id, 'Invalid or missing email address');
                    continue;
                }

                // Count this user
                $users_count++;

                if ($dry_run) {
                    // Dry run: just count, don't send
                    $emails_count++;
                } else {
                    // Send email
                    $sent = $email_sender->send_alert_email($user_id, $queued_projects);

                    if ($sent) {
                        $emails_count++;

                        // Clear queued projects
                        delete_user_meta($user_id, $queued_key);

                        // Update last sent date (use test date if provided)
                        $last_sent_date = isset($current_date) ? $current_date : current_time('Y-m-d');
                        update_user_meta($user_id, $last_sent_key, $last_sent_date);

                        // Log success
                        $this->log_email_sent($user_id, count($queued_projects), $frequency);
                    } else {
                        // Email sending failed - log error but keep queue for retry
                        $this->log_email_error($user_id, 'Failed to send email (wp_mail returned false)');
                    }
                }
            }

            return array(
                'emails_count' => $emails_count,
                'users_count' => $users_count
            );
        }

        /**
         * Check if alert should be sent based on frequency and last sent date
         * @param string $frequency daily, weekly, or monthly
         * @param string $last_sent Last sent date (Y-m-d format)
         * @return bool
         */
        private function should_send_alert($frequency, $last_sent)
        {
            if (empty($last_sent)) {
                return true; // Never sent before
            }

            $today = current_time('Y-m-d');

            // If already sent today, don't send again
            if ($last_sent === $today) {
                return false;
            }

            $last_sent_timestamp = strtotime($last_sent);
            $today_timestamp = strtotime($today);
            $days_since_last = ($today_timestamp - $last_sent_timestamp) / DAY_IN_SECONDS;

            switch ($frequency) {
                case 'daily':
                    return $days_since_last >= 1;

                case 'weekly':
                    return $days_since_last >= 7;

                case 'monthly':
                    // Check if we're in a new month
                    $last_sent_month = date('Y-m', $last_sent_timestamp);
                    $current_month = current_time('Y-m');
                    return $last_sent_month !== $current_month;

                default:
                    return false;
            }
        }

        /**
         * Send alerts by frequency
         * @param string $frequency
         */
        private function send_alerts_by_frequency($frequency)
        {
            global $wpdb;

            require_once FELAN_PLUGIN_DIR . 'includes/partials/project-alert/class-felan-project-alert.php';
            require_once FELAN_PLUGIN_DIR . 'includes/partials/project-alert/class-felan-project-alert-email.php';

            $project_alert = new Felan_Project_Alert();
            $email_sender = new Felan_Project_Alert_Email();

            // OPTIMIZATION: Only get users with queued projects
            $settings_key = FELAN_METABOX_PREFIX . self::META_SETTINGS;
            $queued_key = FELAN_METABOX_PREFIX . self::META_QUEUED;

            // First, get users with queued projects
            $users_with_queues = $wpdb->get_col($wpdb->prepare("
                SELECT DISTINCT um.user_id
                FROM {$wpdb->usermeta} um
                INNER JOIN {$wpdb->users} u ON um.user_id = u.ID
                WHERE um.meta_key = %s
                AND um.meta_value != ''
                AND um.meta_value != 'a:0:{}'
            ", $queued_key));

            if (empty($users_with_queues)) {
                return;
            }

            // Verify these users are freelancers (simpler approach)
            $user_ids = array();
            foreach ($users_with_queues as $user_id) {
                $user = get_user_by('ID', $user_id);
                if ($user && in_array(self::FREELANCER_ROLE, $user->roles)) {
                    $user_ids[] = $user_id;
                }
            }

            if (empty($user_ids)) {
                return;
            }

            // Batch load user meta (PERFORMANCE: 1 query instead of N queries)
            $placeholders = implode(',', array_fill(0, count($user_ids), '%d'));

            $meta_results = $wpdb->get_results($wpdb->prepare("
                SELECT user_id, meta_key, meta_value
                FROM {$wpdb->usermeta}
                WHERE user_id IN ($placeholders)
                AND meta_key IN (%s, %s)
            ", array_merge($user_ids, array($settings_key, $queued_key))));

            // Group meta by user_id
            $user_data = array();
            foreach ($meta_results as $row) {
                if (!isset($user_data[$row->user_id])) {
                    $user_data[$row->user_id] = array();
                }
                $user_data[$row->user_id][$row->meta_key] = maybe_unserialize($row->meta_value);
            }

            // Process users
            foreach ($user_ids as $user_id) {
                $settings = isset($user_data[$user_id][$settings_key])
                    ? $user_data[$user_id][$settings_key]
                    : $project_alert->get_notification_settings($user_id);

                // Check if email is enabled and frequency matches
                if (!$settings['email_enabled'] || $settings['frequency'] !== $frequency) {
                    continue;
                }

                // Validate user email before processing
                $user = get_user_by('ID', $user_id);
                if (!$user || empty($user->user_email) || !is_email($user->user_email)) {
                    // Invalid or missing email - clear queue and log error
                    delete_user_meta($user_id, $queued_key);
                    $this->log_email_error($user_id, 'Invalid or missing email address');
                    continue;
                }

                // Get queued projects for this user
                $queued_projects = isset($user_data[$user_id][$queued_key])
                    ? $user_data[$user_id][$queued_key]
                    : array();

                if (!is_array($queued_projects) || empty($queued_projects)) {
                    continue;
                }

                // Filter out old projects based on frequency (dynamic time range)
                $queued_projects = $this->filter_recent_projects($queued_projects, $frequency);

                if (empty($queued_projects)) {
                    // No recent projects, clear the queue and skip sending email
                    delete_user_meta($user_id, $queued_key);
                    continue; // Skip sending email when no projects found
                }

                // Send email
                $sent = $email_sender->send_alert_email($user_id, $queued_projects);

                if ($sent) {
                    // Clear queued projects
                    delete_user_meta($user_id, $queued_key);

                    // Log success
                    $this->log_email_sent($user_id, count($queued_projects), $frequency);
                } else {
                    // Email sending failed - log error but keep queue for retry
                    $this->log_email_error($user_id, 'Failed to send email (wp_mail returned false)');
                }
            }
        }

        /**
         * Trigger alert when project is published
         * @param string $new_status
         * @param string $old_status
         * @param WP_Post $post
         */
        public function on_project_published($new_status, $old_status, $post)
        {
            // Only for projects
            if ($post->post_type !== 'project') {
                return;
            }

            // Clear cache when project is updated (taxonomies might have changed)
            if ($old_status === 'publish' && $new_status === 'publish') {
                delete_transient('project_taxonomies_' . $post->ID);
                return; // Don't trigger alert again for updates
            }

            // Only when transitioning to publish
            if ($new_status !== 'publish' || $old_status === 'publish') {
                return;
            }

            require_once FELAN_PLUGIN_DIR . 'includes/partials/project-alert/class-felan-project-alert.php';
            $project_alert = new Felan_Project_Alert();

            // Trigger alert
            $project_alert->trigger_project_alert($post->ID);
        }

        /**
         * Get time range in days based on frequency
         * @param string $frequency daily, weekly, or monthly
         * @return int Number of days
         */
        private function get_time_range_by_frequency($frequency)
        {
            switch ($frequency) {
                case 'daily':
                    return 1; // Last 1 day
                case 'weekly':
                    return 7; // Last 7 days
                case 'monthly':
                    return 30; // Last 30 days
                default:
                    return 90; // Fallback to 90 days
            }
        }

        /**
         * Filter projects to only include recent ones based on frequency
         * @param array $project_ids
         * @param string $frequency daily, weekly, or monthly
         * @return array
         */
        private function filter_recent_projects($project_ids, $frequency = 'daily')
        {
            if (empty($project_ids)) {
                return array();
            }

            $days_limit = $this->get_time_range_by_frequency($frequency);
            $limit_timestamp = strtotime("-{$days_limit} days");
            $recent_projects = array();

            foreach ($project_ids as $project_id) {
                $project_date = get_post_field('post_date', $project_id);
                if ($project_date) {
                    $project_timestamp = strtotime($project_date);
                    if ($project_timestamp >= $limit_timestamp) {
                        $recent_projects[] = $project_id;
                    }
                }
            }

            return $recent_projects;
        }

        /**
         * Log email sent
         * @param int $user_id
         * @param int $project_count
         * @param string $frequency
         */
        private function log_email_sent($user_id, $project_count, $frequency)
        {
            $log_entry = array(
                'user_id' => $user_id,
                'project_count' => $project_count,
                'frequency' => $frequency,
                'sent_at' => current_time('mysql'),
                'status' => 'success'
            );

            // Get existing log
            $log = get_option('felan_project_alert_email_log', array());

            // Keep only last 100 entries
            if (count($log) > 100) {
                $log = array_slice($log, -100);
            }

            $log[] = $log_entry;
            update_option('felan_project_alert_email_log', $log);
        }

        /**
         * Log email error
         * @param int $user_id
         * @param string $error_message
         */
        private function log_email_error($user_id, $error_message)
        {
            $log_entry = array(
                'user_id' => $user_id,
                'error' => $error_message,
                'sent_at' => current_time('mysql'),
                'status' => 'error'
            );

            // Get existing log
            $log = get_option('felan_project_alert_email_log', array());

            // Keep only last 100 entries
            if (count($log) > 100) {
                $log = array_slice($log, -100);
            }

            $log[] = $log_entry;
            update_option('felan_project_alert_email_log', $log);
        }

        /**
         * Clear scheduled events (for deactivation)
         */
        public static function clear_scheduled_events()
        {
            // Clear main hourly job
            wp_clear_scheduled_hook('felan_send_hourly_project_alerts');

            // Clear legacy jobs (if any still exist)
            wp_clear_scheduled_hook('felan_send_daily_project_alerts');
            wp_clear_scheduled_hook('felan_send_weekly_project_alerts');
            wp_clear_scheduled_hook('felan_send_monthly_project_alerts');
        }
    }

    // Initialize cron
    new Felan_Project_Alert_Cron();
}
