<?php
/**
 * Class Felan_Project_Alert_Package
 * Handles package limits and validation for project alerts
 */

if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('Felan_Project_Alert_Package')) {
    class Felan_Project_Alert_Package
    {
        /**
         * Check if project alerts feature is enabled
         *
         * @return bool
         */
        private function is_feature_enabled()
        {
            $enable_post_type_project = felan_get_option('enable_post_type_project', '1');
            $enable_project_package_alerts = felan_get_option('enable_project_package_alerts', '1');

            return ($enable_post_type_project == '1' && $enable_project_package_alerts == '1');
        }

        /**
         * Get package limits for project alerts
         *
         * @param int $user_id User ID
         * @return array Package limits structure
         */
        public function get_package_limits($user_id)
        {
            if (!$this->is_feature_enabled()) {
                return array(
                    'alerts' => array(
                        'enabled' => false,
                        'limit' => 0,
                        'current' => 0,
                    ),
                    'taxonomies' => array(),
                );
            }

            $freelancer_package_id = get_user_meta($user_id, FELAN_METABOX_PREFIX . 'freelancer_package_id', true);

            $current_alerts_count = $this->get_current_alerts_count($user_id);

            $alerts_limit = $this->get_alerts_limit($user_id, $freelancer_package_id);
            $alerts_unlimited = $this->is_alerts_unlimited($user_id, $freelancer_package_id);

            $enable_package_specific = felan_get_option('enable_project_package_specific_settings', '1');

            if ($enable_package_specific === '1') {
                if (!$freelancer_package_id) {
                    $alerts_enabled = false;
                } else {
                    $package_limit = get_post_meta($freelancer_package_id, FELAN_METABOX_PREFIX . 'package_number_project_alerts', true);
                    if ($package_limit === '' || $package_limit === false || $package_limit === null || (int) $package_limit <= 0) {
                        $alerts_enabled = false;
                    } else {
                        $alerts_enabled = true;
                    }
                }
            } else {
                $alerts_enabled = true;
            }

            $allowed_taxonomies = $this->get_allowed_taxonomies($user_id, $freelancer_package_id);

            $taxonomies_limits = array();
            foreach ($allowed_taxonomies as $taxonomy_slug) {
                $max_terms = $this->get_max_terms_per_taxonomy($user_id, $taxonomy_slug, $freelancer_package_id);
                $taxonomies_limits[$taxonomy_slug] = array(
                    'enabled' => true,
                    'max_terms' => $max_terms,
                );
            }

            return array(
                'alerts' => array(
                    'enabled' => $alerts_enabled,
                    'limit' => $alerts_unlimited ? 'unlimited' : $alerts_limit,
                    'current' => $current_alerts_count,
                ),
                'taxonomies' => $taxonomies_limits,
            );
        }

        /**
         * Get current alerts count for user
         *
         * @param int $user_id User ID
         * @return int Current alerts count
         */
        public function get_current_alerts_count($user_id)
        {
            $args = array(
                'post_type' => 'project_alerts',
                'post_status' => 'publish',
                'posts_per_page' => -1,
                'author' => absint($user_id),
                'fields' => 'ids',
            );

            $alerts = get_posts($args);
            return count($alerts);
        }

        /**
         * Get alerts limit from package or theme options
         *
         * @param int $user_id User ID
         * @param int|false $package_id Package ID
         * @return int Alert limit
         */
        public function get_alerts_limit($user_id, $package_id = false)
        {
            $enable_package_specific = felan_get_option('enable_project_package_specific_settings', '1');

            if ($enable_package_specific !== '1') {
                $default_limit = felan_get_option('project_package_alerts_default_limit', '3');
                return $default_limit !== '' ? (int) $default_limit : 3;
            }

            if ($package_id) {
                $package_limit = get_post_meta($package_id, FELAN_METABOX_PREFIX . 'package_number_project_alerts', true);
                if ($package_limit !== '' && $package_limit !== false) {
                    return (int) $package_limit;
                }
            }

            $default_limit = felan_get_option('project_package_alerts_default_limit', '3');
            return $default_limit !== '' ? (int) $default_limit : 3;
        }

        /**
         * Check if alerts are unlimited
         *
         * @param int $user_id User ID
         * @param int|false $package_id Package ID
         * @return bool
         */
        public function is_alerts_unlimited($user_id, $package_id = false)
        {
            // Unlimited option has been removed to prevent spam
            return false;
        }

        /**
         * Check if user can create more alerts
         *
         * @param int $user_id User ID
         * @return bool|array Returns true if can create, or array with error message
         */
        public function can_create_alert($user_id)
        {
            // Check if feature is enabled
            if (!$this->is_feature_enabled()) {
                return array(
                    'success' => false,
                    'message' => esc_html__('Project alerts feature is not enabled.', 'felan-framework'),
                );
            }

            $limits = $this->get_package_limits($user_id);

            if (!$limits['alerts']['enabled']) {
                return array(
                    'success' => false,
                    'message' => esc_html__('Project alerts are not enabled for your package.', 'felan-framework'),
                );
            }

            // Unlimited option has been removed to prevent spam
            // Always check limit
            if ($limits['alerts']['current'] >= $limits['alerts']['limit']) {
                return array(
                    'success' => false,
                    'message' => sprintf(
                        esc_html__('You have reached the maximum number of alerts (%d). Please delete an existing alert or upgrade your package.', 'felan-framework'),
                        $limits['alerts']['limit']
                    ),
                );
            }

            return true;
        }

        /**
         * Get allowed taxonomies for package
         *
         * @param int $user_id User ID
         * @param int|false $package_id Package ID
         * @return array Array of allowed taxonomy slugs
         */
        public function get_allowed_taxonomies($user_id, $package_id = false)
        {
            // Check if package-specific settings are enabled
            $enable_package_specific = felan_get_option('enable_project_package_specific_settings', '1');

            if ($enable_package_specific === '1') {
                if ($package_id) {
                    $allowed = array();
                    $taxonomies = array('categories', 'skills', 'language', 'career', 'location', 'state');

                    foreach ($taxonomies as $taxonomy) {
                        $show_field_id = FELAN_METABOX_PREFIX . 'show_package_project_alerts_' . $taxonomy;
                        $show_value = get_post_meta($package_id, $show_field_id, true);
                        if ($show_value === '1' || $show_value === 1) {
                            $allowed[] = $taxonomy;
                        }
                    }

                    if (empty($allowed)) {
                        $allowed = array('categories', 'skills', 'language', 'career', 'location');
                        $enable_option_state = felan_get_option('enable_option_state', '0');
                        if ($enable_option_state === '1') {
                            $allowed[] = 'state';
                        }
                    }

                    return $allowed;
                }

                $allowed = array('categories', 'skills', 'language', 'career', 'location');
                $enable_option_state = felan_get_option('enable_option_state', '0');
                if ($enable_option_state === '1') {
                    $allowed[] = 'state';
                }
                return $allowed;
            } else {
                $global_allowed = felan_get_option('project_package_allowed_taxonomies', array('categories', 'skills', 'language', 'career', 'location'));
                if (!is_array($global_allowed)) {
                    $global_allowed = array('categories', 'skills', 'language', 'career', 'location');
                }

                $enable_option_state = felan_get_option('enable_option_state', '0');
                if ($enable_option_state === '1' && !in_array('state', $global_allowed)) {
                    $global_allowed[] = 'state';
                }

                return $global_allowed;
            }
        }

        /**
         * Get max terms per taxonomy
         *
         * @param int $user_id User ID
         * @param string $taxonomy_slug Taxonomy slug (categories, skills, etc.)
         * @param int|false $package_id Package ID
         * @return int|string Returns max count or 'unlimited'
         */
        public function get_max_terms_per_taxonomy($user_id, $taxonomy_slug, $package_id = false)
        {
            $allowed_taxonomies = $this->get_allowed_taxonomies($user_id, $package_id);

            if (!in_array($taxonomy_slug, $allowed_taxonomies)) {
                return 10; // Default max
            }

            $enable_package_specific = felan_get_option('enable_project_package_specific_settings', '1');

            if ($enable_package_specific !== '1') {
                $global_max = felan_get_option('project_package_max_terms_per_taxonomy', '10');
                $global_max = $global_max !== '' ? (int) $global_max : 10;
                return $global_max > 0 ? $global_max : 'unlimited';
            }

            $global_max = 10;

            if ($package_id) {
                $package_post = get_post($package_id);
                if (!$package_post || $package_post->post_type !== 'freelancer_package') {
                    return $global_max > 0 ? $global_max : 'unlimited';
                }

                $unlimited_key = 'enable_package_project_alerts_unlimited_' . $taxonomy_slug;
                $package_unlimited = get_post_meta($package_id, FELAN_METABOX_PREFIX . $unlimited_key, true);
                if ($package_unlimited === '1' || $package_unlimited === 1) {
                    return 'unlimited';
                }

                $max_key = 'package_project_alerts_max_' . $taxonomy_slug;
                $package_max = get_post_meta($package_id, FELAN_METABOX_PREFIX . $max_key, true);
                if ($package_max !== '' && $package_max !== false && $package_max !== null) {
                    $package_max = (int) $package_max;
                    // If 0, treat as unlimited
                    if ($package_max === 0) {
                        return 'unlimited';
                    }
                    return $package_max;
                }
            }

            return $global_max > 0 ? $global_max : 'unlimited';
        }

        /**
         * Validate alert creation against package
         *
         * @param int $user_id User ID
         * @param array $alert_data Alert data (categories, skills, language, career, location, state)
         * @param int $alert_id Optional. Alert ID if updating
         * @return bool|array Returns true if valid, or array with error message
         */
        public function validate_alert($user_id, $alert_data, $alert_id = 0)
        {
            if (!$this->is_feature_enabled()) {
                return array(
                    'success' => false,
                    'message' => esc_html__('Project alerts feature is not enabled.', 'felan-framework'),
                );
            }

            $limits = $this->get_package_limits($user_id);

            if (!$limits['alerts']['enabled']) {
                return array(
                    'success' => false,
                    'message' => esc_html__('Project alerts are not enabled for your package.', 'felan-framework'),
                );
            }

            if (!$alert_id) {
                $can_create = $this->can_create_alert($user_id);
                if ($can_create !== true) {
                    return $can_create;
                }
            }

            $freelancer_package_id = get_user_meta($user_id, FELAN_METABOX_PREFIX . 'freelancer_package_id', true);
            $allowed_taxonomies = $this->get_allowed_taxonomies($user_id, $freelancer_package_id);

            $taxonomy_map = array(
                'categories' => 'categories',
                'skills' => 'skills',
                'language' => 'language',
                'career' => 'career',
                'location' => 'location',
                'state' => 'state',
            );

            foreach ($taxonomy_map as $key => $taxonomy_slug) {
                if (!isset($alert_data[$key])) {
                    continue;
                }

                $selected_terms = is_array($alert_data[$key]) ? $alert_data[$key] : array();

                if ($taxonomy_slug === 'state') {
                    $enable_option_state = felan_get_option('enable_option_state', '0');
                    if ($enable_option_state !== '1') {
                        continue;
                    }
                }

                if (!empty($selected_terms) && !in_array($taxonomy_slug, $allowed_taxonomies)) {
                    $taxonomy_labels = array(
                        'categories' => esc_html__('Categories', 'felan-framework'),
                        'skills' => esc_html__('Skills', 'felan-framework'),
                        'language' => esc_html__('Languages', 'felan-framework'),
                        'career' => esc_html__('Careers', 'felan-framework'),
                        'location' => esc_html__('Location', 'felan-framework'),
                        'state' => esc_html__('State/Province', 'felan-framework'),
                    );
                    $taxonomy_label = isset($taxonomy_labels[$taxonomy_slug]) ? $taxonomy_labels[$taxonomy_slug] : $taxonomy_slug;

                    return array(
                        'success' => false,
                        'message' => sprintf(
                            esc_html__('The taxonomy "%s" is not allowed for your package.', 'felan-framework'),
                            $taxonomy_label
                        ),
                    );
                }

                if (!empty($selected_terms)) {
                    $max_terms = $this->get_max_terms_per_taxonomy($user_id, $taxonomy_slug, $freelancer_package_id);

                    if ($max_terms !== 'unlimited' && is_numeric($max_terms) && count($selected_terms) > (int) $max_terms) {
                        $taxonomy_labels = array(
                            'categories' => esc_html__('categories', 'felan-framework'),
                            'skills' => esc_html__('skills', 'felan-framework'),
                            'language' => esc_html__('languages', 'felan-framework'),
                            'career' => esc_html__('careers', 'felan-framework'),
                            'location' => esc_html__('locations', 'felan-framework'),
                            'state' => esc_html__('states', 'felan-framework'),
                        );
                        $taxonomy_label = isset($taxonomy_labels[$taxonomy_slug]) ? $taxonomy_labels[$taxonomy_slug] : $taxonomy_slug;

                        return array(
                            'success' => false,
                            'message' => sprintf(
                                esc_html__('You can only select up to %d %s. Please reduce your selection.', 'felan-framework'),
                                $max_terms,
                                $taxonomy_label
                            ),
                        );
                    }
                }
            }

            return true;
        }

        /**
         * Get package info for display
         *
         * @param int $user_id User ID
         * @return array Package info
         */
        public function get_package_info($user_id)
        {
            $limits = $this->get_package_limits($user_id);

            return array(
                'alerts' => array(
                    'current' => $limits['alerts']['current'],
                    'limit' => $limits['alerts']['limit'],
                    'remaining' => $limits['alerts']['limit'] === 'unlimited'
                        ? 'unlimited'
                        : max(0, $limits['alerts']['limit'] - $limits['alerts']['current']),
                ),
                'taxonomies' => $limits['taxonomies'],
            );
        }
    }
}

