<?php
if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('Felan_Project_Alert')) {
    /**
     * Class Felan_Project_Alert
     * Handles project alert functionality for freelancers
     */
    class Felan_Project_Alert
    {
        /**
         * Constructor
         */
        public function __construct()
        {
            add_action('init', array($this, 'create_database_table'));
        }

        /**
         * Create database table for project alerts
         */
        public function create_database_table()
        {
            global $wpdb;
            $table_name = $wpdb->prefix . 'felan_project_alerts';
            $charset_collate = $wpdb->get_charset_collate();

            $sql = "CREATE TABLE IF NOT EXISTS $table_name (
                id bigint(20) NOT NULL AUTO_INCREMENT,
                user_id bigint(20) NOT NULL,
                skill_id bigint(20) NOT NULL,
                status tinyint(1) DEFAULT 1,
                created_at datetime DEFAULT CURRENT_TIMESTAMP,
                updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY  (id),
                KEY user_id (user_id),
                KEY skill_id (skill_id),
                KEY status (status)
            ) $charset_collate;";

            require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
            dbDelta($sql);
        }

        /**
         * Get followed skills for a user
         * @param int $user_id
         * @return array
         */
        public function get_followed_skills($user_id)
        {
            global $wpdb;
            $table_name = $wpdb->prefix . 'felan_project_alerts';

            $results = $wpdb->get_results($wpdb->prepare(
                "SELECT skill_id FROM $table_name WHERE user_id = %d AND status = 1",
                $user_id
            ));

            return wp_list_pluck($results, 'skill_id');
        }

        /**
         * Add skill to follow list
         * @param int $user_id
         * @param int $skill_id
         * @return bool|int
         */
        public function follow_skill($user_id, $skill_id)
        {
            global $wpdb;
            $table_name = $wpdb->prefix . 'felan_project_alerts';

            // Check if already exists
            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT id FROM $table_name WHERE user_id = %d AND skill_id = %d",
                $user_id,
                $skill_id
            ));

            if ($exists) {
                // Update status to active
                return $wpdb->update(
                    $table_name,
                    array('status' => 1),
                    array('id' => $exists),
                    array('%d'),
                    array('%d')
                );
            } else {
                // Insert new record
                return $wpdb->insert(
                    $table_name,
                    array(
                        'user_id' => $user_id,
                        'skill_id' => $skill_id,
                        'status' => 1
                    ),
                    array('%d', '%d', '%d')
                );
            }
        }

        /**
         * Remove skill from follow list
         * @param int $user_id
         * @param int $skill_id
         * @return bool|int
         */
        public function unfollow_skill($user_id, $skill_id)
        {
            global $wpdb;
            $table_name = $wpdb->prefix . 'felan_project_alerts';

            return $wpdb->update(
                $table_name,
                array('status' => 0),
                array(
                    'user_id' => $user_id,
                    'skill_id' => $skill_id
                ),
                array('%d'),
                array('%d', '%d')
            );
        }

        /**
         * Check if user follows a skill
         * @param int $user_id
         * @param int $skill_id
         * @return bool
         */
        public function is_following_skill($user_id, $skill_id)
        {
            global $wpdb;
            $table_name = $wpdb->prefix . 'felan_project_alerts';

            $result = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table_name WHERE user_id = %d AND skill_id = %d AND status = 1",
                $user_id,
                $skill_id
            ));

            return (bool) $result;
        }

        /**
         * Get count of followed skills for a user
         * @param int $user_id
         * @return int
         */
        public function get_followed_skills_count($user_id)
        {
            global $wpdb;
            $table_name = $wpdb->prefix . 'felan_project_alerts';

            $count = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table_name WHERE user_id = %d AND status = 1",
                $user_id
            ));

            return (int) $count;
        }

        /**
         * Get skill follow limit based on package
         * @param int $user_id
         * @return int|string Returns number or 'unlimited'
         */
        public function get_skill_follow_limit($user_id)
        {
            $freelancer_package_id = get_user_meta($user_id, FELAN_METABOX_PREFIX . 'freelancer_package_id', true);

            if (!$freelancer_package_id) {
                return 0;
            }

            $unlimited = get_post_meta($freelancer_package_id, FELAN_METABOX_PREFIX . 'enable_package_skill_alert_unlimited', true);

            if ($unlimited == 1) {
                return 'unlimited';
            }

            $limit = get_post_meta($freelancer_package_id, FELAN_METABOX_PREFIX . 'freelancer_package_number_skill_alert', true);

            return $limit ? (int) $limit : 2; // Default to 2 if not set
        }

        /**
         * Check if user can follow more skills
         * @param int $user_id
         * @return bool
         */
        public function can_follow_more_skills($user_id)
        {
            $limit = $this->get_skill_follow_limit($user_id);

            if ($limit === 'unlimited') {
                return true;
            }

            $current_count = $this->get_followed_skills_count($user_id);

            return $current_count < $limit;
        }

        /**
         * Get notification settings for user
         * @param int $user_id
         * @return array
         */
        public function get_notification_settings($user_id)
        {
            // Get email_time from theme options (global setting)
            $email_time = felan_get_option('project_alerts_email_time', '09:00');

            // Validate email_time format (HH:MM)
            if (!preg_match('/^([01]?[0-9]|2[0-3]):[0-5][0-9]$/', $email_time)) {
                $email_time = '09:00';
            }

            $defaults = array(
                'email_enabled' => true,
                'frequency' => 'daily', // daily, weekly, monthly
                'email_time' => $email_time, // From theme options
                'internal_enabled' => true
            );

            $settings = get_user_meta($user_id, FELAN_METABOX_PREFIX . 'project_alert_settings', true);

            if (!is_array($settings)) {
                $settings = array();
            }

            // Merge with defaults, but always use theme option email_time (not user meta)
            $merged = wp_parse_args($settings, $defaults);
            $merged['email_time'] = $email_time; // Override with theme option

            return $merged;
        }

        /**
         * Update notification settings for user
         * @param int $user_id
         * @param array $settings
         * @return bool
         */
        public function update_notification_settings($user_id, $settings)
        {
            $current_settings = $this->get_notification_settings($user_id);
            $new_settings = wp_parse_args($settings, $current_settings);

            return update_user_meta($user_id, FELAN_METABOX_PREFIX . 'project_alert_settings', $new_settings);
        }

        /**
         * Get users who follow a specific skill
         * @param int $skill_id
         * @return array User IDs
         */
        public function get_users_following_skill($skill_id)
        {
            global $wpdb;
            $table_name = $wpdb->prefix . 'felan_project_alerts';

            $results = $wpdb->get_results($wpdb->prepare(
                "SELECT DISTINCT user_id FROM $table_name WHERE skill_id = %d AND status = 1",
                $skill_id
            ));

            return wp_list_pluck($results, 'user_id');
        }

        /**
         * Get all skills (followed and unfollowed)
         * @param int $user_id
         * @return array
         */
        public function get_all_skills_with_follow_status($user_id)
        {
            $followed_skills = $this->get_followed_skills($user_id);

            $all_skills = get_terms(array(
                'taxonomy' => 'freelancer_skills',
                'hide_empty' => false,
                'orderby' => 'name',
                'order' => 'ASC'
            ));

            $skills_data = array();

            foreach ($all_skills as $skill) {
                $skills_data[] = array(
                    'id' => $skill->term_id,
                    'name' => $skill->name,
                    'slug' => $skill->slug,
                    'followed' => in_array($skill->term_id, $followed_skills)
                );
            }

            return $skills_data;
        }

        /**
         * Send alert when new project is posted
         * @param int $project_id
         */
        public function trigger_project_alert($project_id)
        {
            // Get all project taxonomies
            $project_taxonomies = $this->get_project_taxonomies($project_id);

            // If project has no taxonomies, skip
            if (empty($project_taxonomies)) {
                return;
            }

            // Get all active project alerts (post type)
            $alerts = $this->get_all_active_alerts();

            if (empty($alerts)) {
                return;
            }

            // Batch load user settings for performance
            $user_ids = array_unique(wp_list_pluck($alerts, 'user_id'));
            $settings_key = FELAN_METABOX_PREFIX . 'project_alert_settings';
            $user_settings = $this->batch_load_user_meta($user_ids, $settings_key);

            // Process each alert
            foreach ($alerts as $alert) {
                // Check if alert matches project
                if ($this->alert_matches_project($alert, $project_taxonomies)) {
                    $user_id = $alert['user_id'];
                    $settings = isset($user_settings[$user_id])
                        ? $user_settings[$user_id]
                        : $this->get_notification_settings($user_id);

                    $this->create_notification($user_id, $project_id, $settings);
                }
            }
        }

        /**
         * Get all taxonomies for a project (with caching)
         * @param int $project_id
         * @return array
         */
        private function get_project_taxonomies($project_id)
        {
            // Cache key
            $cache_key = 'project_taxonomies_' . $project_id;

            // Try to get from cache first
            $taxonomies = get_transient($cache_key);
            if (false !== $taxonomies) {
                return $taxonomies;
            }

            $taxonomies = array();

            // Categories
            $categories = wp_get_post_terms($project_id, 'project-categories', array('fields' => 'ids'));
            if (!empty($categories) && !is_wp_error($categories)) {
                $taxonomies['categories'] = $categories;
            }

            // Skills
            $skills = wp_get_post_terms($project_id, 'project-skills', array('fields' => 'ids'));
            if (!empty($skills) && !is_wp_error($skills)) {
                $taxonomies['skills'] = $skills;
            }

            // Language
            $language = wp_get_post_terms($project_id, 'project-language', array('fields' => 'ids'));
            if (!empty($language) && !is_wp_error($language)) {
                $taxonomies['language'] = $language;
            }

            // Career
            $career = wp_get_post_terms($project_id, 'project-career', array('fields' => 'ids'));
            if (!empty($career) && !is_wp_error($career)) {
                $taxonomies['career'] = $career;
            }

            // Location
            $location = wp_get_post_terms($project_id, 'felan_location', array('fields' => 'ids'));
            if (!empty($location) && !is_wp_error($location)) {
                $taxonomies['location'] = $location;
            }

            // State (optional)
            $enable_state = felan_get_option('enable_option_state', '0');
            if ($enable_state === '1') {
                $state = wp_get_post_terms($project_id, 'felan_state', array('fields' => 'ids'));
                if (!empty($state) && !is_wp_error($state)) {
                    $taxonomies['state'] = $state;
                }
            }

            // Cache for 1 hour (taxonomies don't change frequently)
            set_transient($cache_key, $taxonomies, HOUR_IN_SECONDS);

            return $taxonomies;
        }

        /**
         * Get all active project alerts
         * @return array
         */
        private function get_all_active_alerts()
        {
            $args = array(
                'post_type' => 'project_alerts',
                'post_status' => 'publish',
                'posts_per_page' => -1,
                'fields' => 'ids',
            );

            $alert_ids = get_posts($args);

            if (empty($alert_ids)) {
                return array();
            }

            $alerts = array();
            foreach ($alert_ids as $alert_id) {
                $user_id = get_post_meta($alert_id, FELAN_METABOX_PREFIX . 'project_alerts_user_id', true);
                if (empty($user_id)) {
                    continue;
                }

                $alerts[] = array(
                    'id' => $alert_id,
                    'user_id' => (int) $user_id,
                    'categories' => get_post_meta($alert_id, FELAN_METABOX_PREFIX . 'project_alerts_categories', true) ?: array(),
                    'skills' => get_post_meta($alert_id, FELAN_METABOX_PREFIX . 'project_alerts_skills', true) ?: array(),
                    'language' => get_post_meta($alert_id, FELAN_METABOX_PREFIX . 'project_alerts_language', true) ?: array(),
                    'career' => get_post_meta($alert_id, FELAN_METABOX_PREFIX . 'project_alerts_career', true) ?: array(),
                    'location' => get_post_meta($alert_id, FELAN_METABOX_PREFIX . 'project_alerts_location', true) ?: array(),
                    'state' => get_post_meta($alert_id, FELAN_METABOX_PREFIX . 'project_alerts_state', true) ?: array(),
                    'match_type' => get_post_meta($alert_id, FELAN_METABOX_PREFIX . 'project_alerts_match_type', true) ?: 'OR',
                );
            }

            return $alerts;
        }

        /**
         * Check if alert matches project based on taxonomies
         * @param array $alert Alert data
         * @param array $project_taxonomies Project taxonomies
         * @return bool
         */
        private function alert_matches_project($alert, $project_taxonomies)
        {
            $match_type = isset($alert['match_type']) ? $alert['match_type'] : 'OR';
            $matches = array();

            // Check each taxonomy
            $taxonomy_map = array(
                'categories' => 'categories',
                'skills' => 'skills',
                'language' => 'language',
                'career' => 'career',
                'location' => 'location',
                'state' => 'state',
            );

            // Check if alert has any taxonomies selected (early check)
            $has_selected_taxonomies = false;
            foreach ($taxonomy_map as $taxonomy_key => $taxonomy_name) {
                $alert_terms = isset($alert[$taxonomy_key]) ? $alert[$taxonomy_key] : array();
                if (!empty($alert_terms)) {
                    $has_selected_taxonomies = true;
                    break;
                }
            }

            if (!$has_selected_taxonomies) {
                return false; // No taxonomies selected, don't match
            }

            foreach ($taxonomy_map as $taxonomy_key => $taxonomy_name) {
                $alert_terms = isset($alert[$taxonomy_key]) ? $alert[$taxonomy_key] : array();
                $project_terms = isset($project_taxonomies[$taxonomy_key]) ? $project_taxonomies[$taxonomy_key] : array();

                // Skip if alert doesn't have this taxonomy selected
                if (empty($alert_terms)) {
                    continue;
                }

                // Check if there's any intersection
                $intersection = array_intersect($alert_terms, $project_terms);
                if (!empty($intersection)) {
                    $matches[$taxonomy_key] = true;

                    // Early exit for OR mode: if we found a match, return true immediately
                    if ($match_type === 'OR') {
                        return true;
                    }
                }
            }

            // Apply match type logic
            if ($match_type === 'AND') {
                // AND: All selected taxonomies must match
                foreach ($taxonomy_map as $taxonomy_key => $taxonomy_name) {
                    $alert_terms = isset($alert[$taxonomy_key]) ? $alert[$taxonomy_key] : array();
                    if (!empty($alert_terms)) {
                        // This taxonomy is selected in alert, must have match
                        if (!isset($matches[$taxonomy_key])) {
                            return false;
                        }
                    }
                }
                return true;
            } else {
                // OR: At least one taxonomy must match
                return !empty($matches);
            }
        }

        /**
         * Create notification for user
         * @param int $user_id
         * @param int $project_id
         * @param array $settings
         */
        private function create_notification($user_id, $project_id, $settings = null)
        {
            if ($settings === null) {
                $settings = $this->get_notification_settings($user_id);
            }

            // Create internal notification
            if ($settings['internal_enabled']) {
                $this->create_internal_notification($user_id, $project_id);
            }

            // Queue email notification based on frequency
            if ($settings['email_enabled']) {
                $this->queue_email_notification($user_id, $project_id, $settings['frequency']);
            }
        }

        /**
         * Create internal notification
         * @param int $user_id
         * @param int $project_id
         */
        private function create_internal_notification($user_id, $project_id)
        {
            // Store pending notifications in user meta
            $pending_notifications = get_user_meta($user_id, FELAN_METABOX_PREFIX . 'pending_project_alerts', true);

            if (!is_array($pending_notifications)) {
                $pending_notifications = array();
            }

            $pending_notifications[] = array(
                'project_id' => $project_id,
                'created_at' => current_time('mysql')
            );

            update_user_meta($user_id, FELAN_METABOX_PREFIX . 'pending_project_alerts', $pending_notifications);
        }

        /**
         * Get time range in days based on frequency
         * @param string $frequency daily, weekly, or monthly
         * @return int Number of days
         */
        private function get_time_range_by_frequency($frequency)
        {
            switch ($frequency) {
                case 'daily':
                    return 1; // Last 1 day
                case 'weekly':
                    return 7; // Last 7 days
                case 'monthly':
                    return 30; // Last 30 days
                default:
                    return 90; // Fallback to 90 days
            }
        }

        /**
         * Queue email notification
         * @param int $user_id
         * @param int $project_id
         * @param string $frequency
         */
        private function queue_email_notification($user_id, $project_id, $frequency)
        {
            // Get time range based on frequency (dynamic)
            $days_limit = $this->get_time_range_by_frequency($frequency);

            // Only queue projects published within the time range based on frequency
            $project_date = get_post_field('post_date', $project_id);
            if ($project_date) {
                $project_timestamp = strtotime($project_date);
                $limit_timestamp = strtotime("-{$days_limit} days");

                if ($project_timestamp < $limit_timestamp) {
                    // Project is too old for this frequency, skip queueing
                    return;
                }
            }

            // Store projects to be sent in next email batch
            $queued_projects = get_user_meta($user_id, FELAN_METABOX_PREFIX . 'queued_project_alerts', true);

            if (!is_array($queued_projects)) {
                $queued_projects = array();
            }

            // Limit queue size to prevent memory issues (max 100 projects per user)
            if (count($queued_projects) >= 100) {
                // Sort by post_date and keep newest 100 projects
                $project_dates = array();
                foreach ($queued_projects as $pid) {
                    $project_date = get_post_field('post_date', $pid);
                    if ($project_date) {
                        $project_dates[$pid] = strtotime($project_date);
                    } else {
                        $project_dates[$pid] = 0; // Fallback for projects without date
                    }
                }
                // Sort by date descending (newest first)
                arsort($project_dates);
                // Keep only newest 100 projects
                $queued_projects = array_slice(array_keys($project_dates), 0, 100);
            }

            if (!in_array($project_id, $queued_projects)) {
                $queued_projects[] = $project_id;
                update_user_meta($user_id, FELAN_METABOX_PREFIX . 'queued_project_alerts', $queued_projects);
            }
        }

        /**
         * Batch load user meta for multiple users (PERFORMANCE OPTIMIZATION)
         * @param array $user_ids
         * @param string $meta_key
         * @return array Array of user_id => meta_value
         */
        private function batch_load_user_meta($user_ids, $meta_key)
        {
            global $wpdb;

            if (empty($user_ids)) {
                return array();
            }

            $placeholders = implode(',', array_fill(0, count($user_ids), '%d'));
            $query = $wpdb->prepare("
                SELECT user_id, meta_value
                FROM {$wpdb->usermeta}
                WHERE user_id IN ($placeholders) AND meta_key = %s
            ", array_merge($user_ids, array($meta_key)));

            $results = $wpdb->get_results($query);

            $data = array();
            foreach ($results as $row) {
                $data[$row->user_id] = maybe_unserialize($row->meta_value);
            }

            return $data;
        }
    }
}
