<?php
if (!defined('ABSPATH')) {
  exit;
}

if (!class_exists('Felan_Project_Invite_Email')) {
  /**
   * Class Felan_Project_Invite_Email
   * Handles email notifications for project invites
   */
  class Felan_Project_Invite_Email
  {
    /**
     * Get theme colors for email template
     * @return array
     */
    private function get_theme_colors()
    {
      // Get colors from theme customizer
      $text_color = get_theme_mod('text_color', '#444444');
      $accent_color = get_theme_mod('accent_color', '#1f72f2');
      $primary_color = get_theme_mod('primary_color', '#111111');
      $secondary_color = get_theme_mod('secondary_color', '#333333');
      $border_color = get_theme_mod('border_color', '#e5e7eb');
      $bg_color = get_theme_mod('body_background_color', '#ffffff');

      // Fallback to defaults if empty
      if (empty($text_color)) {
        $text_color = '#444444';
      }
      if (empty($accent_color)) {
        $accent_color = '#1f72f2';
      }
      if (empty($primary_color)) {
        $primary_color = '#111111';
      }
      if (empty($secondary_color)) {
        $secondary_color = '#333333';
      }
      if (empty($border_color)) {
        $border_color = '#e5e7eb';
      }
      if (empty($bg_color)) {
        $bg_color = '#ffffff';
      }

      // Create light background from border color (for body bg)
      $light_bg = $this->lighten_color($border_color, 0.95);

      // Create muted text colors (text with opacity effect)
      $muted_text = $this->darken_color($text_color, 0.7);
      $footer_text = $this->darken_color($text_color, 0.6);

      return array(
        'text' => $text_color,
        'accent' => $accent_color,
        'primary' => $primary_color,
        'secondary' => $secondary_color,
        'border' => $border_color,
        'bg' => $bg_color,
        'light_bg' => $light_bg,
        'muted_text' => $muted_text,
        'footer_text' => $footer_text,
      );
    }

    /**
     * Lighten a hex color
     * @param string $hex
     * @param float $percent
     * @return string
     */
    private function lighten_color($hex, $percent = 0.5)
    {
      $hex = str_replace('#', '', $hex);
      $rgb = array(
        hexdec(substr($hex, 0, 2)),
        hexdec(substr($hex, 2, 2)),
        hexdec(substr($hex, 4, 2))
      );

      for ($i = 0; $i < 3; $i++) {
        $rgb[$i] = round($rgb[$i] + (255 - $rgb[$i]) * $percent);
      }

      return sprintf('#%02x%02x%02x', $rgb[0], $rgb[1], $rgb[2]);
    }

    /**
     * Darken a hex color
     * @param string $hex
     * @param float $percent
     * @return string
     */
    private function darken_color($hex, $percent = 0.5)
    {
      $hex = str_replace('#', '', $hex);
      $rgb = array(
        hexdec(substr($hex, 0, 2)),
        hexdec(substr($hex, 2, 2)),
        hexdec(substr($hex, 4, 2))
      );

      for ($i = 0; $i < 3; $i++) {
        $rgb[$i] = round($rgb[$i] * $percent);
      }

      return sprintf('#%02x%02x%02x', $rgb[0], $rgb[1], $rgb[2]);
    }

    /**
     * Get SVG icon for email template
     * @param string $icon_type (location, clock, money, company, invite, tag, skill, language)
     * @param string $color Hex color code
     * @param int $size Icon size in pixels
     * @return string SVG HTML
     */
    private function get_svg_icon($icon_type, $color = '#111111', $size = 18)
    {
      $icons = array(
        'location' => '<svg width="' . $size . '" height="' . $size . '" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9 2.25C6.65279 2.25 4.75 4.15279 4.75 6.5C4.75 9.75 9 15.75 9 15.75C9 15.75 13.25 9.75 13.25 6.5C13.25 4.15279 11.3472 2.25 9 2.25ZM9 8.375C8.30964 8.375 7.75 7.81536 7.75 7.125C7.75 6.43464 8.30964 5.875 9 5.875C9.69036 5.875 10.25 6.43464 10.25 7.125C10.25 7.81536 9.69036 8.375 9 8.375Z" stroke="' . esc_attr($color) . '" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>',
        'clock' => '<svg width="' . $size . '" height="' . $size . '" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9 4.5V9H12.375" stroke="' . esc_attr($color) . '" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/><circle cx="9" cy="9" r="6.75" stroke="' . esc_attr($color) . '" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>',
        'money' => '<svg width="' . $size . '" height="' . $size . '" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9 2.25V15.75M12.75 4.5H7.5C6.67157 4.5 6 5.17157 6 6C6 6.82843 6.67157 7.5 7.5 7.5H10.5C11.3284 7.5 12 8.17157 12 9C12 9.82843 11.3284 10.5 10.5 10.5H6M12.75 13.5H7.5" stroke="' . esc_attr($color) . '" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>',
        'company' => '<svg width="' . $size . '" height="' . $size . '" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M3 15.75H15M3 15.75V6.75C3 6.33579 3.33579 6 3.75 6H14.25C14.6642 6 15 6.33579 15 6.75V15.75M3 15.75H1.5M15 15.75H16.5M6 6V3.75C6 3.33579 6.33579 3 6.75 3H11.25C11.6642 3 12 3.33579 12 3.75V6" stroke="' . esc_attr($color) . '" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/><path d="M6 10.5H12M6 12.75H9" stroke="' . esc_attr($color) . '" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>',
        'invite' => '<svg width="' . $size . '" height="' . $size . '" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9 12.75L12.75 9L9 5.25M5.25 9H12.75" stroke="' . esc_attr($color) . '" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/><circle cx="9" cy="9" r="6.75" stroke="' . esc_attr($color) . '" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>',
        'tag' => '<svg width="' . $size . '" height="' . $size . '" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M8.23529 15.1458L2.76141 9.67189C2.4841 9.39458 2.32612 9.01991 2.32116 8.62777L2.25006 3.01128C2.24471 2.58868 2.58867 2.24471 3.01128 2.25006L8.62777 2.32116C9.0199 2.32612 9.39458 2.4841 9.67189 2.76141L15.1458 8.23529C15.6506 8.74013 16.0394 9.70424 15.4632 10.2804L10.2804 15.4632C9.70424 16.0394 8.74012 15.6506 8.23529 15.1458Z" stroke="' . esc_attr($color) . '" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/><path d="M6.01422 5.66412L5.48389 5.13379" stroke="' . esc_attr($color) . '" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>',
        'skill' => '<svg width="' . $size . '" height="' . $size . '" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M9 1.5L11.3175 6.195L16.5 6.9525L12.75 10.605L13.635 15.765L9 13.3275L4.365 15.765L5.25 10.605L1.5 6.9525L6.6825 6.195L9 1.5Z" stroke="' . esc_attr($color) . '" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/></svg>',
        'language' => '<svg width="' . $size . '" height="' . $size . '" viewBox="0 0 18 18" fill="none" xmlns="http://www.w3.org/2000/svg"><circle cx="9" cy="9" r="6.75" stroke="' . esc_attr($color) . '" stroke-width="1.5"/><path d="M2.25 9H15.75M9 2.25C10.5 4.5 10.5 13.5 9 15.75M9 2.25C7.5 4.5 7.5 13.5 9 15.75" stroke="' . esc_attr($color) . '" stroke-width="1.5" stroke-linecap="round"/></svg>',
      );

      return isset($icons[$icon_type]) ? $icons[$icon_type] : '';
    }

    /**
     * Check if full custom template is enabled via toggle
     * @return bool
     */
    private function use_custom_template()
    {
        $use_custom = felan_get_option('use_custom_mail_project_invite', '0');
        $custom_content = felan_get_option('mail_project_invite', '');
        return ($use_custom == '1' && !empty($custom_content));
    }

    /**
     * Get custom email heading from theme options
     * @return string
     */
    private function get_custom_heading()
    {
        $custom_heading = felan_get_option('mail_project_invite_heading', '');
        if (!empty($custom_heading)) {
            return $custom_heading;
        }
        return __("You've Been Invited!", 'felan-framework');
    }

    /**
     * Get custom greeting from theme options
     * @param string $user_display_name
     * @param string $employer_name
     * @param array $colors
     * @return string
     */
    private function get_custom_greeting($user_display_name, $employer_name, $colors)
    {
        $custom_greeting = felan_get_option('mail_project_invite_greeting', '');
        if (!empty($custom_greeting)) {
            $greeting = str_replace('%employer_name', $employer_name, $custom_greeting);
            return '<p style="margin: 0; font-size: 15px; color: ' . esc_attr($colors['text']) . '; line-height: 1.6;">' . esc_html($greeting) . '</p>';
        }

        // Default greeting
        return '<p style="margin: 0 0 8px 0; font-size: 16px; color: ' . esc_attr($colors['primary']) . '; line-height: 1.6; font-weight: 500;">'
            . ($user_display_name ? sprintf(__('Hi %s,', 'felan-framework'), esc_html($user_display_name)) : __('Hi there,', 'felan-framework'))
            . '</p><p style="margin: 0; font-size: 15px; color: ' . esc_attr($colors['text']) . '; line-height: 1.6;">'
            . sprintf(__('%s has invited you to apply for the following project(s):', 'felan-framework'), '<strong style="color: ' . esc_attr($colors['accent']) . ';">' . esc_html($employer_name) . '</strong>')
            . '</p>';
    }

    /**
     * Get custom footer from theme options
     * @param string $site_name
     * @param string $site_url
     * @param array $colors
     * @return string
     */
    private function get_custom_footer($site_name, $site_url, $colors)
    {
        $custom_footer = felan_get_option('mail_project_invite_footer', '');
        if (!empty($custom_footer)) {
            $footer = str_replace(
                array('%website_name', '%website_url'),
                array($site_name, $site_url),
                $custom_footer
            );
            return '<p style="margin: 0; color: ' . esc_attr($colors['footer_text']) . '; font-size: 13px; line-height: 1.5;">' . esc_html($footer) . '</p>';
        }

        // Default footer
        return '<p style="margin: 0 0 8px 0; color: ' . esc_attr($colors['footer_text']) . '; font-size: 13px; line-height: 1.5;">'
            . sprintf(__('This email was sent by %s', 'felan-framework'), '<a href="' . esc_url($site_url) . '" style="color: ' . esc_attr($colors['accent']) . '; text-decoration: none; font-weight: 500;">' . esc_html($site_name) . '</a>')
            . '</p><p style="margin: 0; color: ' . esc_attr($colors['footer_text']) . '; font-size: 12px; line-height: 1.5;">'
            . __('You are receiving this email because you were invited to apply for a project.', 'felan-framework')
            . '</p>';
    }

    /**
     * Parse custom template and wrap in standard email layout
     * @param string $email
     * @param string $projects_html
     * @param string $employer_name
     * @return string
     */
    private function get_custom_email_template($email, $projects_html, $employer_name)
    {
        $custom_content = felan_get_option('mail_project_invite', '');
        if (empty($custom_content)) {
            return '';
        }

        $site_name = get_bloginfo('name');
        $site_url = home_url();

        // Get project listing URL
        $project_url_slug = felan_get_option('project_url_slug', 'projects');
        $project_listing_url = get_post_type_archive_link('project');
        if (!$project_listing_url) {
            $project_listing_url = home_url('/' . $project_url_slug . '/');
        }

        // Build simple project list for plain text template
        $project_list = strip_tags(str_replace('</tr>', "\n", $projects_html));
        $project_list = preg_replace('/\s+/', ' ', $project_list);

        // Replace placeholders
        $replacements = array(
            '%employer_name' => $employer_name,
            '%project_list' => $project_list,
            '%project_link' => $project_listing_url,
            '%website_name' => $site_name,
            '%website_url' => $site_url,
        );

        $content = str_replace(array_keys($replacements), array_values($replacements), $custom_content);

        // Wrap content in HTML email template
        $html = '<!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <meta name="viewport" content="width=device-width, initial-scale=1.0">
        </head>
        <body style="margin: 0; padding: 20px; background-color: #f5f5f5; font-family: Arial, sans-serif;">
            <div style="max-width: 600px; margin: 0 auto; background: #ffffff; padding: 30px; border-radius: 8px;">
                ' . nl2br(esc_html($content)) . '
            </div>
        </body>
        </html>';
        return $html;
    }

    /**
     * Send project invite email
     * @param string $email
     * @param array $project_ids
     * @param string $employer_name
     * @return bool
     */
    public function send_invite_email($email, $project_ids, $employer_name)
    {
      if (empty($project_ids) || empty($email) || !is_email($email)) {
        return false;
      }

      $projects_html = $this->build_projects_html($project_ids);
      if (empty($projects_html)) {
        return false;
      }

      $email_content = $this->get_email_template($email, $projects_html, $employer_name);
      $subject = $this->get_email_subject(count($project_ids));
      $headers = array('Content-Type: text/html; charset=UTF-8');

      return wp_mail($email, $subject, $email_content, $headers);
    }

    /**
     * Get email subject
     * @param int $count
     * @return string
     */
    private function get_email_subject($count)
    {
      $subject_template = felan_get_option('subject_mail_project_invite', '');
      if (!empty($subject_template)) {
        return $subject_template;
      }

      if ($count == 1) {
        return sprintf(__('You\'ve been invited to apply for a project', 'felan-framework'));
      }
      return sprintf(__('You\'ve been invited to apply for %d projects', 'felan-framework'), $count);
    }

    /**
     * Build HTML for projects
     * @param array $project_ids
     * @return string
     */
    private function build_projects_html($project_ids)
    {
      $html = '';
      $currency_sign = felan_get_option('currency_sign_default', '$');
      // Clean currency sign (remove any extra spaces or special characters)
      $currency_sign = trim($currency_sign);
      $currency_position = felan_get_option('currency_position', 'before');
      $colors = $this->get_theme_colors();

      foreach ($project_ids as $project_id) {
        $project = get_post($project_id);
        if (!$project) {
          continue;
        }

        $project_url = get_permalink($project_id);
        $project_title = get_the_title($project_id);

        // Get company info
        $company_id = get_post_meta($project_id, FELAN_METABOX_PREFIX . 'project_select_company', true);
        $company_name = '';
        $company_logo_url = '';
        if ($company_id) {
          $company_name = get_the_title($company_id);
          $company_logo = get_post_meta($company_id, FELAN_METABOX_PREFIX . 'company_logo', true);
          if (!empty($company_logo) && is_array($company_logo) && isset($company_logo['url'])) {
            $company_logo_url = $company_logo['url'];
          }
        }

        // Get budget type label
        $budget_show = get_post_meta($project_id, FELAN_METABOX_PREFIX . 'project_budget_show', true);
        $budget_type_label = __('Budget base', 'felan-framework');

        // Get budget/salary
        $budget_min = intval(get_post_meta($project_id, FELAN_METABOX_PREFIX . 'project_budget_minimum', true));
        $budget_max = intval(get_post_meta($project_id, FELAN_METABOX_PREFIX . 'project_budget_maximum', true));
        $budget_html = '';

        if ($budget_min || $budget_max) {
          $formatted_min = $budget_min ? number_format($budget_min, 0, '.', ',') : '';
          $formatted_max = $budget_max ? number_format($budget_max, 0, '.', ',') : '';

          if ($budget_show == 'fixed' && $budget_min) {
            $budget_html = ($currency_position === 'before') ? $currency_sign . $formatted_min : $formatted_min . $currency_sign;
          } elseif ($budget_min && $budget_max) {
            if ($currency_position === 'before') {
              $budget_html = $currency_sign . $formatted_min . ' - ' . $currency_sign . $formatted_max;
            } else {
              $budget_html = $formatted_min . $currency_sign . ' - ' . $formatted_max . $currency_sign;
            }
          } elseif ($budget_min) {
            $budget_html = ($currency_position === 'before') ? $currency_sign . $formatted_min : $formatted_min . $currency_sign;
          } elseif ($budget_max) {
            $budget_html = ($currency_position === 'before') ? $currency_sign . $formatted_max : $formatted_max . $currency_sign;
          }
        }

        // Get max time / duration
        $max_time = '';
        if (function_exists('felan_project_maximum_time')) {
          $max_time = trim(felan_project_maximum_time($project_id));
        }

        // Helper to format terms with limit (max 3, show "+X more" if exceeded)
        $max_terms = 3;

        // Get project categories (limit to 3)
        $project_categories = get_the_terms($project_id, 'project-categories');
        $category_html = '';
        if (!empty($project_categories) && !is_wp_error($project_categories)) {
          $category_names = wp_list_pluck($project_categories, 'name');
          $total_count = count($category_names);
          if ($total_count > $max_terms) {
            $category_html = implode(', ', array_slice($category_names, 0, $max_terms)) . ' +' . ($total_count - $max_terms);
          } else {
            $category_html = implode(', ', $category_names);
          }
        }

        // Get project skills (limit to 3)
        $project_skills = get_the_terms($project_id, 'project-skills');
        $skills_html = '';
        if (!empty($project_skills) && !is_wp_error($project_skills)) {
          $skill_names = wp_list_pluck($project_skills, 'name');
          $total_count = count($skill_names);
          if ($total_count > $max_terms) {
            $skills_html = implode(', ', array_slice($skill_names, 0, $max_terms)) . ' +' . ($total_count - $max_terms);
          } else {
            $skills_html = implode(', ', $skill_names);
          }
        }

        // Get project language (limit to 3)
        $project_language = get_the_terms($project_id, 'project-language');
        $language_html = '';
        if (!empty($project_language) && !is_wp_error($project_language)) {
          $language_names = wp_list_pluck($project_language, 'name');
          $total_count = count($language_names);
          if ($total_count > $max_terms) {
            $language_html = implode(', ', array_slice($language_names, 0, $max_terms)) . ' +' . ($total_count - $max_terms);
          } else {
            $language_html = implode(', ', $language_names);
          }
        }

        // Get location (limit to 2 for space)
        $locations = wp_get_post_terms($project_id, 'felan_location', array('fields' => 'names'));
        $location_html = '';
        if (!empty($locations) && !is_wp_error($locations)) {
          $total_count = count($locations);
          if ($total_count > 2) {
            $location_html = implode(', ', array_slice($locations, 0, 2)) . ' +' . ($total_count - 2);
          } else {
            $location_html = implode(', ', $locations);
          }
        }

        // Get project thumbnail with fallback
        $thumbnail_url = '';
        if (has_post_thumbnail($project_id)) {
          $thumbnail_url = get_the_post_thumbnail_url($project_id, 'medium');
          // Double check if URL is valid (sometimes get_the_post_thumbnail_url can return false)
          if (empty($thumbnail_url) || $thumbnail_url === false) {
            $thumbnail_url = '';
          }
        }
        // Use fallback image if thumbnail is empty or not available
        if (empty($thumbnail_url)) {
          $thumbnail_url = FELAN_PLUGIN_URL . 'assets/images/no-image.jpg';
          $thumbnail_url = apply_filters('felan_project_default_thumbnail_url', $thumbnail_url, $project_id);
        }

        // Get post date
        $post_date = get_post_field('post_date', $project_id);
        $posted_time = '';
        if ($post_date) {
          $posted_time = human_time_diff(strtotime($post_date), current_time('timestamp')) . ' ' . __('ago', 'felan-framework');
        }

        $html .= '
                <tr>
                    <td style="padding: 0;">
                        <a href="' . esc_url($project_url) . '" style="display: block; text-decoration: none; color: inherit;">
                            <table cellpadding="0" cellspacing="0" border="0" width="100%" style="background-color: ' . esc_attr($colors['bg']) . '; border: 1px solid ' . esc_attr($colors['border']) . '; border-radius: 12px; margin-bottom: 16px; overflow: hidden; transition: all 0.2s ease;">
                                <tr>
                                    <td style="width: 220px; min-width: 180px; padding: 0; background-image: url(\'' . esc_url($thumbnail_url) . '\'); background-size: cover; background-position: center; background-repeat: no-repeat; background-color: ' . esc_attr($colors['light_bg']) . '; border-radius: 12px 0 0 12px;">
                                        <!--[if gte mso 9]>
                                        <v:rect xmlns:v="urn:schemas-microsoft-com:vml" fill="true" stroke="false" style="width:180px;height:100%;">
                                        <v:fill type="frame" src="' . esc_url($thumbnail_url) . '" />
                                        <v:textbox inset="0,0,0,0">
                                        <![endif]-->
                                        <div style="width: 180px; min-height: 200px;"></div>
                                        <!--[if gte mso 9]>
                                        </v:textbox>
                                        </v:rect>
                                        <![endif]-->
                                    </td>
                                    <td style="padding: 20px; vertical-align: top;">
                                        ' . ($company_name ? '<table cellpadding="0" cellspacing="0" border="0" style="margin-bottom: 10px;">
                                            <tr>
                                                <td style="width: 32px; vertical-align: middle; padding-right: 8px;">
                                                    ' . ($company_logo_url ? '<img src="' . esc_url($company_logo_url) . '" alt="' . esc_attr($company_name) . '" style="width: 32px; height: 32px; border-radius: 50%; object-fit: cover; border: 1px solid ' . esc_attr($colors['border']) . '; display: block;" />' : '<table cellpadding="0" cellspacing="0" border="0" style="width: 32px; height: 32px; background: linear-gradient(135deg, ' . esc_attr($colors['light_bg']) . ' 0%, ' . esc_attr($this->lighten_color($colors['accent'], 0.9)) . ' 100%); border-radius: 50%; border: 1px solid ' . esc_attr($colors['border']) . ';"><tr><td style="text-align: center; vertical-align: middle;">' . $this->get_svg_icon('company', $colors['accent'], 18) . '</td></tr></table>') . '
                                                </td>
                                                <td style="vertical-align: middle;">
                                                    <span style="color: ' . esc_attr($colors['secondary']) . '; font-size: 14px; font-weight: 500; line-height: 1.5;">' . esc_html($company_name) . '</span>
                                                </td>
                                            </tr>
                                        </table>' : '') . '
                                        <h3 style="margin: 0 0 10px 0; font-size: 17px; font-weight: 600; line-height: 1.4; color: ' . esc_attr($colors['primary']) . ';">
                                            ' . esc_html($project_title) . '
                                        </h3>
                                        <!-- Project Meta: Category, Skills, Language, Posted Date -->
                                        <div style="margin-bottom: 12px;">
                                            ' . ($category_html ? '<span style="display: inline-flex; align-items: center; margin-right: 12px; margin-bottom: 6px; color: ' . esc_attr($colors['text']) . '; font-size: 13px; line-height: 1.5; gap: 5px;">' . $this->get_svg_icon('tag', $colors['text'], 14) . ' ' . esc_html($category_html) . '</span>' : '') . '
                                            ' . ($skills_html ? '<span style="display: inline-flex; align-items: center; margin-right: 12px; margin-bottom: 6px; color: ' . esc_attr($colors['text']) . '; font-size: 13px; line-height: 1.5; gap: 5px;">' . $this->get_svg_icon('skill', $colors['accent'], 14) . ' ' . esc_html($skills_html) . '</span>' : '') . '
                                            ' . ($language_html ? '<span style="display: inline-flex; align-items: center; margin-right: 12px; margin-bottom: 6px; color: ' . esc_attr($colors['text']) . '; font-size: 13px; line-height: 1.5; gap: 5px;">' . $this->get_svg_icon('language', $colors['text'], 14) . ' ' . esc_html($language_html) . '</span>' : '') . '
                                            ' . ($posted_time ? '<span style="display: inline-flex; align-items: center; margin-bottom: 6px; color: ' . esc_attr($colors['muted_text']) . '; font-size: 13px; line-height: 1.5; gap: 5px;">' . $this->get_svg_icon('clock', $colors['muted_text'], 14) . ' ' . esc_html($posted_time) . '</span>' : '') . '
                                        </div>
                                        <!-- Project Info: Budget, Duration, Location -->
                                        <div style="padding: 12px; background-color: ' . esc_attr($colors['light_bg']) . '; border-radius: 8px;">
                                            <table cellpadding="0" cellspacing="0" border="0" width="100%">
                                                <tr>
                                                    <td style="vertical-align: middle;">
                                                        <span style="display: block; color: ' . esc_attr($colors['muted_text']) . '; font-size: 12px; margin-bottom: 4px;">' . esc_html($budget_type_label) . '</span>
                                                        <span style="color: ' . esc_attr($colors['accent']) . '; font-size: 16px; font-weight: 600; line-height: 1.3;">' . esc_html($budget_html) . '</span>
                                                    </td>
                                                    ' . ($max_time ? '<td style="vertical-align: middle; text-align: center; border-left: 1px solid ' . esc_attr($colors['border']) . '; padding-left: 16px;">
                                                        <span style="display: block; color: ' . esc_attr($colors['muted_text']) . '; font-size: 12px; margin-bottom: 4px;">' . __('Duration', 'felan-framework') . '</span>
                                                        <span style="color: ' . esc_attr($colors['primary']) . '; font-size: 14px; font-weight: 500; line-height: 1.3;">' . esc_html($max_time) . '</span>
                                                    </td>' : '') . '
                                                    ' . ($location_html ? '<td style="vertical-align: middle; text-align: right; border-left: 1px solid ' . esc_attr($colors['border']) . '; padding-left: 16px;">
                                                        <span style="display: block; color: ' . esc_attr($colors['muted_text']) . '; font-size: 12px; margin-bottom: 4px;">' . __('Location', 'felan-framework') . '</span>
                                                        <span style="display: inline-flex; align-items: center; color: ' . esc_attr($colors['primary']) . '; font-size: 14px; font-weight: 500; line-height: 1.3; gap: 4px;">' . $this->get_svg_icon('location', $colors['primary'], 14) . ' ' . esc_html($location_html) . '</span>
                                                    </td>' : '') . '
                                                </tr>
                                            </table>
                                        </div>
                                    </td>
                                </tr>
                            </table>
                        </a>
                    </td>
                </tr>';
      }

      return $html;
    }

    /**
     * Get email template
     * @param string $email
     * @param string $projects_html
     * @param string $employer_name
     * @return string
     */
    private function get_email_template($email, $projects_html, $employer_name)
    {
      // Check if full custom template is enabled
      if ($this->use_custom_template()) {
          return $this->get_custom_email_template($email, $projects_html, $employer_name);
      }

      $site_name = get_bloginfo('name');
      $site_url = home_url();
      $colors = $this->get_theme_colors();

      // Get project listing URL
      $project_url_slug = felan_get_option('project_url_slug', 'projects');
      $project_listing_url = get_post_type_archive_link('project');
      if (!$project_listing_url) {
        $project_listing_url = home_url('/' . $project_url_slug . '/');
      }

      // Get site logo
      $site_logo_url = '';
      if (class_exists('Felan_Helper')) {
        $site_logo_url = Felan_Helper::get_setting('logo_light');
        if (empty($site_logo_url)) {
          $site_logo_url = Felan_Helper::get_setting('logo_dark');
        }
      }
      if (empty($site_logo_url)) {
        $custom_logo_id = get_theme_mod('custom_logo');
        if ($custom_logo_id) {
          $site_logo_url = wp_get_attachment_image_url($custom_logo_id, 'full');
        }
      }

      // Get user display name if available
      $user = get_user_by('email', $email);
      $user_display_name = $user ? $user->display_name : '';

      return '
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <title>' . __('Project Invitation', 'felan-framework') . '</title>
</head>
<body style="margin: 0; padding: 0; font-family: \'Inter\', -apple-system, BlinkMacSystemFont, \'Segoe UI\', Roboto, \'Helvetica Neue\', Arial, sans-serif; background-color: ' . esc_attr($colors['light_bg']) . ';">
    <table cellpadding="0" cellspacing="0" border="0" width="100%" style="background-color: ' . esc_attr($colors['light_bg']) . '; padding: 32px 16px;">
        <tr>
            <td align="center">
                <table cellpadding="0" cellspacing="0" border="0" width="100%" style="max-width: 900px; background-color: ' . esc_attr($colors['bg']) . '; border-radius: 16px; overflow: hidden; box-shadow: 0 4px 16px rgba(0,0,0,0.08);">
                    <!-- Header -->
                    <tr>
                        <td style="background-color: ' . esc_attr($colors['accent']) . '; padding: 40px 32px; text-align: center;">
                            ' . (!empty($site_logo_url) ? '<table cellpadding="0" cellspacing="0" border="0" width="100%" style="margin-bottom: 20px;">
                                <tr>
                                    <td align="center">
                                        <a href="' . esc_url($site_url) . '" style="display: inline-block; text-decoration: none;">
                                            <img src="' . esc_url($site_logo_url) . '" alt="' . esc_attr($site_name) . '" style="max-width: 180px; height: auto; display: block;" />
                                        </a>
                                    </td>
                                </tr>
                            </table>' : '') . '
                            <h1 style="margin: 0; color: ' . esc_attr($colors['bg']) . '; font-size: 26px; font-weight: 500; line-height: 1.3; letter-spacing: -0.5px;">
                                ' . esc_html($this->get_custom_heading()) . '
                            </h1>
                        </td>
                    </tr>

                    <!-- Greeting -->
                    <tr>
                        <td style="padding: 32px 32px 24px 32px;">
                            ' . $this->get_custom_greeting($user_display_name, $employer_name, $colors) . '
                        </td>
                    </tr>

                    <!-- Projects List -->
                    <tr>
                        <td style="padding: 0 32px 32px 32px;">
                            <table cellpadding="0" cellspacing="0" border="0" width="100%">
                                ' . $projects_html . '
                            </table>
                        </td>
                    </tr>

                    <!-- View Projects Button -->
                    <tr>
                        <td style="padding: 0 32px 32px 32px; text-align: center;">
                            <a href="' . esc_url($project_listing_url) . '" style="display: inline-block; padding: 14px 32px; background-color: ' . esc_attr($colors['accent']) . '; color: ' . esc_attr($colors['bg']) . '; text-decoration: none; border-radius: 10px; font-weight: 500; font-size: 15px; letter-spacing: 0.3px;">
                                ' . __('View Projects', 'felan-framework') . ' →
                            </a>
                        </td>
                    </tr>

                    <!-- Footer Info -->
                    <tr>
                        <td style="padding: 24px 32px; border-top: 1px solid ' . esc_attr($colors['border']) . '; text-align: center; background-color: ' . esc_attr($colors['light_bg']) . ';">
                            ' . $this->get_custom_footer($site_name, $site_url, $colors) . '
                        </td>
                    </tr>
                </table>
            </td>
        </tr>
    </table>
</body>
</html>';
    }
  }
}
