<?php
if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('Felan_Tools_Manager')) {
    /**
     * Class Felan_Tools_Manager
     * Manages registration and display of Felan tools
     */
    class Felan_Tools_Manager
    {
        /**
         * Instance
         * @var Felan_Tools_Manager
         */
        private static $instance = null;

        /**
         * Registered Tools
         * @var array
         */
        private $tools = array();

        /**
         * Get Instance
         */
        public static function get_instance()
        {
            if (self::$instance === null) {
                self::$instance = new self();
            }
            return self::$instance;
        }

        public function __construct()
        {
            add_action('admin_menu', array($this, 'register_tools_menu'), 50);
            add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
            $this->load_tools();
        }

        /**
         * Load all available tools
         */
        private function load_tools()
        {
            // Base path for tools
            $tools_paths = glob(FELAN_PLUGIN_DIR . 'includes/tools/*/*.php');

            foreach ($tools_paths as $path) {
                require_once $path;

                // Determine class name from file name
                $file_name = basename($path, '.php');
                // Remove 'class-' prefix and convert dashes to underscores
                $class_name = str_replace('-', '_', str_replace('class-', '', $file_name));
                // Capitalize each word
                $class_name = str_replace(' ', '_', ucwords(str_replace('_', ' ', $class_name)));

                if (class_exists($class_name) && is_subclass_of($class_name, 'Felan_Tool')) {
                    $tool = new $class_name();
                    $this->tools[$tool->get_id()] = $tool;
                }
            }
        }

        /**
         * Register Admin Menu
         */
        public function register_tools_menu()
        {
            add_submenu_page(
                'felan_welcome',
                esc_html__('Tools', 'felan-framework'),
                esc_html__('Tools', 'felan-framework'),
                'manage_options',
                'felan_tools',
                array($this, 'render_tools_page')
            );
        }

        /**
         * Enqueue Scripts
         */
        public function enqueue_scripts($hook)
        {
            if ($hook !== 'felan_page_felan_tools') {
                return;
            }
            // Load merged admin styles
             wp_enqueue_style('felan-admin-style', FELAN_PLUGIN_URL . 'assets/css/admin/admin.min.css', array(), FELAN_THEME_VERSION);
             wp_enqueue_script('felan-tools-script', FELAN_PLUGIN_URL . 'assets/js/felan-tools.js', array('jquery'), FELAN_THEME_VERSION, true);
        }

        /**
         * Render Tools Page content
         */
        public function render_tools_page()
        {
            $groups = array(
                'all' => esc_html__('All', 'felan-framework'),
                'maintenance' => esc_html__('Maintenance', 'felan-framework'),
                'importers' => esc_html__('Importers', 'felan-framework'),
                'debug' => esc_html__('Debug', 'felan-framework'),
            );

            // Handle Tool Action
            if (isset($_GET['tool']) && isset($_GET['action']) && $_GET['action'] == 'run') {
                $tool_id = sanitize_text_field($_GET['tool']);
                if (isset($this->tools[$tool_id])) {
                    $this->tools[$tool_id]->render();
                    return; // Stop rendering main grid if specific tool is running/rendering
                }
            }

            ?>
            <div class="wrap felan-wrap about-wrap felan-tools-container">
                <h1><?php esc_html_e('Felan Tools', 'felan-framework'); ?></h1>

                <p><?php esc_html_e('Handy tools to help you manage your Felan site.', 'felan-framework'); ?></p>

                <div class="felan-tools-grid">
                    <?php
                    $has_tools = false;
                    foreach ($this->tools as $tool) :
                        $has_tools = true;
                    ?>
                        <div class="felan-tool-card">
                            <div class="felan-tool-body">
                                <div class="felan-tool-header">
                                    <div class="felan-tool-icon">
                                        <?php
                                        $icon = $tool->get_icon();
                                        if (strpos($icon, '<svg') !== false) {
                                            echo $icon; // Raw SVG
                                        } else {
                                            echo '<span class="' . esc_attr($icon) . '"></span>';
                                        }
                                        ?>
                                    </div>
                                    <h3 class="felan-tool-title"><?php echo esc_html($tool->get_name()); ?></h3>
                                </div>
                                <p class="felan-tool-desc">
                                    <?php echo esc_html($tool->get_description()); ?>
                                </p>
                                <div class="felan-tool-action">
                                    <a href="<?php echo esc_url(add_query_arg(array('page' => 'felan_tools', 'tool' => $tool->get_id(), 'action' => 'run'), admin_url('admin.php'))); ?>"
                                       class="button button-primary">
                                        <?php esc_html_e('Run Tool', 'felan-framework'); ?>
                                    </a>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>

                    <?php if (!$has_tools) : ?>
                         <div class="notice notice-info inline"><p><?php esc_html_e('No tools found.', 'felan-framework'); ?></p></div>
                    <?php endif; ?>
                </div>
            </div>
            <?php
        }
    }
}
