<?php
if (!defined('ABSPATH')) {
    exit;
}

if (!class_exists('Felan_Tool_Refresh_Jobs') && class_exists('Felan_Tool')) {
    class Felan_Tool_Refresh_Jobs extends Felan_Tool
    {
        public function __construct()
        {
            $this->id = 'refresh_jobs';
            $this->name = esc_html__('Refresh Jobs Expiration', 'felan-framework');
            $this->description = esc_html__('Recalculate expiration dates and reactivate expired jobs based on custom criteria.', 'felan-framework');
            $this->group = 'maintenance';
            $this->icon = '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="feather feather-refresh-cw"><polyline points="23 4 23 10 17 10"></polyline><polyline points="1 20 1 14 7 14"></polyline><path d="M3.51 9a9 9 0 0 1 14.85-3.36L23 10M1 14l4.64 4.36A9 9 0 0 0 20.49 15"></path></svg>';
        }

        public function run()
        {
            // Logic handled in render view via POST for this specific tool
        }

        public function render()
        {
            $message = '';
            $message_type = ''; // 'success' or 'error'

            if (isset($_POST['felan_tool_refresh_jobs_nonce']) && wp_verify_nonce($_POST['felan_tool_refresh_jobs_nonce'], 'felan_tool_refresh_jobs_action')) {
                $min_days = isset($_POST['min_days']) ? absint($_POST['min_days']) : 55;
                $max_days = isset($_POST['max_days']) ? absint($_POST['max_days']) : 90;

                if ($min_days < 1 || $max_days < 1 || $min_days > $max_days) {
                    $message = esc_html__('Invalid days range. Min days must be less than or equal to Max days.', 'felan-framework');
                    $message_type = 'error';
                } else {
                    $result = $this->fix_all_jobs_expiration($min_days, $max_days);
                    if ($result['success']) {
                        $message = sprintf(
                            esc_html__('Success! Updated %d jobs. Changed %d expired jobs to publish. Range used: %d-%d days.', 'felan-framework'),
                            $result['count'],
                            $result['expired_count'],
                            $min_days,
                            $max_days
                        );
                        $message_type = 'success';
                    } else {
                        $message = $result['message'];
                        $message_type = 'error';
                    }
                }
            }

            $jobs_count = wp_count_posts('jobs');
            $default_min = 55;
            $default_max = 90;

            // Debug information
            $post_type_exists = post_type_exists('jobs');
            $enable_post_type_jobs = felan_get_option('enable_post_type_jobs', '1');
            $actual_jobs = get_posts(array(
                'post_type' => 'jobs',
                'post_status' => 'any',
                'posts_per_page' => -1,
                'fields' => 'ids'
            ));
?>
            <div class="wrap felan-wrap about-wrap felan-tools-container">
                <h1><?php echo esc_html($this->name); ?></h1>

                <p>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=felan_tools')); ?>" class="button">
                        &larr; <?php esc_html_e('Back to Tools', 'felan-framework'); ?>
                    </a>
                </p>

                <?php if ($message) : ?>
                    <div class="notice notice-<?php echo esc_attr($message_type); ?> is-dismissible">
                        <p><strong><?php echo esc_html($message); ?></strong></p>
                    </div>
                <?php endif; ?>

                <!-- Debug Information -->
                <div class="notice notice-info inline" style="margin-bottom: 20px;">
                    <p><strong><?php esc_html_e('Diagnostic Information:', 'felan-framework'); ?></strong></p>
                    <ul style="list-style: disc; margin-left: 20px;">
                        <li>
                            <?php esc_html_e('Post type "jobs" registered:', 'felan-framework'); ?>
                            <strong><?php echo $post_type_exists ? esc_html__('Yes', 'felan-framework') : esc_html__('No', 'felan-framework'); ?></strong>
                        </li>
                        <li>
                            <?php esc_html_e('Jobs enabled in options:', 'felan-framework'); ?>
                            <strong><?php echo $enable_post_type_jobs === '1' ? esc_html__('Yes', 'felan-framework') : esc_html__('No', 'felan-framework'); ?></strong>
                        </li>
                        <li>
                            <?php esc_html_e('Total jobs found in database:', 'felan-framework'); ?>
                            <strong><?php echo count($actual_jobs); ?></strong>
                        </li>
                    </ul>
                </div>

                <div class="felan-tool-runner-wrap">
                    <div class="felan-tool-form-card">
                        <div class="felan-tool-info-box">
                            <h4 class="info-title"><?php echo esc_html__('What this tool does:', 'felan-framework'); ?></h4>
                            <ul class="info-list">
                                <li><?php echo esc_html__('Resets all jobs expiration flags (enable_jobs_expires = 0)', 'felan-framework'); ?></li>
                                <li><?php echo esc_html__('Recalculates expiration dates: (days since publish) + random(min-max days)', 'felan-framework'); ?></li>
                                <li><strong><?php echo esc_html__('ONLY changes "expired" jobs back to "publish" status', 'felan-framework'); ?></strong></li>
                            </ul>
                        </div>

                        <div class="felan-tool-stats">
                            <div class="stat-item">
                                <span class="stat-label"><?php echo esc_html__('Total Jobs', 'felan-framework'); ?></span>
                                <span class="stat-value"><?php echo esc_html(isset($jobs_count->publish) ? ($jobs_count->publish + $jobs_count->expired + $jobs_count->pending + $jobs_count->draft) : 0); ?></span>
                            </div>
                            <div class="stat-item">
                                <span class="stat-label"><?php echo esc_html__('Published', 'felan-framework'); ?></span>
                                <span class="stat-value text-success"><?php echo esc_html(isset($jobs_count->publish) ? $jobs_count->publish : 0); ?></span>
                            </div>
                            <div class="stat-item">
                                <span class="stat-label"><?php echo esc_html__('Expired', 'felan-framework'); ?></span>
                                <span class="stat-value text-warning"><?php echo esc_html(isset($jobs_count->expired) ? $jobs_count->expired : 0); ?></span>
                            </div>
                        </div>

                        <form method="post" id="felan-refresh-jobs-form" class="felan-tool-run-form">
                            <?php wp_nonce_field('felan_tool_refresh_jobs_action', 'felan_tool_refresh_jobs_nonce'); ?>

                            <div class="felan-form-row">
                                <div class="felan-form-group">
                                    <label for="min_days"><?php echo esc_html__('Minimum Days', 'felan-framework'); ?></label>
                                    <input type="number" id="min_days" name="min_days" value="<?php echo esc_attr($default_min); ?>" min="1" max="365" class="small-text">
                                    <p class="description"><?php echo esc_html__('Minimum number of days to add from publish date.', 'felan-framework'); ?></p>
                                </div>
                                <div class="felan-form-group">
                                    <label for="max_days"><?php echo esc_html__('Maximum Days', 'felan-framework'); ?></label>
                                    <input type="number" id="max_days" name="max_days" value="<?php echo esc_attr($default_max); ?>" min="1" max="365" class="small-text">
                                    <p class="description"><?php echo esc_html__('Maximum number of days to add from publish date.', 'felan-framework'); ?></p>
                                </div>
                            </div>

                            <p class="submit">
                                <button type="submit" class="button button-primary button-large">
                                    <?php echo esc_html__('Run Refresh Jobs', 'felan-framework'); ?>
                                </button>
                            </p>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Confirmation Modal -->
            <div id="felan-confirm-modal" class="felan-modal" style="display:none;">
                <div class="felan-modal-backdrop"></div>
                <div class="felan-modal-content">
                    <div class="felan-modal-header">
                        <h3><?php echo esc_html__('Confirm Action', 'felan-framework'); ?></h3>
                    </div>
                    <div class="felan-modal-body">
                        <p><?php echo esc_html__('Are you sure you want to refresh all jobs expiration?', 'felan-framework'); ?></p>
                        <p><strong><?php echo esc_html__('This will:', 'felan-framework'); ?></strong></p>
                        <ul>
                            <li><?php echo esc_html__('Reset expiration flags for all jobs', 'felan-framework'); ?></li>
                            <li><?php echo esc_html__('Recalculate expiration dates', 'felan-framework'); ?></li>
                            <li><?php echo esc_html__('Reactivate expired jobs', 'felan-framework'); ?></li>
                        </ul>
                    </div>
                    <div class="felan-modal-footer">
                        <button type="button" class="button" id="felan-modal-cancel"><?php echo esc_html__('Cancel', 'felan-framework'); ?></button>
                        <button type="button" class="button button-primary" id="felan-modal-confirm"><?php echo esc_html__('Confirm', 'felan-framework'); ?></button>
                    </div>
                </div>
            </div>

            <script>
                jQuery(document).ready(function($) {
                    $('#felan-refresh-jobs-form button[type="submit"]').on('click', function(e) {
                        e.preventDefault();
                        $('#felan-confirm-modal').fadeIn(200);
                    });

                    $('#felan-modal-cancel, .felan-modal-backdrop').on('click', function() {
                        $('#felan-confirm-modal').fadeOut(200);
                    });

                    $('#felan-modal-confirm').on('click', function() {
                        $('#felan-confirm-modal').fadeOut(200);
                        $('#felan-refresh-jobs-form').off('submit').submit();
                    });
                });
            </script>
<?php
        }

        public function fix_all_jobs_expiration($min_days, $max_days)
        {
            try {
                // Ensure constant is defined
                if (!defined('FELAN_METABOX_PREFIX')) {
                    return array(
                        'success' => false,
                        'count' => 0,
                        'expired_count' => 0,
                        'message' => esc_html__('FELAN_METABOX_PREFIX constant is not defined.', 'felan-framework')
                    );
                }

                $args = array(
                    'post_type' => 'jobs',
                    'post_status' => 'any',
                    'posts_per_page' => -1,
                    'suppress_filters' => true,
                    'no_found_rows' => true
                );

                $jobs_query = new WP_Query($args);
                $current_date = date('Y-m-d');
                $updated_count = 0;
                $expired_count = 0;

                while ($jobs_query->have_posts()) {
                    $jobs_query->the_post();
                    $job_id = get_the_ID();
                    $publish_date = get_the_date('Y-m-d');
                    $current_status = get_post_status();

                    if (!$job_id || !$publish_date) {
                        continue;
                    }

                    // Calculate expiration days
                    $days_since_publish = (strtotime($current_date) - strtotime($publish_date)) / 86400;
                    $random_additional_days = $min_days + ($job_id % ($max_days - $min_days + 1));
                    $new_jobs_days_closing = ceil($days_since_publish) + $random_additional_days;

                    // Update meta fields
                    update_post_meta($job_id, FELAN_METABOX_PREFIX . 'enable_jobs_expires', '0');
                    update_post_meta($job_id, FELAN_METABOX_PREFIX . 'enable_jobs_package_expires', '0');
                    update_post_meta($job_id, FELAN_METABOX_PREFIX . 'jobs_days_closing', $new_jobs_days_closing);

                    // Change expired jobs to publish
                    if ($current_status == 'expired') {
                        global $wpdb;
                        // Update directly in database to bypass hooks
                        $wpdb->update(
                            $wpdb->posts,
                            array('post_status' => 'publish'),
                            array('ID' => $job_id),
                            array('%s'),
                            array('%d')
                        );
                        clean_post_cache($job_id);
                        $expired_count++;
                    }

                    $updated_count++;
                }

                wp_reset_postdata();

                return array(
                    'success' => true,
                    'count' => $updated_count,
                    'expired_count' => $expired_count,
                    'message' => ''
                );
            } catch (Exception $e) {
                wp_reset_postdata();
                return array(
                    'success' => false,
                    'count' => 0,
                    'expired_count' => 0,
                    'message' => sprintf(esc_html__('Error: %s', 'felan-framework'), $e->getMessage())
                );
            }
        }
    }
}
