<?php
/**
 * Project Alerts List Template
 * Server-side rendered list of project alerts
 *
 * @var int $user_id Current user ID
 * @var string $user_email Current user email
 * @var bool $alerts_enabled Whether alerts are enabled for user's package
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Extract variables
$user_id = isset($user_id) ? $user_id : 0;
$user_email = isset($user_email) ? $user_email : '';
$alerts_enabled = isset($alerts_enabled) ? $alerts_enabled : true;

// Get user alerts
$args = array(
    'post_type' => 'project_alerts',
    'post_status' => 'publish',
    'posts_per_page' => -1,
    'author' => $user_id,
    'orderby' => 'date',
    'order' => 'DESC',
);

$query = new WP_Query($args);
$alerts = array();

if ($query->have_posts()) {
    while ($query->have_posts()) {
        $query->the_post();
        $post_id = get_the_ID();

        $created_at = get_post_meta($post_id, FELAN_METABOX_PREFIX . 'project_alerts_created_at', true);
        $updated_at = get_post_meta($post_id, FELAN_METABOX_PREFIX . 'project_alerts_updated_at', true);

        // Fallback to post date if created_at not set
        if (empty($created_at)) {
            $created_at = get_post_time('Y-m-d H:i:s', false, $post_id);
        }

        // Fallback to post modified date if updated_at not set
        if (empty($updated_at)) {
            $updated_at = get_post_modified_time('Y-m-d H:i:s', false, $post_id);
        }

        $alerts[] = array(
            'id' => $post_id,
            'name' => get_the_title(),
            'email' => get_post_meta($post_id, FELAN_METABOX_PREFIX . 'project_alerts_email', true),
            'categories' => get_post_meta($post_id, FELAN_METABOX_PREFIX . 'project_alerts_categories', true) ?: array(),
            'skills' => get_post_meta($post_id, FELAN_METABOX_PREFIX . 'project_alerts_skills', true) ?: array(),
            'language' => get_post_meta($post_id, FELAN_METABOX_PREFIX . 'project_alerts_language', true) ?: array(),
            'career' => get_post_meta($post_id, FELAN_METABOX_PREFIX . 'project_alerts_career', true) ?: array(),
            'location' => get_post_meta($post_id, FELAN_METABOX_PREFIX . 'project_alerts_location', true) ?: array(),
            'state' => get_post_meta($post_id, FELAN_METABOX_PREFIX . 'project_alerts_state', true) ?: array(),
            'match_type' => get_post_meta($post_id, FELAN_METABOX_PREFIX . 'project_alerts_match_type', true) ?: 'OR',
            'frequency' => get_post_meta($post_id, FELAN_METABOX_PREFIX . 'project_alerts_frequency', true) ?: 'daily',
            'created_at' => $created_at,
            'updated_at' => $updated_at,
        );
    }
    wp_reset_postdata();
}

// Helper function to format date
function felan_format_alert_date($date_string) {
    if (empty($date_string)) {
        return '';
    }

    $date = strtotime($date_string);
    if (!$date) {
        return $date_string;
    }

    $now = current_time('timestamp');
    $diff = $now - $date;
    $seconds = floor($diff);
    $minutes = floor($seconds / 60);
    $hours = floor($minutes / 60);
    $days = floor($hours / 24);

    if ($seconds < 60) {
        return esc_html__('Just now', 'felan-framework');
    } elseif ($minutes < 60) {
        return sprintf(
            _n('%d minute ago', '%d minutes ago', $minutes, 'felan-framework'),
            $minutes
        );
    } elseif ($hours < 24) {
        return sprintf(
            _n('%d hour ago', '%d hours ago', $hours, 'felan-framework'),
            $hours
        );
    } elseif ($days < 7) {
        return sprintf(
            _n('%d day ago', '%d days ago', $days, 'felan-framework'),
            $days
        );
    } else {
        return date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $date);
    }
}

$has_alerts = !empty($alerts);
$edit_id = isset($_GET['edit']) ? absint($_GET['edit']) : 0;
$is_edit_mode = $edit_id > 0;
?>

<!-- Existing Project Alerts List -->
<div class="project-alerts-list-wrapper" id="project-alerts-list" style="<?php echo $is_edit_mode ? 'display: none;' : ''; ?>">
    <div class="alerts-list-header">
        <h3 class="alerts-list-title">
            <i class="far fa-bell" aria-hidden="true"></i>
            <?php esc_html_e('My Project Alerts', 'felan-framework'); ?>
        </h3>
        <?php if ($has_alerts && $alerts_enabled) : ?>
            <button type="button" class="btn-create-alert" id="btn-create-alert">
                <i class="far fa-plus" aria-hidden="true"></i>
                <span class="btn-text"><?php esc_html_e('Create New Alert', 'felan-framework'); ?></span>
            </button>
        <?php endif; ?>
    </div>
    <div class="alerts-list-content" id="alerts-list-content">
        <?php if ($has_alerts) : ?>
            <div class="alerts-grid">
                <?php foreach ($alerts as $alert) :
                    $is_editing = ($edit_id && $edit_id == $alert['id']);
                    $has_update = !empty($alert['updated_at']) && $alert['updated_at'] !== $alert['created_at'];

                    // Taxonomy config
                    $taxonomy_config = array(
                        'categories' => array('icon' => 'far fa-folder', 'label' => esc_html__('Categories', 'felan-framework')),
                        'skills' => array('icon' => 'far fa-tags', 'label' => esc_html__('Skills', 'felan-framework')),
                        'language' => array('icon' => 'far fa-globe', 'label' => esc_html__('Languages', 'felan-framework')),
                        'career' => array('icon' => 'far fa-briefcase', 'label' => esc_html__('Careers', 'felan-framework')),
                        'location' => array('icon' => 'far fa-map-marker-alt', 'label' => esc_html__('Location', 'felan-framework')),
                        'state' => array('icon' => 'far fa-map', 'label' => esc_html__('State/Province', 'felan-framework')),
                    );

                    $taxonomies = array();
                    $total_count = 0;
                    foreach ($taxonomy_config as $key => $config) {
                        if (!empty($alert[$key]) && is_array($alert[$key]) && count($alert[$key]) > 0) {
                            $taxonomies[] = array(
                                'key' => $key,
                                'count' => count($alert[$key]),
                                'icon' => $config['icon'],
                                'label' => $config['label'],
                            );
                            $total_count += count($alert[$key]);
                        }
                    }
                    ?>
                    <div class="alert-item<?php echo $is_editing ? ' alert-item-editing' : ''; ?>" data-id="<?php echo esc_attr($alert['id']); ?>">
                        <div class="alert-item-header">
                            <div class="alert-item-title-wrapper">
                                <h4 class="alert-item-title"><?php echo esc_html($alert['name'] ?: $alert['email']); ?></h4>
                                <?php if ($is_editing) : ?>
                                    <span class="alert-item-badge editing-badge">
                                        <i class="far fa-edit" aria-hidden="true"></i>
                                        <?php esc_html_e('Editing', 'felan-framework'); ?>
                                    </span>
                                <?php endif; ?>
                            </div>
                            <div class="alert-item-actions">
                                <button type="button" class="btn-edit-alert<?php echo $is_editing ? ' active' : ''; ?>" data-id="<?php echo esc_attr($alert['id']); ?>" aria-label="<?php esc_attr_e('Edit Alert', 'felan-framework'); ?>">
                                    <i class="far fa-edit" aria-hidden="true"></i>
                                    <span class="btn-text"><?php esc_html_e('Edit', 'felan-framework'); ?></span>
                                </button>
                                <button type="button" class="btn-delete-alert" data-id="<?php echo esc_attr($alert['id']); ?>" aria-label="<?php esc_attr_e('Delete', 'felan-framework'); ?>">
                                    <i class="far fa-trash-alt" aria-hidden="true"></i>
                                </button>
                            </div>
                        </div>
                        <div class="alert-item-body">
                            <div class="alert-item-meta">
                                <span class="alert-meta-item">
                                    <i class="far fa-envelope" aria-hidden="true"></i>
                                    <?php echo esc_html($alert['email']); ?>
                                </span>
                                <span class="alert-meta-item">
                                    <i class="far fa-filter" aria-hidden="true"></i>
                                    <?php echo esc_html($alert['match_type'] ?: 'OR'); ?>
                                </span>
                                <span class="alert-meta-item">
                                    <i class="far fa-clock" aria-hidden="true"></i>
                                    <?php echo esc_html($alert['frequency'] ?: 'daily'); ?>
                                </span>
                            </div>

                            <div class="alert-item-timestamps">
                                <?php if ($has_update) : ?>
                                    <span class="alert-timestamp-item" title="<?php echo esc_attr($alert['updated_at']); ?>">
                                        <i class="far fa-edit" aria-hidden="true"></i>
                                        <span class="timestamp-label"><?php esc_html_e('Updated:', 'felan-framework'); ?></span>
                                        <span class="timestamp-value"><?php echo esc_html(felan_format_alert_date($alert['updated_at'])); ?></span>
                                    </span>
                                <?php elseif (!empty($alert['created_at'])) : ?>
                                    <span class="alert-timestamp-item" title="<?php echo esc_attr($alert['created_at']); ?>">
                                        <i class="far fa-calendar-plus" aria-hidden="true"></i>
                                        <span class="timestamp-label"><?php esc_html_e('Created:', 'felan-framework'); ?></span>
                                        <span class="timestamp-value"><?php echo esc_html(felan_format_alert_date($alert['created_at'])); ?></span>
                                    </span>
                                <?php endif; ?>
                            </div>

                            <?php if (!empty($taxonomies)) : ?>
                                <div class="alert-item-taxonomies">
                                    <div class="taxonomy-badges">
                                        <?php foreach ($taxonomies as $tax) : ?>
                                            <span class="taxonomy-badge">
                                                <i class="<?php echo esc_attr($tax['icon']); ?>" aria-hidden="true"></i>
                                                <?php echo esc_html($tax['label']); ?>: <strong><?php echo esc_html($tax['count']); ?></strong>
                                            </span>
                                        <?php endforeach; ?>
                                    </div>
                                    <div class="taxonomy-summary">
                                        <?php esc_html_e('Total:', 'felan-framework'); ?> <strong><?php echo esc_html($total_count); ?></strong> <?php esc_html_e('items selected', 'felan-framework'); ?>
                                    </div>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php else : ?>
            <div class="alerts-empty">
                <i class="far fa-inbox" aria-hidden="true"></i>
                <p><?php esc_html_e('You haven\'t created any project alerts yet.', 'felan-framework'); ?></p>
                <?php if ($alerts_enabled) : ?>
                    <button type="button" class="btn-create-alert-empty" id="btn-create-alert-empty">
                        <i class="far fa-plus" aria-hidden="true"></i>
                        <?php esc_html_e('Create Your First Alert', 'felan-framework'); ?>
                    </button>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>
</div>
