<?php

/**
 * Wire Transfer Modal Template
 * Reusable modal for displaying wire transfer payment details
 *
 * Variables required:
 * - $context: 'freelancer' or 'service' or 'project'
 */

if (!defined('ABSPATH')) {
  exit; // Exit if accessed directly
}

// Determine which options to load based on context
$option_prefix = '';
switch ($context) {
  case 'freelancer':
    $option_prefix = 'freelancer_';
    break;
  case 'service':
    $option_prefix = 'service_';
    break;
  case 'project':
    $option_prefix = 'project_';
    break;
  default:
    $option_prefix = 'freelancer_';
}

$enable_wire_transfer = felan_get_option($option_prefix . 'enable_wire_transfer', '0');

if ($enable_wire_transfer === '1') :
  $wire_transfer_account_number = trim(felan_get_option($option_prefix . 'wire_transfer_account_number', ''));
  $wire_transfer_account_holder = trim(felan_get_option($option_prefix . 'wire_transfer_account_holder', ''));
  $wire_transfer_bank_name = trim(felan_get_option($option_prefix . 'wire_transfer_bank_name', ''));
  $wire_transfer_bank_address = trim(felan_get_option($option_prefix . 'wire_transfer_bank_address', ''));
  $wire_transfer_swift_bic = trim(felan_get_option($option_prefix . 'wire_transfer_swift_bic', ''));
  $wire_transfer_iban = trim(felan_get_option($option_prefix . 'wire_transfer_iban', ''));
  $wire_transfer_routing_number = trim(felan_get_option($option_prefix . 'wire_transfer_routing_number', ''));
  $wire_transfer_country = trim(felan_get_option($option_prefix . 'wire_transfer_country', ''));
  $wire_transfer_instructions = felan_get_option($option_prefix . 'wire_transfer_instructions', '');

  // Check if minimum required fields are configured
  $has_required_fields = !empty($wire_transfer_account_number) &&
    !empty($wire_transfer_account_holder) &&
    !empty($wire_transfer_bank_name);

  // Check if at least one identification field exists (SWIFT/IBAN/Routing)
  $has_identification = !empty($wire_transfer_swift_bic) ||
    !empty($wire_transfer_iban) ||
    !empty($wire_transfer_routing_number);
?>
  <div id="felan-wire-transfer-modal" class="felan-modal" style="display: none;"
    data-context="<?php echo esc_attr($context); ?>"
    data-configured="<?php echo esc_attr($has_required_fields && $has_identification ? '1' : '0'); ?>">
    <div class="felan-modal-overlay"></div>
    <div class="felan-modal-content">
      <div class="felan-modal-header">
        <h2><?php esc_html_e('Wire Transfer Details', 'felan-framework'); ?></h2>
        <button class="felan-modal-close" type="button">&times;</button>
      </div>

      <div class="felan-modal-body">
        <?php if ($has_required_fields && $has_identification) : ?>
          <div class="wire-transfer-notice">
            <p><strong><?php esc_html_e('Please transfer the payment to the following account:', 'felan-framework'); ?></strong></p>
            <p><?php esc_html_e('Make sure to include your Order Number in the transfer reference.', 'felan-framework'); ?></p>
          </div>

          <div class="wire-transfer-details">
            <?php if (!empty($wire_transfer_country)) : ?>
              <div class="transfer-detail-row">
                <span class="detail-label"><?php esc_html_e('Country:', 'felan-framework'); ?></span>
                <span class="detail-value"><?php echo esc_html($wire_transfer_country); ?></span>
              </div>
            <?php endif; ?>

            <?php if (!empty($wire_transfer_account_number)) : ?>
              <div class="transfer-detail-row">
                <span class="detail-label"><?php esc_html_e('Account Number:', 'felan-framework'); ?></span>
                <span class="detail-value"><?php echo esc_html($wire_transfer_account_number); ?></span>
                <button class="copy-btn" data-copy="<?php echo esc_attr($wire_transfer_account_number); ?>" type="button">
                  <i class="far fa-copy"></i>
                </button>
              </div>
            <?php endif; ?>

            <?php if (!empty($wire_transfer_account_holder)) : ?>
              <div class="transfer-detail-row">
                <span class="detail-label"><?php esc_html_e('Account Holder:', 'felan-framework'); ?></span>
                <span class="detail-value"><?php echo esc_html($wire_transfer_account_holder); ?></span>
                <button class="copy-btn" data-copy="<?php echo esc_attr($wire_transfer_account_holder); ?>" type="button">
                  <i class="far fa-copy"></i>
                </button>
              </div>
            <?php endif; ?>

            <?php if (!empty($wire_transfer_bank_name)) : ?>
              <div class="transfer-detail-row">
                <span class="detail-label"><?php esc_html_e('Bank Name:', 'felan-framework'); ?></span>
                <span class="detail-value"><?php echo esc_html($wire_transfer_bank_name); ?></span>
              </div>
            <?php endif; ?>

            <?php if (!empty($wire_transfer_bank_address)) : ?>
              <div class="transfer-detail-row">
                <span class="detail-label"><?php esc_html_e('Bank Address:', 'felan-framework'); ?></span>
                <span class="detail-value"><?php echo nl2br(esc_html($wire_transfer_bank_address)); ?></span>
              </div>
            <?php endif; ?>

            <?php if (!empty($wire_transfer_swift_bic)) : ?>
              <div class="transfer-detail-row">
                <span class="detail-label"><?php esc_html_e('SWIFT/BIC:', 'felan-framework'); ?></span>
                <span class="detail-value"><?php echo esc_html($wire_transfer_swift_bic); ?></span>
                <button class="copy-btn" data-copy="<?php echo esc_attr($wire_transfer_swift_bic); ?>" type="button">
                  <i class="far fa-copy"></i>
                </button>
              </div>
            <?php endif; ?>

            <?php if (!empty($wire_transfer_iban)) : ?>
              <div class="transfer-detail-row">
                <span class="detail-label"><?php esc_html_e('IBAN:', 'felan-framework'); ?></span>
                <span class="detail-value"><?php echo esc_html($wire_transfer_iban); ?></span>
                <button class="copy-btn" data-copy="<?php echo esc_attr($wire_transfer_iban); ?>" type="button">
                  <i class="far fa-copy"></i>
                </button>
              </div>
            <?php endif; ?>

            <?php if (!empty($wire_transfer_routing_number)) : ?>
              <div class="transfer-detail-row">
                <span class="detail-label"><?php esc_html_e('Routing Number:', 'felan-framework'); ?></span>
                <span class="detail-value"><?php echo esc_html($wire_transfer_routing_number); ?></span>
                <button class="copy-btn" data-copy="<?php echo esc_attr($wire_transfer_routing_number); ?>" type="button">
                  <i class="far fa-copy"></i>
                </button>
              </div>
            <?php endif; ?>

            <?php if (!empty($wire_transfer_instructions)) : ?>
              <div class="transfer-detail-instructions">
                <h4><?php esc_html_e('Additional Instructions:', 'felan-framework'); ?></h4>
                <?php echo wp_kses_post($wire_transfer_instructions); ?>
              </div>
            <?php endif; ?>
          </div>
        <?php else : ?>
          <!-- Show warning if wire transfer is not properly configured -->
          <div class="wire-transfer-notice warning">
            <p class="notice-title"><i class="fas fa-exclamation-triangle"></i> <strong><?php esc_html_e('Configuration Incomplete', 'felan-framework'); ?></strong></p>
            <p><?php esc_html_e('Wire transfer payment is currently unavailable. The payment details have not been fully configured by the administrator.', 'felan-framework'); ?></p>
            <p><?php esc_html_e('Please choose another payment method or contact support for assistance.', 'felan-framework'); ?></p>
          </div>
        <?php endif; ?>
      </div>

      <div class="felan-modal-footer">
        <button type="button" class="felan-button felan-button-secondary felan-modal-close">
          <?php esc_html_e('Cancel', 'felan-framework'); ?>
        </button>
        <?php if ($has_required_fields && $has_identification) : ?>
          <button type="button" class="felan-confirm-wire-transfer felan-button felan-button-primary">
            <i class="fas fa-check"></i> <?php esc_html_e('I have completed the transfer', 'felan-framework'); ?>
          </button>
        <?php endif; ?>
      </div>
    </div>
  </div>
<?php endif; ?>
