<?php

namespace TPCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Utils;

if (!defined('ABSPATH'))
    exit; // Exit if accessed directly

/**
 * Tp Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TP_About_Tab extends Widget_Base
{

    use \TPCore\Widgets\TP_Style_Trait;
    use \TPCore\Widgets\TP_Animation_Trait;

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'tp-about-tab';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('About Tab', 'tpcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'tp-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['tpcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['tpcore'];
    }


    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */

    protected function register_controls()
    {
        $this->register_controls_section();
        $this->style_tab_content();
    }
    protected function register_controls_section()
    {

        // layout Panel
        $this->start_controls_section(
            'tp_layout',
            [
                'label' => esc_html__('Design Layout', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_design_style',
            [
                'label' => esc_html__('Select Layout', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'tpcore'),
                    'layout-2' => esc_html__('Layout 2', 'tpcore'),
                ],
                'default' => 'layout-1',
            ]
        );

        $this->end_controls_section();

        // section title 
        $this->tp_section_title_render_controls('about_tab_section_heading', 'Section Heading', 'layout-1');


        // About Tab group
        $this->start_controls_section(
            'tp_about_tab_rep_section',
            [
                'label' => esc_html__('About List', 'tpcore'),
                'description' => esc_html__('Control all the style settings from Style tab', 'tpcore'),
                'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'tp_about_tab_shape_switcher',
            [
                'label' => esc_html__('Image shape', 'tpcore'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'tpcore'),
                'label_off' => esc_html__('No', 'tpcore'),
                'return_value' => 'yes',
                'default' => 0,
                'separator' => 'before',
                'condition' => [
                    'tp_design_style' => 'layout-1',
                ],
            ]
        );

        $repeater = new \Elementor\Repeater();


        $repeater->add_control(
            'repeater_condition',
            [
                'label' => __('Field condition', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'style_1' => __('Style 1', 'tpcore'),
                    'style_2' => __('Style 2', 'tpcore'),
                ],
                'default' => 'style_1',
                'frontend_available' => true,
                'style_transfer' => true,
            ]
        );

        $repeater->add_control(
            'tp_about_tab_active',
            [
                'label' => esc_html__('Active This', 'tpcore'),
                'type' => \Elementor\Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'tpcore'),
                'label_off' => esc_html__('No', 'tpcore'),
                'return_value' => 'yes',
                'default' => 0,
                'separator' => 'before',
            ]
        );

        $repeater->start_controls_tabs(
            'tp_about_tab_tabs'
        );

        // tab heading 
        $repeater->start_controls_tab(
            'tp_about_tab_normal_tab',
            [
                'label' => esc_html__('Heading', 'tpcore'),
            ]
        );

        $repeater->add_control(
            'tp_about_tab_title',
            [
                'label' => esc_html__('Title', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Title here', 'tpcore'),
                'label_block' => true,
            ]
        );

        $repeater->end_controls_tab();


        // Tab content 
        $repeater->start_controls_tab(
            'tp_about_content_tab',
            [
                'label' => esc_html__('Content', 'tpcore'),
            ]
        );

        $repeater->add_control(
            'tp_about_content_tab_image',
            [
                'type' => Controls_Manager::MEDIA,
                'label' => __('Slider Image', 'tpcore'),
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );
        $repeater->add_control(
            'tp_about_content_item_image',
            [
                'type' => Controls_Manager::MEDIA,
                'label' => __('Item Image', 'tpcore'),
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'dynamic' => [
                    'active' => true,
                ],
                'condition' => [
                    'repeater_condition' => 'style_2',
                ],
            ]
        );
        $repeater->add_group_control(
            \Elementor\Group_Control_Image_Size::get_type(),
            [
                'name' => 'content_thumb',
                'exclude' => ['custom'],
                'include' => [],
                'default' => 'full',
            ]
        );

        $repeater->add_control(
            'tp_about_content_tab_title',
            [
                'label' => esc_html__('Slider Title', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Your Slider Title here', 'tpcore'),
                'description' => tp_get_allowed_html_desc('Housing & Dining'),
                'label_block' => true,
                'condition' => [
                    'repeater_condition' => 'style_2',
                ],
            ]
        );

        $repeater->add_control(
            'tp_about_tab_content',
            [
                'label' => esc_html__('Slider Content', 'tpcore'),
                'type' => Controls_Manager::TEXTAREA,
                'default' => esc_html__('Your Slider Content here', 'tpcore'),
                'description' => tp_get_allowed_html_desc('intermediate'),
                'label_block' => true,
            ]
        );

        $repeater->add_control(
            'tp_about_content_tab_btn_title',
            [
                'label' => esc_html__('Button title', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Find out More', 'tpcore'),
                'label_block' => false,
                'condition' => [
                    'repeater_condition' => 'style_1',
                ],
            ]
        );

        tp_render_links_controls($repeater, 'about_tab_content_btn', );

        $repeater->end_controls_tab();

        $repeater->end_controls_tabs();

        $this->add_control(
            'tp_about_lists',
            [
                'label' => esc_html__('About Tab List', 'tpcore'),
                'type' => \Elementor\Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'tp_about_tab_title' => esc_html__('Gorki Campus', 'tpcore'),
                    ],
                    [
                        'tp_about_tab_title' => esc_html__('Skolkovo Campus', 'tpcore')
                    ],
                    [
                        'tp_about_tab_title' => esc_html__('Saint Petersburg Campus', 'tpcore')
                    ]
                ],
                'title_field' => '{{{ tp_about_tab_title }}}',
            ]
        );

        //  image size 
        $this->add_group_control(
            \Elementor\Group_Control_Image_Size::get_type(),
            [
                'name' => 'content_thumbnail', // Usage: `{name}_size` and `{name}_custom_dimension`, in this case `thumbnail_size` and `thumbnail_custom_dimension`.
                'exclude' => ['custom'],
                'include' => [],
                'default' => 'full',
            ]
        );

        $this->end_controls_section();

        $this->tp_creative_animation_multi(null, 'about_tab_left', 'tp_design_style', 'Left Content Animation');
        $this->tp_creative_animation_multi(null, 'about_tab_right', 'tp_design_style', 'Right Content Animation');
    }

    // style_tab_content
    protected function style_tab_content()
    {
        $this->tp_basic_style_controls('about_sec_title', 'Heading Title', '.tp-el-sec-title', 'layout-1');
        $this->tp_basic_style_controls('about_tab_description', 'Heading Description', '.tp-el-sec-desc', 'layout-1');
        $this->tp_basic_style_controls('about_tab_title', 'Tab Title', '.tp-el-tab-title');
        $this->tp_basic_style_controls('about_sec_content', 'Tab Content', '.tp-el-tab-content');
        $this->tp_link_controls_style('', 'btn1_style', 'Tab Button', '.tp-els-btn', );

    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

        ?>
        <?php if ($settings['tp_design_style'] == 'layout-2'): ?>

            <div class="tp-campus-student-content">
                <div class="tab-content" id="myTabContent">
                    <?php foreach ($settings['tp_about_lists'] as $key => $item):

                        $active_show = $item['tp_about_tab_active'] ? 'active show' : NULL;

                        // thumbnail
                        $img = tp_get_img($item, 'tp_about_content_tab_image', 'content_thumb');
                        $content_img = tp_get_img($item, 'tp_about_content_item_image', 'content_thumb');

                        $tp_button_title = $item['tp_about_content_tab_btn_title'];
                        $tp_address = $item['tp_about_tab_content'];

                        ?>
                        <div class="tab-pane fade <?php echo esc_attr($active_show); ?>" id="home-<?php echo esc_attr($key + 1); ?>"
                            role="tabpanel" aria-labelledby="home-tab-<?php echo esc_attr($key + 1); ?>">
                            <div class="tp-campus-student-wrap p-relative">

                                <?php if (!empty($img['tp_about_content_tab_image'])): ?>
                                    <div class="tp-campus-student-thumb">
                                        <img src="<?php echo esc_url($img['tp_about_content_tab_image']) ?>"
                                            alt="<?php echo esc_url($img['tp_about_content_tab_image_alt']) ?>">
                                    </div>
                                <?php endif; ?>


                                <div class="tp-campus-student-item">

                                    <?php if (!empty($item['tp_about_content_tab_title'])): ?>
                                        <h4 class="tp-campus-student-item-title tp-el-tab-title">
                                            <?php echo esc_html($item['tp_about_content_tab_title']); ?>
                                        </h4>
                                    <?php endif; ?>

                                    <?php if (!empty($item['tp_about_tab_content'])): ?>
                                        <p class="tp-el-tab-content">
                                            <?php echo esc_html($item['tp_about_tab_content']); ?>
                                        </p>
                                    <?php endif; ?>

                                    <?php if (!empty($content_img['tp_about_content_item_image'])): ?>
                                        <div class="tp-campus-student-item-thumb">
                                            <img src="<?php echo esc_url($content_img['tp_about_content_item_image']) ?>"
                                                alt="<?php echo esc_url($content_img['tp_about_content_item_image_alt']) ?>">
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>

                </div>
            </div>

            <div class="tp-campus-student-list">
                <ul class="nav nav-tabs " id="myTab" role="tablist">
                    <?php foreach ($settings['tp_about_lists'] as $key => $item):

                        $active_title = $item['tp_about_tab_active'] ? 'active' : NULL;
                        $tp_title = $item['tp_about_tab_title'];
                        ?>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link tp-els-btn <?php echo esc_attr($active_title); ?>"
                                id="home-tab-<?php echo esc_attr($key + 1); ?>" data-bs-toggle="tab"
                                data-bs-target="#home-<?php echo esc_attr($key + 1); ?>" type="button" role="tab" aria-controls="home"
                                aria-selected="true"><?php echo esc_html($tp_title); ?></button>
                        </li>
                    <?php endforeach; ?>

                </ul>
            </div>

        <?php else:
            $this->add_render_attribute('title_args', 'class', 'tp-about-campus-title tp-el-sec-title');

            $animation_left = $this->tp_animation_show_multi($settings, 'about_tab_left');
            $animation_right = $this->tp_animation_show_multi($settings, 'about_tab_right');
            $tp_shape_switcher = $settings['tp_about_tab_shape_switcher'];

            ?>
            <section class="tp-about-campus-area grey-bg p-relative">
                <div class="container">
                    <div class="row">
                        <div class="col-lg-4">
                            <div class="tp-about-campus-tab <?php echo esc_attr($animation_left['animation']); ?>" <?php echo $animation_left['duration'] . ' ' . $animation_left['delay']; ?>>
                                <div class="tp-about-campus-heading">
                                    <?php
                                    if (!empty($settings['tp_about_tab_section_heading_title'])):
                                        printf(
                                            '<%1$s %2$s>%3$s</%1$s>',
                                            tag_escape($settings['tp_about_tab_section_heading_title_tag']),
                                            $this->get_render_attribute_string('title_args'),
                                            tp_kses($settings['tp_about_tab_section_heading_title'])
                                        );
                                    endif;
                                    ?>

                                    <?php if (!empty($settings['tp_about_tab_section_heading_description'])): ?>
                                        <p class="tp-el-sec-desc">
                                            <?php echo tp_kses($settings['tp_about_tab_section_heading_description']); ?>
                                        </p>
                                    <?php endif; ?>
                                </div>
                                <ul class="nav nav-tabs justify-content-center" id="myTab" role="tablist">
                                    <?php foreach ($settings['tp_about_lists'] as $key => $item):

                                        $active_title = $item['tp_about_tab_active'] ? 'active' : NULL;
                                        $tp_title = $item['tp_about_tab_title'];
                                        ?>
                                        <li class="nav-item" role="presentation">
                                            <button class="nav-link tp-el-tab-title <?php echo esc_attr($active_title); ?>"
                                                id="home-tab-<?php echo esc_attr($key + 1); ?>" data-bs-toggle="tab"
                                                data-bs-target="#home-<?php echo esc_attr($key + 1); ?>" type="button" role="tab"
                                                aria-controls="home" aria-selected="true"><?php echo esc_html($tp_title); ?></button>
                                        </li>
                                    <?php endforeach; ?>

                                </ul>
                            </div>
                        </div>
                        <div class="col-lg-8 p-relative">
                            <?php if ("yes" == $tp_shape_switcher): ?>
                                <div class="tp-about-campus-shape">
                                    <div class="shape-1">
                                        <img src="<?php echo get_template_directory_uri(); ?>/assets/img/campus/campuses-shape-1.jpg"
                                            alt="">
                                    </div>
                                    <div class="shape-2">
                                        <img src="<?php echo get_template_directory_uri(); ?>/assets/img/campus/campuses-shape-2.jpg"
                                            alt="">
                                    </div>
                                </div>
                            <?php endif ?>

                            <div class="tab-content <?php echo esc_attr($animation_right['animation']); ?>" <?php echo $animation_right['duration'] . ' ' . $animation_right['delay']; ?>>
                                <?php foreach ($settings['tp_about_lists'] as $key => $item):

                                    $active_show = $item['tp_about_tab_active'] ? 'active show' : NULL;

                                    // thumbnail
                                    $tp_main_image = !empty($item['tp_about_content_tab_image']['id']) ? wp_get_attachment_image_url($item['tp_about_content_tab_image']['id'], $settings['content_thumbnail_size']) : $item['tp_about_content_tab_image']['url'];
                                    $tp_main_image_alt = get_post_meta($item["tp_about_content_tab_image"]["id"], "_wp_attachment_image_alt", true);

                                    $tp_button_title = $item['tp_about_content_tab_btn_title'];
                                    $tp_address = $item['tp_about_tab_content'];

                                    $attrs = tp_get_repeater_links_attr($item, 'about_tab_content_btn');
                                    extract($attrs);

                                    $links_attrs = [
                                        'href' => $link,
                                        'target' => $target,
                                        'rel' => $rel,
                                    ];
                                    ?>
                                    <div class="tab-pane fade <?php echo esc_attr($active_show); ?>"
                                        id="home-<?php echo esc_attr($key + 1); ?>" role="tabpanel"
                                        aria-labelledby="home-tab-<?php echo esc_attr($key + 1); ?>">
                                        <div class="tp-about-campus-box p-relative">
                                            <?php if (!empty($tp_main_image)): ?>
                                                <div class="tp-about-campus-thumb">
                                                    <img src="<?php echo esc_url($tp_main_image); ?>"
                                                        alt="<?php echo esc_attr($tp_main_image_alt); ?>">
                                                </div>
                                            <?php endif; ?>

                                            <div class="tp-about-campus-content d-flex justify-content-between">
                                                <?php if (!empty($tp_address)): ?>
                                                    <div class="tp-about-campus-location">
                                                        <span><svg width="14" height="17" viewBox="0 0 14 17" fill="none"
                                                                xmlns="http://www.w3.org/2000/svg">
                                                                <path
                                                                    d="M3.53162 0.93677C5.71647 -0.332725 8.40201 -0.310536 10.5664 0.994893C12.7094 2.32691 14.0119 4.70417 13.9999 7.26143C13.9499 9.80192 12.5533 12.19 10.8075 14.0361C9.79981 15.1064 8.6726 16.0528 7.44883 16.856C7.32279 16.9289 7.18474 16.9777 7.04147 17C6.90359 16.9941 6.7693 16.9534 6.65074 16.8814C4.78242 15.6745 3.14333 14.134 1.81233 12.3339C0.698586 10.8313 0.0658344 9.01599 0 7.13441L0.00498621 6.86068C0.0959219 4.40464 1.42479 2.16093 3.53162 0.93677ZM7.90725 5.03476C7.01906 4.65723 5.99503 4.86234 5.31331 5.55434C4.63158 6.24633 4.42663 7.28871 4.79415 8.19477C5.16168 9.10082 6.02917 9.69183 6.99159 9.69183C7.62209 9.69636 8.22817 9.44381 8.67479 8.99046C9.12141 8.53711 9.37147 7.92062 9.36924 7.27837C9.37259 6.29802 8.79544 5.4123 7.90725 5.03476Z"
                                                                    fill="currentColor" />
                                                            </svg></span>
                                                        <a class="tp-el-tab-content "><?php echo tp_kses($tp_address); ?></a>
                                                    </div>
                                                <?php endif; ?>

                                                <?php if (!empty($tp_button_title)): ?>
                                                    <div class="tp-about-campus-btn">
                                                        <a class="tp-btn" <?php echo tp_implode_html_attributes($links_attrs); ?>><?php echo tp_kses($tp_button_title); ?></a>
                                                    </div>
                                                <?php endif; ?>
                                            </div>
                                        </div>
                                    </div>

                                <?php endforeach; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </section>
        <?php endif;
    }
}

$widgets_manager->register(new TP_About_Tab());