<?php
namespace TPCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Utils;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Repeater;

if (!defined('ABSPATH'))
    exit; // Exit if accessed directly

/**
 * Tp Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TP_Brand_Slider extends Widget_Base
{

    use \TPCore\Widgets\TP_Style_Trait;
    use \TPCore\Widgets\TP_Animation_Trait;

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'tp-brand-slider';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Brand Slider', 'tpcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'tp-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['tpcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['tpcore'];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */

    protected function register_controls()
    {
        $this->register_controls_section();
        $this->style_tab_content();
    }

    protected function register_controls_section()
    {

        // layout Panel
        $this->tp_design_layout('Select Layout', 3);

        // brand section
        $this->start_controls_section(
            'tp_brand_section',
            [
                'label' => __('Brand Item', 'tpcore'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'tp_brand_sec_title',
            [
                'label' => esc_html__('Section Title', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Default Text', 'tpcore'),
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'tp_brand_image',
            [
                'type' => Controls_Manager::MEDIA,
                'label' => __('Image', 'tpcore'),
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $this->add_control(
            'tp_brand_slides',
            [
                'show_label' => false,
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => esc_html__('Brand Item', 'tpcore'),
                'default' => [
                    [
                        'tp_brand_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                    [
                        'tp_brand_image' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                    ],
                ]
            ]
        );

        $this->add_group_control(
            Group_Control_Image_Size::get_type(),
            [
                'name' => 'thumbnail',
                'default' => 'medium_large',
                'separator' => 'before',
                'exclude' => [
                    'custom'
                ]
            ]
        );

        $this->end_controls_section();

        // animation
        $this->tp_creative_animation(['layout-3', 'layout-6']);

    }

    protected function style_tab_content()
    {
        $this->tp_section_style_controls('section', 'Section - Style', '.tp-el-section');
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();
        ?>

        <?php if ($settings['tp_design_style'] == 'layout-2'): ?>

            <div class="tp-brands-quran-wrap">
                <div class="swiper tp-brands-quran-slider-active">
                    <div class="swiper-wrapper align-items-center">
                        <?php foreach ($settings['tp_brand_slides'] as $key => $item):
                            if (!empty($item['tp_brand_image']['url'])) {
                                $tp_brand_image_url = !empty($item['tp_brand_image']['id']) ? wp_get_attachment_image_url($item['tp_brand_image']['id'], $settings['thumbnail_size']) : $item['tp_brand_image']['url'];
                                $tp_brand_image_alt = get_post_meta($item["tp_brand_image"]["id"], "_wp_attachment_image_alt", true);
                            }
                            ?>
                            <div class="swiper-slide">
                                <div>
                                    <img src="<?php echo esc_url($tp_brand_image_url); ?>"
                                        alt="<?php echo esc_attr($tp_brand_image_alt); ?>">
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>

        <?php elseif ($settings['tp_design_style'] == 'layout-3'):
            $animation = $this->tp_animation_show($settings);

            $attrs = [
                'class' => "tp-brands-chef-wrap p-relative tp-el-section " . $animation['animation'] . ' ' . $animation['duration'] . ' ' . $animation['delay'],
            ];
            $title = $settings['tp_brand_sec_title'];
            ?>

            <div <?php echo tp_implode_html_attributes($attrs); ?>>

                <?php if (!empty($title)): ?>
                    <div class="tp-brands-chef-title">
                        <span>
                            <?php echo esc_html($title); ?>
                        </span>
                    </div>
                <?php endif; ?>

                <div class="swiper tp-brands-chef-slider-active">
                    <div class="swiper-wrapper slide-transtion align-items-center">

                        <?php foreach ($settings['tp_brand_slides'] as $key => $item):
                            if (!empty($item['tp_brand_image']['url'])) {
                                $tp_brand_image_url = !empty($item['tp_brand_image']['id']) ? wp_get_attachment_image_url($item['tp_brand_image']['id'], $settings['thumbnail_size']) : $item['tp_brand_image']['url'];
                                $tp_brand_image_alt = get_post_meta($item["tp_brand_image"]["id"], "_wp_attachment_image_alt", true);
                            }
                            ?>
                            <div class="swiper-slide">
                                <div>
                                    <img src="<?php echo esc_url($tp_brand_image_url); ?>" alt="<?php echo esc_attr($tp_brand_image_alt); ?>">
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>

        <?php elseif ($settings['tp_design_style'] == 'layout-4'): ?>

            <div class="tp-brand-3-slider-wrapper">
                <div class="swiper-container tp-brand-3-slider-active fix">
                    <div class="swiper-wrapper brand-wrapper">

                        <?php foreach ($settings['tp_brand_slides'] as $key => $item):
                            if (!empty($item['tp_brand_image']['url'])) {
                                $tp_brand_image_url = !empty($item['tp_brand_image']['id']) ? wp_get_attachment_image_url($item['tp_brand_image']['id'], $settings['thumbnail_size']) : $item['tp_brand_image']['url'];
                                $tp_brand_image_alt = get_post_meta($item["tp_brand_image"]["id"], "_wp_attachment_image_alt", true);
                            }
                            ?>
                            <div class="swiper-slide">
                                <div class="tp-brand-item">
                                    <img src="<?php echo esc_url($tp_brand_image_url); ?>"
                                        alt="<?php echo esc_attr($tp_brand_image_alt); ?>">
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                </div>
            </div>

        <?php else: ?>

            <div class="row">
                <div class="col-12">
                    <div class="swiper tp-brand-5-active">
                        <div class="swiper-wrapper">

                            <?php foreach ($settings['tp_brand_slides'] as $key => $item):
                                if (!empty($item['tp_brand_image']['url'])) {
                                    $tp_brand_image_url = !empty($item['tp_brand_image']['id']) ? wp_get_attachment_image_url($item['tp_brand_image']['id'], $settings['thumbnail_size']) : $item['tp_brand_image']['url'];
                                    $tp_brand_image_alt = get_post_meta($item["tp_brand_image"]["id"], "_wp_attachment_image_alt", true);
                                }
                                ?>
                                <div class="swiper-slide">
                                    <div class="tp-brand-5-item">
                                        <img src="<?php echo esc_url($tp_brand_image_url); ?>"
                                            alt="<?php echo esc_attr($tp_brand_image_alt); ?>">
                                    </div>
                                </div>
                            <?php endforeach; ?>

                        </div>
                    </div>
                </div>
            </div>


        <?php endif;
    }
}

$widgets_manager->register(new TP_Brand_Slider());