<?php
namespace TPCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;


if (!defined('ABSPATH'))
    exit; // Exit if accessed directly

/**
 * Tp Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TP_Image_Box extends Widget_Base
{

    use TP_Style_Trait, TP_Icon_Trait, TP_Offcanvas_Trait, TP_Menu_Trait, TP_Animation_Trait;

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'tp-image-box';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Image Box', 'tpcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'tp-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['tpcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['tpcore'];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */

    protected function register_controls()
    {
        $this->register_controls_section();
        $this->style_tab_content();
    }

    protected function register_controls_section()
    {
        $this->tp_design_layout('Layout', 2);

        $this->start_controls_section(
            'tp_image_box_section',
            [
                'label' => esc_html__('Image Box Contents', 'tpcore'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'tp_image_box_image',
            [
                'label' => esc_html__('Image', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->add_control(
            'tp_button_text',
            [
                'label' => esc_html__('Button Text', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Button Text', 'tpcore'),
                'placeholder' => esc_html__('Your Button Text', 'tpcore'),
                'label_block' => true,
                'condition' => [
                    'tp_design_style!' => ['layout-1'],
                ]
            ]
        );

        $this->add_control(
            'tp_image_box_title',
            [
                'label' => esc_html__('Title', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Branding', 'tpcore'),
                'placeholder' => esc_html__('Your Text', 'tpcore'),
                'label_block' => true,
            ]
        );

        tp_render_links_controls($this, 'image_box');

        $this->add_control(
            'tp_image_box_desc',
            [
                'label' => esc_html__('Description', 'tpcore'),
                'type' => Controls_Manager::TEXTAREA,
                'default' => esc_html__('Branding is one of the most important ingredients for the success of any business.', 'tpcore'),
                'placeholder' => esc_html__('Your Text', 'tpcore'),
                'label_block' => true,
                'condition' => [
                    'tp_design_style!' => ['layout-3'],
                ]
            ]
        );

        $this->end_controls_section();

        // animation
        $this->tp_creative_animation();

    }

    // style_tab_content
    protected function style_tab_content()
    {
        $this->tp_link_controls_style('', 'btn_1_style', 'Button', '.tp-el-button');
        $this->tp_basic_style_controls('image_box_title', 'Title', '.tp-el-title');
        $this->tp_basic_style_controls('image_box_content', 'Description', '.tp-el-content');
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $animation = $this->tp_animation_show($settings);
        ?>

        <?php if ($settings['tp_design_style'] == 'layout-2'):
            $img = tp_get_img($settings, 'tp_image_box_image', 'full', false);

            $attrs = tp_get_repeater_links_attr($settings, 'image_box');
            extract($attrs);

            $links_attrs = [
                'href' => $link,
                'target' => $target,
                'rel' => $rel,
            ];

            ?>

            <div class="tp-classes-yoga-wrap tp-classes-yoga-3-wrap mb-30 <?php echo esc_attr($animation['animation']); ?>" <?php echo $animation['duration'] . ' ' . $animation['delay']; ?>>
                <div class="tp-classes-yoga-thumb fix p-relative">
                    <?php if (!empty($img['tp_image_box_image'])): ?>
                        <img class="w-100" src="<?php echo esc_url($img['tp_image_box_image']); ?>"
                            alt="<?php echo esc_attr($img['tp_image_box_image_alt']); ?>">
                    <?php endif; ?>

                    <a <?php echo tp_implode_html_attributes($links_attrs); ?> class="tp-classes-yoga-link tp-el-button">
                        <?php echo tp_kses($settings['tp_button_text']); ?>
                    </a>

                </div>
                <?php if (!empty($settings['tp_image_box_title'])): ?>
                    <div class="tp-classes-yoga-content text-center">
                        <h4 class="tp-classes-yoga-title mb-0 tp-el-title">
                            <a <?php echo tp_implode_html_attributes($links_attrs); ?>>
                                <?php echo tp_kses($settings['tp_image_box_title']); ?>
                            </a>
                        </h4>
                        <?php if (!empty($settings['tp_image_box_desc'])): ?>
                            <span class="tp-classes-yoga-tag tp-el-content">
                                <?php echo tp_kses($settings['tp_image_box_desc']); ?>
                            </span>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>

        <?php else:


            $img = tp_get_img($settings, 'tp_image_box_image', 'full', false);

            $attrs = tp_get_repeater_links_attr($settings, 'image_box');
            extract($attrs);

            $links_attrs = [
                'href' => $link,
                'target' => $target,
                'rel' => $rel,
            ];
            ?>

            <div class="tp-service-chef-wrap text-center <?php echo esc_attr($animation['animation']); ?>" <?php echo $animation['duration'] . ' ' . $animation['delay']; ?>>
                <span class="tp-service-chef-icon mb-30 d-inline-block">
                    <img src="<?php echo esc_url($img['tp_image_box_image']); ?>" alt="">
                </span>

                <?php if (!empty($settings['tp_image_box_title'])): ?>
                    <h3 class="tp-service-chef-title mb-10">
                        <a <?php echo tp_implode_html_attributes($links_attrs); ?>>
                            <?php echo tp_kses($settings['tp_image_box_title']); ?>
                        </a>
                    </h3>
                <?php endif; ?>

                <?php if (!empty($settings['tp_image_box_desc'])): ?>
                    <p class="tp-el-content">
                        <?php echo tp_kses($settings['tp_image_box_desc']); ?>
                    </p>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    <?php
    }
}

$widgets_manager->register(new TP_Image_Box());