<?php

namespace TPCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Utils;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Group_Control_Typography;



if (!defined('ABSPATH'))
    exit; // Exit if accessed directly

/**
 * Tp Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TP_Header_04 extends Widget_Base
{

    use TP_Style_Trait, TP_Icon_Trait, TP_Offcanvas_Trait, TP_Menu_Trait;

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'tp-header-04';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Header 4', 'tpcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'tp-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['tpcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['tpcore'];
    }


    /**
     * Menu index.
     *
     * @access protected
     * @var $nav_menu_index
     */
    protected $nav_menu_index = 1;

    /**
     * Retrieve the menu index.
     *
     * Used to get index of nav menu.
     *
     * @since 1.3.0
     * @access protected
     *
     * @return string nav index.
     */
    protected function get_nav_menu_index()
    {
        return $this->nav_menu_index++;
    }

    /**
     * Retrieve the list of available menus.
     *
     * Used to get the list of available menus.
     *
     * @since 1.3.0
     * @access private
     *
     * @return array get WordPress menus list.
     */
    private function get_available_menus()
    {

        $menus = wp_get_nav_menus();

        $options = [];

        foreach ($menus as $menu) {
            $options[$menu->slug] = $menu->name;
        }

        return $options;
    }


    private function get_tutor_lms_dashboard_pages()
    {
        $pages = [];
        $tutorPages = tutor_utils()->tutor_dashboard_nav_ui_items();

        foreach ($tutorPages as $key => $value) {
            if (array_key_exists('type', $value)) {
                continue;
            } else {
                $pages[$key] = $value['title'];
            }
        }

        return $pages;

    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */

    protected function register_controls()
    {
        $this->register_controls_section();
        $this->style_tab_content();
    }

    protected function register_controls_section()
    {

        // layout Panel
        $this->start_controls_section(
            'tp_header_top',
            [
                'label' => esc_html__('Header Info', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_header_sticky',
            [
                'label' => esc_html__('Enable Sticky', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'tpcore'),
                'label_off' => esc_html__('Hide', 'tpcore'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'tp_header_right_switch',
            [
                'label' => esc_html__('Header Right On/Off', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'tpcore'),
                'label_off' => esc_html__('Hide', 'tpcore'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        // header_Login_switch
        $this->add_control(
            'tp_header_login_switch',
            [
                'label' => esc_html__('Enable Login?', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'tpcore'),
                'label_off' => esc_html__('Hide', 'tpcore'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'tp_header_cart_switch',
            [
                'label' => esc_html__('Enable Cart?', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'tpcore'),
                'label_off' => esc_html__('Hide', 'tpcore'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        // cart icon controls
        $this->tp_single_icon_control('header_cart', 'tp_header_cart_switch', 'yes');

        $this->end_controls_section();


        if (function_exists('tutor')) {
            $this->start_controls_section(
                'tp_header_user_sec',
                [
                    'label' => esc_html__('User Menu List', 'tpcore'),
                    'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
                ]
            );


            $this->add_control(
                'tp_header_user_list',
                [
                    'label' => esc_html__('User Menu List', 'tpcore'),
                    'type' => \Elementor\Controls_Manager::SELECT2,
                    'multiple' => true,
                    'label_block' => true,
                    'options' => $this->get_tutor_lms_dashboard_pages(),
                ]
            );

            $this->end_controls_section();
        }

        // _tp_image
        $this->start_controls_section(
            '_tp_image',
            [
                'label' => esc_html__('Site Logo', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_logo_black',
            [
                'label' => esc_html__('Choose Logo Black', 'tpcore'),
                'type' => \Elementor\Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );
        $this->add_control(
            'tp_logo_white',
            [
                'label' => esc_html__('Choose Logo White', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

		$this->add_control(
			'tp_logo_width',
			[
				'type' => \Elementor\Controls_Manager::NUMBER,
				'label' => esc_html__( 'Logo Width', 'tpcore' ),
				'description' => esc_html__( 'This number will count by "PX" and maximum number is 1000', 'tpcore' ),
				'placeholder' => '0',
				'min' => 10,
				'max' => 1000,
				'step' => 1,
				'default' => 125,
                'selectors' => [
                    '{{WRAPPER}} .tp-el-logo img' => 'width: {{VALUE}}px;',
                ],
			]
		);

        $this->end_controls_section();

        // menu
        $this->start_controls_section(
            'section_menu',
            [
                'label' => __('Menu', 'tpcore'),
            ]
        );

        $menus = $this->get_available_menus();

        if (!empty($menus)) {
            $this->add_control(
                'menu',
                [
                    'label' => __('Menu', 'tpcore'),
                    'type' => Controls_Manager::SELECT,
                    'options' => $menus,
                    'default' => array_keys($menus)[0],
                    'save_default' => true,
                    /* translators: %s Nav menu URL */
                    'description' => sprintf(__('Go to the <a href="%s" target="_blank">Menus screen</a> to manage your menus.', 'tpcore'), admin_url('nav-menus.php')),
                ]
            );
        } else {
            $this->add_control(
                'menu',
                [
                    'type' => Controls_Manager::RAW_HTML,
                    /* translators: %s Nav menu URL */
                    'raw' => sprintf(__('<strong>There are no menus in your site.</strong><br>Go to the <a href="%s" target="_blank">Menus screen</a> to create one.', 'tpcore'), admin_url('nav-menus.php?action=edit&menu=0')),
                    'content_classes' => 'elementor-panel-alert elementor-panel-alert-info',
                ]
            );
        }

        $this->add_control(
            'menu_last_item',
            [
                'label' => __('Last Menu Item', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'none' => __('Default', 'tpcore'),
                    'cta' => __('Button', 'tpcore'),
                ],
                'default' => 'none',
                'condition' => [
                    'layout!' => 'expandible',
                ],
            ]
        );

        $this->end_controls_section();

        // offcanvas
        $this->tp_offcanvas_controls('offcanvas', 'Offcanvas');
    }

    // style_tab_content
    protected function style_tab_content()
    {
        $this->start_controls_section(
            'tp_menu_style_section',
            [
                'label' => esc_html__('Mega Menu', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tp_menu_title_typography',
                'label' => esc_html__('Typography', 'tpcore'),
                'selector' => '{{WRAPPER}} .tp-megamenu-home-item a',
            ]
        );

        $this->start_controls_tabs(
            'style_tabs'
        );

        $this->start_controls_tab(
            'style_normal_tab',
            [
                'label' => esc_html__('Normal', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_menu_title_color',
            [
                'label' => esc_html__('Menu Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-megamenu-home-item a' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'style_hover_tab',
            [
                'label' => esc_html__('Hover', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_menu_title_hvr_color',
            [
                'label' => esc_html__('Menu Hover Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-megamenu-home-item:hover a' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();


        // main Menu style
        $this->start_controls_section(
            'tp_main_menu_style_section',
            [
                'label' => esc_html__('Main Menu', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tp_main_menu_title_typography',
                'label' => esc_html__('Typography', 'tpcore'),
                'selector' => '{{WRAPPER}} .main-menu > nav > ul > li > a',
            ]
        );

        $this->start_controls_tabs(
            'main_style_tabs'
        );

        $this->start_controls_tab(
            'main_style_normal_tab',
            [
                'label' => esc_html__('Normal', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_main_menu_title_color',
            [
                'label' => esc_html__('Menu Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .main-menu > nav > ul > li > a,
                    {{WRAPPER}} .main-menu > nav > ul > li.has-dropdown > a::after' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'main_style_hover_tab',
            [
                'label' => esc_html__('Hover', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_main_menu_title_hvr_color',
            [
                'label' => esc_html__('Menu Hover Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .main-menu > nav > ul > li:hover > a,
                    {{WRAPPER}} .main-menu > nav > ul > li:hover > a:after' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();


        // Link Menu style
        $this->start_controls_section(
            'tp_mobile_link_menu_style_section',
            [
                'label' => esc_html__('Link Menu', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tp_mobile_link_menu_title_typography',
                'label' => esc_html__('Typography', 'tpcore'),
                'selector' => '{{WRAPPER}} .tp-megamenu-small-content .tp-megamenu-list a,
                {{WRAPPER}} .main-menu > nav > ul > li > .tp-submenu li > a,
                {{WRAPPER}} .tp-megamenu-fullwidth-list ul li a',
            ]
        );

        $this->start_controls_tabs(
            'mobile_link_style_tabs'
        );

        $this->start_controls_tab(
            'mobile_link_style_normal_tab',
            [
                'label' => esc_html__('Normal', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_mobile_link_menu_title_color',
            [
                'label' => esc_html__('Menu Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-megamenu-small-content .tp-megamenu-list a,
                {{WRAPPER}} .main-menu > nav > ul > li > .tp-submenu li > a,
                {{WRAPPER}} .tp-megamenu-fullwidth-list ul li a' => 'color: {{VALUE}} !important',
                ],
            ]
        );

        $this->add_control(
            'tp_mobile_link_menu_toggle_color',
            [
                'label' => esc_html__('Menu Bar Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-megamenu-small-content .tp-megamenu-list a::before,
                    {{WRAPPER}} .main-menu > nav > ul > li > .tp-submenu li > a::before,
                    {{WRAPPER}} .tp-megamenu-fullwidth-list ul li a:before' => 'background-color: {{VALUE}} !important',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'mobile_link_style_hover_tab',
            [
                'label' => esc_html__('Hover', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_mobile_link_menu_title_hvr_color',
            [
                'label' => esc_html__('Menu Hover Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-megamenu-small-content .tp-megamenu-list a:hover,
                {{WRAPPER}} .main-menu > nav > ul > li > .tp-submenu li > a:hover,
                {{WRAPPER}} .tp-megamenu-fullwidth-list ul li a:hover' => 'color: {{VALUE}} !important',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

        // Mobile Menu style
        $this->start_controls_section(
            'tp_mobile_main_menu_style_section',
            [
                'label' => esc_html__('Mobile Menu', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tp_mobile_main_menu_title_typography',
                'label' => esc_html__('Typography', 'tpcore'),
                'selector' => '{{WRAPPER}} .tp-main-menu-mobile ul li > a',
            ]
        );

        $this->start_controls_tabs(
            'mobile_main_style_tabs'
        );

        $this->start_controls_tab(
            'mobile_main_style_normal_tab',
            [
                'label' => esc_html__('Normal', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_mobile_main_menu_title_color',
            [
                'label' => esc_html__('Menu Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-main-menu-mobile ul li > a' => 'color: {{VALUE}} !important',
                ],
            ]
        );

        $this->add_control(
            'tp_mobile_main_menu_toggle_color',
            [
                'label' => esc_html__('Menu Toggle Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-main-menu-mobile ul li.has-dropdown > a .dropdown-toggle-btn::after' => 'background-color: {{VALUE}} !important',
                    '{{WRAPPER}} .tp-main-menu-mobile ul li.has-dropdown > a .dropdown-toggle-btn::before' => 'background-color: {{VALUE}} !important',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'mobile_main_style_hover_tab',
            [
                'label' => esc_html__('Hover', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_mobile_main_menu_title_hvr_color',
            [
                'label' => esc_html__('Menu Hover Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-main-menu-mobile ul li > a.expanded, 
                    {{WRAPPER}} .tp-main-menu-mobile ul li:hover > a' => 'color: {{VALUE}} !important',
                ],
            ]
        );

        $this->add_control(
            'tp_mobile_main_menu_toggle_hvr_color',
            [
                'label' => esc_html__('Menu Toggle Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-main-menu-mobile ul li.has-dropdown > a.expanded .dropdown-toggle-btn::after,
                    {{WRAPPER}} .offcanvas__2 .tp-main-menu-mobile ul li:hover > a .dropdown-toggle-btn::before,
                    {{WRAPPER}} .offcanvas__2 .tp-main-menu-mobile ul li:hover > a .dropdown-toggle-btn::after' => 'background-color: {{VALUE}} !important',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

        // Close Btn style
        $this->start_controls_section(
            'tp_mobile_close_btn_style_section',
            [
                'label' => esc_html__('Close Button', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tp_mobile_close_btn_typography',
                'label' => esc_html__('Typography', 'tpcore'),
                'selector' => '{{WRAPPER}} .offcanvas__close-btn',
            ]
        );

        $this->start_controls_tabs(
            'close_button_style_tabs'
        );

        $this->start_controls_tab(
            'close_button_style_normal_tab',
            [
                'label' => esc_html__('Normal', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_close_button_menu_title_color',
            [
                'label' => esc_html__('Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .offcanvas__close-btn' => 'color: {{VALUE}} !important',
                ],
            ]
        );

        $this->add_control(
            'tp_close_button_menu_toggle_color',
            [
                'label' => esc_html__('Bg Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .offcanvas__close-btn' => 'background-color: {{VALUE}} !important',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'close_button_style_hover_tab',
            [
                'label' => esc_html__('Hover', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_close_button_menu_title_hvr_color',
            [
                'label' => esc_html__('Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .offcanvas__close-btn:hover' => 'color: {{VALUE}} !important',
                ],
            ]
        );

        $this->add_control(
            'tp_close_button_menu_title_hvrbg_color',
            [
                'label' => esc_html__('Bg Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .offcanvas__close-btn:hover' => 'background-color: {{VALUE}} !important',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();


        if (!empty($settings['menu'])):
            $menu = $settings['menu'];
        else:
            $menu = '';
        endif;


        $menus = $this->get_available_menus();
        require_once get_parent_theme_file_path() . '/inc/class-navwalker.php';

        $args = [
            'echo' => false,
            'menu' => $menu,
            'menu_class' => 'tp-nav-menu ',
            'menu_id' => 'menu-' . $this->get_nav_menu_index() . '-' . $this->get_id(),
            'fallback_cb' => 'Acadia_Navwalker_Class::fallback',
            'container' => '',
            'walker' => new Acadia_Navwalker_Class,
        ];

        $menu_html = wp_nav_menu($args);



        $logo_image_size = tp_get_img_size($settings, 'tp_image_size');


        if (!empty($settings['tp_logo_black']['url'])) {
            $logo_black = !empty($settings['tp_logo_black']['id']) ? wp_get_attachment_image_url($settings['tp_logo_black']['id'], $logo_image_size, true) : $settings['tp_logo_black']['url'];
            $logo_black_alt = get_post_meta($settings["tp_logo_black"]["id"], "_wp_attachment_image_alt", true);
        }

        if (!empty($settings['tp_logo_white']['url'])) {
            $tp_logo_white = !empty($settings['tp_logo_white']['id']) ? wp_get_attachment_image_url($settings['tp_logo_white']['id'], $logo_image_size, true) : $settings['tp_logo_white']['url'];
            $tp_logo_white_alt = get_post_meta($settings["tp_logo_white"]["id"], "_wp_attachment_image_alt", true);
        }

        $is_sticky = $settings['tp_header_sticky'] == 'yes' ? 'header-sticky' : 'no-sticky';

        $edit_class = tp_is_elementor_edit_mode() ? 'acadia-elementor-header-edit-mode ' : '';

        ?>
        <!-- header-area-start -->
        <header class="header-area">
            <div id="<?php echo esc_attr($is_sticky); ?>" class="tp-header-5 <?php echo esc_attr($edit_class); ?>">
                <div class="container-fluid">
                    <div class="row align-items-center">
                        <div class="col-xl-2 col-lg-6 col-6">
                            
                            <div class="tp-header-5-logo tp-el-logo">
                                <a href="<?php print esc_url(home_url('/')); ?>">
                                    <?php if(!empty($tp_logo_white)) : ?>
                                    <img class="logo-1" src="<?php echo esc_url($tp_logo_white); ?>"
                                        alt="<?php echo esc_attr($tp_logo_white_alt); ?>">
                                    <?php endif; ?>

                                    <?php if(!empty($logo_black)) : ?>
                                    <img class="logo-2" src="<?php echo esc_url($logo_black); ?>"
                                        alt="<?php echo esc_attr($logo_black_alt); ?>">
                                    <?php endif; ?>
                                </a>
                            </div>
                        </div>
                        <div class="col-xl-8 d-none d-xl-block">
                            <div class="main-menu">
                                <nav class="tp-main-menu-content">
                                    <?php echo $menu_html; ?>
                                </nav>
                            </div>
                        </div>

                        <?php if (!empty($settings['tp_header_right_switch'])): ?>
                            <div class="col-xl-2 col-lg-6 col-6">
                                <div class="tp-header-5-contact d-flex align-items-center justify-content-end">

                                    <?php
                                        if (!empty($settings['tp_header_login_switch']) && function_exists('tutor')):
                                            if (is_user_logged_in()):
                                                $user_id = get_current_user_id();
                                                $user = get_user_by('ID', $user_id);

                                                $cover_placeholder = tutor()->url . 'assets/images/cover-photo.jpg';
                                                $cover_photo_src = $cover_placeholder;
                                                $cover_photo_id = get_user_meta($user->ID, '_tutor_cover_photo', true);
                                                if ($cover_photo_id) {
                                                    $url = wp_get_attachment_image_url($cover_photo_id, 'full');
                                                    !empty($url) ? $cover_photo_src = $url : 0;
                                                }
                                                $profile_placeholder = apply_filters('tutor_login_default_avatar', tutor()->url . 'assets/images/profile-photo.png');
                                                $profile_photo_src = $profile_placeholder;
                                                $profile_photo_id = get_user_meta($user->ID, '_tutor_profile_photo', true);
                                                if ($profile_photo_id) {
                                                    $url = wp_get_attachment_image_url($profile_photo_id, 'full');
                                                    !empty($url) ? $profile_photo_src = $url : 0;
                                                }
                                                $dashboard_pages = tutor_utils()->tutor_dashboard_nav_ui_items();
                                                ?>
                                            <div class="tp-header-5-login tp-header-user-hover">
                                                <button>
                                                    <span class="tp-header_cart-svg-icon ">
                                                        <svg width="17" height="19" viewBox="0 0 17 19" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                            <path d="M8.40947 9.14901C10.6598 9.14901 12.484 7.32479 12.484 5.07451C12.484 2.82422 10.6598 1 8.40947 1C6.15918 1 4.33496 2.82422 4.33496 5.07451C4.33496 7.32479 6.15918 9.14901 8.40947 9.14901Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            <path d="M15.4102 17.298C15.4102 14.1444 12.2728 11.5937 8.41016 11.5937C4.54752 11.5937 1.41016 14.1444 1.41016 17.298" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path>
                                                        </svg>
                                                    </span>
                                                </button>
                                    
                                                <div class="tp-header-user-box">
                                                    <div class="tp-header-user-content">
                                                        <div class="tp-header-user-profile d-flex align-items-center">
                                                            <div class="tp-header-user-profile-thumb">
                                                                <img src="<?php echo esc_url($profile_photo_src); ?>"
                                                                    alt="<?php echo esc_attr($user->display_name); ?>">
                                                            </div>
                                                            <div class="tp-header-user-profile-content">
                                                                <h4><?php echo esc_html($user->display_name); ?></h4>
                                                                <span><?php echo esc_html(ucwords($user->roles[0])); ?></span>
                                                            </div>
                                                        </div>
                                                        <div class="tp-header-user-list">
                                                            <ul>
                                                                <?php
                                                                $dashboard_pages = tutor_utils()->tutor_dashboard_nav_ui_items();
                                                                $disable = !get_tutor_option('enable_course_review');
                                                                foreach ($dashboard_pages as $dashboard_key => $dashboard_page) {
                                                                    foreach ($settings['tp_header_user_list'] as $key => $value) {
                                                                        if ($dashboard_key == $value) {
                                                                            if ($disable && 'reviews' === $dashboard_key) {
                                                                                continue;
                                                                            }

                                                                            $menu_title = $dashboard_page;
                                                                            $menu_link = tutor_utils()->get_tutor_dashboard_page_permalink($dashboard_key);
                                                                            $separator = false;
                                                                            $menu_icon = '';

                                                                            if (is_array($dashboard_page)) {
                                                                                $menu_title = tutor_utils()->array_get('title', $dashboard_page);
                                                                                $menu_icon_name = tutor_utils()->array_get('icon', $dashboard_page, (isset($dashboard_page['icon']) ? $dashboard_page['icon'] : ''));

                                                                                if (in_array($dashboard_key, get_lsm_dashboard_menu_keys())) {
                                                                                    $menu_icon = get_lsm_dashboard_menu_icon($dashboard_key);
                                                                                } else {
                                                                                    if ($menu_icon_name) {
                                                                                        $menu_icon = "<span class='{$menu_icon_name} tutor-dashboard-menu-item-icon'></span>";
                                                                                    }
                                                                                }

                                                                                if (isset($dashboard_page['url'])) {
                                                                                    $menu_link = $dashboard_page['url'];
                                                                                }
                                                                                if (isset($dashboard_page['type']) && $dashboard_page['type'] == 'separator') {
                                                                                    $separator = true;
                                                                                }
                                                                            }
                                                                            if ($separator) {
                                                                                // Optionally, add code to handle separators
                                                                            } else {
                                                                                $li_class = "acadia-dashboard-menu-{$dashboard_key}";
                                                                                if ('index' === $dashboard_key) {
                                                                                    $dashboard_key = '';
                                                                                }

                                                                                $data_no_instant = 'logout' == $dashboard_key ? 'data-no-instant' : '';
                                                                                $menu_link = apply_filters('tutor_dashboard_menu_link', $menu_link, $menu_title);
                                                                                ?>
                                                                                <?php if ($dashboard_key == 'settings'): ?>
                                                                                    <li class="hr-border"></li>
                                                                                <?php endif; ?>
                                                                                <li class='acadia-dashboard-menu-item <?php echo esc_attr($li_class); ?>'>
                                                                                    <a <?php echo esc_html($data_no_instant); ?> href="<?php echo esc_url($menu_link); ?>">
                                                                                        <span>
                                                                                            <?php echo tp_kses($menu_icon); ?>
                                                                                        </span>
                                                                                        <?php echo esc_html($menu_title); ?>
                                                                                    </a>
                                                                                </li>
                                                                                <?php
                                                                            }
                                                                        }
                                                                    }
                                                                }
                                                                ?>
                                                            </ul>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        <?php else: ?>
                                            <!-- User is not logged in -->
                                            <div class="tp-header-5-login tp-header-user-hover">
                                                <a href="<?php echo esc_url(tutor_utils()->tutor_dashboard_url()); ?>">
                                                    <button>
                                                        <span class="tp-header_cart-svg-icon ">
                                                            <svg width="17" height="19" viewBox="0 0 17 19" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                                <path d="M8.40947 9.14901C10.6598 9.14901 12.484 7.32479 12.484 5.07451C12.484 2.82422 10.6598 1 8.40947 1C6.15918 1 4.33496 2.82422 4.33496 5.07451C4.33496 7.32479 6.15918 9.14901 8.40947 9.14901Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path>
                                                                <path d="M15.4102 17.298C15.4102 14.1444 12.2728 11.5937 8.41016 11.5937C4.54752 11.5937 1.41016 14.1444 1.41016 17.298" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path>
                                                            </svg>
                                                        </span>
                                                    </button>
                                                </a>
                                            </div>
                                        <?php endif; ?>
                                    <?php endif; ?>

                                    <?php if ($settings['tp_header_cart_switch'] == 'yes' && class_exists('WooCommerce')): ?>
                                        <div class="tp-header-5-cart ml-35">
                                            <button class="cartmini-open-btn">
                                                <span class="d-none">
                                                    <?php echo esc_html__('Cart', 'tpcore'); ?>
                                                </span>
                                                <?php tp_render_signle_icon_html($settings, 'header_cart'); ?>

                                                <?php
                                                $cart_count = !is_null(WC()->cart) ? WC()->cart->get_cart_contents_count() : 0;
                                                ?>
                                                <i><?php echo esc_html($cart_count); ?></i>
                                            </button>
                                        </div>
                                    <?php endif; ?>

                                    <div class="tp-header-5-sidebar offcanvas-btns ml-35">
                                        <button class="offcanvas-open-btn"><span></span></button>
                                    </div>
                                </div>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </header>
        <!-- header-area-end -->

        <?php if (class_exists('WooCommerce') && $settings['tp_header_cart_switch'] == 'yes'): ?>
            <?php print acadia_shopping_cart(); ?>
            <div class="body-overlay"></div>
        <?php endif; ?>


        <?php include(TPCORE_ELEMENTS_PATH . '/header/header-offcanvas.php'); ?>


        <?php

    }
}

$widgets_manager->register(new TP_Header_04());
