<?php

namespace TPCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Group_Control_Typography;



if (!defined('ABSPATH'))
    exit; // Exit if accessed directly

/**
 * Tp Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TP_Header_lang_academy extends Widget_Base
{

    use TP_Style_Trait, TP_Icon_Trait, TP_Offcanvas_Trait, TP_Menu_Trait, TP_Column_Trait, TP_Query_Trait;

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'tp-header-lang-academy';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Header Lang Academy', 'tpcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'tp-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['tpcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['tpcore'];
    }

    /**
     * Menu index.
     *
     * @access protected
     * @var $nav_menu_index
     */
    protected $nav_menu_index = 1;

    /**
     * Retrieve the menu index.
     *
     * Used to get index of nav menu.
     *
     * @since 1.3.0
     * @access protected
     *
     * @return string nav index.
     */
    protected function get_nav_menu_index()
    {
        return $this->nav_menu_index++;
    }

    /**
     * Retrieve the list of available menus.
     *
     * Used to get the list of available menus.
     *
     * @since 1.3.0
     * @access private
     *
     * @return array get WordPress menus list.
     */
    private function get_available_menus()
    {

        $menus = wp_get_nav_menus();

        $options = [];

        foreach ($menus as $menu) {
            $options[$menu->slug] = $menu->name;
        }

        return $options;
    }


    private function get_tutor_lms_dashboard_pages()
    {
        $pages = [];
        $tutorPages = tutor_utils()->tutor_dashboard_nav_ui_items();

        foreach ($tutorPages as $key => $value) {
            if (array_key_exists('type', $value)) {
                continue;
            } else {
                $pages[$key] = $value['title'];
            }
        }

        return $pages;

    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */

    protected function register_controls()
    {
        $this->register_controls_section();
        $this->style_tab_content();
    }

    protected function register_controls_section()
    {

        $this->start_controls_section(
            'tp_header_info',
            [
                'label' => esc_html__('Header Info', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_header_sticky',
            [
                'label' => esc_html__('Enable Sticky', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'tpcore'),
                'label_off' => esc_html__('Hide', 'tpcore'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->end_controls_section();

        // header top
        $this->start_controls_section(
            'tp_header_top',
            [
                'label' => esc_html__('Header Info', 'tpcore'),
            ]
        );

        $this->add_control(
            'top_contact_info',
            [
                'label' => esc_html__('Contact Info', 'tpcore'),
                'type' => Controls_Manager::TEXTAREA,
                'rows' => 10,
                'default' => esc_html__('Contact Code here', 'tpcore'),
                'description' => esc_html__('Enter your HTML code', 'tpcore'),
            ]
        );

        $this->add_control(
            'top_discount_text',
            [
                'label' => esc_html__('Discount Info', 'tpcore'),
                'type' => Controls_Manager::TEXTAREA,
                'rows' => 5,
                'default' => esc_html__('intro price ! get online courses @25% Offer', 'tpcore'),
            ]
        );

        $this->add_control(
            'top_lang_switch',
            [
                'label' => esc_html__('Enable Language Switch', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'tpcore'),
                'label_off' => esc_html__('Hide', 'tpcore'),
                'return_value' => 'yes',
                'default' => 'yes',
            ]
        );

        $this->add_control(
            'top_currency_text',
            [
                'label' => esc_html__('Currency Text', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Currency', 'tpcore'),
            ]
        );


        $this->end_controls_section();

        // Header Right
        $this->start_controls_section(
            'tp_header_right',
            [
                'label' => esc_html__('Header Right', 'tpcore'),
            ]
        );

        // header_cart_switch
        $this->add_control(
            'tp_header_cart_switch',
            [
                'label' => esc_html__('Enable Cart?', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'tpcore'),
                'label_off' => esc_html__('Hide', 'tpcore'),
                'return_value' => 'yes',
                'default' => 'yes',
                'separator' => 'before',
            ]
        );

        // cart icon controls
        $this->tp_single_icon_control('header_cart', 'tp_header_cart_switch', 'yes');

        $this->add_control(
            'tp_header_login_switch',
            [
                'label' => esc_html__('Enable Login?', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'tpcore'),
                'label_off' => esc_html__('Hide', 'tpcore'),
                'return_value' => 'yes',
                'default' => 'yes',
                'separator' => 'before',
            ]
        );

        // header_login icon controls
        $this->tp_single_icon_control('header_login_icon', 'tp_header_login_switch', 'yes');

        $this->add_control(
            'tp_header_login_text',
            [
                'label' => esc_html__('Admin Text', 'tpcore'),
                'description' => esc_html__('admin'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('admin', 'tpcore'),
                'placeholder' => esc_html__('admin', 'tpcore'),
                'label_block' => true,
                'condition' => [
                    'tp_header_login_switch' => 'yes',
                ],
            ]
        );

        $this->end_controls_section();

        if (function_exists('tutor')) {
            $this->start_controls_section(
                'tp_header_user_sec',
                [
                    'label' => esc_html__('User Menu List', 'tpcore'),
                    'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
                ]
            );


            $this->add_control(
                'tp_header_user_list',
                [
                    'label' => esc_html__('User Menu List', 'tpcore'),
                    'type' => \Elementor\Controls_Manager::SELECT2,
                    'multiple' => true,
                    'label_block' => true,
                    'options' => $this->get_tutor_lms_dashboard_pages(),
                ]
            );

            $this->end_controls_section();
        }

        // header button controls
        $this->tp_link_render_controls('header_btn', 'Button');


        // _tp_image
        $this->start_controls_section(
            '_tp_image',
            [
                'label' => esc_html__('Site Logo', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_logo_black',
            [
                'label' => esc_html__('Choose Logo Black', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => \Elementor\Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->add_control(
            'tp_logo_width',
            [
                'type' => Controls_Manager::NUMBER,
                'label' => esc_html__('Logo Width', 'tpcore'),
                'description' => esc_html__('This number will count by "PX" and maximum number is 1000', 'tpcore'),
                'placeholder' => '0',
                'min' => 10,
                'max' => 1000,
                'step' => 1,
                'default' => 125,
                'selectors' => [
                    '{{WRAPPER}} .tp-el-logo img' => 'width: {{VALUE}}px;',
                ],
            ]
        );

        $this->end_controls_section();

        // menu
        $this->start_controls_section(
            'section_menu',
            [
                'label' => __('Menu', 'tpcore'),
            ]
        );

        $menus = $this->get_available_menus();

        if (!empty($menus)) {
            $this->add_control(
                'menu',
                [
                    'label' => __('Menu', 'tpcore'),
                    'type' => Controls_Manager::SELECT,
                    'options' => $menus,
                    'default' => array_keys($menus)[0],
                    'save_default' => true,
                    /* translators: %s Nav menu URL */
                    'description' => sprintf(__('Go to the <a href="%s" target="_blank">Menus screen</a> to manage your menus.', 'tpcore'), admin_url('nav-menus.php')),
                ]
            );
        } else {
            $this->add_control(
                'menu',
                [
                    'type' => Controls_Manager::RAW_HTML,
                    /* translators: %s Nav menu URL */
                    'raw' => sprintf(__('<strong>There are no menus in your site.</strong><br>Go to the <a href="%s" target="_blank">Menus screen</a> to create one.', 'tpcore'), admin_url('nav-menus.php?action=edit&menu=0')),
                    'content_classes' => 'elementor-panel-alert elementor-panel-alert-info',
                ]
            );
        }

        $this->add_control(
            'menu_last_item',
            [
                'label' => __('Last Menu Item', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'none' => __('Default', 'tpcore'),
                    'cta' => __('Button', 'tpcore'),
                ],
                'default' => 'none',
                'condition' => [
                    'layout!' => 'expandible',
                ],
            ]
        );

        $this->end_controls_section();

        // offcanvas
        $this->tp_offcanvas_controls('offcanvas', 'Offcanvas');
    }

    // style_tab_content
    protected function style_tab_content()
    {

        // menu
        $this->start_controls_section(
            'section_header_style',
            [
                'label' => __('Header Style', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'background',
                'types' => ['classic', 'gradient', 'video'],
                'exclude' => ['video'],
                'selector' => '{{WRAPPER}} .tp_el_header',
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'tp_box_shadow',
                'selector' => '{{WRAPPER}} .tp_el_header',
            ]
        );
        $this->add_control(
            'tp_header_padding',
            [
                'label' => esc_html__('Padding', 'tpcore'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .tp_el_header' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        // header sticky
        $this->add_control(
            '_content_header_sticky',
            [
                'type' => Controls_Manager::HEADING,
                'label' => __('Header Sticky', 'tpcore'),
                'separator' => 'before'
            ]
        );

        $this->add_group_control(
            \Elementor\Group_Control_Background::get_type(),
            [
                'name' => 'tp_sticky_background',
                'types' => ['classic', 'gradient', 'video'],
                'exclude' => ['video'],
                'selector' => '{{WRAPPER}} .tp_el_header.tp-header-sticky',
            ]
        );
        $this->add_group_control(
            \Elementor\Group_Control_Box_Shadow::get_type(),
            [
                'name' => 'tp_sticky_box_shadow',
                'selector' => '{{WRAPPER}} .tp_el_header.tp-header-sticky',
            ]
        );

        $this->end_controls_section();

        $this->tp_link_controls_style('', 'btn2_style', 'Button', '.tp-el-btn2');


        $this->start_controls_section(
            'tp_menu_style_section',
            [
                'label' => esc_html__('Mega Menu', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
                'separator' => 'before',
            ]
        );


        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tp_menuss_title_typography',
                'label' => esc_html__('Typography', 'tpcore'),
                'selector' => '{{WRAPPER}} .tp-megamenu-home-item a',
            ]
        );

        $this->start_controls_tabs(
            'style_tabs'
        );

        $this->start_controls_tab(
            'style_normal_tab',
            [
                'label' => esc_html__('Normal', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_menu_title_color',
            [
                'label' => esc_html__('Menu Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-megamenu-home-item a' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'style_hover_tab',
            [
                'label' => esc_html__('Hover', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_menu_title_hvr_color',
            [
                'label' => esc_html__('Menu Hover Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-megamenu-home-item:hover a' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();


        // main Menu style
        $this->start_controls_section(
            'tp_main_menu_style_section',
            [
                'label' => esc_html__('Main Menu', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tp_main_menu_title_typography',
                'label' => esc_html__('Typography', 'tpcore'),
                'selector' => '{{WRAPPER}} .main-menu > nav > ul > li > a',
            ]
        );

        $this->start_controls_tabs(
            'main_style_tabs'
        );

        $this->start_controls_tab(
            'main_style_normal_tab',
            [
                'label' => esc_html__('Normal', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_main_menu_title_color',
            [
                'label' => esc_html__('Menu Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .main-menu > nav > ul > li > a,
                    {{WRAPPER}} .main-menu > nav > ul > li.has-dropdown > a::after' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'main_style_hover_tab',
            [
                'label' => esc_html__('Hover', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_main_menu_title_hvr_color',
            [
                'label' => esc_html__('Menu Hover Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .main-menu > nav > ul > li:hover > a,
                    {{WRAPPER}} .main-menu > nav > ul > li:hover > a:after' => 'color: {{VALUE}}',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();


        // Link Menu style
        $this->start_controls_section(
            'tp_mobile_link_menu_style_section',
            [
                'label' => esc_html__('Link Menu', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tp_mobile_link_menu_title_typography',
                'label' => esc_html__('Typography', 'tpcore'),
                'selector' => '{{WRAPPER}} .tp-megamenu-small-content .tp-megamenu-list a,
                {{WRAPPER}} .main-menu > nav > ul > li > .tp-submenu li > a,
                {{WRAPPER}} .tp-megamenu-fullwidth-list ul li a',
            ]
        );

        $this->start_controls_tabs(
            'mobile_link_style_tabs'
        );

        $this->start_controls_tab(
            'mobile_link_style_normal_tab',
            [
                'label' => esc_html__('Normal', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_mobile_link_menu_title_color',
            [
                'label' => esc_html__('Menu Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-megamenu-small-content .tp-megamenu-list a,
                {{WRAPPER}} .main-menu > nav > ul > li > .tp-submenu li > a,
                {{WRAPPER}} .tp-megamenu-fullwidth-list ul li a' => 'color: {{VALUE}} !important',
                ],
            ]
        );

        $this->add_control(
            'tp_mobile_link_menu_toggle_color',
            [
                'label' => esc_html__('Menu Bar Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-megamenu-small-content .tp-megamenu-list a::before,
                    {{WRAPPER}} .main-menu > nav > ul > li > .tp-submenu li > a::before,
                    {{WRAPPER}} .tp-megamenu-fullwidth-list ul li a:before' => 'background-color: {{VALUE}} !important',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'mobile_link_style_hover_tab',
            [
                'label' => esc_html__('Hover', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_mobile_link_menu_title_hvr_color',
            [
                'label' => esc_html__('Menu Hover Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-megamenu-small-content .tp-megamenu-list a:hover,
                {{WRAPPER}} .main-menu > nav > ul > li > .tp-submenu li > a:hover,
                {{WRAPPER}} .tp-megamenu-fullwidth-list ul li a:hover' => 'color: {{VALUE}} !important',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

        // Mobile Menu style
        $this->start_controls_section(
            'tp_mobile_main_menu_style_section',
            [
                'label' => esc_html__('Mobile Menu', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tp_mobile_main_menu_title_typography',
                'label' => esc_html__('Typography', 'tpcore'),
                'selector' => '{{WRAPPER}} .tp-main-menu-mobile ul li > a',
            ]
        );

        $this->start_controls_tabs(
            'mobile_main_style_tabs'
        );

        $this->start_controls_tab(
            'mobile_main_style_normal_tab',
            [
                'label' => esc_html__('Normal', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_mobile_main_menu_title_color',
            [
                'label' => esc_html__('Menu Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-main-menu-mobile ul li > a' => 'color: {{VALUE}} !important',
                ],
            ]
        );

        $this->add_control(
            'tp_mobile_main_menu_toggle_color',
            [
                'label' => esc_html__('Menu Toggle Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-main-menu-mobile ul li.has-dropdown > a .dropdown-toggle-btn::after' => 'background-color: {{VALUE}} !important',
                    '{{WRAPPER}} .tp-main-menu-mobile ul li.has-dropdown > a .dropdown-toggle-btn::before' => 'background-color: {{VALUE}} !important',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'mobile_main_style_hover_tab',
            [
                'label' => esc_html__('Hover', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_mobile_main_menu_title_hvr_color',
            [
                'label' => esc_html__('Menu Hover Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-main-menu-mobile ul li > a.expanded, 
                    {{WRAPPER}} .tp-main-menu-mobile ul li:hover > a' => 'color: {{VALUE}} !important',
                ],
            ]
        );

        $this->add_control(
            'tp_mobile_main_menu_toggle_hvr_color',
            [
                'label' => esc_html__('Menu Toggle Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-main-menu-mobile ul li.has-dropdown > a.expanded .dropdown-toggle-btn::after,
                    {{WRAPPER}} .offcanvas__2 .tp-main-menu-mobile ul li:hover > a .dropdown-toggle-btn::before,
                    {{WRAPPER}} .offcanvas__2 .tp-main-menu-mobile ul li:hover > a .dropdown-toggle-btn::after' => 'background-color: {{VALUE}} !important',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

        // Close Btn style
        $this->start_controls_section(
            'tp_mobile_close_btn_style_section',
            [
                'label' => esc_html__('Close Button', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tp_mobile_close_btn_typography',
                'label' => esc_html__('Typography', 'tpcore'),
                'selector' => '{{WRAPPER}} .offcanvas__close-btn',
            ]
        );

        $this->start_controls_tabs(
            'close_button_style_tabs'
        );

        $this->start_controls_tab(
            'close_button_style_normal_tab',
            [
                'label' => esc_html__('Normal', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_close_button_menu_title_color',
            [
                'label' => esc_html__('Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .offcanvas__close-btn' => 'color: {{VALUE}} !important',
                ],
            ]
        );

        $this->add_control(
            'tp_close_button_menu_toggle_color',
            [
                'label' => esc_html__('Bg Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .offcanvas__close-btn' => 'background-color: {{VALUE}} !important',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->start_controls_tab(
            'close_button_style_hover_tab',
            [
                'label' => esc_html__('Hover', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_close_button_menu_title_hvr_color',
            [
                'label' => esc_html__('Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .offcanvas__close-btn:hover' => 'color: {{VALUE}} !important',
                ],
            ]
        );

        $this->add_control(
            'tp_close_button_menu_title_hvrbg_color',
            [
                'label' => esc_html__('Bg Color', 'tpcore'),
                'type' => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .offcanvas__close-btn:hover' => 'background-color: {{VALUE}} !important',
                ],
            ]
        );

        $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();

        // Button style
        $this->tp_link_controls_style('', 'btn1_style', 'Button', '.tp-el-btn');

    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $control_id = 'header_btn';

        $btn_id = 'theme_btn';

        if (!empty($settings['menu'])):
            $menu = $settings['menu'];
        else:
            $menu = '';
        endif;

        $menus = $this->get_available_menus();
        require_once get_parent_theme_file_path() . '/inc/class-navwalker.php';

        $args = [
            'echo' => false,
            'menu' => $menu,
            'menu_class' => 'tp-nav-menu ',
            'menu_id' => 'menu-' . $this->get_nav_menu_index() . '-' . $this->get_id(),
            'fallback_cb' => 'Acadia_Navwalker_Class::fallback',
            'container' => '',
            'walker' => new Acadia_Navwalker_Class,
        ];

        $menu_html = wp_nav_menu($args);
        

        $img = tp_get_img($settings, 'tp_logo_black', 'full', false);

        $is_sticky = $settings['tp_header_sticky'] == 'yes' ? 'header-sticky' : 'no-sticky';

        $edit_class = tp_is_elementor_edit_mode() ? 'acadia-elementor-header-edit-mode' : '';

        $this->tp_link_attributes_render('header_btn', 'tp-btn-inner tp-el-btn', $this->get_settings());


        $contact_info = $settings['top_contact_info'];
        $discount_text = $settings['top_discount_text'];
        $currency_text = $settings['top_currency_text'];
        $lang_switch = $settings['top_lang_switch'];

        ?>

        <!-- header-area-start -->

        <header class="header-area tp-header-transparent p-relative tp_el_header">

            <div class="tp-header-11-top">
                <div class="container-fluid">
                    <div class="row">
                        <?php if (!empty($contact_info)): ?>
                            <div class="col-lg-4 col-6">
                                <div class="tp-header-11-top-left">
                                    <div class="tp-heder-info d-flex  align-items-center">
                                        <?php echo tp_kses($contact_info); ?>
                                    </div>
                                </div>
                            </div>
                        <?php endif; ?>

                        <?php if (!empty($discount_text)): ?>
                            <div class="col-lg-4 d-none d-lg-block">
                                <div class="tp-header-11-offer text-center d-none d-xl-block">
                                    <h4>
                                        <?php echo tp_kses($discount_text); ?>
                                    </h4>
                                </div>
                            </div>
                        <?php endif; ?>

                        <div class="col-lg-4 col-6">
                            <div class="tp-header-11-top-right d-flex justify-content-end">
                                <?php if (!empty($lang_switch)): ?>
                                    <div class="tp-header-info-item">

                                        <?php if (!empty(has_nav_menu('tp-language-menu'))): ?>
                                            <div class="header-bottom__lang">
                                                <?php acadia_language_menu(); ?>
                                            </div>
                                        <?php else: ?>
                                            <div class="header-bottom__lang">
                                                <ul>
                                                    <li>
                                                        <a id="header-bottom__lang-toggle" href="javascript:void(0)">
                                                            <span>
                                                                <img src="<?php echo get_template_directory_uri(); ?>/assets/img/flag/flag-1.png" alt="">
                                                                <?php echo esc_html__('English', 'tpcore'); ?>
                                                            </span>

                                                            <span>
                                                                <svg xmlns="http://www.w3.org/2000/svg" width="10" height="6"
                                                                    viewBox="0 0 10 6" fill="none">
                                                                    <path d="M1 1L5 5L9 1" stroke="black" stroke-width="1.5"
                                                                        stroke-linecap="round" stroke-linejoin="round"></path>
                                                                </svg>
                                                            </span>
                                                        </a>
                                                        <ul class="header-bottom__lang-submenu">
                                                            <li><a href="#"><?php echo esc_html__('English', 'tpcore') ?></a></li>
                                                            <li><a href="#"><?php echo esc_html__('Spanish', 'tpcore') ?></a></li>
                                                            <li><a href="#"><?php echo esc_html__('French', 'tpcore') ?></a></li>
                                                        </ul>
                                                    </li>
                                                </ul>
                                            </div>

                                            
                                        <?php endif; ?>
                                    </div>
                                <?php endif; ?>

                                <?php if (!empty($currency_text)): ?>
                                    <div class="tp-header-11-info d-none d-sm-block">
                                        <span>
                                            <svg xmlns="http://www.w3.org/2000/svg" width="8" height="15" viewBox="0 0 8 15"
                                                fill="none">
                                                <path
                                                    d="M2.87888 14.55V12.8121C2.19158 12.6504 1.5983 12.3406 1.09903 11.8825C0.599766 11.4244 0.233422 10.7778 0 9.9425L1.43944 9.33625C1.63396 9.98292 1.92249 10.4747 2.30505 10.8115C2.6876 11.1483 3.19011 11.3167 3.81257 11.3167C4.34425 11.3167 4.79489 11.192 5.16447 10.9428C5.53406 10.6936 5.71885 10.3062 5.71885 9.78083C5.71885 9.30931 5.5762 8.93545 5.29091 8.65927C5.00561 8.38309 4.34425 8.06986 3.30682 7.71958C2.19158 7.35583 1.42647 6.92135 1.0115 6.41615C0.596524 5.91094 0.389037 5.29458 0.389037 4.56708C0.389037 3.69139 0.661364 3.01104 1.20602 2.52604C1.75067 2.04104 2.30829 1.76486 2.87888 1.6975V0H4.43503V1.6975C5.08342 1.80528 5.61835 2.05115 6.03981 2.4351C6.46126 2.81906 6.76925 3.28722 6.96377 3.83958L5.52433 4.48625C5.36872 4.05514 5.14826 3.73181 4.86297 3.51625C4.57767 3.30069 4.18864 3.19292 3.69586 3.19292C3.12527 3.19292 2.69084 3.32427 2.39258 3.58698C2.09432 3.84969 1.94519 4.17639 1.94519 4.56708C1.94519 5.01167 2.13971 5.36194 2.52874 5.61792C2.91778 5.87389 3.59211 6.14333 4.55174 6.42625C5.44652 6.69569 6.1241 7.12344 6.58446 7.70948C7.04482 8.29552 7.275 8.9725 7.275 9.74042C7.275 10.6969 7.00267 11.4244 6.45802 11.9229C5.91337 12.4214 5.23904 12.7312 4.43503 12.8525V14.55H2.87888Z"
                                                    fill="#1E1E2F" />
                                            </svg>
                                            <?php echo tp_kses($currency_text); ?>
                                        </span>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <div id="<?php echo esc_attr($is_sticky); ?>" class="tp-header-mob-space tp-header-11-menu">
                <div class="container custom-container-larg">
                    <div class="row align-items-center">
                        <div class="col-xxl-3 col-xl-2 col-lg-6 col-md-6 col-6">
                            <?php if (!empty($img['tp_logo_black'])): ?>
                                <div class="tp-header-logo-11 tp-header-logo tp-el-logo">
                                    <a href="<?php print esc_url(home_url('/')); ?>">
                                        <img src="<?php echo esc_url($img['tp_logo_black']) ?>" alt="<?php echo esc_attr($img['tp_logo_black_alt']) ?>">
                                    </a>
                                </div>
                            <?php endif; ?>
                        </div>

                        <div class="col-xxl-6 col-xl-7 d-none d-xl-block">
                            <div class="main-menu text-center">
                                <nav class="tp-main-menu-content">
                                    <?php echo $menu_html; ?>
                                </nav>
                            </div>
                        </div>

                        <div class="col-xxl-3 col-xl-3 col-lg-6 col-md-6 col-6">
                            <div class="tp-header-11-content d-flex align-items-center justify-content-end">
                                <div class="tp-header-2-contact d-flex align-items-center justify-content-end">
                                   
                                    <?php if ($settings['tp_header_cart_switch'] == 'yes' && class_exists('WooCommerce')): ?>
                                        <div class="tp-header-2-cart d-none d-xl-block">
                                            <button class="cartmini-open-btn">
                                                <?php tp_render_signle_icon_html($settings, 'header_cart'); ?>
                                                <?php
                                                $cart_count = !is_null(WC()->cart) ? WC()->cart->get_cart_contents_count() : 0;
                                                ?>
                                                <i><?php echo esc_html($cart_count); ?></i>
                                            </button>
                                        </div>
                                    <?php endif; ?>

                                    <?php
                                        if (!empty($settings['tp_header_login_switch']) && function_exists('tutor')):
                                            if (is_user_logged_in()):
                                                $user_id = get_current_user_id();
                                                $user = get_user_by('ID', $user_id);

                                                $cover_placeholder = tutor()->url . 'assets/images/cover-photo.jpg';
                                                $cover_photo_src = $cover_placeholder;
                                                $cover_photo_id = get_user_meta($user->ID, '_tutor_cover_photo', true);
                                                if ($cover_photo_id) {
                                                    $url = wp_get_attachment_image_url($cover_photo_id, 'full');
                                                    !empty($url) ? $cover_photo_src = $url : 0;
                                                }
                                                $profile_placeholder = apply_filters('tutor_login_default_avatar', tutor()->url . 'assets/images/profile-photo.png');
                                                $profile_photo_src = $profile_placeholder;
                                                $profile_photo_id = get_user_meta($user->ID, '_tutor_profile_photo', true);
                                                if ($profile_photo_id) {
                                                    $url = wp_get_attachment_image_url($profile_photo_id, 'full');
                                                    !empty($url) ? $profile_photo_src = $url : 0;
                                                }
                                                $dashboard_pages = tutor_utils()->tutor_dashboard_nav_ui_items();
                                                ?>                                     

                                                <div class="tp-header-6-login tp-header-user-hover d-none d-md-block">
                                                    <button>
                                                        <span class="tp-header_cart-svg-icon ">
                                                            <?php tp_render_signle_icon_html($settings, 'header_login_icon'); ?>
                                                            <?php echo tp_kses($settings['tp_header_login_text']); ?>
                                                        </span>
                                                    </button>
                                        
                                                    <div class="tp-header-user-box">
                                                        <div class="tp-header-user-content">
                                                            <div class="tp-header-user-profile d-flex align-items-center">
                                                                <div class="tp-header-user-profile-thumb">
                                                                    <img src="<?php echo esc_url($profile_photo_src); ?>"
                                                                        alt="<?php echo esc_attr($user->display_name); ?>">
                                                                </div>
                                                                <div class="tp-header-user-profile-content">
                                                                    <h4><?php echo esc_html($user->display_name); ?></h4>
                                                                    <span><?php echo esc_html(ucwords($user->roles[0])); ?></span>
                                                                </div>
                                                            </div>
                                                            <div class="tp-header-user-list">
                                                                <ul>
                                                                    <?php
                                                                    $dashboard_pages = tutor_utils()->tutor_dashboard_nav_ui_items();
                                                                    $disable = !get_tutor_option('enable_course_review');
                                                                    foreach ($dashboard_pages as $dashboard_key => $dashboard_page) {
                                                                        foreach ($settings['tp_header_user_list'] as $key => $value) {
                                                                            if ($dashboard_key == $value) {
                                                                                if ($disable && 'reviews' === $dashboard_key) {
                                                                                    continue;
                                                                                }

                                                                                $menu_title = $dashboard_page;
                                                                                $menu_link = tutor_utils()->get_tutor_dashboard_page_permalink($dashboard_key);
                                                                                $separator = false;
                                                                                $menu_icon = '';

                                                                                if (is_array($dashboard_page)) {
                                                                                    $menu_title = tutor_utils()->array_get('title', $dashboard_page);
                                                                                    $menu_icon_name = tutor_utils()->array_get('icon', $dashboard_page, (isset($dashboard_page['icon']) ? $dashboard_page['icon'] : ''));

                                                                                    if (in_array($dashboard_key, get_lsm_dashboard_menu_keys())) {
                                                                                        $menu_icon = get_lsm_dashboard_menu_icon($dashboard_key);
                                                                                    } else {
                                                                                        if ($menu_icon_name) {
                                                                                            $menu_icon = "<span class='{$menu_icon_name} tutor-dashboard-menu-item-icon'></span>";
                                                                                        }
                                                                                    }

                                                                                    if (isset($dashboard_page['url'])) {
                                                                                        $menu_link = $dashboard_page['url'];
                                                                                    }
                                                                                    if (isset($dashboard_page['type']) && $dashboard_page['type'] == 'separator') {
                                                                                        $separator = true;
                                                                                    }
                                                                                }
                                                                                if ($separator) {
                                                                                    // Optionally, add code to handle separators
                                                                                } else {
                                                                                    $li_class = "acadia-dashboard-menu-{$dashboard_key}";
                                                                                    if ('index' === $dashboard_key) {
                                                                                        $dashboard_key = '';
                                                                                    }

                                                                                    $data_no_instant = 'logout' == $dashboard_key ? 'data-no-instant' : '';
                                                                                    $menu_link = apply_filters('tutor_dashboard_menu_link', $menu_link, $menu_title);
                                                                                    ?>
                                                                                    <?php if ($dashboard_key == 'settings'): ?>
                                                                                        <li class="hr-border"></li>
                                                                                    <?php endif; ?>
                                                                                    <li class='acadia-dashboard-menu-item <?php echo esc_attr($li_class); ?>'>
                                                                                        <a <?php echo esc_html($data_no_instant); ?> href="<?php echo esc_url($menu_link); ?>">
                                                                                            <span>
                                                                                                <?php echo tp_kses($menu_icon); ?>
                                                                                                <?php echo esc_html($menu_title); ?>
                                                                                            </span>
                                                                                        </a>
                                                                                    </li>
                                                                                    <?php
                                                                                }
                                                                            }
                                                                        }
                                                                    }
                                                                    ?>
                                                                </ul>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            <?php else: ?>
                                            <!-- User is not logged in -->

                                                <div class="tp-header-6-login tp-header-user-hover d-none d-md-block">
                                                    <a href="<?php echo esc_url(tutor_utils()->tutor_dashboard_url()); ?>">
                                                        <button>
                                                            <span class="tp-header_cart-svg-icon ">
                                                                <svg width="17" height="19" viewBox="0 0 17 19" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                                    <path d="M8.40947 9.14901C10.6598 9.14901 12.484 7.32479 12.484 5.07451C12.484 2.82422 10.6598 1 8.40947 1C6.15918 1 4.33496 2.82422 4.33496 5.07451C4.33496 7.32479 6.15918 9.14901 8.40947 9.14901Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path>
                                                                    <path d="M15.4102 17.298C15.4102 14.1444 12.2728 11.5937 8.41016 11.5937C4.54752 11.5937 1.41016 14.1444 1.41016 17.298" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"></path>
                                                                </svg>
                                                                <?php echo esc_html__('Admin', 'tpcore'); ?>
                                                            </span>
                                                        </button>
                                                    </a>
                                                </div>
                                            <?php endif; ?>
                                    <?php endif; ?>

                                    <?php if (!empty($settings['tp_' . $control_id . '_text']) && $settings['tp_' . $control_id . '_button_show'] == 'yes'): ?>
                                        <div class="ml-20 d-none d-xxl-block">
                                            <a <?php echo $this->get_render_attribute_string('tp-button-arg' . $control_id . ''); ?>>
                                                <?php echo $settings['tp_' . $control_id . '_text']; ?>
                                            </a>
                                        </div>
                                    <?php endif; ?>

                                    <div class="offcanvas-btn black-color d-xl-none ml-30">
                                        <button class="offcanvas-open-btn">
                                            <i class="fa-solid fa-bars"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </header>
        <!-- header-area-end -->

        <?php if (class_exists('WooCommerce') && $settings['tp_header_cart_switch'] == 'yes'): ?>
            <?php print acadia_shopping_cart(); ?>
            <div class="body-overlay"></div>
        <?php endif; ?>

        <?php include(TPCORE_ELEMENTS_PATH . '/header/header-offcanvas.php'); ?>

        <?php
    }
}

$widgets_manager->register(new TP_Header_lang_academy());
