<?php

namespace TPCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Utils;

if (!defined('ABSPATH'))
    exit; // Exit if accessed directly

/**
 * Tp Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TP_Hero_Art_Design extends Widget_Base
{

    use TP_Style_Trait, TP_Icon_Trait, TP_Animation_Trait;

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'tp-hero-art-design';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Hero Art Design', 'tpcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'tp-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['tpcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['tpcore'];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */

    protected function register_controls()
    {
        $this->register_controls_section();
        $this->style_tab_content();
    }

    protected function register_controls_section()
    {

        // layout Panel
        $this->tp_design_layout('Layout Style', 1);


        // title/content
        $this->tp_section_title_render_controls('section', 'Section Title');

        // main image section
        $this->start_controls_section(
            'tp_main_image_sec',
            [
                'label' => esc_html__('Main Image', 'tpcore'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'tp_hero_bg_image',
            [
                'label' => esc_html__('Background Image', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->add_control(
            'tp_hero_left_image',
            [
                'label' => esc_html__('Left Image', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->add_control(
            'tp_hero_right_image',
            [
                'label' => esc_html__('Right Image', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->add_control(
            'tp_heading_shape_image',
            [
                'label' => esc_html__('Heading Shape Image', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->end_controls_section();

        // shape image section
        $this->start_controls_section(
            'tp_shape_image_sec',
            [
                'label' => esc_html__('Shape Image', 'tpcore'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'tp_shape_class_name',
            [
                'label' => esc_html__('Class Name Add', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Your class name', 'tpcore'),
            ]
        );

        $repeater->add_control(
            'tp_hero_shape_image',
            [
                'label' => esc_html__('Shape Image', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $repeater->add_control(
            'tp_parallax_y_offset',
            [
                'label' => esc_html__('Parallax Y Offset', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('200', 'tpcore'),
            ]
        );

        $repeater->add_control(
            'tp_parallax_smoothness',
            [
                'label' => esc_html__('Parallax Smoothness', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('30', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_hero_shape_section',
            [
                'label' => esc_html__('Section Label', 'tpcore'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'tp_shape_class_name' => esc_html__('Default Class', 'tpcore'),
                    ],
                ],
                'title_field' => '{{{ tp_shape_class_name }}}',
            ]
        );

        $this->end_controls_section();

        // button
        $this->start_controls_section(
            'tp_hero_online_button_sec',
            [
                'label' => esc_html__('Button', 'tpcore'),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'tp_design_style' => ['layout-1'],
                ]
            ]
        );

        $this->add_control(
            'tp_button_text',
            [
                'label' => esc_html__('Button text', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'default' => esc_html__('Button Text', 'tpcore'),
                'placeholder' => esc_html__('Placeholder Text', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_theme_btn_icon_show',
            [
                'label' => esc_html__('Add Icon ?', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'tpcore'),
                'label_off' => esc_html__('Hide', 'tpcore'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->tp_single_icon_control('theme_btn', 'tp_theme_btn_icon_show', 'yes');

        tp_render_links_controls($this, 'button_url');

        $this->end_controls_section();
    }

    protected function style_tab_content()
    {
        $this->tp_section_style_controls('section', 'Section - Style', '.tp-el-section');
        $this->tp_basic_style_controls('heading_subtitle', 'Section - Subtitle', '.tp-el-subtitle');
        $this->tp_basic_style_controls('heading_title', 'Section - Title', '.tp-el-title');
        $this->tp_basic_style_controls('heading_desc', 'Section - Description', '.tp-el-content', 'layout-1');
        $this->tp_link_controls_style('', 'btn1_style', 'Button', '.tp-el-btn');
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();
        ?>

        <?php if ($settings['tp_design_style'] == 'layout-2'): ?>

        <?php else:

            $subtitle = $settings['tp_section_sub_title'];
            $desc = $settings['tp_section_description'];

            // thumbnail
            $bg_img = tp_get_img($settings, 'tp_hero_bg_image', 'full', false);
            $left_img = tp_get_img($settings, 'tp_hero_left_image', 'full', false);
            $right_img = tp_get_img($settings, 'tp_hero_right_image', 'full', false);
            $heading_shape = tp_get_img($settings, 'tp_heading_shape_image', 'full', false);

            $button_text = $settings['tp_button_text'];

            $attrs = tp_get_repeater_links_attr($settings, 'button_url');
            extract($attrs);

            $links_attrs = [
                'href' => $link,
                'target' => $target,
                'rel' => $rel,
                'class' => 'tp-btn-border green-bg tp-el-btn'
            ];

            $title = $settings['tp_section_title'];
            $title_tag = $settings['tp_section_title_tag'];

            $this->add_render_attribute('title_args', 'class', 'tp-hero-12-title wow fadeInUp tp-el-title');
            $this->add_render_attribute('title_args', 'data-delay', '.5s');

            ?>

            <!-- hero-area-start -->
            <div class="tp-hero-12-area tp-hero-12-ptb include-bg p-relative tp-el-section"
                style="background-image: url(<?php echo esc_url($bg_img['tp_hero_bg_image']); ?>)">

                <div class="tp-hero-12-shapes">

                    <?php foreach ($settings['tp_hero_shape_section'] as $key => $item):
                        $class = $item['tp_shape_class_name'];
                        $img = tp_get_img($item, 'tp_hero_shape_image', 'full', false);

                        $smoothness = $item['tp_parallax_smoothness'];
                        $y_offset = $item['tp_parallax_y_offset'];
                        ?>
                        <div class="<?php echo esc_attr($class); ?>"
                            data-parallax='{"y":  <?php echo esc_attr($y_offset); ?>, "smoothness": <?php echo esc_attr($smoothness); ?>}'>
                            <img src="<?php echo esc_url($img['tp_hero_shape_image']); ?>" alt="">
                        </div>
                    <?php endforeach; ?>
                </div>

                <div class="tp-hero-12-thumbs">
                    <?php if (!empty($left_img['tp_hero_left_image'])): ?>
                        <div class="hero-thumb-1">
                            <img src="<?php echo esc_url($left_img['tp_hero_left_image']); ?>" alt="">
                        </div>
                    <?php endif; ?>

                    <?php if (!empty($right_img['tp_hero_right_image'])): ?>
                        <div class="hero-thumb-2">
                            <img src="<?php echo esc_url($right_img['tp_hero_right_image']); ?>" alt="">
                        </div>
                    <?php endif; ?>
                </div>

                <div class="container">
                    <div class="tp-hero-12-wrapper p-relative">
                        <div class="row justify-content-center">
                            <div class="col-xl-10">
                                <div class="tp-hero-12-content text-center p-relative pb-60">
                                    <?php if (!empty($heading_shape['tp_heading_shape_image'])): ?>
                                        <div class="tp-hero-12-text-shape">
                                            <img src="<?php echo esc_url($heading_shape['tp_heading_shape_image']); ?>" alt="">
                                        </div>
                                    <?php endif; ?>

                                    <?php if (!empty($subtitle)): ?>
                                        <span class="tp-hero-6-subtitle color-black wow fadeInUp tp-el-subtitle" data-wow-delay=".3s">
                                            <?php echo tp_kses($subtitle); ?>
                                        </span>
                                    <?php endif; ?>

                                    <?php
                                    if (!empty($title)):
                                        printf(
                                            '<%1$s %2$s %3$s>%4$s</%1$s>',
                                            tag_escape($title_tag),
                                            $this->get_render_attribute_string('title_args'),
                                            $this->get_render_attribute_string('title_data'),
                                            tp_kses($title)
                                        );
                                    endif;
                                    ?>

                                    <?php if (!empty($desc)): ?>
                                        <p class="wow fadeInUp tp-el-content" data-wow-delay=".5s">
                                            <?php echo tp_kses($desc); ?>
                                        </p>
                                    <?php endif; ?>

                                    <div class="tp-hero-12-btn wow fadeInUp" data-wow-delay=".7s">
                                        <a <?php echo tp_implode_html_attributes($links_attrs); ?>>
                                            <?php echo tp_kses($button_text); ?>

                                            <?php tp_render_signle_icon_html($settings, 'theme_btn', 'icon'); ?>
                                        </a>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <!-- hero-area-end -->
        <?php endif;
    }
}

$widgets_manager->register(new TP_Hero_Art_Design());