<?php

namespace TPCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Utils;

if (!defined('ABSPATH'))
    exit; // Exit if accessed directly

/**
 * Tp Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TP_Hero_Kitchen_Coach extends Widget_Base
{

    use \TPCore\Widgets\TP_Style_Trait;
    use \TPCore\Widgets\TP_Animation_Trait;
    use \TPCore\Widgets\TP_Query_Trait;

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'tp-hero-kitchen-coach';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Hero Kitchen Coach', 'tpcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'tp-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['tpcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['tpcore'];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */

    protected function register_controls()
    {
        $this->register_controls_section();
        $this->style_tab_content();
    }

    protected function register_controls_section()
    {

        // layout Panel
        $this->tp_design_layout('Layout Style', 1);


        // title/content
        $this->tp_section_title_render_controls('section', 'Section Title');


        //  main image
        $this->start_controls_section(
            'tp_hero_main_image_sec',
            [
                'label' => esc_html__('Main Image', 'tpcore'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control(
            'tp_hero_main_image',
            [
                'label' => esc_html__('Main Image', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );
        $this->end_controls_section();

        // Shape Image Section
        $this->start_controls_section(
            'tp_hero_online_shape_sec',
            [
                'label' => esc_html__('Shape Image Section', 'tpcore'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_responsive_control(
            'left_shape_image',
            [
                'label' => esc_html__('Left Image', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->add_responsive_control(
            'tp_hero_middle_shape',
            [
                'label' => esc_html__('Middle Image', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->end_controls_section();


        // Social Section
        $this->start_controls_section(
            'tp_social_sec',
            [
                'label' => esc_html__('Social', 'tpcore'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'tp_social_text',
            [
                'label' => esc_html__('Social Title', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Follow Us', 'tpcore'),
                'label_block' => true,
            ]
        );

        $repeater = new \Elementor\Repeater();

        $repeater->add_control(
            'repeater_condition',
            [
                'label' => __('Field condition', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'style_1' => __('Style 1', 'tpcore'),
                ],
                'default' => 'style_1',
                'frontend_available' => true,
                'style_transfer' => true,
            ]
        );

        $repeater->add_control(
            'hero_social_link',
            [
                'label' => esc_html__('Link', 'tpcore'),
                'type' => Controls_Manager::URL,
                'label_block' => true,
            ]
        );

        tp_rep_render_icon_controls($repeater, 'hero_social_btn_icon', 'style_1');

        $this->add_control(
            'tp_social_list',
            [
                'label' => esc_html__('Section Label', 'tpcore'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'hero_social_link' => 'url',
                    ],
                ],
                'title_field' => '{{{ hero_social_link.url }}}',
            ]
        );

        $this->end_controls_section();

        // button
        $this->start_controls_section(
            'tp_hero_online_button_sec',
            [
                'label' => esc_html__('Button', 'tpcore'),
                'tab' => Controls_Manager::TAB_CONTENT,
                'condition' => [
                    'tp_design_style' => ['layout-1'],
                ]
            ]
        );

        $this->add_control(
            'tp_button_text',
            [
                'label' => esc_html__('Button text', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'default' => esc_html__('Button Text', 'tpcore'),
                'placeholder' => esc_html__('Placeholder Text', 'tpcore'),
            ]
        );

        tp_render_links_controls($this, 'button_url');

        $this->end_controls_section();

        // video section
        $this->start_controls_section(
            'tp_video_section',
            [
                'label' => esc_html__('Video Section', 'tpcore'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );
        $this->add_control(
            'tp_video_title',
            [
                'label' => esc_html__('Video Title', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Watch Chef Video', 'tpcore'),
                'label_block' => true,
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );
        $this->add_control(
            'tp_video_url',
            [
                'label' => esc_html__('Video URL', 'tpcore'),
                'type' => Controls_Manager::TEXTAREA,
                'rows' => 5,
                'default' => esc_html__('https://www.youtube.com/watch?v=SHptCWeDjD4', 'tpcore'),
            ]
        );

        $this->end_controls_section();

        // client section
        $this->start_controls_section(
            'tp_client_section',
            [
                'label' => esc_html__('Client Section', 'tpcore'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'tp_client_number',
            [
                'label' => esc_html__('Client Number', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('2.5K', 'tpcore'),
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $this->add_control(
            'tp_client_text',
            [
                'label' => esc_html__('Client Text', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('We consultant client world-wide', 'tpcore'),
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $this->add_responsive_control(
            'tp_client_image',
            [
                'label' => esc_html__('Image', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
            ]
        );

        $this->end_controls_section();

        // courses
        $this->tp_query_controls('course_card', 'Course Controls', 'courses', 'course-category');

        $this->start_controls_section(
            'tp_course_section',
            [
                'label' => esc_html__('Course Section', 'tpcore'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'tp_course_text',
            [
                'label' => esc_html__('Course Text', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('New cooking show', 'tpcore'),
                'dynamic' => [
                    'active' => true,
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function style_tab_content()
    {
        $this->tp_section_style_controls('section', 'Section - Style', '.tp-el-section');
        $this->tp_basic_style_controls('heading_subtitle', 'Section - Subtitle', '.tp-el-subtitle');
        $this->tp_basic_style_controls('heading_title', 'Section - Title', '.tp-el-title');
        $this->tp_basic_style_controls('heading_desc', 'Section - Description', '.tp-el-content', 'layout-1');
        $this->tp_link_controls_style('', 'btn1_style', 'Button', '.tp-el-btn');
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();
        ?>

        <?php if ($settings['tp_design_style'] == 'layout-2'): ?>

        <?php else:
            // thumbnail
            $img = tp_get_img($settings, 'tp_hero_main_image', 'full', false);
            $img2 = tp_get_img($settings, 'left_shape_image', 'full', false);
            $img3 = tp_get_img($settings, 'tp_hero_middle_shape', 'full', false);
            $img5 = tp_get_img($settings, 'tp_client_image', 'full', false);

            $button_text = $settings['tp_button_text'];

            $attrs = tp_get_repeater_links_attr($settings, 'button_url');
            extract($attrs);

            $links_attrs = [
                'href' => $link,
                'target' => $target,
                'rel' => $rel,
                'class' => 'tp-chef-btn-lg tp-chef-btn tp-chef-btn-squre tp-chef-btn-shadow-4 mb-70 wow fadeInUp tp-el-btn'
            ];

            $title = $settings['tp_section_title'];
            $title_tag = $settings['tp_section_title_tag'];

            $this->add_render_attribute('title_args', 'class', 'tp-hero-chef-title mb-40 wow fadeInUp');
            $this->add_render_attribute('title_args', 'data-delay', '.5s');

            ?>

            <!-- hero-area-start -->
            <div class="tp-hero-chef-area fix p-relative chef-bg tp-el-section">
                <div class="tp-hero-chef-shape-2 p-absolute d-none d-sm-block">
                    <img src="<?php echo esc_url($img2['left_shape_image']); ?>" alt="">
                </div>
                <div class="tp-hero-chef-social d-none d-md-block">
                    <?php if (!empty($settings['tp_social_text'])): ?>
                        <span class="tp-el-social-title">
                            <?php echo tp_kses($settings['tp_social_text']); ?>
                        </span>
                    <?php endif; ?>

                    <ul>
                        <?php foreach ($settings['tp_social_list'] as $item): ?>
                            <li>
                                <a href="<?php echo esc_url($item['hero_social_link']['url']); ?>">
                                    <?php tp_render_single_icon_html($item, 'rep_hero_social_btn_icon', ); ?>
                                </a>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
                <div class="container-fluid p-0">
                    <div class="row">
                        <div class="col-xl-7">
                            <div class="tp-hero-chef-content p-relative">
                                <div class="tp-hero-chef-shape p-absolute d-none d-sm-block">
                                    <img src="<?php echo esc_url($img3['tp_hero_middle_shape']); ?>" alt="">
                                </div>

                                <div class="tp-hero-chef-video-wrap mb-30 d-flex align-items-center wow fadeInUp"
                                    data-wow-delay=".3s">
                                    <a class="tp-hero-chef-video popup-video mr-10"
                                        href="<?php echo esc_url($settings['tp_video_url']); ?>">
                                        <span>
                                            <i class="fa-solid fa-play"></i>
                                        </span>
                                    </a>
                                    <?php if (!empty($settings['tp_video_title'])): ?>
                                        <span class="text">
                                            <?php echo tp_kses($settings['tp_video_title']); ?>
                                        </span>
                                    <?php endif; ?>
                                </div>

                                <?php
                                if (!empty($title)):
                                    printf(
                                        '<%1$s %2$s %3$s>%4$s</%1$s>',
                                        tag_escape($title_tag),
                                        $this->get_render_attribute_string('title_args'),
                                        $this->get_render_attribute_string('title_data'),
                                        tp_kses($title)
                                    );
                                endif;
                                ?>

                                <?php if (!empty($links_attrs['href'])): ?>
                                    <a <?php echo tp_implode_html_attributes($links_attrs); ?> data-wow-delay=".7s">
                                        <?php echo tp_kses($button_text); ?>
                                    </a>
                                <?php endif; ?>

                                <div class="tp-hero-chef-clients-wrap d-flex wow fadeInUp" data-wow-delay=".9s">
                                    <div class="tp-hero-chef-clients mt-35">
                                        <?php if (!empty($settings['tp_client_number'])): ?>
                                            <h3>
                                                <?php echo tp_kses($settings['tp_client_number']); ?>
                                            </h3>
                                        <?php endif; ?>

                                        <?php if (!empty($settings['tp_client_text'])): ?>
                                            <span>
                                                <?php echo tp_kses($settings['tp_client_text']); ?>
                                            </span>
                                        <?php endif; ?>
                                    </div>
                                    <span class="tp-borders"></span>
                                    <div class="rounded-text p-relative d-inline-block">
                                        <span class="tp-hero-chef-icon">
                                            <svg width="33" height="32" viewBox="0 0 33 32" fill="none"
                                                xmlns="http://www.w3.org/2000/svg">
                                                <path d="M7.9292 24.9289L25.6129 7.24521" stroke="#444C64" stroke-width="2"
                                                    stroke-miterlimit="10" />
                                                <path d="M25.835 21.1649C21.9265 17.2564 21.8269 11.0306 25.6124 7.2452"
                                                    stroke="#444C64" stroke-width="2" stroke-miterlimit="10" />
                                                <path d="M25.6127 7.24521C21.8273 11.0307 15.6015 10.9311 11.693 7.02264"
                                                    stroke="#444C64" stroke-width="2" stroke-miterlimit="10" />
                                            </svg>
                                        </span>
                                        <img class="tp-hero-chef-text" src="<?php echo esc_url($img5['tp_client_image']); ?>"
                                            alt="">
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="col-xl-5">
                            <div class="tp-hero-chef-thumb p-relative">

                                <div class="tp-courses-chef-wrap tp-hero-chef-card  wow fadeInUp z-index-11" data-wow-delay=".5s">

                                    <?php if (!empty($settings['tp_course_text'])): ?>
                                        <h4 class="tp-courses-chef-title">
                                            <?php echo tp_kses($settings['tp_course_text']); ?>
                                        </h4>
                                    <?php endif; ?>

                                    <div class="tp-hero-kitchen-course-card">
                                        <?php

                                        $args = tp_query_args('courses', 'course-category', $this->get_settings());
                                        $main_query = new \WP_Query($args);

                                        ?>
                                        <?php if ($main_query->have_posts()): ?>
                                            <?php while ($main_query->have_posts()):
                                                $main_query->the_post();
                                                global $post, $authordata;

                                                $tutor_course_img = get_tutor_course_thumbnail_src();
                                                $course_id = get_the_ID();

                                                $tutor_lesson_count = tutor_utils()->get_lesson_count_by_course(get_the_ID());
                                                $course_students = apply_filters('tutor_course_students', tutor_utils()->count_enrolled_users_by_course($course_id), $course_id);

                                                $show_course_ratings = apply_filters('tutor_show_course_ratings', true, get_the_ID());
                                                $course_rating = tutor_utils()->get_course_rating();

                                                $show_course_ratings = apply_filters('tutor_show_course_ratings', true, get_the_ID());
                                                $course_rating = tutor_utils()->get_course_rating(); 
                                                ?>
                                                <div class="tp-hero-kitchen-course-wrapper">
                                                    <div class="tp-courses-chef-thumb mb-15">
                                                        <a href="<?php the_permalink(); ?>">
                                                            <img class="w-100" src="<?php echo esc_url($tutor_course_img); ?>"
                                                                alt="<?php the_title(); ?>" loading="lazy">
                                                        </a>
                                                    </div>

                                                    <?php if ($show_course_ratings): ?>
                                                        <div class="tp-courses-chef-ratings mb-10">    
                                                            <span>
                                                                <?php
                                                                $course_rating = tutor_utils()->get_course_rating();
                                                                tutor_utils()->star_rating_generator_course($course_rating->rating_avg);
                                                                ?>
                                                            </span>
                                                        </div>
                                                    <?php endif; ?>

                                                    <h3 class="tp-courses-chef-title mb-15">
                                                        <a href="<?php the_permalink(); ?>">
                                                            <?php echo wp_trim_words(get_the_title(), $settings['tp_post_title_word'], ''); ?>
                                                        </a>
                                                    </h3>

                                                    <?php if (tutor_utils()->get_option('enable_course_total_enrolled') || !empty($tutor_lesson_count)): ?>
                                                        <div class="tp-courses-chef-studebts">
                                                            <?php if (!empty($tutor_lesson_count)): ?>
                                                                <span>
                                                                    <svg width="15" height="14" viewBox="0 0 15 14" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                                        <path
                                                                            d="M13.9228 10.0426V2.29411C13.9228 1.51825 13.2949 0.953997 12.5252 1.01445H12.4847C11.1276 1.12529 9.07163 1.82055 7.91706 2.53596L7.80567 2.6065C7.62337 2.71733 7.30935 2.71733 7.11692 2.6065L6.9549 2.50573C5.81046 1.79033 3.75452 1.1152 2.3974 1.00437C1.62768 0.943911 0.999756 1.51827 0.999756 2.28405V10.0426C0.999756 10.6573 1.50613 11.2417 2.12393 11.3122L2.30622 11.3425C3.70386 11.5238 5.87126 12.2392 7.10685 12.9143L7.1372 12.9244C7.30937 13.0252 7.59293 13.0252 7.75498 12.9244C8.99057 12.2393 11.1681 11.5339 12.5758 11.3425L12.7885 11.3122C13.4164 11.2417 13.9228 10.6674 13.9228 10.0426Z"
                                                                            stroke="#94928E" stroke-width="1.2" stroke-linecap="round" stroke-linejoin="round" />
                                                                        <path d="M7.46118 2.81787V12.4506" stroke="#94928E" stroke-width="1.2" stroke-linecap="round"
                                                                            stroke-linejoin="round" />
                                                                    </svg>
                                                                    <?php printf(_n('%d Lesson', '%d Lessons', $tutor_lesson_count, 'acadia'), $tutor_lesson_count); ?>
                                                                </span>
                                                            <?php endif; ?>
                                                    
                                                            <?php if (!empty($course_students)): ?>
                                                                <span>
                                                                    <svg width="13" height="15" viewBox="0 0 13 15" fill="none" xmlns="http://www.w3.org/2000/svg">
                                                                        <path
                                                                            d="M6.57134 7.5C8.36239 7.5 9.81432 6.04493 9.81432 4.25C9.81432 2.45507 8.36239 1 6.57134 1C4.7803 1 3.32837 2.45507 3.32837 4.25C3.32837 6.04493 4.7803 7.5 6.57134 7.5Z"
                                                                            stroke="#94928E" stroke-width="1.2" stroke-linecap="round" stroke-linejoin="round" />
                                                                        <path
                                                                            d="M12.1426 14C12.1426 11.4845 9.64553 9.44995 6.57119 9.44995C3.49684 9.44995 0.999756 11.4845 0.999756 14"
                                                                            stroke="#94928E" stroke-width="1.2" stroke-linecap="round" stroke-linejoin="round" />
                                                                    </svg>
                                                                    <?php printf(_n('%d Student', '%d Students', $course_students, 'acadia'), $course_students); ?>
                                                                </span>
                                                            <?php endif; ?>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                            <?php endwhile;
                                            wp_reset_query();
                                        endif; ?>
                                    </div>
                                </div>

                                <div class=" wow fadeInRight" data-wow-delay=".5s">
                                    <img class="thumb w-100" src="<?php echo esc_url($img['tp_hero_main_image']); ?>" alt="">
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <!-- hero-area-end -->
        <?php endif;
    }
}

$widgets_manager->register(new TP_Hero_Kitchen_Coach());