<?php

namespace TPCore\Widgets;

use Elementor\Widget_Base;
use \Elementor\Group_Control_Image_Size;

if (!defined('ABSPATH'))
	exit; // Exit if accessed directly

/**
 * Tp Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TP_Image extends Widget_Base
{

	use \TPCore\Widgets\TP_Style_Trait;
	use \TPCore\Widgets\TP_Animation_Trait;

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'tp-image';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return __('Image', 'tpcore');
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'tp-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['tpcore'];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends()
	{
		return ['tpcore'];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */

	protected function register_controls()
	{
		$this->register_controls_section();
		$this->style_tab_content();
	}

	protected function register_controls_section()
	{

		// layout Panel
		$this->tp_design_layout('Layout Style', 6);

		// image
		$this->start_controls_section(
			'thumb_section',
			[
				'label' => esc_html__('Image Section', 'tpcore'),
			]
		);

		$this->add_control(
			'tp_image',
			[
				'label' => esc_html__('Choose Image', 'tpcore'),
				'type' => \Elementor\Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
			]
		);

		$this->add_control(
			'tp_shape_image',
			[
				'label' => esc_html__('Shape Image', 'tpcore'),
				'type' => \Elementor\Controls_Manager::MEDIA,
				'default' => [
					'url' => \Elementor\Utils::get_placeholder_image_src(),
				],
				'condition' => [
					'tp_design_style!' => ['layout-4', 'layout-5', 'layout-6'],
				]
			]
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			[
				'name' => 'tp_image_size',
				'default' => 'full',
				'exclude' => [
					'custom'
				],
				'condition' => [
					'tp_design_style!' => ['layout-3', 'layout-4', 'layout-5', 'layout-6'],
				]
			]
		);

		$this->end_controls_section();

		// animation
		$this->tp_creative_animation(['layout-3', 'layout-6']);
	}

	protected function style_tab_content()
	{

		$this->start_controls_section(
			'tp_image_style_sec',
			[
				'label' => esc_html__('Image Styles', 'tpcore'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,

			]
		);

		$this->add_control(
			'tp_image_style_width',
			[
				'label' => esc_html__('Width', 'tpcore'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em', 'rem', 'custom'],
				'selectors' => [
					'{{WRAPPER}} .tp-el-image img' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->add_control(
			'tp_image_style_height',
			[
				'label' => esc_html__('Height', 'tpcore'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em', 'rem', 'custom'],
				'selectors' => [
					'{{WRAPPER}} .tp-el-image img' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'tp_image_object_style',
			[
				'label' => esc_html__('Object Fit', 'tpcore'),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'cover',
				'options' => [
					'none' => esc_html__('None', 'tpcore'),
					'contain' => esc_html__('Contain', 'tpcore'),
					'cover' => esc_html__('Cover', 'tpcore'),
					'fill' => esc_html__('fill', 'tpcore'),
				],
				'selectors' => [
					'{{WRAPPER}} .tp-el-image img' => 'object-fit: {{VALUE}};',
				],
			]
		);
		$this->add_control(
			'tp_image_border_radius',
			[
				'label' => esc_html__('Border Radius', 'tpcore'),
				'type' => \Elementor\Controls_Manager::DIMENSIONS,
				'size_units' => ['px', '%', 'em'],
				'selectors' => [
					'{{WRAPPER}} .tp-el-image img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section();


		$this->start_controls_section(
			'tp_video_style_sec',
			[
				'label' => esc_html__('Video Styles', 'tpcore'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'tp_design_style' => 'layout-5',
				]
			]
		);

		$this->add_control(
			'tp_video_style_width',
			[
				'label' => esc_html__('Width', 'tpcore'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em', 'rem', 'custom'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 5,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => '%',
					'size' => 100,
				],
				'selectors' => [
					'{{WRAPPER}} .tp-el-video' => 'width: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->add_control(
			'tp_video_style_height',
			[
				'label' => esc_html__('Height', 'tpcore'),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => ['px', '%', 'em', 'rem', 'custom'],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 1000,
						'step' => 5,
					],
					'%' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 230,
				],
				'selectors' => [
					'{{WRAPPER}} .tp-el-video' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();
		$animation = $this->tp_animation_show($settings);
		?>

		<?php if ($settings['tp_design_style'] == 'layout-6'):
			$img = tp_get_img($settings, 'tp_image', 'full', false);
			?>
			<div class="tp-about-yoga-thumb tp-el-image <?php echo esc_attr($animation['animation']); ?>" <?php echo $animation['duration'] . ' ' . $animation['delay']; ?>>
				<img class="w-100" src="<?php echo esc_url($img['tp_image']); ?>" alt="">
			</div>


		<?php elseif ($settings['tp_design_style'] == 'layout-5'):
			$img = tp_get_img($settings, 'tp_image', 'full', false);
			?>
			<div class="tp-courses-chef-shape tp-el-image" data-parallax='{"y": 100, "smoothness": 30}'>
				<img src="<?php echo esc_url($img['tp_image']); ?>" alt="">
			</div>

		<?php elseif ($settings['tp_design_style'] == 'layout-4'):
			$img = tp_get_img($settings, 'tp_image', 'full', false);
			?>

			<div class="tp-feature-chef-shape tp-el-image d-none d-md-block" data-parallax='{"x": 80, "smoothness": 30}'>
				<img src="<?php echo esc_url($img['tp_image']); ?>" alt="">
			</div>

		<?php elseif ($settings['tp_design_style'] == 'layout-3'):
			$img = tp_get_img($settings, 'tp_image', 'full', false);
			$img2 = tp_get_img($settings, 'tp_shape_image', 'full', false);
			$animation = $this->tp_animation_show($settings);
			?>

			<div class="tp-faq-quran-thumb-wrap p-relative  <?php echo esc_attr($animation['animation']); ?>" <?php echo $animation['duration'] . ' ' . $animation['delay']; ?>>

				<div class="tp-faq-quran-thumb tp-el-image">
					<img src="<?php echo esc_url($img['tp_image']); ?>" alt="">
				</div>

				<div class="tp-faq-quran-thumb-2 p-absolute mr-30" data-parallax='{"y": -160, "smoothness": 20}'>
					<img src="<?php echo esc_url($img2['tp_shape_image']); ?>" alt="">
				</div>
			</div>

		<?php elseif ($settings['tp_design_style'] == 'layout-2'):

			$img = tp_get_img($settings, 'tp_image', 'full', false);

			?>

			<div class="tp-custom-hover-bg">
				<img src="<?php echo esc_url($img['tp_image']); ?>" alt="<?php echo esc_attr($img['tp_image_alt']); ?>">
			</div>

		<?php else:
			// thumbnail
			$img = tp_get_img($settings, 'tp_image', 'tp_image_size');
			?>

			<div class="offcanvas-gallery-img fix tp-el-image">
				<img src="<?php echo esc_url($img['tp_image']); ?>" alt="<?php echo esc_url($img['tp_image_alt']); ?>">
			</div>

		<?php endif;
	}
}

$widgets_manager->register(new TP_Image());
