<?php
namespace TPCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Utils;
use \Elementor\Control_Media;

use \Elementor\Group_Control_Border;
use \Elementor\Group_Control_Box_Shadow;
use \Elementor\Group_Control_Text_Shadow;
use \Elementor\Group_Control_Typography;
use \Elementor\Core\Schemes\Typography;
use \Elementor\Group_Control_Background;
use \Elementor\Group_Control_Image_Size;
use \Elementor\Repeater;

if (!defined('ABSPATH'))
    exit; // Exit if accessed directly

/**
 * Tp Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TP_Price_Box extends Widget_Base
{

    use \TPCore\Widgets\TP_Style_Trait;
    use \TPCore\Widgets\TP_Animation_Trait;

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'tp-price-box';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Price Box', 'tpcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'tp-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['tpcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['tpcore'];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */

    private static function get_currency_symbol($symbol_name)
    {
        $symbols = [
            'baht' => '&#3647;',
            'bdt' => '&#2547;',
            'dollar' => '&#36;',
            'euro' => '&#128;',
            'franc' => '&#8355;',
            'guilder' => '&fnof;',
            'indian_rupee' => '&#8377;',
            'pound' => '&#163;',
            'peso' => '&#8369;',
            'peseta' => '&#8359',
            'lira' => '&#8356;',
            'ruble' => '&#8381;',
            'shekel' => '&#8362;',
            'rupee' => '&#8360;',
            'real' => 'R$',
            'krona' => 'kr',
            'won' => '&#8361;',
            'yen' => '&#165;',
        ];

        return isset($symbols[$symbol_name]) ? $symbols[$symbol_name] : '';
    }

    private static function get_currency_symbol_text($symbol_text)
    {
        $symbols = [
            'baht' => 'THB',
            'bdt' => 'BDT',
            'dollar' => 'USD',
            'euro' => 'EUR',
            'franc' => 'EUR',
            'guilder' => 'GLD',
            'indian_rupee' => 'INR',
            'pound' => 'GBP',
            'peso' => 'MXN',
            'lira' => 'TRY',
            'ruble' => 'RUB',
            'shekel' => 'ILS',
            'real' => 'BRL',
            'krona' => 'KR',
            'won' => 'KRW',
            'yen' => 'JPY',
        ];

        return isset($symbols[$symbol_text]) ? $symbols[$symbol_text] : '';
    }


    protected function register_controls()
    {
        $this->register_controls_section();
        $this->style_tab_content();
    }

    protected function register_controls_section()
    {

        // layout Panel
        $this->tp_design_layout('Layout Style', 2);


        $this->start_controls_section(
            '_section_pricing',
            [
                'label' => __('Pricing', 'tpcore'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'currency',
            [
                'label' => __('Currency', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'label_block' => false,
                'options' => [
                    '' => __('None', 'tpcore'),
                    'baht' => '&#3647; ' . _x('Baht', 'Currency Symbol', 'tpcore'),
                    'bdt' => '&#2547; ' . _x('BD Taka', 'Currency Symbol', 'tpcore'),
                    'dollar' => '&#36; ' . _x('Dollar', 'Currency Symbol', 'tpcore'),
                    'euro' => '&#128; ' . _x('Euro', 'Currency Symbol', 'tpcore'),
                    'franc' => '&#8355; ' . _x('Franc', 'Currency Symbol', 'tpcore'),
                    'guilder' => '&fnof; ' . _x('Guilder', 'Currency Symbol', 'tpcore'),
                    'krona' => 'kr ' . _x('Krona', 'Currency Symbol', 'tpcore'),
                    'lira' => '&#8356; ' . _x('Lira', 'Currency Symbol', 'tpcore'),
                    'peso' => '&#8369; ' . _x('Peso', 'Currency Symbol', 'tpcore'),
                    'pound' => '&#163; ' . _x('Pound Sterling', 'Currency Symbol', 'tpcore'),
                    'real' => 'R$ ' . _x('Real', 'Currency Symbol', 'tpcore'),
                    'ruble' => '&#8381; ' . _x('Ruble', 'Currency Symbol', 'tpcore'),
                    'indian_rupee' => '&#8377; ' . _x('Rupee (Indian)', 'Currency Symbol', 'tpcore'),
                    'shekel' => '&#8362; ' . _x('Shekel', 'Currency Symbol', 'tpcore'),
                    'won' => '&#8361; ' . _x('Won', 'Currency Symbol', 'tpcore'),
                    'yen' => '&#165; ' . _x('Yen/Yuan', 'Currency Symbol', 'tpcore'),
                    'custom' => __('Custom', 'tpcore'),
                ],
                'default' => 'dollar',
            ]
        );

        $this->add_control(
            'currency_custom',
            [
                'label' => __('Custom Symbol', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'condition' => [
                    'currency' => 'custom',
                ],
                'dynamic' => [
                    'active' => true,
                ]
            ]
        );

        $this->add_control(
            'price_duration',
            [
                'label' => __('Duration', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => '/monthly',
                'dynamic' => [
                    'active' => true
                ]
            ]
        );

        $this->add_control(
            'price_main',
            [
                'label' => __('Price', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => '59.99',
                'dynamic' => [
                    'active' => true
                ]
            ]
        );


        $this->end_controls_section();

        $this->start_controls_section(
            'tp_pricing_header',
            [
                'label' => esc_html__('Header Controls', 'tpcore'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'tp_pricing_active_switcher',
            [
                'label' => esc_html__('Is Active Price?', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Show', 'tpcore'),
                'label_off' => esc_html__('Hide', 'tpcore'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $this->add_control(
            'tp_pricing_header_subtitle',
            [
                'label' => esc_html__('Subtitle', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('This is subtitle', 'tpcore'),
                'placeholder' => esc_html__('Your Text', 'tpcore'),
                'label_block' => true,
            ]
        );

        $this->add_control(
            'tp_pricing_header_title',
            [
                'label' => esc_html__('Title', 'tpcore'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Gold Membership', 'tpcore'),
                'placeholder' => esc_html__('Your Gold Membership Text', 'tpcore'),
                'label_block' => true,
            ]
        );

        $this->end_controls_section();


        $this->start_controls_section(
            '_section_features',
            [
                'label' => __('Features', 'tpcore'),
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'tp_feature_switcher',
            [
                'label' => esc_html__('Active Feature', 'tpcore'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__('Yes', 'tpcore'),
                'label_off' => esc_html__('No', 'tpcore'),
                'return_value' => 'yes',
                'default' => '0',

            ]
        );
        $repeater->add_control(
            'tp_features_icon_type',
            [
                'label' => esc_html__('Select Icon Type', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'default' => 'icon',
                'options' => [
                    'image' => esc_html__('Image', 'tpcore'),
                    'icon' => esc_html__('Icon', 'tpcore'),
                    'svg' => esc_html__('SVG', 'tpcore'),
                ],
            ]
        );

        $repeater->add_control(
            'tp_features_image',
            [
                'label' => esc_html__('Upload Icon Image', 'tpcore'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'tp_features_icon_type' => 'image',
                ]

            ]
        );

        $repeater->add_control(
            'tp_features_icon_svg',
            [
                'show_label' => false,
                'type' => Controls_Manager::TEXTAREA,
                'label_block' => true,
                'placeholder' => esc_html__('SVG Code Here', 'tpcore'),
                'condition' => [
                    'tp_features_icon_type' => 'svg'
                ]
            ]
        );

        $repeater->add_control(
            'tp_features_icon',
            [
                'label' => esc_html__('Icon', 'tpcore'),
                'type' => Controls_Manager::ICONS,
                'default' => [
                    'value' => 'fas fa-smile',
                    'library' => 'fa-solid',
                ],
                'recommended' => [
                    'fa-solid' => [
                        'circle',
                        'dot-circle',
                        'square-full',
                    ],
                    'fa-regular' => [
                        'circle',
                        'dot-circle',
                        'square-full',
                    ],
                ],
                'condition' => [
                    'tp_features_icon_type' => 'icon',
                ],
            ]
        );

        $repeater->add_control(
            'feature_text',
            [
                'label' => __('Feature Text', 'tpcore'),
                'type' => Controls_Manager::TEXTAREA,
                'default' => __('Exciting Feature', 'tpcore'),
                'dynamic' => [
                    'active' => true
                ]
            ]
        );

        $this->add_control(
            'features_list',
            [
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'show_label' => false,
                'default' => [
                    [
                        'feature_text' => __('Standard Feature', 'tpcore'),
                        'tp_check_icon' => 'fa fa-check',
                    ],
                    [
                        'feature_text' => __('Another Great Feature', 'tpcore'),
                        'tp_check_icon' => 'fa fa-check',
                    ],
                    [
                        'feature_text' => __('Obsolete Feature', 'tpcore'),
                        'tp_check_icon' => 'fa fa-close',
                    ],
                ],
                'title_field' => '{{{ feature_text }}}',
            ]
        );

        $this->end_controls_section();

        // price_btn
        $this->tp_link_render_controls('price_btn', 'Button');

        // animation
        $this->tp_creative_animation(['layout-1']);
    }

    protected function style_tab_content()
    {
        $this->tp_section_style_controls('section', 'Section', '.tp-el-section');
        $this->tp_basic_style_controls('pricing_heading_title', 'Heading Title', '.tp-el-title');
        $this->tp_basic_style_controls('pricing_heading_desc', 'Heading Description', '.tp-el-desc');

        $this->tp_basic_style_controls('pricing_heading_symbol', 'Heading Symbol', '.tp-el-symbol');
        $this->tp_basic_style_controls('pricing_heading_price', 'Heading Price', '.tp-el-price');
        $this->tp_basic_style_controls('pricing_heading_month', 'Heading Month', '.tp-el-month');

        // feature item

        $this->start_controls_section(
            'tp_pricing_f_sec',
            [
                'label' => esc_html__('Feature Item', 'tpcore'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'tp_pricing_f_tab_typo',
                'label' => esc_html__('Active Typography', 'tpcore'),
                'selector' => '{{WRAPPER}} .tp-el-f-title',
            ]
        );
        $this->start_controls_tabs(
            'tp_pricing_f_tabs',
        );

        $this->start_controls_tab(
            'tp_pricing_f_tab_active',
            [
                'label' => esc_html__('Active', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_pricing_f_tab_active_color',
            [
                'label' => esc_html__('Color', 'tpcore'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-el-f-title' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->end_controls_tab();

        // inactive 
        $this->start_controls_tab(
            'tp_pricing_f_tab_inactive',
            [
                'label' => esc_html__('Inactive', 'tpcore'),
            ]
        );

        $this->add_control(
            'tp_pricing_f_tab_inactive_color',
            [
                'label' => esc_html__('Color', 'tpcore'),
                'type' => \Elementor\Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .tp-el-f-title.inactive' => 'color: {{VALUE}}',
                ],
            ]
        );
        $this->end_controls_tab();


        $this->end_controls_tabs();

        $this->end_controls_section();


        $this->tp_link_controls_style('', 'pricing_btn', 'Button', '.tp-el-price-btn');
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();
        ?>

        <?php if ($settings['tp_design_style'] == 'layout-2'):
            $control_id = 'price_btn';

            $animation = $this->tp_animation_show($settings);

            if ($settings['currency'] === 'custom') {
                $currency = $settings['currency_custom'];
            } else {
                $currency = self::get_currency_symbol($settings['currency']);
            }

            $is_active_price = $settings['tp_pricing_active_switcher'] == 'yes' ? true : false;
            $active_class = $is_active_price ? 'active' : '';

            $btn_class = $is_active_price ? 'tp-btn-4' : 'tp-btn-4';

            $this->tp_link_attributes_render('price_btn', '' . $btn_class . '  tp-el-price-btn tp-btn-4', $this->get_settings());
            ?>

            <div class="tp-pricing-5-item tp-pricing-inner-head tp-el-section <?php echo esc_attr($active_class); ?> mb-40 <?php echo esc_attr($animation['animation']); ?>"
                <?php echo $animation['duration'] . ' ' . $animation['delay']; ?>             <?php if ($is_active_price): ?>             <?php endif; ?>>
                <div class="tp-pricing-5-head tp-pricing-inner-head">

                    <?php if (!empty($settings['tp_pricing_header_title'])): ?>
                        <h4 class="tp-pricing-5-head-title tp-el-title">
                            <?php echo tp_kses($settings['tp_pricing_header_title']); ?>
                        </h4>
                    <?php endif; ?>

                    <?php if (!empty($settings['tp_pricing_header_subtitle'])): ?>
                        <span class="tp-el-desc">
                            <?php echo tp_kses($settings['tp_pricing_header_subtitle']); ?>
                        </span>
                    <?php endif; ?>

                    <h2 class="tp-pricing-5-price tp-el-price">
                        <span class="tp-price-monthly tp-el-symbol">
                            <?php echo esc_html($currency); ?>
                        </span>
                        <?php echo tp_kses($settings['price_main']); ?>
                        <b class="tp-el-month"><?php echo tp_kses($settings['price_duration']); ?></b>
                    </h2>
                </div>
                <div class="tp-pricing-5-list">
                    <ul>
                        <?php foreach ($settings['features_list'] as $key => $item):

                            $is_active_feature = $item['tp_feature_switcher'] == 'yes' ? true : false;
                            $active_feature_class = $is_active_feature ? 'inactive' : '';
                            ?>
                            <li class="tp-el-f-title <?php echo esc_attr($active_feature_class); ?>">
                                <?php tp_render_signle_icon_html($item, 'features') ?>
                                <?php echo tp_kses($item['feature_text']); ?>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </div>

                <?php if (!empty($settings['tp_' . $control_id . '_text']) && $settings['tp_' . $control_id . '_button_show'] == 'yes'): ?>
                    <div class="tp-pricing-inner-btn">
                        <a <?php echo $this->get_render_attribute_string('tp-button-arg' . $control_id . ''); ?>>
                            <?php echo $settings['tp_' . $control_id . '_text']; ?>
                        </a>
                    </div>
                <?php endif; ?>
            </div>

        <?php else:

            $control_id = 'price_btn';

            $animation = $this->tp_animation_show($settings);

            if ($settings['currency'] === 'custom') {
                $currency = $settings['currency_custom'];
            } else {
                $currency = self::get_currency_symbol($settings['currency']);
            }

            $is_active_price = $settings['tp_pricing_active_switcher'] == 'yes' ? true : false;
            $active_class = $is_active_price ? 'active' : '';

            $btn_class = $is_active_price ? 'tp-btn-4' : 'tp-btn-4';

            $this->tp_link_attributes_render('price_btn', '' . $btn_class . '  tp-el-price-btn tp-btn-4', $this->get_settings());
            ?>

            <div class="tp-pricing-5-item tp-el-section <?php echo esc_attr($active_class); ?> mb-40 <?php echo esc_attr($animation['animation']); ?>"
                <?php echo $animation['duration'] . ' ' . $animation['delay']; ?>             <?php if ($is_active_price): ?>             <?php endif; ?>>
                <div class="tp-pricing-5-head">
                    <div class="tp-pricing-5-head-text">

                        <?php if (!empty($settings['tp_pricing_header_title'])): ?>
                            <h4 class="tp-pricing-5-head-title tp-el-title">
                                <?php echo tp_kses($settings['tp_pricing_header_title']); ?>
                            </h4>
                        <?php endif; ?>


                        <?php if (!empty($settings['tp_pricing_header_subtitle'])): ?>
                            <span class="tp-el-desc">
                                <?php echo tp_kses($settings['tp_pricing_header_subtitle']); ?>
                            </span>
                        <?php endif; ?>

                    </div>

                    <h2 class="tp-pricing-5-price tp-el-price">
                        <span class="tp-price-monthly tp-el-symbol">
                            <?php echo esc_html($currency); ?>
                        </span>
                        <?php echo tp_kses($settings['price_main']); ?>
                        <b class=""><?php echo tp_kses($settings['price_duration']); ?></b>
                    </h2>
                </div>
                <div class="tp-pricing-5-list">
                    <ul>
                        <?php foreach ($settings['features_list'] as $key => $item):

                            $is_active_feature = $item['tp_feature_switcher'] == 'yes' ? true : false;
                            $active_feature_class = $is_active_feature ? 'inactive' : '';
                            ?>
                            <li class="tp-el-f-title <?php echo esc_attr($active_feature_class); ?>">
                                <?php tp_render_signle_icon_html($item, 'features') ?>
                                <?php echo tp_kses($item['feature_text']); ?>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                </div>

                <?php if (!empty($settings['tp_' . $control_id . '_text']) && $settings['tp_' . $control_id . '_button_show'] == 'yes'): ?>
                    <div class="tp-pricing-5-btn">
                        <a <?php echo $this->get_render_attribute_string('tp-button-arg' . $control_id . ''); ?>>
                            <?php echo $settings['tp_' . $control_id . '_text']; ?>
                        </a>
                    </div>
                <?php endif; ?>
            </div>

        <?php endif;
    }
}

$widgets_manager->register(new TP_Price_Box());