<?php

namespace TPCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use \Elementor\Utils;
use \Elementor\Group_Control_Image_Size;

if (!defined('ABSPATH'))
	exit; // Exit if accessed directly

/**
 * Tp Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TP_Video_Pop_Up extends Widget_Base
{

	use \TPCore\Widgets\TP_Style_Trait;
	use \TPCore\Widgets\TP_Animation_Trait;

	/**
	 * Retrieve the widget name.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget name.
	 */
	public function get_name()
	{
		return 'tp-video-pop-up';
	}

	/**
	 * Retrieve the widget title.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget title.
	 */
	public function get_title()
	{
		return __('Video Pop Up', 'tpcore');
	}

	/**
	 * Retrieve the widget icon.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return string Widget icon.
	 */
	public function get_icon()
	{
		return 'tp-icon';
	}

	/**
	 * Retrieve the list of categories the widget belongs to.
	 *
	 * Used to determine where to display the widget in the editor.
	 *
	 * Note that currently Elementor supports only one category.
	 * When multiple categories passed, Elementor uses the first one.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget categories.
	 */
	public function get_categories()
	{
		return ['tpcore'];
	}

	/**
	 * Retrieve the list of scripts the widget depended on.
	 *
	 * Used to set scripts dependencies required to run the widget.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 *
	 * @return array Widget scripts dependencies.
	 */
	public function get_script_depends()
	{
		return ['tpcore'];
	}

	/**
	 * Register the widget controls.
	 *
	 * Adds different input fields to allow the user to change and customize the widget settings.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */

	protected function register_controls()
	{
		$this->register_controls_section();
		$this->style_tab_content();
	}

	protected function register_controls_section()
	{

		// layout Panel
		$this->tp_design_layout('Layout Style', 2);


		// layout Panel
		$this->start_controls_section(
			'video_content',
			[
				'label' => esc_html__('Video Content', 'tpcore'),
			]
		);

		$this->add_control(
			'tp_video_url',
			[
				'label' => esc_html__('Video URL', 'tpcore'),
				'type' => Controls_Manager::TEXTAREA,
				'default' => 'https://html.hixstudio.net/videos/acadia/acadia.mp4',
				'label_block' => true,
				'description' => __("We recommended to put 'mp4' format video.", 'tpcore')
			]
		);

		$this->add_control(
			'tp_tag_icon_type',
			[
				'label' => esc_html__('Video Icon Type', 'tpcore'),
				'type' => Controls_Manager::CHOOSE,
				'label_block' => false,
				'options' => [
					'icon' => [
						'title' => esc_html__('Icon', 'tpcore'),
						'icon' => 'eicon-nerd-wink',
					],
					'image' => [
						'title' => esc_html__('Image', 'tpcore'),
						'icon' => 'fa fa-image',
					],
					'svg' => [
						'title' => esc_html__('Svg', 'tpcore'),
						'icon' => 'fas fa-code',
					],
				],
				'default' => 'icon',
				'toggle' => false,
				'style_transfer' => true,
			]
		);

		$this->add_control(
			'image',
			[
				'label' => esc_html__('Image', 'tpcore'),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
				'condition' => [
					'tp_tag_icon_type' => 'image',
				],
			]
		);

		$this->add_control(
			'icon',
			[
				'label' => esc_html__('Icon', 'tpcore'),
				'type' => Controls_Manager::ICONS,
				'default' => [
					'value' => 'fas fa-star',
					'library' => 'solid',

				],
				'condition' => [
					'tp_tag_icon_type' => 'icon',
				]
			]
		);

		$this->add_control(
			'svg',
			[
				'type' => Controls_Manager::TEXTAREA,
				'label_block' => true,
				'label' => __('Svg Code', 'tpcore'),
				'default' => __('Svg Code Here', 'tpcore'),
				'placeholder' => __('Type Svg Code here', 'tpcore'),
				'dynamic' => [
					'active' => true,
				],
				'condition' => [
					'tp_tag_icon_type' => 'svg',
				],
			]
		);



		$this->add_control(
			'tp_video_image',
			[
				'label' => esc_html__('Video Image', 'tpcore'),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				]
			]
		);

		$this->end_controls_section();
	}

	protected function style_tab_content()
	{
		$this->tp_section_style_controls('section', 'Section - Style', '.tp-el-section');

		$this->start_controls_section(
		 'tp_image_style_section',
			 [
			   'label' => esc_html__( 'Image Control', 'textdomain' ),
			   'tab'   => \Elementor\Controls_Manager::TAB_STYLE,
			 ]
		);
		

		$this->add_responsive_control(
				'tp_image_width',
				[
					'label' => esc_html__( 'Image Width', 'textdomain' ),
					'type' => \Elementor\Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%', 'custom' ],
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 1000,
							'step' => 5,
						],
						'%' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default' => [
						'unit' => '%',
						'size' => 100,
					],
					'selectors' => [
						'{{WRAPPER}} .tp-el-width' => 'width: {{SIZE}}{{UNIT}}; object-fit: cover;',
					],
				]
			);
		
		
		$this->end_controls_section();

		$this->tp_icon_style('layout-1', 'video_Ply', '.tp-el-play-icon');

		$this->start_controls_section(
			'tp_v_P_btn_section',
			[
				'label' => esc_html__('Play Button', 'tpcore'),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
				'condition' => [
					'tp_design_style' => 'layout-2'
				]
			]
		);

		$this->add_control(
			'tp_v_P_btn_icon_color',
			[
				'label' => esc_html__('Icon Color', 'tpcore'),
				'type' => \Elementor\Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .tp_v_P_btn svg' => 'color: {{VALUE}}',
				],
			]
		);
		$this->end_controls_section();
	}

	/**
	 * Render the widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 *
	 * @access protected
	 */
	protected function render()
	{
		$settings = $this->get_settings_for_display();
		?>

		<?php if ($settings['tp_design_style'] == 'layout-2'):?>
		<?php else:
			$img = tp_get_img($settings, 'tp_video_image', 'full', false);
			?>
			<?php if (!empty($settings['tp_video_url'])): ?>
				<?php if (!empty($img['tp_video_image'])): ?>
					<div class="tp-practice-yoga-thumb p-relative">
						<img class="tp-el-width" src="<?php echo esc_url($img['tp_video_image']) ?>" alt="<?php echo esc_url($img['tp_video_image_alt']) ?>">
					<?php endif; ?>

					<div class="tp-practice-yoga-video tp-el-section">
						<a class="video-border-animation popup-video tp_v_P_btn" href="<?php echo esc_url($settings['tp_video_url']); ?>">
							<span class="tp-el-play-icon">
								<?php if ($settings['tp_tag_icon_type'] === 'image' && ($settings['image']['url'] || $settings['image']['id'])):
									$this->get_render_attribute_string('image');
									$settings['hover_animation'] = 'disable-animation';
									?>
									<?php echo Group_Control_Image_Size::get_attachment_image_html($settings, 'image'); ?>
								<?php elseif (!empty($settings['icon'])): ?>
									<?php \Elementor\Icons_Manager::render_icon($settings['icon'], ['aria-hidden' => 'true']); ?>
								<?php elseif (!empty($settings['svg'])): ?>
									<?php echo $settings['svg']; ?>
								<?php endif; ?>
							</span>
						</a>
					</div>
				</div>
			<?php endif; ?>
		<?php endif;
	}
}

$widgets_manager->register(new TP_Video_Pop_Up());
