<?php
namespace TPCore\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;

if (!defined('ABSPATH'))
    exit; // Exit if accessed directly

/**
 * Tp Core
 *
 * Elementor widget for hello world.
 *
 * @since 1.0.0
 */
class TP_Blog_Post extends Widget_Base
{

    use TP_Style_Trait, TP_Query_Trait, TP_Column_Trait, TP_Animation_Trait;

    /**
     * Retrieve the widget name.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget name.
     */
    public function get_name()
    {
        return 'tp-blog-post';
    }

    /**
     * Retrieve the widget title.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget title.
     */
    public function get_title()
    {
        return __('Blog Post', 'tpcore');
    }

    /**
     * Retrieve the widget icon.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return string Widget icon.
     */
    public function get_icon()
    {
        return 'tp-icon';
    }

    /**
     * Retrieve the list of categories the widget belongs to.
     *
     * Used to determine where to display the widget in the editor.
     *
     * Note that currently Elementor supports only one category.
     * When multiple categories passed, Elementor uses the first one.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget categories.
     */
    public function get_categories()
    {
        return ['tpcore'];
    }

    /**
     * Retrieve the list of scripts the widget depended on.
     *
     * Used to set scripts dependencies required to run the widget.
     *
     * @since 1.0.0
     *
     * @access public
     *
     * @return array Widget scripts dependencies.
     */
    public function get_script_depends()
    {
        return ['tpcore'];
    }

    /**
     * Register the widget controls.
     *
     * Adds different input fields to allow the user to change and customize the widget settings.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function register_controls()
    {
        $this->register_controls_section();
        $this->style_tab_content();
    }

    protected function register_controls_section()
    {

        // layout Panel
        $this->start_controls_section(
            'tp_layout',
            [
                'label' => esc_html__('Design Layout', 'tpcore'),
            ]
        );
        $this->add_control(
            'tp_design_style',
            [
                'label' => esc_html__('Select Layout', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'layout-1' => esc_html__('Layout 1', 'tpcore'),
                    'layout-2' => esc_html__('Layout 2', 'tpcore'),
                    'layout-3' => esc_html__('Layout 3', 'tpcore'),
                    'layout-4' => esc_html__('Layout 4', 'tpcore'),
                ],
                'default' => 'layout-1',

            ]
        );

        $this->add_control(
            'tp_item_style',
            [
                'label' => esc_html__('Item Style', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'default' => esc_html__('Default', 'tpcore'),
                    'tp-blog-6-item' => esc_html__('Border Style', 'tpcore'),
                    'tp-blog-chef-wrap' => esc_html__('Box Shadow Style', 'tpcore'),
                ],
                'default' => 'default',
                'condition' => [
                    'tp_design_style' => 'layout-2',
                ]
            ]
        );

        $this->end_controls_section();

        // section column
        $this->tp_columns('col', ['layout-1', 'layout-2']);

        // Query Panel
        $this->tp_query_controls('blog', 'Blog', 'post', 'category', 6, 10, 6, 0, 'date', 'desc', true, true, true, true);

        $this->tp_query_meta_controls('blog_grid_meta', 'Meta Controls', true, true, true, '');

        // animation
        $this->tp_creative_animation(['layout-1', 'layout-2']);
    }

    // style_tab_content
    protected function style_tab_content()
    {
        $this->tp_section_style_controls('section', 'Section - Style', '.tp-el-section');
        $this->tp_basic_style_controls('blog_post_tag_title', 'Tag Title', '.tp-el-tag-title');
        $this->tp_basic_style_controls('blog_post_title', 'Title', '.tp-el-title');
        $this->tp_basic_style_controls('blog_post_meta', 'Meta Text', '.tp-el-meta-text-title > span');
        $this->tp_basic_style_controls('blog_post_author', 'Author Name', '.tp-el-avator-name');

        $this->start_controls_section(
            'tp_blog_post_avator_section',
            [
                'label' => esc_html__('Avator Image', 'tpcore'),
                'tab' => \Elementor\Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_responsive_control(
            'tp_blog_post_avator_w',
            [
                'type' => \Elementor\Controls_Manager::SLIDER,
                'label' => esc_html__('Image Width', 'tpcore'),
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 1,
                        'max' => 2000,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .tp-el-avator img' => 'min-width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );
        $this->add_responsive_control(
            'tp_blog_post_avator_h',
            [
                'type' => \Elementor\Controls_Manager::SLIDER,
                'label' => esc_html__('Image Height', 'tpcore'),
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 1,
                        'max' => 2000,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .tp-el-avator' => 'min-height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'tp_image_section_padding',
            [
                'label' => esc_html__('Border Radius', 'tpcore'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .tp-el-avator img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();

        $this->start_controls_section(
            'tp_blog_post_image',
            [
                'label' => esc_html__('Post Image', 'tpcore'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'tp_image_section_border_radius',
            [
                'label' => esc_html__('Border Radius', 'tpcore'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%', 'em'],
                'selectors' => [
                    '{{WRAPPER}} .tp-el-post-img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'tp_blog_post_avator_widths',
            [
                'type' => Controls_Manager::SLIDER,
                'label' => esc_html__('Image Width', 'tpcore'),
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 1,
                        'max' => 2000,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .tp-el-post-img img' => 'width: {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'tp_design_style!' => 'layout-4',
                ],
            ]
        );

        $this->add_responsive_control(
            'tp_blog_post_avato_width',
            [
                'type' => Controls_Manager::SLIDER,
                'label' => esc_html__('Image Width', 'tpcore'),
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 1,
                        'max' => 2000,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .tp-el-post-img img' => 'width: {{SIZE}}{{UNIT}}; flex: 0 0 {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .tp-el-post-img' => 'width: {{SIZE}}{{UNIT}}; flex: 0 0 {{SIZE}}{{UNIT}};',
                ],
                'condition' => [
                    'tp_design_style' => 'layout-4',
                ],
            ]
        );

        $this->add_responsive_control(
            'tp_blog_post_avator_heights',
            [
                'type' => Controls_Manager::SLIDER,
                'label' => esc_html__('Image Height', 'tpcore'),
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 1,
                        'max' => 2000,
                    ],
                ],
                'selectors' => [
                    '{{WRAPPER}} .tp-el-post-img img' => 'height: {{SIZE}}{{UNIT}};',
                    '{{WRAPPER}} .tp-el-post-img' => 'height: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_control(
            'tp_blog_post_object_fit',
            [
                'label' => esc_html__('Object Fit', 'tpcore'),
                'type' => Controls_Manager::SELECT,
                'options' => [
                    'fill' => esc_html__('Fill', 'tpcore'),
                    'contain' => esc_html__('Contain', 'tpcore'),
                    'cover' => esc_html__('Cover', 'tpcore'),
                    'none' => esc_html__('None', 'tpcore'),
                    'scale-down' => esc_html__('Scale Down', 'tpcore'),
                ],
                'default' => 'cover',
                'selectors' => [
                    '{{WRAPPER}} .tp-el-post-img img' => 'object-fit: {{VALUE}};',
                ]
            ]
        );

        $this->end_controls_section();
    }

    /**
     * Render the widget output on the frontend.
     *
     * Written in PHP and used to generate the final HTML.
     *
     * @since 1.0.0
     *
     * @access protected
     */
    protected function render()
    {
        $settings = $this->get_settings_for_display();

        ?>

        <?php if ($settings['tp_design_style'] == 'layout-4'):
            $query_args = tp_query_args('post', 'category', $this->get_settings());
            $the_query = new \WP_Query($query_args);

            ?>
            <div class="tp-blog-12-item-2-wrapper">

                <?php
                if ($the_query->have_posts()):
                    while ($the_query->have_posts()):
                        $the_query->the_post();

                        $categories = get_the_category();
                        ?>
                        <div class="tp-blog-12-item-2 d-flex">

                            <?php if (has_post_thumbnail()): ?>
                                <div class="tp-blog-12-item-2-thumb tp-el-post-img">
                                    <a href="<?php the_permalink(); ?>">
                                        <?php the_post_thumbnail(); ?>
                                    </a>
                                </div>
                            <?php endif; ?>

                            <div class="tp-blog-12-item-2-content">
                                <div class="tp-blog-12-item-tag">
                                    <p>
                                        <?php if (!empty($settings['tp_post_category'])): ?>
                                            <?php if (!empty($categories[0]->name)): ?>
                                                <a class="tp-el-tag-title"
                                                    href="<?php print esc_url(get_category_link($categories[0]->term_id)); ?>">
                                                    <span>
                                                        <?php echo esc_html($categories[0]->name); ?>
                                                    </span>
                                                </a>
                                            <?php endif; ?>
                                        <?php endif; ?>

                                        <?php if ($settings['tp_post_date'] == 'yes'): ?>
                                            <?php if (!empty($settings['tp_post_date_format'])):
                                                $date_format = $settings['tp_post_date_format'] == 'default' ? get_option('date_format') : $settings['tp_post_date_format'];
                                                $date_format = $settings['tp_post_date_format'] == 'custom' ? $settings['tp_post_date_custom_format'] : $date_format;
                                                ?>
                                                <?php the_time($date_format); ?>
                                            <?php endif; ?>
                                        <?php endif; ?>
                                    </p>
                                </div>

                                <h3 class="tp-blog-12-item-title tp-el-title">
                                    <a href="<?php the_permalink(); ?>">
                                        <?php echo wp_trim_words(get_the_title(), $settings['tp_post_title_word'], '...'); ?>
                                    </a>
                                </h3>

                                <?php if (!empty($settings['post_read_more_btn_text'])): ?>
                                    <div class="tp-blog-12-item-btn">
                                        <a href="<?php the_permalink(); ?>">
                                            <?php echo tp_kses($settings['post_read_more_btn_text']); ?>
                                            <span>
                                                <svg xmlns="http://www.w3.org/2000/svg" width="13" height="12" viewBox="0 0 13 12" fill="none">
                                                    <path d="M8.07422 1.73193L12.0019 6.01664L8.07422 10.3013M1 6.01709H11.89"
                                                        stroke="currentColor" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round"
                                                        stroke-linejoin="round" />
                                                </svg>
                                            </span>
                                        </a>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php
                    endwhile;
                    wp_reset_postdata();
                endif;
                ?>

                <?php if (($settings['tp_post_pagination'] == 'yes') && ('-1' != $settings['posts_per_page'])): ?>
                    <div class="row">
                        <div class="col-12">
                            <div class="basic-pagination tp-pagination tp-el-pagination-alignment">
                                <?php
                                $big = 999999999;

                                if (get_query_var('paged')) {
                                    $paged = get_query_var('paged');
                                } else if (get_query_var('page')) {
                                    $paged = get_query_var('page');
                                } else {
                                    $paged = 1;
                                }

                                echo paginate_links(
                                    array(
                                        'base' => str_replace($big, '%#%', get_pagenum_link($big)),
                                        'format' => '?paged=%#%',
                                        'current' => $paged,
                                        'total' => $the_query->max_num_pages,
                                        'type' => 'list',
                                        'prev_text' => '<i class="fa-regular fa-arrow-left icon"></i>',
                                        'next_text' => '<i class="fa-regular fa-arrow-right icon"></i>',
                                        'show_all' => false,
                                        'end_size' => 1,
                                        'mid_size' => 4,
                                    )
                                );
                                ?>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
            </div>

        <?php elseif ($settings['tp_design_style'] == 'layout-3'):
            $query_args = tp_query_args('post', 'category', $this->get_settings());
            $the_query = new \WP_Query($query_args);

            ?>

            <div class="tp-acadia-blog-12-wrapper">
                <?php
                if ($the_query->have_posts()):
                    while ($the_query->have_posts()):
                        $the_query->the_post();

                        $categories = get_the_category();
                        ?>

                        <div class="tp-blog-12-item">

                            <?php if (has_post_thumbnail()): ?>
                                <div class="tp-blog-12-item-thumb tp-el-post-img">
                                    <a href="<?php the_permalink(); ?>">
                                        <?php the_post_thumbnail(); ?>
                                    </a>
                                </div>
                            <?php endif; ?>

                            <div class="tp-blog-12-item-content">
                                <div class="tp-blog-12-item-tag">
                                    <p>
                                        <?php if (!empty($settings['tp_post_category'])): ?>
                                            <?php if (!empty($categories[0]->name)): ?>
                                                <a class="tp-el-tag-title"
                                                    href="<?php print esc_url(get_category_link($categories[0]->term_id)); ?>">
                                                    <span>
                                                        <?php echo esc_html($categories[0]->name); ?>
                                                    </span>
                                                </a>
                                            <?php endif; ?>
                                        <?php endif; ?>

                                        <?php if ($settings['tp_post_date'] == 'yes'): ?>
                                            <?php if (!empty($settings['tp_post_date_format'])):
                                                $date_format = $settings['tp_post_date_format'] == 'default' ? get_option('date_format') : $settings['tp_post_date_format'];
                                                $date_format = $settings['tp_post_date_format'] == 'custom' ? $settings['tp_post_date_custom_format'] : $date_format;
                                                ?>
                                                <?php the_time($date_format); ?>
                                            <?php endif; ?>
                                        <?php endif; ?>
                                    </p>
                                </div>

                                <h3 class="tp-blog-12-item-title tp-el-title">
                                    <a href="<?php the_permalink(); ?>">
                                        <?php echo wp_trim_words(get_the_title(), $settings['tp_post_title_word'], '...'); ?>
                                    </a>
                                </h3>

                                <?php if (!empty($settings['post_read_more_btn_text'])): ?>
                                    <div class="tp-blog-12-item-btn">
                                        <a href="<?php the_permalink(); ?>">
                                            <?php echo tp_kses($settings['post_read_more_btn_text']); ?>
                                            <span>
                                                <svg xmlns="http://www.w3.org/2000/svg" width="13" height="12" viewBox="0 0 13 12" fill="none">
                                                    <path d="M8.07422 1.73193L12.0019 6.01664L8.07422 10.3013M1 6.01709H11.89"
                                                        stroke="currentColor" stroke-width="1.5" stroke-miterlimit="10" stroke-linecap="round"
                                                        stroke-linejoin="round" />
                                                </svg>
                                            </span>
                                        </a>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                        <?php
                    endwhile;
                    wp_reset_postdata();
                endif;
                ?>

                <?php if (($settings['tp_post_pagination'] == 'yes') && ('-1' != $settings['posts_per_page'])): ?>
                    <div class="row">
                        <div class="col-12">
                            <div class="basic-pagination tp-pagination tp-el-pagination-alignment">
                                <?php
                                $big = 999999999;

                                if (get_query_var('paged')) {
                                    $paged = get_query_var('paged');
                                } else if (get_query_var('page')) {
                                    $paged = get_query_var('page');
                                } else {
                                    $paged = 1;
                                }

                                echo paginate_links(
                                    array(
                                        'base' => str_replace($big, '%#%', get_pagenum_link($big)),
                                        'format' => '?paged=%#%',
                                        'current' => $paged,
                                        'total' => $the_query->max_num_pages,
                                        'type' => 'list',
                                        'prev_text' => '<i class="fa-regular fa-arrow-left icon"></i>',
                                        'next_text' => '<i class="fa-regular fa-arrow-right icon"></i>',
                                        'show_all' => false,
                                        'end_size' => 1,
                                        'mid_size' => 4,
                                    )
                                );
                                ?>
                            </div>
                        </div>
                    </div>
                <?php endif; ?>
            </div>

        <?php elseif ($settings['tp_design_style'] == 'layout-2'):
            $query_args = tp_query_args('post', 'category', $this->get_settings());
            $the_query = new \WP_Query($query_args);

            $animation = $this->tp_animation_show($settings);

            ?>
            <div class="row">
                <?php

                if ($the_query->have_posts()):
                    while ($the_query->have_posts()):
                        $the_query->the_post();

                        $categories = get_the_category();
                        ?>

                        <div class="<?php echo esc_attr($this->col_show($settings)); ?>">
                            <div class="tp-el-section tp-blog-stories-item p-relative mb-50 <?php echo esc_attr($settings['tp_item_style']); ?> <?php echo esc_attr($animation['animation']); ?>"
                                <?php echo $animation['duration'] . ' ' . $animation['delay']; ?>>
                                <?php if (has_post_thumbnail()): ?>
                                    <div class="tp-blog-stories-thumb tp-el-post-img">
                                        <a href="<?php the_permalink(); ?>">
                                            <?php the_post_thumbnail(); ?>
                                        </a>
                                    </div>
                                <?php endif; ?>

                                <div class="tp-blog-stories-content">

                                    <div class="tp-blog-stories-tag-wrap tp-el-meta-text-title d-flex">
                                        <?php if (!empty($settings['tp_post_category'])): ?>
                                            <?php if (!empty($categories[0]->name)):
                                                $color = get_term_meta($categories[0]->term_id, '_acadia_post_cat_color', true)
                                                    ?>
                                                <a data-bg-color="<?php echo esc_attr($color); ?>"
                                                    class="tp-blog-tag tp-blog-categorize tp-el-tag-title"
                                                    href="<?php print esc_url(get_category_link($categories[0]->term_id)); ?>">
                                                    <?php echo esc_html($categories[0]->name); ?>
                                                </a>
                                            <?php endif; ?>
                                        <?php endif; ?>

                                        <?php if ($settings['tp_post_date'] == 'yes'): ?>
                                            <?php if (!empty($settings['tp_post_date_format'])):
                                                $date_format = $settings['tp_post_date_format'] == 'default' ? get_option('date_format') : $settings['tp_post_date_format'];
                                                $date_format = $settings['tp_post_date_format'] == 'custom' ? $settings['tp_post_date_custom_format'] : $date_format;
                                                ?>
                                                <span>
                                                    <?php the_time($date_format); ?>
                                                </span>
                                            <?php endif; ?>
                                        <?php endif; ?>
                                    </div>

                                    <h4 class="tp-blog-stories-title tp-el-title">
                                        <a href="<?php the_permalink(); ?>">
                                            <?php echo wp_trim_words(get_the_title(), $settings['tp_post_title_word'], '...'); ?>
                                        </a>
                                    </h4>

                                    <?php if (!empty($settings['tp_post_author'] == 'yes')): ?>
                                        <div class="tp-blog-stories-user-box d-flex align-items-center">
                                            <div class="tp-blog-stories-user-thumb tp-el-avator">
                                                <img class="" src="<?php print get_avatar_url(get_the_author_meta('ID')); ?>" alt="img-blog">
                                            </div>
                                            <span class="tp-el-avator-name">
                                                <?php print get_the_author(); ?>
                                            </span>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>
                        </div>
                        <?php
                    endwhile;
                    wp_reset_postdata();
                endif;
                ?>
            </div>

            <?php if (($settings['tp_post_pagination'] == 'yes') && ('-1' != $settings['posts_per_page'])): ?>
                <div class="row">
                    <div class="col-12">
                        <div class="basic-pagination tp-pagination tp-el-pagination-alignment">
                            <?php
                            $big = 999999999;

                            if (get_query_var('paged')) {
                                $paged = get_query_var('paged');
                            } else if (get_query_var('page')) {
                                $paged = get_query_var('page');
                            } else {
                                $paged = 1;
                            }

                            echo paginate_links(
                                array(
                                    'base' => str_replace($big, '%#%', get_pagenum_link($big)),
                                    'format' => '?paged=%#%',
                                    'current' => $paged,
                                    'total' => $the_query->max_num_pages,
                                    'type' => 'list',
                                    'prev_text' => '<i class="fa-regular fa-arrow-left icon"></i>',
                                    'next_text' => '<i class="fa-regular fa-arrow-right icon"></i>',
                                    'show_all' => false,
                                    'end_size' => 1,
                                    'mid_size' => 4,
                                )
                            );
                            ?>
                        </div>
                    </div>
                </div>
            <?php endif; ?>

        <?php else:

            $query_args = tp_query_args('post', 'category', $this->get_settings());
            $the_query = new \WP_Query($query_args);

            $animation = $this->tp_animation_show($settings);

            ?>
            <div class="row">
                <?php

                if ($the_query->have_posts()):
                    while ($the_query->have_posts()):
                        $the_query->the_post();

                        $categories = get_the_category();
                        ?>

                        <div class="<?php echo esc_attr($this->col_show($settings)); ?>">
                            <div class="tp-blog-item text-center tp-el-section mb-40 <?php echo esc_attr($animation['animation']); ?>" <?php echo $animation['duration'] . ' ' . $animation['delay']; ?>>
                                <?php if (has_post_thumbnail()): ?>
                                    <div class="tp-blog-thumb fix">
                                        <a href="<?php the_permalink(); ?>">
                                            <?php the_post_thumbnail(); ?>
                                        </a>
                                    </div>
                                <?php endif; ?>

                                <div class="tp-blog-content">

                                    <?php if (!empty($settings['tp_post_category'])): ?>
                                        <?php if (!empty($categories[0]->name)): ?>
                                            <span class="tp-blog-tag tp-el-tag-title">
                                                <a href="<?php print esc_url(get_category_link($categories[0]->term_id)); ?>">
                                                    <?php echo esc_html($categories[0]->name); ?>
                                                </a>
                                            </span>
                                        <?php endif; ?>
                                    <?php endif; ?>

                                    <h4 class="tp-blog-title tp-el-title">
                                        <a href="<?php the_permalink(); ?>">
                                            <?php echo wp_trim_words(get_the_title(), $settings['tp_post_title_word'], '...'); ?>
                                        </a>
                                    </h4>

                                    <div class="tp-blog-meta tp-el-meta-text-title">
                                        <?php if (!empty($settings['tp_post_author'] == 'yes')): ?>
                                            <span>
                                                <span>
                                                    <a href="<?php print esc_url(get_author_posts_url(get_the_author_meta('ID'))); ?>">
                                                        <svg width="14" height="16" viewBox="0 0 14 16" fill="none"
                                                            xmlns="http://www.w3.org/2000/svg">
                                                            <path
                                                                d="M13 14.5V13C13 12.2044 12.6839 11.4413 12.1213 10.8787C11.5587 10.3161 10.7956 10 10 10H4C3.20435 10 2.44129 10.3161 1.87868 10.8787C1.31607 11.4413 1 12.2044 1 13V14.5"
                                                                stroke="currentColor" stroke-width="1.5" stroke-linecap="round"
                                                                stroke-linejoin="round" />
                                                            <path
                                                                d="M7 7C8.65685 7 10 5.65685 10 4C10 2.34315 8.65685 1 7 1C5.34315 1 4 2.34315 4 4C4 5.65685 5.34315 7 7 7Z"
                                                                stroke="currentColor" stroke-width="1.5" stroke-linecap="round"
                                                                stroke-linejoin="round" />
                                                        </svg>
                                                        <?php print get_the_author(); ?>
                                                    </a>
                                                </span>
                                            </span>
                                        <?php endif; ?>

                                        <?php if ($settings['tp_post_date'] == 'yes'): ?>
                                            <?php if (!empty($settings['tp_post_date_format'])):
                                                $date_format = $settings['tp_post_date_format'] == 'default' ? get_option('date_format') : $settings['tp_post_date_format'];
                                                $date_format = $settings['tp_post_date_format'] == 'custom' ? $settings['tp_post_date_custom_format'] : $date_format;
                                                ?>
                                                <span>
                                                    <span>
                                                        <svg width="15" height="15" viewBox="0 0 15 15" fill="none"
                                                            xmlns="http://www.w3.org/2000/svg">
                                                            <path
                                                                d="M7.5 14C11.0899 14 14 11.0899 14 7.5C14 3.91015 11.0899 1 7.5 1C3.91015 1 1 3.91015 1 7.5C1 11.0899 3.91015 14 7.5 14Z"
                                                                stroke="currentColor" stroke-width="1.5" stroke-linecap="round"
                                                                stroke-linejoin="round" />
                                                            <path d="M7.5 3.6001V7.5001L10.1 8.8001" stroke="currentColor" stroke-width="1.5"
                                                                stroke-linecap="round" stroke-linejoin="round" />
                                                        </svg>
                                                        <?php the_time($date_format); ?>
                                                    </span>
                                                </span>
                                            <?php endif; ?>
                                        <?php endif; ?>
                                    </div>
                                    <div class="tp-blog-btn">
                                        <a href="<?php the_permalink(); ?>">
                                            <span class="tp-blog-btn-bg blog-btn-bg"></span>
                                            <span class="tp-blog-btn-border blog-btn-border"></span>
                                            <span class="icon">
                                                <svg width="12" height="12" viewBox="0 0 12 12" fill="none"
                                                    xmlns="http://www.w3.org/2000/svg">
                                                    <path d="M1 6H11" stroke="currentColor" stroke-width="2" stroke-linecap="round"
                                                        stroke-linejoin="round" />
                                                    <path d="M6 1L11 6L6 11" stroke="currentColor" stroke-width="2" stroke-linecap="round"
                                                        stroke-linejoin="round" />
                                                </svg>
                                            </span>
                                        </a>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <?php
                    endwhile;
                    wp_reset_postdata();
                endif;
                ?>
            </div>

            <?php if (($settings['tp_post_pagination'] == 'yes') && ('-1' != $settings['posts_per_page'])): ?>
                <div class="row">
                    <div class="col-12">
                        <div class="basic-pagination tp-pagination tp-el-pagination-alignment">
                            <?php
                            $big = 999999999;

                            if (get_query_var('paged')) {
                                $paged = get_query_var('paged');
                            } else if (get_query_var('page')) {
                                $paged = get_query_var('page');
                            } else {
                                $paged = 1;
                            }

                            echo paginate_links(
                                array(
                                    'base' => str_replace($big, '%#%', get_pagenum_link($big)),
                                    'format' => '?paged=%#%',
                                    'current' => $paged,
                                    'total' => $the_query->max_num_pages,
                                    'type' => 'list',
                                    'prev_text' => '<i class="fa-regular fa-arrow-left icon"></i>',
                                    'next_text' => '<i class="fa-regular fa-arrow-right icon"></i>',
                                    'show_all' => false,
                                    'end_size' => 1,
                                    'mid_size' => 4,
                                )
                            );
                            ?>
                        </div>
                    </div>
                </div>
            <?php endif; ?>

        <?php endif;
    }

}

$widgets_manager->register(new TP_Blog_Post());