/*! iScroll v5.2.0 ~ (c) 2008-2016 Matteo Spinelli ~ http://cubiq.org/license */
(function (window, document, Math) {
	var rAF = window.requestAnimationFrame	||
			  window.webkitRequestAnimationFrame	||
			  window.mozRequestAnimationFrame		||
			  window.oRequestAnimationFrame		||
			  window.msRequestAnimationFrame		||
			  function (callback) { window.setTimeout(callback, 1000 / 60); };

	var utils = (function () {
		var me = {};

		var _elementStyle = document.createElement('div').style;
		var _vendor = (function () {
			var vendors = ['t', 'webkitT', 'MozT', 'msT', 'OT'],
				transform,
				i = 0,
				l = vendors.length;

			for ( ; i < l; i++ ) {
				transform = vendors[i] + 'ransform';
				if ( transform in _elementStyle ) return vendors[i].substr(0, vendors[i].length-1);
			}

			return false;
		})();

		function _prefixStyle (style) {
			if ( _vendor === false ) return false;
			if ( _vendor === '' ) return style;
			return _vendor + style.charAt(0).toUpperCase() + style.substr(1);
		}

		me.getTime = Date.now || function getTime () { return new Date().getTime(); };

		me.extend = function (target, obj) {
			for ( var i in obj ) {
				target[i] = obj[i];
			}
		};

		me.addEvent = function (el, type, fn, capture) {
			el.addEventListener(type, fn, !!capture);
		};

		me.removeEvent = function (el, type, fn, capture) {
			el.removeEventListener(type, fn, !!capture);
		};

		me.prefixPointerEvent = function (pointerEvent) {
			return window.MSPointerEvent ?
			'MSPointer' + pointerEvent.charAt(7).toUpperCase() + pointerEvent.substr(8):
				pointerEvent;
		};

		me.momentum = function (current, start, time, lowerMargin, wrapperSize, deceleration) {
			var distance = current - start,
				speed = Math.abs(distance) / time,
				destination,
				duration;

			deceleration = deceleration === undefined ? 0.0006 : deceleration;

			destination = current + ( speed * speed ) / ( 2 * deceleration ) * ( distance < 0 ? -1 : 1 );
			duration = speed / deceleration;

			if ( destination < lowerMargin ) {
				destination = wrapperSize ? lowerMargin - ( wrapperSize / 2.5 * ( speed / 8 ) ) : lowerMargin;
				distance = Math.abs(destination - current);
				duration = distance / speed;
			} else if ( destination > 0 ) {
				destination = wrapperSize ? wrapperSize / 2.5 * ( speed / 8 ) : 0;
				distance = Math.abs(current) + destination;
				duration = distance / speed;
			}

			return {
				destination: Math.round(destination),
				duration: duration
			};
		};

		var _transform = _prefixStyle('transform');

		me.extend(me, {
			hasTransform: _transform !== false,
			hasPerspective: _prefixStyle('perspective') in _elementStyle,
			hasTouch: 'ontouchstart' in window,
			hasPointer: !!(window.PointerEvent || window.MSPointerEvent), // IE10 is prefixed
			hasTransition: _prefixStyle('transition') in _elementStyle
		});

		/*
		 This should find all Android browsers lower than build 535.19 (both stock browser and webview)
		 - galaxy S2 is ok
		 - 2.3.6 : `AppleWebKit/533.1 (KHTML, like Gecko) Version/4.0 Mobile Safari/533.1`
		 - 4.0.4 : `AppleWebKit/534.30 (KHTML, like Gecko) Version/4.0 Mobile Safari/534.30`
		 - galaxy S3 is badAndroid (stock brower, webview)
		 `AppleWebKit/534.30 (KHTML, like Gecko) Version/4.0 Mobile Safari/534.30`
		 - galaxy S4 is badAndroid (stock brower, webview)
		 `AppleWebKit/534.30 (KHTML, like Gecko) Version/4.0 Mobile Safari/534.30`
		 - galaxy S5 is OK
		 `AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Mobile Safari/537.36 (Chrome/)`
		 - galaxy S6 is OK
		 `AppleWebKit/537.36 (KHTML, like Gecko) Version/4.0 Mobile Safari/537.36 (Chrome/)`
		 */
		me.isBadAndroid = (function() {
			var appVersion = window.navigator.appVersion;
			// Android browser is not a chrome browser.
			if (/Android/.test(appVersion) && !(/Chrome\/\d/.test(appVersion))) {
				var safariVersion = appVersion.match(/Safari\/(\d+.\d)/);
				if(safariVersion && typeof safariVersion === "object" && safariVersion.length >= 2) {
					return parseFloat(safariVersion[1]) < 535.19;
				} else {
					return true;
				}
			} else {
				return false;
			}
		})();

		me.extend(me.style = {}, {
			transform: _transform,
			transitionTimingFunction: _prefixStyle('transitionTimingFunction'),
			transitionDuration: _prefixStyle('transitionDuration'),
			transitionDelay: _prefixStyle('transitionDelay'),
			transformOrigin: _prefixStyle('transformOrigin')
		});

		me.hasClass = function (e, c) {
			var re = new RegExp("(^|\\s)" + c + "(\\s|$)");
			return re.test(e.className);
		};

		me.addClass = function (e, c) {
			if ( me.hasClass(e, c) ) {
				return;
			}

			var newclass = e.className.split(' ');
			newclass.push(c);
			e.className = newclass.join(' ');
		};

		me.removeClass = function (e, c) {
			if ( !me.hasClass(e, c) ) {
				return;
			}

			var re = new RegExp("(^|\\s)" + c + "(\\s|$)", 'g');
			e.className = e.className.replace(re, ' ');
		};

		me.offset = function (el) {
			var left = -el.offsetLeft,
				top = -el.offsetTop;

			// jshint -W084
			while (el = el.offsetParent) {
				left -= el.offsetLeft;
				top -= el.offsetTop;
			}
			// jshint +W084

			return {
				left: left,
				top: top
			};
		};

		me.preventDefaultException = function (el, exceptions) {
			for ( var i in exceptions ) {
				if ( exceptions[i].test(el[i]) ) {
					return true;
				}
			}

			return false;
		};

		me.extend(me.eventType = {}, {
			touchstart: 1,
			touchmove: 1,
			touchend: 1,

			mousedown: 2,
			mousemove: 2,
			mouseup: 2,

			pointerdown: 3,
			pointermove: 3,
			pointerup: 3,

			MSPointerDown: 3,
			MSPointerMove: 3,
			MSPointerUp: 3
		});

		me.extend(me.ease = {}, {
			quadratic: {
				style: 'cubic-bezier(0.25, 0.46, 0.45, 0.94)',
				fn: function (k) {
					return k * ( 2 - k );
				}
			},
			circular: {
				style: 'cubic-bezier(0.1, 0.57, 0.1, 1)',	// Not properly "circular" but this looks better, it should be (0.075, 0.82, 0.165, 1)
				fn: function (k) {
					return Math.sqrt( 1 - ( --k * k ) );
				}
			},
			back: {
				style: 'cubic-bezier(0.175, 0.885, 0.32, 1.275)',
				fn: function (k) {
					var b = 4;
					return ( k = k - 1 ) * k * ( ( b + 1 ) * k + b ) + 1;
				}
			},
			bounce: {
				style: '',
				fn: function (k) {
					if ( ( k /= 1 ) < ( 1 / 2.75 ) ) {
						return 7.5625 * k * k;
					} else if ( k < ( 2 / 2.75 ) ) {
						return 7.5625 * ( k -= ( 1.5 / 2.75 ) ) * k + 0.75;
					} else if ( k < ( 2.5 / 2.75 ) ) {
						return 7.5625 * ( k -= ( 2.25 / 2.75 ) ) * k + 0.9375;
					} else {
						return 7.5625 * ( k -= ( 2.625 / 2.75 ) ) * k + 0.984375;
					}
				}
			},
			elastic: {
				style: '',
				fn: function (k) {
					var f = 0.22,
						e = 0.4;

					if ( k === 0 ) { return 0; }
					if ( k == 1 ) { return 1; }

					return ( e * Math.pow( 2, - 10 * k ) * Math.sin( ( k - f / 4 ) * ( 2 * Math.PI ) / f ) + 1 );
				}
			}
		});

		me.tap = function (e, eventName) {
			var ev = document.createEvent('Event');
			ev.initEvent(eventName, true, true);
			ev.pageX = e.pageX;
			ev.pageY = e.pageY;
			e.target.dispatchEvent(ev);
		};

		me.click = function (e) {
			var target = e.target,
				ev;

			if ( !(/(SELECT|INPUT|TEXTAREA)/i).test(target.tagName) ) {
				// https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/initMouseEvent
				// initMouseEvent is deprecated.
				ev = document.createEvent(window.MouseEvent ? 'MouseEvents' : 'Event');
				ev.initEvent('click', true, true);
				ev.view = e.view || window;
				ev.detail = 1;
				ev.screenX = target.screenX || 0;
				ev.screenY = target.screenY || 0;
				ev.clientX = target.clientX || 0;
				ev.clientY = target.clientY || 0;
				ev.ctrlKey = !!e.ctrlKey;
				ev.altKey = !!e.altKey;
				ev.shiftKey = !!e.shiftKey;
				ev.metaKey = !!e.metaKey;
				ev.button = 0;
				ev.relatedTarget = null;
				ev._constructed = true;
				target.dispatchEvent(ev);
			}
		};

		return me;
	})();
	function IScroll (el, options) {
		this.wrapper = typeof el == 'string' ? document.querySelector(el) : el;
		this.scroller = this.wrapper.children[0];
		this.scrollerStyle = this.scroller.style;		// cache style for better performance

		this.options = {

			resizeScrollbars: true,

			mouseWheelSpeed: 20,

			snapThreshold: 0.334,

			// INSERT POINT: OPTIONS
			disablePointer : !utils.hasPointer,
			disableTouch : utils.hasPointer || !utils.hasTouch,
			disableMouse : utils.hasPointer || utils.hasTouch,
			startX: 0,
			startY: 0,
			scrollY: true,
			directionLockThreshold: 5,
			momentum: true,

			bounce: true,
			bounceTime: 600,
			bounceEasing: '',

			preventDefault: true,
			preventDefaultException: { tagName: /^(INPUT|TEXTAREA|BUTTON|SELECT)$/ },

			HWCompositing: true,
			useTransition: true,
			useTransform: true,
			bindToWrapper: typeof window.onmousedown === "undefined"
		};

		for ( var i in options ) {
			this.options[i] = options[i];
		}

		// Normalize options
		this.translateZ = this.options.HWCompositing && utils.hasPerspective ? ' translateZ(0)' : '';

		this.options.useTransition = utils.hasTransition && this.options.useTransition;
		this.options.useTransform = utils.hasTransform && this.options.useTransform;

		this.options.eventPassthrough = this.options.eventPassthrough === true ? 'vertical' : this.options.eventPassthrough;
		this.options.preventDefault = !this.options.eventPassthrough && this.options.preventDefault;

		// If you want eventPassthrough I have to lock one of the axes
		this.options.scrollY = this.options.eventPassthrough == 'vertical' ? false : this.options.scrollY;
		this.options.scrollX = this.options.eventPassthrough == 'horizontal' ? false : this.options.scrollX;

		// With eventPassthrough we also need lockDirection mechanism
		this.options.freeScroll = this.options.freeScroll && !this.options.eventPassthrough;
		this.options.directionLockThreshold = this.options.eventPassthrough ? 0 : this.options.directionLockThreshold;

		this.options.bounceEasing = typeof this.options.bounceEasing == 'string' ? utils.ease[this.options.bounceEasing] || utils.ease.circular : this.options.bounceEasing;

		this.options.resizePolling = this.options.resizePolling === undefined ? 60 : this.options.resizePolling;

		if ( this.options.tap === true ) {
			this.options.tap = 'tap';
		}

		// https://github.com/cubiq/iscroll/issues/1029
		if (!this.options.useTransition && !this.options.useTransform) {
			if(!(/relative|absolute/i).test(this.scrollerStyle.position)) {
				this.scrollerStyle.position = "relative";
			}
		}

		if ( this.options.shrinkScrollbars == 'scale' ) {
			this.options.useTransition = false;
		}

		this.options.invertWheelDirection = this.options.invertWheelDirection ? -1 : 1;

		// INSERT POINT: NORMALIZATION

		// Some defaults
		this.x = 0;
		this.y = 0;
		this.directionX = 0;
		this.directionY = 0;
		this._events = {};

		// INSERT POINT: DEFAULTS

		this._init();
		this.refresh();

		this.scrollTo(this.options.startX, this.options.startY);
		this.enable();
	}

	IScroll.prototype = {
		version: '5.2.0',

		_init: function () {
			this._initEvents();

			if ( this.options.scrollbars || this.options.indicators ) {
				this._initIndicators();
			}

			if ( this.options.mouseWheel ) {
				this._initWheel();
			}

			if ( this.options.snap ) {
				this._initSnap();
			}

			if ( this.options.keyBindings ) {
				this._initKeys();
			}

			// INSERT POINT: _init

		},

		destroy: function () {
			this._initEvents(true);
			clearTimeout(this.resizeTimeout);
			this.resizeTimeout = null;
			this._execEvent('destroy');
		},

		_transitionEnd: function (e) {
			if ( e.target != this.scroller || !this.isInTransition ) {
				return;
			}

			this._transitionTime();
			if ( !this.resetPosition(this.options.bounceTime) ) {
				this.isInTransition = false;
				this._execEvent('scrollEnd');
			}
		},

		_start: function (e) {
			// React to left mouse button only
			if ( utils.eventType[e.type] != 1 ) {
				// for button property
				// http://unixpapa.com/js/mouse.html
				var button;
				if (!e.which) {
					/* IE case */
					button = (e.button < 2) ? 0 :
						((e.button == 4) ? 1 : 2);
				} else {
					/* All others */
					button = e.button;
				}
				if ( button !== 0 ) {
					return;
				}
			}

			if ( !this.enabled || (this.initiated && utils.eventType[e.type] !== this.initiated) ) {
				return;
			}

			if ( this.options.preventDefault && !utils.isBadAndroid && !utils.preventDefaultException(e.target, this.options.preventDefaultException) ) {
				e.preventDefault();
			}

			var point = e.touches ? e.touches[0] : e,
				pos;

			this.initiated	= utils.eventType[e.type];
			this.moved		= false;
			this.distX		= 0;
			this.distY		= 0;
			this.directionX = 0;
			this.directionY = 0;
			this.directionLocked = 0;

			this.startTime = utils.getTime();

			if ( this.options.useTransition && this.isInTransition ) {
				this._transitionTime();
				this.isInTransition = false;
				pos = this.getComputedPosition();
				this._translate(Math.round(pos.x), Math.round(pos.y));
				this._execEvent('scrollEnd');
			} else if ( !this.options.useTransition && this.isAnimating ) {
				this.isAnimating = false;
				this._execEvent('scrollEnd');
			}

			this.startX    = this.x;
			this.startY    = this.y;
			this.absStartX = this.x;
			this.absStartY = this.y;
			this.pointX    = point.pageX;
			this.pointY    = point.pageY;

			this._execEvent('beforeScrollStart');
		},

		_move: function (e) {
			if ( !this.enabled || utils.eventType[e.type] !== this.initiated ) {
				return;
			}

			if ( this.options.preventDefault ) {	// increases performance on Android? TODO: check!
				e.preventDefault();
			}

			var point		= e.touches ? e.touches[0] : e,
				deltaX		= point.pageX - this.pointX,
				deltaY		= point.pageY - this.pointY,
				timestamp	= utils.getTime(),
				newX, newY,
				absDistX, absDistY;

			this.pointX		= point.pageX;
			this.pointY		= point.pageY;

			this.distX		+= deltaX;
			this.distY		+= deltaY;
			absDistX		= Math.abs(this.distX);
			absDistY		= Math.abs(this.distY);

			// We need to move at least 10 pixels for the scrolling to initiate
			if ( timestamp - this.endTime > 300 && (absDistX < 10 && absDistY < 10) ) {
				return;
			}

			// If you are scrolling in one direction lock the other
			if ( !this.directionLocked && !this.options.freeScroll ) {
				if ( absDistX > absDistY + this.options.directionLockThreshold ) {
					this.directionLocked = 'h';		// lock horizontally
				} else if ( absDistY >= absDistX + this.options.directionLockThreshold ) {
					this.directionLocked = 'v';		// lock vertically
				} else {
					this.directionLocked = 'n';		// no lock
				}
			}

			if ( this.directionLocked == 'h' ) {
				if ( this.options.eventPassthrough == 'vertical' ) {
					e.preventDefault();
				} else if ( this.options.eventPassthrough == 'horizontal' ) {
					this.initiated = false;
					return;
				}

				deltaY = 0;
			} else if ( this.directionLocked == 'v' ) {
				if ( this.options.eventPassthrough == 'horizontal' ) {
					e.preventDefault();
				} else if ( this.options.eventPassthrough == 'vertical' ) {
					this.initiated = false;
					return;
				}

				deltaX = 0;
			}

			deltaX = this.hasHorizontalScroll ? deltaX : 0;
			deltaY = this.hasVerticalScroll ? deltaY : 0;

			newX = this.x + deltaX;
			newY = this.y + deltaY;

			// Slow down if outside of the boundaries
			if ( newX > 0 || newX < this.maxScrollX ) {
				newX = this.options.bounce ? this.x + deltaX / 3 : newX > 0 ? 0 : this.maxScrollX;
			}
			if ( newY > 0 || newY < this.maxScrollY ) {
				newY = this.options.bounce ? this.y + deltaY / 3 : newY > 0 ? 0 : this.maxScrollY;
			}

			this.directionX = deltaX > 0 ? -1 : deltaX < 0 ? 1 : 0;
			this.directionY = deltaY > 0 ? -1 : deltaY < 0 ? 1 : 0;

			if ( !this.moved ) {
				this._execEvent('scrollStart');
			}

			this.moved = true;

			this._translate(newX, newY);

			/* REPLACE START: _move */

			if ( timestamp - this.startTime > 300 ) {
				this.startTime = timestamp;
				this.startX = this.x;
				this.startY = this.y;
			}

			/* REPLACE END: _move */

		},

		_end: function (e) {
			if ( !this.enabled || utils.eventType[e.type] !== this.initiated ) {
				return;
			}

			if ( this.options.preventDefault && !utils.preventDefaultException(e.target, this.options.preventDefaultException) ) {
				e.preventDefault();
			}

			var point = e.changedTouches ? e.changedTouches[0] : e,
				momentumX,
				momentumY,
				duration = utils.getTime() - this.startTime,
				newX = Math.round(this.x),
				newY = Math.round(this.y),
				distanceX = Math.abs(newX - this.startX),
				distanceY = Math.abs(newY - this.startY),
				time = 0,
				easing = '';

			this.isInTransition = 0;
			this.initiated = 0;
			this.endTime = utils.getTime();

			// reset if we are outside of the boundaries
			if ( this.resetPosition(this.options.bounceTime) ) {
				return;
			}

			this.scrollTo(newX, newY);	// ensures that the last position is rounded

			// we scrolled less than 10 pixels
			if ( !this.moved ) {
				if ( this.options.tap ) {
					utils.tap(e, this.options.tap);
				}

				if ( this.options.click ) {
					utils.click(e);
				}

				this._execEvent('scrollCancel');
				return;
			}

			if ( this._events.flick && duration < 200 && distanceX < 100 && distanceY < 100 ) {
				this._execEvent('flick');
				return;
			}

			// start momentum animation if needed
			if ( this.options.momentum && duration < 300 ) {
				momentumX = this.hasHorizontalScroll ? utils.momentum(this.x, this.startX, duration, this.maxScrollX, this.options.bounce ? this.wrapperWidth : 0, this.options.deceleration) : { destination: newX, duration: 0 };
				momentumY = this.hasVerticalScroll ? utils.momentum(this.y, this.startY, duration, this.maxScrollY, this.options.bounce ? this.wrapperHeight : 0, this.options.deceleration) : { destination: newY, duration: 0 };
				newX = momentumX.destination;
				newY = momentumY.destination;
				time = Math.max(momentumX.duration, momentumY.duration);
				this.isInTransition = 1;
			}


			if ( this.options.snap ) {
				var snap = this._nearestSnap(newX, newY);
				this.currentPage = snap;
				time = this.options.snapSpeed || Math.max(
						Math.max(
							Math.min(Math.abs(newX - snap.x), 1000),
							Math.min(Math.abs(newY - snap.y), 1000)
						), 300);
				newX = snap.x;
				newY = snap.y;

				this.directionX = 0;
				this.directionY = 0;
				easing = this.options.bounceEasing;
			}

			// INSERT POINT: _end

			if ( newX != this.x || newY != this.y ) {
				// change easing function when scroller goes out of the boundaries
				if ( newX > 0 || newX < this.maxScrollX || newY > 0 || newY < this.maxScrollY ) {
					easing = utils.ease.quadratic;
				}

				this.scrollTo(newX, newY, time, easing);
				return;
			}

			this._execEvent('scrollEnd');
		},

		_resize: function () {
			var that = this;

			clearTimeout(this.resizeTimeout);

			this.resizeTimeout = setTimeout(function () {
				that.refresh();
			}, this.options.resizePolling);
		},

		resetPosition: function (time) {
			var x = this.x,
				y = this.y;

			time = time || 0;

			if ( !this.hasHorizontalScroll || this.x > 0 ) {
				x = 0;
			} else if ( this.x < this.maxScrollX ) {
				x = this.maxScrollX;
			}

			if ( !this.hasVerticalScroll || this.y > 0 ) {
				y = 0;
			} else if ( this.y < this.maxScrollY ) {
				y = this.maxScrollY;
			}

			if ( x == this.x && y == this.y ) {
				return false;
			}

			this.scrollTo(x, y, time, this.options.bounceEasing);

			return true;
		},

		disable: function () {
			this.enabled = false;
		},

		enable: function () {
			this.enabled = true;
		},

		refresh: function () {
			var rf = this.wrapper.offsetHeight;		// Force reflow

			this.wrapperWidth	= this.wrapper.clientWidth;
			this.wrapperHeight	= this.wrapper.clientHeight;

			/* REPLACE START: refresh */

			this.scrollerWidth	= this.scroller.offsetWidth;
			this.scrollerHeight	= this.scroller.offsetHeight;

			this.maxScrollX		= this.wrapperWidth - this.scrollerWidth;
			this.maxScrollY		= this.wrapperHeight - this.scrollerHeight;

			/* REPLACE END: refresh */

			this.hasHorizontalScroll	= this.options.scrollX && this.maxScrollX < 0;
			this.hasVerticalScroll		= this.options.scrollY && this.maxScrollY < 0;

			if ( !this.hasHorizontalScroll ) {
				this.maxScrollX = 0;
				this.scrollerWidth = this.wrapperWidth;
			}

			if ( !this.hasVerticalScroll ) {
				this.maxScrollY = 0;
				this.scrollerHeight = this.wrapperHeight;
			}

			this.endTime = 0;
			this.directionX = 0;
			this.directionY = 0;

			this.wrapperOffset = utils.offset(this.wrapper);

			this._execEvent('refresh');

			this.resetPosition();

			// INSERT POINT: _refresh

		},

		on: function (type, fn) {
			if ( !this._events[type] ) {
				this._events[type] = [];
			}

			this._events[type].push(fn);
		},

		off: function (type, fn) {
			if ( !this._events[type] ) {
				return;
			}

			var index = this._events[type].indexOf(fn);

			if ( index > -1 ) {
				this._events[type].splice(index, 1);
			}
		},

		_execEvent: function (type) {
			if ( !this._events[type] ) {
				return;
			}

			var i = 0,
				l = this._events[type].length;

			if ( !l ) {
				return;
			}

			for ( ; i < l; i++ ) {
				this._events[type][i].apply(this, [].slice.call(arguments, 1));
			}
		},

		scrollBy: function (x, y, time, easing) {
			x = this.x + x;
			y = this.y + y;
			time = time || 0;

			this.scrollTo(x, y, time, easing);
		},

		scrollTo: function (x, y, time, easing) {
			easing = easing || utils.ease.circular;

			this.isInTransition = this.options.useTransition && time > 0;
			var transitionType = this.options.useTransition && easing.style;
			if ( !time || transitionType ) {
				if(transitionType) {
					this._transitionTimingFunction(easing.style);
					this._transitionTime(time);
				}
				this._translate(x, y);
			} else {
				this._animate(x, y, time, easing.fn);
			}
		},

		scrollToElement: function (el, time, offsetX, offsetY, easing) {
			el = el.nodeType ? el : this.scroller.querySelector(el);

			if ( !el ) {
				return;
			}

			var pos = utils.offset(el);

			pos.left -= this.wrapperOffset.left;
			pos.top  -= this.wrapperOffset.top;

			// if offsetX/Y are true we center the element to the screen
			if ( offsetX === true ) {
				offsetX = Math.round(el.offsetWidth / 2 - this.wrapper.offsetWidth / 2);
			}
			if ( offsetY === true ) {
				offsetY = Math.round(el.offsetHeight / 2 - this.wrapper.offsetHeight / 2);
			}

			pos.left -= offsetX || 0;
			pos.top  -= offsetY || 0;

			pos.left = pos.left > 0 ? 0 : pos.left < this.maxScrollX ? this.maxScrollX : pos.left;
			pos.top  = pos.top  > 0 ? 0 : pos.top  < this.maxScrollY ? this.maxScrollY : pos.top;

			time = time === undefined || time === null || time === 'auto' ? Math.max(Math.abs(this.x-pos.left), Math.abs(this.y-pos.top)) : time;

			this.scrollTo(pos.left, pos.top, time, easing);
		},

		_transitionTime: function (time) {
			if (!this.options.useTransition) {
				return;
			}
			time = time || 0;
			var durationProp = utils.style.transitionDuration;
			if(!durationProp) {
				return;
			}

			this.scrollerStyle[durationProp] = time + 'ms';

			if ( !time && utils.isBadAndroid ) {
				this.scrollerStyle[durationProp] = '0.0001ms';
				// remove 0.0001ms
				var self = this;
				rAF(function() {
					if(self.scrollerStyle[durationProp] === '0.0001ms') {
						self.scrollerStyle[durationProp] = '0s';
					}
				});
			}


			if ( this.indicators ) {
				for ( var i = this.indicators.length; i--; ) {
					this.indicators[i].transitionTime(time);
				}
			}


			// INSERT POINT: _transitionTime

		},

		_transitionTimingFunction: function (easing) {
			this.scrollerStyle[utils.style.transitionTimingFunction] = easing;


			if ( this.indicators ) {
				for ( var i = this.indicators.length; i--; ) {
					this.indicators[i].transitionTimingFunction(easing);
				}
			}


			// INSERT POINT: _transitionTimingFunction

		},

		_translate: function (x, y) {
			if ( this.options.useTransform ) {

				/* REPLACE START: _translate */

				this.scrollerStyle[utils.style.transform] = 'translate(' + x + 'px,' + y + 'px)' + this.translateZ;

				/* REPLACE END: _translate */

			} else {
				x = Math.round(x);
				y = Math.round(y);
				this.scrollerStyle.left = x + 'px';
				this.scrollerStyle.top = y + 'px';
			}

			this.x = x;
			this.y = y;


			if ( this.indicators ) {
				for ( var i = this.indicators.length; i--; ) {
					this.indicators[i].updatePosition();
				}
			}


			// INSERT POINT: _translate

		},

		_initEvents: function (remove) {
			var eventType = remove ? utils.removeEvent : utils.addEvent,
				target = this.options.bindToWrapper ? this.wrapper : window;

			eventType(window, 'orientationchange', this);
			eventType(window, 'resize', this);

			if ( this.options.click ) {
				eventType(this.wrapper, 'click', this, true);
			}

			if ( !this.options.disableMouse ) {
				eventType(this.wrapper, 'mousedown', this);
				eventType(target, 'mousemove', this);
				eventType(target, 'mousecancel', this);
				eventType(target, 'mouseup', this);
			}

			if ( utils.hasPointer && !this.options.disablePointer ) {
				eventType(this.wrapper, utils.prefixPointerEvent('pointerdown'), this);
				eventType(target, utils.prefixPointerEvent('pointermove'), this);
				eventType(target, utils.prefixPointerEvent('pointercancel'), this);
				eventType(target, utils.prefixPointerEvent('pointerup'), this);
			}

			if ( utils.hasTouch && !this.options.disableTouch ) {
				eventType(this.wrapper, 'touchstart', this);
				eventType(target, 'touchmove', this);
				eventType(target, 'touchcancel', this);
				eventType(target, 'touchend', this);
			}

			eventType(this.scroller, 'transitionend', this);
			eventType(this.scroller, 'webkitTransitionEnd', this);
			eventType(this.scroller, 'oTransitionEnd', this);
			eventType(this.scroller, 'MSTransitionEnd', this);
		},

		getComputedPosition: function () {
			var matrix = window.getComputedStyle(this.scroller, null),
				x, y;

			if ( this.options.useTransform ) {
				matrix = matrix[utils.style.transform].split(')')[0].split(', ');
				x = +(matrix[12] || matrix[4]);
				y = +(matrix[13] || matrix[5]);
			} else {
				x = +matrix.left.replace(/[^-\d.]/g, '');
				y = +matrix.top.replace(/[^-\d.]/g, '');
			}

			return { x: x, y: y };
		},
		_initIndicators: function () {
			var interactive = this.options.interactiveScrollbars,
				customStyle = typeof this.options.scrollbars != 'string',
				indicators = [],
				indicator;

			var that = this;

			this.indicators = [];

			if ( this.options.scrollbars ) {
				// Vertical scrollbar
				if ( this.options.scrollY ) {
					indicator = {
						el: createDefaultScrollbar('v', interactive, this.options.scrollbars),
						interactive: interactive,
						defaultScrollbars: true,
						customStyle: customStyle,
						resize: this.options.resizeScrollbars,
						shrink: this.options.shrinkScrollbars,
						fade: this.options.fadeScrollbars,
						listenX: false
					};

					this.wrapper.appendChild(indicator.el);
					indicators.push(indicator);
				}

				// Horizontal scrollbar
				if ( this.options.scrollX ) {
					indicator = {
						el: createDefaultScrollbar('h', interactive, this.options.scrollbars),
						interactive: interactive,
						defaultScrollbars: true,
						customStyle: customStyle,
						resize: this.options.resizeScrollbars,
						shrink: this.options.shrinkScrollbars,
						fade: this.options.fadeScrollbars,
						listenY: false
					};

					this.wrapper.appendChild(indicator.el);
					indicators.push(indicator);
				}
			}

			if ( this.options.indicators ) {
				// TODO: check concat compatibility
				indicators = indicators.concat(this.options.indicators);
			}

			for ( var i = indicators.length; i--; ) {
				this.indicators.push( new Indicator(this, indicators[i]) );
			}

			// TODO: check if we can use array.map (wide compatibility and performance issues)
			function _indicatorsMap (fn) {
				if (that.indicators) {
					for ( var i = that.indicators.length; i--; ) {
						fn.call(that.indicators[i]);
					}
				}
			}

			if ( this.options.fadeScrollbars ) {
				this.on('scrollEnd', function () {
					_indicatorsMap(function () {
						this.fade();
					});
				});

				this.on('scrollCancel', function () {
					_indicatorsMap(function () {
						this.fade();
					});
				});

				this.on('scrollStart', function () {
					_indicatorsMap(function () {
						this.fade(1);
					});
				});

				this.on('beforeScrollStart', function () {
					_indicatorsMap(function () {
						this.fade(1, true);
					});
				});
			}


			this.on('refresh', function () {
				_indicatorsMap(function () {
					this.refresh();
				});
			});

			this.on('destroy', function () {
				_indicatorsMap(function () {
					this.destroy();
				});

				delete this.indicators;
			});
		},

		_initWheel: function () {
			utils.addEvent(this.wrapper, 'wheel', this);
			utils.addEvent(this.wrapper, 'mousewheel', this);
			utils.addEvent(this.wrapper, 'DOMMouseScroll', this);

			this.on('destroy', function () {
				clearTimeout(this.wheelTimeout);
				this.wheelTimeout = null;
				utils.removeEvent(this.wrapper, 'wheel', this);
				utils.removeEvent(this.wrapper, 'mousewheel', this);
				utils.removeEvent(this.wrapper, 'DOMMouseScroll', this);
			});
		},

		_wheel: function (e) {
			if ( !this.enabled ) {
				return;
			}

			e.preventDefault();

			var wheelDeltaX, wheelDeltaY,
				newX, newY,
				that = this;

			if ( this.wheelTimeout === undefined ) {
				that._execEvent('scrollStart');
			}

			// Execute the scrollEnd event after 400ms the wheel stopped scrolling
			clearTimeout(this.wheelTimeout);
			this.wheelTimeout = setTimeout(function () {
				if(!that.options.snap) {
					that._execEvent('scrollEnd');
				}
				that.wheelTimeout = undefined;
			}, 400);

			if ( 'deltaX' in e ) {
				if (e.deltaMode === 1) {
					wheelDeltaX = -e.deltaX * this.options.mouseWheelSpeed;
					wheelDeltaY = -e.deltaY * this.options.mouseWheelSpeed;
				} else {
					wheelDeltaX = -e.deltaX;
					wheelDeltaY = -e.deltaY;
				}
			} else if ( 'wheelDeltaX' in e ) {
				wheelDeltaX = e.wheelDeltaX / 120 * this.options.mouseWheelSpeed;
				wheelDeltaY = e.wheelDeltaY / 120 * this.options.mouseWheelSpeed;
			} else if ( 'wheelDelta' in e ) {
				wheelDeltaX = wheelDeltaY = e.wheelDelta / 120 * this.options.mouseWheelSpeed;
			} else if ( 'detail' in e ) {
				wheelDeltaX = wheelDeltaY = -e.detail / 3 * this.options.mouseWheelSpeed;
			} else {
				return;
			}

			wheelDeltaX *= this.options.invertWheelDirection;
			wheelDeltaY *= this.options.invertWheelDirection;

			if ( !this.hasVerticalScroll ) {
				wheelDeltaX = wheelDeltaY;
				wheelDeltaY = 0;
			}

			if ( this.options.snap ) {
				newX = this.currentPage.pageX;
				newY = this.currentPage.pageY;

				if ( wheelDeltaX > 0 ) {
					newX--;
				} else if ( wheelDeltaX < 0 ) {
					newX++;
				}

				if ( wheelDeltaY > 0 ) {
					newY--;
				} else if ( wheelDeltaY < 0 ) {
					newY++;
				}

				this.goToPage(newX, newY);

				return;
			}

			newX = this.x + Math.round(this.hasHorizontalScroll ? wheelDeltaX : 0);
			newY = this.y + Math.round(this.hasVerticalScroll ? wheelDeltaY : 0);

			this.directionX = wheelDeltaX > 0 ? -1 : wheelDeltaX < 0 ? 1 : 0;
			this.directionY = wheelDeltaY > 0 ? -1 : wheelDeltaY < 0 ? 1 : 0;

			if ( newX > 0 ) {
				newX = 0;
			} else if ( newX < this.maxScrollX ) {
				newX = this.maxScrollX;
			}

			if ( newY > 0 ) {
				newY = 0;
			} else if ( newY < this.maxScrollY ) {
				newY = this.maxScrollY;
			}

			this.scrollTo(newX, newY, 0);

			// INSERT POINT: _wheel
		},

		_initSnap: function () {
			this.currentPage = {};

			if ( typeof this.options.snap == 'string' ) {
				this.options.snap = this.scroller.querySelectorAll(this.options.snap);
			}

			this.on('refresh', function () {
				var i = 0, l,
					m = 0, n,
					cx, cy,
					x = 0, y,
					stepX = this.options.snapStepX || this.wrapperWidth,
					stepY = this.options.snapStepY || this.wrapperHeight,
					el;

				this.pages = [];

				if ( !this.wrapperWidth || !this.wrapperHeight || !this.scrollerWidth || !this.scrollerHeight ) {
					return;
				}

				if ( this.options.snap === true ) {
					cx = Math.round( stepX / 2 );
					cy = Math.round( stepY / 2 );

					while ( x > -this.scrollerWidth ) {
						this.pages[i] = [];
						l = 0;
						y = 0;

						while ( y > -this.scrollerHeight ) {
							this.pages[i][l] = {
								x: Math.max(x, this.maxScrollX),
								y: Math.max(y, this.maxScrollY),
								width: stepX,
								height: stepY,
								cx: x - cx,
								cy: y - cy
							};

							y -= stepY;
							l++;
						}

						x -= stepX;
						i++;
					}
				} else {
					el = this.options.snap;
					l = el.length;
					n = -1;

					for ( ; i < l; i++ ) {
						if ( i === 0 || el[i].offsetLeft <= el[i-1].offsetLeft ) {
							m = 0;
							n++;
						}

						if ( !this.pages[m] ) {
							this.pages[m] = [];
						}

						x = Math.max(-el[i].offsetLeft, this.maxScrollX);
						y = Math.max(-el[i].offsetTop, this.maxScrollY);
						cx = x - Math.round(el[i].offsetWidth / 2);
						cy = y - Math.round(el[i].offsetHeight / 2);

						this.pages[m][n] = {
							x: x,
							y: y,
							width: el[i].offsetWidth,
							height: el[i].offsetHeight,
							cx: cx,
							cy: cy
						};

						if ( x > this.maxScrollX ) {
							m++;
						}
					}
				}

				this.goToPage(this.currentPage.pageX || 0, this.currentPage.pageY || 0, 0);

				// Update snap threshold if needed
				if ( this.options.snapThreshold % 1 === 0 ) {
					this.snapThresholdX = this.options.snapThreshold;
					this.snapThresholdY = this.options.snapThreshold;
				} else {
					this.snapThresholdX = Math.round(this.pages[this.currentPage.pageX][this.currentPage.pageY].width * this.options.snapThreshold);
					this.snapThresholdY = Math.round(this.pages[this.currentPage.pageX][this.currentPage.pageY].height * this.options.snapThreshold);
				}
			});

			this.on('flick', function () {
				var time = this.options.snapSpeed || Math.max(
						Math.max(
							Math.min(Math.abs(this.x - this.startX), 1000),
							Math.min(Math.abs(this.y - this.startY), 1000)
						), 300);

				this.goToPage(
					this.currentPage.pageX + this.directionX,
					this.currentPage.pageY + this.directionY,
					time
				);
			});
		},

		_nearestSnap: function (x, y) {
			if ( !this.pages.length ) {
				return { x: 0, y: 0, pageX: 0, pageY: 0 };
			}

			var i = 0,
				l = this.pages.length,
				m = 0;

			// Check if we exceeded the snap threshold
			if ( Math.abs(x - this.absStartX) < this.snapThresholdX &&
				 Math.abs(y - this.absStartY) < this.snapThresholdY ) {
				return this.currentPage;
			}

			if ( x > 0 ) {
				x = 0;
			} else if ( x < this.maxScrollX ) {
				x = this.maxScrollX;
			}

			if ( y > 0 ) {
				y = 0;
			} else if ( y < this.maxScrollY ) {
				y = this.maxScrollY;
			}

			for ( ; i < l; i++ ) {
				if ( x >= this.pages[i][0].cx ) {
					x = this.pages[i][0].x;
					break;
				}
			}

			l = this.pages[i].length;

			for ( ; m < l; m++ ) {
				if ( y >= this.pages[0][m].cy ) {
					y = this.pages[0][m].y;
					break;
				}
			}

			if ( i == this.currentPage.pageX ) {
				i += this.directionX;

				if ( i < 0 ) {
					i = 0;
				} else if ( i >= this.pages.length ) {
					i = this.pages.length - 1;
				}

				x = this.pages[i][0].x;
			}

			if ( m == this.currentPage.pageY ) {
				m += this.directionY;

				if ( m < 0 ) {
					m = 0;
				} else if ( m >= this.pages[0].length ) {
					m = this.pages[0].length - 1;
				}

				y = this.pages[0][m].y;
			}

			return {
				x: x,
				y: y,
				pageX: i,
				pageY: m
			};
		},

		goToPage: function (x, y, time, easing) {
			easing = easing || this.options.bounceEasing;

			if ( x >= this.pages.length ) {
				x = this.pages.length - 1;
			} else if ( x < 0 ) {
				x = 0;
			}

			if ( y >= this.pages[x].length ) {
				y = this.pages[x].length - 1;
			} else if ( y < 0 ) {
				y = 0;
			}

			var posX = this.pages[x][y].x,
				posY = this.pages[x][y].y;

			time = time === undefined ? this.options.snapSpeed || Math.max(
				Math.max(
					Math.min(Math.abs(posX - this.x), 1000),
					Math.min(Math.abs(posY - this.y), 1000)
				), 300) : time;

			this.currentPage = {
				x: posX,
				y: posY,
				pageX: x,
				pageY: y
			};

			this.scrollTo(posX, posY, time, easing);
		},

		next: function (time, easing) {
			var x = this.currentPage.pageX,
				y = this.currentPage.pageY;

			x++;

			if ( x >= this.pages.length && this.hasVerticalScroll ) {
				x = 0;
				y++;
			}

			this.goToPage(x, y, time, easing);
		},

		prev: function (time, easing) {
			var x = this.currentPage.pageX,
				y = this.currentPage.pageY;

			x--;

			if ( x < 0 && this.hasVerticalScroll ) {
				x = 0;
				y--;
			}

			this.goToPage(x, y, time, easing);
		},

		_initKeys: function (e) {
			// default key bindings
			var keys = {
				pageUp: 33,
				pageDown: 34,
				end: 35,
				home: 36,
				left: 37,
				up: 38,
				right: 39,
				down: 40
			};
			var i;

			// if you give me characters I give you keycode
			if ( typeof this.options.keyBindings == 'object' ) {
				for ( i in this.options.keyBindings ) {
					if ( typeof this.options.keyBindings[i] == 'string' ) {
						this.options.keyBindings[i] = this.options.keyBindings[i].toUpperCase().charCodeAt(0);
					}
				}
			} else {
				this.options.keyBindings = {};
			}

			for ( i in keys ) {
				this.options.keyBindings[i] = this.options.keyBindings[i] || keys[i];
			}

			utils.addEvent(window, 'keydown', this);

			this.on('destroy', function () {
				utils.removeEvent(window, 'keydown', this);
			});
		},

		_key: function (e) {
			if ( !this.enabled ) {
				return;
			}

			var snap = this.options.snap,	// we are using this alot, better to cache it
				newX = snap ? this.currentPage.pageX : this.x,
				newY = snap ? this.currentPage.pageY : this.y,
				now = utils.getTime(),
				prevTime = this.keyTime || 0,
				acceleration = 0.250,
				pos;

			if ( this.options.useTransition && this.isInTransition ) {
				pos = this.getComputedPosition();

				this._translate(Math.round(pos.x), Math.round(pos.y));
				this.isInTransition = false;
			}

			this.keyAcceleration = now - prevTime < 200 ? Math.min(this.keyAcceleration + acceleration, 50) : 0;

			switch ( e.keyCode ) {
				case this.options.keyBindings.pageUp:
					if ( this.hasHorizontalScroll && !this.hasVerticalScroll ) {
						newX += snap ? 1 : this.wrapperWidth;
					} else {
						newY += snap ? 1 : this.wrapperHeight;
					}
					break;
				case this.options.keyBindings.pageDown:
					if ( this.hasHorizontalScroll && !this.hasVerticalScroll ) {
						newX -= snap ? 1 : this.wrapperWidth;
					} else {
						newY -= snap ? 1 : this.wrapperHeight;
					}
					break;
				case this.options.keyBindings.end:
					newX = snap ? this.pages.length-1 : this.maxScrollX;
					newY = snap ? this.pages[0].length-1 : this.maxScrollY;
					break;
				case this.options.keyBindings.home:
					newX = 0;
					newY = 0;
					break;
				case this.options.keyBindings.left:
					newX += snap ? -1 : 5 + this.keyAcceleration>>0;
					break;
				case this.options.keyBindings.up:
					newY += snap ? 1 : 5 + this.keyAcceleration>>0;
					break;
				case this.options.keyBindings.right:
					newX -= snap ? -1 : 5 + this.keyAcceleration>>0;
					break;
				case this.options.keyBindings.down:
					newY -= snap ? 1 : 5 + this.keyAcceleration>>0;
					break;
				default:
					return;
			}

			if ( snap ) {
				this.goToPage(newX, newY);
				return;
			}

			if ( newX > 0 ) {
				newX = 0;
				this.keyAcceleration = 0;
			} else if ( newX < this.maxScrollX ) {
				newX = this.maxScrollX;
				this.keyAcceleration = 0;
			}

			if ( newY > 0 ) {
				newY = 0;
				this.keyAcceleration = 0;
			} else if ( newY < this.maxScrollY ) {
				newY = this.maxScrollY;
				this.keyAcceleration = 0;
			}

			this.scrollTo(newX, newY, 0);

			this.keyTime = now;
		},

		_animate: function (destX, destY, duration, easingFn) {
			var that = this,
				startX = this.x,
				startY = this.y,
				startTime = utils.getTime(),
				destTime = startTime + duration;

			function step () {
				var now = utils.getTime(),
					newX, newY,
					easing;

				if ( now >= destTime ) {
					that.isAnimating = false;
					that._translate(destX, destY);

					if ( !that.resetPosition(that.options.bounceTime) ) {
						that._execEvent('scrollEnd');
					}

					return;
				}

				now = ( now - startTime ) / duration;
				easing = easingFn(now);
				newX = ( destX - startX ) * easing + startX;
				newY = ( destY - startY ) * easing + startY;
				that._translate(newX, newY);

				if ( that.isAnimating ) {
					rAF(step);
				}
			}

			this.isAnimating = true;
			step();
		},
		handleEvent: function (e) {
			switch ( e.type ) {
				case 'touchstart':
				case 'pointerdown':
				case 'MSPointerDown':
				case 'mousedown':
					this._start(e);
					break;
				case 'touchmove':
				case 'pointermove':
				case 'MSPointerMove':
				case 'mousemove':
					this._move(e);
					break;
				case 'touchend':
				case 'pointerup':
				case 'MSPointerUp':
				case 'mouseup':
				case 'touchcancel':
				case 'pointercancel':
				case 'MSPointerCancel':
				case 'mousecancel':
					this._end(e);
					break;
				case 'orientationchange':
				case 'resize':
					this._resize();
					break;
				case 'transitionend':
				case 'webkitTransitionEnd':
				case 'oTransitionEnd':
				case 'MSTransitionEnd':
					this._transitionEnd(e);
					break;
				case 'wheel':
				case 'DOMMouseScroll':
				case 'mousewheel':
					this._wheel(e);
					break;
				case 'keydown':
					this._key(e);
					break;
				case 'click':
					if ( this.enabled && !e._constructed ) {
						e.preventDefault();
						e.stopPropagation();
					}
					break;
			}
		}
	};
	function createDefaultScrollbar (direction, interactive, type) {
		var scrollbar = document.createElement('div'),
			indicator = document.createElement('div');

		if ( type === true ) {
			scrollbar.style.cssText = 'position:absolute;z-index:9999';
			indicator.style.cssText = '-webkit-box-sizing:border-box;-moz-box-sizing:border-box;box-sizing:border-box;position:absolute;background:rgba(0,0,0,0.5);border:1px solid rgba(255,255,255,0.9);border-radius:3px';
		}

		indicator.className = 'iScrollIndicator';

		if ( direction == 'h' ) {
			if ( type === true ) {
				scrollbar.style.cssText += ';height:7px;left:2px;right:2px;bottom:0';
				indicator.style.height = '100%';
			}
			scrollbar.className = 'iScrollHorizontalScrollbar';
		} else {
			if ( type === true ) {
				scrollbar.style.cssText += ';width:7px;bottom:2px;top:2px;right:1px';
				indicator.style.width = '100%';
			}
			scrollbar.className = 'iScrollVerticalScrollbar';
		}

		scrollbar.style.cssText += ';overflow:hidden';

		if ( !interactive ) {
			scrollbar.style.pointerEvents = 'none';
		}

		scrollbar.appendChild(indicator);

		return scrollbar;
	}

	function Indicator (scroller, options) {
		this.wrapper = typeof options.el == 'string' ? document.querySelector(options.el) : options.el;
		this.wrapperStyle = this.wrapper.style;
		this.indicator = this.wrapper.children[0];
		this.indicatorStyle = this.indicator.style;
		this.scroller = scroller;

		this.options = {
			listenX: true,
			listenY: true,
			interactive: false,
			resize: true,
			defaultScrollbars: false,
			shrink: false,
			fade: false,
			speedRatioX: 0,
			speedRatioY: 0
		};

		for ( var i in options ) {
			this.options[i] = options[i];
		}

		this.sizeRatioX = 1;
		this.sizeRatioY = 1;
		this.maxPosX = 0;
		this.maxPosY = 0;

		if ( this.options.interactive ) {
			if ( !this.options.disableTouch ) {
				utils.addEvent(this.indicator, 'touchstart', this);
				utils.addEvent(window, 'touchend', this);
			}
			if ( !this.options.disablePointer ) {
				utils.addEvent(this.indicator, utils.prefixPointerEvent('pointerdown'), this);
				utils.addEvent(window, utils.prefixPointerEvent('pointerup'), this);
			}
			if ( !this.options.disableMouse ) {
				utils.addEvent(this.indicator, 'mousedown', this);
				utils.addEvent(window, 'mouseup', this);
			}
		}

		if ( this.options.fade ) {
			this.wrapperStyle[utils.style.transform] = this.scroller.translateZ;
			var durationProp = utils.style.transitionDuration;
			if(!durationProp) {
				return;
			}
			this.wrapperStyle[durationProp] = utils.isBadAndroid ? '0.0001ms' : '0ms';
			// remove 0.0001ms
			var self = this;
			if(utils.isBadAndroid) {
				rAF(function() {
					if(self.wrapperStyle[durationProp] === '0.0001ms') {
						self.wrapperStyle[durationProp] = '0s';
					}
				});
			}
			this.wrapperStyle.opacity = '0';
		}
	}

	Indicator.prototype = {
		handleEvent: function (e) {
			switch ( e.type ) {
				case 'touchstart':
				case 'pointerdown':
				case 'MSPointerDown':
				case 'mousedown':
					this._start(e);
					break;
				case 'touchmove':
				case 'pointermove':
				case 'MSPointerMove':
				case 'mousemove':
					this._move(e);
					break;
				case 'touchend':
				case 'pointerup':
				case 'MSPointerUp':
				case 'mouseup':
				case 'touchcancel':
				case 'pointercancel':
				case 'MSPointerCancel':
				case 'mousecancel':
					this._end(e);
					break;
			}
		},

		destroy: function () {
			if ( this.options.fadeScrollbars ) {
				clearTimeout(this.fadeTimeout);
				this.fadeTimeout = null;
			}
			if ( this.options.interactive ) {
				utils.removeEvent(this.indicator, 'touchstart', this);
				utils.removeEvent(this.indicator, utils.prefixPointerEvent('pointerdown'), this);
				utils.removeEvent(this.indicator, 'mousedown', this);

				utils.removeEvent(window, 'touchmove', this);
				utils.removeEvent(window, utils.prefixPointerEvent('pointermove'), this);
				utils.removeEvent(window, 'mousemove', this);

				utils.removeEvent(window, 'touchend', this);
				utils.removeEvent(window, utils.prefixPointerEvent('pointerup'), this);
				utils.removeEvent(window, 'mouseup', this);
			}

			if ( this.options.defaultScrollbars ) {
				this.wrapper.parentNode.removeChild(this.wrapper);
			}
		},

		_start: function (e) {
			var point = e.touches ? e.touches[0] : e;

			e.preventDefault();
			e.stopPropagation();

			this.transitionTime();

			this.initiated = true;
			this.moved = false;
			this.lastPointX	= point.pageX;
			this.lastPointY	= point.pageY;

			this.startTime	= utils.getTime();

			if ( !this.options.disableTouch ) {
				utils.addEvent(window, 'touchmove', this);
			}
			if ( !this.options.disablePointer ) {
				utils.addEvent(window, utils.prefixPointerEvent('pointermove'), this);
			}
			if ( !this.options.disableMouse ) {
				utils.addEvent(window, 'mousemove', this);
			}

			this.scroller._execEvent('beforeScrollStart');
		},

		_move: function (e) {
			var point = e.touches ? e.touches[0] : e,
				deltaX, deltaY,
				newX, newY,
				timestamp = utils.getTime();

			if ( !this.moved ) {
				this.scroller._execEvent('scrollStart');
			}

			this.moved = true;

			deltaX = point.pageX - this.lastPointX;
			this.lastPointX = point.pageX;

			deltaY = point.pageY - this.lastPointY;
			this.lastPointY = point.pageY;

			newX = this.x + deltaX;
			newY = this.y + deltaY;

			this._pos(newX, newY);

			// INSERT POINT: indicator._move

			e.preventDefault();
			e.stopPropagation();
		},

		_end: function (e) {
			if ( !this.initiated ) {
				return;
			}

			this.initiated = false;

			e.preventDefault();
			e.stopPropagation();

			utils.removeEvent(window, 'touchmove', this);
			utils.removeEvent(window, utils.prefixPointerEvent('pointermove'), this);
			utils.removeEvent(window, 'mousemove', this);

			if ( this.scroller.options.snap ) {
				var snap = this.scroller._nearestSnap(this.scroller.x, this.scroller.y);

				var time = this.options.snapSpeed || Math.max(
						Math.max(
							Math.min(Math.abs(this.scroller.x - snap.x), 1000),
							Math.min(Math.abs(this.scroller.y - snap.y), 1000)
						), 300);

				if ( this.scroller.x != snap.x || this.scroller.y != snap.y ) {
					this.scroller.directionX = 0;
					this.scroller.directionY = 0;
					this.scroller.currentPage = snap;
					this.scroller.scrollTo(snap.x, snap.y, time, this.scroller.options.bounceEasing);
				}
			}

			if ( this.moved ) {
				this.scroller._execEvent('scrollEnd');
			}
		},

		transitionTime: function (time) {
			time = time || 0;
			var durationProp = utils.style.transitionDuration;
			if(!durationProp) {
				return;
			}

			this.indicatorStyle[durationProp] = time + 'ms';

			if ( !time && utils.isBadAndroid ) {
				this.indicatorStyle[durationProp] = '0.0001ms';
				// remove 0.0001ms
				var self = this;
				rAF(function() {
					if(self.indicatorStyle[durationProp] === '0.0001ms') {
						self.indicatorStyle[durationProp] = '0s';
					}
				});
			}
		},

		transitionTimingFunction: function (easing) {
			this.indicatorStyle[utils.style.transitionTimingFunction] = easing;
		},

		refresh: function () {
			this.transitionTime();

			if ( this.options.listenX && !this.options.listenY ) {
				this.indicatorStyle.display = this.scroller.hasHorizontalScroll ? 'block' : 'none';
			} else if ( this.options.listenY && !this.options.listenX ) {
				this.indicatorStyle.display = this.scroller.hasVerticalScroll ? 'block' : 'none';
			} else {
				this.indicatorStyle.display = this.scroller.hasHorizontalScroll || this.scroller.hasVerticalScroll ? 'block' : 'none';
			}

			if ( this.scroller.hasHorizontalScroll && this.scroller.hasVerticalScroll ) {
				utils.addClass(this.wrapper, 'iScrollBothScrollbars');
				utils.removeClass(this.wrapper, 'iScrollLoneScrollbar');

				if ( this.options.defaultScrollbars && this.options.customStyle ) {
					if ( this.options.listenX ) {
						this.wrapper.style.right = '8px';
					} else {
						this.wrapper.style.bottom = '8px';
					}
				}
			} else {
				utils.removeClass(this.wrapper, 'iScrollBothScrollbars');
				utils.addClass(this.wrapper, 'iScrollLoneScrollbar');

				if ( this.options.defaultScrollbars && this.options.customStyle ) {
					if ( this.options.listenX ) {
						this.wrapper.style.right = '2px';
					} else {
						this.wrapper.style.bottom = '2px';
					}
				}
			}

			var r = this.wrapper.offsetHeight;	// force refresh

			if ( this.options.listenX ) {
				this.wrapperWidth = this.wrapper.clientWidth;
				if ( this.options.resize ) {
					this.indicatorWidth = Math.max(Math.round(this.wrapperWidth * this.wrapperWidth / (this.scroller.scrollerWidth || this.wrapperWidth || 1)), 8);
					this.indicatorStyle.width = this.indicatorWidth + 'px';
				} else {
					this.indicatorWidth = this.indicator.clientWidth;
				}

				this.maxPosX = this.wrapperWidth - this.indicatorWidth;

				if ( this.options.shrink == 'clip' ) {
					this.minBoundaryX = -this.indicatorWidth + 8;
					this.maxBoundaryX = this.wrapperWidth - 8;
				} else {
					this.minBoundaryX = 0;
					this.maxBoundaryX = this.maxPosX;
				}

				this.sizeRatioX = this.options.speedRatioX || (this.scroller.maxScrollX && (this.maxPosX / this.scroller.maxScrollX));
			}

			if ( this.options.listenY ) {
				this.wrapperHeight = this.wrapper.clientHeight;
				if ( this.options.resize ) {
					this.indicatorHeight = Math.max(Math.round(this.wrapperHeight * this.wrapperHeight / (this.scroller.scrollerHeight || this.wrapperHeight || 1)), 8);
					this.indicatorStyle.height = this.indicatorHeight + 'px';
				} else {
					this.indicatorHeight = this.indicator.clientHeight;
				}

				this.maxPosY = this.wrapperHeight - this.indicatorHeight;

				if ( this.options.shrink == 'clip' ) {
					this.minBoundaryY = -this.indicatorHeight + 8;
					this.maxBoundaryY = this.wrapperHeight - 8;
				} else {
					this.minBoundaryY = 0;
					this.maxBoundaryY = this.maxPosY;
				}

				this.maxPosY = this.wrapperHeight - this.indicatorHeight;
				this.sizeRatioY = this.options.speedRatioY || (this.scroller.maxScrollY && (this.maxPosY / this.scroller.maxScrollY));
			}

			this.updatePosition();
		},

		updatePosition: function () {
			var x = this.options.listenX && Math.round(this.sizeRatioX * this.scroller.x) || 0,
				y = this.options.listenY && Math.round(this.sizeRatioY * this.scroller.y) || 0;

			if ( !this.options.ignoreBoundaries ) {
				if ( x < this.minBoundaryX ) {
					if ( this.options.shrink == 'scale' ) {
						this.width = Math.max(this.indicatorWidth + x, 8);
						this.indicatorStyle.width = this.width + 'px';
					}
					x = this.minBoundaryX;
				} else if ( x > this.maxBoundaryX ) {
					if ( this.options.shrink == 'scale' ) {
						this.width = Math.max(this.indicatorWidth - (x - this.maxPosX), 8);
						this.indicatorStyle.width = this.width + 'px';
						x = this.maxPosX + this.indicatorWidth - this.width;
					} else {
						x = this.maxBoundaryX;
					}
				} else if ( this.options.shrink == 'scale' && this.width != this.indicatorWidth ) {
					this.width = this.indicatorWidth;
					this.indicatorStyle.width = this.width + 'px';
				}

				if ( y < this.minBoundaryY ) {
					if ( this.options.shrink == 'scale' ) {
						this.height = Math.max(this.indicatorHeight + y * 3, 8);
						this.indicatorStyle.height = this.height + 'px';
					}
					y = this.minBoundaryY;
				} else if ( y > this.maxBoundaryY ) {
					if ( this.options.shrink == 'scale' ) {
						this.height = Math.max(this.indicatorHeight - (y - this.maxPosY) * 3, 8);
						this.indicatorStyle.height = this.height + 'px';
						y = this.maxPosY + this.indicatorHeight - this.height;
					} else {
						y = this.maxBoundaryY;
					}
				} else if ( this.options.shrink == 'scale' && this.height != this.indicatorHeight ) {
					this.height = this.indicatorHeight;
					this.indicatorStyle.height = this.height + 'px';
				}
			}

			this.x = x;
			this.y = y;

			if ( this.scroller.options.useTransform ) {
				this.indicatorStyle[utils.style.transform] = 'translate(' + x + 'px,' + y + 'px)' + this.scroller.translateZ;
			} else {
				this.indicatorStyle.left = x + 'px';
				this.indicatorStyle.top = y + 'px';
			}
		},

		_pos: function (x, y) {
			if ( x < 0 ) {
				x = 0;
			} else if ( x > this.maxPosX ) {
				x = this.maxPosX;
			}

			if ( y < 0 ) {
				y = 0;
			} else if ( y > this.maxPosY ) {
				y = this.maxPosY;
			}

			x = this.options.listenX ? Math.round(x / this.sizeRatioX) : this.scroller.x;
			y = this.options.listenY ? Math.round(y / this.sizeRatioY) : this.scroller.y;

			this.scroller.scrollTo(x, y);
		},

		fade: function (val, hold) {
			if ( hold && !this.visible ) {
				return;
			}

			clearTimeout(this.fadeTimeout);
			this.fadeTimeout = null;

			var time = val ? 250 : 500,
				delay = val ? 0 : 300;

			val = val ? '1' : '0';

			this.wrapperStyle[utils.style.transitionDuration] = time + 'ms';

			this.fadeTimeout = setTimeout((function (val) {
				this.wrapperStyle.opacity = val;
				this.visible = +val;
			}).bind(this, val), delay);
		}
	};

	IScroll.utils = utils;

	if ( typeof module != 'undefined' && module.exports ) {
		module.exports = IScroll;
	} else if ( typeof define == 'function' && define.amd ) {
		define( function () { return IScroll; } );
	} else {
		window.IScroll = IScroll;
	}

})(window, document, Math);

/*!
 * fullPage 2.8.8
 * https://github.com/alvarotrigo/fullPage.js
 * @license MIT licensed
 *
 * Copyright (C) 2015 alvarotrigo.com - A project by Alvaro Trigo
 */
(
	function( global, factory ) {
		'use strict';
		if ( typeof define === 'function' && define.amd ) {
			define( ['jquery'], function( $ ) {
				return factory( $, global, global.document, global.Math );
			} );
		} else if ( typeof exports === "object" && exports ) {
			module.exports = factory( require( 'jquery' ), global, global.document, global.Math );
		} else {
			factory( jQuery, global, global.document, global.Math );
		}
	}
)( typeof window !== 'undefined' ? window : this, function( $, window, document, Math, undefined ) {
	'use strict';

	// keeping central set of classnames and selectors
	var WRAPPER     = 'fullpage-wrapper';
	var WRAPPER_SEL = '.' + WRAPPER;

	// slimscroll
	var SCROLLABLE     = 'fp-scrollable';
	var SCROLLABLE_SEL = '.' + SCROLLABLE;

	// util
	var RESPONSIVE     = 'fp-responsive';
	var NO_TRANSITION  = 'fp-notransition';
	var DESTROYED      = 'fp-destroyed';
	var ENABLED        = 'fp-enabled';
	var VIEWING_PREFIX = 'fp-viewing';
	var ACTIVE         = 'active';
	var ACTIVE_SEL     = '.' + ACTIVE;
	var COMPLETELY     = 'fp-completely';
	var COMPLETELY_SEL = '.' + COMPLETELY;

	// section
	var SECTION_DEFAULT_SEL = '.section';
	var SECTION             = 'fp-section';
	var SECTION_SEL         = '.' + SECTION;
	var SECTION_ACTIVE_SEL  = SECTION_SEL + ACTIVE_SEL;
	var SECTION_FIRST_SEL   = SECTION_SEL + ':first';
	var SECTION_LAST_SEL    = SECTION_SEL + ':last';
	var TABLE_CELL          = 'fp-tableCell';
	var TABLE_CELL_SEL      = '.' + TABLE_CELL;
	var AUTO_HEIGHT         = 'fp-auto-height';
	var AUTO_HEIGHT_SEL     = '.fp-auto-height';
	var NORMAL_SCROLL       = 'fp-normal-scroll';
	var NORMAL_SCROLL_SEL   = '.fp-normal-scroll';

	// section nav
	var SECTION_NAV             = 'fp-nav';
	var SECTION_NAV_SEL         = '#' + SECTION_NAV;
	var SECTION_NAV_TOOLTIP     = 'fp-tooltip';
	var SECTION_NAV_TOOLTIP_SEL = '.' + SECTION_NAV_TOOLTIP;
	var SHOW_ACTIVE_TOOLTIP     = 'fp-show-active';

	// slide
	var SLIDE_DEFAULT_SEL    = '.tm-onepage-slide';
	var SLIDE                = 'fp-slide';
	var SLIDE_SEL            = '.' + SLIDE;
	var SLIDE_ACTIVE_SEL     = SLIDE_SEL + ACTIVE_SEL;
	var SLIDES_WRAPPER       = 'fp-slides';
	var SLIDES_WRAPPER_SEL   = '.' + SLIDES_WRAPPER;
	var SLIDES_CONTAINER     = 'fp-slidesContainer';
	var SLIDES_CONTAINER_SEL = '.' + SLIDES_CONTAINER;
	var TABLE                = 'fp-table';

	// slide nav
	var SLIDES_NAV            = 'fp-slidesNav';
	var SLIDES_NAV_SEL        = '.' + SLIDES_NAV;
	var SLIDES_NAV_LINK_SEL   = SLIDES_NAV_SEL + ' a';
	var SLIDES_ARROW          = 'fp-controlArrow';
	var SLIDES_ARROW_SEL      = '.' + SLIDES_ARROW;
	var SLIDES_PREV           = 'fp-prev';
	var SLIDES_PREV_SEL       = '.' + SLIDES_PREV;
	var SLIDES_ARROW_PREV     = SLIDES_ARROW + ' ' + SLIDES_PREV;
	var SLIDES_ARROW_PREV_SEL = SLIDES_ARROW_SEL + SLIDES_PREV_SEL;
	var SLIDES_NEXT           = 'fp-next';
	var SLIDES_NEXT_SEL       = '.' + SLIDES_NEXT;
	var SLIDES_ARROW_NEXT     = SLIDES_ARROW + ' ' + SLIDES_NEXT;
	var SLIDES_ARROW_NEXT_SEL = SLIDES_ARROW_SEL + SLIDES_NEXT_SEL;

	var $window   = $( window );
	var $document = $( document );

	// Default options for iScroll.js used when using scrollOverflow
	var iscrollOptions = {
		scrollbars           : true,
		mouseWheel           : true,
		hideScrollbars       : false,
		fadeScrollbars       : false,
		disableMouse         : true,
		interactiveScrollbars: true
	};

	$.fn.fullpage = function( options ) {
		//only once my friend!
		if ( $( 'html' ).hasClass( ENABLED ) ) {
			displayWarnings();
			return;
		}

		// common jQuery objects
		var $htmlBody = $( 'html, body' );
		var $body     = $( 'body' );

		var FP = $.fn.fullpage;

		// Create some defaults, extending them with any options that were provided
		options = $.extend( {
			//navigation
			menu              : false,
			anchors           : [],
			lockAnchors       : false,
			navigation        : false,
			navigationPosition: 'right',
			navigationTooltips: [],
			showActiveTooltip : false,
			slidesNavigation  : false,
			slidesNavPosition : 'bottom',
			scrollBar         : false,
			hybrid            : false,

			//scrolling
			css3                             : true,
			scrollingSpeed                   : 700,
			autoScrolling                    : true,
			fitToSection                     : true,
			fitToSectionDelay                : 1000,
			easing                           : 'easeInOutCubic',
			easingcss3                       : 'ease',
			loopBottom                       : false,
			loopTop                          : false,
			loopHorizontal                   : true,
			continuousVertical               : false,
			continuousHorizontal             : false,
			scrollHorizontally               : false,
			interlockedSlides                : false,
			resetSliders                     : false,
			fadingEffect                     : false,
			normalScrollElements             : null,
			scrollOverflow                   : false,
			scrollOverflowHandler            : iscrollHandler,
			scrollOverflowOptions            : null,
			touchSensitivity                 : 5,
			normalScrollElementTouchThreshold: 5,
			bigSectionsDestination           : null,

			//Accessibility
			keyboardScrolling: true,
			animateAnchor    : true,
			recordHistory    : true,

			//design
			controlArrows    : true,
			controlArrowColor: '#fff',
			verticalCentered : true,
			sectionsColor    : [],
			paddingTop       : 0,
			paddingBottom    : 0,
			fixedElements    : null,
			responsive       : 0, //backwards compabitility with responsiveWiddth
			responsiveWidth  : 0,
			responsiveHeight : 0,
			responsiveSlides : false,

			//Custom selectors
			sectionSelector: SECTION_DEFAULT_SEL,
			slideSelector  : SLIDE_DEFAULT_SEL,

			//events
			afterLoad      : null,
			onLeave        : null,
			afterRender    : null,
			afterResize    : null,
			afterReBuild   : null,
			afterSlideLoad : null,
			onSlideLeave   : null,
			afterResponsive: null,

			lazyLoading: true
		}, options );

		//flag to avoid very fast sliding for landscape sliders
		var slideMoving = false;

		var isTouchDevice   = navigator.userAgent.match( /(iPhone|iPod|iPad|Android|playbook|silk|BlackBerry|BB10|Windows Phone|Tizen|Bada|webOS|IEMobile|Opera Mini)/ );
		var isTouch         = (
			(
				'ontouchstart' in window
			) || (
				navigator.msMaxTouchPoints > 0
			) || (
				navigator.maxTouchPoints
			)
		);
		var container       = $( this );
		var windowsHeight   = getAvailHeight();
		var isResizing      = false;
		var isWindowFocused = true;
		var lastScrolledDestiny;
		var lastScrolledSlide;
		var canScroll       = true;
		var scrollings      = [];
		var controlPressed;
		var isScrollAllowed = {};
		isScrollAllowed.m   = {
			'up'   : true,
			'down' : true,
			'left' : true,
			'right': true
		};
		isScrollAllowed.k   = $.extend( true, {}, isScrollAllowed.m );

		//timeouts
		var resizeId;
		var afterSectionLoadsId;
		var afterSlideLoadsId;
		var scrollId;
		var scrollId2;
		var keydownId;
		var originals = $.extend( true, {}, options ); //deep copy

		displayWarnings();

		//fixing bug in iScroll with links: https://github.com/cubiq/iscroll/issues/783
		iscrollOptions.click = isTouch; // see #2035

		//extending iScroll options with the user custom ones
		iscrollOptions = $.extend( iscrollOptions, options.scrollOverflowOptions );

		//easeInOutCubic animation included in the plugin
		$.extend( $.easing, {
			easeInOutCubic: function( x, t, b, c, d ) {
				if ( (
						 t /= d / 2
					 ) < 1 ) {
					return c / 2 * t * t * t + b;
				}
				return c / 2 * (
						(
							t -= 2
						) * t * t + 2
					) + b;
			}
		} );

		/* Added by Arden.
		 *
		 * Returns the available height of the container (window height without admin bar)
		 */
		function getAvailHeight() {
			var availHeight       = $( window )
				.height(), offset = 0, adminBar = $( '#wpadminbar' );

			if ( adminBar.length > 0 ) {
				offset += adminBar.outerHeight();
			}

			availHeight = availHeight - offset;

			return availHeight;
		}

		/**
		 * Sets the autoScroll option.
		 * It changes the scroll bar visibility and the history of the site as a result.
		 */
		function setAutoScrolling( value, type ) {
			setVariableState( 'autoScrolling', value, type );

			var element = $( SECTION_ACTIVE_SEL );

			if ( options.autoScrolling && ! options.scrollBar ) {
				$htmlBody.css( {
					'overflow': 'hidden',
					'height'  : '100%'
				} );

				setRecordHistory( originals.recordHistory, 'internal' );

				//for IE touch devices
				container.css( {
					'-ms-touch-action': 'none',
					'touch-action'    : 'none'
				} );

				if ( element.length ) {
					//moving the container up
					silentScroll( element.position().top );
				}

			} else {
				$htmlBody.css( {
					'overflow': 'visible',
					'height'  : 'initial'
				} );

				setRecordHistory( false, 'internal' );

				//for IE touch devices
				container.css( {
					'-ms-touch-action': '',
					'touch-action'    : ''
				} );

				silentScroll( 0 );

				//scrolling the page to the section with no animation
				if ( element.length ) {
					$htmlBody.scrollTop( element.position().top );
				}
			}
		}

		/**
		 * Defines wheter to record the history for each hash change in the URL.
		 */
		function setRecordHistory( value, type ) {
			setVariableState( 'recordHistory', value, type );
		}

		/**
		 * Defines the scrolling speed
		 */
		function setScrollingSpeed( value, type ) {
			setVariableState( 'scrollingSpeed', value, type );
		}

		/**
		 * Sets fitToSection
		 */
		function setFitToSection( value, type ) {
			setVariableState( 'fitToSection', value, type );
		}

		/**
		 * Sets lockAnchors
		 */
		function setLockAnchors( value ) {
			options.lockAnchors = value;
		}

		/**
		 * Adds or remove the possiblity of scrolling through sections by using the mouse wheel or the trackpad.
		 */
		function setMouseWheelScrolling( value ) {
			if ( value ) {
				addMouseWheelHandler();
				addMiddleWheelHandler();
			} else {
				removeMouseWheelHandler();
				removeMiddleWheelHandler();
			}
		}

		/**
		 * Adds or remove the possiblity of scrolling through sections by using the mouse wheel/trackpad or touch gestures.
		 * Optionally a second parameter can be used to specify the direction for which the action will be applied.
		 *
		 * @param directions string containing the direction or directions separated by comma.
		 */
		function setAllowScrolling( value, directions ) {
			if ( typeof directions !== 'undefined' ) {
				directions = directions.replace( / /g, '' ).split( ',' );

				$.each( directions, function( index, direction ) {
					setIsScrollAllowed( value, direction, 'm' );
				} );
			} else if ( value ) {
				setMouseWheelScrolling( true );
				addTouchHandler();
			} else {
				setMouseWheelScrolling( false );
				removeTouchHandler();
			}
		}

		/**
		 * Adds or remove the possiblity of scrolling through sections by using the keyboard arrow keys
		 */
		function setKeyboardScrolling( value, directions ) {
			if ( typeof directions !== 'undefined' ) {
				directions = directions.replace( / /g, '' ).split( ',' );

				$.each( directions, function( index, direction ) {
					setIsScrollAllowed( value, direction, 'k' );
				} );
			} else {
				options.keyboardScrolling = value;
			}
		}

		/**
		 * Moves the page up one section.
		 */
		function moveSectionUp() {
			var prev = $( SECTION_ACTIVE_SEL ).prev( SECTION_SEL );

			//looping to the bottom if there's no more sections above
			if ( ! prev.length && (
					options.loopTop || options.continuousVertical
				) ) {
				prev = $( SECTION_SEL ).last();
			}

			if ( $( 'body' ).hasClass( 'tm-one-page-footer-expanded' ) ) {
				setTimeout( function() {
					$( 'body' ).removeClass( 'tm-one-page-footer-expanded' );
				}, 600 );
				return false;
			}

			if ( prev.length ) {
				scrollPage( prev, null, true );
			}
		}

		/**
		 * Moves the page down one section.
		 */
		function moveSectionDown() {
			var next = $( SECTION_ACTIVE_SEL ).next( SECTION_SEL );

			//looping to the top if there's no more sections below
			if ( ! next.length && (
					options.loopBottom || options.continuousVertical
				) ) {
				next = $( SECTION_SEL ).first();
			}

			if ( ! next.length && $( '#page-footer-wrapper' ).length > 0 && ! $( 'body' )
					.hasClass( 'tm-one-page-footer-expanded' ) ) {
				$( 'body' ).addClass( 'tm-one-page-footer-expanded' );
			}

			if ( next.length ) {
				scrollPage( next, null, false );
			}
		}

		/**
		 * Moves the page to the given section and slide with no animation.
		 * Anchors or index positions can be used as params.
		 */
		function silentMoveTo( sectionAnchor, slideAnchor ) {
			setScrollingSpeed( 0, 'internal' );
			moveTo( sectionAnchor, slideAnchor );
			setScrollingSpeed( originals.scrollingSpeed, 'internal' );
		}

		/**
		 * Moves the page to the given section and slide.
		 * Anchors or index positions can be used as params.
		 */
		function moveTo( sectionAnchor, slideAnchor ) {
			var destiny = getSectionByAnchor( sectionAnchor );

			if ( typeof slideAnchor !== 'undefined' ) {
				scrollPageAndSlide( sectionAnchor, slideAnchor );
			} else if ( destiny.length > 0 ) {
				scrollPage( destiny );
			}
			if ( $( 'body' ).hasClass( 'tm-one-page-footer-expanded' ) ) {
				$( 'body' ).removeClass( 'tm-one-page-footer-expanded' );
			}
		}

		/**
		 * Slides right the slider of the active section.
		 * Optional `section` param.
		 */
		function moveSlideRight( section ) {
			moveSlide( 'right', section );
		}

		/**
		 * Slides left the slider of the active section.
		 * Optional `section` param.
		 */
		function moveSlideLeft( section ) {
			moveSlide( 'left', section );
		}

		/**
		 * When resizing is finished, we adjust the slides sizes and positions
		 */
		function reBuild( resizing ) {
			if ( container.hasClass( DESTROYED ) || ! canScroll ) {
				return;
			}  //nothing to do if the plugin was destroyed

			isResizing = true;

			windowsHeight = getAvailHeight();  //updating global var

			if ( container.hasClass( 'tm-enable-onepage-animation' ) ) {
				container.css( 'height', windowsHeight );
			}

			$( SECTION_SEL ).each( function() {
				var slidesWrap = $( this ).find( SLIDES_WRAPPER_SEL );
				var slides     = $( this ).find( SLIDE_SEL );

				//adjusting the height of the table-cell for IE and Firefox
				if ( options.verticalCentered ) {
					$( this ).find( TABLE_CELL_SEL ).css( 'height', getTableHeight( $( this ) ) + 'px' );
				}

				$( this ).css( 'height', windowsHeight + 'px' );

				//resizing the scrolling divs
				if ( options.scrollOverflow ) {
					if ( slides.length ) {
						slides.each( function() {
							createScrollBar( $( this ) );
						} );
					} else {
						createScrollBar( $( this ) );
					}
				}

				//adjusting the position fo the FULL WIDTH slides...
				if ( slides.length > 1 ) {
					landscapeScroll( slidesWrap, slidesWrap.find( SLIDE_ACTIVE_SEL ) );
				}
			} );

			var activeSection = $( SECTION_ACTIVE_SEL );
			var sectionIndex  = activeSection.index( SECTION_SEL );

			//isn't it the first section?
			if ( sectionIndex ) {
				//adjusting the position for the current section
				silentMoveTo( sectionIndex + 1 );
			}

			isResizing = false;
			$.isFunction( options.afterResize ) && resizing && options.afterResize.call( container );
			$.isFunction( options.afterReBuild ) && ! resizing && options.afterReBuild.call( container );
		}

		/**
		 * Turns fullPage.js to normal scrolling mode when the viewport `width` or `height`
		 * are smaller than the set limit values.
		 */
		function setResponsive( active ) {
			var isResponsive = $body.hasClass( RESPONSIVE );

			if ( active ) {
				if ( ! isResponsive ) {
					setAutoScrolling( false, 'internal' );
					setFitToSection( false, 'internal' );
					$( SECTION_NAV_SEL ).hide();
					$body.addClass( RESPONSIVE );
					$.isFunction( options.afterResponsive ) && options.afterResponsive.call( container, active );
				}
			} else if ( isResponsive ) {
				setAutoScrolling( originals.autoScrolling, 'internal' );
				setFitToSection( originals.autoScrolling, 'internal' );
				$( SECTION_NAV_SEL ).show();
				$body.removeClass( RESPONSIVE );
				$.isFunction( options.afterResponsive ) && options.afterResponsive.call( container, active );
			}
		}

		function getFullpageData() {
			return {
				options  : options,
				internals: {
					getXmovement         : getXmovement,
					removeAnimation      : removeAnimation,
					getTransforms        : getTransforms,
					lazyLoad             : lazyLoad,
					addAnimation         : addAnimation,
					performHorizontalMove: performHorizontalMove,
					silentLandscapeScroll: silentLandscapeScroll,
					keepSlidesPosition   : keepSlidesPosition,
					silentScroll         : silentScroll,
					styleSlides          : styleSlides
				}
			};
		}

		if ( $( this ).length ) {
			//public functions
			FP.setAutoScrolling       = setAutoScrolling;
			FP.setRecordHistory       = setRecordHistory;
			FP.setScrollingSpeed      = setScrollingSpeed;
			FP.setFitToSection        = setFitToSection;
			FP.setLockAnchors         = setLockAnchors;
			FP.setMouseWheelScrolling = setMouseWheelScrolling;
			FP.setAllowScrolling      = setAllowScrolling;
			FP.setKeyboardScrolling   = setKeyboardScrolling;
			FP.moveSectionUp          = moveSectionUp;
			FP.moveSectionDown        = moveSectionDown;
			FP.silentMoveTo           = silentMoveTo;
			FP.moveTo                 = moveTo;
			FP.moveSlideRight         = moveSlideRight;
			FP.moveSlideLeft          = moveSlideLeft;
			FP.reBuild                = reBuild;
			FP.setResponsive          = setResponsive;
			FP.getFullpageData        = getFullpageData;
			FP.destroy                = destroy;

			init();

			bindEvents();
		}

		function init() {
			//if css3 is not supported, it will use jQuery animations
			if ( options.css3 ) {
				options.css3 = support3d();
			}

			options.scrollBar = options.scrollBar || options.hybrid;

			setOptionsFromDOM();

			prepareDom();
			setAllowScrolling( true );

			setAutoScrolling( options.autoScrolling, 'internal' );

			//the starting point is a slide?
			var activeSlide = $( SECTION_ACTIVE_SEL ).find( SLIDE_ACTIVE_SEL );

			//the active section isn't the first one? Is not the first slide of the first section? Then we load that section/slide by default.
			if ( activeSlide.length && (
					$( SECTION_ACTIVE_SEL ).index( SECTION_SEL ) !== 0 || (
						$( SECTION_ACTIVE_SEL ).index( SECTION_SEL ) === 0 && activeSlide.index() !== 0
					)
				) ) {
				silentLandscapeScroll( activeSlide );
			}

			responsive();

			//setting the class for the body element
			setBodyClass();

			if ( document.readyState === 'complete' ) {
				scrollToAnchor();
			}
			$window.on( 'load', scrollToAnchor );
		}

		function bindEvents() {
			$window
			//when scrolling...
				.on( 'scroll', scrollHandler )

				//detecting any change on the URL to scroll to the given anchor link
				//(a way to detect back history button as we play with the hashes on the URL)
				.on( 'hashchange', hashChangeHandler )

				//when opening a new tab (ctrl + t), `control` won't be pressed when comming back.
				.blur( blurHandler )

				//when resizing the site, we adjust the heights of the sections, slimScroll...
				.resize( resizeHandler );

			$document
			//Sliding with arrow keys, both, vertical and horizontal
				.keydown( keydownHandler )

				//to prevent scrolling while zooming
				.keyup( keyUpHandler )

				//Scrolls to the section when clicking the navigation bullet
				.on( 'click touchstart', SECTION_NAV_SEL + ' a', sectionBulletHandler )

				//Scrolls the slider to the given slide destination for the given section
				.on( 'click touchstart', SLIDES_NAV_LINK_SEL, slideBulletHandler )

				.on( 'click', SECTION_NAV_TOOLTIP_SEL, tooltipTextHandler );

			//Scrolling horizontally when clicking on the slider controls.
			$( SECTION_SEL ).on( 'click touchstart', SLIDES_ARROW_SEL, slideArrowHandler );

			/**
			 * Applying normalScroll elements.
			 * Ignoring the scrolls over the specified selectors.
			 */
			if ( options.normalScrollElements ) {
				$document.on( 'mouseenter', options.normalScrollElements, function() {
					setMouseWheelScrolling( false );
				} );

				$document.on( 'mouseleave', options.normalScrollElements, function() {
					setMouseWheelScrolling( true );
				} );
			}
		}

		/**
		 * Setting options from DOM elements if they are not provided.
		 */
		function setOptionsFromDOM() {
			var sections = container.find( options.sectionSelector );

			//no anchors option? Checking for them in the DOM attributes
			if ( ! options.anchors.length ) {
				options.anchors = sections.filter( '[data-anchor]' ).map( function() {
					return $( this ).data( 'anchor' ).toString();
				} ).get();
			}

			//no tooltipos option? Checking for them in the DOM attributes
			if ( ! options.navigationTooltips.length ) {
				options.navigationTooltips = sections.filter( '[data-tooltip]' ).map( function() {
					return $( this ).data( 'tooltip' ).toString();
				} ).get();
			}
		}

		/**
		 * Works over the DOM structure to set it up for the current fullpage optionss.
		 */
		function prepareDom() {
			container.css( {
				'height'  : '100%',
				'position': 'relative'
			} );

			//adding a class to recognize the container internally in the code
			container.addClass( WRAPPER );
			$( 'html' ).addClass( ENABLED );

			//due to https://github.com/alvarotrigo/fullPage.js/issues/1502
			windowsHeight = getAvailHeight();

			container.removeClass( DESTROYED ); //in case it was destroyed before initilizing it again

			addInternalSelectors();

			if ( container.hasClass( 'tm-enable-onepage-animation' ) ) {
				container.css( 'height', windowsHeight );
			}

			//styling the sections / slides / menu
			$( SECTION_SEL ).each( function( index ) {
				var section   = $( this );
				var slides    = section.find( SLIDE_SEL );
				var numSlides = slides.length;

				styleSection( section, index );
				styleMenu( section, index );

				// if there's any slide
				if ( numSlides > 0 ) {
					styleSlides( section, slides, numSlides );
				} else {
					if ( options.verticalCentered ) {
						addTableClass( section );
					}
				}
			} );

			//fixed elements need to be moved out of the plugin container due to problems with CSS3.
			if ( options.fixedElements && options.css3 ) {
				$( options.fixedElements ).appendTo( $body );
			}

			//vertical centered of the navigation + active bullet
			if ( options.navigation ) {
				addVerticalNavigation();
			}

			enableYoutubeAPI();

			if ( options.scrollOverflow ) {
				if ( document.readyState === 'complete' ) {
					createScrollBarHandler();
				}
				//after DOM and images are loaded
				$window.on( 'load', createScrollBarHandler );
			} else {
				afterRenderActions();
			}
		}

		/**
		 * Styles the horizontal slides for a section.
		 */
		function styleSlides( section, slides, numSlides ) {
			var sliderWidth = numSlides * 100;
			var slideWidth  = 100 / numSlides;

			slides.wrapAll( '<div class="' + SLIDES_CONTAINER + '" />' );
			slides.parent().wrap( '<div class="' + SLIDES_WRAPPER + '" />' );

			section.find( SLIDES_CONTAINER_SEL ).css( 'width', sliderWidth + '%' );

			if ( numSlides > 1 ) {
				if ( options.controlArrows ) {
					createSlideArrows( section );
				}

				if ( options.slidesNavigation ) {
					addSlidesNavigation( section, numSlides );
				}
			}

			slides.each( function( index ) {
				$( this ).css( 'width', slideWidth + '%' );

				if ( options.verticalCentered ) {
					addTableClass( $( this ) );
				}
			} );

			var startingSlide = section.find( SLIDE_ACTIVE_SEL );

			//if the slide won't be an starting point, the default will be the first one
			//the active section isn't the first one? Is not the first slide of the first section? Then we load that section/slide by default.
			if ( startingSlide.length && (
					$( SECTION_ACTIVE_SEL ).index( SECTION_SEL ) !== 0 || (
						$( SECTION_ACTIVE_SEL ).index( SECTION_SEL ) === 0 && startingSlide.index() !== 0
					)
				) ) {
				silentLandscapeScroll( startingSlide );
			} else {
				slides.eq( 0 ).addClass( ACTIVE );
			}
		}

		/**
		 * Styling vertical sections
		 */
		function styleSection( section, index ) {
			//if no active section is defined, the 1st one will be the default one
			if ( ! index && $( SECTION_ACTIVE_SEL ).length === 0 ) {
				section.addClass( ACTIVE );
			}

			section.css( 'height', windowsHeight + 'px' );

			if ( options.paddingTop ) {
				section.css( 'padding-top', options.paddingTop );
			}

			if ( options.paddingBottom ) {
				section.css( 'padding-bottom', options.paddingBottom );
			}

			if ( typeof options.sectionsColor[index] !== 'undefined' ) {
				section.css( 'background-color', options.sectionsColor[index] );
			}

			if ( typeof options.anchors[index] !== 'undefined' ) {
				section.attr( 'data-anchor', options.anchors[index] );
			}
		}

		/**
		 * Sets the data-anchor attributes to the menu elements and activates the current one.
		 */
		function styleMenu( section, index ) {
			if ( typeof options.anchors[index] !== 'undefined' ) {
				//activating the menu / nav element on load
				if ( section.hasClass( ACTIVE ) ) {
					activateMenuAndNav( options.anchors[index], index );
				}
			}

			//moving the menu outside the main container if it is inside (avoid problems with fixed positions when using CSS3 tranforms)
			if ( options.menu && options.css3 && $( options.menu ).closest( WRAPPER_SEL ).length ) {
				$( options.menu ).appendTo( $body );
			}
		}

		/**
		 * Adds internal classes to be able to provide customizable selectors
		 * keeping the link with the style sheet.
		 */
		function addInternalSelectors() {
			container.find( options.sectionSelector ).addClass( SECTION );
			container.find( options.slideSelector ).addClass( SLIDE );
		}

		/**
		 * Creates the control arrows for the given section
		 */
		function createSlideArrows( section ) {
			section.find( SLIDES_WRAPPER_SEL )
				   .after( '<div class="' + SLIDES_ARROW_PREV + '"></div><div class="' + SLIDES_ARROW_NEXT + '"></div>' );

			if ( options.controlArrowColor != '#fff' ) {
				section.find( SLIDES_ARROW_NEXT_SEL )
					   .css( 'border-color', 'transparent transparent transparent ' + options.controlArrowColor );
				section.find( SLIDES_ARROW_PREV_SEL )
					   .css( 'border-color', 'transparent ' + options.controlArrowColor + ' transparent transparent' );
			}

			if ( ! options.loopHorizontal ) {
				section.find( SLIDES_ARROW_PREV_SEL ).hide();
			}
		}

		/**
		 * Creates a vertical navigation bar.
		 */
		function addVerticalNavigation() {
			$body.append( '<div id="' + SECTION_NAV + '"><ul></ul></div>' );
			var nav = $( SECTION_NAV_SEL );

			nav.addClass( function() {
				return options.showActiveTooltip ? SHOW_ACTIVE_TOOLTIP + ' ' + options.navigationPosition : options.navigationPosition;
			} );

			for ( var i = 0; i < $( SECTION_SEL ).length; i ++ ) {
				var link = '';
				if ( options.anchors.length ) {
					link = options.anchors[i];
				}

				var li = '<li><a href="#' + link + '"><span></span></a>';

				// Only add tooltip if needed (defined by user)
				var tooltip = options.navigationTooltips[i];

				if ( typeof tooltip !== 'undefined' && tooltip !== '' ) {
					li += '<div class="' + SECTION_NAV_TOOLTIP + ' ' + options.navigationPosition + '">' + tooltip + '</div>';
				}

				li += '</li>';

				nav.find( 'ul' ).append( li );
			}

			//centering it vertically
			$( SECTION_NAV_SEL )
				.css( 'margin-top', '-' + (
						$( SECTION_NAV_SEL ).height() / 2
					) + 'px' );

			//activating the current active section
			$( SECTION_NAV_SEL )
				.find( 'li' )
				.eq( $( SECTION_ACTIVE_SEL ).index( SECTION_SEL ) )
				.find( 'a' )
				.addClass( ACTIVE );
		}

		/**
		 * Creates the slim scroll scrollbar for the sections and slides inside them.
		 */
		function createScrollBarHandler() {
			$( SECTION_SEL ).each( function() {
				var slides = $( this ).find( SLIDE_SEL );

				if ( slides.length ) {
					slides.each( function() {
						createScrollBar( $( this ) );
					} );
				} else {
					createScrollBar( $( this ) );
				}

			} );
			afterRenderActions();
		}

		/*
		 * Enables the Youtube videos API so we can control their flow if necessary.
		 */
		function enableYoutubeAPI() {
			container.find( 'iframe[src*="youtube.com/embed/"]' ).each( function() {
				addURLParam( $( this ), 'enablejsapi=1' );
			} );
		}

		/**
		 * Adds a new parameter and its value to the `src` of a given element
		 */
		function addURLParam( element, newParam ) {
			var originalSrc = element.attr( 'src' );
			element.attr( 'src', originalSrc + getUrlParamSign( originalSrc ) + newParam );
		}

		/*
		 * Returns the prefix sign to use for a new parameter in an existen URL.
		 *
		 * @return {String}  ? | &
		 */
		function getUrlParamSign( url ) {
			return (
				! /\?/.test( url )
			) ? '?' : '&';
		}

		/**
		 * Actions and callbacks to fire afterRender
		 */
		function afterRenderActions() {
			var section = $( SECTION_ACTIVE_SEL );

			section.addClass( COMPLETELY );

			if ( options.scrollOverflowHandler.afterRender ) {
				options.scrollOverflowHandler.afterRender( section );
			}
			lazyLoad( section );
			playMedia( section );

			$.isFunction( options.afterLoad ) && options.afterLoad.call( section, section.data( 'anchor' ), (
				section.index( SECTION_SEL ) + 1
			) );
			$.isFunction( options.afterRender ) && options.afterRender.call( container );
		}


		var isScrolling = false;
		var lastScroll  = 0;

		//when scrolling...
		function scrollHandler() {
			var currentSection;

			if ( ! options.autoScrolling || options.scrollBar ) {
				var currentScroll       = $window.scrollTop();
				var scrollDirection     = getScrollDirection( currentScroll );
				var visibleSectionIndex = 0;
				var screen_mid          = currentScroll + (
						$window.height() / 2.0
					);
				var isAtBottom          = $body.height() - $window.height() === currentScroll;
				var sections            = document.querySelectorAll( SECTION_SEL );

				//when using `auto-height` for a small last section it won't be centered in the viewport
				if ( isAtBottom ) {
					visibleSectionIndex = sections.length - 1;
				}
				//is at top? when using `auto-height` for a small first section it won't be centered in the viewport
				else if ( ! currentScroll ) {
					visibleSectionIndex = 0;
				}

				//taking the section which is showing more content in the viewport
				else {
					for ( var i = 0; i < sections.length; ++ i ) {
						var section = sections[i];

						// Pick the the last section which passes the middle line of the screen.
						if ( section.offsetTop <= screen_mid ) {
							visibleSectionIndex = i;
						}
					}
				}

				if ( isCompletelyInViewPort( scrollDirection ) ) {
					if ( ! $( SECTION_ACTIVE_SEL ).hasClass( COMPLETELY ) ) {
						$( SECTION_ACTIVE_SEL ).addClass( COMPLETELY ).siblings().removeClass( COMPLETELY );
					}
				}

				//geting the last one, the current one on the screen
				currentSection = $( sections ).eq( visibleSectionIndex );

				//setting the visible section as active when manually scrolling
				//executing only once the first time we reach the section
				if ( ! currentSection.hasClass( ACTIVE ) ) {
					isScrolling             = true;
					var leavingSection      = $( SECTION_ACTIVE_SEL );
					var leavingSectionIndex = leavingSection.index( SECTION_SEL ) + 1;
					var yMovement           = getYmovement( currentSection );
					var anchorLink          = currentSection.data( 'anchor' );
					var sectionIndex        = currentSection.index( SECTION_SEL ) + 1;
					var activeSlide         = currentSection.find( SLIDE_ACTIVE_SEL );
					var slideIndex;
					var slideAnchorLink;

					if ( activeSlide.length ) {
						slideAnchorLink = activeSlide.data( 'anchor' );
						slideIndex      = activeSlide.index();
					}

					if ( canScroll ) {
						currentSection.addClass( ACTIVE ).siblings().removeClass( ACTIVE );

						$.isFunction( options.onLeave ) && options.onLeave.call( leavingSection, leavingSectionIndex, sectionIndex, yMovement );
						$.isFunction( options.afterLoad ) && options.afterLoad.call( currentSection, anchorLink, sectionIndex );

						stopMedia( leavingSection );
						lazyLoad( currentSection );
						playMedia( currentSection );

						activateMenuAndNav( anchorLink, sectionIndex - 1 );

						if ( options.anchors.length ) {
							//needed to enter in hashChange event when using the menu with anchor links
							lastScrolledDestiny = anchorLink;
						}
						setState( slideIndex, slideAnchorLink, anchorLink, sectionIndex );
					}

					//small timeout in order to avoid entering in hashChange event when scrolling is not finished yet
					clearTimeout( scrollId );
					scrollId = setTimeout( function() {
						isScrolling = false;
					}, 100 );
				}

				if ( options.fitToSection ) {
					//for the auto adjust of the viewport to fit a whole section
					clearTimeout( scrollId2 );

					scrollId2 = setTimeout( function() {
						//checking fitToSection again in case it was set to false before the timeout delay
						if ( canScroll && options.fitToSection ) {
							//allows to scroll to an active section and
							//if the section is already active, we prevent firing callbacks
							if ( $( SECTION_ACTIVE_SEL ).is( currentSection ) ) {
								isResizing = true;
							}
							scrollPage( $( SECTION_ACTIVE_SEL ) );

							isResizing = false;
						}
					}, options.fitToSectionDelay );
				}
			}
		}

		/**
		 * Determines whether the active section has seen in its whole or not.
		 */
		function isCompletelyInViewPort( movement ) {
			var top    = $( SECTION_ACTIVE_SEL ).position().top;
			var bottom = top + $window.height();

			if ( movement == 'up' ) {
				return bottom >= (
						$window.scrollTop() + $window.height()
					);
			}
			return top <= $window.scrollTop();
		}

		/**
		 * Gets the directon of the the scrolling fired by the scroll event.
		 */
		function getScrollDirection( currentScroll ) {
			var direction = currentScroll > lastScroll ? 'down' : 'up';

			lastScroll = currentScroll;

			//needed for auto-height sections to determine if we want to scroll to the top or bottom of the destination
			previousDestTop = currentScroll;

			return direction;
		}

		/**
		 * Determines the way of scrolling up or down:
		 * by 'automatically' scrolling a section or by using the default and normal scrolling.
		 */
		function scrolling( type, scrollable ) {
			if ( ! isScrollAllowed.m[type] || ! canScroll ) {
				return;
			}
			var check         = (
				type === 'down'
			) ? 'bottom' : 'top';
			var scrollSection = (
				type === 'down'
			) ? moveSectionDown : moveSectionUp;

			if ( scrollable.length > 0 ) {
				//is the scrollbar at the start/end of the scroll?
				if ( options.scrollOverflowHandler.isScrolled( check, scrollable ) ) {
					scrollSection();
				} else {
					return true;
				}
			} else {
				// moved up/down
				scrollSection();
			}
		}

		/*
		 * Preventing bouncing in iOS #2285
		 */
		function preventBouncing( event ) {
			var e = event.originalEvent;
			if ( options.autoScrolling && isReallyTouch( e ) ) {
				//preventing the easing on iOS devices
				event.preventDefault();
			}
		}

		var touchStartY = 0;
		var touchStartX = 0;
		var touchEndY   = 0;
		var touchEndX   = 0;

		/* Detecting touch events

		 * As we are changing the top property of the page on scrolling, we can not use the traditional way to detect it.
		 * This way, the touchstart and the touch moves shows an small difference between them which is the
		 * used one to determine the direction.
		 */
		function touchMoveHandler( event ) {
			var e             = event.originalEvent;
			var activeSection = $( e.target ).closest( SECTION_SEL );

			// additional: if one of the normalScrollElements isn't within options.normalScrollElementTouchThreshold hops up the DOM chain
			if ( ! checkParentForNormalScrollElement( event.target ) && isReallyTouch( e ) ) {

				if ( options.autoScrolling ) {
					//preventing the easing on iOS devices
					event.preventDefault();
				}

				var scrollable = options.scrollOverflowHandler.scrollable( activeSection );

				if ( canScroll && ! slideMoving ) { //if theres any #
					var touchEvents = getEventsPage( e );

					touchEndY = touchEvents.y;
					touchEndX = touchEvents.x;

					//if movement in the X axys is greater than in the Y and the currect section has slides...
					if ( activeSection.find( SLIDES_WRAPPER_SEL ).length && Math.abs( touchStartX - touchEndX ) > (
							Math.abs( touchStartY - touchEndY )
						) ) {

						//is the movement greater than the minimum resistance to scroll?
						if ( Math.abs( touchStartX - touchEndX ) > (
								$window.outerWidth() / 100 * options.touchSensitivity
							) ) {
							if ( touchStartX > touchEndX ) {
								if ( isScrollAllowed.m.right ) {
									moveSlideRight( activeSection ); //next
								}
							} else {
								if ( isScrollAllowed.m.left ) {
									moveSlideLeft( activeSection ); //prev
								}
							}
						}
					}

					//vertical scrolling (only when autoScrolling is enabled)
					else if ( options.autoScrolling ) {

						//is the movement greater than the minimum resistance to scroll?
						if ( Math.abs( touchStartY - touchEndY ) > (
								$window.height() / 100 * options.touchSensitivity
							) ) {
							if ( touchStartY > touchEndY ) {
								scrolling( 'down', scrollable );
							} else if ( touchEndY > touchStartY ) {
								scrolling( 'up', scrollable );
							}
						}
					}
				}
			}

		}

		/**
		 * recursive function to loop up the parent nodes to check if one of them exists in options.normalScrollElements
		 * Currently works well for iOS - Android might need some testing
		 * @param  {Element} el  target element / jquery selector (in subsequent nodes)
		 * @param  {int}     hop current hop compared to options.normalScrollElementTouchThreshold
		 * @return {boolean} true if there is a match to options.normalScrollElements
		 */
		function checkParentForNormalScrollElement( el, hop ) {
			hop        = hop || 0;
			var parent = $( el ).parent();

			if ( hop < options.normalScrollElementTouchThreshold && parent.is( options.normalScrollElements ) ) {
				return true;
			} else if ( hop == options.normalScrollElementTouchThreshold ) {
				return false;
			} else {
				return checkParentForNormalScrollElement( parent, ++ hop );
			}
		}

		/**
		 * As IE >= 10 fires both touch and mouse events when using a mouse in a touchscreen
		 * this way we make sure that is really a touch event what IE is detecting.
		 */
		function isReallyTouch( e ) {
			//if is not IE   ||  IE is detecting `touch` or `pen`
			return typeof e.pointerType === 'undefined' || e.pointerType != 'mouse';
		}

		/**
		 * Handler for the touch start event.
		 */
		function touchStartHandler( event ) {
			var e = event.originalEvent;

			//stopping the auto scroll to adjust to a section
			if ( options.fitToSection ) {
				$htmlBody.stop();
			}

			if ( isReallyTouch( e ) ) {
				var touchEvents = getEventsPage( e );
				touchStartY     = touchEvents.y;
				touchStartX     = touchEvents.x;
			}
		}

		/**
		 * Gets the average of the last `number` elements of the given array.
		 */
		function getAverage( elements, number ) {
			var sum = 0;

			//taking `number` elements from the end to make the average, if there are not enought, 1
			var lastElements = elements.slice( Math.max( elements.length - number, 1 ) );

			for ( var i = 0; i < lastElements.length; i ++ ) {
				sum = sum + lastElements[i];
			}

			return Math.ceil( sum / number );
		}

		/**
		 * Detecting mousewheel scrolling
		 *
		 * http://blogs.sitepointstatic.com/examples/tech/mouse-wheel/index.html
		 * http://www.sitepoint.com/html5-javascript-mouse-wheel/
		 */
		var prevTime = new Date().getTime();

		function MouseWheelHandler( e ) {
			var curTime        = new Date().getTime();
			var isNormalScroll = $( COMPLETELY_SEL ).hasClass( NORMAL_SCROLL );

			//autoscrolling and not zooming?
			if ( options.autoScrolling && ! controlPressed && ! isNormalScroll ) {
				// cross-browser wheel delta
				e         = e || window.event;
				var value = e.wheelDelta || - e.deltaY || - e.detail;
				var delta = Math.max( - 1, Math.min( 1, value ) );

				var horizontalDetection   = typeof e.wheelDeltaX !== 'undefined' || typeof e.deltaX !== 'undefined';
				var isScrollingVertically = (
												Math.abs( e.wheelDeltaX ) < Math.abs( e.wheelDelta )
											) || (
											Math.abs( e.deltaX ) < Math.abs( e.deltaY ) || ! horizontalDetection
											);

				//Limiting the array to 150 (lets not waste memory!)
				if ( scrollings.length > 149 ) {
					scrollings.shift();
				}

				//keeping record of the previous scrollings
				scrollings.push( Math.abs( value ) );

				//preventing to scroll the site on mouse wheel when scrollbar is present
				if ( options.scrollBar ) {
					e.preventDefault ? e.preventDefault() : e.returnValue = false;
				}

				var activeSection = $( SECTION_ACTIVE_SEL );
				var scrollable    = options.scrollOverflowHandler.scrollable( activeSection );

				//time difference between the last scroll and the current one
				var timeDiff = curTime - prevTime;
				prevTime     = curTime;

				//haven't they scrolled in a while?
				//(enough to be consider a different scrolling action to scroll another section)
				if ( timeDiff > 200 ) {
					//emptying the array, we dont care about old scrollings for our averages
					scrollings = [];
				}

				if ( canScroll ) {
					var averageEnd     = getAverage( scrollings, 10 );
					var averageMiddle  = getAverage( scrollings, 70 );
					var isAccelerating = averageEnd >= averageMiddle;

					//to avoid double swipes...
					if ( isAccelerating && isScrollingVertically ) {
						//scrolling down?
						if ( delta < 0 ) {
							scrolling( 'down', scrollable );

							//scrolling up?
						} else {
							scrolling( 'up', scrollable );
						}
					}
				}

				return false;
			}

			if ( options.fitToSection ) {
				//stopping the auto scroll to adjust to a section
				$htmlBody.stop();
			}
		}

		/**
		 * Slides a slider to the given direction.
		 * Optional `section` param.
		 */
		function moveSlide( direction, section ) {
			var activeSection = typeof section === 'undefined' ? $( SECTION_ACTIVE_SEL ) : section;
			var slides        = activeSection.find( SLIDES_WRAPPER_SEL );
			var numSlides     = slides.find( SLIDE_SEL ).length;

			// more than one slide needed and nothing should be sliding
			if ( ! slides.length || slideMoving || numSlides < 2 || ! canScroll ) {
				return;
			}

			var currentSlide = slides.find( SLIDE_ACTIVE_SEL );
			var destiny      = null;

			if ( direction === 'left' ) {
				destiny = currentSlide.prev( SLIDE_SEL );
			} else {
				destiny = currentSlide.next( SLIDE_SEL );
			}

			//isn't there a next slide in the secuence?
			if ( ! destiny.length ) {
				//respect loopHorizontal settin
				if ( ! options.loopHorizontal ) {
					return;
				}

				if ( direction === 'left' ) {
					destiny = currentSlide.siblings( ':last' );
				} else {
					destiny = currentSlide.siblings( ':first' );
				}
			}

			slideMoving = true;

			landscapeScroll( slides, destiny, direction );
		}

		/**
		 * Maintains the active slides in the viewport
		 * (Because the `scroll` animation might get lost with some actions, such as when using continuousVertical)
		 */
		function keepSlidesPosition() {
			$( SLIDE_ACTIVE_SEL ).each( function() {
				silentLandscapeScroll( $( this ), 'internal' );
			} );
		}

		var previousDestTop = 0;

		/**
		 * Returns the destination Y position based on the scrolling direction and
		 * the height of the section.
		 */
		function getDestinationPosition( element ) {
			var elemPosition = element.position();

			//top of the desination will be at the top of the viewport
			var position               = elemPosition.top;
			var isScrollingDown        = elemPosition.top > previousDestTop;
			var sectionBottom          = position - windowsHeight + element.outerHeight();
			var bigSectionsDestination = options.bigSectionsDestination;

			//is the destination element bigger than the viewport?
			if ( element.outerHeight() > windowsHeight ) {
				//scrolling up?
				if ( ! isScrollingDown && ! bigSectionsDestination || bigSectionsDestination === 'bottom' ) {
					position = sectionBottom;
				}
			}

			//sections equal or smaller than the viewport height && scrolling down? ||  is resizing and its in the last section
			else if ( isScrollingDown || (
					isResizing && element.is( ':last-child' )
				) ) {
				//The bottom of the destination will be at the bottom of the viewport
				position = sectionBottom;
			}

			/*
			 Keeping record of the last scrolled position to determine the scrolling direction.
			 No conventional methods can be used as the scroll bar might not be present
			 AND the section might not be active if it is auto-height and didnt reach the middle
			 of the viewport.
			 */
			previousDestTop = position;
			return position;
		}

		/**
		 * Scrolls the site to the given element and scrolls to the slide if a callback is given.
		 */
		function scrollPage( element, callback, isMovementUp ) {
			if ( typeof element === 'undefined' || ! canScroll ) {
				return;
			} //there's no element to scroll, leaving the function

			var dtop = getDestinationPosition( element );
			var slideAnchorLink;
			var slideIndex;

			//local variables
			var v = {
				element       : element,
				callback      : callback,
				isMovementUp  : isMovementUp,
				dtop          : dtop,
				yMovement     : getYmovement( element ),
				anchorLink    : element.data( 'anchor' ),
				sectionIndex  : element.index( SECTION_SEL ),
				activeSlide   : element.find( SLIDE_ACTIVE_SEL ),
				activeSection : $( SECTION_ACTIVE_SEL ),
				leavingSection: $( SECTION_ACTIVE_SEL ).index( SECTION_SEL ) + 1,

				//caching the value of isResizing at the momment the function is called
				//because it will be checked later inside a setTimeout and the value might change
				localIsResizing: isResizing
			};

			//quiting when destination scroll is the same as the current one
			if ( (
					 v.activeSection.is( element ) && ! isResizing
				 ) || (
					 options.scrollBar && $window.scrollTop() === v.dtop && ! element.hasClass( AUTO_HEIGHT )
				 ) ) {
				return;
			}

			if ( v.activeSlide.length ) {
				slideAnchorLink = v.activeSlide.data( 'anchor' );
				slideIndex      = v.activeSlide.index();
			}

			// If continuousVertical && we need to wrap around
			if ( options.autoScrolling && options.continuousVertical && typeof (
					v.isMovementUp
				) !== "undefined" && (
					 (
						 ! v.isMovementUp && v.yMovement == 'up'
					 ) || // Intending to scroll down but about to go up or
					 (
						 v.isMovementUp && v.yMovement == 'down'
					 )
				 ) ) { // intending to scroll up but about to go down

				v = createInfiniteSections( v );
			}

			//callback (onLeave) if the site is not just resizing and readjusting the slides
			if ( $.isFunction( options.onLeave ) && ! v.localIsResizing ) {
				if ( options.onLeave.call( v.activeSection, v.leavingSection, (
						v.sectionIndex + 1
					), v.yMovement ) === false ) {
					return;
				}
			}

			stopMedia( v.activeSection );

			options.scrollOverflowHandler.beforeLeave();
			element.addClass( ACTIVE ).siblings().removeClass( ACTIVE );
			lazyLoad( element );
			options.scrollOverflowHandler.onLeave();


			//preventing from activating the MouseWheelHandler event
			//more than once if the page is scrolling
			canScroll = false;

			setState( slideIndex, slideAnchorLink, v.anchorLink, v.sectionIndex );

			performMovement( v );

			//flag to avoid callingn `scrollPage()` twice in case of using anchor links
			lastScrolledDestiny = v.anchorLink;

			//avoid firing it twice (as it does also on scroll)
			activateMenuAndNav( v.anchorLink, v.sectionIndex );
		}

		var tmAnimations = {
			options        : {
				activeSection: $(),
				element      : null,
				yMovement    : 'up'
			},
			transformOrigin: {
				transformOut: '50% 50%',
				transformIn : '50% 50%'
			},
			beforeTransform: {
				transformOut: 'translate3d(-50%,-50%,0)',
				transformIn : 'translate3d(-50%,-50%,0)'
			},
			transform      : {
				transformOut: 'translate3d(-50%,-50%,0)',
				transformIn : 'translate3d(-50%,-50%,0)'
			},
			init           : function( options, style ) {
				var animation = 'tmAnimation' + style;
				if ( options ) {
					this.options = options;
				}
				if ( typeof this[animation] != 'undefined' ) {
					this[animation]();
					this.animate();
				}
			},
			tmAnimation1   : function() {
				if ( this.options.yMovement == 'up' ) {
					this.transformOrigin.transformOut = '50% 150%';
					this.transformOrigin.transformIn  = '50% 0%';
					this.beforeTransform.transformOut = 'translate3d(0%,0%,0)';
					this.beforeTransform.transformIn  = 'translate3d(0%,-100%,0)';
					this.transform.transformOut       = 'translate3d(0%,50%,0)';
					this.transform.transformIn        = 'translate3d(0%,0%,0)';
				} else {
					this.transformOrigin.transformOut = '50% 0%';
					this.transformOrigin.transformIn  = '50% 150%';
					this.beforeTransform.transformOut = 'translate3d(0,0,0)';
					this.beforeTransform.transformIn  = 'translate3d(0,100%,0)';
					this.transform.transformOut       = 'translate3d(0,-50%,0)';
					this.transform.transformIn        = 'translate3d(0,0,0)';
				}
			},
			tmAnimation2   : function() {
				if ( this.options.yMovement == 'up' ) {
					this.transformOrigin.transformOut = '50% 90%';
					this.transformOrigin.transformIn  = '50% 0%';
					this.beforeTransform.transformOut = 'perspective(100px) translate3d(0%,0%,0) scale(1)';
					this.beforeTransform.transformIn  = 'perspective(0) translate3d(0%,-100%,0) scale(.8)';
					this.transform.transformOut       = 'perspective(0) translate3d(0,0,0) scale(.8)';
					this.transform.transformIn        = 'perspective(100px) translate3d(0,0,0) scale(1)';
				} else {
					this.transformOrigin.transformOut = '50% 10%';
					this.transformOrigin.transformIn  = '50% 150%';
					this.beforeTransform.transformOut = 'perspective(100px) translate3d(0,0,0) scale(1)';
					this.beforeTransform.transformIn  = 'perspective(0) translate3d(0,100%,0) scale(.8)';
					this.transform.transformOut       = 'perspective(0) translate3d(0,0,0) scale(.8)';
					this.transform.transformIn        = 'perspective(100px) translate3d(0,0,0) scale(1)';
				}
			},
			tmAnimation3   : function() {
				if ( this.options.yMovement == 'up' ) {
					this.transformOrigin.transformOut = '50% 0%';
					this.transformOrigin.transformIn  = '50% 50%';
					this.beforeTransform.transformOut = 'perspective(1200px) translate3d(0%,0%,0) rotateX(0)';
					this.beforeTransform.transformIn  = 'perspective(1200px) translate3d(0%,-100%,0)';
					this.transform.transformOut       = 'perspective(1200px) translate3d(0%,100%,0) rotateX(-55deg)';
					this.transform.transformIn        = 'perspective(1200px) translate3d(0,0,0)';
				} else {
					this.transformOrigin.transformOut = '50% 100%';
					this.transformOrigin.transformIn  = '50% 50%';
					this.beforeTransform.transformOut = 'perspective(1200px) translate3d(0%,0%,0) rotateX(0)';
					this.beforeTransform.transformIn  = 'perspective(1200px) translate3d(0%,100%,0)';
					this.transform.transformOut       = 'perspective(1200px) translate3d(0,-100%,0) rotateX(55deg)';
					this.transform.transformIn        = 'perspective(1200px) translate3d(0,0,0)';
				}
			},
			animate        : function() {
				var self = this, speed = originals.scrollingSpeed - 100;

				this.options.activeSection.css( getTransformsOrigin( this.transformOrigin.transformOut ) );
				this.options.activeSection.css( getTransforms( this.beforeTransform.transformOut ) );

				this.options.element.css( getTransformsOrigin( this.transformOrigin.transformIn ) );
				this.options.element.css( getTransforms( this.beforeTransform.transformIn ) );

				setTimeout( function() {
					self.options.activeSection.css( {
						'will-change'       : 'transform',
						'-webkit-transition': 'all ' + speed + 'ms cubic-bezier(.31,.24,.29,.97)',
						'-moz-transition'   : 'all ' + speed + 'ms cubic-bezier(.31,.24,.29,.97)',
						'-ms-transition'    : 'all ' + speed + 'ms cubic-bezier(.31,.24,.29,.97)',
						'-o-transition'     : 'all ' + speed + 'ms cubic-bezier(.31,.24,.29,.97)',
						'transition'        : 'all ' + speed + 'ms cubic-bezier(.31,.24,.29,.97)',
						'-webkit-transform' : self.transform.transformOut,
						'-moz-transform'    : self.transform.transformOut,
						'-ms-transform'     : self.transform.transformOut,
						'-o-transform'      : self.transform.transformOut,
						'transform'         : self.transform.transformOut,
					} );
					self.options.element.css( {
						'will-change'       : 'transform',
						'-webkit-transition': 'all ' + speed + 'ms cubic-bezier(.31,.24,.29,.97)',
						'-moz-transition'   : 'all ' + speed + 'ms cubic-bezier(.31,.24,.29,.97)',
						'-ms-transition'    : 'all ' + speed + 'ms cubic-bezier(.31,.24,.29,.97)',
						'-o-transition'     : 'all ' + speed + 'ms cubic-bezier(.31,.24,.29,.97)',
						'transition'        : 'all ' + speed + 'ms cubic-bezier(.31,.24,.29,.97)',
						'-webkit-transform' : self.transform.transformIn,
						'-moz-transform'    : self.transform.transformIn,
						'-ms-transform'     : self.transform.transformIn,
						'-o-transform'      : self.transform.transformIn,
						'transform'         : self.transform.transformIn,
					} );
				}, 100 );
			}
		};

		/**
		 * Performs the vertical movement (by CSS3 or by jQuery)
		 */
		function performMovement( v ) {
			var animatioEffect = container.hasClass( 'tm-enable-onepage-animation' );
			// using CSS3 translate functionality
			if ( options.css3 && options.autoScrolling && ! options.scrollBar ) {
				if ( ! animatioEffect ) {
					// The first section can have a negative value in iOS 10. Not quite sure why: -0.0142822265625
					// that's why we round it to 0.
					var translate3d = 'translate3d(0px, -' + Math.round( v.dtop ) + 'px, 0px)';
					transformContainer( translate3d, true );
				}

				$( 'body' ).on( 'vc-front-reinited', function() {
					transformContainer( 'translate3d(0px, 0px, 0px)', true );
				} );

				//even when the scrollingSpeed is 0 there's a little delay, which might cause the
				//scrollingSpeed to change in case of using silentMoveTo();
				if ( options.scrollingSpeed ) {
					clearTimeout( afterSectionLoadsId );
					afterSectionLoadsId = setTimeout( function() {
						afterSectionLoads( v );
					}, options.scrollingSpeed );
				} else {
					afterSectionLoads( v );
				}
			}

			// using jQuery animate
			else {
				var scrollSettings = getScrollSettings( v );
				if ( animatioEffect ) {
					scrollSettings.options = { 'top': 0 };
				}

				$( scrollSettings.element )
					.animate( scrollSettings.options, options.scrollingSpeed, options.easing )
					.promise()
					.done( function() { //only one single callback in case of animating  `html, body`
						if ( options.scrollBar ) {

							/* Hack!
							 The timeout prevents setting the most dominant section in the viewport as "active" when the user
							 scrolled to a smaller section by using the mousewheel (auto scrolling) rather than draging the scroll bar.

							 When using scrollBar:true It seems like the scroll events still getting propagated even after the scrolling animation has finished.
							 */
							setTimeout( function() {
								afterSectionLoads( v );
							}, 30 );
						} else {
							afterSectionLoads( v );
						}
					} );
			}
			if ( animatioEffect ) {
				var anim = container.attr( 'class' );
				if ( anim.indexOf( "tm-3d-style-" ) != - 1 ) {
					anim = anim.substring( anim.toLowerCase().indexOf( "tm-3d-style-" ) );
					anim = anim.substring( 12, 13 );
				}
				if ( typeof tmAnimations != 'undefined' && anim.length == 1 ) {
					tmAnimations.init( v, anim );
				}
			}
		}

		/**
		 * Gets the scrolling settings depending on the plugin autoScrolling option
		 */
		function getScrollSettings( v ) {
			var scroll = {};

			if ( options.autoScrolling && ! options.scrollBar ) {
				scroll.options = { 'top': - v.dtop };
				scroll.element = WRAPPER_SEL;
			} else {
				scroll.options = { 'scrollTop': v.dtop };
				scroll.element = 'html, body';
			}

			return scroll;
		}

		/**
		 * Adds sections before or after the current one to create the infinite effect.
		 */
		function createInfiniteSections( v ) {
			// Scrolling down
			if ( ! v.isMovementUp ) {
				// Move all previous sections to after the active section
				$( SECTION_ACTIVE_SEL ).after( v.activeSection.prevAll( SECTION_SEL ).get().reverse() );
			} else { // Scrolling up
				// Move all next sections to before the active section
				$( SECTION_ACTIVE_SEL ).before( v.activeSection.nextAll( SECTION_SEL ) );
			}

			// Maintain the displayed position (now that we changed the element order)
			silentScroll( $( SECTION_ACTIVE_SEL ).position().top );

			// Maintain the active slides visible in the viewport
			keepSlidesPosition();

			// save for later the elements that still need to be reordered
			v.wrapAroundElements = v.activeSection;

			// Recalculate animation variables
			v.dtop      = v.element.position().top;
			v.yMovement = getYmovement( v.element );

			return v;
		}

		/**
		 * Fix section order after continuousVertical changes have been animated
		 */
		function continuousVerticalFixSectionOrder( v ) {
			// If continuousVertical is in effect (and autoScrolling would also be in effect then),
			// finish moving the elements around so the direct navigation will function more simply
			if ( ! v.wrapAroundElements || ! v.wrapAroundElements.length ) {
				return;
			}

			if ( v.isMovementUp ) {
				$( SECTION_FIRST_SEL ).before( v.wrapAroundElements );
			} else {
				$( SECTION_LAST_SEL ).after( v.wrapAroundElements );
			}

			silentScroll( $( SECTION_ACTIVE_SEL ).position().top );

			// Maintain the active slides visible in the viewport
			keepSlidesPosition();
		}


		/**
		 * Actions to do once the section is loaded.
		 */
		function afterSectionLoads( v ) {
			continuousVerticalFixSectionOrder( v );

			//callback (afterLoad) if the site is not just resizing and readjusting the slides
			$.isFunction( options.afterLoad ) && ! v.localIsResizing && options.afterLoad.call( v.element, v.anchorLink, (
				v.sectionIndex + 1
			) );
			options.scrollOverflowHandler.afterLoad();

			playMedia( v.element );
			v.element.addClass( COMPLETELY ).siblings().removeClass( COMPLETELY );

			canScroll = true;

			$.isFunction( v.callback ) && v.callback.call( this );
		}

		/**
		 * Lazy loads image, video and audio elements.
		 */
		function lazyLoad( destiny ) {
			if ( ! options.lazyLoading ) {
				return;
			}

			var panel = getSlideOrSection( destiny );
			var element;

			panel.find( 'img[data-src], source[data-src], audio[data-src], iframe[data-src]' ).each( function() {
				element = $( this );
				element.attr( 'src', element.data( 'src' ) );
				element.removeAttr( 'data-src' );

				if ( element.is( 'source' ) ) {
					element.closest( 'video' ).get( 0 ).load();
				}
			} );
		}

		/**
		 * Plays video and audio elements.
		 */
		function playMedia( destiny ) {
			var panel = getSlideOrSection( destiny );

			//playing HTML5 media elements
			panel.find( 'video, audio' ).each( function() {
				var element = $( this ).get( 0 );

				if ( element.hasAttribute( 'data-autoplay' ) && typeof element.play === 'function' ) {
					//                    element.play();
				}
			} );

			//youtube videos
			panel.find( 'iframe[src*="youtube.com/embed/"]' ).each( function() {
				var element = $( this ).get( 0 );

				if ( element.hasAttribute( 'data-autoplay' ) ) {
					//                    playYoutube(element);
				}

				//in case the URL was not loaded yet. On page load we need time for the new URL (with the API string) to load.
				element.onload = function() {
					if ( element.hasAttribute( 'data-autoplay' ) ) {
						//                        playYoutube(element);
					}
				};
			} );
		}

		/**
		 * Plays a youtube video
		 */
		function playYoutube( element ) {
			element.contentWindow.postMessage( '{"event":"command","func":"playVideo","args":""}', '*' );
		}

		/**
		 * Stops video and audio elements.
		 */
		function stopMedia( destiny ) {
			var panel = getSlideOrSection( destiny );

			//stopping HTML5 media elements
			panel.find( 'video, audio' ).each( function() {
				var element = $( this ).get( 0 );

				if ( ! element.hasAttribute( 'data-keepplaying' ) && typeof element.pause === 'function' ) {
					//                    element.pause();
				}
			} );

			//youtube videos
			panel.find( 'iframe[src*="youtube.com/embed/"]' ).each( function() {
				var element = $( this ).get( 0 );

				if ( /youtube\.com\/embed\//.test( $( this )
						.attr( 'src' ) ) && ! element.hasAttribute( 'data-keepplaying' ) ) {
					//                    $(this).get(0).contentWindow.postMessage('{"event":"command","func":"pauseVideo","args":""}','*');
				}
			} );
		}

		/**
		 * Gets the active slide (or section) for the given section
		 */
		function getSlideOrSection( destiny ) {
			var slide = destiny.find( SLIDE_ACTIVE_SEL );
			if ( slide.length ) {
				destiny = $( slide );
			}

			return destiny;
		}

		/**
		 * Scrolls to the anchor in the URL when loading the site
		 */
		function scrollToAnchor() {
			//getting the anchor link in the URL and deleting the `#`
			var value         = window.location.hash.replace( '#', '' ).split( '/' );
			var sectionAnchor = decodeURIComponent( value[0] );
			var slideAnchor   = decodeURIComponent( value[1] );

			if ( sectionAnchor ) {  //if theres any #
				if ( options.animateAnchor ) {
					scrollPageAndSlide( sectionAnchor, slideAnchor );
				} else {
					silentMoveTo( sectionAnchor, slideAnchor );
				}
			}
		}

		/**
		 * Detecting any change on the URL to scroll to the given anchor link
		 * (a way to detect back history button as we play with the hashes on the URL)
		 */
		function hashChangeHandler() {
			if ( ! isScrolling && ! options.lockAnchors ) {
				var value         = window.location.hash.replace( '#', '' ).split( '/' );
				var sectionAnchor = decodeURIComponent( value[0] );
				var slideAnchor   = decodeURIComponent( value[1] );

				//when moving to a slide in the first section for the first time (first time to add an anchor to the URL)
				var isFirstSlideMove  = (
					typeof lastScrolledDestiny === 'undefined'
				);
				var isFirstScrollMove = (
					typeof lastScrolledDestiny === 'undefined' && typeof slideAnchor === 'undefined' && ! slideMoving
				);


				if ( sectionAnchor.length ) {
					/*in order to call scrollpage() only once for each destination at a time
					 It is called twice for each scroll otherwise, as in case of using anchorlinks `hashChange`
					 event is fired on every scroll too.*/
					if ( (
						 sectionAnchor && sectionAnchor !== lastScrolledDestiny
						 ) && ! isFirstSlideMove || isFirstScrollMove || (
							 ! slideMoving && lastScrolledSlide != slideAnchor
						 ) ) {
						scrollPageAndSlide( sectionAnchor, slideAnchor );
					}
				}
			}
		}

		//Sliding with arrow keys, both, vertical and horizontal
		function keydownHandler( e ) {

			clearTimeout( keydownId );

			var activeElement = $( ':focus' );

			if ( ! activeElement.is( 'textarea' ) && ! activeElement.is( 'input' ) && ! activeElement.is( 'select' ) && activeElement.attr( 'contentEditable' ) !== "true" && activeElement.attr( 'contentEditable' ) !== '' && options.keyboardScrolling && options.autoScrolling ) {
				var keyCode = e.which;

				//preventing the scroll with arrow keys & spacebar & Page Up & Down keys
				var keyControls = [
					40,
					38,
					32,
					33,
					34
				];
				if ( $.inArray( keyCode, keyControls ) > - 1 ) {
					e.preventDefault();
				}

				controlPressed = e.ctrlKey;

				keydownId = setTimeout( function() {
					onkeydown( e );
				}, 150 );
			}
		}

		function tooltipTextHandler() {
			$( this ).prev().trigger( 'click' );
		}

		//to prevent scrolling while zooming
		function keyUpHandler( e ) {
			if ( isWindowFocused ) { //the keyup gets fired on new tab ctrl + t in Firefox
				controlPressed = e.ctrlKey;
			}
		}

		//binding the mousemove when the mouse's middle button is released
		function mouseDownHandler( e ) {
			//middle button
			if ( e.which == 2 ) {
				oldPageY = e.pageY;
				container.on( 'mousemove', mouseMoveHandler );
			}
		}

		//unbinding the mousemove when the mouse's middle button is released
		function mouseUpHandler( e ) {
			//middle button
			if ( e.which == 2 ) {
				container.off( 'mousemove' );
			}
		}

		//Scrolling horizontally when clicking on the slider controls.
		function slideArrowHandler() {
			var section = $( this ).closest( SECTION_SEL );

			if ( $( this ).hasClass( SLIDES_PREV ) ) {
				if ( isScrollAllowed.m.left ) {
					moveSlideLeft( section );
				}
			} else {
				if ( isScrollAllowed.m.right ) {
					moveSlideRight( section );
				}
			}
		}

		//when opening a new tab (ctrl + t), `control` won't be pressed when comming back.
		function blurHandler() {
			isWindowFocused = false;
			controlPressed  = false;
		}

		//Scrolls to the section when clicking the navigation bullet
		function sectionBulletHandler( e ) {
			e.preventDefault();
			var index = $( this ).parent().index();
			scrollPage( $( SECTION_SEL ).eq( index ) );
			if ( $( 'body' ).hasClass( 'tm-one-page-footer-expanded' ) ) {
				$( 'body' ).removeClass( 'tm-one-page-footer-expanded' );
			}
		}

		//Scrolls the slider to the given slide destination for the given section
		function slideBulletHandler( e ) {
			e.preventDefault();
			var slides  = $( this ).closest( SECTION_SEL ).find( SLIDES_WRAPPER_SEL );
			var destiny = slides.find( SLIDE_SEL ).eq( $( this ).closest( 'li' ).index() );

			landscapeScroll( slides, destiny );
		}

		/**
		 * Keydown event
		 */
		function onkeydown( e ) {
			var shiftPressed = e.shiftKey;

			switch ( e.which ) {
				//up
				case 38:
				case 33:
					if ( isScrollAllowed.k.up ) {
						moveSectionUp();
					}
					break;

				//down
				case 32: //spacebar
					if ( shiftPressed && isScrollAllowed.k.up ) {
						moveSectionUp();
						break;
					}
				/* falls through */
				case 40:
				case 34:
					if ( isScrollAllowed.k.down ) {
						moveSectionDown();
					}
					break;

				//Home
				case 36:
					if ( isScrollAllowed.k.up ) {
						moveTo( 1 );
					}
					break;

				//End
				case 35:
					if ( isScrollAllowed.k.down ) {
						moveTo( $( SECTION_SEL ).length );
					}
					break;

				//left
				case 37:
					if ( isScrollAllowed.k.left ) {
						moveSlideLeft();
					}
					break;

				//right
				case 39:
					if ( isScrollAllowed.k.right ) {
						moveSlideRight();
					}
					break;

				default:
					return; // exit this handler for other keys
			}
		}

		/**
		 * Detecting the direction of the mouse movement.
		 * Used only for the middle button of the mouse.
		 */
		var oldPageY = 0;

		function mouseMoveHandler( e ) {
			if ( canScroll ) {
				// moving up
				if ( e.pageY < oldPageY && isScrollAllowed.m.up ) {
					moveSectionUp();
				}

				// moving down
				else if ( e.pageY > oldPageY && isScrollAllowed.m.down ) {
					moveSectionDown();
				}
			}
			oldPageY = e.pageY;
		}

		/**
		 * Scrolls horizontal sliders.
		 */
		function landscapeScroll( slides, destiny, direction ) {
			var section = slides.closest( SECTION_SEL );
			var v       = {
				slides        : slides,
				destiny       : destiny,
				direction     : direction,
				destinyPos    : destiny.position(),
				slideIndex    : destiny.index(),
				section       : section,
				sectionIndex  : section.index( SECTION_SEL ),
				anchorLink    : section.data( 'anchor' ),
				slidesNav     : section.find( SLIDES_NAV_SEL ),
				slideAnchor   : getAnchor( destiny ),
				prevSlide     : section.find( SLIDE_ACTIVE_SEL ),
				prevSlideIndex: section.find( SLIDE_ACTIVE_SEL ).index(),

				//caching the value of isResizing at the momment the function is called
				//because it will be checked later inside a setTimeout and the value might change
				localIsResizing: isResizing
			};
			v.xMovement = getXmovement( v.prevSlideIndex, v.slideIndex );

			//important!! Only do it when not resizing
			if ( ! v.localIsResizing ) {
				//preventing from scrolling to the next/prev section when using scrollHorizontally
				canScroll = false;
			}

			if ( options.onSlideLeave ) {

				//if the site is not just resizing and readjusting the slides
				if ( ! v.localIsResizing && v.xMovement !== 'none' ) {
					if ( $.isFunction( options.onSlideLeave ) ) {
						if ( options.onSlideLeave.call( v.prevSlide, v.anchorLink, (
								v.sectionIndex + 1
							), v.prevSlideIndex, v.xMovement, v.slideIndex ) === false ) {
							slideMoving = false;
							return;
						}
					}
				}
			}

			destiny.addClass( ACTIVE ).siblings().removeClass( ACTIVE );

			if ( ! v.localIsResizing ) {
				stopMedia( v.prevSlide );
				lazyLoad( destiny );
			}

			if ( ! options.loopHorizontal && options.controlArrows ) {
				//hidding it for the fist slide, showing for the rest
				section.find( SLIDES_ARROW_PREV_SEL ).toggle( v.slideIndex !== 0 );

				//hidding it for the last slide, showing for the rest
				section.find( SLIDES_ARROW_NEXT_SEL ).toggle( ! destiny.is( ':last-child' ) );
			}

			//only changing the URL if the slides are in the current section (not for resize re-adjusting)
			if ( section.hasClass( ACTIVE ) ) {
				setState( v.slideIndex, v.slideAnchor, v.anchorLink, v.sectionIndex );
			}

			performHorizontalMove( slides, v, true );
		}


		function afterSlideLoads( v ) {
			activeSlidesNavigation( v.slidesNav, v.slideIndex );

			//if the site is not just resizing and readjusting the slides
			if ( ! v.localIsResizing ) {
				$.isFunction( options.afterSlideLoad ) && options.afterSlideLoad.call( v.destiny, v.anchorLink, (
					v.sectionIndex + 1
				), v.slideAnchor, v.slideIndex );

				//needs to be inside the condition to prevent problems with continuousVertical and scrollHorizontally
				//and to prevent double scroll right after a windows resize
				canScroll = true;
			}

			playMedia( v.destiny );

			//letting them slide again
			slideMoving = false;
		}

		/**
		 * Performs the horizontal movement. (CSS3 or jQuery)
		 *
		 * @param fireCallback {Bool} - determines whether or not to fire the callback
		 */
		function performHorizontalMove( slides, v, fireCallback ) {
			var destinyPos = v.destinyPos;

			if ( options.css3 ) {
				var translate3d = 'translate3d(-' + Math.round( destinyPos.left ) + 'px, 0px, 0px)';

				addAnimation( slides.find( SLIDES_CONTAINER_SEL ) ).css( getTransforms( translate3d ) );

				afterSlideLoadsId = setTimeout( function() {
					fireCallback && afterSlideLoads( v );
				}, options.scrollingSpeed, options.easing );
			} else {
				slides.animate( {
					scrollLeft: Math.round( destinyPos.left )
				}, options.scrollingSpeed, options.easing, function() {

					fireCallback && afterSlideLoads( v );
				} );
			}
		}

		/**
		 * Sets the state for the horizontal bullet navigations.
		 */
		function activeSlidesNavigation( slidesNav, slideIndex ) {
			slidesNav.find( ACTIVE_SEL ).removeClass( ACTIVE );
			slidesNav.find( 'li' ).eq( slideIndex ).find( 'a' ).addClass( ACTIVE );
		}

		var previousHeight = windowsHeight;

		//when resizing the site, we adjust the heights of the sections, slimScroll...
		function resizeHandler() {
			//checking if it needs to get responsive
			responsive();

			// rebuild immediately on touch devices
			if ( isTouchDevice ) {
				var activeElement = $( document.activeElement );

				//if the keyboard is NOT visible
				if ( ! activeElement.is( 'textarea' ) && ! activeElement.is( 'input' ) && ! activeElement.is( 'select' ) ) {
					var currentHeight = $window.height();

					//making sure the change in the viewport size is enough to force a rebuild. (20 % of the window to avoid problems when hidding scroll bars)
					if ( Math.abs( currentHeight - previousHeight ) > (
							20 * Math.max( previousHeight, currentHeight ) / 100
						) ) {
						reBuild( true );
						previousHeight = currentHeight;
					}
				}
			} else {
				//in order to call the functions only when the resize is finished
				//http://stackoverflow.com/questions/4298612/jquery-how-to-call-resize-event-only-once-its-finished-resizing
				clearTimeout( resizeId );

				resizeId = setTimeout( function() {
					reBuild( true );
				}, 350 );
			}
		}

		/**
		 * Checks if the site needs to get responsive and disables autoScrolling if so.
		 * A class `fp-responsive` is added to the plugin's container in case the user wants to use it for his own responsive CSS.
		 */
		function responsive() {
			var widthLimit  = options.responsive || options.responsiveWidth; //backwards compatiblity
			var heightLimit = options.responsiveHeight;

			//only calculating what we need. Remember its called on the resize event.
			var isBreakingPointWidth  = widthLimit && $window.outerWidth() < widthLimit;
			var isBreakingPointHeight = heightLimit && $window.height() < heightLimit;

			if ( widthLimit && heightLimit ) {
				setResponsive( isBreakingPointWidth || isBreakingPointHeight );
			} else if ( widthLimit ) {
				setResponsive( isBreakingPointWidth );
			} else if ( heightLimit ) {
				setResponsive( isBreakingPointHeight );
			}
		}

		/**
		 * Adds transition animations for the given element
		 */
		function addAnimation( element ) {
			var transition = 'all ' + options.scrollingSpeed + 'ms ' + options.easingcss3;

			element.removeClass( NO_TRANSITION );
			return element.css( {
				'-webkit-transition': transition,
				'transition'        : transition
			} );
		}

		/**
		 * Remove transition animations for the given element
		 */
		function removeAnimation( element ) {
			return element.addClass( NO_TRANSITION );
		}

		/**
		 * Activating the vertical navigation bullets according to the given slide name.
		 */
		function activateNavDots( name, sectionIndex ) {
			if ( options.navigation ) {
				$( SECTION_NAV_SEL ).find( ACTIVE_SEL ).removeClass( ACTIVE );
				if ( name ) {
					$( SECTION_NAV_SEL ).find( 'a[href="#' + name + '"]' ).addClass( ACTIVE );
				} else {
					$( SECTION_NAV_SEL ).find( 'li' ).eq( sectionIndex ).find( 'a' ).addClass( ACTIVE );
				}
			}
		}

		/**
		 * Activating the website main menu elements according to the given slide name.
		 */
		function activateMenuElement( name ) {
			if ( options.menu ) {
				$( options.menu ).find( ACTIVE_SEL ).removeClass( ACTIVE );
				$( options.menu ).find( '[data-menuanchor="' + name + '"]' ).addClass( ACTIVE );
			}
		}

		/**
		 * Sets to active the current menu and vertical nav items.
		 */
		function activateMenuAndNav( anchor, index ) {
			activateMenuElement( anchor );
			activateNavDots( anchor, index );
		}

		/**
		 * Retuns `up` or `down` depending on the scrolling movement to reach its destination
		 * from the current section.
		 */
		function getYmovement( destiny ) {
			var fromIndex = $( SECTION_ACTIVE_SEL ).index( SECTION_SEL );
			var toIndex   = destiny.index( SECTION_SEL );
			if ( fromIndex == toIndex ) {
				return 'none';
			}
			if ( fromIndex > toIndex ) {
				return 'up';
			}
			return 'down';
		}

		/**
		 * Retuns `right` or `left` depending on the scrolling movement to reach its destination
		 * from the current slide.
		 */
		function getXmovement( fromIndex, toIndex ) {
			if ( fromIndex == toIndex ) {
				return 'none';
			}
			if ( fromIndex > toIndex ) {
				return 'left';
			}
			return 'right';
		}

		/**
		 * Checks if the element needs scrollbar and if the user wants to apply it.
		 * If so it creates it.
		 *
		 * @param {Object} element   jQuery object of the section or slide
		 */
		function createScrollBar( element ) {
			//User doesn't want scrollbar here? Sayonara baby!
			if ( element.hasClass( 'fp-noscroll' ) ) {
				return;
			}

			//needed to make `scrollHeight` work under Opera 12
			element.css( 'overflow', 'hidden' );

			var scrollOverflowHandler = options.scrollOverflowHandler;
			var wrap                  = scrollOverflowHandler.wrapContent();
			//in case element is a slide
			var section               = element.closest( SECTION_SEL );
			var scrollable            = scrollOverflowHandler.scrollable( element );
			var contentHeight;
			var headerHeight          = 0;
			var $header               = $( '#page-header-inner' );
			//if there was scroll, the contentHeight will be the one in the scrollable section
			if ( scrollable.length ) {
				//				element.find('.fp-scroller').css('height','auto');
				contentHeight = scrollOverflowHandler.scrollHeight( element );
			} else {
				contentHeight = element.get( 0 ).scrollHeight;
				if ( options.verticalCentered ) {
					contentHeight = element.find( TABLE_CELL_SEL ).get( 0 ).scrollHeight;
				}
			}

			var scrollHeight = windowsHeight - parseInt( section.css( 'padding-bottom' ) ) - parseInt( section.css( 'padding-top' ) );

			//needs scroll?
			if ( contentHeight > scrollHeight ) {
				//did we already have an scrollbar ? Updating it
				if ( scrollable.length ) {
					scrollOverflowHandler.update( element, scrollHeight );
				}
				//creating the scrolling
				else {
					if ( options.verticalCentered ) {
						element.find( TABLE_CELL_SEL ).wrapInner( wrap );
					} else {
						element.wrapInner( wrap );
					}
					element.find( '.fp-scroller' ).css( 'height', contentHeight + headerHeight );
					scrollOverflowHandler.create( element, scrollHeight );
				}
			}
			//removing the scrolling when it is not necessary anymore
			else {
				scrollOverflowHandler.remove( element );
			}

			//undo
			element.css( 'overflow', '' );
		}

		function addTableClass( element ) {
			//In case we are styling for the 2nd time as in with reponsiveSlides
			if ( ! element.hasClass( TABLE ) ) {
				if ( $( 'body' ).hasClass( 'compose-mode' ) ) {
					element.addClass( TABLE )
						   .find( '> .vc-row-wrapper' )
						   .wrapInner( '<div class="' + TABLE_CELL + '" style="height:' + getTableHeight( element ) + 'px;" />' );
				} else {
					element.addClass( TABLE )
						   .wrapInner( '<div class="' + TABLE_CELL + '" style="height:' + getTableHeight( element ) + 'px;" />' );
				}
			}
		}

		function getTableHeight( element ) {
			var sectionHeight = windowsHeight;

			if ( options.paddingTop || options.paddingBottom ) {
				var section = element;
				if ( ! section.hasClass( SECTION ) ) {
					section = element.closest( SECTION_SEL );
				}

				var paddings  = parseInt( section.css( 'padding-top' ) ) + parseInt( section.css( 'padding-bottom' ) );
				sectionHeight = (
					windowsHeight - paddings
				);
			}

			return sectionHeight;
		}

		/**
		 * Adds a css3 transform property to the container class with or without animation depending on the animated param.
		 */
		function transformContainer( translate3d, animated ) {
			if ( animated ) {
				addAnimation( container );
			} else {
				removeAnimation( container );
			}

			container.css( getTransforms( translate3d ) );

			//syncronously removing the class after the animation has been applied.
			setTimeout( function() {
				container.removeClass( NO_TRANSITION );
			}, 10 );
		}

		/**
		 * Adds a css3 transform property to the element given with or without animation depending on the animated param.
		 */
		function transformElement( translate3d, animated, element ) {
			if ( animated ) {
				addAnimation( element );
			} else {
				removeAnimation( element );
			}

			element.css( getTransforms( translate3d ) );

			//syncronously removing the class after the animation has been applied.
			setTimeout( function() {
				element.removeClass( NO_TRANSITION );
			}, 0 );
		}

		/**
		 * Gets a section by its anchor / index
		 */
		function getSectionByAnchor( sectionAnchor ) {
			//section
			var section = container.find( SECTION_SEL + '[data-anchor="' + sectionAnchor + '"]' );
			if ( ! section.length ) {
				section = $( SECTION_SEL )
					.eq( (
						sectionAnchor - 1
					) );
			}

			return section;
		}

		/**
		 * Gets a slide inside a given section by its anchor / index
		 */
		function getSlideByAnchor( slideAnchor, section ) {
			var slides = section.find( SLIDES_WRAPPER_SEL );
			var slide  = slides.find( SLIDE_SEL + '[data-anchor="' + slideAnchor + '"]' );

			if ( ! slide.length ) {
				slide = slides.find( SLIDE_SEL ).eq( slideAnchor );
			}

			return slide;
		}

		/**
		 * Scrolls to the given section and slide anchors
		 */
		function scrollPageAndSlide( destiny, slide ) {
			var section = getSectionByAnchor( destiny );

			//do nothing if there's no section with the given anchor name
			if ( ! section.length || ! canScroll ) {
				return;
			}

			//default slide
			if ( typeof slide === 'undefined' ) {
				slide = 0;
			}

			//we need to scroll to the section and then to the slide
			if ( destiny !== lastScrolledDestiny && ! section.hasClass( ACTIVE ) ) {
				scrollPage( section, function() {
					scrollSlider( section, slide );
				} );
			}
			//if we were already in the section
			else {
				scrollSlider( section, slide );
			}
		}

		/**
		 * Scrolls the slider to the given slide destination for the given section
		 */
		function scrollSlider( section, slideAnchor ) {
			if ( typeof slideAnchor !== 'undefined' ) {
				var slides  = section.find( SLIDES_WRAPPER_SEL );
				var destiny = getSlideByAnchor( slideAnchor, section );

				if ( destiny.length ) {
					landscapeScroll( slides, destiny );
				}
			}
		}

		/**
		 * Creates a landscape navigation bar with dots for horizontal sliders.
		 */
		function addSlidesNavigation( section, numSlides ) {
			section.append( '<div class="' + SLIDES_NAV + '"><ul></ul></div>' );
			var nav = section.find( SLIDES_NAV_SEL );

			//top or bottom
			nav.addClass( options.slidesNavPosition );

			for ( var i = 0; i < numSlides; i ++ ) {
				nav.find( 'ul' ).append( '<li><a href="#"><span></span></a></li>' );
			}

			//centering it
			nav.css( 'margin-left', '-' + (
					nav.width() / 2
				) + 'px' );

			nav.find( 'li' ).first().find( 'a' ).addClass( ACTIVE );
		}


		/**
		 * Sets the state of the website depending on the active section/slide.
		 * It changes the URL hash when needed and updates the body class.
		 */
		function setState( slideIndex, slideAnchor, anchorLink, sectionIndex ) {
			var sectionHash = '';

			if ( options.anchors.length && ! options.lockAnchors ) {

				//isn't it the first slide?
				if ( slideIndex ) {
					if ( typeof anchorLink !== 'undefined' ) {
						sectionHash = anchorLink;
					}

					//slide without anchor link? We take the index instead.
					if ( typeof slideAnchor === 'undefined' ) {
						slideAnchor = slideIndex;
					}

					lastScrolledSlide = slideAnchor;
					setUrlHash( sectionHash + '/' + slideAnchor );

					//first slide won't have slide anchor, just the section one
				} else if ( typeof slideIndex !== 'undefined' ) {
					lastScrolledSlide = slideAnchor;
					setUrlHash( anchorLink );
				}

				//section without slides
				else {
					setUrlHash( anchorLink );
				}
			}

			setBodyClass();
		}

		/**
		 * Sets the URL hash.
		 */
		function setUrlHash( url ) {
			if ( options.recordHistory ) {
				location.hash = url;
			} else {
				//Mobile Chrome doesn't work the normal way, so... lets use HTML5 for phones :)
				if ( isTouchDevice || isTouch ) {
					window.history.replaceState( undefined, undefined, '#' + url );
				} else {
					var baseUrl = window.location.href.split( '#' )[0];
					window.location.replace( baseUrl + '#' + url );
				}
			}
		}

		/**
		 * Gets the anchor for the given slide / section. Its index will be used if there's none.
		 */
		function getAnchor( element ) {
			var anchor = element.data( 'anchor' );
			var index  = element.index();

			//Slide without anchor link? We take the index instead.
			if ( typeof anchor === 'undefined' ) {
				anchor = index;
			}

			return anchor;
		}

		/**
		 * Sets a class for the body of the page depending on the active section / slide
		 */
		function setBodyClass() {
			var section = $( SECTION_ACTIVE_SEL );
			var slide   = section.find( SLIDE_ACTIVE_SEL );

			var sectionAnchor = getAnchor( section );
			var slideAnchor   = getAnchor( slide );

			var text = String( sectionAnchor );

			if ( slide.length ) {
				text = text + '-' + slideAnchor;
			}

			//changing slash for dash to make it a valid CSS style
			text = text.replace( '/', '-' ).replace( '#', '' );

			//removing previous anchor classes
			var classRe        = new RegExp( '\\b\\s?' + VIEWING_PREFIX + '-[^\\s]+\\b', "g" );
			$body[0].className = $body[0].className.replace( classRe, '' );

			//adding the current anchor
			$body.addClass( VIEWING_PREFIX + '-' + text );
		}

		/**
		 * Checks for translate3d support
		 * @return boolean
		 * http://stackoverflow.com/questions/5661671/detecting-transform-translate3d-support
		 */
		function support3d() {
			var el = document.createElement( 'p' ), has3d, transforms = {
				'webkitTransform': '-webkit-transform',
				'OTransform'     : '-o-transform',
				'msTransform'    : '-ms-transform',
				'MozTransform'   : '-moz-transform',
				'transform'      : 'transform'
			};

			// Add it to the body to get the computed style.
			document.body.insertBefore( el, null );

			for ( var t in transforms ) {
				if ( el.style[t] !== undefined ) {
					el.style[t] = 'translate3d(1px,1px,1px)';
					has3d       = window.getComputedStyle( el ).getPropertyValue( transforms[t] );
				}
			}

			document.body.removeChild( el );

			return (
				has3d !== undefined && has3d.length > 0 && has3d !== 'none'
			);
		}

		/**
		 * Removes the auto scrolling action fired by the mouse wheel and trackpad.
		 * After this function is called, the mousewheel and trackpad movements won't scroll through sections.
		 */
		function removeMouseWheelHandler() {
			if ( document.addEventListener ) {
				document.removeEventListener( 'mousewheel', MouseWheelHandler, false ); //IE9, Chrome, Safari, Oper
				document.removeEventListener( 'wheel', MouseWheelHandler, false ); //Firefox
				document.removeEventListener( 'MozMousePixelScroll', MouseWheelHandler, false ); //old Firefox
			} else {
				document.detachEvent( 'onmousewheel', MouseWheelHandler ); //IE 6/7/8
			}
		}

		/**
		 * Adds the auto scrolling action for the mouse wheel and trackpad.
		 * After this function is called, the mousewheel and trackpad movements will scroll through sections
		 * https://developer.mozilla.org/en-US/docs/Web/Events/wheel
		 */
		function addMouseWheelHandler() {
			var prefix = '';
			var _addEventListener;

			if ( window.addEventListener ) {
				_addEventListener = "addEventListener";
			} else {
				_addEventListener = "attachEvent";
				prefix            = 'on';
			}

			// detect available wheel event
			var support = 'onwheel' in document.createElement( 'div' ) ? 'wheel' : // Modern browsers support "wheel"
				document.onmousewheel !== undefined ? 'mousewheel' : // Webkit and IE support at least "mousewheel"
					'DOMMouseScroll'; // let's assume that remaining browsers are older Firefox


			if ( support == 'DOMMouseScroll' ) {
				document[_addEventListener]( prefix + 'MozMousePixelScroll', MouseWheelHandler, false );
			}

			//handle MozMousePixelScroll in older Firefox
			else {
				document[_addEventListener]( prefix + support, MouseWheelHandler, false );
			}
		}

		/**
		 * Binding the mousemove when the mouse's middle button is pressed
		 */
		function addMiddleWheelHandler() {
			container
				.on( 'mousedown', mouseDownHandler )
				.on( 'mouseup', mouseUpHandler );
		}

		/**
		 * Unbinding the mousemove when the mouse's middle button is released
		 */
		function removeMiddleWheelHandler() {
			container
				.off( 'mousedown', mouseDownHandler )
				.off( 'mouseup', mouseUpHandler );
		}

		/**
		 * Adds the possibility to auto scroll through sections on touch devices.
		 */
		function addTouchHandler() {
			if ( options.autoScrolling && (
					isTouchDevice || isTouch
				) ) {
				//Microsoft pointers
				var MSPointer = getMSPointer();

				$body.off( 'touchmove ' + MSPointer.move ).on( 'touchmove ' + MSPointer.move, preventBouncing );

				$( WRAPPER_SEL )
					.off( 'touchstart ' + MSPointer.down ).on( 'touchstart ' + MSPointer.down, touchStartHandler )
					.off( 'touchmove ' + MSPointer.move ).on( 'touchmove ' + MSPointer.move, touchMoveHandler );
			}
		}

		/**
		 * Removes the auto scrolling for touch devices.
		 */
		function removeTouchHandler() {
			if ( isTouchDevice || isTouch ) {
				//Microsoft pointers
				var MSPointer = getMSPointer();

				$( WRAPPER_SEL )
					.off( 'touchstart ' + MSPointer.down )
					.off( 'touchmove ' + MSPointer.move );
			}
		}

		/*
		 * Returns and object with Microsoft pointers (for IE<11 and for IE >= 11)
		 * http://msdn.microsoft.com/en-us/library/ie/dn304886(v=vs.85).aspx
		 */
		function getMSPointer() {
			var pointer;

			//IE >= 11 & rest of browsers
			if ( window.PointerEvent ) {
				pointer = {
					down: 'pointerdown',
					move: 'pointermove'
				};
			}

			//IE < 11
			else {
				pointer = {
					down: 'MSPointerDown',
					move: 'MSPointerMove'
				};
			}

			return pointer;
		}

		/**
		 * Gets the pageX and pageY properties depending on the browser.
		 * https://github.com/alvarotrigo/fullPage.js/issues/194#issuecomment-34069854
		 */
		function getEventsPage( e ) {
			var events = [];

			events.y = (
				typeof e.pageY !== 'undefined' && (
					e.pageY || e.pageX
				) ? e.pageY : e.touches[0].pageY
			);
			events.x = (
				typeof e.pageX !== 'undefined' && (
					e.pageY || e.pageX
				) ? e.pageX : e.touches[0].pageX
			);

			//in touch devices with scrollBar:true, e.pageY is detected, but we have to deal with touch events. #1008
			if ( isTouch && isReallyTouch( e ) && options.scrollBar ) {
				events.y = e.touches[0].pageY;
				events.x = e.touches[0].pageX;
			}

			return events;
		}

		/**
		 * Slides silently (with no animation) the active slider to the given slide.
		 * @param noCallback {bool} true or defined -> no callbacks
		 */
		function silentLandscapeScroll( activeSlide, noCallbacks ) {
			setScrollingSpeed( 0, 'internal' );

			if ( typeof noCallbacks !== 'undefined' ) {
				//preventing firing callbacks afterSlideLoad etc.
				isResizing = true;
			}

			landscapeScroll( activeSlide.closest( SLIDES_WRAPPER_SEL ), activeSlide );

			if ( typeof noCallbacks !== 'undefined' ) {
				isResizing = false;
			}

			setScrollingSpeed( originals.scrollingSpeed, 'internal' );
		}

		/**
		 * Scrolls silently (with no animation) the page to the given Y position.
		 */
		function silentScroll( top ) {
			// The first section can have a negative value in iOS 10. Not quite sure why: -0.0142822265625
			// that's why we round it to 0.
			var roundedTop = Math.round( top );

			if ( options.scrollBar ) {
				container.scrollTop( roundedTop );
			} else if ( options.css3 ) {
				var translate3d = 'translate3d(0px, -' + roundedTop + 'px, 0px)';
				transformContainer( translate3d, false );
			} else {
				container.css( 'top', - roundedTop );
			}
		}

		/**
		 * Returns the cross-browser transform string.
		 */
		function getTransforms( translate3d ) {
			return {
				'-webkit-transform': translate3d,
				'-moz-transform'   : translate3d,
				'-ms-transform'    : translate3d,
				'transform'        : translate3d
			};
		}

		/**
		 * Returns the cross-browser transform string.
		 */
		function getTransformsOrigin( transformOrigin ) {
			return {
				'-webkit-transform-origin': transformOrigin,
				'-moz-transform-origin'   : transformOrigin,
				'-ms-transform-origin'    : transformOrigin,
				'transform-origin'        : transformOrigin
			};
		}

		/**
		 * Allowing or disallowing the mouse/swipe scroll in a given direction. (not for keyboard)
		 * @type  m (mouse) or k (keyboard)
		 */
		function setIsScrollAllowed( value, direction, type ) {
			switch ( direction ) {
				case 'up':
					isScrollAllowed[type].up = value;
					break;
				case 'down':
					isScrollAllowed[type].down = value;
					break;
				case 'left':
					isScrollAllowed[type].left = value;
					break;
				case 'right':
					isScrollAllowed[type].right = value;
					break;
				case 'all':
					if ( type == 'm' ) {
						setAllowScrolling( value );
					} else {
						setKeyboardScrolling( value );
					}
			}
		}

		/*
		 * Destroys fullpage.js plugin events and optinally its html markup and styles
		 */
		function destroy( all ) {
			setAutoScrolling( false, 'internal' );
			setAllowScrolling( false );
			setKeyboardScrolling( false );
			container.addClass( DESTROYED );

			clearTimeout( afterSlideLoadsId );
			clearTimeout( afterSectionLoadsId );
			clearTimeout( resizeId );
			clearTimeout( scrollId );
			clearTimeout( scrollId2 );

			$window
				.off( 'scroll', scrollHandler )
				.off( 'hashchange', hashChangeHandler )
				.off( 'resize', resizeHandler );

			$document
				.off( 'click touchstart', SECTION_NAV_SEL + ' a' )
				.off( 'mouseenter', SECTION_NAV_SEL + ' li' )
				.off( 'mouseleave', SECTION_NAV_SEL + ' li' )
				.off( 'click touchstart', SLIDES_NAV_LINK_SEL )
				.off( 'mouseover', options.normalScrollElements )
				.off( 'mouseout', options.normalScrollElements );

			$( SECTION_SEL )
				.off( 'click touchstart', SLIDES_ARROW_SEL );

			clearTimeout( afterSlideLoadsId );
			clearTimeout( afterSectionLoadsId );

			//lets make a mess!
			if ( all ) {
				destroyStructure();
			}
		}

		/*
		 * Removes inline styles added by fullpage.js
		 */
		function destroyStructure() {
			//resetting the `top` or `translate` properties to 0
			silentScroll( 0 );

			//loading all the lazy load content
			container.find( 'img[data-src], source[data-src], audio[data-src], iframe[data-src]' ).each( function() {
				$( this ).attr( 'src', $( this ).data( 'src' ) );
				$( this ).removeAttr( 'data-src' );
			} );

			$( SECTION_NAV_SEL + ', ' + SLIDES_NAV_SEL + ', ' + SLIDES_ARROW_SEL ).remove();

			//removing inline styles
			$( SECTION_SEL ).css( {
				'height'          : '',
				'background-color': '',
				'padding'         : ''
			} );

			$( SLIDE_SEL ).css( {
				'width': ''
			} );

			container.css( {
				'height'          : '',
				'position'        : '',
				'-ms-touch-action': '',
				'touch-action'    : ''
			} );

			$htmlBody.css( {
				'overflow': '',
				'height'  : ''
			} );

			// remove .fp-enabled class
			$( 'html' ).removeClass( ENABLED );

			// remove .fp-responsive class
			$body.removeClass( RESPONSIVE );

			// remove all of the .fp-viewing- classes
			$.each( $body.get( 0 ).className.split( /\s+/ ), function( index, className ) {
				if ( className.indexOf( VIEWING_PREFIX ) === 0 ) {
					$body.removeClass( className );
				}
			} );

			//removing added classes
			$( SECTION_SEL + ', ' + SLIDE_SEL ).each( function() {
				options.scrollOverflowHandler.remove( $( this ) );
				$( this ).removeClass( TABLE + ' ' + ACTIVE );
			} );

			removeAnimation( container );

			//Unwrapping content
			container.find( TABLE_CELL_SEL + ', ' + SLIDES_CONTAINER_SEL + ', ' + SLIDES_WRAPPER_SEL )
					 .each( function() {
						 //unwrap not being use in case there's no child element inside and its just text
						 $( this ).replaceWith( this.childNodes );
					 } );

			//scrolling the page to the top with no animation
			$htmlBody.scrollTop( 0 );

			//removing selectors
			var usedSelectors = [
				SECTION,
				SLIDE,
				SLIDES_CONTAINER
			];
			$.each( usedSelectors, function( index, value ) {
				$( '.' + value ).removeClass( value );
			} );
		}

		/*
		 * Sets the state for a variable with multiple states (original, and temporal)
		 * Some variables such as `autoScrolling` or `recordHistory` might change automatically its state when using `responsive` or `autoScrolling:false`.
		 * This function is used to keep track of both states, the original and the temporal one.
		 * If type is not 'internal', then we assume the user is globally changing the variable.
		 */
		function setVariableState( variable, value, type ) {
			options[variable] = value;
			if ( type !== 'internal' ) {
				originals[variable] = value;
			}
		}

		/**
		 * Displays warnings
		 */
		function displayWarnings() {
			var extensions = [
				'fadingEffect',
				'continuousHorizontal',
				'scrollHorizontally',
				'interlockedSlides',
				'resetSliders',
				'responsiveSlides'
			];
			if ( $( 'html' ).hasClass( ENABLED ) ) {
				showError( 'error', 'Fullpage.js can only be initialized once and you are doing it multiple times!' );
				return;
			}

			// Disable mutually exclusive settings
			if ( options.continuousVertical && (
					options.loopTop || options.loopBottom
				) ) {
				options.continuousVertical = false;
				showError( 'warn', 'Option `loopTop/loopBottom` is mutually exclusive with `continuousVertical`; `continuousVertical` disabled' );
			}

			if ( options.scrollBar && options.scrollOverflow ) {
				showError( 'warn', 'Option `scrollBar` is mutually exclusive with `scrollOverflow`. Sections with scrollOverflow might not work well in Firefox' );
			}

			if ( options.continuousVertical && options.scrollBar ) {
				options.continuousVertical = false;
				showError( 'warn', 'Option `scrollBar` is mutually exclusive with `continuousVertical`; `continuousVertical` disabled' );
			}

			//using extensions? Wrong file!
			extensions.forEach( function( extension ) {
				//is the option set to true?
				if ( options[extension] ) {
					showError( 'warn', 'fullpage.js extensions require jquery.fullpage.extensions.min.js file instead of the usual jquery.fullpage.js. Requested: ' + extension );
				}
			} );

			//anchors can not have the same value as any element ID or NAME
			$.each( options.anchors, function( index, name ) {

				//case insensitive selectors (http://stackoverflow.com/a/19465187/1081396)
				var nameAttr = $document.find( '[name]' ).filter( function() {
					return $( this ).attr( 'name' ) && $( this ).attr( 'name' ).toLowerCase() == name.toLowerCase();
				} );

				var idAttr = $document.find( '[id]' ).filter( function() {
					return $( this ).attr( 'id' ) && $( this ).attr( 'id' ).toLowerCase() == name.toLowerCase();
				} );

				if ( idAttr.length || nameAttr.length ) {
					showError( 'error', 'data-anchor tags can not have the same value as any `id` element on the site (or `name` element for IE).' );
					idAttr.length && showError( 'error', '"' + name + '" is is being used by another element `id` property' );
					nameAttr.length && showError( 'error', '"' + name + '" is is being used by another element `name` property' );
				}
			} );
		}

		/**
		 * Shows a message in the console of the given type.
		 */
		function showError( type, text ) {
			console && console[type] && console[type]( 'fullPage: ' + text );
		}

	}; //end of $.fn.fullpage

	if ( typeof IScroll !== 'undefined' ) {
		/*
		 * Turns iScroll `mousewheel` option off dynamically
		 * https://github.com/cubiq/iscroll/issues/1036
		 */
		IScroll.prototype.wheelOn = function() {
			this.wrapper.addEventListener( 'wheel', this );
			this.wrapper.addEventListener( 'mousewheel', this );
			this.wrapper.addEventListener( 'DOMMouseScroll', this );
		};

		/*
		 * Turns iScroll `mousewheel` option on dynamically
		 * https://github.com/cubiq/iscroll/issues/1036
		 */
		IScroll.prototype.wheelOff = function() {
			this.wrapper.removeEventListener( 'wheel', this );
			this.wrapper.removeEventListener( 'mousewheel', this );
			this.wrapper.removeEventListener( 'DOMMouseScroll', this );
		};
	}

	/**
	 * An object to handle overflow scrolling.
	 * This uses jquery.slimScroll to accomplish overflow scrolling.
	 * It is possible to pass in an alternate scrollOverflowHandler
	 * to the fullpage.js option that implements the same functions
	 * as this handler.
	 *
	 * @type {Object}
	 */
	var iscrollHandler = {
		refreshId       : null,
		iScrollInstances: [],

		/**
		 * Turns off iScroll for the destination section.
		 * When scrolling very fast on some trackpads (and Apple laptops) the inertial scrolling would
		 * scroll the destination section/slide before the sections animations ends.
		 */
		onLeave: function() {
			var scroller = $( SECTION_ACTIVE_SEL ).find( SCROLLABLE_SEL ).data( 'iscrollInstance' );
			if ( typeof scroller !== 'undefined' && scroller ) {
				scroller.wheelOff();
			}
		},

		// Turns off iScroll for the leaving section
		beforeLeave: function() {
			iscrollHandler.onLeave()
		},

		// Turns on iScroll on section load
		afterLoad: function() {
			var scroller = $( SECTION_ACTIVE_SEL ).find( SCROLLABLE_SEL ).data( 'iscrollInstance' );
			if ( typeof scroller !== 'undefined' && scroller ) {
				scroller.wheelOn();
			}
		},

		/**
		 * Called when overflow scrolling is needed for a section.
		 *
		 * @param  {Object} element      jQuery object containing current section
		 * @param  {Number} scrollHeight Current window height in pixels
		 */
		create: function( element, scrollHeight ) {
			var scrollable = element.find( SCROLLABLE_SEL );

			scrollable.height( scrollHeight );
			scrollable.each( function() {
				var $this           = jQuery( this );
				var iScrollInstance = $this.data( 'iscrollInstance' );
				if ( iScrollInstance ) {
					$.each( iscrollHandler.iScrollInstances, function() {
						$( this ).destroy();
					} );
				}

				iScrollInstance = new IScroll( $this.get( 0 ), iscrollOptions );
				iscrollHandler.iScrollInstances.push( iScrollInstance );

				//off by default until the section gets active
				//				iScrollInstance.wheelOff();

				$this.data( 'iscrollInstance', iScrollInstance );
			} );
		},

		/**
		 * Return a boolean depending on whether the scrollable element is a
		 * the end or at the start of the scrolling depending on the given type.
		 *
		 * @param  {String}  type       Either 'top' or 'bottom'
		 * @param  {Object}  scrollable jQuery object for the scrollable element
		 * @return {Boolean}
		 */
		isScrolled: function( type, scrollable ) {
			var scroller = scrollable.data( 'iscrollInstance' );

			//no scroller?
			if ( ! scroller ) {
				return true;
			}

			if ( type === 'top' ) {
				return scroller.y >= 0 && ! scrollable.scrollTop();
			} else if ( type === 'bottom' ) {
				return (
						   0 - scroller.y
					   ) + scrollable.scrollTop() + 1 + scrollable.innerHeight() >= scrollable[0].scrollHeight;
			}
		},

		/**
		 * Returns the scrollable element for the given section.
		 * If there are landscape slides, will only return a scrollable element
		 * if it is in the active slide.
		 *
		 * @param  {Object}  activeSection jQuery object containing current section
		 * @return {Boolean}
		 */
		scrollable: function( activeSection ) {
			// if there are landscape slides, we check if the scrolling bar is in the current one or not
			if ( activeSection.find( SLIDES_WRAPPER_SEL ).length ) {
				return activeSection.find( SLIDE_ACTIVE_SEL ).find( SCROLLABLE_SEL );
			}
			return activeSection.find( SCROLLABLE_SEL );
		},

		/**
		 * Returns the scroll height of the wrapped content.
		 * If this is larger than the window height minus section padding,
		 * overflow scrolling is needed.
		 *
		 * @param  {Object} element jQuery object containing current section
		 * @return {Number}
		 */
		scrollHeight: function( element ) {
			return element.find( SCROLLABLE_SEL ).children().first().get( 0 ).scrollHeight;
		},

		/**
		 * Called when overflow scrolling is no longer needed for a section.
		 *
		 * @param  {Object} element      jQuery object containing current section
		 */
		remove: function( element ) {
			var scrollable = element.find( SCROLLABLE_SEL );
			if ( scrollable.length ) {
				var iScrollInstance = scrollable.data( 'iscrollInstance' );
				iScrollInstance.destroy();

				scrollable.data( 'iscrollInstance', null );
			}
			element.find( SCROLLABLE_SEL ).children().first().children().first().unwrap().unwrap();
		},

		/**
		 * Called when overflow scrolling has already been setup but the
		 * window height has potentially changed.
		 *
		 * @param  {Object} element      jQuery object containing current section
		 * @param  {Number} scrollHeight Current window height in pixels
		 */
		update: function( element, scrollHeight ) {
			//using a timeout in order to execute the refresh function only once when `update` is called multiple times in a
			//short period of time.
			//it also comes on handy because iScroll requires the use of timeout when using `refresh`.
			clearTimeout( iscrollHandler.refreshId );
			iscrollHandler.refreshId = setTimeout( function() {
				$.each( iscrollHandler.iScrollInstances, function() {
					$( this ).get( 0 ).refresh();
				} );
			}, 150 );

			//updating the wrappers height
			element.find( SCROLLABLE_SEL )
				   .css( 'height', scrollHeight + 'px' )
				   .parent()
				   .css( 'height', scrollHeight + 'px' );
		},

		/**
		 * Called to get any additional elements needed to wrap the section
		 * content in order to facilitate overflow scrolling.
		 *
		 * @return {String|Object} Can be a string containing HTML,
		 *                         a DOM element, or jQuery object.
		 */
		wrapContent: function() {
			return '<div class="' + SCROLLABLE + '"><div class="fp-scroller"></div></div>';
		}
	};
} );

(
    function ( $ ) {
        "use strict";
        var $body              = $( 'body' );
        var $container         = $( '#one-page-scroll' );
        var dots               = true;
        var insightInitOnePage = function () {
            $container.fullpage( {
                navigation: dots,
                navigationPosition: 'right',
                lazyLoading: true,
                scrollBar: false,
                css3: true,
                scrollingSpeed: 900,
                scrollOverflow: true,
                scrollOverflowOptions: {
                    click: true
                },
                verticalCentered: true,
                afterLoad: function ( anchorLink, index ) {
                    var $currentRow = $container.children( '.active' );
                    var skin        = $currentRow.attr( 'data-skin' );
                    $body.attr( 'data-row-skin', skin );

                    $currentRow.find( '.tm-animation' ).addClass( 'animate' );

                    $container.find( '> div' ).css( {
                        'will-change': 'auto',
                        '-webkit-transform': 'translate3d(0,0,0)',
                        '-moz-transform': 'translate3d(0,0,0)',
                        '-ms-transform': 'translate3d(0,0,0)',
                        '-o-transform': 'translate3d(0,0,0)',
                        'transform': 'translate3d(0,0,0)',
                        '-webkit-transition': 'none',
                        '-moz-transition': 'none',
                        '-ms-transition': 'none',
                        '-o-transition': 'none',
                        'transition': 'none'
                    } );
                }
            } );
        };

        if ( $container.length > 0 ) {
            $( document ).ready( function () {
                if ( $( '#wpadminbar' ).length > 0 ) {
                    $( 'html' ).addClass( 'admin-bar-enabled' );
                }
                dots = $container.data( 'enable-dots' );
                insightInitOnePage();
            } );
        }
    }
)( jQuery );
