<?php
/**
 * Bifrost Plugin Updater
 *
 * Handles automatic updates for bundled plugins (Revolution Slider, Neuron Core)
 * through the WordPress update system using the NeuronThemes API.
 *
 * @package Bifrost
 * @since 2.4.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Bifrost Plugin Updater Class
 */
class Bifrost_Plugin_Updater {

	/**
	 * API URL for Revolution Slider info
	 */
	const REVSLIDER_API_URL = 'https://support.neuronthemes.com/api/plugin/revslider/info';

	/**
	 * API URL for Bifrost theme info (includes Neuron Core version)
	 */
	const BIFROST_API_URL = 'https://support.neuronthemes.com/api/themes/bifrost/info';

	/**
	 * Cache key for Bifrost API
	 */
	const BIFROST_CACHE_KEY = 'bifrost_theme_update_info';

	/**
	 * Cache duration in seconds (12 hours)
	 */
	const CACHE_DURATION = 43200;

	/**
	 * Plugins we manage updates for
	 */
	private $managed_plugins = array();

	/**
	 * Singleton instance
	 */
	private static $instance = null;

	/**
	 * Get singleton instance
	 */
	public static function get_instance() {
		if ( self::$instance === null ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Constructor
	 */
	private function __construct() {
		$this->managed_plugins = array(
			'revslider' => array(
				'basename'    => 'revslider/revslider.php',
				'slug'        => 'revslider',
				'api_url'     => self::REVSLIDER_API_URL,
				'cache_key'   => 'bifrost_revslider_update_info',
				'name'        => 'Revolution Slider',
			),
			'neuron-core-bifrost' => array(
				'basename'    => 'neuron-core-bifrost/neuron-core-bifrost.php',
				'slug'        => 'neuron-core-bifrost',
				'api_url'     => self::BIFROST_API_URL,
				'cache_key'   => self::BIFROST_CACHE_KEY,
				'name'        => 'Neuron Core Bifrost',
				'api_key'     => 'plugin', // Key in API response
			),
		);

		// Hook into WordPress update system
		add_filter( 'pre_set_site_transient_update_plugins', array( $this, 'check_for_updates' ) );
		add_filter( 'plugins_api', array( $this, 'plugin_info' ), 20, 3 );

		// Add action links on plugins page
		add_filter( 'plugin_action_links_revslider/revslider.php', array( $this, 'add_check_update_link' ) );
		add_filter( 'plugin_action_links_neuron-core-bifrost/neuron-core-bifrost.php', array( $this, 'add_check_update_link' ) );

		// AJAX handler for manual update check
		add_action( 'wp_ajax_bifrost_check_plugin_updates', array( $this, 'ajax_check_updates' ) );

		// Enqueue admin scripts
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_scripts' ) );

		// Clear update cache when theme is updated
		add_action( 'after_switch_theme', array( $this, 'clear_update_cache' ) );
	}

	/**
	 * Enqueue admin scripts for the plugins page
	 */
	public function enqueue_admin_scripts( $hook ) {
		if ( $hook !== 'plugins.php' ) {
			return;
		}

		wp_add_inline_script( 'jquery', $this->get_inline_script() );
		wp_add_inline_style( 'wp-admin', $this->get_inline_styles() );
	}

	/**
	 * Get inline JavaScript for check updates functionality
	 */
	private function get_inline_script() {
		return "
		jQuery(document).ready(function($) {
			$('.bifrost-check-updates').on('click', function(e) {
				e.preventDefault();
				var \$link = $(this);
				var originalText = \$link.text();

				\$link.addClass('bifrost-checking').text('" . esc_js( __( 'Checking...', 'bifrost' ) ) . "');

				$.ajax({
					url: ajaxurl,
					type: 'POST',
					data: {
						action: 'bifrost_check_plugin_updates',
						nonce: \$link.data('nonce')
					},
					success: function(response) {
						if (response.success) {
							if (response.data.has_updates) {
								location.reload();
							} else {
								\$link.text('" . esc_js( __( 'Up to date', 'bifrost' ) ) . "');
								setTimeout(function() {
									\$link.removeClass('bifrost-checking').text(originalText);
								}, 2000);
							}
						} else {
							\$link.text(response.data.message || '" . esc_js( __( 'Error', 'bifrost' ) ) . "');
							setTimeout(function() {
								\$link.removeClass('bifrost-checking').text(originalText);
							}, 2000);
						}
					},
					error: function() {
						\$link.text('" . esc_js( __( 'Connection error', 'bifrost' ) ) . "');
						setTimeout(function() {
							\$link.removeClass('bifrost-checking').text(originalText);
						}, 2000);
					}
				});
			});
		});
		";
	}

	/**
	 * Get inline CSS styles
	 */
	private function get_inline_styles() {
		return "
		.bifrost-check-updates {
			cursor: pointer;
		}
		.bifrost-check-updates.bifrost-checking {
			opacity: 0.7;
			pointer-events: none;
		}
		";
	}

	/**
	 * Add check for updates link on plugins page
	 *
	 * @param array $links Plugin action links
	 * @return array Modified links
	 */
	public function add_check_update_link( $links ) {
		$links['check_updates'] = sprintf(
			'<a href="#" class="bifrost-check-updates" data-nonce="%s">%s</a>',
			wp_create_nonce( 'bifrost_check_plugin_updates' ),
			__( 'Check for updates', 'bifrost' )
		);
		return $links;
	}

	/**
	 * AJAX handler for manual update check
	 */
	public function ajax_check_updates() {
		check_ajax_referer( 'bifrost_check_plugin_updates', 'nonce' );

		if ( ! current_user_can( 'update_plugins' ) ) {
			wp_send_json_error( array( 'message' => __( 'Permission denied.', 'bifrost' ) ) );
		}

		// Clear cache and fetch fresh info
		$this->clear_update_cache();

		$has_updates = false;
		$results = array();

		foreach ( $this->managed_plugins as $plugin_key => $plugin_data ) {
			if ( ! $this->is_plugin_installed( $plugin_data['basename'] ) ) {
				continue;
			}

			$remote_info = $this->get_remote_info( $plugin_key, true );
			$current_version = $this->get_plugin_version( $plugin_data['basename'] );

			if ( $remote_info && isset( $remote_info['version'] ) ) {
				$update_available = version_compare( $current_version, $remote_info['version'], '<' );
				if ( $update_available ) {
					$has_updates = true;
				}
				$results[ $plugin_key ] = array(
					'current' => $current_version,
					'latest'  => $remote_info['version'],
					'update'  => $update_available,
				);
			}
		}

		// Force WordPress to re-check updates
		delete_site_transient( 'update_plugins' );

		wp_send_json_success( array(
			'has_updates' => $has_updates,
			'plugins'     => $results,
			'message'     => $has_updates
				? __( 'Updates available!', 'bifrost' )
				: __( 'All plugins are up to date.', 'bifrost' ),
		) );
	}

	/**
	 * Check for plugin updates
	 *
	 * @param object $transient Update transient
	 * @return object Modified transient
	 */
	public function check_for_updates( $transient ) {
		if ( empty( $transient->checked ) ) {
			return $transient;
		}

		foreach ( $this->managed_plugins as $plugin_key => $plugin_data ) {
			// Skip if plugin is not installed
			if ( ! $this->is_plugin_installed( $plugin_data['basename'] ) ) {
				continue;
			}

			$remote_info = $this->get_remote_info( $plugin_key );

			if ( ! $remote_info || empty( $remote_info['version'] ) ) {
				continue;
			}

			$current_version = $this->get_plugin_version( $plugin_data['basename'] );

			if ( ! $current_version ) {
				continue;
			}

			// Compare versions
			if ( version_compare( $current_version, $remote_info['version'], '<' ) ) {
				$update = (object) array(
					'id'            => $plugin_data['basename'],
					'slug'          => $plugin_data['slug'],
					'plugin'        => $plugin_data['basename'],
					'new_version'   => $remote_info['version'],
					'url'           => isset( $remote_info['homepage'] ) ? $remote_info['homepage'] : '',
					'package'       => isset( $remote_info['download_url'] ) ? $remote_info['download_url'] : '',
					'tested'        => isset( $remote_info['tested_wp'] ) ? $remote_info['tested_wp'] : '',
					'requires_php'  => isset( $remote_info['requires_php'] ) ? $remote_info['requires_php'] : '7.4',
					'requires'      => isset( $remote_info['requires_wp'] ) ? $remote_info['requires_wp'] : '5.0',
					'icons'         => array(),
					'banners'       => array(),
					'banners_rtl'   => array(),
				);

				$transient->response[ $plugin_data['basename'] ] = $update;
			} else {
				// No update available - add to no_update list
				$transient->no_update[ $plugin_data['basename'] ] = (object) array(
					'id'            => $plugin_data['basename'],
					'slug'          => $plugin_data['slug'],
					'plugin'        => $plugin_data['basename'],
					'new_version'   => $current_version,
					'url'           => '',
				);
			}
		}

		return $transient;
	}

	/**
	 * Plugin information for the update modal
	 *
	 * @param mixed  $result Default result
	 * @param string $action API action
	 * @param object $args   API arguments
	 * @return mixed Plugin info or default result
	 */
	public function plugin_info( $result, $action, $args ) {
		if ( $action !== 'plugin_information' ) {
			return $result;
		}

		if ( ! isset( $args->slug ) ) {
			return $result;
		}

		// Check if this is one of our managed plugins
		$plugin_key = null;
		foreach ( $this->managed_plugins as $key => $plugin_data ) {
			if ( $plugin_data['slug'] === $args->slug ) {
				$plugin_key = $key;
				break;
			}
		}

		if ( ! $plugin_key ) {
			return $result;
		}

		$plugin_data = $this->managed_plugins[ $plugin_key ];
		$remote_info = $this->get_remote_info( $plugin_key );

		if ( ! $remote_info ) {
			return $result;
		}

		$current_version = $this->get_plugin_version( $plugin_data['basename'] );

		return (object) array(
			'name'              => isset( $remote_info['name'] ) ? $remote_info['name'] : $plugin_data['name'],
			'slug'              => $plugin_data['slug'],
			'version'           => isset( $remote_info['version'] ) ? $remote_info['version'] : $current_version,
			'author'            => '<a href="https://neuronthemes.com">NeuronThemes</a>',
			'author_profile'    => 'https://neuronthemes.com',
			'homepage'          => isset( $remote_info['homepage'] ) ? $remote_info['homepage'] : 'https://neuronthemes.com',
			'requires'          => isset( $remote_info['requires_wp'] ) ? $remote_info['requires_wp'] : '5.0',
			'tested'            => isset( $remote_info['tested_wp'] ) ? $remote_info['tested_wp'] : '',
			'requires_php'      => isset( $remote_info['requires_php'] ) ? $remote_info['requires_php'] : '7.4',
			'downloaded'        => 0,
			'last_updated'      => isset( $remote_info['last_updated'] ) ? $remote_info['last_updated'] : '',
			'sections'          => array(
				'description' => $this->get_plugin_description( $plugin_key ),
				'changelog'   => isset( $remote_info['changelog'] ) ? $remote_info['changelog'] : '<p>See plugin changelog for details.</p>',
			),
			'download_link'     => isset( $remote_info['download_url'] ) ? $remote_info['download_url'] : '',
		);
	}

	/**
	 * Get remote plugin info from API
	 *
	 * @param string $plugin_key Plugin key
	 * @param bool   $force_refresh Force refresh cache
	 * @return array|false Remote info or false on failure
	 */
	public function get_remote_info( $plugin_key, $force_refresh = false ) {
		if ( ! isset( $this->managed_plugins[ $plugin_key ] ) ) {
			return false;
		}

		$plugin_data = $this->managed_plugins[ $plugin_key ];
		$cache_key = $plugin_data['cache_key'];

		// Try to get from cache
		if ( ! $force_refresh ) {
			$cached = get_transient( $cache_key );
			if ( $cached !== false ) {
				// For Neuron Core, extract from cached Bifrost API response
				if ( isset( $plugin_data['api_key'] ) && $plugin_data['api_key'] === 'plugin' ) {
					return isset( $cached['plugin'] ) ? $cached['plugin'] : false;
				}
				return $cached;
			}
		}

		// Fetch from API
		$response = wp_remote_get( $plugin_data['api_url'], array(
			'timeout' => 15,
			'headers' => array(
				'Accept' => 'application/json',
			),
		) );

		if ( is_wp_error( $response ) ) {
			return false;
		}

		$code = wp_remote_retrieve_response_code( $response );
		if ( $code !== 200 ) {
			return false;
		}

		$body = wp_remote_retrieve_body( $response );
		$data = json_decode( $body, true );

		if ( empty( $data ) ) {
			return false;
		}

		// Cache the result
		set_transient( $cache_key, $data, self::CACHE_DURATION );

		// For Neuron Core, extract plugin info from Bifrost API response
		if ( isset( $plugin_data['api_key'] ) && $plugin_data['api_key'] === 'plugin' ) {
			return isset( $data['plugin'] ) ? $data['plugin'] : false;
		}

		// For RevSlider, check if version exists directly
		if ( ! isset( $data['version'] ) ) {
			return false;
		}

		return $data;
	}

	/**
	 * Check if a plugin is installed
	 *
	 * @param string $basename Plugin basename
	 * @return bool
	 */
	private function is_plugin_installed( $basename ) {
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$plugins = get_plugins();
		return isset( $plugins[ $basename ] );
	}

	/**
	 * Get installed plugin version
	 *
	 * @param string $basename Plugin basename
	 * @return string|false Version string or false
	 */
	private function get_plugin_version( $basename ) {
		if ( ! function_exists( 'get_plugins' ) ) {
			require_once ABSPATH . 'wp-admin/includes/plugin.php';
		}

		$plugins = get_plugins();

		if ( isset( $plugins[ $basename ] ) && isset( $plugins[ $basename ]['Version'] ) ) {
			return $plugins[ $basename ]['Version'];
		}

		return false;
	}

	/**
	 * Get plugin description
	 *
	 * @param string $plugin_key Plugin key
	 * @return string Description HTML
	 */
	private function get_plugin_description( $plugin_key ) {
		$descriptions = array(
			'revslider' => 'Revolution Slider is an innovative, responsive WordPress slider plugin that displays your content the beautiful way on desktop, tablet, and mobile devices.',
			'neuron-core-bifrost' => 'Neuron Core Bifrost provides essential functionality for the Bifrost theme including widgets, customizer options, and demo import features.',
		);

		return isset( $descriptions[ $plugin_key ] ) ? '<p>' . $descriptions[ $plugin_key ] . '</p>' : '';
	}

	/**
	 * Clear update cache for all managed plugins
	 */
	public function clear_update_cache() {
		foreach ( $this->managed_plugins as $plugin_data ) {
			delete_transient( $plugin_data['cache_key'] );
		}

		// Also clear WordPress plugin update transient
		delete_site_transient( 'update_plugins' );
	}

	/**
	 * Manually check for updates (useful for debugging)
	 *
	 * @return array Update status for all managed plugins
	 */
	public function check_all_updates() {
		$this->clear_update_cache();

		$results = array();

		foreach ( $this->managed_plugins as $plugin_key => $plugin_data ) {
			if ( ! $this->is_plugin_installed( $plugin_data['basename'] ) ) {
				$results[ $plugin_key ] = array(
					'status' => 'not_installed',
				);
				continue;
			}

			$remote_info = $this->get_remote_info( $plugin_key, true );
			$current_version = $this->get_plugin_version( $plugin_data['basename'] );

			if ( ! $remote_info ) {
				$results[ $plugin_key ] = array(
					'status'          => 'error',
					'current_version' => $current_version,
					'message'         => 'Could not fetch remote info',
				);
				continue;
			}

			$has_update = version_compare( $current_version, $remote_info['version'], '<' );

			$results[ $plugin_key ] = array(
				'status'          => $has_update ? 'update_available' : 'up_to_date',
				'current_version' => $current_version,
				'latest_version'  => $remote_info['version'],
				'has_update'      => $has_update,
			);
		}

		return $results;
	}
}

// Initialize the updater
Bifrost_Plugin_Updater::get_instance();
