<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Cassio_Widget_Portfolio_Masonry_Grid extends Arts_Widget_Base {
	protected static $_instance, $_posts, $_post_type = 'arts_portfolio_item';
	protected static $_data_static_fields             = array( 'title', 'permalink', 'categories_names', 'image' );

	public function get_name() {
		return 'cassio-widget-portfolio-masonry-grid';
	}

	public function get_title() {
		return esc_html__( 'Portfolio Masonry Grid', 'cassio' );
	}

	public function get_icon() {
		return 'eicon-sitemap';
	}

	public function get_categories() {
		return array( 'cassio-dynamic' );
	}

	protected function is_dynamic_content(): bool {
		return true;
	}

	protected function register_controls() {

		// posts toggles & posts amount
		$this->add_controls_posts_toggles();

		$this->start_controls_section(
			'settings_section',
			array(
				'label' => esc_html__( 'Elements', 'cassio' ),
				'tab'   => Controls_Manager::TAB_SETTINGS,
			)
		);

		$this->add_control(
			'enable_categories',
			array(
				'label'   => esc_html__( 'Show Categories', 'cassio' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'layout_section',
			array(
				'label' => esc_html__( 'Layout', 'cassio' ),
				'tab'   => Controls_Manager::TAB_LAYOUT,
			)
		);

		$this->add_control(
			'heading_grid',
			array(
				'label' => esc_html__( 'Grid', 'cassio' ),
				'type'  => Controls_Manager::HEADING,
			)
		);

		$this->add_control(
			'enable_fancy',
			array(
				'label'     => esc_html__( 'Enable Fancy Grid', 'cassio' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => array(
					'columns!' => '12',
				),
			)
		);

		$selector_value_string = 'flex: 0 0 calc({{VALUE}} / 12  * 100%); max-width: calc({{VALUE}} / 12 * 100%);';
		$this->add_responsive_control(
			'columns',
			array(
				'label'           => esc_html__( 'Columns', 'cassio' ),
				'type'            => Controls_Manager::SELECT,
				'options'         => array(
					3  => esc_html__( 'Four Columns', 'cassio' ),
					4  => esc_html__( 'Three Columns', 'cassio' ),
					6  => esc_html__( 'Two Columns', 'cassio' ),
					12 => esc_html__( 'Single Column', 'cassio' ),
				),
				'selectors'       => array(
					'{{WRAPPER}} .grid__sizer' => $selector_value_string,
					'{{WRAPPER}} .grid__item'  => $selector_value_string,
				),
				'render_type'     => 'template',
				'default'         => 4,
				'desktop_default' => 4,
				'tablet_default'  => 6,
				'mobile_default'  => 12,
				'frontend_available' => true,
			)
		);

		$this->add_responsive_control(
			'space_between',
			array(
				'label'           => esc_html__( 'Space Between', 'cassio' ),
				'type'            => Controls_Manager::SLIDER,
				'range'           => array(
					'px' => array(
						'min'  => 0,
						'max'  => 200,
						'step' => 1,
					),
					'vw' => array(
						'min'  => 0,
						'max'  => 100,
						'step' => 1,
					),
				),
				'size_units'      => array( 'px', 'vw', 'vh' ),
				'devices'         => array( 'desktop', 'tablet', 'mobile' ),
				'default' => array(
					'size' => 6,
					'unit' => 'vw',
				),
				'desktop_default' => array(
					'size' => 6,
					'unit' => 'vw',
				),
				'tablet_default'  => array(
					'size' => 40,
					'unit' => 'px',
				),
				'mobile_default'  => array(
					'size' => 20,
					'unit' => 'px',
				),
				'selectors'       => array(
					'{{WRAPPER}}.elementor-widget_vertical-captions .elementor-widget-container' => 'padding: 0 calc({{SIZE}}{{UNIT}});',
					'{{WRAPPER}} .grid'       => 'margin: calc(-{{SIZE}}{{UNIT}}) calc(-{{SIZE}}{{UNIT}});',
					'{{WRAPPER}} .grid__item' => 'padding: calc({{SIZE}}{{UNIT}});',
					'{{WRAPPER}} .grid_fancy:not(.grid_fancy-disable) .grid__item:nth-child(3)' => 'margin-top: {{SIZE}}{{UNIT}};',
					'(tablet){{WRAPPER}} .grid_fancy:not(.grid_fancy-disable) .grid__item:nth-child(3)' => 'margin-top: 0;',
					'(mobile){{WRAPPER}} .grid_fancy:not(.grid_fancy-disable) .grid__item:nth-child(3)' => 'margin-top: 0;',
				),
				'render_type'     => 'template',
				'frontend_available' => true,
			)
		);

		$this->add_control(
			'heading_images',
			array(
				'label'     => esc_html__( 'Images', 'cassio' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'image_type',
			array(
				'label'   => esc_html__( 'Priority Image', 'cassio' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'primary',
				'options' => array(
					'primary'   => esc_html__( 'Primary Featured Image', 'cassio' ),
					'secondary' => esc_html__( 'Secondary Featured Image', 'cassio' ),
				),
			)
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			array(
				'label'   => esc_html__( 'Thumbnail Size', 'cassio' ),
				'name'    => 'image', // Usage: `{name}_size` and `{name}_custom_dimension`, in this case `image_size` and `image_custom_dimension`.
				'default' => 'full',
			)
		);

		$this->add_control(
			'image_type_info',
			array(
				'type'            => Controls_Manager::RAW_HTML,
				'raw'             => sprintf(
					'%1$s<br><br>%2$s <a href="%3$s" target="_blank">%4$s</a>',
					esc_html__( 'If a secondary featured image is not set for a post then it will fallback to a primary featured image.', 'cassio' ),
					esc_html__( 'You can edit your posts and adjust the featured images', 'cassio' ),
					admin_url( 'edit.php?post_type=arts_portfolio_item' ),
					esc_html__( 'in WordPress admin panel', 'cassio' )
				),
				'content_classes' => 'elementor-panel-alert elementor-panel-alert-warning',
				'condition'       => array(
					'image_type' => 'secondary',
				),
			)
		);

		$this->add_control(
			'heading_filter',
			array(
				'label'     => esc_html__( 'Filter', 'cassio' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'enable_filter',
			array(
				'label' => esc_html__( 'Enable Grid Filter', 'cassio' ),
				'type'  => Controls_Manager::SWITCHER,
			)
		);

		$this->add_control(
			'filter_all_label',
			array(
				'label'     => esc_html__( '"All" label', 'cassio' ),
				'type'      => Controls_Manager::TEXT,
				'default'   => esc_html__( 'All', 'cassio' ),
				'condition' => array(
					'enable_filter' => 'yes',
				),
			)
		);

		$this->add_control(
			'filter_row_class',
			array(
				'label'     => esc_html__( 'Justify Filter Items', 'cassio' ),
				'type'      => Controls_Manager::CHOOSE,
				'options'   => array(
					'justify-content-lg-between justify-content-center text-center' => array(
						'title' => esc_html__( 'Space Between', 'cassio' ),
						'icon'  => 'eicon-text-align-justify',
					),
					'justify-content-start text-left'    => array(
						'title' => esc_html__( 'Start', 'cassio' ),
						'icon'  => 'eicon-text-align-left',
					),
					'justify-content-center text-center' => array(
						'title' => esc_html__( 'Center', 'cassio' ),
						'icon'  => 'eicon-text-align-center',
					),
					'justify-content-end text-right'     => array(
						'title' => esc_html__( 'End', 'cassio' ),
						'icon'  => 'eicon-text-align-right',
					),
				),
				'default'   => 'justify-content-start text-left',
				'condition' => array(
					'enable_filter' => 'yes',
				),
				'toggle'    => false,

			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'style_section',
			array(
				'label' => esc_html__( 'Style', 'cassio' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'items_style',
			array(
				'label'   => esc_html__( 'Items Style', 'cassio' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'figure-portfolio-big',
				'options' => array(
					'figure-portfolio-big'        => esc_html__( 'Normal', 'cassio' ),
					'figure-portfolio-item_hover' => esc_html__( 'Hover Curtain', 'cassio' ),
				),
			)
		);

		$this->add_control(
			'captions_alignment',
			array(
				'label'     => esc_html__( 'Captions Alignment', 'cassio' ),
				'type'      => Controls_Manager::CHOOSE,
				'options'   => array(
					'text-left'   => array(
						'title' => esc_html__( 'Left', 'cassio' ),
						'icon'  => 'eicon-text-align-left',
					),
					'text-center' => array(
						'title' => esc_html__( 'Center', 'cassio' ),
						'icon'  => 'eicon-text-align-center',
					),
					'text-right'  => array(
						'title' => esc_html__( 'Right', 'cassio' ),
						'icon'  => 'eicon-text-align-right',
					),
				),
				'default'   => 'text-right',
				'toggle'    => false,
				'condition' => array(
					'items_style' => 'figure-portfolio-big',
				),
			)
		);

		$this->add_control(
			'show_letter',
			array(
				'label'     => esc_html__( 'Show Parallax Letters', 'cassio' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => array(
					'items_style' => 'figure-portfolio-big',
				),
			)
		);

		$this->add_control(
			'letters_amount',
			array(
				'label'      => esc_html__( 'Amount of Letters', 'cassio' ),
				'type'       => Controls_Manager::SLIDER,
				'default'    => array(
					'size' => 1,
					'unit' => 'px',
				),
				'size_units' => array( 'px' ),
				'range'      => array(
					'px' => array(
						'min'  => 1,
						'max'  => 3,
						'step' => 1,
					),
				),
				'condition'  => array(
					'show_letter' => 'yes',
				),
			)
		);

		$this->add_control(
			'letters_adjustment_1',
			array(
				'type'       => Controls_Manager::HIDDEN,
				'default'    => 'yes',
				'conditions' => array(
					'terms' => array(
						array(
							'name'     => 'show_letter',
							'operator' => '==',
							'value'    => 'yes',
						),
						array(
							'name'     => 'letters_amount[size]',
							'operator' => '==',
							'value'    => '2',
						),
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .figure-portfolio-big__letter' => 'transform: translateX(-50%);',
				),
				'frontend_available' => true,
			)
		);

		$this->add_control(
			'letters_adjustment_2',
			array(
				'type'       => Controls_Manager::HIDDEN,
				'default'    => 'yes',
				'conditions' => array(
					'terms' => array(
						array(
							'name'     => 'show_letter',
							'operator' => '==',
							'value'    => 'yes',
						),
						array(
							'name'     => 'letters_amount[size]',
							'operator' => '>',
							'value'    => '2',
						),
					),
				),
				'selectors'  => array(
					'{{WRAPPER}} .figure-portfolio-big__letter' => 'transform: translateX(-25%);',
				),
				'frontend_available' => true,
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'typography_section',
			array(
				'label' => esc_html__( 'Typography', 'cassio' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'heading_tag',
			array(
				'label'   => esc_html__( 'Headings HTML Tag', 'cassio' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'h2',
				'options' => array(
					'h2' => esc_html( '<h2>' ),
					'h3' => esc_html( '<h3>' ),
					'h4' => esc_html( '<h4>' ),
					'h5' => esc_html( '<h5>' ),
					'h6' => esc_html( '<h6>' ),
				),
			)
		);

		$this->add_control(
			'heading_size',
			array(
				'label'   => esc_html__( 'Headings Size', 'cassio' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'h1',
				'options' => array(
					'h1'       => esc_html__( 'H1', 'cassio' ),
					'h2'       => esc_html__( 'H2', 'cassio' ),
					'h3'       => esc_html__( 'H3', 'cassio' ),
					'h4'       => esc_html__( 'H4', 'cassio' ),
					'h5'       => esc_html__( 'H5', 'cassio' ),
					'h6'       => esc_html__( 'H6', 'cassio' ),
					'text-xl'  => esc_html__( 'XL', 'cassio' ),
					'text-xxl' => esc_html__( 'XXL', 'cassio' ),
				),
			)
		);

		$this->add_control(
			'heading_weight',
			array(
				'label'   => esc_html__( 'Headings Weight', 'cassio' ),
				'type'    => Controls_Manager::SELECT,
				'default' => '',
				'options' => array(
					''              => esc_html__( 'Auto', 'cassio' ),
					'heading-light' => esc_html__( 'Normal', 'cassio' ),
					'heading-bold'  => esc_html__( 'Bold', 'cassio' ),
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'animation_section',
			array(
				'label' => esc_html__( 'Animation', 'cassio' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'transition_text_enabled',
			array(
				'label'   => esc_html__( 'Enable Text Transition', 'cassio' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'enable_animation',
			array(
				'label'   => esc_html__( 'Enable on-scroll animation', 'cassio' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => '',
			)
		);

		$this->add_control(
			'heading_parallax',
			array(
				'label'     => esc_html__( 'Parallax', 'cassio' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'image_parallax',
			array(
				'label'   => esc_html__( 'Enable parallax', 'cassio' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'image_parallax_speed',
			array(
				'label'     => esc_html__( 'Parallax Speed', 'cassio' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'factor' => array(
						'min'  => -0.5,
						'max'  => 0.5,
						'step' => 0.01,
					),
				),
				'default'   => array(
					'unit' => 'factor',
					'size' => 0.1,
				),
				'condition' => array(
					'image_parallax' => 'yes',
				),
			)
		);

		$this->end_controls_section();

	}

	protected function render() {
		$settings          = $this->get_settings_for_display();
		$posts             = $this->get_posts_to_display();
		$active_taxonomies = $settings['enable_filter'] ? $this->get_taxonomies_to_display( $posts ) : array();

		$columns_desktop = array_key_exists( 'columns', $settings ) ? $settings['columns'] : 12;
		$columns_tablet  = array_key_exists( 'columns_tablet', $settings ) ? $settings['columns_tablet'] : 12;
		$columns_mobile  = array_key_exists( 'columns_mobile', $settings ) ? $settings['columns_mobile'] : 12;

		$col_desktop = 'grid__item_desktop-' . $this->translate_columns_settings_inverse( $columns_desktop );
		$col_tablet  = 'grid__item_tablet-' . $this->translate_columns_settings_inverse( $columns_tablet );
		$col_mobile  = 'grid__item_mobile-' . $this->translate_columns_settings_inverse( $columns_mobile );

		$this->add_render_attribute(
			'section',
			array(
				'class' => array( 'section', 'section-portfolio' ),
			)
		);

		$this->add_render_attribute(
			'grid',
			array(
				'class'                    => array( 'grid', 'js-grid' ),
				'data-grid-columns'        => 12 / $this->translate_columns_settings( $columns_desktop ),
				'data-grid-columns-tablet' => 12 / $this->translate_columns_settings( $columns_tablet ),
				'data-grid-columns-mobile' => 12 / $this->translate_columns_settings( $columns_mobile ),
			)
		);

		$this->add_render_attribute(
			'sizerAtts',
			array(
				'class' => array( 'grid__sizer', 'js-grid__sizer', $col_desktop, $col_tablet, $col_mobile ),
			)
		);

		$this->add_render_attribute(
			'filterRow',
			array(
				'class' => array( 'row', $settings['filter_row_class'] ),
			)
		);

		if ( $settings['enable_fancy'] ) {

			$this->add_render_attribute(
				'grid',
				array(
					'class' => array( 'grid_fancy', 'js-grid' ),
				)
			);

		}

		$this->add_render_attribute(
			'heading',
			array(
				'class' => array( $settings['heading_size'], $settings['heading_weight'] ),
			)
		);

		if ( $settings['transition_text_enabled'] ) {
			$this->add_render_attribute(
				'heading',
				array(
					'class'                => array( 'split-text', 'js-text-to-fly', 'js-split-text' ),
					'data-split-text-type' => 'lines, words, chars',
				)
			);
		}

		$this->add_render_attribute(
			'category',
			array(
				'class' => array( 'figure-portfolio__category' ),
			)
		);

		if ( $settings['items_style'] == 'figure-portfolio-item_hover' ) {

			$this->add_render_attribute(
				'heading',
				array(
					'class' => array( 'figure-portfolio__heading' ),
				)
			);

		} else {

			$this->add_render_attribute(
				'heading',
				array(
					'class' => array( 'figure-portfolio-big__heading' ),
				)
			);
			$this->add_render_attribute(
				'category',
				array(
					'class'                => array( 'split-text', 'js-split-text', 'figure-portfolio-big__category' ),
					'data-split-text-type' => 'lines, words, chars',
				)
			);

		}

		if ( $settings['enable_animation'] ) {

			$this->add_render_attribute(
				'section',
				array(
					'data-os-animation' => 'data-os-animation',
				)
			);

			$this->add_render_attribute(
				'heading',
				array(
					'data-split-text-set' => 'chars',
				)
			);

			$this->add_render_attribute(
				'category',
				array(
					'data-split-text-set' => 'chars',
				)
			);

		}

		?>

		<div <?php $this->print_render_attribute_string( 'section' ); ?>>
			<?php if ( ! empty( $active_taxonomies ) && $settings['enable_filter'] ) : ?>
				<div class="section-portfolio__wrapper-tabs">
					<div class="filter js-filter">
						<div class="filter__inner">
							<div <?php $this->print_render_attribute_string( 'filterRow' ); ?>>
								<div class="col-xl-auto col-12 filter__item filter__item_active js-filter__item" data-filter="*">
									<div><?php echo esc_html( $settings['filter_all_label'] ); ?></div>
								</div>
								<?php foreach ( $active_taxonomies as $item ) : ?>
									<div class="col-xl-auto col-12 filter__item js-filter__item" data-filter=".category-<?php echo $item['slug']; ?>">
										<div><?php echo $item['name']; ?></div>
									</div>
								<?php endforeach; ?>
								<div class="filter__underline js-filter__underline"></div>
							</div>
						</div>
					</div>
				</div>
			<?php endif; ?>

			<?php if ( ! empty( $posts ) ) : ?>
				<div class="section-portfolio__wrapper-grid">
					<div <?php $this->print_render_attribute_string( 'grid' ); ?>>
						<div <?php $this->print_render_attribute_string( 'sizerAtts' ); ?>></div>
						<?php foreach ( $posts as $index => $item ) : ?>
								<?php
									$categories = array();

									$this->add_render_attribute(
										'itemAtts' . $index,
										array(
											'class' => array( 'grid__item', 'js-grid__item', $col_desktop, $col_tablet, $col_mobile ),
										)
									);

									$this->add_render_attribute(
										'linkAtts' . $index,
										array(
											'class' => array( 'figure-portfolio', $settings['items_style'] ),
											'href'  => $item['permalink'],
										)
									);

								if ( $settings['transition_text_enabled'] ) {
									$this->add_render_attribute(
										'linkAtts' . $index,
										'data-pjax-link',
										'flyingHeading'
									);
								}

								if ( $settings['items_style'] == 'figure-portfolio-big' ) {
									$this->add_render_attribute(
										'linkAtts' . $index,
										array(
											'class' => $settings['captions_alignment'],
										)
									);
								}

								if ( $settings['enable_animation'] ) {
									$this->add_render_attribute(
										'linkAtts' . $index,
										array(
											'data-os-animation' => 'data-os-animation',
										)
									);
								}

								// categories
								if ( array_key_exists( 'categories', $item ) && is_array( $item['categories'] ) ) {
									foreach ( $item['categories'] as $taxonomy ) {
										$this->add_render_attribute(
											'itemAtts' . $index,
											array(
												'class' => 'category-' . esc_attr( $taxonomy->slug ),
											)
										);
										$categories[] .= $taxonomy->name;
									}
								}

								if ( array_key_exists( 'permalink_is_external', $item ) && ! ! $item['permalink_is_external'] ) {
									$this->add_render_attribute( 'linkAtts' . $index, 'target', '_blank' );
								}

								if ( array_key_exists( 'permalink_nofollow', $item ) && ! ! $item['permalink_nofollow'] ) {
									$this->add_render_attribute( 'linkAtts' . $index, 'rel', 'nofollow' );
								}

								if ( ! empty( $item['title'] ) ) {
									$this->add_render_attribute( 'linkAtts' . $index, 'aria-label', $item['title'] );
								} else {
									$this->add_render_attribute( 'linkAtts' . $index, 'aria-label', esc_html__( 'View Project', 'cassio' ) );
								}

								$categories_string = implode( '&nbsp;&nbsp;/&nbsp;&nbsp;', $categories );

								?>
								<div <?php $this->print_render_attribute_string( 'itemAtts' . $index ); ?>>
									<a <?php $this->print_render_attribute_string( 'linkAtts' . $index ); ?>>
										<?php if ( $settings['items_style'] == 'figure-portfolio-item_hover' ) : ?>
											<?php
												$this->add_render_attribute(
													'wrapperOuter' . $index,
													array(
														'class' => 'figure-portfolio__wrapper-img',
													)
												);
											?>
											<div <?php $this->print_render_attribute_string( 'wrapperOuter' . $index ); ?>>
												<div class="figure-portfolio__wrapper-img-inner">
													<div class="figure-portfolio__wrapper-img-zoom">
														<?php
															arts_the_lazy_image(
																array(
																	'id'   => $this->get_priority_image_id_to_display( $item, $settings['image_type'] ),
																	'type' => 'image',
																	'size' => $settings['image_size'],
																	'parallax' => array(
																		'enabled' => $settings['image_parallax'],
																		'factor' => $settings['image_parallax_speed'] ? $settings['image_parallax_speed']['size'] : 0,
																	),
																)
															);
														?>
													</div>
												</div>
												<div class="figure-portfolio__content">
													<?php if ( $settings['enable_categories'] && ! empty( $categories_string ) ) : ?>
														<div <?php $this->print_render_attribute_string( 'category' ); ?>><?php echo $categories_string; ?></div>
													<?php endif; ?>
													<?php if ( ! empty( $item['title'] ) ) : ?>
														<div class="figure-portfolio__header">
															<<?php echo $settings['heading_tag']; ?> <?php $this->print_render_attribute_string( 'heading' ); ?>><?php echo $item['title']; ?></<?php echo $settings['heading_tag']; ?>>
															<div class="figure-portfolio__icon material-icons">keyboard_arrow_right</div>
														</div>
														<div class="figure-portfolio__curtain"></div>
													<?php endif; ?>
												</div>
											</div>
										<?php else : ?>
											<?php if ( $settings['show_letter'] ) : ?>
												<?php
													$amount_letters = intval( $settings['letters_amount']['size'] );
													$letter         = mb_substr( $item['title'], 0, $amount_letters );
												if ( $amount_letters === 1 ) {
													$letter = mb_strtoupper( $letter );
												}
												?>
												<div class="figure-portfolio-big__wrapper-letter">
													<div class="figure-portfolio-big__letter" data-art-parallax="element" data-art-parallax-y="-75%"><?php echo $letter; ?></div>
												</div>
											<?php endif; ?>
											<?php
												$this->add_render_attribute(
													'wrapperOuter' . $index,
													array(
														'class' => 'figure-portfolio-big__wrapper-img',
													)
												);
											?>
											<div <?php $this->print_render_attribute_string( 'wrapperOuter' . $index ); ?>>
												<div class="figure-portfolio-big__wrapper-img-inner">
													<div class="figure-portfolio-big__wrapper-img-zoom">
														<?php
															arts_the_lazy_image(
																array(
																	'id'   => $this->get_priority_image_id_to_display( $item, $settings['image_type'] ),
																	'type' => 'image',
																	'size' => $settings['image_size'],
																	'parallax' => array(
																		'wrapper' => true,
																		'enabled' => $settings['image_parallax'],
																		'factor' => $settings['image_parallax_speed'] ? $settings['image_parallax_speed']['size'] : 0,
																	),
																)
															);
														?>
													</div>
												</div>
											</div>
											<div class="figure-portfolio-big__content">
												<<?php echo $settings['heading_tag']; ?> <?php $this->print_render_attribute_string( 'heading' ); ?>><?php echo $item['title']; ?></<?php echo $settings['heading_tag']; ?>>
												<?php if ( $settings['enable_categories'] ) : ?>
													<div <?php $this->print_render_attribute_string( 'category' ); ?>><?php echo $categories_string; ?></div>
												<?php endif; ?>
											</div>
										<?php endif; ?>
									</a>
								</div>
						<?php endforeach; ?>
					</div>
				</div>
			<?php endif; ?>

		</div>

		<?php
	}
}
