<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Cassio_Widget_Slider_Images extends Arts_Widget_Base {
	protected static $_instance = null;

	public function get_name() {
		return 'cassio-widget-slider-images';
	}

	public function get_title() {
		return esc_html__( 'Slider Images', 'cassio' );
	}

	public function get_icon() {
		return 'eicon-plug';
	}

	public function get_categories() {
		return array( 'cassio-static' );
	}

	public function get_script_depends() {
		return array( 'swiper' );
	}

	public function get_style_depends() {
		return array( 'swiper' );
	}

	protected function register_controls() {

		$this->start_controls_section(
			'content_section',
			array(
				'label' => esc_html__( 'Images', 'cassio' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		$this->add_control(
			'images_gallery',
			array(
				'type'    => Controls_Manager::GALLERY,
				'default' => array(),
			)
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			array(
				'label'   => esc_html__( 'Thumbnail Size', 'cassio' ),
				'name'    => 'image', // Usage: `{name}_size` and `{name}_custom_dimension`, in this case `image_size` and `image_custom_dimension`.
				'default' => 'full',
			)
		);

		$this->add_control(
			'enable_captions',
			array(
				'label'   => esc_html__( 'Enable Captions', 'cassio' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->end_controls_section();

		/**
		 * Section Slider
		 */
		$this->start_controls_section(
			'slider_section',
			array(
				'label' => esc_html__( 'Slider', 'cassio' ),
				'tab'   => Controls_Manager::TAB_SETTINGS,
			)
		);

		$this->add_responsive_control(
			'slides_per_view',
			array(
				'label'              => esc_html__( 'Slides Per Screen', 'cassio' ),
				'type'               => Controls_Manager::SLIDER,
				'range'              => array(
					'number' => array(
						'min'  => 1,
						'max'  => 4,
						'step' => 1,
					),
				),
				'devices'            => array( 'desktop', 'tablet', 'mobile' ),
				'default'            => array(
					'size' => 3.5,
					'unit' => 'number',
				),
				'desktop_default'    => array(
					'size' => 3.5,
					'unit' => 'number',
				),
				'tablet_default'     => array(
					'size' => 1.33,
					'unit' => 'number',
				),
				'mobile_default'     => array(
					'size' => 1.33,
					'unit' => 'number',
				),
				'frontend_available' => true,
			)
		);

		$this->add_responsive_control(
			'centered_slides',
			array(
				'label'              => esc_html__( 'Horizontaly Centered Slides', 'cassio' ),
				'label_block'        => true,
				'type'               => Controls_Manager::SWITCHER,
				'default'            => true,
				'desktop_default'    => true,
				'tablet_default'     => true,
				'mobile_default'     => true,
				'frontend_available' => true,
			)
		);

		$this->add_control(
			'vertical_centered_slides',
			array(
				'label'   => esc_html__( 'Vertically Centered Slides', 'cassio' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_responsive_control(
			'space_between',
			array(
				'label'              => esc_html__( 'Space Between Slides', 'cassio' ),
				'type'               => Controls_Manager::SLIDER,
				'range'              => array(
					'px' => array(
						'min'  => 0,
						'max'  => 160,
						'step' => 1,
					),
				),
				'devices'            => array( 'desktop', 'tablet', 'mobile' ),
				'default'            => array(
					'size' => 30,
					'unit' => 'px',
				),
				'desktop_default'    => array(
					'size' => 30,
					'unit' => 'px',
				),
				'tablet_default'     => array(
					'size' => 15,
					'unit' => 'px',
				),
				'mobile_default'     => array(
					'size' => 15,
					'unit' => 'px',
				),
				'frontend_available' => true,
			)
		);

		$this->add_control(
			'enable_autoplay',
			array(
				'label'   => esc_html__( 'Autoplay', 'cassio' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'autoplay_delay',
			array(
				'label'     => esc_html__( 'Autoplay Delay (ms)', 'cassio' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'ms' => array(
						'min'  => 1000,
						'max'  => 60000,
						'step' => 100,
					),
				),
				'default'   => array(
					'unit' => 'ms',
					'size' => 6000,
				),
				'condition' => array(
					'enable_autoplay' => 'yes',
				),
			)
		);

		$this->add_control(
			'speed',
			array(
				'label'   => esc_html__( 'Speed', 'cassio' ),
				'type'    => Controls_Manager::SLIDER,
				'range'   => array(
					'ms' => array(
						'min'  => 100,
						'max'  => 10000,
						'step' => 100,
					),
				),
				'default' => array(
					'unit' => 'ms',
					'size' => 1200,
				),
			)
		);

		$this->add_control(
			'touch_ratio',
			array(
				'label'   => esc_html__( 'Touch Ratio', 'cassio' ),
				'type'    => Controls_Manager::SLIDER,
				'range'   => array(
					'ms' => array(
						'min'  => 1,
						'max'  => 4,
						'step' => 0.1,
					),
				),
				'default' => array(
					'unit' => 'px',
					'size' => 3,
				),
			)
		);

		$this->add_control(
			'heading_controls',
			array(
				'label'     => esc_html__( 'Controls', 'cassio' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'controls_position',
			array(
				'label'   => esc_html__( 'Controls Position', 'cassio' ),
				'type'    => Controls_Manager::CHOOSE,
				'options' => array(
					'slider-images_footer-top'    => array(
						'title' => esc_html__( 'Top', 'cassio' ),
						'icon'  => 'eicon-v-align-top',
					),
					'slider-images_footer-bottom' => array(
						'title' => esc_html__( 'Bottom', 'cassio' ),
						'icon'  => 'eicon-v-align-bottom',
					),
				),
				'default' => 'slider-images_footer-top',
				'toggle'  => false,
			)
		);

		$this->add_control(
			'enable_navigation',
			array(
				'label'   => esc_html__( 'Enable Navigation Arrows', 'cassio' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'enable_dots',
			array(
				'label'   => esc_html__( 'Enable Dots', 'cassio' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'captions_dots_note',
			array(
				'type'            => Controls_Manager::RAW_HTML,
				'raw'             => esc_html__( 'Please note that slider dots won\'t be rendered unless you have enabled images captions.', 'cassio' ),
				'content_classes' => 'elementor-panel-alert elementor-panel-alert-warning',
				'condition'       => array(
					'enable_dots'     => 'yes',
					'enable_captions' => 'yes',
				),
			)
		);

		$this->add_control(
			'enable_counter',
			array(
				'label'   => esc_html__( 'Enable Counter', 'cassio' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'counter_zeros',
			array(
				'label'     => esc_html__( 'Counter Prefix', 'cassio' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 2,
				'options'   => array(
					0 => esc_html__( 'None', 'cassio' ),
					1 => esc_html__( '1 Zero', 'cassio' ),
					2 => esc_html__( '2 Zeros', 'cassio' ),
				),
				'condition' => array(
					'enable_counter' => 'yes',
				),
			)
		);

		$this->end_controls_section();

	}

	protected function render() {

		$settings = $this->get_settings_for_display();

		$this->add_render_attribute(
			'section',
			array(
				'class' => array( 'slider', 'slider-images', $settings['controls_position'] ),
			)
		);

		$this->add_render_attribute(
			'swiper',
			array(
				'class'                => array( 'swiper', 'swiper-container', 'js-slider-images', 'slider_draggable' ),
				'data-speed'           => $settings['speed']['size'],
				'data-slides-per-view' => $settings['slides_per_view']['size'],
				'data-space-between'   => $settings['space_between']['size'],
				'data-centered-slides' => $settings['centered_slides'],
				'data-touch-ratio'     => $settings['touch_ratio']['size'],
				'data-counter-zeros'   => $settings['counter_zeros'],
			)
		);

		if ( array_key_exists( 'slides_per_view_tablet', $settings ) ) {
			$this->add_render_attribute( 'swiper', 'data-slides-per-view-tablet', $settings['slides_per_view_tablet']['size'] );
		}

		if ( array_key_exists( 'slides_per_view_mobile', $settings ) ) {
			$this->add_render_attribute( 'swiper', 'data-slides-per-view-mobile', $settings['slides_per_view_mobile']['size'] );
		}

		if ( array_key_exists( 'space_between_tablet', $settings ) ) {
			$this->add_render_attribute( 'swiper', 'data-space-between-tablet', $settings['space_between_tablet']['size'] );
		}

		if ( array_key_exists( 'space_between_mobile', $settings ) ) {
			$this->add_render_attribute( 'swiper', 'data-space-between-mobile', $settings['space_between_mobile']['size'] );
		}

		if ( array_key_exists( 'centered_slides_tablet', $settings ) ) {
			$this->add_render_attribute( 'swiper', 'data-centered-slides-tablet', $settings['centered_slides_tablet'] ? 'true' : 'false' );
		}

		if ( array_key_exists( 'centered_slides_mobile', $settings ) ) {
			$this->add_render_attribute( 'swiper', 'data-centered-slides-mobile', $settings['centered_slides_mobile'] ? 'true' : 'false' );
		}

		if ( $settings['enable_autoplay'] ) {
			$this->add_render_attribute(
				'swiper',
				array(
					'data-autoplay-enabled' => 'true',
					'data-autoplay-delay'   => $settings['autoplay_delay']['size'],
				)
			);
		}

		if ( $settings['vertical_centered_slides'] ) {

			$this->add_render_attribute(
				'swiper',
				array(
					'class' => 'slider_vertical-centered',
				)
			);

		}

		$class_footer        = 'justify-content-between';
		$class_nav_arrows    = 'text-left';
		$class_dots_captions = 'text-center';
		$class_counter       = 'text-right';

		// at least one navigation item
		if ( $this->isSingleNavItem(
			array(
				$settings['enable_navigation'],
				$settings['enable_dots'],
				$settings['enable_counter'],
			)
		) || $this->isSingleNavItem(
			array(
				$settings['enable_navigation'],
				$settings['enable_captions'],
				$settings['enable_counter'],
			)
		)
		) {
			$class_footer        = 'justify-content-center';
			$class_nav_arrows    = 'text-center';
			$class_dots_captions = 'text-center';
			$class_counter       = 'text-center';
		}

		// at least one navigation item
		?>

		<div <?php $this->print_render_attribute_string( 'section' ); ?>>
			<div <?php $this->print_render_attribute_string( 'swiper' ); ?>>
				<div class="swiper-wrapper">
					<?php if ( ! empty( $settings['images_gallery'] ) ) : ?>
						<?php foreach ( $settings['images_gallery'] as $image ) : ?>
							<div class="swiper-slide slider__slide slider-images__slide">
								<div class="slider__slide-inner slider-images__inner">
									<?php
										arts_the_lazy_image(
											array(
												'id'    => $image['id'],
												'size'  => $settings['image_size'],
												'type'  => 'image',
												'class' => array(
													'wrapper' => false,
													'image'   => array( 'swiper-lazy' ),
												),
											)
										);
									?>
								</div>
							</div>
						<?php endforeach; ?>
					<?php endif; ?>
				</div>
			</div>
			<div class="container-fluid slider-images__footer">
				<div class="row align-items-center <?php echo esc_attr( $class_footer ); ?>">
					<?php if ( $settings ['enable_navigation'] ) : ?>
						<div class="col-auto col-md-4 text-left slider__footer-col order-md-1 order-1 <?php echo esc_attr( $class_nav_arrows ); ?>">
							<div class="slider__arrows">
								<div class="slider__arrow js-slider-images__prev"><i class="material-icons">keyboard_arrow_left</i></div>
								<div class="slider__arrows-divider"></div>
								<div class="slider__arrow js-slider-images__next"><i class="material-icons">keyboard_arrow_right</i></div>
							</div>
						</div>
						<!-- - slider nav arrows -->
					<?php endif; ?>
					<?php if ( $settings['enable_captions'] ) : ?>
						<div class="col-12 col-md-4 slider__footer-col order-md-2 order-0 <?php echo esc_attr( $class_dots_captions ); ?>">
							<div class="slider-images__captions">
								<div class="swiper swiper-container slider-images__slider-captions js-slider-images__captions">
									<div class="swiper-wrapper slider-images__slider-captions-wrapper">
										<?php foreach ( $settings['images_gallery'] as $image ) : ?>
											<?php $caption = wp_get_attachment_caption( $image['id'] ); ?>
											<div class="swiper-slide slider-images__slider-captions-slide">
												<div class="slider-images__caption"><?php echo $caption; ?></div>
											</div>
										<?php endforeach; ?>
									</div>
								</div>
							</div>
						</div>
						<!-- - slider captions -->
					<?php else : ?>
						<?php if ( $settings['enable_dots'] ) : ?> 
							<div class="col-auto col-md-4 text-center slider__footer-col d-none d-md-block order-md-3 order-2">
								<div class="slider__dots js-slider-images__dots">
									<div class="slider__dot slider__dot_active">
										<svg viewBox="0 0 152 152" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
											<g fill="none" fill-rule="evenodd">
												<g transform="translate(-134.000000, -98.000000)">
													<path class="circle" d="M135,174a75,75 0 1,0 150,0a75,75 0 1,0 -150,0"></path>
												</g>
											</g>
										</svg>
									</div>
									<div class="slider__dot">
										<svg viewBox="0 0 152 152" version="1.1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink">
											<g fill="none" fill-rule="evenodd">
												<g transform="translate(-134.000000, -98.000000)">
													<path class="circle" d="M135,174a75,75 0 1,0 150,0a75,75 0 1,0 -150,0"></path>
												</g>
											</g>
										</svg>
									</div>
								</div>
							</div>
						<!-- - slider nav dots -->
						<?php endif; ?>
					<?php endif; ?>
					<?php if ( $settings['enable_counter'] ) : ?>
						<div class="col-auto col-md-4 slider-images__footer-col order-md-3 order-2 <?php echo esc_attr( $class_counter ); ?>">
							<div class="slider__progress">
								<div class="swiper swiper-container slider__counter slider__counter_current js-slider-images__counter-current">
									<div class="swiper-wrapper"></div>
								</div>
								<div class="slider__counter-divider"></div>
								<div class="slider__counter slider__counter_total js-slider-images__counter-total">001</div>
							</div>
						</div>
						<!-- - slider counter -->
					<?php endif; ?>
				</div>
			</div>
		</div>

		<?php
	}

	private function isSingleNavItem( $arr = array() ) {

		if ( ! $arr ) {
			return false;
		}

		$values = array_count_values( $arr );

		if ( array_key_exists( 'yes', $values ) && $values['yes'] === 1 ) {
			return true;
		} else {
			return false;
		}

	}

}
