<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

use \Arts\Utilities\Utilities;

class Arts_Widget_Base extends Widget_Base {
	protected static $_instance;

	public static function instance() {
		if ( is_null( static::$_instance ) ) {
			static::$_instance = new static();
		}

		return static::$_instance;
	}

	public function get_name() {}

	public function get_title() {}

	public function get_icon() {
		return 'eicon-plug';
	}

	public function get_categories() {
		return array( 'cassio-static' );
	}

	protected static $_posts, $_post_type, $_data_static_fields = array();

	protected function is_dynamic_content(): bool {
		return false;
	}

	public function has_widget_inner_wrapper(): bool {
		return ! \Elementor\Plugin::$instance->experiments->is_feature_active( 'e_optimized_markup' );
	}

	/**
	 * WordPress DB query for the posts
	 *
	 * @return void
	 */
	protected static function _get_posts() {

		// get class name in lowercase
		$class_name = ( new \ReflectionClass( static::class ) )->getShortName();
		$class_name = strtolower( $class_name );

		// filter to change current widget post type
		$args = apply_filters(
			'arts/elementor/' . $class_name . '/query_args',
			array(
				'post_type'      => static::$_post_type,
				'posts_per_page' => -1,
			)
		);

		$posts      = array();
		$taxonomies = array();
		$counter    = 0;

		$loop = new \WP_Query( $args );

		if ( $loop->have_posts() ) {

			while ( $loop->have_posts() ) {

				$loop->the_post();

				$posts[ $counter ]['id']                 = get_the_ID();
				$posts[ $counter ]['title']              = get_the_title();
				$posts[ $counter ]['permalink']          = get_the_permalink();
				$posts[ $counter ]['image_id']           = get_post_thumbnail_id();
				$posts[ $counter ]['image_secondary_id'] = Utilities::acf_get_field( 'secondary_image', $posts[ $counter ]['id'] );
				$posts[ $counter ]['video']              = Utilities::acf_get_field( 'video', $posts[ $counter ]['id'] );
				$posts[ $counter ]['subheading']         = Utilities::acf_get_field( 'subheading', $posts[ $counter ]['id'] );
				$post_categories                         = get_the_terms( $posts[ $counter ]['id'], 'arts_portfolio_category' );
				$posts[ $counter ]['categories']         = $post_categories;
				$posts[ $counter ]['categories_names']   = array();

				if ( is_array( $post_categories ) ) {
					foreach ( $post_categories as $item ) {

						$arr = array(
							'slug' => $item->slug,
							'name' => $item->name,
						);

						array_push( $posts[ $counter ]['categories_names'], $item->name );

						// don't add the same item multiple times
						if ( ! in_array( $arr, $taxonomies ) ) {
							array_push( $taxonomies, $arr );
						}
					}
				}

				$counter++;

			}

			wp_reset_postdata();

		}

		static::$_posts = $posts;
	}

	/**
	 * Get all posts by a pre-set type
	 *
	 * @return array
	 */
	public function get_posts() {
		if ( is_null( static::$_posts ) ) {
			static::_get_posts();
		}

		return static::$_posts;
	}

	/**
	 * Filter out disabled posts
	 *
	 * @return array
	 */
	public function get_posts_to_display() {
		$posts    = $this->get_posts();
		$settings = $this->get_settings_for_display();

		// static data source
		if ( array_key_exists( 'posts_data_source', $settings ) && $settings['posts_data_source'] === 'static' ) {

			if ( is_array( $settings['posts_static'] ) && ! empty( $settings['posts_static'] ) ) {
				foreach ( $settings['posts_static'] as $key => $value ) {

					$settings['posts_static'][ $key ]['id'] = $key;

					if ( array_key_exists( 'permalink', $settings['posts_static'][ $key ] ) ) {
						$settings['posts_static'][ $key ]['permalink_is_external'] = array_key_exists( 'is_external', $settings['posts_static'][ $key ]['permalink'] ) && ! ! $settings['posts_static'][ $key ]['permalink']['is_external'];
						$settings['posts_static'][ $key ]['permalink_nofollow']    = array_key_exists( 'nofollow', $settings['posts_static'][ $key ]['permalink'] ) && ! ! $settings['posts_static'][ $key ]['permalink']['nofollow'];
						$settings['posts_static'][ $key ]['permalink']             = $settings['posts_static'][ $key ]['permalink']['url'];
					}

					if ( array_key_exists( 'image', $settings['posts_static'][ $key ] ) ) {
						$settings['posts_static'][ $key ]['image_id']           = $settings['posts_static'][ $key ]['image']['id'];
						$settings['posts_static'][ $key ]['image_secondary_id'] = null;
					}

					if ( array_key_exists( 'video', $settings['posts_static'][ $key ] ) ) {
						$settings['posts_static'][ $key ]['video'] = $settings['posts_static'][ $key ]['video']['url'];
					}

					if ( array_key_exists( 'categories_names', $settings['posts_static'][ $key ] ) ) {
						$category_name                                        = $settings['posts_static'][ $key ]['categories_names'];
						$settings['posts_static'][ $key ]['categories_names'] = array( $category_name );
						$settings['posts_static'][ $key ]['categories'][0]    = (object) array(
							'name' => $category_name,
							'slug' => sanitize_title( $category_name ),
						);
					}

					if ( array_key_exists( 'subheading', $settings['posts_static'][ $key ] ) ) {
						$settings['posts_static'][ $key ]['subheading'] = $settings['posts_static'][ $key ]['subheading'];
					}
				}
			}

			$posts = $settings['posts_static'];

		} else { // dynamic data source

			// limit posts amount
			if ( $settings['posts_amount']['size'] > 0 ) {
				array_splice( $posts, $settings['posts_amount']['size'] );
			}

			// only "enabled" posts
			$posts = array_filter(
				$posts,
				function( $item ) {
					$settings = $this->get_settings_for_display();
					return ( array_key_exists( 'enabled' . $item['id'], $settings ) ) && ( $settings[ 'enabled' . $item['id'] ] );
				}
			);

		}

		return $posts;
	}

	/**
	 * Collect the posts categories, remove duplicates
	 *
	 * @return array
	 */
	public function get_taxonomies_to_display( $posts ) {
		$active_taxonomies   = array();
		$hicpo_options       = get_option( 'hicpo_options' );
		$is_sortable         = false;
		$categories_taxonomy = 'arts_portfolio_category';

		if ( class_exists( 'Hicpo' ) && is_array( $hicpo_options ) && array_key_exists( 'tags', $hicpo_options ) && ! empty( $hicpo_options['tags'] ) ) {
			$hicpo_tags  = $hicpo_options['tags'];
			$is_sortable = in_array( $categories_taxonomy, $hicpo_tags );
		}

		foreach ( $posts as $item ) {
			if ( is_array( $item['categories'] ) ) {
				foreach ( $item['categories'] as $taxonomy ) {

					$arr = array(
						'slug' => $taxonomy->slug,
						'name' => $taxonomy->name,
					);

					// don't add the same item multiple times
					if ( ! in_array( $arr, $active_taxonomies ) ) {
						if ( $is_sortable ) {
							$active_taxonomies[ $taxonomy->term_order ] = $arr;
						} else {
							array_push( $active_taxonomies, $arr );
						}
					}
				}
			}
		}

		if ( $is_sortable ) {
			ksort( $active_taxonomies );
		}

		return $active_taxonomies;
	}

	/**
	 * Helper function: print HTML tag
	 *
	 * @param string $settings_tag_key
	 * @return void
	 */
	public function print_html_tag( $settings_tag_key ) {
		$html_tag = $this->get_settings( esc_attr( $settings_tag_key ) );

		// fallback
		if ( empty( $html_tag ) ) {
			$html_tag = 'div';
		}

		echo $html_tag;
	}

	/**
	 * Used for widgets with dynamically fetched posts
	 * Prints posts toggles set in the control panel
	 *
	 * @return void
	 */
	public function add_controls_posts_toggles() {
		$posts         = $this->get_posts();
		$post_type     = static::$_post_type;
		$post_type_obj = get_post_type_object( $post_type );

		$this->start_controls_section(
			'content_section',
			array(
				'label' => esc_html__( 'Posts', 'cassio' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		$this->add_controls_static_fields();

		if ( ! $posts ) {
			$this->end_controls_section();
			return;
		}

		$this->add_control(
			'dynamic_content_info',
			array(
				'type'            => Controls_Manager::RAW_HTML,
				'raw'             => sprintf(
					'%1$s <strong>%2$s.</strong> %3$s<br><br>%4$s <a href="%5$s" target="_blank">%6$s</a>',
					esc_html__( 'This widget displays content dynamically from the existing', 'cassio' ),
					$post_type_obj->labels->name,
					esc_html__( 'It\'s not editable directly through Elementor Page Builder.', 'cassio' ),
					esc_html__( 'You can edit or re-order your posts', 'cassio' ),
					admin_url( 'edit.php?post_type=' . $post_type ),
					esc_html__( 'in WordPress admin panel', 'cassio' )
				),
				'content_classes' => 'elementor-panel-alert elementor-panel-alert-warning',
				'condition'       => array(
					'posts_data_source' => 'dynamic',
				),
			)
		);

		$this->add_control(
			'posts_amount',
			array(
				'label'     => esc_html__( 'Posts to Display (0 for all)', 'cassio' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'number' => array(
						'min'  => 0,
						'max'  => 16,
						'step' => 1,
					),
				),
				'default'   => array(
					'unit' => 'number',
					'size' => 0,
				),
				'separator' => 'after',
				'condition' => array(
					'posts_data_source' => 'dynamic',
				),
			)
		);

		foreach ( $posts as $index => $item ) {
			/**
			 * Heading Toggle
			 */
			$id = 'heading_toggle' . $item['id'];
			$this->add_control(
				$id,
				array(
					'raw'        => sprintf(
						'<h3 class="elementor-control-title"><strong>%1$s</strong>&nbsp;&nbsp;<a href="%2$s" target="_blank"><i class="eicon-edit"></i></a></h3>',
						$item['title'],
						admin_url( 'post.php?post=' . $item['id'] . '&action=edit' ),
						esc_html__( 'Edit', 'cassio' )
					),
					'type'       => Controls_Manager::RAW_HTML,
					'separator'  => 'before',
					'conditions' => array(
						'relation' => 'and',
						'terms'    => array(
							array(
								'relation' => 'or',
								'terms'    => array(
									array(
										'name'     => 'posts_amount[size]',
										'operator' => '>',
										'value'    => $index,
									),
									array(
										'name'     => 'posts_amount[size]',
										'operator' => '<=',
										'value'    => '0',
									),
								),
							),
							array(
								'relation' => 'and',
								'terms'    => array(
									array(
										'name'     => 'posts_data_source',
										'operator' => '==',
										'value'    => 'dynamic',
									),
								),
							),
						),
					),
				)
			);

			/**
			 * Toggle
			 */
			$id = 'enabled' . $item['id'];
			$this->add_control(
				$id,
				array(
					'label'      => esc_html__( 'Enabled', 'cassio' ),
					'type'       => Controls_Manager::SWITCHER,
					'default'    => 'yes',
					'separator'  => 'after',
					'conditions' => array(
						'relation' => 'and',
						'terms'    => array(
							array(
								'relation' => 'or',
								'terms'    => array(
									array(
										'name'     => 'posts_amount[size]',
										'operator' => '>',
										'value'    => $index,
									),
									array(
										'name'     => 'posts_amount[size]',
										'operator' => '<=',
										'value'    => '0',
									),
								),
							),
							array(
								'relation' => 'and',
								'terms'    => array(
									array(
										'name'     => 'posts_data_source',
										'operator' => '==',
										'value'    => 'dynamic',
									),
								),
							),
						),
					),
				)
			);

		}

		$this->end_controls_section();
	}

	/**
	 * Adds controls to handle static data
	 *
	 * @return void
	 */
	public function add_controls_static_fields() {
		$static_fields = static::$_data_static_fields;

		if ( empty( $static_fields ) ) {

			// fallback data source type is "dynamic"
			$this->add_control(
				'posts_data_source',
				array(
					'label'   => esc_html__( 'Data Source Type', 'cassio' ),
					'type'    => Controls_Manager::HIDDEN,
					'default' => 'dynamic',
				)
			);

			return;
		}

		$this->add_control(
			'posts_data_source',
			array(
				'label'   => esc_html__( 'Data Source Type', 'cassio' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					'dynamic' => esc_html__( 'Dynamic', 'cassio' ),
					'static'  => esc_html__( 'Static', 'cassio' ),
				),
				'default' => 'dynamic',
			)
		);

		$repeater = new Repeater();

		if ( in_array( 'title', $static_fields ) ) {
			$repeater->add_control(
				'title',
				array(
					'label'   => esc_html__( 'Title', 'cassio' ),
					'type'    => Controls_Manager::TEXT,
					'default' => esc_html__( 'Item...', 'cassio' ),
				)
			);
		}

		if ( in_array( 'categories_names', $static_fields ) ) {
			$repeater->add_control(
				'categories_names',
				array(
					'label'   => esc_html__( 'Categories', 'cassio' ),
					'type'    => Controls_Manager::TEXT,
					'default' => '',
				)
			);
		}

		if ( in_array( 'subheading', $static_fields ) ) {
			$repeater->add_control(
				'subheading',
				array(
					'label'   => esc_html__( 'Subheading', 'cassio' ),
					'type'    => Controls_Manager::TEXTAREA,
					'default' => '',
				)
			);
		}

		if ( in_array( 'permalink', $static_fields ) ) {
			$repeater->add_control(
				'permalink',
				array(
					'label'         => esc_html__( 'Link', 'cassio' ),
					'type'          => Controls_Manager::URL,
					'placeholder'   => 'https://...',
					'show_external' => true,
					'default'       => array(
						'url'         => '#',
						'is_external' => false,
						'nofollow'    => false,
					),
				)
			);
		}

		if ( in_array( 'image', $static_fields ) ) {
			$repeater->add_control(
				'image',
				array(
					'label'   => esc_html__( 'Choose Image', 'cassio' ),
					'type'    => Controls_Manager::MEDIA,
					'default' => array(
						'url' => Utils::get_placeholder_image_src(),
					),
				)
			);
		}

		if ( in_array( 'video', $static_fields ) ) {
			$repeater->add_control(
				'video',
				array(
					'label'      => esc_html__( 'Choose Video', 'cassio' ),
					'type'       => Controls_Manager::MEDIA,
					'media_type' => 'video',
				)
			);
		}

		$this->add_control(
			'posts_static',
			array(
				'type'          => Controls_Manager::REPEATER,
				'fields'        => $repeater->get_controls(),
				'title_field'   => '{{{ title }}}',
				'prevent_empty' => false,
				'condition'     => array(
					'posts_data_source' => 'static',
				),
			)
		);
	}

	/**
	 * Select which image to display – primary or secondary
	 * If a secondary featured image is selected but is not set for a post
	 * then it will fallback to a primary featured image.
	 *
	 * @param array  $post
	 * @param string $image_type
	 * @param array  $args
	 * @return void
	 */
	public function get_priority_image_id_to_display( $post, $image_type = 'primary', $args = array() ) {
		if ( ! $post ) {
			return;
		}

		$image_id = '';
		$settings = $this->get_settings_for_display();
		$defaults = array(
			'image_size'         => 'full',
			'id_primary'         => 'image_id',
			'id_secondary'       => 'image_secondary_id',
			'id_video'           => 'video',
			'use_featured_video' => array_key_exists( 'enable_video', $settings ) ? $settings['enable_video'] : false,
		);

		$args = wp_parse_args( $args, $defaults );

		if ( $args['use_featured_video'] ) {

			$path = wp_get_attachment_image_src( Utilities::acf_get_field( $post[ $args['id_video'] ], $post['id'] ) );

			if ( $path ) {
				$image_id = $post[ $args['id_video'] ];
				return $image_id;
			}
		}

		if ( $image_type === 'primary' ) {

			$image_id = $post[ $args['id_primary'] ];
		} else {
			$path = wp_get_attachment_image_src( $post[ $args['id_secondary'] ], $args['image_size'] );
			if ( $path ) {
				$image_id = $post[ $args['id_secondary'] ];
			} else {
				$image_id = $post[ $args['id_primary'] ];
			}
		}

		return $image_id;
	}

	/**
	 * Translate the string values to the correct columns proportion value
	 */
	public function translate_columns_settings( $option ) {
		if ( ! $option ) {
			return 12;
		}

		if ( $option == '2dot4' ) {
			return 2.4;
		}

		return $option;
	}

	public function translate_columns_settings_inverse( $option ) {
		if ( ! $option ) {
			return 12;
		}

		if ( $option == 2.4 ) {
			return '2dot4';
		}

		return $option;
	}
}
