<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Cassio_Widget_Portfolio_Slider extends Arts_Widget_Base {
	protected static $_instance, $_posts, $_post_type = 'arts_portfolio_item';
	protected static $_data_static_fields             = array( 'title', 'permalink', 'categories_names', 'image' );

	public function get_name() {
		return 'cassio-widget-portfolio-slider';
	}

	public function get_title() {
		return esc_html__( 'Portfolio Slider', 'cassio' );
	}

	public function get_icon() {
		return 'eicon-sitemap';
	}

	public function get_categories() {
		return array( 'cassio-dynamic' );
	}

	protected function is_dynamic_content(): bool {
		return true;
	}

	public function get_script_depends() {
		return array( 'swiper' );
	}

	public function get_style_depends() {
		return array( 'swiper' );
	}

	protected function register_controls() {

		// posts toggles & posts amount
		$this->add_controls_posts_toggles();

		$this->start_controls_section(
			'settings_section',
			array(
				'label' => esc_html__( 'Elements', 'cassio' ),
				'tab'   => Controls_Manager::TAB_SETTINGS,
			)
		);

		$this->add_control(
			'enable_categories',
			array(
				'type'    => Controls_Manager::SWITCHER,
				'label'   => esc_html__( 'Show Categories', 'cassio' ),
				'default' => 'yes',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'slider_section',
			array(
				'label' => esc_html__( 'Slider', 'cassio' ),
				'tab'   => Controls_Manager::TAB_SETTINGS,
			)
		);

		$this->add_responsive_control(
			'slides_per_view',
			array(
				'label'              => esc_html__( 'Slides Per Screen', 'cassio' ),
				'type'               => Controls_Manager::SLIDER,
				'range'              => array(
					'number' => array(
						'min'  => 1,
						'max'  => 4,
						'step' => 1,
					),
				),
				'devices'            => array( 'desktop', 'tablet', 'mobile' ),
				'default'            => array(
					'size' => 3.5,
					'unit' => 'number',
				),
				'desktop_default'    => array(
					'size' => 3.5,
					'unit' => 'number',
				),
				'tablet_default'     => array(
					'size' => 1.33,
					'unit' => 'number',
				),
				'mobile_default'     => array(
					'size' => 1.33,
					'unit' => 'number',
				),
				'render_type'        => 'template',
				'frontend_available' => true,
			)
		);

		$this->add_responsive_control(
			'centered_slides',
			array(
				'label'              => esc_html__( 'Horizontaly Centered Slides', 'cassio' ),
				'type'               => Controls_Manager::SWITCHER,
				'default'            => true,
				'desktop_default'    => true,
				'tablet_default'     => true,
				'mobile_default'     => true,
				'render_type'        => 'template',
				'frontend_available' => true,
			)
		);

		$this->add_control(
			'vertical_centered_slides',
			array(
				'label'   => esc_html__( 'Vertically Centered Slides', 'cassio' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_responsive_control(
			'space_between',
			array(
				'label'              => esc_html__( 'Space Between Slides', 'cassio' ),
				'type'               => Controls_Manager::SLIDER,
				'range'              => array(
					'px' => array(
						'min'  => 0,
						'max'  => 160,
						'step' => 1,
					),
				),
				'devices'            => array( 'desktop', 'tablet', 'mobile' ),
				'default'            => array(
					'size' => 30,
					'unit' => 'px',
				),
				'desktop_default'    => array(
					'size' => 30,
					'unit' => 'px',
				),
				'tablet_default'     => array(
					'size' => 15,
					'unit' => 'px',
				),
				'mobile_default'     => array(
					'size' => 15,
					'unit' => 'px',
				),
				'render_type'        => 'template',
				'frontend_available' => true,
			)
		);

		$this->add_control(
			'enable_autoplay',
			array(
				'label'   => esc_html__( 'Autoplay', 'cassio' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'autoplay_delay',
			array(
				'label'     => esc_html__( 'Autoplay Delay (ms)', 'cassio' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'ms' => array(
						'min'  => 1000,
						'max'  => 60000,
						'step' => 100,
					),
				),
				'default'   => array(
					'unit' => 'ms',
					'size' => 6000,
				),
				'condition' => array(
					'enable_autoplay' => 'yes',
				),
			)
		);

		$this->add_control(
			'speed',
			array(
				'label'   => esc_html__( 'Speed', 'cassio' ),
				'type'    => Controls_Manager::SLIDER,
				'range'   => array(
					'ms' => array(
						'min'  => 100,
						'max'  => 10000,
						'step' => 100,
					),
				),
				'default' => array(
					'unit' => 'ms',
					'size' => 1200,
				),
			)
		);

		$this->add_control(
			'touch_ratio',
			array(
				'label'   => esc_html__( 'Touch Ratio', 'cassio' ),
				'type'    => Controls_Manager::SLIDER,
				'range'   => array(
					'ms' => array(
						'min'  => 1,
						'max'  => 4,
						'step' => 0.1,
					),
				),
				'default' => array(
					'unit' => 'px',
					'size' => 3,
				),
			)
		);

		$this->add_control(
			'heading_controls',
			array(
				'label'     => esc_html__( 'Controls', 'cassio' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'enable_counter',
			array(
				'label'   => esc_html__( 'Enable Counter', 'cassio' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'counter_zeros',
			array(
				'label'     => esc_html__( 'Counter Prefix', 'cassio' ),
				'type'      => Controls_Manager::SELECT,
				'default'   => 2,
				'options'   => array(
					0 => esc_html__( 'None', 'cassio' ),
					1 => esc_html__( '1 Zero', 'cassio' ),
					2 => esc_html__( '2 Zeros', 'cassio' ),
				),
				'condition' => array(
					'enable_counter' => 'yes',
				),
			)
		);

		$this->add_control(
			'enable_navigation',
			array(
				'label'   => esc_html__( 'Enable Navigation Arrows', 'cassio' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'layout_section',
			array(
				'label' => esc_html__( 'Layout', 'cassio' ),
				'tab'   => Controls_Manager::TAB_LAYOUT,
			)
		);

		$this->add_control(
			'heading_image',
			array(
				'label'     => esc_html__( 'Images', 'cassio' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
			)
		);

		$this->add_control(
			'image_type',
			array(
				'label'   => esc_html__( 'Priority Image', 'cassio' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'primary',
				'options' => array(
					'primary'   => esc_html__( 'Primary Featured Image', 'cassio' ),
					'secondary' => esc_html__( 'Secondary Featured Image', 'cassio' ),
				),
			)
		);

		$this->add_group_control(
			Group_Control_Image_Size::get_type(),
			array(
				'label'   => esc_html__( 'Thumbnail Size', 'cassio' ),
				'name'    => 'image', // Usage: `{name}_size` and `{name}_custom_dimension`, in this case `image_size` and `image_custom_dimension`.
				'default' => 'large',
			)
		);

		$this->add_control(
			'image_type_info',
			array(
				'type'            => Controls_Manager::RAW_HTML,
				'raw'             => sprintf(
					'%1$s<br><br>%2$s <a href="%3$s" target="_blank">%4$s</a>',
					esc_html__( 'If a secondary featured image is not set for a post then it will fallback to a primary featured image.', 'cassio' ),
					esc_html__( 'You can edit your posts and adjust the featured images', 'cassio' ),
					admin_url( 'edit.php?post_type=arts_portfolio_item' ),
					esc_html__( 'in WordPress admin panel', 'cassio' )
				),
				'content_classes' => 'elementor-panel-alert elementor-panel-alert-warning',
				'condition'       => array(
					'image_type' => 'secondary',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'typography_section',
			array(
				'label' => esc_html__( 'Typography', 'cassio' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'heading_tag',
			array(
				'label'   => esc_html__( 'Headings HTML Tag', 'cassio' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'h3',
				'options' => array(
					'h2' => esc_html( '<h2>' ),
					'h3' => esc_html( '<h3>' ),
					'h4' => esc_html( '<h4>' ),
					'h5' => esc_html( '<h5>' ),
					'h6' => esc_html( '<h6>' ),
				),
			)
		);

		$this->add_control(
			'heading_size',
			array(
				'label'   => esc_html__( 'Headings Size', 'cassio' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'h3',
				'options' => array(
					'h1'       => esc_html__( 'H1', 'cassio' ),
					'h2'       => esc_html__( 'H2', 'cassio' ),
					'h3'       => esc_html__( 'H3', 'cassio' ),
					'h4'       => esc_html__( 'H4', 'cassio' ),
					'h5'       => esc_html__( 'H5', 'cassio' ),
					'h6'       => esc_html__( 'H6', 'cassio' ),
					'text-xl'  => esc_html__( 'XL', 'cassio' ),
					'text-xxl' => esc_html__( 'XXL', 'cassio' ),
				),
			)
		);

		$this->add_control(
			'heading_weight',
			array(
				'label'   => esc_html__( 'Headings Weight', 'cassio' ),
				'type'    => Controls_Manager::SELECT,
				'default' => '',
				'options' => array(
					''              => esc_html__( 'Auto', 'cassio' ),
					'heading-light' => esc_html__( 'Normal', 'cassio' ),
					'heading-bold'  => esc_html__( 'Bold', 'cassio' ),
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'animation_section',
			array(
				'label' => esc_html__( 'Animation', 'cassio' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'transition_text_enabled',
			array(
				'label'   => esc_html__( 'Enable Text Transition', 'cassio' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->end_controls_section();
	}

	protected function render() {
		$settings = $this->get_settings_for_display();
		$posts    = $this->get_posts_to_display();

		$this->add_render_attribute(
			'section',
			array(
				'class' => array( 'section', 'section-portfolio' ),
			)
		);

		$this->add_render_attribute(
			'swiper',
			array(
				'class'                => array( 'swiper', 'swiper-container', 'slider', 'slider-projects', 'slider_draggable', 'js-slider-projects' ),
				'data-speed'           => $settings['speed']['size'],
				'data-slides-per-view' => $settings['slides_per_view']['size'],
				'data-space-between'   => $settings['space_between']['size'],
				'data-centered-slides' => $settings['centered_slides'],
				'data-touch-ratio'     => $settings['touch_ratio']['size'],
				'data-counter-zeros'   => $settings['counter_zeros'],
			)
		);

		if ( array_key_exists( 'slides_per_view_tablet', $settings ) ) {
			$this->add_render_attribute( 'swiper', 'data-slides-per-view-tablet', $settings['slides_per_view_tablet']['size'] );
		}

		if ( array_key_exists( 'slides_per_view_mobile', $settings ) ) {
			$this->add_render_attribute( 'swiper', 'data-slides-per-view-mobile', $settings['slides_per_view_mobile']['size'] );
		}

		if ( array_key_exists( 'space_between_tablet', $settings ) ) {
			$this->add_render_attribute( 'swiper', 'data-space-between-tablet', $settings['space_between_tablet']['size'] );
		}

		if ( array_key_exists( 'space_between_mobile', $settings ) ) {
			$this->add_render_attribute( 'swiper', 'data-space-between-mobile', $settings['space_between_mobile']['size'] );
		}

		if ( array_key_exists( 'centered_slides_tablet', $settings ) ) {
			$this->add_render_attribute( 'swiper', 'data-centered-slides-tablet', $settings['centered_slides_tablet'] ? 'true' : 'false' );
		}

		if ( array_key_exists( 'centered_slides_mobile', $settings ) ) {
			$this->add_render_attribute( 'swiper', 'data-centered-slides-mobile', $settings['centered_slides_mobile'] ? 'true' : 'false' );
		}

		$this->add_render_attribute(
			'heading',
			array(
				'class' => array( $settings['heading_size'], $settings['heading_weight'], 'figure-portfolio__heading' ),
			)
		);

		if ( $settings['transition_text_enabled'] ) {
			$this->add_render_attribute(
				'heading',
				array(
					'class'                => array( 'split-text', 'js-text-to-fly', 'js-split-text' ),
					'data-split-text-type' => 'lines, words, chars',
					'data-split-text-set'  => 'chars',
				)
			);
		}

		if ( $settings['enable_autoplay'] ) {
			$this->add_render_attribute(
				'swiper',
				array(
					'data-autoplay-enabled' => 'true',
					'data-autoplay-delay'   => $settings['autoplay_delay']['size'],
				)
			);
		}

		if ( $settings['vertical_centered_slides'] ) {

			$this->add_render_attribute(
				'swiper',
				array(
					'class' => 'slider_vertical-centered',
				)
			);

		}

		$class_footer     = 'justify-content-between';
		$class_nav_arrows = 'text-left';
		$class_counter    = 'text-right';

		if ( $this->is_single_nav_item(
			array(
				$settings['enable_navigation'],
				$settings['enable_counter'],
			)
		)
		) {
			$class_footer     = 'justify-content-center';
			$class_nav_arrows = 'text-center';
			$class_counter    = 'text-center';
		}

		?>

		<div <?php $this->print_render_attribute_string( 'section' ); ?>>
			<div <?php $this->print_render_attribute_string( 'swiper' ); ?>>
				<div class="swiper-wrapper">
					<?php foreach ( $posts as $index => $item ) : ?>
						<?php
							$categories = array();

							$this->add_render_attribute(
								'linkAtts' . $index,
								array(
									'class'          => array(
										'figure-portfolio',
										'figure-portfolio-item_hover',
									),
									'href'           => $item['permalink'],
									'data-pjax-link' => 'flyingHeading',
								)
							);

						if ( $settings['transition_text_enabled'] ) {
							$this->add_render_attribute(
								'linkAtts' . $index,
								'data-pjax-link',
								'flyingHeading'
							);
						}

							// categories
						if ( array_key_exists( 'categories', $item ) && is_array( $item['categories'] ) ) {
							foreach ( $item['categories'] as $taxonomy ) {
								$categories[] .= $taxonomy->name;
							}
						}

						if ( array_key_exists( 'permalink_is_external', $item ) && ! ! $item['permalink_is_external'] ) {
							$this->add_render_attribute( 'linkAtts' . $index, 'target', '_blank' );
						}

						if ( array_key_exists( 'permalink_nofollow', $item ) && ! ! $item['permalink_nofollow'] ) {
							$this->add_render_attribute( 'linkAtts' . $index, 'rel', 'nofollow' );
						}

							$categories_string = implode( '&nbsp;&nbsp;/&nbsp;&nbsp;', $categories );
						?>
						<div class="swiper-slide">
							<a <?php $this->print_render_attribute_string( 'linkAtts' . $index ); ?>>
								<div class="figure-portfolio__wrapper-img">
									<div class="figure-portfolio__wrapper-img-inner">
										<div class="figure-portfolio__wrapper-img-zoom">
											<?php
												arts_the_lazy_image(
													array(
														'id'   => $this->get_priority_image_id_to_display( $item, $settings['image_type'] ),
														'type' => 'image',
														'size' => $settings['image_size'],
														'class' => array(
															'wrapper' => array(),
															'image' => array( 'swiper-lazy' ),
														),
													)
												);
											?>
										</div>
									</div>
									<div class="figure-portfolio__content">
										<?php if ( $settings['enable_categories'] && ! empty( $categories_string ) ) : ?>
											<div class="figure-portfolio__category"><?php echo $categories_string; ?></div>
										<?php endif; ?>
										<?php if ( ! empty( $item['title'] ) ) : ?>
											<div class="figure-portfolio__header">
												<<?php echo $settings['heading_tag']; ?> <?php $this->print_render_attribute_string( 'heading' ); ?>><?php echo $item['title']; ?></<?php echo $settings['heading_tag']; ?>>
												<div class="figure-portfolio__icon material-icons">keyboard_arrow_right</div>
											</div>
										<?php endif; ?>
										<div class="figure-portfolio__curtain"></div>
									</div>
								</div>
							</a>
						</div>
					<?php endforeach; ?>
				</div>
			</div>
			<div class="container slider__footer-relative <?php echo $settings['enable_navigation'] || $settings['enable_counter'] ? '' : 'hidden_absolute'; ?>">
				<div class="row align-items-center <?php echo $class_footer; ?>">
					<?php if ( $settings['enable_navigation'] ) : ?>
						<div class="col-auto col-md-4 text-left slider__footer-col <?php echo esc_attr( $class_nav_arrows ); ?>">
							<div class="slider__arrows">
								<div class="slider__arrow js-slider-projects__prev">
									<i class="material-icons">keyboard_arrow_left</i>
								</div>
								<div class="slider__arrows-divider"></div>
								<div class="slider__arrow js-slider-projects__next">
									<i class="material-icons">keyboard_arrow_right</i>
								</div>
							</div>
						</div>
						<!-- - slider nav arrows -->
					<?php endif; ?>
					<div class="col-auto col-md-4 slider__footer-col <?php echo $settings['enable_counter'] ? '' : 'hidden_absolute'; ?> <?php echo esc_attr( $class_counter ); ?>">
						<div class="slider__progress">
							<div class="swiper swiper-container slider__counter slider__counter_current js-slider-projects__counter-current">
								<div class="swiper-wrapper"></div>
							</div>
							<div class="slider__counter-divider"></div>
							<div class="slider__counter slider__counter_total js-slider-projects__counter-total">001</div>
						</div>
					</div>
					<!-- - slider counter -->
				</div>
			</div>
		</div>

		<?php
	}

	private function is_single_nav_item( $arr = array() ) {

		if ( ! $arr ) {
			return false;
		}

		$values = array_count_values( $arr );

		if ( array_key_exists( 'yes', $values ) && $values['yes'] === 1 ) {
			return true;
		} else {
			return false;
		}
	}
}
