<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Cassio_Widget_Google_Map extends Arts_Widget_Base {
	protected static $_instance = null;

	public function get_name() {
		return 'cassio-widget-google-map';
	}

	public function get_title() {
		return esc_html__( 'Google Map', 'cassio' );
	}

	public function get_icon() {
		return 'eicon-plug';
	}

	public function get_categories() {
		return array( 'cassio-static' );
	}

	public function get_script_depends() {
		return array( 'googlemap' );
	}

	private function check_api_key() {
		$gmap_api_key = arts_get_gmap_key();

		return ! empty( $gmap_api_key );
	}

	public function wpml_widgets_to_translate_filter( $widgets ) {

		$name  = $this->get_name();
		$title = $this->get_title();

		$widgets[ $name ] = array(
			'conditions'        => array( 'widgetType' => $name ),
			'fields'            => array(),
			'integration-class' => 'WPML_Elementor_Cassio_Widget_Google_Map',
		);

		return $widgets;
	}

	public function add_wpml_support() {
		add_filter( 'wpml_elementor_widgets_to_translate', array( $this, 'wpml_widgets_to_translate_filter' ) );
	}

	protected function register_controls() {

		/**
		 * Section Markers
		 */
		$this->start_controls_section(
			'markers_section',
			array(
				'label' => esc_html__( 'Markers', 'cassio' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		$this->add_control(
			'is_key',
			array(
				'type'    => Controls_Manager::HIDDEN,
				'default' => $this->check_api_key(),
			)
		);

		$this->add_control(
			'message_api',
			array(
				'type'            => Controls_Manager::RAW_HTML,
				'raw'             => sprintf(
					'<strong>%1$s</strong><br>%2$s <strong><a href="%3$s" target="_blank" rel="nofollow">%4$s</a></strong> %5$s',
					esc_html__( 'Could not load Google Maps', 'cassio' ),
					esc_html__( 'Please ensure API Key is set in', 'cassio' ),
					admin_url( 'admin.php?page=elementor-settings#tab-integrations' ),
					esc_html__( 'Elementor -> Settings -> Integrations -> Google Maps Embed API', 'cassio' ),
					esc_html__( 'panel.', 'cassio' ),
				),
				'content_classes' => 'elementor-panel-alert elementor-panel-alert-warning',
				'condition'       => array(
					'is_key' => false,
				),
			)
		);

		$repeater = new Repeater();

		$repeater->add_control(
			'marker_content',
			array(
				'label'       => esc_html__( 'Content Box', 'cassio' ),
				'description' => esc_html__( 'That content will appear in a small window on marker click. You can place a helpful information here (e.g. company address).', 'cassio' ),
				'type'        => Controls_Manager::TEXTAREA,
				'default'     => '',
			)
		);

		$repeater->add_control(
			'marker_lat',
			array(
				'label'   => esc_html__( 'Latitude', 'cassio' ),
				'type'    => Controls_Manager::TEXT,
				'default' => '40.6700',
			)
		);

		$repeater->add_control(
			'marker_lon',
			array(
				'label'   => esc_html__( 'Longitude', 'cassio' ),
				'type'    => Controls_Manager::TEXT,
				'default' => '-73.9400',
			)
		);

		$repeater->add_control(
			'marker_icon',
			array(
				'label'       => esc_html__( 'Icon', 'cassio' ),
				'description' => esc_html__( 'Upload a PNG image of a nice map pin.', 'cassio' ),
				'type'        => Controls_Manager::MEDIA,
			)
		);

		$repeater->add_control(
			'marker_icon_size',
			array(
				'label'     => esc_html__( 'Image Size', 'cassio' ),
				'type'      => Controls_Manager::IMAGE_DIMENSIONS,
				'default'   => array(
					'width'  => '',
					'height' => '',
				),
				'condition' => array(
					'marker_icon!' => array(
						'id'  => '',
						'url' => '',
					),
				),
			)
		);

		$this->add_control(
			'markers',
			array(
				'type'          => Controls_Manager::REPEATER,
				'fields'        => $repeater->get_controls(),
				'title_field'   => '{{{ marker_content }}}',
				'prevent_empty' => true,
				'condition'     => array(
					'is_key' => true,
				),
			)
		);

		$this->end_controls_section();

		/**
		 * Section Style
		 */
		$this->start_controls_section(
			'style_section',
			array(
				'label' => esc_html__( 'Style', 'cassio' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'map_zoom',
			array(
				'label'       => esc_html__( 'Zoom Level', 'cassio' ),
				'description' => esc_html__( 'Applicable if there is only one marker on the map', 'cassio' ),
				'type'        => Controls_Manager::SLIDER,
				'range'       => array(
					'px' => array(
						'min' => 1,
						'max' => 20,
					),
				),
				'default'     => array(
					'unit' => 'px',
					'size' => 14,
				),
				'condition'   => array(
					'is_key' => true,
				),
			)
		);

		$this->add_responsive_control(
			'map_height',
			array(
				'label'           => esc_html__( 'Map Height', 'cassio' ),
				'type'            => Controls_Manager::SLIDER,
				'range'           => array(
					'px' => array(
						'min' => 0,
						'max' => 1440,
					),
					'vh' => array(
						'min' => 0,
						'max' => 100,
					),
				),
				'default'         => array(
					'size' => 900,
				),
				'desktop_default' => array(
					'size' => 900,
				),
				'tablet_default'  => array(
					'size' => 70,
					'unit' => 'vh',
				),
				'mobile_default'  => array(
					'size' => 50,
					'unit' => 'vh',
				),
				'size_units'      => array( 'px', 'vh' ),
				'selectors'       => array(
					'{{WRAPPER}} .gmap__container' => 'height: {{SIZE}}{{UNIT}};',
				),
			)
		);

		$this->add_control(
			'snazzy_styles',
			array(
				'label'       => esc_html__( 'Snazzy Maps Styles', 'cassio' ),
				'type'        => Controls_Manager::TEXTAREA,
				'description' => sprintf(
					'%1$s <a href="https://snazzymaps.com/explore" target="_blank">%2$s</a> %3$s',
					esc_html__( 'You can preview and generate custom map styles on', 'cassio' ),
					esc_html__( 'Snazzy Maps.', 'cassio' ),
					esc_html__( 'Just copy-paste JavaScript style array in this field.', 'cassio' )
				),
				'default'     => '[{"featureType":"all","elementType":"labels.text.fill","stylers":[{"saturation":36},{"color":"#000000"},{"lightness":40}]},{"featureType":"all","elementType":"labels.text.stroke","stylers":[{"visibility":"on"},{"color":"#000000"},{"lightness":16}]},{"featureType":"all","elementType":"labels.icon","stylers":[{"visibility":"off"}]},{"featureType":"administrative","elementType":"geometry.fill","stylers":[{"color":"#000000"},{"lightness":20}]},{"featureType":"administrative","elementType":"geometry.stroke","stylers":[{"color":"#000000"},{"lightness":17},{"weight":1.2}]},{"featureType":"landscape","elementType":"geometry","stylers":[{"color":"#000000"},{"lightness":20}]},{"featureType":"poi","elementType":"geometry","stylers":[{"color":"#000000"},{"lightness":21}]},{"featureType":"road.highway","elementType":"geometry.fill","stylers":[{"color":"#000000"},{"lightness":17}]},{"featureType":"road.highway","elementType":"geometry.stroke","stylers":[{"color":"#000000"},{"lightness":29},{"weight":0.2}]},{"featureType":"road.arterial","elementType":"geometry","stylers":[{"color":"#000000"},{"lightness":18}]},{"featureType":"road.local","elementType":"geometry","stylers":[{"color":"#000000"},{"lightness":16}]},{"featureType":"transit","elementType":"geometry","stylers":[{"color":"#000000"},{"lightness":19}]},{"featureType":"water","elementType":"geometry","stylers":[{"color":"#000000"},{"lightness":17}]}]',
			)
		);

		$this->end_controls_section();
	}

	protected function render() {

		$settings = $this->get_settings_for_display();

		$this->add_render_attribute(
			'gmap',
			array(
				'class'                   => 'gmap',
				'data-gmap-zoom'          => $settings['map_zoom'] ? $settings['map_zoom']['size'] : 1,
				'data-gmap-snazzy-styles' => preg_replace( '^\n|\r|\s+|\s+^', '', $settings['snazzy_styles'] ),
			)
		);

		?>

		<div class="section section-map">
			<div <?php $this->print_render_attribute_string( 'gmap' ); ?>>
				<div class="gmap__container"></div>
				<?php if ( $settings['markers'] ) : ?>
					<?php foreach ( $settings['markers'] as $index => $item ) : ?>
						<?php
							$rowKey = $this->get_repeater_setting_key( 'marker', 'markers', $index );
							$this->add_render_attribute(
								$rowKey,
								array(
									'class'               => 'gmap__marker d-none',
									'data-marker-lat'     => $item['marker_lat'],
									'data-marker-lon'     => $item['marker_lon'],
									'data-marker-content' => $item['marker_content'],
									'data-marker-img'     => $item['marker_icon']['url'],
									'data-marker-width'   => $item['marker_icon_size']['width'],
									'data-marker-height'  => $item['marker_icon_size']['height'],
								)
							);
						?>
						<div <?php $this->print_render_attribute_string( $rowKey ); ?>></div>
					<?php endforeach; ?>
				<?php endif; ?>
			</div>
		</div>

		<?php
	}
}
