<?php

namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Cassio_Widget_Hero extends Arts_Widget_Base {
	protected static $_instance = null;

	public function get_name() {
		return 'cassio-widget-hero';
	}

	public function get_title() {
		return esc_html__( 'Hero', 'cassio' );
	}

	public function get_icon() {
		return 'eicon-plug';
	}

	public function get_categories() {
		return array( 'cassio-static' );
	}

	public function wpml_widgets_to_translate_filter( $widgets ) {
		$name  = $this->get_name();
		$title = $this->get_title();

		$widgets[ $name ] = array(
			'conditions' => array( 'widgetType' => $name ),
			'fields'     => array(
				array(
					'field'       => 'heading',
					'type'        => sprintf( '<strong>%1$s</strong><br>%2$s', $title, esc_html__( 'Heading', 'cassio' ) ),
					'editor_type' => 'LINE',
				),
				array(
					'field'       => 'subheading',
					'type'        => sprintf( '<strong>%1$s</strong><br>%2$s', $title, esc_html__( 'Subheading', 'cassio' ) ),
					'editor_type' => 'LINE',
				),
				array(
					'field'       => 'text',
					'type'        => sprintf( '<strong>%1$s</strong><br>%2$s', $title, esc_html__( 'Text', 'cassio' ) ),
					'editor_type' => 'VISUAL',
				),
				array(
					'field'       => 'button_text',
					'type'        => sprintf( '<strong>%1$s</strong><br>%2$s', $title, esc_html__( 'Button Title', 'cassio' ) ),
					'editor_type' => 'LINE',
				),
				array(
					'field'       => 'button_link',
					'type'        => sprintf( '<strong>%1$s</strong><br>%2$s', $title, esc_html__( 'Button Link', 'cassio' ) ),
					'editor_type' => 'LINE',
				),
			),
		);

		return $widgets;
	}

	public function add_wpml_support() {
		add_filter( 'wpml_elementor_widgets_to_translate', array( $this, 'wpml_widgets_to_translate_filter' ) );
	}

	protected function register_controls() {

		$this->start_controls_section(
			'content_section',
			array(
				'label' => esc_html__( 'Content', 'cassio' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		$this->add_control(
			'heading',
			array(
				'label'       => esc_html__( 'Heading', 'cassio' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__( 'Heading...', 'cassio' ),
			)
		);

		$this->add_control(
			'subheading',
			array(
				'label'       => esc_html__( 'Subheading', 'cassio' ),
				'label_block' => true,
				'type'        => Controls_Manager::TEXT,
				'default'     => esc_html__( 'Subheading...', 'cassio' ),
			)
		);

		$this->add_control(
			'text',
			array(
				'label'       => esc_html__( 'Text', 'cassio' ),
				'type'        => Controls_Manager::WYSIWYG,
				'description' => sprintf(
					'%1s <ul><li><strong>%2s</strong></li><strong>%3s</strong></li></ul> %4s',
					esc_html__( 'To override the font weight, please use', 'cassio' ),
					esc_html( 'heading-light' ),
					esc_html( 'heading-bold' ),
					esc_html__( 'helper classes for the elements in editor "Text" tab' )
				),
				'default'     => esc_html__( 'Text...', 'cassio' ),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'button_section',
			array(
				'label' => esc_html__( 'Button', 'cassio' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		$this->add_control(
			'button_type',
			array(
				'label'   => esc_html__( 'Type', 'cassio' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'link',
				'options' => array(
					'button' => esc_html__( 'Button', 'cassio' ),
					'link'   => esc_html__( 'Link', 'cassio' ),
				),
			)
		);

		$this->add_control(
			'button_text',
			array(
				'label'   => esc_html__( 'Title', 'cassio' ),
				'type'    => Controls_Manager::TEXT,
				'default' => esc_html__( 'Button...', 'cassio' ),
			)
		);

		$this->add_control(
			'button_link',
			array(
				'label'         => esc_html__( 'Link', 'cassio' ),
				'type'          => Controls_Manager::URL,
				'placeholder'   => 'https://...',
				'show_external' => true,
				'default'       => array(
					'url'         => '#',
					'is_external' => true,
					'nofollow'    => true,
				),
			)
		);

		$this->add_control(
			'button_color',
			array(
				'label'       => esc_html__( 'Color', 'cassio' ),
				'label_block' => true,
				'type'        => Controls_Manager::SELECT,
				'options'     => array(
					'bg-white'              => esc_html__( 'White', 'cassio' ),
					'bg-off-white'          => esc_html__( 'Off white', 'cassio' ),
					'bg-light'              => esc_html__( 'Light', 'cassio' ),
					'bg-black'              => esc_html__( 'Black', 'cassio' ),
					'bg-dark'               => esc_html__( 'Dark', 'cassio' ),
					'bg-dark-2'             => esc_html__( 'Dark 2', 'cassio' ),
					'bg-dark-3'             => esc_html__( 'Dark 3', 'cassio' ),
					'bg-accent-primary-1'   => esc_html__( 'Accent Primary 1', 'cassio' ),
					'bg-accent-primary-2'   => esc_html__( 'Accent Primary 2', 'cassio' ),
					'bg-accent-primary-3'   => esc_html__( 'Accent Primary 3', 'cassio' ),
					'bg-accent-primary-4'   => esc_html__( 'Accent Primary 4', 'cassio' ),
					'bg-accent-secondary-1' => esc_html__( 'Accent Secondary 1', 'cassio' ),
					'bg-accent-secondary-2' => esc_html__( 'Accent Secondary 2', 'cassio' ),
					'bg-accent-secondary-3' => esc_html__( 'Accent Secondary 3', 'cassio' ),
					'bg-accent-secondary-4' => esc_html__( 'Accent Secondary 4', 'cassio' ),
				),
				'default'     => 'bg-accent-primary-3',
				'condition'   => array(
					'button_type' => 'button',
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'background_section',
			array(
				'label' => esc_html__( 'Background', 'cassio' ),
				'tab'   => Controls_Manager::TAB_CONTENT,
			)
		);

		$this->add_control(
			'background_type',
			array(
				'label'   => esc_html__( 'Type', 'cassio' ),
				'type'    => Controls_Manager::CHOOSE,
				'options' => array(
					''      => array(
						'title' => esc_html__( 'None', 'cassio' ),
						'icon'  => 'eicon-ban',
					),
					'image' => array(
						'title' => esc_html__( 'Image', 'cassio' ),
						'icon'  => 'eicon-image-bold',
					),
					'video' => array(
						'title' => esc_html__( 'Video', 'cassio' ),
						'icon'  => 'eicon-video-camera',
					),
				),
				'default' => 'image',
				'toggle'  => false,
			)
		);
		$this->add_control(
			'image',
			array(
				'label'     => esc_html__( 'Choose Image', 'cassio' ),
				'type'      => Controls_Manager::MEDIA,
				'default'   => array(
					'url' => Utils::get_placeholder_image_src(),
				),
				'condition' => array(
					'background_type' => 'image',
				),
			)
		);

		$this->add_control(
			'video',
			array(
				'label'      => esc_html__( 'Choose Video', 'cassio' ),
				'type'       => Controls_Manager::MEDIA,
				'media_type' => 'video',
				'condition'  => array(
					'background_type' => 'video',
				),
			)
		);

		$this->add_control(
			'overlay_background_color',
			array(
				'label'     => esc_html__( 'Overlay', 'cassio' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => 'rgba(0,0,0,.6)',
				'selectors' => array(
					'{{WRAPPER}} .section-image__overlay' => 'background-color: {{VALUE}};',
				),
				'condition' => array(
					'background_type!' => '',
				),
			)
		);

		$this->add_control(
			'heading_parallax',
			array(
				'label'     => esc_html__( 'Parallax', 'cassio' ),
				'type'      => Controls_Manager::HEADING,
				'separator' => 'before',
				'condition' => array(
					'background_type!' => '',
				),
			)
		);

		$this->add_control(
			'image_parallax',
			array(
				'label'     => esc_html__( 'Enable Parallax', 'cassio' ),
				'type'      => Controls_Manager::SWITCHER,
				'default'   => 'yes',
				'condition' => array(
					'background_type!' => '',
					'image!'           => array(
						'id'  => '',
						'url' => '',
					),
				),
			)
		);

		$this->add_control(
			'image_parallax_speed',
			array(
				'label'     => esc_html__( 'Parallax Speed', 'cassio' ),
				'type'      => Controls_Manager::SLIDER,
				'range'     => array(
					'factor' => array(
						'min'  => -0.5,
						'max'  => 0.5,
						'step' => 0.01,
					),
				),
				'default'   => array(
					'unit' => 'factor',
					'size' => 0.1,
				),
				'condition' => array(
					'background_type!' => '',
					'image_parallax'   => 'yes',
					'image!'           => array(
						'id'  => '',
						'url' => '',
					),
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'layout_section',
			array(
				'label' => esc_html__( 'Layout', 'cassio' ),
				'tab'   => Controls_Manager::TAB_LAYOUT,
			)
		);

		$this->add_control(
			'container',
			array(
				'label'   => esc_html__( 'Container', 'cassio' ),
				'type'    => Controls_Manager::SELECT,
				'options' => array(
					''                => esc_html__( 'None', 'cassio' ),
					'container'       => esc_html__( 'Boxed', 'cassio' ),
					'container-fluid' => esc_html__( 'Fullwidth', 'cassio' ),
				),
				'default' => 'container',
			)
		);

		$this->add_control(
			'layout',
			array(
				'label'   => esc_html__( 'Content Layout', 'cassio' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'solid',
				'options' => array(
					'columns' => esc_html__( '2 Columns', 'cassio' ),
					'stack'   => esc_html__( 'Stack', 'cassio' ),
				),
				'default' => 'columns',
			)
		);

		$this->add_control(
			'content_position',
			array(
				'label'     => esc_html__( 'Content Position', 'cassio' ),
				'type'      => Controls_Manager::CHOOSE,
				'options'   => array(
					'justify-content-start'  => array(
						'title' => esc_html__( 'Left', 'cassio' ),
						'icon'  => 'eicon-h-align-left',
					),
					'justify-content-center' => array(
						'title' => esc_html__( 'Center', 'cassio' ),
						'icon'  => 'eicon-h-align-center',
					),
					'justify-content-end'    => array(
						'title' => esc_html__( 'Right', 'cassio' ),
						'icon'  => 'eicon-h-align-right',
					),
				),
				'default'   => 'justify-content-start',
				'toggle'    => false,
				'condition' => array(
					'layout' => 'stack',
				),
			)
		);

		$this->add_control(
			'content_alignment',
			array(
				'label'   => esc_html__( 'Alignment', 'cassio' ),
				'type'    => Controls_Manager::CHOOSE,
				'options' => array(
					'text-left'   => array(
						'title' => esc_html__( 'Left', 'cassio' ),
						'icon'  => 'eicon-text-align-left',
					),
					'text-center' => array(
						'title' => esc_html__( 'Center', 'cassio' ),
						'icon'  => 'eicon-text-align-center',
					),
					'text-right'  => array(
						'title' => esc_html__( 'Right', 'cassio' ),
						'icon'  => 'eicon-text-align-right',
					),
				),
				'default' => 'text-left',
				'toggle'  => false,
			)
		);

		$this->add_control(
			'content_col',
			array(
				'label'   => esc_html__( 'Content Proportions', 'cassio' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'col-lg-8',
				'options' => array(
					'col-lg-4'  => esc_html__( '4 Columns (33%)', 'cassio' ),
					'col-lg-5'  => esc_html__( '5 Columns (41%)', 'cassio' ),
					'col-lg-6'  => esc_html__( '6 Columns (50%)', 'cassio' ),
					'col-lg-7'  => esc_html__( '7 Columns (58%)', 'cassio' ),
					'col-lg-8'  => esc_html__( '8 Columns (66%)', 'cassio' ),
					'col-lg-9'  => esc_html__( '9 Columns (75%)', 'cassio' ),
					'col-lg-10' => esc_html__( '10 Columns (83%)', 'cassio' ),
					'col-lg-11' => esc_html__( '11 Columns (92%)', 'cassio' ),
					'col-lg-12' => esc_html__( '12 Columns (100%)', 'cassio' ),
				),
			)
		);

		$this->add_control(
			'fit_to_screen',
			array(
				'label'   => esc_html__( 'Enable Fit to Screen', 'cassio' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->add_control(
			'enable_scroll_down',
			array(
				'type'    => \Elementor\Controls_Manager::SWITCHER,
				'label'   => esc_html__( 'Show Scroll Down', 'cassio' ),
				'default' => '',
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'typography_section',
			array(
				'label' => esc_html__( 'Typography', 'cassio' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'heading_tag',
			array(
				'label'   => esc_html__( 'Heading HTML Tag', 'cassio' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'h1',
				'options' => array(
					'h1' => esc_html( '<h1>' ),
					'h2' => esc_html( '<h2>' ),
					'h3' => esc_html( '<h3>' ),
					'h4' => esc_html( '<h4>' ),
					'h5' => esc_html( '<h5>' ),
					'h6' => esc_html( '<h6>' ),
					'p'  => esc_html( '<p>' ),
				),
			)
		);

		$this->add_control(
			'heading_size',
			array(
				'label'   => esc_html__( 'Heading Size', 'cassio' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'text-xl',
				'options' => array(
					'h1'       => esc_html__( 'H1', 'cassio' ),
					'h2'       => esc_html__( 'H2', 'cassio' ),
					'h3'       => esc_html__( 'H3', 'cassio' ),
					'h4'       => esc_html__( 'H4', 'cassio' ),
					'h5'       => esc_html__( 'H5', 'cassio' ),
					'h6'       => esc_html__( 'H6', 'cassio' ),
					'text-xl'  => esc_html__( 'XL', 'cassio' ),
					'text-xxl' => esc_html__( 'XXL', 'cassio' ),
				),
			)
		);

		$this->add_control(
			'heading_weight',
			array(
				'label'   => esc_html__( 'Heading Weight', 'cassio' ),
				'type'    => Controls_Manager::SELECT,
				'default' => '',
				'options' => array(
					''              => esc_html__( 'Auto', 'cassio' ),
					'heading-light' => esc_html__( 'Normal', 'cassio' ),
					'heading-bold'  => esc_html__( 'Bold', 'cassio' ),
				),
			)
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'animation_section',
			array(
				'label' => esc_html__( 'Animation', 'cassio' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			)
		);

		$this->add_control(
			'enable_animation',
			array(
				'label'   => esc_html__( 'Enable on-scroll animation', 'cassio' ),
				'type'    => Controls_Manager::SWITCHER,
				'default' => 'yes',
			)
		);

		$this->end_controls_section();

	}

	protected function render() {

		$settings = $this->get_settings_for_display();

		$this->add_inline_editing_attributes( 'heading' );
		$this->add_inline_editing_attributes( 'subheading' );
		$this->add_inline_editing_attributes( 'text' );
		$this->add_inline_editing_attributes( 'button_text' );

		$this->add_render_attribute(
			'section',
			array(
				'class' => array( 'section', 'section-intro', 'section-masthead', $settings['content_alignment'] ),
			)
		);

		$this->add_render_attribute(
			'inner',
			array(
				'class' => array( 'section-masthead__inner', 'section-intro__inner', $settings['container'] ),
			)
		);

		$this->add_render_attribute(
			'heading',
			array(
				'class' => array( 'section-masthead__heading', 'section-intro__heading', $settings['heading_size'], $settings['heading_weight'] ),
			)
		);

		$this->add_render_attribute(
			'subheading',
			array(
				'class' => array( 'section-masthead__subheading', 'section-intro__subheading', 'subheading', 'h6' ),
			)
		);

		$this->add_render_attribute(
			'text',
			array(
				'class' => array( 'section-masthead__text' ),
			)
		);

		$this->add_render_attribute(
			'row',
			array(
				'class' => array( 'row', 'section-intro__header', 'section-masthead__header', $settings['content_position'] ),
			)
		);

		$this->add_render_attribute(
			'col',
			array(
				'class' => $settings['content_col'],
			)
		);

		if ( $settings['fit_to_screen'] ) {

			$this->add_render_attribute(
				'section',
				array(
					'class' => array( 'section-fullheight' ),
				)
			);

			$this->add_render_attribute(
				'inner',
				array(
					'class' => array( 'section-fullheight__inner' ),
				)
			);

		}

		if ( $settings['button_type'] == 'button' ) {

			$this->add_render_attribute(
				'button',
				array(
					'class' => array( 'button', 'button_icon', $settings['button_color'] ),
					'href'  => $settings['button_link']['url'],
				)
			);

			$this->add_render_attribute(
				'button_label',
				array(
					'class' => 'button__label',
				)
			);

			$this->add_render_attribute(
				'button_icon',
				array(
					'class' => array( 'button__icon' ),
				)
			);

		} else {

			$this->add_render_attribute(
				'button',
				array(
					'class' => array( 'link-arrow' ),
					'href'  => $settings['button_link']['url'],
				)
			);

			$this->add_render_attribute(
				'button_label',
				array(
					'class' => array( 'link-arrow__label' ),
				)
			);

			$this->add_render_attribute(
				'button_icon',
				array(
					'class' => array( 'link-arrow__icon' ),
				)
			);

		}

		if ( $settings['button_link']['is_external'] ) {
			$this->add_render_attribute( 'button', 'target', '_blank' );
		}

		if ( $settings['button_link']['nofollow'] ) {
			$this->add_render_attribute( 'button', 'rel', 'nofollow' );
		}

		if ( $settings['enable_animation'] ) {

			$this->add_render_attribute( 'section', 'data-os-animation' );

			$this->add_render_attribute(
				'heading',
				array(
					'class'                => array( 'split-text', 'js-split-text' ),
					'data-split-text-type' => 'lines, words, chars',
					'data-split-text-set'  => 'chars',
				)
			);

			$this->add_render_attribute(
				'subheading',
				array(
					'class'                => array( 'split-text', 'js-split-text' ),
					'data-split-text-type' => 'lines, words, chars',
					'data-split-text-set'  => 'chars',
				)
			);

			$this->add_render_attribute(
				'text',
				array(
					'class'                => array( 'split-text', 'js-split-text' ),
					'data-split-text-type' => 'lines',
					'data-split-text-set'  => 'lines',
				)
			);

		}

		?>

		<div <?php $this->print_render_attribute_string( 'section' ); ?>>
			<div <?php $this->print_render_attribute_string( 'inner' ); ?>>
				<?php if ( ! empty( $settings['heading'] ) || ! empty( $settings['subheading'] ) ) : ?>
					<header <?php $this->print_render_attribute_string( 'row' ); ?>>
						<div <?php $this->print_render_attribute_string( 'col' ); ?>>
							<?php if ( ! empty( $settings['subheading'] ) ) : ?>
								<div <?php $this->print_render_attribute_string( 'subheading' ); ?>><?php echo $settings['subheading']; ?></div>
								<div class="w-100"></div>
							<?php endif; ?>
							<?php if ( ! empty( $settings['heading'] ) ) : ?>
								<<?php echo $settings['heading_tag']; ?> <?php $this->print_render_attribute_string( 'heading' ); ?>><?php echo $settings['heading']; ?></<?php echo $settings['heading_tag']; ?>>
							<?php endif; ?>
							<?php if ( ! empty( $settings['button_text'] && $settings['layout'] == 'columns' ) ) : ?>
								<div class="section-masthead__wrapper-button d-lg-block d-none">
									<a <?php $this->print_render_attribute_string( 'button' ); ?>>
										<div <?php $this->print_render_attribute_string( 'button_label' ); ?>>
											<span <?php $this->print_render_attribute_string( 'button_text' ); ?>><?php echo $settings['button_text']; ?></span>
										</div>
										<div <?php $this->print_render_attribute_string( 'button_icon' ); ?>><i class="material-icons">keyboard_arrow_right</i></div>
									</a>
								</div>
							<?php endif; ?>
						</div>
						<?php if ( $settings['layout'] == 'stack' ) : ?>
							<div class="w-100"></div>
						<?php endif; ?>
						<?php if ( ! empty( $settings['text'] ) ) : ?>
							<div <?php $this->print_render_attribute_string( 'col' ); ?>>
								<div <?php $this->print_render_attribute_string( 'text' ); ?>><?php echo $settings['text']; ?></div>
							</div>
						<?php endif; ?>
					</header>
				<?php endif; ?>
				<?php if ( ! empty( $settings['button_text'] ) ) : ?>
					<div class="section-masthead__wrapper-button <?php echo $settings['layout'] == 'columns' ? 'd-lg-none' : ''; ?>">
						<a <?php $this->print_render_attribute_string( 'button' ); ?>>
							<div <?php $this->print_render_attribute_string( 'button_label' ); ?>>
								<span <?php $this->print_render_attribute_string( 'button_text' ); ?>><?php echo $settings['button_text']; ?></span>
							</div>
							<div <?php $this->print_render_attribute_string( 'button_icon' ); ?>><i class="material-icons">keyboard_arrow_right</i></div>
						</a>
					</div>
				<?php endif; ?>
			</div>
			<?php if ( $settings['background_type'] ) : ?>
				<?php if ( $settings['background_type'] === 'video' ) : ?>
					<?php
						arts_the_lazy_image(
							array(
								'id'       => $settings['video']['id'],
								'type'     => 'background-video',
								'class'    => array(
									'wrapper' => array( 'section-masthead__background', 'section-masthead__background_fullscreen' ),
									'image'   => array( 'art-parallax__bg' ),
								),
								'parallax' => array(
									'wrapper' => true,
									'enabled' => $settings['image_parallax'],
									'factor'  => $settings['image_parallax_speed'] ? $settings['image_parallax_speed']['size'] : 0,
								),
							)
						);
					?>
				<?php else : ?>
					<?php
						arts_the_lazy_image(
							array(
								'id'       => $settings['image']['id'],
								'class'    => array(
									'wrapper' => array( 'section-masthead__background', 'section-masthead__background_fullscreen' ),
									'image'   => array( 'art-parallax__bg' ),
								),
								'parallax' => array(
									'wrapper' => true,
									'enabled' => $settings['image_parallax'],
									'factor'  => $settings['image_parallax_speed'] ? $settings['image_parallax_speed']['size'] : 0,
								),
							)
						);
					?>
				<?php endif; ?>
				<div class="overlay overlay_dark section-image__overlay"></div>
			<?php endif; ?>
			<?php if ( $settings['enable_scroll_down'] ) : ?>
				<div class="section-masthead__wrapper-scroll-down">
					<div class="scroll-down js-scroll-down">
						<div class="scroll-down__circle-1"></div>
						<div class="scroll-down__line"></div>
						<div class="scroll-down__circle-2"></div>
					</div>
				</div>
			<?php endif; ?>
		</div>

		<?php
	}
}
