<?php

namespace Arts\PluginsUpdater\Managers;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Abstract class BaseManager
 *
 * This class serves as a base for managers of the plugin.
 *
 * @package Arts\PluginsUpdater\Managers
 */
abstract class BaseManager {
	/**
	 * Arguments for the manager.
	 *
	 * @var array
	 */
	protected $args;

	/**
	 * Array of text strings used by the manager.
	 *
	 * @var array
	 */
	protected $strings;

	/**
	 * Other managers used by the current manager.
	 *
	 * @var \stdClass
	 */
	protected $managers;

	/**
	 * The theme ID associated with the plugin.
	 *
	 * @var string
	 * @example Example `my-theme-slug`
	 */
	protected $theme_id;

	/**
	 * The plugin ID, typically the plugin's directory and main file.
	 *
	 * @var string
	 * @example Example `my-plugin/my_plugin.php`
	 */
	protected $plugin_id;

	/**
	 * The endpoint URL for fetching remote plugin data.
	 *
	 * @var string
	 */
	protected $endpoint;

	/**
	 * The HTTP method to use when fetching remote plugin data.
	 *
	 * Must be either 'POST' or 'GET'.
	 */
	protected $method;

	/**
	 * The license key for the plugin.
	 *
	 * @var string
	 */
	protected $license_key;

	/**
	 * The license key status for the plugin.
	 *
	 * @var string
	 */
	protected $license_key_status;

	/**
	 * The timeout for remote requests in seconds.
	 *
	 * @var int
	 */
	protected $timeout;

	/**
	 * Request-level cache for remote plugin data.
	 *
	 * @var array
	 */
	protected $remote_data_cache = array();

	/**
	 * Constructor for the BaseManager class.
	 *
	 * @param array $args    Arguments for the manager.
	 * @param array $strings Array of text strings used by the manager.
	 */
	public function __construct( $theme_id, $args = array(), $strings = array() ) {
		$this->theme_id           = $theme_id;
		$this->license_key        = get_option( "{$this->theme_id}_license_key" );
		$this->license_key_status = get_option( "{$this->theme_id}_license_key_status" );
		$this->timeout            = isset( $args['timeout'] ) ? absint( $args['timeout'] ) : 10;
		$this->args               = $args;
		$this->strings            = $strings;
	}

	/**
	 * Initialize the manager with other managers.
	 *
	 * @param \stdClass $managers Other managers used by the current manager.
	 */
	public function init( $managers ) {
		$this->add_managers( $managers );
		$this->add_actions();
		$this->add_filters();
	}

	/**
	 * Add other managers to the current manager.
	 *
	 * @param \stdClass $managers Other managers used by the current manager.
	 *
	 * @return BaseManager Returns the current instance for method chaining.
	 */
	protected function add_managers( $managers ) {
		if ( ! isset( $this->managers ) ) {
			$this->managers = new \stdClass();
		}

		foreach ( $managers as $key => $manager ) {
			// Prevent adding self to the managers property to avoid infinite loop.
			if ( $manager !== $this ) {
				$this->managers->$key = $manager;
			}
		}

		return $this;
	}

	/**
	 * Add WordPress actions for the manager.
	 *
	 * This method can be overridden by subclasses to add specific actions.
	 *
	 * @return void
	 */
	protected function add_actions() {
		// Default implementation does nothing.
	}

	/**
	 * Add WordPress filters for the manager.
	 *
	 * This method can be overridden by subclasses to add specific filters.
	 *
	 * @return void
	 */
	protected function add_filters() {
		// Default implementation does nothing.
	}

	/**
	 * Fetches remote data from a specified endpoint using either POST or GET method.
	 *
	 * This method sends a request to the specified endpoint with the provided key and URL.
	 * It supports both POST and GET methods for the request.
	 *
	 * @param string $key The key to be sent in the request.
	 * @param string $endpoint The endpoint URL to send the request to.
	 * @param string $method Optional. The HTTP method to use for the request. Default 'POST'.
	 * @return object|WP_Error The response object on success, or WP_Error on failure.
	 */
	protected function fetch_remote_data( $key, $endpoint, $method = 'POST' ) {
		// Sanitize inputs
		$key      = sanitize_text_field( $key );
		$endpoint = esc_url_raw( $endpoint );

		// Validate method
		$method = strtoupper( $method );
		if ( ! in_array( $method, array( 'POST', 'GET' ) ) ) {
			$method = 'POST'; // Default to POST if invalid method provided
		}

		$body = array(
			'url' => esc_url( home_url( '/' ) ),
			'key' => rawurlencode( $key ),
		);

		if ( strtoupper( $method ) === 'GET' ) {
			$endpoint = add_query_arg( $body, $endpoint );
			$request  = wp_remote_get( $endpoint, array( 'timeout' => $this->timeout ) );
		} else {
			$args    = array(
				'timeout' => $this->timeout,
				'body'    => $body,
			);
			$request = wp_remote_post( $endpoint, $args );
		}

		if ( is_wp_error( $request ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'ArtsPluginsUpdater: Remote request failed: ' . $request->get_error_message() );
			}
			return $request;
		}

		if ( wp_remote_retrieve_response_code( $request ) === 200 ) {
			$response = json_decode( wp_remote_retrieve_body( $request ), true );

			if ( json_last_error() === JSON_ERROR_NONE ) {
				return (object) $response;
			} else {
				return new \WP_Error( 'json_decode_error', $this->strings['error-json-decode'] );
			}
		}

		return new \WP_Error( 'fetch_remote_data_failed', $this->strings['error-fetch'], $request );
	}

	/**
	 * Fetches remote data with request-level caching.
	 *
	 * This method wraps fetch_remote_data() with caching to prevent redundant API calls
	 * within the same request. The cache is stored in memory and cleared on each new request.
	 *
	 * @return object|WP_Error The cached or freshly fetched response object.
	 */
	protected function get_cached_remote_data() {
		if ( ! isset( $this->remote_data_cache[ $this->plugin_id ] ) ) {
			$this->remote_data_cache[ $this->plugin_id ] = $this->fetch_remote_data(
				$this->license_key,
				$this->endpoint,
				$this->method
			);
		}

		return $this->remote_data_cache[ $this->plugin_id ];
	}
}
