<?php

namespace Arts\PluginsUpdater\Managers;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Class BundledPlugin
 *
 * Manages the updates for theme bundled plugin.
 *
 * @package Arts\PluginsUpdater\Managers
 */
class BundledPlugin extends BaseManager {

	public function init( $managers ) {
		$this->plugin_id = $this->args['plugin_id'];
		$this->endpoint  = $this->args['endpoint'];
		$this->method    = $this->args['method'];

		$this->add_managers( $managers );
		$this->add_actions();
		$this->add_filters();
	}

	protected function add_actions() {
		// Remove the update message for the bundled plugin if the theme license key is valid.
		if ( $this->license_key && $this->license_key_status === 'valid' ) {
			add_action( 'admin_init', array( $this, 'remove_update_message' ) );
		}

		return $this;
	}

	protected function add_filters() {
		// Modify the plugins transient to include the download URL for the bundled plugin.
		// Hook on both save (pre_set) and retrieve (site_transient) to ensure package URL persists.
		add_filter( 'pre_set_site_transient_update_plugins', array( $this, 'modify_plugins_transient' ), 99, 1 );

		// Always hook on site_transient - we'll check if we should process inside the method
		add_filter( 'site_transient_update_plugins', array( $this, 'modify_plugins_transient' ), 99, 1 );

		return $this;
	}

	/**
	 * Check if we should process the update.
	 *
	 * @return bool
	 */
	protected function should_process_update() {
		if ( ! is_admin() ) {
			return false;
		}

		global $pagenow;

		if ( $pagenow && in_array( $pagenow, array( 'plugins.php', 'update-core.php' ) ) ) {
			return true;
		}

		// Only process AJAX requests for plugin updates
		if ( wp_doing_ajax() ) {
			$action = isset( $_REQUEST['action'] ) ? sanitize_text_field( $_REQUEST['action'] ) : '';
			return $action === 'update-plugin';
		}

		return false;
	}

	/**
	 * Modifies the plugins transient to include update data for the bundled plugin.
	 *
	 * @param object $transient The plugins transient object.
	 * @return object The modified plugins transient object.
	 */
	public function modify_plugins_transient( $transient ) {
		if ( ! is_object( $transient ) || ! isset( $transient->response ) || ! is_array( $transient->response ) ) {
			return $transient;
		}

		if ( empty( $this->license_key ) || $this->license_key_status !== 'valid' ) {
			return $transient;
		}

		if ( ! $this->should_process_update() ) {
			return $transient;
		}

		// Get cached remote data to prevent redundant API calls within the same request
		$remote_plugin_data = $this->get_cached_remote_data();

		// If we got valid remote data with a download URL
		if ( ! is_wp_error( $remote_plugin_data ) && ! empty( $remote_plugin_data->download_url ) ) {
			// Get installed version from transient
			$installed_version = isset( $transient->checked[ $this->plugin_id ] ) ? $transient->checked[ $this->plugin_id ] : '0';

			// Only add to response if remote version is newer than installed version
			if ( version_compare( $remote_plugin_data->version, $installed_version, '>' ) ) {
				// Check if package URL is already correctly set
				$current_package = isset( $transient->response[ $this->plugin_id ]->package ) ? $transient->response[ $this->plugin_id ]->package : '';

				if ( $current_package === $remote_plugin_data->download_url ) {
					return $transient;
				}

				// Update server returns complete plugin data - use it directly
				$remote_plugin_data->new_version = $remote_plugin_data->version;
				$remote_plugin_data->package     = $remote_plugin_data->download_url;

				// Set this plugin in the response
				$transient->response[ $this->plugin_id ] = $remote_plugin_data;

				$this->remove_update_message();
			} else {
				// Remove from response if versions match or installed is newer
				if ( isset( $transient->response[ $this->plugin_id ] ) ) {
					unset( $transient->response[ $this->plugin_id ] );
				}
			}
		}

		return $transient;
	}

	/**
	 * Removes all update messages for the specified plugin.
	 *
	 * @return void
	 */
	public function remove_update_message() {
		remove_all_actions( 'in_plugin_update_message-' . $this->plugin_id );
	}
}
