<?php
/**
 *
 * @package Cocoon
 *
 * @since 1.0
 *
 * ========================
 * THEME MAIN FUNCTIONS
 * ========================
 **/

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/*
=====================================================
COCOON THEME SETUP
=====================================================
*/

function cocoon_setup() {

	/* Set the content width */
	$GLOBALS['content_width'] = apply_filters( 'cocoon_content_width', 1200 );
	if ( ! isset( $content_width ) ) {
		$content_width = 1200;
	}

	/* Enable Support for Post Thumbnails */
	add_theme_support( 'post-thumbnails' );
	set_post_thumbnail_size( 840, 350, true ); // size of thumbnails.
	add_image_size( 'cocoon-blog', 1000, 563, true );

	/* Enable Support for Post Formats */
	add_theme_support( 'post-formats', [ 'audio', 'image', 'gallery', 'quote', 'video' ] );

	/* Change default markup to output valid HTML5  */
	add_theme_support( 'html5', [ 'search-form', 'comment-form', 'comment-list', 'gallery', 'caption' ] );

	/* Add default posts and comments RSS feed links to head. */
	add_theme_support( 'automatic-feed-links' );

	/*
	 * Let WordPress manage the document title.
	 * By adding theme support, we declare that this theme does not use a
	 * hard-coded <title> tag in the document head, and expect WordPress to
	 * provide it for us.
	 */
	add_theme_support( 'title-tag' );

	/* Enable Support for WooCommerce */
	add_theme_support( 'woocommerce' );

	/* Support images for Gutenberg */
	add_theme_support( 'align-wide' );

	/* === REGISTER MENUS === */
	register_nav_menus(
		[
			'primary' => esc_html__( 'Primary Menu', 'cocoon' ),
		]
	);
}

add_action( 'after_setup_theme', 'cocoon_setup' );

/**
 * Load Theme Textdomain
 *
 * @since 1.4.1
 */
function cocoon_load_translations() {
	load_theme_textdomain( 'cocoon', get_template_directory() . '/lang' );
}

add_action( 'init', 'cocoon_load_translations' );

/*
=====================================================
CHECK IF WC IS ACTIVE FUNCTION
=====================================================
*/

if ( ! function_exists( 'cocoon_is_woocommerce_activated' ) ) {
	function cocoon_wc_is_activated() {
		return class_exists( 'woocommerce' ) ? true : false;
	}
}

/**
 * Customize Kirki configuration support for PHP 8.2+.
 *
 * @since 1.4.1
 *
 * @param array $config Kirki config array.
 */
function cocoon_kirki_config( $config ) {
	if ( isset( $config['compiler'] ) ) {
		unset( $config['compiler'] );
	}
	return $config;
}

add_filter( 'kirki/config', 'cocoon_kirki_config', 999 );

/*
=====================================================
COCOON CORE FUNCTION
=====================================================
*/

function cocoon_functions_config( $Cocoon_Core ) {
	$Cocoon_Core->offsetSet( 'themeName', 'Cocoon' );
	$Cocoon_Core->offsetSet( 'themePath', get_stylesheet_directory() );
	$Cocoon_Core->offsetSet( 'themeUri', get_stylesheet_directory_uri() );
	$Cocoon_Core['Cocoon_AdminMenu'] = new Cocoon_AdminMenu( $Cocoon_Core );
	$Cocoon_Core['Cocoon_Plugins']   = new Cocoon_Plugins( $Cocoon_Core );
	$Cocoon_Core['Cocoon_Redirect']  = new Cocoon_Redirect( $Cocoon_Core );
}

add_filter( 'cocoon/config', 'cocoon_functions_config', 10, 1 );

function cocoon_functions_core() {
	if ( ! class_exists( 'Cocoon_Core' ) ) {
		$Cocoon_Core = [];
		global $Cocoon_Core;
		$Cocoon_Core['themeName'] = 'cocoon';
		$Cocoon_Core['themePath'] = get_stylesheet_directory();
		$start_menu               = new Cocoon_AdminMenu( $Cocoon_Core );
		$redirect                 = new Cocoon_Redirect( $Cocoon_Core );
		$start_menu->run();
		$redirect->run();

		if ( function_exists( 'cocoon_stat_display' ) ) {
			$default_plugins = new Cocoon_Plugins( $Cocoon_Core );
			$default_plugins->run();
		}
	}
}

add_action( 'init', 'cocoon_functions_core', 10, 1 );

/*
=====================================================
MAIN MENU FALLBACK
=====================================================
*/

function cocoon_menu_fallback( $output ) {
	if ( current_user_can( 'administrator' ) ) {
		echo( '
        <ul class="navbar-nav mx-auto" id="main-menu">
            <li class="nav-item"><a href="' . esc_url( admin_url( 'nav-menus.php' ) ) . '">' . esc_html__( 'Add a Menu', 'cocoon' ) . '</a></li>
        </ul>' );
	} else {
		echo( '
        <ul class="navbar-nav mx-auto" id="main-menu">
            <li class="nav-item"></li>
        </ul>' );
	}
}

/*
=====================================================
	PAGING NAVIGATION
=====================================================
*/

/**
 * Navigation function for pagination.
 *
 * @since  1.0
 */
if ( ! function_exists( 'cocoon_paging_nav' ) ) {
	function cocoon_paging_nav() {
		cocoon_numeric_pagination();
	}
}

/**
 * Numeric Pagination
 *
 * @since  1.0.0
 */
if ( ! function_exists( 'cocoon_numeric_pagination' ) ) {
	function cocoon_numeric_pagination() {
		global $wp_query;

		if ( $wp_query->max_num_pages < 2 ) {
			return;
		}

		?>
		<div class="col-md-12 col-12">
			<nav class="pagination">
				<?php
				$big  = 999999999;
				$args = [
					'base'      => str_replace( $big, '%#%', esc_url( get_pagenum_link( $big ) ) ),
					'total'     => $wp_query->max_num_pages,
					'current'   => max( 1, get_query_var( 'paged' ) ),
					'prev_text' => esc_html__( 'Previous', 'cocoon' ),
					'next_text' => esc_html__( 'Next', 'cocoon' ),
					'type'      => 'list',
				];

				echo paginate_links( $args );
				?>
			</nav>
		</div>
		<?php
	}
}

/*
=====================================================
	BLOG POST THUMBNAIL
=====================================================
*/

/**
 * Post Thumbnail
 *
 * @since  1.2.0
 */

if ( ! function_exists( 'cocoon_post_thumb' ) ) {
	function cocoon_post_thumb( $args = [] ) {
		global $post;

		$defaults = [
			'size'  => 'large',
			'class' => 'post-image',
		];

		$args        = wp_parse_args( $args, $defaults );
		$post_format = get_post_format();

		if ( cocoon_get_option( 'cocoon_lazyload' ) ) {
			cocoon_add_lazy_load_attrs();
		}

		// Standard or Image Post.
		if ( $post_format === false || $post_format == 'standard' || $post_format == 'image' ) {
			if ( has_post_thumbnail() ) {
				?>
			<!-- Blog Post Thumbnail -->
				<div class="blog-thumbnail mb30">

					<a href="<?php the_permalink(); ?>" title="<?php the_title(); ?>">
						<?php the_post_thumbnail(); ?>
					</a>

				</div>
				<?php
			}
		}

		// Gallery Post.
		if ( $post_format == 'gallery' ) {
			if ( function_exists( 'cocoon_plugin_functions' ) ) {

				$images = rwmb_meta( 'cocoon_blog_gallery', 'type=image_advanced&size=blog' );

				if ( ! empty( $images ) ) {
					?>
					<!-- Blog Post Gallery Thumbnail -->
					<div class="gallery-post-wrapper">
						<div class="owl-carousel gallery-post mb30">

							<?php
							foreach ( $images as $image ) {
								echo '<div class="item"><img src="' . esc_url( $image['url'] ) . '" alt="' . esc_attr( $image['alt'] ) . '" width="' . esc_attr( $image['width'] ) . '" height="' . esc_attr( $image['height'] ) . '"/></div>';
							}
							?>

						</div>
					</div>
					<?php
				}
			}
		}

		// Video Post.
		if ( $post_format == 'video' ) {
			$video = get_post_meta( $post->ID, 'cocoon_blog_video_embed', true );
			if ( $video !== '' ) {
				?>
				<!-- Embed Video -->
				<div class="embed-responsive embed-responsive-16by9 mb30">
					<?php
					if ( wp_oembed_get( $video ) ) {
						echo wp_oembed_get( $video );
					} else {
						$allowed_tags = wp_kses_allowed_html( 'post' );
						echo wp_kses( $video, $allowed_tags );
					}
					?>
				</div>
				<?php
			}
		}

		// Audio Post.
		if ( $post_format == 'audio' ) {
			$audio = get_post_meta( $post->ID, 'cocoon_blog_audio', true );
			if ( $audio !== '' ) {
				?>
				<!-- Embed Audio -->
				<div class="audio-wrapper mb30">
					<?php
					if ( wp_oembed_get( $audio ) ) {
						echo wp_oembed_get( $audio );
					} else {
						$allowed_tags = wp_kses_allowed_html( 'post' );
						echo wp_kses( $audio, $allowed_tags );
					}
					?>
				</div>
				<?php
			}
		}

		if ( cocoon_get_option( 'cocoon_lazyload' ) ) {
			cocoon_remove_lazy_load_attrs();
		}

	}
}

/**
 * Single Post Thumbnail
 *
 * @since  1.2.0
 */
if ( ! function_exists( 'cocoon_single_post_thumb' ) ) {
	function cocoon_single_post_thumb() {
		global $post;

		$post_format = get_post_format();

		if ( cocoon_get_option( 'cocoon_lazyload' ) ) {
			cocoon_add_lazy_load_attrs();
		}

		// Standard or Image Post.
		if ( $post_format == false || $post_format == 'image' ) {
			if ( has_post_thumbnail() ) {
				?>
				<!-- Blog Post Thumbnail -->
				<div class="blog-thumbnail">
					<?php the_post_thumbnail(); ?>
				</div>
				<?php
			}
		}

		// Gallery Post.
		if ( $post_format == 'gallery' ) {
			if ( function_exists( 'cocoon_plugin_functions' ) ) {
				?>
				<div class="owl-carousel gallery-post mb30">
					<?php
					$images = rwmb_meta( 'cocoon_blog_gallery', 'type=image_advanced&size=blog' );

					foreach ( $images as $image ) {
						echo '<div class="item"><img src="' . esc_url( $image['url'] ) . '" alt="' . esc_attr( $image['alt'] ) . '" width="' . esc_attr( $image['width'] ) . '" height="' . esc_attr( $image['height'] ) . '"/></div>';
					}
					?>
				</div>
				<?php
			}
		}

		// Quote Post.
		if ( $post_format == 'quote' ) {
			$quote_content = get_post_meta( $post->ID, 'cocoon_blog_quote_content', true );
			$quote_author  = get_post_meta( $post->ID, 'cocoon_blog_quote_author', true );
			$quote_source  = get_post_meta( $post->ID, 'cocoon_blog_quote_source', true );
			$allowed_tags  = wp_kses_allowed_html( 'post' );

			if ( ! empty( $quote_content ) && ! empty( $quote_author ) ) {
				?>
				<!-- Blog Post Quote -->
				<figure class="post-quote mb30">
					<span class="icon"></span>
					<blockquote>

						<h4><?php echo esc_html( $quote_content ); ?></h4>

						<?php if ( ! empty( $quote_source ) ) { ?>
							<a href="<?php echo esc_url( $quote_source ); ?>">
						<?php } ?>
								<h6 class="pt20">
								<?php
								echo esc_html( '- ' );
								echo wp_kses( $quote_author, $allowed_tags );
								?>
								</h6>
						<?php if ( ! empty( $quote_source ) ) { ?>
							</a> 
						<?php } ?>

					</blockquote>
				</figure>
				<?php
			}
		}

		// Audio Post.
		if ( $post_format == 'audio' ) {
			$audio = get_post_meta( $post->ID, 'cocoon_blog_audio', true );
			if ( ! empty( $audio ) ) {
				?>

				<!-- Embed Audio -->
				<div class="audio-wrapper mb30">
					<?php
					if ( wp_oembed_get( $audio ) ) {
							echo wp_oembed_get( $audio );
					} else {
						$allowed_tags = wp_kses_allowed_html( 'post' );
						echo wp_kses( $audio, $allowed_tags );
					}
					?>
				</div>
				<?php
			}
		}

		// Video Post.
		if ( $post_format == 'video' ) {
			$video_embed = get_post_meta( $post->ID, 'cocoon_blog_video_embed', true );
			if ( ! empty( $video_embed ) ) {
				?>

				<!-- Embed Video -->
				<div class="embed-responsive embed-responsive-16by9 mb30">
					<?php
					if ( wp_oembed_get( $video_embed ) ) {
						echo wp_oembed_get( $video_embed );
					} else {
						$allowed_tags = wp_kses_allowed_html( 'post' );
						echo wp_kses( $video_embed, $allowed_tags );
					}
					?>
				</div>

				<?php
			}
		}

		if ( cocoon_get_option( 'cocoon_lazyload' ) ) {
			cocoon_remove_lazy_load_attrs();
		}

	}
}

/*
=====================================================
	BLOG LOOP CUSTOM FUNCTION
=====================================================
*/

/**
 *  Meta info for blog posts
 *
 * @since  1.0
 */
function cocoon_posted_meta() {
	echo '<div class="entry-meta">';

	if ( is_single() ) {
		$metas = cocoon_get_option( 'cocoon_single_blog_meta' );

		if ( in_array( 'author', $metas ) ) {
			echo '<span class="author"><i class="fa fa-keyboard-o"></i>';
			echo esc_html__( 'By', 'cocoon' ) . ' <a class="author-link" rel="author" href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">';
			the_author_meta( 'display_name' );
			echo '</a>';
			echo '</span>';
		}
		if ( in_array( 'date', $metas ) ) {
			$time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
			if ( get_the_time( 'U' ) !== get_the_modified_time( 'U' ) ) {
				$time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time>';
			}

			$time_string = sprintf(
				$time_string,
				esc_attr( get_the_date( 'c' ) ),
				esc_html( get_the_date() ),
				esc_attr( get_the_modified_date( 'c' ) ),
				esc_html( get_the_modified_date() )
			);

			echo '<span class="published"><i class="fa fa-clock-o"></i>' . $time_string . '</span>';

		}
		if ( in_array( 'cat', $metas, true ) ) {
			if ( has_category() ) {
				echo '<span class="category"><i class="fa fa-folder-open-o"></i>';
				the_category( ', ' );
				echo '</span>';
			}
		}
		if ( in_array( 'tags', $metas, true ) ) {
			if ( has_tag() ) {
				echo '<span class="tags"><i class="fa fa-tags"></i>';
				the_tags( '', ', ' );
				echo '</span>';
			}
		}
		if ( in_array( 'com', $metas, true ) ) {
			echo '<span class="comments"><i class="fa fa-comment-o"></i>';
			comments_popup_link(
				esc_html__( '0 comments', 'cocoon' ),
				esc_html__( '1 comment', 'cocoon' ),
				esc_html__( '% comments', 'cocoon' ),
				'comments-link',
				esc_html__( 'Comments are off', 'cocoon' )
			);
			echo '</span>';
		}
	} else {
		$metas = cocoon_get_option( 'cocoon_blog_meta' );

		if ( in_array( 'author', $metas, true ) ) {
			echo '<span class="author"><i class="fa fa-keyboard-o"></i>';
			echo esc_html__( 'By', 'cocoon' ) . ' <a class="author-link" rel="author" href="' . esc_url( get_author_posts_url( get_the_author_meta( 'ID' ) ) ) . '">';
			the_author_meta( 'display_name' );
			echo '</a>';
			echo '</span>';
		}
		if ( in_array( 'date', $metas, true ) ) {
			$time_string = '<time class="entry-date published updated" datetime="%1$s">%2$s</time>';
			if ( get_the_time( 'U' ) !== get_the_modified_time( 'U' ) ) {
				$time_string = '<time class="entry-date published" datetime="%1$s">%2$s</time>';
			}

			$time_string = sprintf(
				$time_string,
				esc_attr( get_the_date( 'c' ) ),
				esc_html( get_the_date() ),
				esc_attr( get_the_modified_date( 'c' ) ),
				esc_html( get_the_modified_date() )
			);

			echo '<span class="published"><i class="fa fa-clock-o"></i>' . $time_string . '</span>';

		}
		if ( in_array( 'cat', $metas, true ) ) {
			if ( has_category() ) {
				echo '<span class="category"><i class="fa fa-folder-open-o"></i>';
				the_category( ', ' );
				echo '</span>';
			}
		}
		if ( in_array( 'tags', $metas, true ) ) {
			if ( has_tag() ) {
				echo '<span class="tags"><i class="fa fa-tags"></i>';
				the_tags( '', ', ' );
				echo '</span>';
			}
		}
		if ( in_array( 'com', $metas, true ) ) {
			echo '<span class="comments"><i class="fa fa-comment-o"></i>';
			comments_popup_link(
				esc_html__( '0 comments', 'cocoon' ),
				esc_html__( '1 comment', 'cocoon' ),
				esc_html__( '% comments', 'cocoon' ),
				'comments-link',
				esc_html__( 'Comments are off', 'cocoon' )
			);
			echo '</span>';
		}
	}
	echo '</div>';
}

/*
=====================================================
	SINGLE POST CUSTOM FUNCTIONS
=====================================================
*/

/**
 *  Display navigation to next/previous set of posts when applicable.
 *
 * @since  1.0
 */
function cocoon_get_post_navigation() {
	require_once get_template_directory() . '/templates/extra/post-nav.php';
}

/**
 *  Display comment navigation.
 *
 * @since  1.0
 */
function cocoon_get_comment_navigation() {
	if ( get_comment_pages_count() > 1 && get_option( 'page_comments' ) ) {
		require_once get_template_directory() . '/templates/comments/comment-nav.php';
	}
}

/**
 *  Add support for Vertical Featured Images.
 *
 * @since  1.0
 */
if ( ! function_exists( 'cocoon_vertical_featured_image' ) ) {
	function cocoon_vertical_featured_image( $html, $post_id, $post_thumbnail_id, $size, $attr ) {
		$image_data = wp_get_attachment_image_src( $post_thumbnail_id, 'large' );

		// Get the image width and height from the data provided by wp_get_attachment_image_src().
		$width  = $image_data[1];
		$height = $image_data[2];

		if ( $height > $width ) {
			$html = str_replace( 'attachment-', 'vertical-image attachment-', $html );
		}
		return $html;
	}

	add_filter( 'post_thumbnail_html', 'cocoon_vertical_featured_image', 10, 5 );
}

/*
=====================================================
	COMMENT FUNCTIONS
=====================================================
*/

/**
 * Comment Callback Function.
 *
 * @since 1.0.0
 */
if ( ! function_exists( 'cocoon_comment' ) ) {
	require_once get_template_directory() . '/templates/comments/comments.php';
} // ends check for cocoon_comment()

/**
 * Comment Form.
 *
 * @since 1.0.0
 */
function cocoon_comment_form( $args ) {

	$commenter    = wp_get_current_commenter();
	$current_user = wp_get_current_user();
	$req          = get_option( 'require_name_email' );
	$aria_req     = ( $req ? " aria-required='true'" : '' );

	$comment_author       = esc_attr( $commenter['comment_author'] );
	$comment_author_email = esc_attr( $commenter['comment_author_email'] );
	$comment_author_url   = esc_attr( $commenter['comment_author_url'] );

	$name    = ( $comment_author ) ? '' : esc_html__( 'Name *', 'cocoon' );
	$email   = ( $comment_author_email ) ? '' : esc_html__( 'Email *', 'cocoon' );
	$website = ( $comment_author_url ) ? '' : esc_html__( 'Website', 'cocoon' );

	$fields = [
		'author' =>
			'<div class="col-md-6 form-group"><input id="author" name="author" type="text" class="form-control" value="' . esc_attr( $commenter['comment_author'] ) . '" placeholder="' . esc_attr__( 'Your Name', 'cocoon' ) . '" required="required" /></div>',

		'email'  =>
			'<div class="col-md-6 form-group"><input id="email" name="email" class="form-control" type="text" value="' . esc_attr( $commenter['comment_author_email'] ) . '" placeholder="' . esc_attr__( 'Your Email', 'cocoon' ) . '" required="required" /></div>',
	];

	$args = [
		'class_form'           => esc_attr( 'row comment-form mb50' ),
		'title_reply'          => esc_html__( 'Leave a Comment', 'cocoon' ),
		'title_reply_to'       => esc_html__( 'Leave a Reply to %s', 'cocoon' ),
		'title_reply_before'   => '<h4 id="reply-title" class="comment-reply-title nomargin">',
		'title_reply_after'    => '</h4>',
		'submit_button'        => '<input name="%1$s" type="submit" id="%2$s" class="%3$s" value="%4$s" />',
		'submit_field'         => '<div class="col-12">%1$s %2$s</div>',
		'class_submit'         => esc_attr( 'btn btn-main' ),
		'label_submit'         => esc_attr__( 'send comment', 'cocoon' ),
		'comment_field'        => '<div class="col-12 form-group"><textarea id="comment" class="form-control" name="comment" rows="8" placeholder="' . esc_attr__( 'Type your comment...', 'cocoon' ) . '" required="required"></textarea></div>',
		'comment_notes_before' => '<div class="col-12"><p><em>' . esc_html__( 'Your email address will not be published.', 'cocoon' ) . '</em></p></div>',
		'logged_in_as'         => '<div class="col-12"><p class="logged-in-as">' .
			sprintf(
				esc_html__( 'Logged in as ', 'cocoon' ) . '<a href="%1$s">%2$s</a>. <a href="%3$s" title="' . esc_html( 'Log out of this account', 'cocoon' ) . '">' . esc_html( 'Log out?', 'cocoon' ) . '</a>',
				esc_url( admin_url( 'profile.php' ) ),
				$current_user->user_login,
				wp_logout_url( apply_filters( 'the_permalink', get_permalink() ) )
			) . '</p></div>',
		'cancel_reply_link'    => esc_html__( 'Cancel Reply', 'cocoon' ),
		'fields'               => apply_filters( 'comment_form_default_fields', $fields ),
	];

	return $args;

}

add_filter( 'comment_form_defaults', 'cocoon_comment_form' );

/*
=====================================================
	PAGE HEADER
=====================================================
*/

/**
 * Fetching the pages titles.
 *
 * @since 1.0.0
 */
if ( ! function_exists( 'cocoon_get_the_title' ) ) {
	function cocoon_get_the_title() {

		if ( cocoon_wc_is_activated() ) {
			if ( is_woocommerce() ) {
				if ( is_single() && ! is_attachment() ) {
					echo get_the_title();
				} elseif ( ! is_single() ) {
					woocommerce_page_title();
				}

				return;
			}
		}

		if ( is_404() ) {
			return esc_html__( 'Error 404', 'cocoon' );
		}

		// Homepage and Single Page.
		if ( is_home() || is_single() || is_404() ) {
			return get_the_title();
		}

		// Search Page.
		if ( is_search() ) {
			return sprintf( esc_html__( 'Search Results for: %s', 'cocoon' ), '<span>' . get_search_query() . '</span>' );
		}

		// Archive Pages.
		if ( is_archive() ) {
			if ( is_author() ) {
				return sprintf( esc_html__( 'All posts by %s', 'cocoon' ), get_the_author() );
			} elseif ( is_day() ) {
				return sprintf( esc_html__( 'Day: %s', 'cocoon' ), get_the_date() );
			} elseif ( is_month() ) {
				return sprintf( esc_html__( 'Month: %s', 'cocoon' ), get_the_date( _x( 'F Y', 'monthly archives date format', 'cocoon' ) ) );
			} elseif ( is_year() ) {
				return sprintf( esc_html__( 'Year: %s', 'cocoon' ), get_the_date( _x( 'Y', 'yearly archives date format', 'cocoon' ) ) );
			} elseif ( is_tag() ) {
				return sprintf( esc_html__( 'Tag: %s', 'cocoon' ), single_tag_title( '', false ) );
			} elseif ( is_category() ) {
				return sprintf( esc_html__( 'Category: %s', 'cocoon' ), single_cat_title( '', false ) );
			} elseif ( is_tax( 'post_format', 'post-format-aside' ) ) {
				return esc_html__( 'Asides', 'cocoon' );
			} elseif ( is_tax( 'post_format', 'post-format-video' ) ) {
				return esc_html__( 'Videos', 'cocoon' );
			} elseif ( is_tax( 'post_format', 'post-format-audio' ) ) {
				return esc_html__( 'Audio', 'cocoon' );
			} elseif ( is_tax( 'post_format', 'post-format-quote' ) ) {
				return esc_html__( 'Quotes', 'cocoon' );
			} elseif ( is_tax( 'post_format', 'post-format-gallery' ) ) {
				return esc_html__( 'Galleries', 'cocoon' );
			} else {
				return esc_html__( 'Archives', 'cocoon' );
			}
		}

		return get_the_title();
	}
}

/*
=====================================================
	PHOTOSWIPE DIALOG ELEMENT
=====================================================
*/

function cocoon_product_images_lightbox() {
	if ( ! cocoon_wc_is_activated() ) {
		return;
	}

	if ( ! is_singular() ) {
		return;
	}
	?>

	<div id="pswp" class="pswp" tabindex="-1" role="dialog" aria-hidden="true">

		<div class="pswp__bg"></div>

		<div class="pswp__scroll-wrap">

			<div class="pswp__container">
				<div class="pswp__item"></div>
				<div class="pswp__item"></div>
				<div class="pswp__item"></div>
			</div>

			<div class="pswp__ui pswp__ui--hidden">

				<div class="pswp__top-bar">


					<div class="pswp__counter"></div>

					<button class="pswp__button pswp__button--close" title="<?php esc_attr_e( 'Close (Esc)', 'cocoon' ); ?>"></button>

					<button class="pswp__button pswp__button--share" title="<?php esc_attr_e( 'Share', 'cocoon' ); ?>"></button>

					<button class="pswp__button pswp__button--fs" title="<?php esc_attr_e( 'Toggle fullscreen', 'cocoon' ); ?>"></button>

					<button class="pswp__button pswp__button--zoom" title="<?php esc_attr_e( 'Zoom in/out', 'cocoon' ); ?>"></button>

					<div class="pswp__preloader">
						<div class="pswp__preloader__icn">
							<div class="pswp__preloader__cut">
								<div class="pswp__preloader__donut"></div>
							</div>
						</div>
					</div>
				</div>

				<div class="pswp__share-modal pswp__share-modal--hidden pswp__single-tap">
					<div class="pswp__share-tooltip"></div>
				</div>

				<button class="pswp__button pswp__button--arrow--left" title="<?php esc_attr_e( 'Previous (arrow left)', 'cocoon' ); ?>">
				</button>

				<button class="pswp__button pswp__button--arrow--right" title="<?php esc_attr_e( 'Next (arrow right)', 'cocoon' ); ?>">
				</button>

				<div class="pswp__caption">
					<div class="pswp__caption__center"></div>
				</div>

			</div>

		</div>

	</div>
	<?php
}

add_action( 'wp_footer', 'cocoon_product_images_lightbox' );

/*
=====================================================
	EXTRA FUNCTIONS
=====================================================
*/

/**
 *  Limits number of words from string.
 *
 * @since  1.0
 */
if ( ! function_exists( 'cocoon_string_limit_words' ) ) {
	function cocoon_string_limit_words( $string, $word_limit ) {
		$words = explode( ' ', $string, ( $word_limit + 1 ) );
		if ( count( $words ) > $word_limit ) {
			array_pop( $words );
			return implode( ' ', $words );
		} else {
			return implode( ' ', $words );
		}
	}
}

/**
 *  Remove attributes from Embeded Iframes
 *
 * @since  1.0
 */
function cocoon_oembed( $return, $data, $url ) {

	if ( is_object( $data ) && property_exists( $data, 'provider_name' ) && 'Vimeo' === $data->provider_name || 'YouTube' === $data->provider_name || 'SoundCloud' === $data->provider_name ) {
		// Remove the unwanted attributes.
		$return = str_ireplace(
			[
				'frameborder="0"',
				'frameborder="no"',
				'webkitallowfullscreen',
				'mozallowfullscreen',
				'scrolling="no"',
			],
			'',
			$return
		);
	}

	return $return;
}

add_filter( 'oembed_dataparse', 'cocoon_oembed', 10, 3 );

/**
 * Get Instagram Photos
 *
 * @since  1.1.2
 */
function cocoon_instagram_photos( $hashtag, $numbers, $title, $columns, $autoplay, $instagram_access_token = false, $image_size = 'low_resolution' ) {
	global $post;

	if ( empty( $instagram_access_token ) ) {
		return '';
	}

	$url = 'https://api.instagram.com/v1/users/self/media/recent?access_token=' . $instagram_access_token;
	if ( $hashtag ) {
		$url = 'https://api.instagram.com/v1/tags/' . $hashtag . '/media/recent?access_token=' . $instagram_access_token;
	}

	$remote = wp_remote_get( $url );

	if ( is_wp_error( $remote ) ) {
		return esc_html__( 'Unable to communicate with Instagram.', 'cocoon' );
	}

	if ( 200 != wp_remote_retrieve_response_code( $remote ) ) {
		return esc_html__( 'Instagram did not return a 200.', 'cocoon' );
	}

	$insta_array = json_decode( $remote['body'], true );

	if ( ! $insta_array ) {
		return esc_html__( 'Instagram has returned invalid data.', 'cocoon' );
	}

	$results = [];

	if ( isset( $insta_array['data'] ) ) {
		$results = $insta_array['data'];
	} else {
		return esc_html__( 'Instagram has returned invalid data.', 'cocoon' );
	}

	if ( ! is_array( $results ) ) {
		return esc_html__( 'Instagram has returned invalid data.', 'cocoon' );
	}

	$columns = intval( $columns );

	$output   = [];
	$output[] = '<div class="cocoon-product-instagram">';
	$output[] = sprintf( '<h3>%s</h3>', wp_kses( $title, wp_kses_allowed_html( 'post' ) ) );
	$output[] = '<ul class="products" data-columns="' . esc_attr( $columns ) . '" data-auto="' . esc_attr( $autoplay ) . '">';

	$count = 0;

	if ( $results ) {
		foreach ( $results as $item ) {
			$image_link = $item['images'][ $image_size ]['url'];
			$image_url  = $item['link'];

			$image_html  = '';
			$image_trans = get_template_directory_uri() . '/images/transparent.png';

			if ( intval( cocoon_get_option( 'cocoon_lazyload' ) ) ) {
				$image_html = sprintf( '<img class="lazyload" data-src="%s" alt="%s">', esc_url( $image_link ), esc_attr( '' ) );
			} else {
				$image_html = sprintf( '<img src="%s" alt="%s">', esc_url( $image_link ), esc_attr( '' ) );
			}

			$output[] = '<li class="product">' . '<a class="insta-item" href="' . esc_url( $image_url ) . '" target="_blank">' . $image_html . '<i class="fa fa-instagram"></i></a>' . '</li>' . "\n";
			$count ++;
			$numbers = intval( $numbers );
			if ( $numbers > 0 ) {
				if ( $count == $numbers ) {
					break;
				}
			}
		}
	} else {
		$output[] = sprintf( '<li>%s</li>', esc_html__( 'No Result found.', 'cocoon' ) );
	}

	$output[] = '</ul></div>';

	return implode( ' ', $output );
}

/**
 * Add classes to body
 *
 * @since  1.2.0
 */
function cocoon_body_classes( $classes ) {
	if ( cocoon_get_option( 'cocoon_ajax_shop' ) == '1' ) {
		$classes[] = 'ajax_shop_active';
	}

	return $classes;
}

add_filter( 'body_class', 'cocoon_body_classes' );

/**
 * Lazy Load Functions
 *
 * @since  1.2.0
 */
function cocoon_lazy_loading( $attr, $attachment, $size ) {
	$attr['data-original'] = $attr['src'];
	$attr['data-lazy']     = $attr['src'];
	$image                 = wp_get_attachment_image_src( $attachment->ID, $size );
	$attr['src']           = get_template_directory_uri() . '/assets/images/transparent.png';
	$lazy_class            = 'lazy';
	$attr['class']         = ( isset( $attr['class'] ) ? $attr['class'] . " {$lazy_class}" : $lazy_class );

	$attr['class'] = ( isset( $attr['class'] ) ? $attr['class'] : '' );

	return $attr;
}

function cocoon_add_lazy_load_attrs() {
	add_filter( 'wp_get_attachment_image_attributes', 'cocoon_lazy_loading', 10, 3 );
}

function cocoon_remove_lazy_load_attrs() {
	remove_action( 'wp_get_attachment_image_attributes', 'cocoon_lazy_loading', 10 );
}

/*
=====================================================
	COOKIE BAR - LAW INFO
=====================================================
*/

/**
 * Cookie Law Info
 *
 * @since  1.1.2
 */
if ( ! function_exists( 'cocoon_cookie_bar' ) ) {
	function cocoon_cookie_bar() {
		if ( cocoon_get_option( 'cocoon_cookie_notice' ) == '0' ) {
			return;
		}

		$text_msg    = cocoon_get_option( 'cookie_notice_message' );
		$policy_page = cocoon_get_option( 'cookie_policy_page' );
		?>

		<div class="cocoon-cookies-bar">
			<div class="cocoon-cookies-inner">
				<div class="cookies-info-text">
					<?php echo esc_html( $text_msg ); ?>
				</div>
				<div class="cookies-buttons">
					<a href="#" class="btn btn-small btn-round btn-main cookies-accept-btn"><?php esc_html_e( 'Accept', 'cocoon' ); ?></a>
					<?php if ( $policy_page ) { ?>
						<a href="<?php echo esc_url( get_page_link( $policy_page ) ); ?>" class="cookies-more-btn" target="_blank">
							<?php esc_html_e( 'More info', 'cocoon' ); ?>
						</a>
					<?php } ?>
				</div>
			</div>
		</div>
		<?php

	}

	add_action( 'wp_footer', 'cocoon_cookie_bar' );
}

/*
=====================================================
	HEADER SWITCHERS
=====================================================
*/

/**
 * Currency Switcher (WooCommerce Currency Switcher required)
 *
 * @since  1.1.9
 */
if ( ! function_exists( 'cocoon_currency_switcher' ) ) {
	function cocoon_currency_switcher( $show_desc = false ) {
		$currency_switch = '';

		if ( class_exists( 'WOOCS' ) && cocoon_wc_is_activated() ) {
			global $WOOCS;

			$key_cur = 'name';
			if ( $show_desc ) {
				$key_cur = 'description';
			}

			$currencies    = $WOOCS->get_currencies();
			$currency_list = [];
			foreach ( $currencies as $key => $currency ) {
				if ( $WOOCS->current_currency == $key ) {
					array_unshift(
						$currency_list,
						sprintf(
							'<li class="active"><a href="#" class="woocs_flag_view_item woocs_flag_view_item_current" data-currency="%s">%s</a></li>',
							esc_attr( $currency['name'] ),
							esc_html( $currency[ $key_cur ] )
						)
					);
				} else {
					$currency_list[] = sprintf(
						'<li><a href="#" class="woocs_flag_view_item" data-currency="%s">%s</a></li>',
						esc_attr( $currency['name'] ),
						esc_html( $currency[ $key_cur ] )
					);
				}
			}

			$currency_switch = sprintf(
				'<span class="nav-link current">%s<span class="toggle-children"></span></span>' .
				'<ul>%s</ul>',
				$currencies[ $WOOCS->current_currency ][ $key_cur ],
				implode( "\n\t", $currency_list )
			);

		}

		return $currency_switch;
	}
}

/**
 * Language Switcher (WPML required)
 *
 * @since  1.1.9
 */
if ( ! function_exists( 'cocoon_language_switcher' ) ) {
	function cocoon_language_switcher( $type = 'code' ) {
		$language_switcher = '';

		if ( function_exists( 'icl_get_languages' ) ) {
			$languages = icl_get_languages();

			if ( $languages ) {
				$lang_list = [];
				$current   = '';
				foreach ( (array) $languages as $code => $language ) {
					$lang = $code;
					if ( $type == 'name' ) {
						$lang = $language['translated_name'];
					} elseif ( $type == 'tag' ) {
						$lang = $language['tag'];
					}

					if ( ! $language['active'] ) {
						$lang_list[] = sprintf(
							'<li class="%s"><a href="%s">%s</a></li>',
							esc_attr( $code ),
							esc_url( $language['url'] ),
							$lang
						);
					} else {
						$current = $language;
						array_unshift(
							$lang_list,
							sprintf(
								'<li class="active %s"><a href="%s">%s</a></li>',
								esc_attr( $code ),
								esc_url( $language['url'] ),
								$lang
							)
						);
					}
				}

				$lang = esc_html( $current['language_code'] );
				if ( $type == 'name' ) {
					$lang = $current['translated_name'];
				} elseif ( $type == 'tag' ) {
					$lang = $current['tag'];
				}

				$language_switcher = sprintf(
					'<span class="nav-link current">%s<span class="toggle-children"></span></span>' .
					'<ul>%s</ul>',
					$lang,
					implode( "\n\t", $lang_list )
				);
			}
		}

		return $language_switcher;
	}
}
