import gulp from 'gulp';
import autoprefixer from 'gulp-autoprefixer';
import lec from 'gulp-line-ending-corrector';
import notify from 'gulp-notify';
import rename from 'gulp-rename';
import rtlcss from 'gulp-rtlcss';
import _uglify from 'gulp-uglify-es';
import uglifycss from 'gulp-uglifycss';
import nodeSass from 'node-sass';
import gulpSass from 'gulp-sass';
import wpPot from 'gulp-wp-pot';
const sass = gulpSass(nodeSass);
import zip from 'gulp-zip';
const uglify = _uglify.default;

// Project information.
const info = {
	name: 'colormag-pro',
	slug: 'colormag-pro',
	url: 'https://themegrill.com/plugins/zakra-pro/',
	author: 'ThemeGrill',
	authorUrl: 'https://themegrill.com/',
	authorEmail: 'themegrill@gmail.com',
	teamEmail: 'team@themegrill.com',
};

const dirs = {
	js: 'js',
	css: 'css'
};

// Define paths.
const paths = {

	styles : {
		src  : './assets/sass/**/*.scss',
		dest : './'
	},

	js : {
		src  : [
			'./assets/js/*.js',
			'!./assets/js/*.min.js',
			// './assets/js/**/*.js',
			// '!./assets/js/**/*.min.js'
		],
		dest : './assets/js/'
	},

	customizePreviewJS : {
		src : [
			'./inc/customizer/assets/js/*.js',
			'!./inc/customizer/assets/js/*.min.js',
		],
		dest : './inc/customizer/assets/js/',
	},

	elementorStyles : {
		scss : {
			src  : './inc/compatibility/elementor/assets/SCSS/**/*.scss',
			dest : './inc/compatibility/elementor/assets/css/'
		},
		cssmin : {
			src  : [
				'./inc/compatibility/elementor/assets/css/*.css',
				'!./inc/compatibility/elementor/assets/css/*.min.css',
				'!./inc/compatibility/elementor/assets/css/*-rtl.css'
			],
			dest : './inc/compatibility/elementor/assets/css/'
		}
	},

	elementorJS : {
		jsmin : {
			src  : [
				'./inc/compatibility/elementor/assets/js/**/*.js',
				'!./inc/compatibility/elementor/assets/js/**/*.min.js'
			],
			dest : './inc/compatibility/elementor/assets/js/'
		}
	},

	metaBoxes : {
		scss      : {
			src  : './inc/meta-boxes/assets/scss/*.scss',
			dest : './inc/meta-boxes/assets/css'
		},
		cssmin    : {
			src  : [
				'./inc/meta-boxes/assets/css/*.css',
				'!./inc/meta-boxes/assets/css/*.min.css',
				'!./inc/meta-boxes/assets/css/*-rtl.css'
			],
			dest : './inc/meta-boxes/assets/css'
		},
		jsmin     : {
			src  : [
				'./inc/meta-boxes/assets/js/*.js',
				'!./inc/meta-boxes/assets/js/*.min.js'
			],
			dest : './inc/meta-boxes/assets/js'
		}
	},

	rtlcss : {
		style             : {
			src  : [ './style.css' ],
			dest : './'
		},
		woocommerceStyle             : {
			src  : [ './woocommerce.css' ],
			dest : './'
		},
		blockStyle        : {
			src  : [ './style-editor-block.css' ],
			dest : './'
		},
		elementorStyles   : {
			src  : [
				'./inc/compatibility/elementor/assets/css/elementor.css',
				'./inc/compatibility/elementor/assets/css/elementor.min.css'
			],
			dest : './inc/compatibility/elementor/assets/css'
		},
		metaBoxes         : {
			src  : [
				'./inc/meta-boxes/assets/css/meta-boxes.css',
				'./inc/meta-boxes/assets/css/meta-boxes.min.css'
			],
			dest : './inc/meta-boxes/assets/css'
		}
	},
	adminStyles : {
		src  : './inc/admin/sass/*.scss',
		dest : './inc/admin/css/'
	},
	zip: {
		src: [
			'**',
			'!assets/sass/**',
			'!.*',
			'!*.md',
			'!*.zip',
			'!.*/**',
			'!dist/**',
			'!bower.json',
			'!gulpfile.js',
			'!Gruntfile.js',
			'!package.json',
			'!node_modules/**',
			'!package-lock.json'
		],
		dest: './dist',
	},

};

// Compiles SASS into CSS.
function sassCompile() {
	return gulp.src( paths.styles.src )
		.pipe( sass( {
			indentType  : 'tab',
			indentWidth : 1,
			outputStyle : 'expanded',
			linefeed    : 'crlf'
		} ))
		.pipe( autoprefixer() )
		.pipe( lec( { verbose : true, eolc : 'LF', encoding : 'utf8' } ) )
		.pipe( gulp.dest( paths.styles.dest ) );
}

function elementorStylesCompile() {
	return gulp.src( paths.elementorStyles.scss.src )
		.pipe( sass( {
			indentType  : 'tab',
			indentWidth : 1,
			outputStyle : 'expanded',
			linefeed    : 'crlf'
		} ).on( 'error', sass.logError ) )
		.pipe(autoprefixer())
		.pipe( lec( { verbose : true, eolc : 'LF', encoding : 'utf8' } ) )
		.pipe( gulp.dest( paths.elementorStyles.scss.dest ) );
}

// Minifies the elementor js files.
async function minifyelementorJs() {
	return gulp
		.src( paths.elementorJS.jsmin.src )
		.pipe( uglify() )
		.pipe( rename( { suffix : '.min' } ) )
		.pipe( lec( { verbose : true, eolc : 'LF', encoding : 'utf8' } ) )
		.pipe( gulp.dest( paths.elementorJS.jsmin.dest ) )
		.on( 'error', notify.onError() );
}

// Compiles Admin SASS into CSS.
function adminSassCompile() {
	return gulp.src( paths.adminStyles.src )
		.pipe( sass( {
			indentType  : 'tab',
			indentWidth : 1,
			outputStyle : 'expanded',
			linefeed    : 'crlf'
		} ).on( 'error', sass.logError ) )
		.pipe( lec( { verbose : true, eolc : 'LF', encoding : 'utf8' } ) )
		.pipe( gulp.dest( paths.adminStyles.dest ) );
}

// Minify elementor styles css file.
function minifyelementorStyles() {
	return gulp
		.src( paths.elementorStyles.cssmin.src )
		.pipe( uglifycss() )
		.pipe( rename( { suffix : '.min' } ) )
		.pipe( lec( { verbose : true, eolc : 'LF', encoding : 'utf8' } ) )
		.pipe( gulp.dest( paths.elementorStyles.cssmin.dest ) );
}

// Minifies the customizer js files.
function minifyCustomizerJs() {
	return gulp
		.src( paths.customizePreviewJS.src )
		.pipe( uglify() )
		.pipe( rename( { suffix : '.min' } ) )
		.pipe( lec( { verbose : true, eolc : 'LF', encoding : 'utf8' } ) )
		.pipe( gulp.dest( paths.customizePreviewJS.dest ) )
		.on( 'error', notify.onError() );
}

// Compile meta boxes styles.
function compileMetaBoxSass() {
	return gulp
		.src( paths.metaBoxes.scss.src )
		.pipe( sass( {
			indentType  : 'tab',
			indentWidth : 1,
			outputStyle : 'expanded',
			linefeed    : 'crlf'
		} ).on( 'error', sass.logError ) )
		.pipe(autoprefixer())
		.pipe( lec( { verbose : true, eolc : 'LF', encoding : 'utf8' } ) )
		.pipe( gulp.dest( paths.metaBoxes.scss.dest ) )
		.on( 'error', notify.onError() );
}

// Minify meta box css file.
function minifyMetaBoxCSS() {
	return gulp
		.src( paths.metaBoxes.cssmin.src )
		.pipe( uglifycss() )
		.pipe( rename( { suffix : '.min' } ) )
		.pipe( lec( { verbose : true, eolc : 'LF', encoding : 'utf8' } ) )
		.pipe( gulp.dest( paths.metaBoxes.cssmin.dest ) );
}

// Minifies the metabox js files.
function minifyMetaBoxJs() {
	return gulp
		.src( paths.metaBoxes.jsmin.src )
		.pipe( uglify() )
		.pipe( rename( { suffix : '.min' } ) )
		.pipe( lec( { verbose : true, eolc : 'LF', encoding : 'utf8' } ) )
		.pipe( gulp.dest( paths.metaBoxes.jsmin.dest ) )
		.on( 'error', notify.onError() );
}

// Generates RTL CSS file.
function generateRTLCSS() {
	return gulp
		.src( paths.rtlcss.style.src )
		.pipe( rtlcss() )
		.pipe( rename( { suffix: '-rtl' } ) )
		.pipe( lec( { verbose : true, eolc : 'LF', encoding : 'utf8' } ) )
		.pipe( gulp.dest( paths.rtlcss.style.dest ) )
		.on( 'error', notify.onError() );
}

// Generates Block Style RTL CSS file.
function generateBlockStyleRTLCSS() {
	return gulp
		.src( paths.rtlcss.blockStyle.src )
		.pipe( rtlcss() )
		.pipe( rename( { suffix: '-rtl' } ) )
		.pipe( lec( { verbose : true, eolc : 'LF', encoding : 'utf8' } ) )
		.pipe( gulp.dest( paths.rtlcss.blockStyle.dest ) )
		.on( 'error', notify.onError() );
}

// Generates Elementor RTL CSS file.
function generateElementorRTLCSS() {
	return gulp
		.src( paths.rtlcss.elementorStyles.src )
		.pipe( rtlcss() )
		.pipe( rename( { suffix: '-rtl' } ) )
		.pipe( lec( { verbose : true, eolc : 'LF', encoding : 'utf8' } ) )
		.pipe( gulp.dest( paths.rtlcss.elementorStyles.dest ) )
		.on( 'error', notify.onError() );
}

// Generates Meta Boxes RTL CSS file.
function generateMetaBoxesRTLCSS() {
	return gulp
		.src( paths.rtlcss.metaBoxes.src )
		.pipe( rtlcss() )
		.pipe( rename( { suffix: '-rtl' } ) )
		.pipe( lec( { verbose : true, eolc : 'LF', encoding : 'utf8' } ) )
		.pipe( gulp.dest( paths.rtlcss.metaBoxes.dest ) )
		.on( 'error', notify.onError() );
}

// From grunt

// Compress plugin into a zip file.
function compressZip() {
	return gulp
		.src(paths.zip.src, {
			encoding: false
		})
		.pipe(
			rename(function (path) {
				path.dirname = info.name + '/' + path.dirname;
			}),
		)
		.pipe(zip(info.name + '.zip'))
		.pipe(gulp.dest(paths.zip.dest));
}

// Generate POT files.
function makePot() {
	return gulp
		.src('**/*.php')
		.pipe(
			wpPot({
				domain: 'colormag',
				package: 'ColorMag Pro',
				bugReport: 'themegrill@gmail.com',
				team: 'LANGUAGE <EMAIL@ADDRESS>'
			}),
		)
		.pipe(gulp.dest('languages/colormag-pro.pot'));
}

// Minify all .js files.
async function minifyJs() {
	return gulp
		.src( paths.js.src )
		.pipe( uglify() )
		.pipe( rename( { suffix : '.min' } ) )
		.pipe( lec( { verbose : true, eolc : 'LF', encoding : 'utf8' } ) )
		.pipe( gulp.dest( paths.js.dest ) )
		.on( 'error', notify.onError() );
}

// JavaScript linting with JSHint.
// function lintJs() {
// 	return gulp.src([
// 		'gulpfile.mjs',
// 		`${dirs.js}/*.js`,
// 		`!${dirs.js}/*.min.js`,
// 		`!${dirs.js}/html5shiv.js`,
// 		`!${dirs.js}/jquery.bxslider.js`
// 	])
// 		.pipe(jshint('.jshintrc'))
// 		.pipe(jshint.reporter('default'));
// }

// Watch changes for assets.
// function watchFiles() {
// 	gulp.watch([
// 		`${dirs.js}/*.js`,
// 		`!${dirs.js}/*.min.js`,
// 		`!${dirs.js}/html5shiv.js`,
// 		`!${dirs.js}/jquery.bxslider.js`
// 	], gulp.series(lintJs, minifyJs));
// }

// Check textdomain errors.
// function checkTextDomain() {
// 	return gulp.src([
// 		'**/*.php',
// 		'!node_modules/**'
// 	])
// 		.pipe(checktextdomain({
// 			text_domain: 'colormag',
// 			keywords: [
// 				'__:1,2d',
// 				'_e:1,2d',
// 				'_x:1,2c,3d',
// 				'esc_html__:1,2d',
// 				'esc_html_e:1,2d',
// 				'esc_html_x:1,2c,3d',
// 				'esc_attr__:1,2d',
// 				'esc_attr_e:1,2d',
// 				'esc_attr_x:1,2c,3d',
// 				'_ex:1,2c,3d',
// 				'_n:1,2,4d',
// 				'_nx:1,2,4c,5d',
// 				'_n_noop:1,2,3d',
// 				'_nx_noop:1,2,3c,4d'
// 			]
// 		}));
// }

// Download language files.
// function downloadLanguages() {
// 	return gulp.src('languages')
// 		.pipe(glotpress({
// 			domainPath: 'languages',
// 			url: 'http://translate.themegrill.com/glotpress',
// 			textdomain: 'colormag',
// 			slug: 'themes/colormag-pro',
// 			file_format: '%domainPath%/%wp_locale%.%format%'
// 		}));
// }

// Copy files.
// function copyFiles() {
// 	return gulp.src([
// 		'bower_components/font-awesome/css/**/*.css',
// 		'bower_components/font-awesome/fonts/**/*',
// 		'bower_components/magnific-popup/dist/**/*',
// 		'bower_components/fitvids/**/*.js',
// 		'bower_components/bxslider-4/dist/**/*.js',
// 		'bower_components/bxslider-4/dist/**/*.css',
// 		'bower_components/easytabs/lib/**/*.js',
// 		'bower_components/theia-sticky-sidebar/dist/**/*.js'
// 	], { base: '.' })
// 		.pipe(copy('dest'));
// }

// Bower update.
// function bowerUpdate() {
// 	return bower();
// }

function watch(){
	gulp.watch( paths.styles.src, sassCompile );
	gulp.watch( paths.styles.src, adminSassCompile );
	gulp.watch( paths.elementorStyles.scss.src, elementorStylesCompile );
	gulp.watch( paths.elementorStyles.cssmin.src, minifyelementorStyles );
	gulp.watch( paths.elementorJS.jsmin.src, minifyelementorJs );
	gulp.watch( paths.js.src, minifyJs );
	gulp.watch( paths.js.src, minifyCustomizerJs );
	gulp.watch( paths.metaBoxes.scss.src, compileMetaBoxSass );
	gulp.watch( paths.metaBoxes.cssmin.src, minifyMetaBoxCSS );
	gulp.watch( paths.metaBoxes.jsmin.src, minifyMetaBoxJs );
	gulp.watch( paths.rtlcss.style.src, generateRTLCSS );
	gulp.watch( paths.rtlcss.blockStyle.src, generateBlockStyleRTLCSS );
	gulp.watch( paths.rtlcss.elementorStyles.src, generateElementorRTLCSS );
	gulp.watch( paths.rtlcss.metaBoxes.src, generateMetaBoxesRTLCSS );
}

// Build
const build = gulp.series(
	generateRTLCSS,
	generateBlockStyleRTLCSS,
	generateElementorRTLCSS,
	generateMetaBoxesRTLCSS,
	minifyJs,
	adminSassCompile,
	minifyCustomizerJs,
	minifyMetaBoxJs,
	minifyMetaBoxCSS,
	minifyelementorJs,
	minifyelementorStyles,
	sassCompile,
	elementorStylesCompile,
	compileMetaBoxSass,
	makePot,
	compressZip,
);

// Watch for file changes.
// function watch() {
// 	gulp.watch( paths.styles.src, sassCompile );
// 	gulp.watch( paths.elementorStyles.scss.src, elementorStylesCompile );
// 	gulp.watch( paths.elementorStyles.cssmin.src, minifyelementorStyles );
// 	gulp.watch( paths.elementorJS.jsmin.src, minifyelementorJs );
// 	gulp.watch( paths.js.src, minifyJs );
// 	gulp.watch( paths.js.src, minifyCustomizerJs );
// 	gulp.watch( paths.metaBoxes.scss.src, compileMetaBoxSass );
// 	gulp.watch( paths.metaBoxes.cssmin.src, minifyMetaBoxCSS );
// 	gulp.watch( paths.metaBoxes.jsmin.src, minifyMetaBoxJs );
// 	gulp.watch( paths.rtlcss.style.src, generateRTLCSS );
// 	gulp.watch( paths.rtlcss.blockStyle.src, generateBlockStyleRTLCSS );
// 	gulp.watch( paths.rtlcss.elementorStyles.src, generateElementorRTLCSS );
// 	gulp.watch( paths.rtlcss.metaBoxes.src, generateMetaBoxesRTLCSS );
// }

// Define series of tasks.
const styles            = gulp.series( sassCompile, generateRTLCSS, generateBlockStyleRTLCSS ),
	scripts           = gulp.series( minifyJs ),
	elementorStyles   = gulp.series( elementorStylesCompile, minifyelementorStyles, minifyelementorJs, generateElementorRTLCSS ),
	metaBoxes         = gulp.series( compileMetaBoxSass, minifyMetaBoxCSS, minifyMetaBoxJs, generateMetaBoxesRTLCSS ),
	compile           = gulp.series( styles, scripts, elementorStyles, metaBoxes );

export {
	build,
	sassCompile,
	adminSassCompile,
	elementorStylesCompile,
	minifyelementorStyles,
	minifyelementorJs,
	styles,
	scripts,
	elementorStyles,
	metaBoxes,
	compile,
	minifyJs,
	minifyCustomizerJs,
	compileMetaBoxSass,
	minifyMetaBoxCSS,
	minifyMetaBoxJs,
	generateRTLCSS,
	generateBlockStyleRTLCSS,
	generateElementorRTLCSS,
	generateMetaBoxesRTLCSS,
	makePot,
	compressZip,
	watch
}
