<?php
/**
 * ColorMag lazy loading images functionality.
 *
 * @package ColorMag
 *
 * @since ColorMag 3.0.0
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'ColorMag_Lazy_Load' ) ) {

	/**
	 * Lazy Loading Images.
	 */
	class ColorMag_Lazy_Load {

		/**
		 * Instance.
		 *
		 * @access private
		 * @var object
		 */
		private static $instance;

		/**
		 * Initiator.
		 */
		public static function get_instance() {
			if ( ! isset( self::$instance ) ) {
				self::$instance = new self();
			}
			return self::$instance;
		}

		/**
		 * Constructor.
		 */
		public function __construct() {
			$this->setup_hooks();
		}

		/**
		 * Define hooks.
		 *
		 * @return void
		 */
		public function setup_hooks() {
			// Skip if lazy loading is disabled
			if ( ! $this->is_lazy_load_enabled() ) {
				return;
			}

			// Content images
			add_filter( 'the_content', array( $this, 'add_lazy_load_to_content_images' ) );

			// Post thumbnails
			add_filter( 'post_thumbnail_html', array( $this, 'add_lazy_load_to_thumbnails' ) );

			// Avatar images
			add_filter( 'get_avatar', array( $this, 'add_lazy_load_to_avatars' ) );

			// Widget images
			add_filter( 'widget_text', array( $this, 'add_lazy_load_to_content_images' ) );

			// Gallery images
			add_filter( 'gallery_style', array( $this, 'add_lazy_load_to_gallery' ) );

		}

		/**
		 * Check if lazy loading is enabled.
		 *
		 * @return bool
		 */
		public function is_lazy_load_enabled() {
			return (bool) get_theme_mod( 'colormag_enable_lazy_load', false );
		}

		/**
		 * Add lazy loading to content images.
		 *
		 * @param string $content The post content.
		 * @return string Modified content.
		 */
		public function add_lazy_load_to_content_images( $content ) {
			// Skip if admin or feed
			if ( is_admin() || is_feed() ) {
				return $content;
			}

			// Replace <img> tags with lazy loading attribute if not already present
			return preg_replace_callback(
				'/<img([^>]+)>/i',
				function ( $matches ) {
					// Skip if already has loading attribute
					if ( strpos( $matches[1], 'loading=' ) !== false ) {
						return $matches[0];
					}

					return '<img' . $matches[1] . ' loading="lazy">';
				},
				$content
			);
		}

		/**
		 * Add lazy loading to post thumbnails.
		 *
		 * @param string $html The post thumbnail HTML.
		 * @return string Modified HTML.
		 */
		public function add_lazy_load_to_thumbnails( $html ) {
			// Skip if admin or feed
			if ( is_admin() || is_feed() ) {
				return $html;
			}

			// Skip if already has loading attribute
			if ( strpos( $html, 'loading=' ) !== false ) {
				return $html;
			}

			return str_replace( '<img', '<img loading="lazy"', $html );
		}

		/**
		 * Add lazy loading to avatars.
		 *
		 * @param string $avatar Avatar image HTML.
		 * @return string Modified HTML.
		 */
		public function add_lazy_load_to_avatars( $avatar ) {
			// Skip if admin or feed
			if ( is_admin() || is_feed() ) {
				return $avatar;
			}

			// Skip if already has loading attribute
			if ( strpos( $avatar, 'loading=' ) !== false ) {
				return $avatar;
			}

			return str_replace( '<img', '<img loading="lazy"', $avatar );
		}

		/**
		 * Add lazy loading to gallery images.
		 *
		 * @param string $style Gallery style HTML.
		 * @return string Modified style HTML.
		 */
		public function add_lazy_load_to_gallery( $style ) {
			// Add script to lazy load gallery images
			add_action(
				'wp_footer',
				function () {
					?>
				<script>
					document.addEventListener('DOMContentLoaded', function() {
						document.querySelectorAll('.gallery img').forEach(function(img) {
							if (!img.hasAttribute('loading')) {
								img.setAttribute('loading', 'lazy');
							}
						});
					});
				</script>
					<?php
				}
			);

			return $style;
		}
	}
}

// Initialize the lazy loading
ColorMag_Lazy_Load::get_instance();
