import {
	Button,
	__experimentalInputControl as InputControl,
	Modal,
} from '@wordpress/components';

import apiFetch from '@wordpress/api-fetch';
import { __, sprintf } from '@wordpress/i18n';
import { maskString } from 'mask-sensitive-data';
import React from 'react';
import { Controller, useForm } from 'react-hook-form';
import { useMutation } from 'react-query';
import { z } from 'zod';
import { useDisclosure } from '../../../hook/useDisclosure';
import { localized } from '../../../utils/global';

const formSchema = z.object({
	email: z.string().email(),
	user_name: z.string(),
	ownership: z.string(),
	transfer_type: z.string(),
	license_key: z.string().optional(),
});

function isValidEmail(email: string) {
	const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
	return emailRegex.test(email);
}

type FormSchema = z.infer<typeof formSchema>;

export const AccountDetails = () => {
	const deactivateLicenseDisclosure = useDisclosure();
	const deactivateLicenseFormRef = React.useRef<HTMLFormElement | null>(null);
	const syncFormRef = React.useRef<HTMLFormElement | null>(null);
	const syncLicenseFormRef = React.useRef<HTMLFormElement | null>(null);
	const editEmailFormRef = React.useRef<HTMLFormElement | null>(null);
	const editNameFormRef = React.useRef<HTMLFormElement | null>(null);
	const [mask, setMask] = React.useState({
		secretKey: true,
		licenseKey: true,
	});
	const updateLicenseDisclosure = useDisclosure();
	const editNameDisclosure = useDisclosure();
	const editEmailDisclosure = useDisclosure();
	const betaDisclosure = useDisclosure();

	const updateBetaProgramMutation = useMutation((beta: boolean) => {
		const formData = new FormData();
		formData.append('action', localized.fs?.account?.set_beta_mode?.action);
		formData.append('security', localized.fs?.account?.set_beta_mode?.security);
		formData.append('module_id', localized.fs?.module_id);
		formData.append('is_beta', beta ? 'true' : 'false');
		return new Promise((resolve, reject) => {
			apiFetch({
				url: localized.fs?.ajax_url,
				method: 'POST',
				body: formData,
			})
				.then((res: any) => {
					if (!res.success) {
						reject(new Error(res.error));
					} else {
						resolve(res);
					}
				})
				.catch((e) => {
					reject(e);
				});
		});
	}, {
		onSuccess: () => {
			// Success handled in the component
		},
		onError: (error: Error) => {
			console.error('Beta program update failed:', error);
		},
	});

	const emailAddressProgramMutation = useMutation(
		() => {
			const formData = new FormData();
			formData.append(
				'action',
				localized.fs?.account?.update_email_address?.action,
			);
			formData.append(
				'security',
				localized.fs?.account?.update_email_address?.security,
			);
			formData.append('module_id', localized.fs?.module_id);
			formData.append('transfer_type', form.getValues().transfer_type || 'transfer');
			formData.append('email_address', form.getValues().email);
			return new Promise((resolve, reject) => {
				apiFetch({
					url: localized.fs?.ajax_url,
					method: 'POST',
					body: formData,
				})
					.then((res: any) => {
						if (!res.success) {
							reject(new Error(res.error || 'Email update failed'));
						} else {
							resolve(res);
						}
					})
					.catch((e) => {
						reject(e);
					});
			});
		},
		{
			onSuccess: () => {
				editEmailDisclosure.onClose();
				window.location.reload();
			},
			onError: (error: Error) => {
				console.error('Email update failed:', error);
			},
		},
	);

	const updateLicenseMutation = useMutation(
		(licenseKey: string) => {
			const formData = new FormData();
			formData.append('action', localized.fs?.account?.update_license?.action);
			formData.append('security', localized.fs?.account?.update_license?.security);
			formData.append('module_id', localized.fs?.module_id);
			formData.append('license_key', licenseKey);
			return new Promise((resolve, reject) => {
				apiFetch({
					url: localized.fs?.ajax_url,
					method: 'POST',
					body: formData,
				})
					.then((res: any) => {
						if (!res.success) {
							reject(new Error(res.error));
						} else {
							resolve(res);
						}
					})
					.catch((e) => {
						reject(e);
					});
			});
		},
		{
			onSuccess: () => {
				updateLicenseDisclosure.onClose();
				form.reset();
				window.location.reload();
			},
			onError: (error: Error) => {
				console.error('License update failed:', error);
			},
		},
	);

	const [beta, setBeta] = React.useState(
		localized.fs?.account?.data?.beta_program,
	);
	const [isSyncing, setIsSyncing] = React.useState(false);
	const form = useForm<FormSchema>({
		defaultValues: {},
	});

	// Reset syncing state when component unmounts
	React.useEffect(() => {
		return () => {
			setIsSyncing(false);
		};
	}, []);
	const watchEmail = form.watch('email');
	const watchOwnership = form.watch('ownership');
	let username = form.watch('user_name');

	return (
		<div>
			<div className="flex items-center justify-between">
				<h2>{__('Account Details', 'colormag')}</h2>
				<div className="flex items-center">
					<form
						action={window.location.href}
						method="post"
						ref={deactivateLicenseFormRef}
					>
						<div
							dangerouslySetInnerHTML={{
								__html: localized.fs?.account?.deactivate_license,
							}}
						/>
						<Button size={'small'} onClick={deactivateLicenseDisclosure.onOpen}>
							{__('Deactivate License', 'colormag')}
						</Button>
					</form>
					<span>|</span>
					<form action={window.location.href} method="post" ref={syncFormRef}>
						<div
							dangerouslySetInnerHTML={{
								__html: localized.fs?.account?.sync,
							}}
						/>
						<Button
							size={'small'}
							onClick={() => {
								syncFormRef.current?.requestSubmit();
							}}
						>
							{__('Sync', 'colormag')}
						</Button>
					</form>
				</div>
			</div>
			<div>
				<div className="relative overflow-hidden cm-account-details">
					<table className="table-fixed w-full text-left border-collapse">
						<tbody
							className="bg-white text-[#6b7280]"
							style={{
								backgroundColor: '#FFFFFF',
								color: '#6b7280',
							}}
						>
							<tr className="py-5">
								<td className="py-4 px-6 text-left">
									{__('Name:', 'colormag')}
								</td>
								<td className="py-4 px-6 text-left text-sm">
									<span>{localized.fs?.account?.data?.user_name}</span>
								</td>
								<td className="py-4 px-6 text-right">
									<form
										action={window.location.href}
										method="post"
										ref={editNameFormRef}
									>
										<div
											dangerouslySetInnerHTML={{
												__html: localized.fs?.account?.user_name,
											}}
										/>
										<Button
											size={'small'}
											className="p-0 h-auto border border-solid border-[#E1E1E1] !px-1"
											onClick={editNameDisclosure.onOpen}
										>
											<svg
												fill="#958989"
												width="18px"
												height="18px"
												xmlns="http://www.w3.org/2000/svg"
												viewBox="0 0 24 24"
											>
												<path
													xmlns="http://www.w3.org/2000/svg"
													fillRule="evenodd"
													d="M13.793 5.293a3.828 3.828 0 1 1 5.414 5.414l-.998.999-.002.001-.001.002-9.499 9.498A1 1 0 0 1 8 21.5H4a1 1 0 0 1-1-1v-4a1 1 0 0 1 .293-.707l10.5-10.5ZM13.5 8.414l-8.5 8.5V19.5h2.586l8.5-8.5L13.5 8.414Zm4 1.172L14.914 7l.293-.293a1.829 1.829 0 0 1 2.586 2.586l-.293.293Z"
													clipRule="evenodd"
												></path>
											</svg>
										</Button>
									</form>
								</td>
							</tr>
							<tr className="py-5">
								<td className="py-4 px-6 text-left">
									{__('Email:', 'colormag')}
								</td>
								<td className="py-4 px-6 text-left text-sm">
									<span>{localized.fs?.account?.data?.email}</span>
								</td>
								<td className="py-4 px-6 text-right">
									<form
										action={window.location.href}
										method="post"
										ref={editEmailFormRef}
									>
										<div
											dangerouslySetInnerHTML={{
												__html: localized.fs?.account?.email,
											}}
										/>
										<Button
											size={'small'}
											className="p-0 h-auto border border-solid border-[#E1E1E1] !px-1"
											onClick={editEmailDisclosure.onOpen}
										>
											<svg
												fill="#958989"
												width="18px"
												height="18px"
												xmlns="http://www.w3.org/2000/svg"
												viewBox="0 0 24 24"
											>
												<path
													xmlns="http://www.w3.org/2000/svg"
													fillRule="evenodd"
													d="M13.793 5.293a3.828 3.828 0 1 1 5.414 5.414l-.998.999-.002.001-.001.002-9.499 9.498A1 1 0 0 1 8 21.5H4a1 1 0 0 1-1-1v-4a1 1 0 0 1 .293-.707l10.5-10.5ZM13.5 8.414l-8.5 8.5V19.5h2.586l8.5-8.5L13.5 8.414Zm4 1.172L14.914 7l.293-.293a1.829 1.829 0 0 1 2.586 2.586l-.293.293Z"
													clipRule="evenodd"
												></path>
											</svg>
										</Button>
									</form>
								</td>
							</tr>
							<tr className="py-5">
								<td className="py-4 px-6 text-left">
									{__('User ID:', 'colormag')}
								</td>
								<td className="py-4 px-6 text-left text-sm">
									<span>{localized.fs?.account?.data?.user_id}</span>
								</td>
								<td className="py-4 px-6 text-right"></td>
							</tr>
							<tr className="py-5">
								<td className="py-4 px-6 text-left">
									{__('Plugin', 'colormag')}
								</td>
								<td className="py-4 px-6 text-left text-sm">
									<span>{localized.fs?.account?.data?.product_id}</span>
								</td>
								<td className="py-4 px-6 text-right"></td>
							</tr>
							<tr className="py-5">
								<td className="py-4 px-6 text-left">
									{__('Site ID:', 'colormag')}
								</td>
								<td className="py-4 px-6 text-left text-sm">
									<span>{localized.fs?.account?.data?.site_id}</span>
								</td>
								<td className="py-4 px-6 text-right"></td>
							</tr>
							<tr className="py-5">
								<td className="py-4 px-6 text-left ">
									{__('Public Key:', 'colormag')}
								</td>
								<td className="py-4 px-6 text-left text-sm">
									<span>{localized.fs?.account?.data?.site_public_key}</span>
								</td>
								<td className="py-4 px-6 text-right"></td>
							</tr>
							<tr className="py-5">
								<td className="py-4 px-6 text-left">
									{__('Secret Key:', 'colormag')}
								</td>
								<td className="py-4 px-6 text-left">
									<span>
										{mask.secretKey
											? maskString(localized.fs?.account?.data?.site_secret_key)
											: localized.fs?.account?.data?.site_secret_key}
									</span>
								</td>
								<td className="py-4 px-6 text-right">
									<Button
										onClick={() =>
											setMask((prev) => ({
												...prev,
												secretKey: !prev.secretKey,
											}))
										}
									>
										{mask.secretKey ? (
											<svg
												xmlns="http://www.w3.org/2000/svg"
												fill="#000000"
												height="18px"
												width="18px"
												viewBox="0 0 56 56"
											>
												<path d="M 43.9492 47.3227 C 44.2544 47.6280 44.6821 47.7909 45.0686 47.7909 C 45.8832 47.7909 46.6361 47.0580 46.6361 46.2234 C 46.6361 45.8163 46.4735 45.4092 46.1679 45.1038 L 12.1120 11.0682 C 11.8066 10.7629 11.3995 10.6204 10.9924 10.6204 C 10.1781 10.6204 9.4250 11.3532 9.4250 12.1674 C 9.4250 12.5949 9.5675 13.0020 9.8728 13.2870 Z M 45.8628 41.5619 C 52.2546 37.4295 56.0000 32.0555 56.0000 29.6738 C 56.0000 25.5415 44.7025 12.3710 28.0102 12.3710 C 24.5497 12.3710 21.3130 12.9613 18.3410 13.9384 L 23.6540 19.2311 C 24.9771 18.6611 26.4428 18.3354 28.0102 18.3354 C 34.3207 18.3354 39.3892 23.3226 39.3892 29.6738 C 39.3892 31.2209 39.0636 32.7069 38.4324 34.0097 Z M 28.0102 46.9766 C 31.7761 46.9766 35.2774 46.3049 38.4124 45.2056 L 33.0179 39.8112 C 31.5318 40.5848 29.8219 41.0122 28.0102 41.0122 C 21.6591 41.0122 16.6310 35.8621 16.6107 29.6738 C 16.6107 27.8418 17.0382 26.1115 17.8117 24.5848 L 10.7278 17.4600 C 4.0102 21.5924 0 27.2310 0 29.6738 C 0 33.7858 11.5013 46.9766 28.0102 46.9766 Z M 34.4835 29.2463 C 34.4835 25.6840 31.6133 22.7934 28.0102 22.7934 C 27.7456 22.7934 27.4809 22.8137 27.2367 22.8341 L 34.4428 30.0402 C 34.4632 29.7960 34.4835 29.5110 34.4835 29.2463 Z M 21.5166 29.2056 C 21.5166 32.7883 24.4682 35.6789 28.0306 35.6789 C 28.3156 35.6789 28.5802 35.6586 28.8652 35.6382 L 21.5573 28.3303 C 21.5369 28.6153 21.5166 28.9206 21.5166 29.2056 Z" />
											</svg>
										) : (
											<svg
												xmlns="http://www.w3.org/2000/svg"
												fill="#000000"
												height="18px"
												width="18px"
												version="1.1"
												id="Capa_1"
												viewBox="0 0 488.85 488.85"
											>
												<g>
													<path d="M244.425,98.725c-93.4,0-178.1,51.1-240.6,134.1c-5.1,6.8-5.1,16.3,0,23.1c62.5,83.1,147.2,134.2,240.6,134.2   s178.1-51.1,240.6-134.1c5.1-6.8,5.1-16.3,0-23.1C422.525,149.825,337.825,98.725,244.425,98.725z M251.125,347.025   c-62,3.9-113.2-47.2-109.3-109.3c3.2-51.2,44.7-92.7,95.9-95.9c62-3.9,113.2,47.2,109.3,109.3   C343.725,302.225,302.225,343.725,251.125,347.025z M248.025,299.625c-33.4,2.1-61-25.4-58.8-58.8c1.7-27.6,24.1-49.9,51.7-51.7   c33.4-2.1,61,25.4,58.8,58.8C297.925,275.625,275.525,297.925,248.025,299.625z" />
												</g>
											</svg>
										)}
									</Button>
								</td>
							</tr>
							<tr className="py-5">
								<td className="py-4 px-6 text-left">
									{__('Version:', 'colormag')}
								</td>
								<td className="py-4 px-6 text-left text-sm">
									<span>{localized.fs?.account?.data?.version}</span>
									<div className="flex gap-2 mt-2 items-center">
										<label className="flex items-center gap-2 cursor-pointer">
											<input
												checked={beta}
												type="checkbox"
												onChange={(e) => {
													if (e.target.checked) {
														setBeta(true);
														betaDisclosure.onOpen();
													} else {
														setBeta(false);
														updateBetaProgramMutation.mutate(false);
													}
												}}
												className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 rounded focus:ring-blue-500 focus:ring-2"
											/>
											<span className="text-sm text-gray-700">
												{updateBetaProgramMutation.isLoading
													? __('Processing...', 'colormag')
													: __('Join the beta program', 'colormag')}
											</span>
										</label>
									</div>
								</td>
								<td className="py-4 px-6 text-right">
									{localized.fs?.account?.data?.update && (
										<Button
											variant={'primary'}
											size={'small'}
											className="p-0 h-auto"
											href={localized.fs?.account?.data?.update?.update_url}
										>
											{sprintf(
												__('Install Update Now [%s]', 'blockart'),
												localized.fs?.account?.data?.update?.update_version,
											)}
										</Button>
									)}
								</td>
							</tr>
							<tr className="py-5">
								<td className="py-4 px-6 text-left">
									{__('Plugin Plan:', 'colormag')}
								</td>
								<td className="py-4 px-6 text-left text-sm">
									<span>{localized.fs?.account?.data?.plan}</span>
								</td>
								<td className="py-4 px-6 text-right">
									<form
										action={window.location.href}
										method="post"
										ref={syncLicenseFormRef}
									>
										<div
											dangerouslySetInnerHTML={{
												__html: localized.fs?.account?.sync,
											}}
										/>
										<Button
											size={'default'}
											className=" h-auto border border-solid border-[#2563EB] px-3 py-[6px] rounded-ss-sm text-[#2563EB] hover:bg-[#2563EB] hover:text-white transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
											onClick={() => {
												setIsSyncing(true);
												syncLicenseFormRef.current?.requestSubmit();
											}}
											disabled={isSyncing}
										>
											{isSyncing ? (
												<div className="flex items-center gap-2">
													<svg className="animate-spin h-4 w-4 text-[#2563EB]" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
														<circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
														<path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
													</svg>
													{__('Syncing...', 'colormag')}
												</div>
											) : (
												__('Sync License', 'colormag')
											)}
										</Button>
									</form>
								</td>
							</tr>
							{localized.fs?.account?.data?.bundle_plan && (
								<tr className="py-5">
									<td className="py-4 px-6 text-left">
										{__('Bundle Plan:', 'colormag')}
									</td>
									<td className="py-4 px-6 text-left text-sm">
										{localized.fs?.account?.data?.bundle_plan}
									</td>
									<td className="py-4 px-6 text-right"></td>
								</tr>
							)}
							<tr className="py-5">
								<td className="py-4 px-6 text-left">
									{__('License Key:', 'colormag')}
								</td>
								<td className="py-4 px-6 text-left text-sm">
									<span>
										{mask.licenseKey
											? maskString(localized.fs?.account?.data?.license_key)
											: localized.fs?.account?.data?.license_key}
									</span>
								</td>
								<td className="py-4 px-6 text-right flex items-center gap-2 justify-end">
									<Button
										onClick={() =>
											setMask((prev) => ({
												...prev,
												licenseKey: !prev.licenseKey,
											}))
										}
									>
										{mask.licenseKey ? (
											<svg
												xmlns="http://www.w3.org/2000/svg"
												fill="#000000"
												height="18px"
												width="18px"
												viewBox="0 0 56 56"
											>
												<path d="M 43.9492 47.3227 C 44.2544 47.6280 44.6821 47.7909 45.0686 47.7909 C 45.8832 47.7909 46.6361 47.0580 46.6361 46.2234 C 46.6361 45.8163 46.4735 45.4092 46.1679 45.1038 L 12.1120 11.0682 C 11.8066 10.7629 11.3995 10.6204 10.9924 10.6204 C 10.1781 10.6204 9.4250 11.3532 9.4250 12.1674 C 9.4250 12.5949 9.5675 13.0020 9.8728 13.2870 Z M 45.8628 41.5619 C 52.2546 37.4295 56.0000 32.0555 56.0000 29.6738 C 56.0000 25.5415 44.7025 12.3710 28.0102 12.3710 C 24.5497 12.3710 21.3130 12.9613 18.3410 13.9384 L 23.6540 19.2311 C 24.9771 18.6611 26.4428 18.3354 28.0102 18.3354 C 34.3207 18.3354 39.3892 23.3226 39.3892 29.6738 C 39.3892 31.2209 39.0636 32.7069 38.4324 34.0097 Z M 28.0102 46.9766 C 31.7761 46.9766 35.2774 46.3049 38.4124 45.2056 L 33.0179 39.8112 C 31.5318 40.5848 29.8219 41.0122 28.0102 41.0122 C 21.6591 41.0122 16.6310 35.8621 16.6107 29.6738 C 16.6107 27.8418 17.0382 26.1115 17.8117 24.5848 L 10.7278 17.4600 C 4.0102 21.5924 0 27.2310 0 29.6738 C 0 33.7858 11.5013 46.9766 28.0102 46.9766 Z M 34.4835 29.2463 C 34.4835 25.6840 31.6133 22.7934 28.0102 22.7934 C 27.7456 22.7934 27.4809 22.8137 27.2367 22.8341 L 34.4428 30.0402 C 34.4632 29.7960 34.4835 29.5110 34.4835 29.2463 Z M 21.5166 29.2056 C 21.5166 32.7883 24.4682 35.6789 28.0306 35.6789 C 28.3156 35.6789 28.5802 35.6586 28.8652 35.6382 L 21.5573 28.3303 C 21.5369 28.6153 21.5166 28.9206 21.5166 29.2056 Z" />
											</svg>
										) : (
											<svg
												xmlns="http://www.w3.org/2000/svg"
												fill="#000000"
												height="18px"
												width="18px"
												version="1.1"
												id="Capa_1"
												viewBox="0 0 488.85 488.85"
											>
												<g>
													<path d="M244.425,98.725c-93.4,0-178.1,51.1-240.6,134.1c-5.1,6.8-5.1,16.3,0,23.1c62.5,83.1,147.2,134.2,240.6,134.2   s178.1-51.1,240.6-134.1c5.1-6.8,5.1-16.3,0-23.1C422.525,149.825,337.825,98.725,244.425,98.725z M251.125,347.025   c-62,3.9-113.2-47.2-109.3-109.3c3.2-51.2,44.7-92.7,95.9-95.9c62-3.9,113.2,47.2,109.3,109.3   C343.725,302.225,302.225,343.725,251.125,347.025z M248.025,299.625c-33.4,2.1-61-25.4-58.8-58.8c1.7-27.6,24.1-49.9,51.7-51.7   c33.4-2.1,61,25.4,58.8,58.8C297.925,275.625,275.525,297.925,248.025,299.625z" />
												</g>
											</svg>
										)}
									</Button>
									<Button
										className=" h-auto border border-solid border-[#2563EB] px-3 py-[6px] rounded-ss-sm text-[#2563EB]"
										onClick={updateLicenseDisclosure.onOpen}
									>
										{__('Change License', 'colormag')}
									</Button>
								</td>
							</tr>
						</tbody>
					</table>
				</div>
			</div>
			{editNameDisclosure.isOpen && (
				<Modal
					className="cm-modal-header"
					onRequestClose={editNameDisclosure.onClose}
				>
					<Controller
						name={'user_name'}
						control={form.control}
						render={({ field }) => (
							<InputControl
								className="mt-4 cm-username"
								label={__('Enter the new name:', 'colormag')}
								onChange={(v) => {
									field.onChange(v);
									if (undefined == v || !editNameFormRef.current) return;
									const _username =
										editNameFormRef.current.querySelector<HTMLInputElement>(
											'input[name="fs_user_name_colormag-theme"]',
										);
									if (_username) {
										_username.value = v;
									}
								}}
								value={field.value || ''}
								placeholder="Name"
							/>
						)}
					/>
					<div className="flex items-center justify-end gap-4">
						<Button
							variant={'primary'}
							size={'small'}
							onClick={editNameDisclosure.onClose}
							className="mt-4 py-4 border border-solid border-[#2563eb] rounded text-[#2563eb]"
						>
							{__('Close', 'colormag')}
						</Button>
						<Button
							type="submit"
							variant={'primary'}
							size={'small'}
							onClick={() => {
								if (!editNameFormRef.current) return;
								editNameFormRef.current.requestSubmit();
							}}
							className="mt-4 py-4 bg-[#2563eb] rounded text-[#FFFFFF] focus:text-[#FFFFFF]"
						>
							{__('Ok', 'colormag')}
						</Button>
					</div>
				</Modal>
			)}

			{editEmailDisclosure.isOpen && (
				<Modal
					className="cm-email-modal"
					onRequestClose={editEmailDisclosure.onClose}
				>
					<div className="text-center">
						<h1>{__('Email Address Update', 'colormag')}</h1>
						<p>
							{__(
								'Please enter the license key that you received in the email right after the purchase:',
								'colormag',
							)}
						</p>
					</div>
					<Controller
						name={'email'}
						control={form.control}
						rules={{
							required: true,
							validate: (value) => {
								return isValidEmail(value);
							},
						}}
						render={({ field }) => (
							<InputControl
								className="mt-4"
								label={__('Enter the new email address:', 'colormag')}
								onChange={(v) => {
									field.onChange(v);
									if (undefined == v || !editEmailFormRef.current) return;
									const _username =
										editEmailFormRef.current.querySelector<HTMLInputElement>(
											'input[name="fs_email_colormag-theme"]',
										);
									if (_username) {
										_username.value = v;
									}
								}}
								value={field.value || ''}
								placeholder="Email Address"
								type="email"
							/>
						)}
					/>
					{form.formState.errors.email && (
						<p className="text-red-500 text-sm mt-2">
							{form.formState.errors.email.message || __('Please enter a valid email address', 'colormag')}
						</p>
					)}
					{isValidEmail(watchEmail) && (
						<>
							<p>
								{sprintf(
									__('Are both %s and %s your email addresses?', 'colormag'),
									localized.fs?.account?.data?.email,
									watchEmail,
								)}
							</p>
							<Controller
								control={form.control}
								name="ownership"
								rules={{
									required: true,
								}}
								render={({ field }) => (
									<div className="flex flex-col gap-3">
										<label className="flex items-center gap-3 cursor-pointer">
											<input
												type="radio"
												value="both"
												name="ownership"
												checked={field.value === 'both'}
												onChange={field.onChange}
												className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 focus:ring-blue-500 focus:ring-2"
											/>
											<span className="text-sm text-gray-700">
												{__('Yes - both addresses are mine', 'colormag')}
											</span>
										</label>
										<label className="flex items-center gap-3 cursor-pointer">
											<input
												type="radio"
												value="client"
												name="ownership"
												checked={field.value === 'client'}
												onChange={field.onChange}
												className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 focus:ring-blue-500 focus:ring-2"
											/>
											<span className="text-sm text-gray-700">
												{sprintf(
													__(`%s is my clients's email address`, 'colormag'),
													watchEmail,
												)}
											</span>
										</label>
										<label className="flex items-center gap-3 cursor-pointer">
											<input
												type="radio"
												value="new"
												name="ownership"
												checked={field.value === 'new'}
												onChange={field.onChange}
												className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 focus:ring-blue-500 focus:ring-2"
											/>
											<span className="text-sm text-gray-700">
												{sprintf(
													__('%s is my email address', 'colormag'),
													watchEmail,
												)}
											</span>
										</label>
									</div>
								)}
							/>
							{form.formState.errors.ownership && (
								<p className="text-red-500 text-sm mt-2">
									{__('Please select an ownership option', 'colormag')}
								</p>
							)}
							{watchOwnership === 'both' && (
								<>
									<p>
										{sprintf(
											__('Would you like to merge %s and %s?', 'colormag'),
											localized.fs?.account?.data?.email,
											watchEmail,
										)}
									</p>
									<Controller
										control={form.control}
										name="transfer_type"
										rules={{
											required: watchOwnership === 'both',
										}}
										render={({ field }) => (
											<div className="flex flex-col gap-3">
												<label className="flex items-center gap-3 cursor-pointer">
													<input
														name="transfer_type"
														type="radio"
														value="merge"
														checked={field.value === 'merge'}
														onChange={field.onChange}
														className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 focus:ring-blue-500 focus:ring-2"
													/>
													<span className="text-sm text-gray-700">
														{sprintf(
															__(
																`Yes - move all my data and assets from %s to %s`,
																'colormag',
															),
															localized.fs?.account?.data?.email,
															watchEmail,
														)}
													</span>
												</label>
												<label className="flex items-center gap-3 cursor-pointer">
													<input
														name="transfer_type"
														type="radio"
														value="transfer"
														checked={field.value === 'transfer'}
														onChange={field.onChange}
														className="w-4 h-4 text-blue-600 bg-gray-100 border-gray-300 focus:ring-blue-500 focus:ring-2"
													/>
													<span className="text-sm text-gray-700">
														{sprintf(
															__(
																`No - only move this site's data to %s`,
																'colormag',
															),
															watchEmail,
														)}
													</span>
												</label>
											</div>
										)}
									/>
									{form.formState.errors.transfer_type && (
										<p className="text-red-500 text-sm mt-2">
											{__('Please select a transfer type', 'colormag')}
										</p>
									)}
								</>
							)}
						</>
					)}
					<div className="flex items-center justify-end gap-3 flex-col">
						<Button
							variant={'primary'}
							size={'default'}
							onClick={async () => {
								if (!editEmailFormRef.current) return;
								const isValid = await form.trigger();
								if (!isValid) {
									return;
								}
								try {
									await emailAddressProgramMutation.mutateAsync();
								} catch (error) {
									console.error('Email update failed:', error);
								}
							}}
							disabled={emailAddressProgramMutation.isLoading || !form.formState.isValid}
							className="mt-4 py-3 bg-[#2563eb] rounded text-[#FFFFFF] focus:text-[#FFFFFF] w-full justify-center h-auto disabled:opacity-50 disabled:cursor-not-allowed"
						>
							{emailAddressProgramMutation.isLoading ? (
								<div className="flex items-center gap-2">
									<svg className="animate-spin h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
										<circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
										<path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
									</svg>
									{__('Updating...', 'colormag')}
								</div>
							) : (
								__('Update', 'colormag')
							)}
						</Button>
						<Button
							variant={'secondary'}
							size={'default'}
							onClick={editEmailDisclosure.onClose}
							disabled={emailAddressProgramMutation.isLoading}
							className="py-3 border border-solid border-[#2563eb] rounded text-[#2563eb] w-full justify-center h-auto disabled:opacity-50 disabled:cursor-not-allowed"
						>
							{__('Close', 'colormag')}
						</Button>
					</div>
				</Modal>
			)}

			{updateLicenseDisclosure.isOpen && (
				<Modal
					className="cm-email-modal"
					onRequestClose={updateLicenseDisclosure.onClose}
				>
					<div className="text-center">
						<h1>{__('Update License Key', 'colormag')}</h1>
						<p>
							{__(
								'Please enter the new license key that you received in the email right after the purchase:',
								'colormag',
							)}
						</p>
					</div>
					<div className="mt-4">
						<Controller
							name={'license_key'}
							control={form.control}
							rules={{
								required: true,
								validate: (value) => {
									return value && value.length === 32;
								},
							}}
							render={({ field }) => (
								<InputControl
									className="w-full"
									label={__('Enter the new license key:', 'colormag')}
									onChange={field.onChange}
									value={field.value || ''}
									placeholder="Enter License Key"
									type="password"
								/>
							)}
						/>
						{form.formState.errors.license_key && (
							<p className="text-red-500 text-sm mt-2">
								{form.formState.errors.license_key.message || __('Please enter a valid 32-character license key', 'colormag')}
							</p>
						)}
					</div>
					<div className="flex items-center justify-end gap-3 flex-col mt-4">
						<Button
							variant={'primary'}
							size={'default'}
							onClick={async () => {
								const isValid = await form.trigger('license_key');
								if (!isValid) {
									return;
								}
								const licenseKey = form.getValues().license_key;
								if (licenseKey && licenseKey.length === 32) {
									try {
										await updateLicenseMutation.mutateAsync(licenseKey);
									} catch (error) {
										console.error('License update failed:', error);
									}
								}
							}}
							disabled={updateLicenseMutation.isLoading || !form.formState.isValid}
							className="mt-4 py-3 bg-[#2563eb] rounded text-[#FFFFFF] focus:text-[#FFFFFF] w-full justify-center h-auto disabled:opacity-50 disabled:cursor-not-allowed"
						>
							{updateLicenseMutation.isLoading ? (
								<div className="flex items-center gap-2">
									<svg className="animate-spin h-4 w-4 text-white" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
										<circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
										<path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
									</svg>
									{__('Updating...', 'colormag')}
								</div>
							) : (
								__('Update License', 'colormag')
							)}
						</Button>
						<Button
							variant={'secondary'}
							size={'default'}
							onClick={updateLicenseDisclosure.onClose}
							disabled={updateLicenseMutation.isLoading}
							className="py-3 border border-solid border-[#2563eb] rounded text-[#2563eb] w-full justify-center h-auto disabled:opacity-50 disabled:cursor-not-allowed"
						>
							{__('Close', 'colormag')}
						</Button>
					</div>
				</Modal>
			)}

			{betaDisclosure.isOpen && (
				<Modal
					className="cm-modal-header"
					onRequestClose={betaDisclosure.onClose}
				>
					<div className="text-center">
						<h2>{__('Join Beta Program', 'colormag')}</h2>
						<p>
							{__(
								'By joining the beta program, you will receive early access to new features and updates. These versions may be less stable than the regular releases. Are you sure you want to join?',
								'colormag',
							)}
						</p>
					</div>
					<div className="flex items-center justify-end gap-4">
						<Button
							variant={'secondary'}
							size={'small'}
							onClick={betaDisclosure.onClose}
							className="mt-4 py-4 border border-solid border-[#2563eb] rounded text-[#2563eb]"
						>
							{__('Cancel', 'colormag')}
						</Button>
						<Button
							variant={'primary'}
							size={'small'}
							onClick={() => {
								updateBetaProgramMutation.mutate(true);
								betaDisclosure.onClose();
							}}
							disabled={updateBetaProgramMutation.isLoading}
							className="mt-4 py-4 bg-[#2563eb] rounded text-[#FFFFFF] focus:text-[#FFFFFF]"
						>
							{updateBetaProgramMutation.isLoading
								? __('Joining...', 'colormag')
								: __('Join Beta Program', 'colormag')}
						</Button>
					</div>
				</Modal>
			)}

			{deactivateLicenseDisclosure.isOpen && (
				<Modal
					className="cm-modal-header"
					onRequestClose={deactivateLicenseDisclosure.onClose}
					style={{
						maxWidth: '450px',
					}}
				>
					<p>
						{' '}
						{__(
							'Deactivating your license will block all premium features, but will enable activating the license on another site. Are you sure you want to proceed?',
							'colormag',
						)}
					</p>
					<div className="flex items-center justify-center gap-4">
						<Button
							variant={'secondary'}
							size={'default'}
							onClick={deactivateLicenseDisclosure.onClose}
							className="mt-4 py-4 px-4 border border-solid border-[#2563eb] rounded text-[#2563eb] text-base hover:bg-[#fafafc]"
						>
							{__('Close', 'colormag')}
						</Button>
						<Button
							size={'default'}
							onClick={() => {
								deactivateLicenseFormRef.current?.requestSubmit();
								// Redirect to license page after deactivation
								setTimeout(() => {
									window.location.href = 'admin.php?page=colormag#/license';
								}, 1000);
							}}
							className="mt-4 py-4 px-4 text-base  rounded text-[#FFFFFF] focus:text-[#FFFFFF] bg-red-500 hover:bg-red-600 hover:border-none hover:!text-[#FFFFFF] active:!text-[#FFFFFF]"
						>
							{__('Ok', 'colormag')}
						</Button>
					</div>
				</Modal>
			)}
		</div>
	);
};
