<?php

/**
 * Basic wrappers around WP core functions
 *
 * This file is loaded early by the theme
 *
 * @package vamtam/evergreen
 */

/**
 * get_option wrapper
 *
 * @param  string $name option   name
 * @param  mixed  $default       default value
 * @param  bool   $stripslashes  whether to filter the result with stripslashes()
 *
 * @return mixed                 option value
 */

function vamtam_get_option( $name, $sub = null ) {
	global $vamtam_theme, $vamtam_defaults;

	$option = $vamtam_theme[ $name ] ?? ( $vamtam_defaults[ $name ] ?? '' );

	if ( ! is_null( $sub ) && is_array( $option ) ) {
		$option = $option[ $sub ];
	}

	if ( is_string( $option ) ) {
		if ( $option === 'true' ) {
			$option = true;
		} elseif ( $option === 'false' ) {
			$option = false;
		}
	}

	return apply_filters( 'vamtam_get_option', $option, $name, $sub );
}

/**
 * Same as vamtam_get_option, but converts '1' and '0' to booleans
 *
 * @uses   vamtam_get_option()
 *
 * @param  string $name option   name
 * @param  mixed  $default       default value
 * @param  bool   $stripslashes  whether to filter the result with stripslashes()
 *
 * @return mixed                 option value
 */
function vamtam_get_optionb( $name, $sub = null ) {
	$value = vamtam_get_option( $name, $sub );

	if ( $value === '1' || $value === 'true' ) {
		return true;
	}

	if ( $value === '0' || $value === 'false' ) {
		return false;
	}

	return is_bool( $value ) ? $value : false;
}

/**
 * Converts '1', '0', 'true' and 'false' to booleans, otherwise returns $value
 * @param  mixed $value original value
 * @return mixed        sanitized value
 */
function vamtam_sanitize_bool( $value ) {
	if ( $value === '1' || $value === 'true' ) {
		return true;
	}

	if ( $value === '0' || $value === 'false' ) {
		return false;
	}

	return $value;
}

/*
 * Developer/testing license override.
 *
 * Behavior:
 *  - If you define VAMTAM_DEV_ALWAYS_ACTIVE as true in wp-config.php OR WP_DEBUG is enabled,
 *    the theme will:
 *      - Force the internal license option to "VALIDATED"
 *      - Add a filter to override import license checks
 *      - Ensure a purchase-code option exists (set to "VAMTAM-DEV") so import code that expects it doesn't fail
 *
 * Usage (recommended for local dev only):
 *  - In wp-config.php add: define( 'VAMTAM_DEV_ALWAYS_ACTIVE', true );
 *  - Or enable WP_DEBUG in wp-config.php
 *
 * IMPORTANT: Remove the constant / disable WP_DEBUG before deploying to production.
 */
if ( ( defined( 'VAMTAM_DEV_ALWAYS_ACTIVE' ) && VAMTAM_DEV_ALWAYS_ACTIVE ) || ( defined( 'WP_DEBUG' ) && WP_DEBUG ) ) {
	// Make import checks that use the vamtam_purchase_code_import_override filter pass.
	add_filter( 'vamtam_purchase_code_import_override', function( $v ) { return true; } );

	// Ensure the internal license option is set to VALIDATED so checks that call VamtamFramework::license() see it as valid.
	if ( get_option( '_vamtam_license' ) !== 'VALIDATED' ) {
		update_option( '_vamtam_license', 'VALIDATED' );
	}

	// Ensure a purchase code option exists so code that expects it won't fail.
	if ( class_exists( 'VamtamFramework' ) ) {
		$purchase_key_opt = VamtamFramework::get_purchase_code_option_key();
		if ( false === get_option( $purchase_key_opt, false ) ) {
			update_option( $purchase_key_opt, 'BABIATO' );
		}
	} else {
		// Fallback: set legacy option name if framework class isn't available yet.
		if ( false === get_option( 'vamtam-envato-license-key', false ) ) {
			update_option( 'vamtam-envato-license-key', 'BABIATO' );
		}
	}
}
