<?php
/**
 *  Expedition Elementor Functions
 *
 * @author: Mountain-Themes
 * @version: 2.0.0
 */

 if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Expedition_Elementor_Widget {

	 /**
	 * Minimum Elementor Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum Elementor version required to run the plugin.
	 */
	const MINIMUM_ELEMENTOR_VERSION = '3.5.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.0.0
	 *
	 * @var string Minimum PHP version required to run the plugin.
	 */
	const MINIMUM_PHP_VERSION = '7.5';

	/**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 *
	 * @var Expedition_Elementor_Widget The single instance of the class.
	 */
	private static $instance = null;

	public static function get_instance() {
	    if ( ! self::$instance )
	       self::$instance = new self;
	    return self::$instance;
	}

	public function init(){

		// Check if Elementor installed and activated
		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', array( $this, 'admin_notice_missing_main_plugin' ) );
			return;
		}

		// Check for required Elementor version
		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', array( $this, 'admin_notice_minimum_elementor_version' ) );
			return;
		}

		// Check for required PHP version
		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', array( $this, 'admin_notice_minimum_php_version' ) );
			return;
		}

		add_action( 'elementor/widgets/widgets_registered', array( $this, 'widgets_registered' ) );
		add_action( 'elementor/widgets/widgets_registered', array( $this, 'register_extensions' ) );
		add_action( 'elementor/frontend/before_register_scripts', array($this, 'register_frontend_scripts' ) );
		add_action( 'elementor/frontend/after_enqueue_styles', array( $this, 'register_frontend_styles' ) );
		add_action( 'elementor/editor/after_enqueue_styles', array( $this, 'register_editor_styles' ) );
		add_action( 'elementor/elements/categories_registered', array( $this, 'elementor_widget_categories' ) );
		add_action( 'elementor/theme/register_locations', [ $this, 'register_elementor_locations' ] );
		add_action( 'wp_enqueue_scripts', [ $this, 'register_widget_scripts' ] );
		add_action( 'elementor/controls/controls_registered', [ $this, 'init_controls' ] );
	}

	public function widgets_registered() {
		//Require all PHP files in the /elementor/widgets directory
		foreach( glob( plugin_dir_path( __FILE__ ) . "widgets/*.php" ) as $file ) {
		    require $file;
		}
	}

	private function include_extensions_files() {
		require_once plugin_dir_path( __FILE__ ) . 'extensions/icons-integration.php';
		require_once plugin_dir_path( __FILE__ ) . 'extensions/aos-animation.php';
	}

	public function register_extensions() {
		$this->include_extensions_files();
		new \Elementor\Icons_Integration();
		new \Elementor\AOS_Animation();
	}

	public function register_elementor_locations( $elementor_theme_manager ) {
		$elementor_theme_manager->register_location( 'header' );
		$elementor_theme_manager->register_location( 'footer' );
		$elementor_theme_manager->register_location( '404' );
	}

	public function register_frontend_scripts() {
		wp_enqueue_script( 'expedition-frontend-widgets-scripts',  plugin_dir_url( __FILE__ ) . 'assets/js/front-end-widgets.js', array('jquery'), false, true);
		wp_enqueue_script( 'expedition-aos-scripts',  plugin_dir_url( __FILE__ ) . 'assets/js/aos.min.js', array('jquery'), false, true);
	}

	public function register_frontend_styles() {
		wp_enqueue_style( 'expedition-aos-style', plugin_dir_url( __FILE__ ) . 'assets/css/aos.min.css', null, 1.0 );
	}

	public function register_editor_styles() {
		wp_enqueue_style( 'expedition-elementor-style', plugin_dir_url( __FILE__ ) . 'assets/css/style.css', null, 1.0 );
	}

	public function elementor_widget_categories( $elements_manager ) {
		$elements_manager->add_category(
			'expedition-category',
			[
				'title' => esc_html__( 'Expedition Elements', 'expedition-plugin' ),
			]
		);
	}

	/**
	 * Init Controls
	 *
	 * Include control files and register them
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init_controls() {

		// Include Widget files
		require_once plugin_dir_path( __FILE__ ) . 'controls/fileselect-control.php';

		// Register controls
		\Elementor\Plugin::$instance->controls_manager->register_control( 'file-select', new \FileSelect_Control() );
	}

	// Lib and effects for widget Content Cover
	public function register_widget_scripts() {

		// Effects
		wp_register_script( 'winter-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/winter.min.js', null, 1.0  );
		wp_register_script( 'low-poly-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/low-poly.min.js', null, 1.0 );
		wp_register_script( 'constellation-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/constellation.min.js', null, 1.0 );
		wp_register_script( 'particles-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/particles.min.js', null, 1.0 );
		wp_register_script( 'rainbow-squares-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/rainbow-squares.min.js', null, 1.0 );
		wp_register_script( 'shooting-stars-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/shooting-stars.min.js', null, 1.0 );
		wp_register_script( 'starfield-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/starfield.min.js', null, 1.0 );
		wp_register_script( 'gravity-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/gravity.min.js', null, 1.0 );
		wp_register_script( 'fireworks-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/fireworks.min.js', null, 1.0 );
		wp_register_script( 'geometric-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/geometric.min.js', null, 1.0 );
		wp_register_script( 'bubbles-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/bubbles.min.js', null, 1.0 );
		wp_register_script( 'mosaic-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/mosaic.min.js', null, 1.0 );
		wp_register_script( 'rain-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/rain.min.js', null, 1.0 );
		wp_register_script( 'surface-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/surface.min.js', null, 1.0 );
		wp_register_script( 'magneto-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/magneto.min.js', null, 1.0 );
		wp_register_script( 'liquid-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/liquid.min.js', null, 1.0 );
		wp_register_script( 'confetti-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/confetti.min.js', null, 1.0 );
		wp_register_script( 'orbit-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/orbit.min.js', null, 1.0 );
		wp_register_script( 'swirl-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/swirl.min.js', null, 1.0 );
		wp_register_script( 'shift-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/shift.min.js', null, 1.0 );
		wp_register_script( 'coalesce-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/coalesce.min.js', null, 1.0 );
		wp_register_script( 'pipeline-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/pipeline.min.js', null, 1.0 );
		wp_register_script( 'aurora-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/aurora.min.js', null, 1.0 );
		wp_register_script( 'stardust-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/stardust.min.js', null, 1.0 );
		wp_register_script( 'abstractart-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/abstractart.min.js', null, 1.0 );
		wp_register_script( 'wave-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/wave.min.js', null, 1.0 );
		wp_register_script( 'connections-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/connections.min.js', null, 1.0 );
		wp_register_script( 'alien-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/alien.min.js', null, 1.0 );
		wp_register_script( 'balls-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/balls.min.js', null, 1.0 );
		wp_register_script( 'landscape-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/landscape.min.js', null, 1.0 );
		wp_register_script( 'boreal-sky-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/boreal-sky.min.js', null, 1.0 );
		wp_register_script( 'colors-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/colors.min.js', null, 1.0 );
		wp_register_script( 'confetti-lines-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/confetti-lines.min.js', null, 1.0 );
		wp_register_script( 'stars-lines-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/stars-lines.min.js', null, 1.0 );
		wp_register_script( 'quantum-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/quantum.min.js', null, 1.0 );
		wp_register_script( 'information-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/information.min.js', null, 1.0 );
		wp_register_script( 'spacetime-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/spacetime.min.js', null, 1.0 );
		wp_register_script( 'physics-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/physics.min.js', null, 1.0 );
		wp_register_script( 'heuristics-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/heuristics.min.js', null, 1.0 );
		wp_register_script( 'star-wars-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/star-wars.min.js', null, 1.0 );
		wp_register_script( 'brick-tunnel-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/brick-tunnel.min.js', null, 1.0 );
		wp_register_script( 'triangle-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/triangle.min.js', null, 1.0 );
		wp_register_script( 'birds-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/birds.min.js', null, 1.0 );
		wp_register_script( 'fog-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/fog.min.js', null, 1.0 );
		wp_register_script( 'waves-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/waves.min.js', null, 1.0 );
		wp_register_script( 'clouds-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/clouds.min.js', null, 1.0 );
		wp_register_script( 'globe-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/globe.min.js', null, 1.0 );
		wp_register_script( 'net-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/net.min.js', null, 1.0 );
		wp_register_script( 'cells-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/cells.min.js', null, 1.0 );
		wp_register_script( 'trunk-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/trunk.min.js', null, 1.0 );
		wp_register_script( 'topology-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/topology.min.js', null, 1.0 );
		wp_register_script( 'dots-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/dots.min.js', null, 1.0 );
		wp_register_script( 'rings-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/rings.min.js', null, 1.0 );
		wp_register_script( 'halo-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/halo.min.js', null, 1.0 );
		wp_register_script( 'glitch-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/glitch.min.js', null, 1.0 );
		wp_register_script( 'sand-particles-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/sand-particles.min.js', null, 1.0 );
		wp_register_script( 'hills-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/hills.min.js', null, 1.0 );
		wp_register_script( 'plasmatic-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/plasmatic.min.js', null, 1.0 );
		wp_register_script( 'coronavirus-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/coronavirus.min.js', null, 1.0 );
		wp_register_script( 'audio-visualizers-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/audio-visualizers.min.js', null, 1.0 );
		wp_register_script( 'water-droplet-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/water-droplet.min.js', null, 1.0 );
		wp_register_script( 'noisy-blobs-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/noisy-blobs.min.js', null, 1.0 );
		wp_register_script( 'nucleus-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/nucleus.min.js', null, 1.0 );
		wp_register_script( 'tv-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/tv.min.js', null, 1.0 );
		wp_register_script( 'floor-lava-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/floor-lava.min.js', null, 1.0 );
		wp_register_script( 'circuits-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/circuits.min.js', null, 1.0 );
		wp_register_script( 'turbulent-noise-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/turbulent-noise.min.js', null, 1.0 );
		wp_register_script( 'ribbons-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/ribbons.min.js', null, 1.0 );
		wp_register_script( 'dotted-ocean-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/dotted-ocean.min.js', null, 1.0 );
		wp_register_script( 'discoid-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/discoid.min.js', null, 1.0 );
		wp_register_script( 'mechanical-grass-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/mechanical-grass.min.js', null, 1.0 );
		wp_register_script( 'gradient-animation-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/gradient-animation.min.js', null, 1.0 );
		wp_register_script( 'funky-perlin-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/funky-perlin.min.js', null, 1.0 );
		wp_register_script( 'snowfall-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/snowfall.min.js', null, 1.0 );
		wp_register_script( 'trails-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/trails.min.js', null, 1.0 );
		wp_register_script( 'blood-flow-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/blood-flow.min.js', null, 1.0 );
		wp_register_script( 'night-sky-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/night-sky.min.js', null, 1.0 );
		wp_register_script( 'text-particles-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/text-particles.min.js', null, 1.0 );
		wp_register_script( 'pixel-grid-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/pixel-grid.min.js', null, 1.0 );
		wp_register_script( 'swarming-bugs-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/swarming-bugs.min.js', null, 1.0 );
		wp_register_script( 'blobs-effect', plugin_dir_url( __FILE__ ) . 'assets/js/effects/blobs.min.js', null, 1.0 );

		// Lib
		wp_register_script( 'raf-lib', plugin_dir_url( __FILE__ ) . 'assets/js/lib/rAF.min.js', null, 1.0  );
		wp_register_script( 'sketch-lib', plugin_dir_url( __FILE__ ) . 'assets/js/lib/sketch.min.js', null, 1.0  );
		wp_register_script( 'lo-dash-lib', plugin_dir_url( __FILE__ ) . 'assets/js/lib/lo-dash.min.js', null, 1.0  );
		wp_register_script( 'noise-lib', plugin_dir_url( __FILE__ ) . 'assets/js/lib/noise.min.js', null, 1.0  );
		wp_register_script( 'filter-polyfill-lib', plugin_dir_url( __FILE__ ) . 'assets/js/lib/filter-polyfill.min.js', null, 1.0  );
		wp_register_script( 'pixi-lib', plugin_dir_url( __FILE__ ) . 'assets/js/lib/pixi.min.js', null, 1.0  );
		wp_register_script( 'matter-lib', plugin_dir_url( __FILE__ ) . 'assets/js/lib/matter.min.js', null, 1.0  );
		wp_register_script( 'gsap-lib', plugin_dir_url( __FILE__ ) . 'assets/js/lib/gsap.min.js', null, 1.0  );
		wp_register_script( 'perlin-lib', plugin_dir_url( __FILE__ ) . 'assets/js/lib/perlin.min.js', null, 1.0  );
		wp_register_script( 'backstretch-lib', plugin_dir_url( __FILE__ ) . 'assets/js/lib/backstretch.min.js', null, 1.0  );
		wp_register_script( 'ytplayer-lib', plugin_dir_url( __FILE__ ) . 'assets/js/lib/jquery.mb.YTPlayer.min.js', null, 1.0  );
		wp_register_script( 'vimeo-player-lib', plugin_dir_url( __FILE__ ) . 'assets/js/lib/jquery.mb.vimeo_player.min.js', null, 1.0  );
		wp_register_script( 'html5-videobg-lib', plugin_dir_url( __FILE__ ) . 'assets/js/lib/html5-videobg.min.js', null, 1.0  );
		wp_register_script( 'countdown-lib', plugin_dir_url( __FILE__ ) . 'assets/js/lib/countdown.min.js', null, 1.0  );
		wp_register_script( 'chroma-lib', plugin_dir_url( __FILE__ ) . 'assets/js/lib/chroma.min.js', null, 1.0  );
		wp_register_script( 'doyle-spirals-lib', plugin_dir_url( __FILE__ ) . 'assets/js/lib/doyle-spirals.min.js', null, 1.0  );
		wp_register_script( 'matrix-lib', plugin_dir_url( __FILE__ ) . 'assets/js/lib/matrix.min.js', null, 1.0  );
		wp_register_script( 'gradientify-lib', plugin_dir_url( __FILE__ ) . 'assets/js/lib/gradientify.min.js', null, 1.0  );
		wp_register_script( 'shader-program-lib', plugin_dir_url( __FILE__ ) . 'assets/js/lib/shader-program.min.js', null, 1.0  );
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {
		if ( isset( $_GET['activate'] ) ) {
			unset( $_GET['activate'] );
		}

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'expedition-plugin' ),
			'<strong>' . esc_html__( 'Expedition Helper Plugin', 'expedition-plugin' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'expedition-plugin' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {
		if ( isset( $_GET['activate'] ) ) {
			unset( $_GET['activate'] );
		}

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'expedition-plugin' ),
			'<strong>' . esc_html__( 'Expedition Helper Plugin', 'expedition-plugin' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'expedition-plugin' ) . '</strong>',
			self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );
	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {
		if ( isset( $_GET['activate'] ) ) {
			unset( $_GET['activate'] );
		}

		$message = sprintf(
			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'expedition-plugin' ),
			'<strong>' . esc_html__( 'Expedition Helper Plugin', 'expedition-plugin' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'expedition-plugin' ) . '</strong>',
			self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-error is-dismissible"><p>%1$s</p></div>', $message );
	}

}

Expedition_Elementor_Widget::get_instance()->init();
