<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Expedition Google Map Widget
 *
 * @author: Mountain-Themes
 * @version: 2.0.0
 */

class Expedition_Google_Map_Widget extends Widget_Base {

	public function get_name() {
		return 'expedition-google-map';
	}

	public function get_title() {
		return esc_html__( 'Google Map', 'expedition-plugin' );
	}

	public function get_icon() {
		return 'eicon-google-maps expedition-badge';
	}

	public function get_categories() {
		return [ 'expedition-category' ];
	}

	public function get_keywords() {
		return [ 'map', 'google', 'locate' ];
	}

	/**
	 * Register widget controls.
	 */

	protected function register_controls() {
		
		$this->start_controls_section(
			'section_map', [
				'label' => esc_html__( 'Google Map', 'expedition-plugin' ),
			]
		);

        $this->add_control(
			'important_note',
			[
				'label' => '',
				'type' => Controls_Manager::RAW_HTML,
				'raw' => esc_html__( 'Enter your Google Map Api Key in Customizing -> General Settings -> Additional Settings -> Google Map API Key ', 'expedition-plugin' ),
				'content_classes' => 'elementor-descriptor',
			]
		);

		$this->add_control(
			'map_latitude', [
				'label' => esc_html__( 'Latitude', 'expedition-plugin' ),
				'description' => __( '<a href="https://www.latlong.net/" target="_blank">Here is a tool</a> where you can find Latitude &amp; Longitude of your location.' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => '-37.8602828',
				'default' => '-37.8602828',
			]
		);

		$this->add_control(
			'map_longitude', [
				'label' => esc_html__( 'Longitude', 'expedition-plugin' ),
				'description' => __( '<a href="https://www.latlong.net/" target="_blank">Here is a tool</a> where you can find Latitude &amp; Longitude of your location.' ),
				'type' => Controls_Manager::TEXT,
				'placeholder' => '145.079616',
				'default' => '145.079616',
			]
		);

		$this->add_control(
			'map_zoom', [
				'label' => esc_html__( 'Zoom Level', 'expedition-plugin' ),
				'type' => Controls_Manager::SLIDER,
				'default' => [
					'size' => 10,
				],
				'range' => [
					'px' => [
						'min' => 1,
						'max' => 25,
					],
				],
				'separator' => 'before',
			]
		);


        $this->add_responsive_control(
			'map_height', [
				'label' => esc_html__( 'Height', 'expedition-plugin' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'em', 'rem', '%', 'vh' ],
				'default' => [
					'unit' => 'px',
					'size' => 450,
				],
				'range' => [
					'vh' => [
						'min' => 10,
						'max' => 100,
					],
					'px' => [
						'min' => 40,
						'max' => 1440,
					],
				],
				'selectors' => [
					'{{WRAPPER}} #google-container' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);

        $this->add_control(
			'map_type', [
				'label' => esc_html__( 'Map Type', 'expedition-plugin' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'roadmap' => esc_html__( 'Road Map', 'expedition-plugin' ),
					'satellite' => esc_html__( 'Satellite', 'expedition-plugin' ),
					'hybrid' => esc_html__( 'Hybrid', 'expedition-plugin' ),
					'terrain' => esc_html__( 'Terrain', 'expedition-plugin' ),
				],
				'default' => 'roadmap',
				'separator' => 'before',
			]
		);

        $this->add_control(
			'map_zoom_control', [
				'label' => esc_html__( 'Zoom Control', 'expedition-plugin' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
				'separator' => 'before',
			]
		);

        $this->add_control(
			'zoom_icon_color',
			[
				'label' => esc_html__( 'Zoom color', 'expedition-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => 'rgba(61, 74, 110, 0.9)',
                'condition' => [
					'map_zoom_control' => 'yes',
				],
				'selectors' => [
					'{{WRAPPER}} #zoom-in' => 'background-color: {{VALUE}};',
                    '{{WRAPPER}} #zoom-out' => 'background-color: {{VALUE}};',
				],
			]
		);

        $this->add_control(
			'zoom_icon_hover_color',
			[
				'label' => esc_html__( 'Zoom hover color', 'expedition-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '#3d4a6e',
                'condition' => [
					'map_zoom_control' => 'yes',
				],
				'selectors' => [
					'{{WRAPPER}} #zoom-in:hover' => 'background-color: {{VALUE}};',
                    '{{WRAPPER}} #zoom-out:hover' => 'background-color: {{VALUE}};',
				],
			]
		);
        
        $this->add_control(
			'map_marker_title', [
				'label' => esc_html__( 'Marker Title', 'expedition-plugin' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__( 'Melbourne, Australia' , 'expedition-plugin' ),
				'label_block' => true,
			]
		);

        $this->add_control(
			'map_marker_icon',
			[
				'label' => esc_html__( 'Marker icon', 'expedition-plugin' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => '',
				],
			]
		);

        $this->add_control(
			'map_info_window', [
				'label' => esc_html__( 'Info Window', 'expedition-plugin' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'no',
				'separator' => 'before',
			]
		);

        $this->add_control(
			'map_title_info_window', [
				'label' => esc_html__( 'Title Info Window', 'expedition-plugin' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__( 'CONTACT INFORMATION.' , 'expedition-plugin' ),
                'condition' => [
					'map_info_window' => 'yes',
				],
				'label_block' => true,
			]
		);

		$repeater = new Repeater();

		$repeater->add_control(
			'info_icon',
			[
				'label' => esc_html__( 'Icon', 'expedition-plugin' ),
				'type' => Controls_Manager::ICONS,
				'fa4compatibility' => 'icon',
			]
		);	

		$repeater->add_control(
			'info_text', [
				'label' => esc_html__( 'Info Text', 'expedition-plugin' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__( 'Info Text' , 'expedition-plugin' ),
				'label_block' => true,
			]
		);

		$repeater->add_control(
			'map_info_link_check', [
				'label' => esc_html__( 'Info link', 'expedition-plugin' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'no',
				'separator' => 'before',
			]
		);
		
		$repeater->add_control(
			'info_link', [
				'label' => esc_html__( 'Link', 'expedition-plugin' ),
				'type' => Controls_Manager::URL,
				'placeholder' => 'https://your-link.com',
				'default' => [
					'url'=> '#',
				],
				'condition' => [
					'map_info_link_check' => 'yes',
				],
			]
		);

		$this->add_control(
			'info_list',
			[
				'label' => esc_html__( 'Info List', 'expedition-plugin' ),
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[	
						'info_icon' => [
							'value' => ' fa-solid fa-location-dot',
							'library' => 'faprosolid'
						],
						'info_text' => esc_html__( 'Melbourne, Australia', 'expedition-plugin' ),
						'map_info_link_check' => 'no',
						'info_link' => [ 'url' => 'https://your-link.com' ]
					],
						[	'info_icon' => [
							'value' => 'fa-solid fa-phone',
							'library' => 'faprosolid'
						],
						'info_text' => esc_html__( '765-302-2878', 'expedition-plugin' ),
						'map_info_link_check' => 'no',
						'info_link' => [ 'url' => 'https://your-link.com' ]
					],
					[
						'info_icon' => [
							'value' => 'fa-solid fa-paper-plane',
							'library' => 'faprosolid'
						],
						'info_text' => esc_html__( 'name@domain.com', 'expedition-plugin' ),
						'map_info_link_check' => 'yes',
						'info_link' => [ 'url' => 'https://your-link.com' ]
					],
					[
						'info_icon' => [
							'value' => 'fa-solid fa-globe',
							'library' => 'faprosolid'
						],
						'info_text' => esc_html__( 'mycompanyname.com', 'expedition-plugin' ),
						'map_info_link_check' => 'yes',
						'info_link' => [ 'url' => 'https://your-link.com' ]
					],
				],
				'condition' => [
					'map_info_window' => 'yes',
				],
				'title_field' => '{{{ info_text }}}',
			]
		);

        $this->add_control(
			'map_style', [
				'label' => esc_html__( 'Map Style', 'expedition-plugin' ),
				'type' => Controls_Manager::TEXTAREA,
				'description' => __( 'Add style from <a href="https://mapstyle.withgoogle.com/" target="_blank">Google Map Styling Wizard</a> or <a href="https://snazzymaps.com/explore" target="_blank">Snazzy Maps</a>. Copy and Paste the style in the textarea.' ),
				'condition' => [
					'map_type' => 'roadmap',
				],
				'separator' => 'before'
			]
		);

		$this->add_control(
			'contact_info_window', [
				'label' => esc_html__( 'Contact info', 'expedition-plugin' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'no',
				'separator' => 'before',
			]
		);

		$this->add_control(
			'contact_info_bg_color',
			[
				'label' => esc_html__( 'Background color', 'expedition-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => 'rgba(255, 255, 255, 0.85)',
                'condition' => [
					'contact_info_window' => 'yes',
				],
				'selectors' => [
					'{{WRAPPER}} .contact-info-map' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'background_image_overlay',
			array(
				'label'     => esc_html__( 'Image Overlay', 'expedition-plugin' ),
				'type'      => Controls_Manager::MEDIA,
				'default'   => array(
					'url' => Utils::get_placeholder_image_src(),
				),
				'media_types' => [ 'image' ],
                'selectors' => [
                '{{WRAPPER}} .contact-info-map' => 'background-image: url( {{URL}} );',
                ],
                'condition' => [
					'contact_info_window' => 'yes',
				],
			)
		);

		$this->add_control(
			'contact_info_text_color',
			[
				'label' => esc_html__( 'Text color', 'expedition-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '#3d4a6e',
                'condition' => [
					'contact_info_window' => 'yes',
				],
				'selectors' => [
					'{{WRAPPER}} .contact-info-map' => 'color: {{VALUE}};',
				],
			]
		);

        $this->add_control(
			'contact_info_icon1',
			[
				'label' => esc_html__( 'Icon #1', 'expedition-plugin' ),
				'type' => Controls_Manager::ICONS,
				'default' => [
					'value' => 'fa-solid fa-location-dot',
					'library' => 'faprosolid',
				],
				'condition' => [
					'contact_info_window' => 'yes',
				],
			]
		);

		$this->add_control(
			'contact_info_text1', [
				'label' => esc_html__( 'Info Text #1', 'expedition-plugin' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__( 'Melbourne, Australia' , 'expedition-plugin' ),
				'label_block' => true,
				'condition' => [
					'contact_info_window' => 'yes',
				],
			]
		);

		$this->add_control(
			'contact_info_icon2',
			[
				'label' => esc_html__( 'Icon #2', 'expedition-plugin' ),
				'type' => Controls_Manager::ICONS,
				'default' => [
					'value' => 'fa-solid fa-phone',
					'library' => 'faprosolid',
				],
				'condition' => [
					'contact_info_window' => 'yes',
				],
			]
		);

		$this->add_control(
			'contact_info_text2', [
				'label' => esc_html__( 'Info Text #2', 'expedition-plugin' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__( '+61 3 8376 6284' , 'expedition-plugin' ),
				'label_block' => true,
				'condition' => [
					'contact_info_window' => 'yes',
				],
			]
		);

		$this->add_control(
			'contact_info_icon3',
			[
				'label' => esc_html__( 'Icon #3', 'expedition-plugin' ),
				'type' => Controls_Manager::ICONS,
				'default' => [
					'value' => 'fa-solid fa-paper-plane',
					'library' => 'faprosolid',
				],
				'condition' => [
					'contact_info_window' => 'yes',
				],
			]
		);

		$this->add_control(
			'contact_info_text3', [
				'label' => esc_html__( 'Info Text #3', 'expedition-plugin' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__( 'youremail@youremailaddress.com' , 'expedition-plugin' ),
				'label_block' => true,
				'condition' => [
					'contact_info_window' => 'yes',
				],
			]
		);

		$this->end_controls_section();

	}

	protected function render() {

		$settings = $this->get_settings_for_display();
		$shortcode = $this->get_settings_for_display( 'shortcode' );

		$shortcode = do_shortcode( shortcode_unautop( $shortcode ) );

        // prepend map style
		$map_style = $settings[ 'map_style' ];
		$map_style = strip_tags( $map_style );
		$map_style = preg_replace( '/\s/', '', $map_style );

        $this->add_render_attribute( 'google-map', [
			'id' => 'google-container',
			'data-latitude' => $settings[ 'map_latitude' ],
			'data-longitude' => $settings[ 'map_longitude' ],
			'data-zoom' => $settings[ 'map_zoom' ][ 'size' ],
            'data-type' => $settings[ 'map_type' ],
            'data-zoom-control' => $settings[ 'map_zoom_control' ],
            'data-marker-title' => $settings[ 'map_marker_title' ],
            'data-marker-icon' => $settings[ 'map_marker_icon' ]['url'],
			'data-info-window' => $settings[ 'map_info_window' ],
            'data-style' => $settings[ 'map_style' ],
		] );

	
        ?>
			<div <?php echo $this->get_render_attribute_string( 'google-map' ); ?>></div>
			<?php if ( $settings[ 'contact_info_window' ] == 'yes' ) {
				echo '<div class="contact-info-map">';
				echo '<div class="container">';
				echo '<div class="row">';
				echo '<div class="col-md-12">';
				echo '<div class="row">';
								
				echo '<div class="col-md-4 contact-box">';
				Icons_Manager::render_icon( $settings['contact_info_icon1'] );	
				echo $settings[ 'contact_info_text1' ];
				echo '</div>';
								
				echo '<div class="col-md-4 contact-box">';
				Icons_Manager::render_icon( $settings['contact_info_icon2'] );	
				echo $settings[ 'contact_info_text2' ];
				echo '</div>';

				echo '<div class="col-md-4 contact-box">';
				Icons_Manager::render_icon( $settings['contact_info_icon3'] );	
				echo $settings[ 'contact_info_text3' ];
				echo '</div>';
								
				echo '</div>';  
				echo '</div>';
				echo '</div>';
				echo '</div>';
				echo '</div>';
			}
			?>	

        <?php

            if ( $settings[ 'map_zoom_control' ] == 'yes' ) {
                echo '<div id="zoom-in"></div>';
                echo '<div id="zoom-out"></div>';
            }

            if ( $settings[ 'map_info_window' ] == 'yes' ) {

                echo '<script>';
                echo 'var contentString = \'<div class="contact-info-box">\'+';
                echo '\'<h3>' . esc_html($settings['map_title_info_window']) . '</h3>\'+\'';
				echo '<ul>\'+\'';
				foreach ( $settings[ 'info_list' ] as $item ) : 

					if ( $item[ 'map_info_link_check' ] == 'yes' ) {

						if ( ! empty( $item[ 'info_link' ][ 'url' ] ) ) {
			
							$this->add_render_attribute( 'map-info-link', 'href', $item[ 'info_link' ][ 'url' ] );
			
							if ( $item[ 'info_link' ][ 'is_external' ] ) {
								$this->add_render_attribute( 'map-info-link', 'target', '_blank' );
							}
			
							if ( $item[ 'info_link' ][ 'nofollow' ] ) {
								$this->add_render_attribute( 'map-info-link', 'rel', 'nofollow' );
							}
			
						}
					}

					echo '<li>';

					if ( $item[ 'info_icon' ][ 'value' ] ) :

						Icons_Manager::render_icon( $item[ 'info_icon' ], [ 'aria-hidden' => 'true' ] );

					endif;

					if ( ! empty( $item[ 'info_link' ][ 'url' ] ) ) : 
						echo "<a " . $this->get_render_attribute_string( 'map-info-link' ) . ">";
					endif;

					echo $item[ 'info_text' ];

					if ( ! empty( $item[ 'info_link' ][ 'url' ] ) ) :
						echo "</a>";
					endif;
						

					echo '</li>\'+\'';

				endforeach;
				echo '</ul>';
                echo '\'+\'</div>\';';
                echo '</script>';

           }

        ?>


		<?php

	}

}

Plugin::instance()->widgets_manager->register( new Expedition_Google_Map_Widget() );
