<?php
namespace Elementor;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Expedition Team Widget
 *
 * @author: Mountain-Themes
 * @version: 2.0.0
 */

class Expedition_Team_Widget extends Widget_Base {

	public function get_name() {
		return 'expedition-team';
	}

	public function get_title() {
		return esc_html__( 'Team', 'expedition-plugin' );
	}

	public function get_icon() {
		return 'eicon-person expedition-badge';
	}

	public function get_categories() {
		return [ 'expedition-category' ];
	}

	public function get_keywords() {
		return [ 'team', 'member', 'avatar' ];
	}

	/**
	 * Register widget controls.
	 */

	protected function register_controls() {
		

		$this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Content', 'expedition-plugin' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$this->add_control(
			'team_image', [
				'label' => esc_html__( 'Image', 'expedition-plugin' ),
				'type' => Controls_Manager::MEDIA,
				'default'   => array(
					'url' => Utils::get_placeholder_image_src(),
				),
			]
		);

        $this->add_group_control(
			Group_Control_Image_Size::get_type(), [
				'name' => 'team_image', // Usage: `{name}_size` and `{name}_custom_dimension`, in this case `thumbnail_size` and `thumbnail_custom_dimension`.
			]
		);

        $this->add_control(
			'team_title', [
				'label' => esc_html__( 'Title', 'expedition-plugin' ),
				'type' => Controls_Manager::TEXT,
				'label_block' => true,
				'default' => 'Paul Torres',
			]
		);

		$this->add_control(
			'team_text',
			array(
				'type'        => Controls_Manager::WYSIWYG,
				'label' => esc_html__( 'Text', 'expedition-plugin' ),
				'description' => sprintf(
					'%1s <strong>%2s</strong> %3s',
					esc_html__( 'Use class', 'expedition-plugin' ),
					'has-drop-cap',
					esc_html__( 'on an HTML element to set a dropcap.', 'expedition-plugin' )
				),
				'default' => '<p>We don\'t want to conquer the cosmos, we simply want to extend the boundaries of Earth to the frontiers of the cosmos.</p>',
			)
		);

		$this->end_controls_section();

        $this->start_controls_section(
			'section_socials', [
				'label' => esc_html__( 'Social Profiles', 'expedition-plugin' )
			]
		);

        $this->add_control(
			'show_socials', [
				'label' => esc_html__( 'Show Socials', 'expedition-plugin' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
			]
		);

        $repeater = new Repeater();

		$repeater->add_control(
			'team_icon', [
				'label' => esc_html__( 'Icon', 'expedition-plugin' ),
				'type' => Controls_Manager::ICONS,
				'fa4compatibility' => 'social',
			]
		);

		$repeater->add_control(
			'team_link', [
				'label' => esc_html__( 'Link', 'expedition-plugin' ),
				'type' => Controls_Manager::URL,
				'placeholder' => 'https://your-link.com',
			]
		);

		$this->add_control(
			'team_socials', [
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'team_icon' => [
							'value' => 'fa-brands fa-x-twitter',
							'library' => 'faprobrands'
						],
						'team_link' => [ 'url' => '#' ]
					],
					[
						'team_icon' => [
							'value' => 'fa-brands fa-facebook-f',
							'library' => 'faprobrands'
						],
						'team_link' => [ 'url' => '#' ]
					],
					[
						'team_icon' => [
							'value' => 'fa-brands fa-dribbble',
							'library' => 'faprobrands'
						],
						'team_link' => [ 'url' => '#' ]
					],
                    [
						'team_icon' => [
							'value' => 'fa-brands fa-instagram',
							'library' => 'faprobrands'
						],
						'team_link' => [ 'url' => '#' ]
					]
				],
				'condition' => [
					'show_socials' => 'yes',
				]
			]
		);

		$this->end_controls_section();

        $this->start_controls_section(
			'section_styling', [
				'label' => esc_html__( 'Team Style', 'expedition-plugin' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(), [
				'name' => 'title_typography',
				'label' => esc_html__( 'Typography title', 'expedition-plugin' ),
				'selector' => '{{WRAPPER}} .team-box h2',
				'fields_options' => [
					'typography' => [
						'default' => 'yes'
					],
					// Inner control name
					'font_weight' => [
						// Inner control settings
						'default' => '600',
					],
					'font_family' => [
						'default' => 'Aldrich',
					],
					'text_transform' => [
						'default' => 'uppercase',
					],
					'font_size'   => [
						'default' => [ 'unit' => 'px', 'size' => 14 ]
					],
				],
			]
		);

		$this->add_group_control(
			\Elementor\Group_Control_Typography::get_type(), [
				'name' => 'text_typography',
				'label' => esc_html__( 'Typography text', 'expedition-plugin' ),
				'selector' => '{{WRAPPER}} .team-box p',
				'fields_options' => [
					'typography' => [
						'default' => 'yes'
					],
					// Inner control name
					'font_weight' => [
						// Inner control settings
						'default' => '400',
					],
					'font_family' => [
						'default' => 'Exo',
					],
					'text_transform' => [
						'default' => 'none',
					],
					'font_size'   => [
						'default' => [ 'unit' => 'px', 'size' => 13 ]
					],
				],
			]
		);

        $this->add_control(
			'team_title_color',
			[
				'label' => esc_html__( 'Title color', 'expedition-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '#3d4a6e',
				'selectors' => [
					'{{WRAPPER}} .team-box h2' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'services_line_color',
			[
				'label' => esc_html__( 'Line color', 'expedition-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '#3d4a6e',
				'selectors' => [
					'{{WRAPPER}} .team-box .line:before' => 'background-color: {{VALUE}};',
					'{{WRAPPER}} .team-box .line:after' => 'background-color: {{VALUE}};',
				],
			]
		);

        $this->add_control(
			'team_text_color',
			[
				'label' => esc_html__( 'Text color', 'expedition-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '#3d4a6e',
				'selectors' => [
					'{{WRAPPER}} .team-box p' => 'color: {{VALUE}};',
				],
			]
		);

        $this->add_control(
			'team_icon_color',
			[
				'label' => esc_html__( 'Icon color', 'expedition-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => '#3d4a6e',
				'selectors' => [
					'{{WRAPPER}} .team-box .social-icons li i' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'team_icon_hover_color',
			[
				'label' => esc_html__( 'Icon hover color', 'expedition-plugin' ),
				'type' => Controls_Manager::COLOR,
				'default'   => 'rgba(61, 74, 110, 0.59)',
				'selectors' => [
					'{{WRAPPER}} .team-box .social-icons li a:hover i' => 'color: {{VALUE}};',
				],
			]
		);

	}

    protected function render_socials( $instance ) {

		$this->add_render_attribute( 'team_socials', 'class', 'social-icons' );

		if ( $instance[ 'team_socials' ] ) : ?>

			<div <?php echo $this->get_render_attribute_string( 'team_socials' ); ?>>

             <ul>

                        <?php foreach ( $instance[ 'team_socials' ] as $item ) : ?>

                            <?php if ( ! empty( $item[ 'team_link' ][ 'url' ] ) ) : ?>

                                <?php

                                    $this->add_render_attribute( 'social-link-' . $item[ '_id' ], [
                                        'href' => $item[ 'team_link' ][ 'url' ]
                                    ] );

                                    if ( $item[ 'team_link' ][ 'is_external' ] ) {
                                        $this->add_render_attribute( 'social-link-' . $item[ '_id' ], 'target', '_blank' );
                                    }

                                    if ( $item[ 'team_link' ][ 'nofollow' ] ) {
                                        $this->add_render_attribute( 'social-link-' . $item[ '_id' ], 'rel', 'nofollow' );
                                    }

                                ?>

                                <li>

                                    <a <?php echo $this->get_render_attribute_string( 'social-link-' . $item[ '_id' ] ); ?>>

                                        <?php

                                            if ( $item[ 'team_icon' ][ 'value' ] ) :

                                                Icons_Manager::render_icon( $item[ 'team_icon' ], [ 'aria-hidden' => 'true' ] );

                                            endif;

                                        ?>

                                    </a>

                                </li>

                            <?php endif; ?>

                        <?php endforeach; ?>

                </ul>

			</div>

		<?php endif;

	}

	/**
	 * Render widget output on the frontend.
	 *
	 * @since 1.0
	 */
	protected function render() {

		$settings = $this->get_settings_for_display();

        $this->add_render_attribute( 'team', 'class', 'team-box' );

        ?>

        <div <?php echo $this->get_render_attribute_string( 'team' ); ?>>

            <?php

                if ( ! empty( $settings[ 'team_image' ][ 'url' ] ) ) :

                    echo '<div class="team-image-box">';

                        echo Group_Control_Image_Size::get_attachment_image_html( $settings, 'team_image' );

                    echo '</div>';

                endif;

            echo '<h2>' . esc_html($settings['team_title']) . '</h2>';
			echo '<div class="line"><span></span></div>';
            echo $settings['team_text'];
            if ( $settings[ 'show_socials' ] == 'yes' ) {
                $this->render_socials( $settings );
            }


            ?>

        </div>

        <?php

	}

}

Plugin::instance()->widgets_manager->register( new Expedition_Team_Widget() );
