<?php

/**
 * Infinity functions and definitions
 *
 * @package expedition
 */


if ( ! function_exists( 'expedition_setup' ) ) :

/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */

function expedition_setup() {

	/*
	* Make theme available for translation.
	* Translations can be filed in the /languages/ directory.
	* If you're building a theme based on component_s, use a find and replace
	* to change 'component_s' to the name of your theme in all the template files
	*/
	load_theme_textdomain( 'expedition', get_template_directory() . '/languages' );

	// Add default posts and comments RSS feed links to head.
	add_theme_support( 'automatic-feed-links' );

	/*
	* Let WordPress manage the document title.
	* By adding theme support, we declare that this theme does not use a
	* hard-coded <title> tag in the document head, and expect WordPress to
	* provide it for us.
	*/
	add_theme_support( 'title-tag' );

	/*
	* Enable support for Post Thumbnails on posts and pages.
	*
	* @link http://codex.wordpress.org/Function_Reference/add_theme_support#Post_Thumbnails
	*/
	add_theme_support( 'post-thumbnails' );
	set_post_thumbnail_size( 1920, 9999 );

	// This theme uses wp_nav_menu() in one location.
	register_nav_menus( array(
			'primary-menu' => esc_html__( 'Menu', 'expedition' ),
	) );

	/*
	* Switch default core markup for search form, comment form, and comments
	* to output valid HTML5.
	*/
	add_theme_support( 'html5', array(
		'search-form',
		'comment-form',
		'comment-list',
		'gallery',
		'caption',
	) );

	/*
	* Enable support for Post Formats.
	* See https://codex.wordpress.org/Post_Formats
	*/
	if ( class_exists('acf') ) : 

	add_theme_support( 'post-formats', array(
		'gallery', 'link', 'image', 'quote', 'video', 'audio', 'chat',
	) );

	endif;

	// Indicate widget sidebars can use selective refresh in the Customizer.
	add_theme_support( 'customize-selective-refresh-widgets' );

	// Add support for full and wide align images.
	add_theme_support( 'align-wide' );

	// Add support for responsive embedded content.
	add_theme_support( 'responsive-embeds' );
	
}
endif; // component_s_setup
add_action( 'after_setup_theme', 'expedition_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 * ===========================================================================
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */

function expedition_content_width() {

	$GLOBALS['content_width'] = apply_filters( 'expedition_content_width', 1140 );

}
add_action( 'after_setup_theme', 'expedition_content_width');


/* =============================== */
/* :::: Register widget areas :::: */
/* =============================== */

function expedition_widgets_init() {

	register_sidebar( array(
		'name'          => esc_html__( 'Blog Sidebar', 'expedition' ),
		'id'            => 'sidebar-1',
		'description'   => esc_html__( 'Main sidebar that appears on the right.', 'expedition' ),
		'before_widget' => '<div id="%1$s" class="mt-widget %2$s">',
		'after_widget'  => '</div>',
		'before_title'  => '<h4 class="expedition-widget-title">',
		'after_title'   => '</h4>',
	) );

	if ( 'widgets' == get_theme_mod( 'variants_footer' ) ) : 

		$column_footer = get_theme_mod( 'column_footer', 'col-md-3' );

		register_sidebar( array(
				'name'          => esc_html__( 'Footer', 'expedition' ),
				'id'            => 'footer-sidebar',
				'description'   => esc_html__( 'Widgets that appears on the footer.', 'expedition' ),
				'before_widget' => '<div id="%1$s" class="mt-widget %2$s ' . $column_footer . '">',
				'after_widget'  => '</div>',
				'before_title'  => '<h1 class="expedition-widget-title">',
				'after_title'   => '</h1>',
		) );

	endif;

	if ( class_exists( 'WooCommerce' ) ) :

		register_sidebar( array(
			'name'          => esc_html__( 'Shop Sidebar', 'expedition' ),
			'id'            => 'shop-sidebar',
			'description'   => esc_html__( 'Sidebar that appears on the shop.', 'expedition' ),
			'before_widget' => '<div id="%1$s" class="mt-widget %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<h1 class="expedition-widget-title">',
			'after_title'   => '</h1>',
		) );

	endif;

}
add_action( 'widgets_init', 'expedition_widgets_init' );


/* =============================== */
/* ::: Personal logo and link :::: */
/* =============================== */

function expedition_edit_admin_logo() {
		 
	$image_admin = get_theme_mod( 'image_admin', get_template_directory_uri() . '/assets/img/logo.png' );
	$width_img_admin = get_theme_mod( 'width_img_admin', '255' );
	$height_img_admin = get_theme_mod( 'height_img_admin', '220' );

	echo '<style type="text/css">
	.login h1 a {
	background: transparent url(' . esc_url($image_admin) . ') 50% 50% no-repeat!important;
	width:' . esc_attr($width_img_admin) . 'px!important;
	height: ' . esc_attr($height_img_admin) . 'px!important; 
	}
	.login h1 a:focus {
	box-shadow: none; 
	}
	</style>';

}
add_action( 'login_enqueue_scripts', 'expedition_edit_admin_logo' );


function expedition_edit_admin_logo_link() {

	return home_url('/');

}
add_filter('login_headerurl', 'expedition_edit_admin_logo_link');

/* =============================== */
/* ::::::: Post navigation ::::::: */
/* =============================== */

function expedition_post_navigation() {
	
	$nav = '<div class="post-navigation">';

	$prev_post = get_previous_post();

	if (!empty( $prev_post )): 
	$prev = get_previous_post_link( '<a href="' . get_permalink( $prev_post->ID ) . '" class="post-link-nav">' . esc_html__('Next post', 'expedition') . '<i class="fa fa-angle-right"></i></a>' );
	$nav .= '<div class="next-button">' . $prev . '</div>';
	endif;

	$next_post = get_next_post();

	if (!empty( $next_post )): 
	$next = get_next_post_link( '<a href="' . get_permalink( $next_post->ID ) . '" class="post-link-nav"><i class="fa fa-angle-left"></i>' . esc_html__('Prev post', 'expedition') . '</a>' );
	$nav .= '<div class="prev-button">' . $next . '</div>';
	endif;
	
	$nav .= '</div>';
	
	return $nav;
	
}

/* ==================================== */
/* ::::::: Cutting text in post ::::::: */
/* ==================================== */

function expedition_excerpt_length($length) {
    return 70;
}
add_filter('excerpt_length', 'expedition_excerpt_length');

function expedition_excerpt_more($more) {
    global $post;
    return esc_html__( '... ', 'expedition' ) . '<div class="post-button">
			   <a href="'. get_permalink($post->ID) . '">' . esc_html('Read More', 'expedition') . '</a>
			</div>';
}
add_filter('excerpt_more', 'expedition_excerpt_more');


/* =============================== */
/* ::::::: Page pagination ::::::: */
/* =============================== */

function expedition_page_pagination( $template, $class ) {

	return '
	<nav class="%1$s" role="navigation">
		<div class="nav-links">%3$s</div>
	</nav>    
	';

}
add_filter('navigation_markup_template', 'expedition_page_pagination', 10, 2 );

/* ==================================== */
/* :::::::: Comment navigation :::::::: */
/* ==================================== */

function expedition_comment_pagination() {

	$output = '';

	if ( get_comment_pages_count() > 1 ):
		$output .= '<nav class="comments-navigation">';
		if ( get_previous_comments_link() ) {
			add_filter( 'previous_comments_link_attributes', function() {
				return 'class="prev-comments"';
			} );
			$output .= get_previous_comments_link( '<i class="fa fa-arrow-left"></i>' );
		}
		if ( get_next_comments_link() ) {
			add_filter( 'next_comments_link_attributes', function() {
				return 'class="next-comments"';
			} );
			$output .= get_next_comments_link( '<i class="fa fa-arrow-right"></i>' );
		}
		$output .= '</nav>';
	endif;

	return apply_filters( 'expedition/comment_pagination', $output );

}

/* ==================================== */
/* :::::::::: Comments style :::::::::: */
/* ==================================== */

function expedition_comments_style( $comment, $args, $depth ) {
	$GLOBALS['comment'] = $comment;
	global $post;
	?>

	<li id="comment-<?php comment_ID(); ?>" <?php comment_class(); ?>>

		  <div class="avatar-author">

			<?php if ( 0 != $args['avatar_size'] && get_avatar( $comment ) ): ?>
				<?php echo get_avatar( $comment, $args['avatar_size'] ); ?>
			<?php endif; ?>

		  </div>

			<div class="comment-info">

			  <div class="name-author">
					<h5><?php comment_author(); ?></h5>
					<span><?php echo get_comment_date(); ?>
					<?php edit_comment_link( '(Edit)' ); ?></span>
			   </div>


					<div id="comment-content-<?php comment_ID(); ?>" class="body-comment">				

						<?php if ( '0' == $comment->comment_approved ): ?>
							<p class="comment-awaiting-moderation">
								<?php esc_html_e( 'Your comment is awaiting moderation.', 'expedition' ); ?>
							</p>
						<?php endif; ?>

						<?php comment_text(); ?>

						 <div class="reply">

							<?php
								comment_reply_link( array_merge( $args, array(
									'add_below' => 'comment',
									'depth' => $depth,
									'max_depth' => $args['max_depth'],
									'reply_text' => sprintf( esc_html__('Reply', 'expedition')),
								) ) );
							?>
						</div>

					</div>

	 		</div>
	<!-- </li> is added by WordPress automatically -->
	<?php
}

/* =============================== */
/* :::::: Column thumbnail ::::::: */
/* =============================== */

function expedition_add_thumbnail_column_post($columns) {

	$column_thumbnail = array(
		'thumbnail' => esc_html__('Thumbnail', 'expedition')
	);
	$columns = array_slice($columns, 0, 2, true) + $column_thumbnail + array_slice($columns, 1, NULL, true);
	return $columns;

}

function expedition_display_thumbnail_post($column) {

	global $post;
	switch ( $column ) {
		case 'thumbnail':
			echo get_the_post_thumbnail($post->ID, array(
				50,
				50
			));
			break;
	}

}
add_filter('manage_edit-post_columns', 'expedition_add_thumbnail_column_post', 10, 1);
add_action('manage_post_posts_custom_column', 'expedition_display_thumbnail_post', 10, 1);


/* =================================== */
/* :::::::: Font-Awesome 6 PRO ::::::: */
/* =================================== */

function fontawesome_dashboard() {
	wp_enqueue_style('fontawesome-pro', get_template_directory_uri() . '/assets/fonts/css/all.min.css', '', '6.4.2', 'all');
 }
 add_action('admin_init', 'fontawesome_dashboard');

/* =================================== */
/* ::::::: If devmode disabled ::::::: */
/* =================================== */

if ( false == get_theme_mod( 'acf_show_admin_panel', false ) ) {
	add_filter('acf/settings/show_admin', '__return_false');
}

/* ============================== */
/* ::::::: Fallback menu :::::::: */
/* ============================== */

if ( ! function_exists( 'expedition_fallback_menu' ) ) {

	function expedition_fallback_menu() {

		if ( current_user_can( 'administrator' ) ) {
			echo '<p class="no-menu-message">' . esc_html__( 'Please register navigation from', 'expedition' ) . ' ' . '<a href="' . esc_url( admin_url( 'nav-menus.php' ) ) . '" target="_blank">' . esc_html__( 'Appearance > Menus', 'expedition' ) . '</a></p>';
		}

	}
}

/* ============================================= */
/* :::::: Save acf options to json files ::::::: */
/* ============================================= */

if(!function_exists('expedition_acf_save_json')) {
	function expedition_acf_save_json($path) {
	    $path = trailingslashit(get_template_directory() ) . 'lib/custom-fields';
	    return $path;
	}
}
add_filter('acf/settings/save_json', 'expedition_acf_save_json');


if(!function_exists('expedition_acf_load_json')) {
	function expedition_acf_load_json($paths) {
	    unset($paths[0]);
	    $paths[] = trailingslashit(get_template_directory() ) . 'lib/custom-fields';
	    return $paths;
	}
}
add_filter('acf/settings/load_json', 'expedition_acf_load_json');

/* ==================================== */
/* ::::: Disable Kirki Telemetry :::::: */
/* ==================================== */

add_filter( 'kirki_telemetry', '__return_false' );

/* ==================================== */
/* :::: Disable CF7 Auto Paragraph :::: */
/* ==================================== */

add_filter('wpcf7_autop_or_not', '__return_false');

add_filter('wpcf7_form_elements', function($content) {
    $content = preg_replace('/<(span).*?class="\s*(?:.*\s)?wpcf7-form-control-wrap(?:\s[^"]+)?\s*"[^\>]*>(.*)<\/\1>/i', '\2', $content);

    return $content;
});

/* ======================== */
/* ::::: Demo import :::::: */
/* ======================== */

if ( ! function_exists( 'expedition_demo_import_files' ) ) {
	function expedition_demo_import_files() {
		return array(
			array(
				'import_file_name' => esc_html__( 'Demo Import', 'expedition' ),
				'local_import_file' => trailingslashit(get_template_directory() ) . 'inc/demo/demo-content.xml',
				'local_import_widget_file' => trailingslashit(get_template_directory() ) . 'inc/demo/widgets.wie',
				'local_import_customizer_file' => trailingslashit(get_template_directory() ) . 'inc/demo/customizer.dat',
				'import_notice'              =>     esc_html__( 'Before importing demo data you must have to install required plugins', 'expedition' ),
			),
		);
	}
}
add_filter( 'pt-ocdi/import_files', 'expedition_demo_import_files' );

add_filter( 'pt-ocdi/regenerate_thumbnails_in_content_import', '__return_false' );

if ( ! function_exists( 'expedition_after_import_setup' ) ) {
	function expedition_after_import_setup() {

		global $wp_rewrite;

		// Set menus
		$primary_menu = get_term_by( 'name', 'expedition', 'nav_menu' );

		set_theme_mod( 'nav_menu_locations',
			array(
				'primary-menu' => $primary_menu->term_id,
			)
		);

		// Set pages
		$front_page_id = get_page_by_title( html_entity_decode('Classic - single image') );
		update_option( 'show_on_front', 'page' );
		update_option( 'page_on_front', $front_page_id->ID );

		// Update permalink
		$wp_rewrite->set_permalink_structure( '/%postname%/' );

		// Update option
		update_option( 'date_format', 'd M, Y' );

	

	}
}
add_action( 'pt-ocdi/after_import', 'expedition_after_import_setup' );

function expedition_theme_ocdi_plugin_page_setup( $default_settings ) {
    $default_settings['parent_slug'] = 'themes.php';
    $default_settings['page_title']  = esc_html__( 'Expedition Demo Import' , 'expedition' );
    $default_settings['menu_title']  = esc_html__( 'Demo Importer' , 'expedition' );
    $default_settings['capability']  = 'import';
    $default_settings['menu_slug']   = 'expedition-one-click-demo-import';

    return $default_settings;
}
add_filter( 'pt-ocdi/plugin_page_setup', 'expedition_theme_ocdi_plugin_page_setup' );

function expedition_ocdi_plugin_intro_text($default_text)
{
	$default_text .= '<div class="ocdi__intro-text">';
	$default_text .= 	'<p>';
	$default_text .= 		'<b><span style="color: #ff0000;">' . esc_html__('Important:', 'expedition') . '</span> <a target="_blank" href="https://elementor.com/help/requirements/">' . esc_html__('Please check out here for the system requirements you need in order to import demo data and use Elementor.', 'expedition') . '</a> ' . esc_html__('(If you are not sure whether or not your server support this, contact your host.)', 'expedition') . '</b>';
	$default_text .= 		'<br>';
	$default_text .= 		'<br>';
	$default_text .= 	'</p>';
	$default_text .= '</div>';
	
	return $default_text;
}

add_filter('ocdi/plugin_intro_text', 'expedition_ocdi_plugin_intro_text');
