<?php
if ( ! defined( 'ABSPATH' ) ) {
	die( '-1' );
}

/**
 *
 * Class Vc_IconPicker
 * @since 4.4
 * See example usage in shortcode 'vc_icon'
 *
 *      `` example
 *        array(
 *            'type' => 'iconpicker',
 *            'heading' => esc_html__( 'Icon', 'js_composer' ),
 *            'param_name' => 'icon_fontawesome',
 *            'settings' => array(
 *                'emptyIcon' => false, // default true, display an "EMPTY"
 *     icon? - if false it will display first icon from set as default.
 *                'iconsPerPage' => 200, // default 100, how many icons
 *     per/page to display
 *            ),
 *            'dependency' => array(
 *                'element' => 'type',
 *                'value' => 'fontawesome',
 *            ),
 *        ),
 * vc_filter: vc_iconpicker-type-{your_icon_font_name} - filter to add new icon
 *     font type. see example for vc_iconpicker-type-fontawesome in bottom of
 *     this file Also // SEE HOOKS FOLDER FOR FONTS REGISTERING/ENQUEUE IN BASE
 * @path "/include/autoload/hook-vc-iconpicker-param.php"
 */
class Vc_IconPicker {
	/**
	 * @since 4.4
	 * @var array - save current param data array from vc_map
	 */
	protected $settings;
	/**
	 * @since 4.4
	 * @var string - save a current field value
	 */
	protected $value;
	/**
	 * @since 4.4
	 * @var array - optional, can be used as self source from self array., you
	 *     can pass it also with filter see Vc_IconPicker::setDefaults
	 */
	protected $source = array();

	/**
	 * @param $settings - param field data array
	 * @param $value - param field value
	 * @since 4.4
	 *
	 */
	public function __construct( $settings, $value ) {
		$this->settings = $settings;
		$this->setDefaults();

		$this->value = $value; // param field value
	}

	/**
	 * Set default function will extend current settings with defaults
	 * It can be used in Vc_IconPicker::render, but also it is passed to input
	 * field and was hooked in composer-atts.js file See vc.atts.iconpicker in
	 * wp-content/plugins/js_composer/assets/js/params/composer-atts.js init
	 * method
	 *  - it initializes javascript logic, you can provide ANY default param to
	 * it with 'settings' key
	 * @since 4.4
	 */
	protected function setDefaults() {
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['type'] ) ) {
			$this->settings['settings']['type'] = 'fontawesome'; // Default type for icons
		}

		// More about this you can read in http://codeb.it/fonticonpicker/
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['hasSearch'] ) ) {
			// Whether or not to show the search bar.
			$this->settings['settings']['hasSearch'] = true;
		}
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['emptyIcon'] ) ) {
			// Whether or not empty icon should be shown on the icon picker
			$this->settings['settings']['emptyIcon'] = true;
		}
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['allCategoryText'] ) ) {
			// If categorized then use this option
			$this->settings['settings']['allCategoryText'] = esc_html__( 'From all categories', 'js_composer' );
		}
		if ( ! isset( $this->settings['settings'], $this->settings['settings']['unCategorizedText'] ) ) {
			// If categorized then use this option
			$this->settings['settings']['unCategorizedText'] = esc_html__( 'Uncategorized', 'js_composer' );
		}

		/**
		 * Source for icons, can be passed via "mapping" or with filter vc_iconpicker-type-{your_type} (default fontawesome)
		 * vc_filter: vc_iconpicker-type-{your_type} (default fontawesome)
		 */
		if ( isset( $this->settings['settings'], $this->settings['settings']['source'] ) ) {
			$this->source = $this->settings['settings']['source'];
			unset( $this->settings['settings']['source'] ); // We don't need this on frontend.(js)
		}
	}

	/**
	 * Render edit form field type 'iconpicker' with selected settings and
	 * provided value. It uses javascript file vc-icon-picker
	 * (vc_iconpicker_base_register_js, vc_iconpicker_editor_jscss), see
	 * wp-content/plugins/js_composer/include/autoload/hook-vc-iconpicker-param.php
	 * folder
	 * @return string - rendered param field for editor panel
	 * @since 4.4
	 */
	public function render() {

		$output = '<div class="vc-iconpicker-wrapper"><select class="vc-iconpicker">';

		// call filter vc_iconpicker-type-{your_type}, e.g. vc_iconpicker-type-fontawesome with passed source from shortcode(default empty array). to get icons
		$arr = apply_filters( 'vc_iconpicker-type-' . esc_attr( $this->settings['settings']['type'] ), $this->source );
		if ( isset( $this->settings['settings'], $this->settings['settings']['emptyIcon'] ) && true === $this->settings['settings']['emptyIcon'] ) {
			array_unshift( $arr, array() );
		}
		if ( ! empty( $arr ) ) {
			foreach ( $arr as $group => $icons ) {
				if ( ! is_array( $icons ) || ! is_array( current( $icons ) ) ) {
					$class_key = key( $icons );
					$output .= '<option value="' . esc_attr( $class_key ) . '" ' . ( strcmp( $class_key, $this->value ) === 0 ? 'selected' : '' ) . '>' . esc_html( current( $icons ) ) . '</option>' . "\n";
				} else {
					$output .= '<optgroup label="' . esc_attr( $group ) . '">' . "\n";
					foreach ( $icons as $key => $label ) {
						$class_key = key( $label );
						$output .= '<option value="' . esc_attr( $class_key ) . '" ' . ( strcmp( $class_key, $this->value ) === 0 ? 'selected' : '' ) . '>' . esc_html( current( $label ) ) . '</option>' . "\n";
					}
					$output .= '</optgroup>' . "\n";
				}
			}
		}
		$output .= '</select></div>';

		$output .= '<input name="' . esc_attr( $this->settings['param_name'] ) . '" class="wpb_vc_param_value  ' . esc_attr( $this->settings['param_name'] ) . ' ' . esc_attr( $this->settings['type'] ) . '_field" type="hidden" value="' . esc_attr( $this->value ) . '" ' . ( ( isset( $this->settings['settings'] ) && ! empty( $this->settings['settings'] ) ) ? ' data-settings="' . esc_attr( wp_json_encode( $this->settings['settings'] ) ) . '" ' : '' ) . ' />';

		return $output;
	}
}

/**
 * Function for rendering param in edit form (add element)
 * Parse settings from vc_map and entered values.
 *
 * @param $settings
 * @param $value
 * @param $tag
 *
 * @return string - rendered template for params in edit form
 *
 * @since 4.4
 */
function vc_iconpicker_form_field( $settings, $value, $tag ) {

	$icon_picker = new Vc_IconPicker( $settings, $value );

	return apply_filters( 'vc_iconpicker_render_filter', $icon_picker->render() );
}

// SEE HOOKS FOLDER FOR FONTS REGISTERING/ENQUEUE IN BASE @path "/include/autoload/hook-vc-iconpicker-param.php"

add_filter( 'vc_iconpicker-type-fontawesome', 'vc_iconpicker_type_fontawesome' );

/**
 * Fontawesome icons from FontAwesome :)
 *
 * @param $icons - taken from filter - vc_map param field settings['source']
 *     provided icons (default empty array). If array categorized it will
 *     auto-enable category dropdown
 *
 * @return array - of icons for iconpicker, can be categorized, or not.
 * @since 4.4
 */
function vc_iconpicker_type_fontawesome( $icons ) {
	// Categorized icons ( you can also output simple array ( key=> value ), where key = icon class, value = icon readable name ).
	/**
	 * @version 4.7
	 */
	$fontawesome_icons = array(
		'Accessibility' => array(
			array( 'fab fa-accessible-icon' => 'Accessible Icon (accessibility,handicap,person,wheelchair,wheelchair-alt)' ),
			array( 'fas fa-american-sign-language-interpreting' => 'American Sign Language Interpreting (asl,deaf,finger,hand,interpret,speak)' ),
			array( 'fas fa-assistive-listening-systems' => 'Assistive Listening Systems (amplify,audio,deaf,ear,headset,hearing,sound)' ),
			array( 'fas fa-audio-description' => 'Audio Description (blind,narration,video,visual)' ),
			array( 'fas fa-blind' => 'Blind (cane,disability,person,sight)' ),
			array( 'fas fa-braille' => 'Braille (alphabet,blind,dots,raised,vision)' ),
			array( 'fas fa-closed-captioning' => 'Closed Captioning (cc,deaf,hearing,subtitle,subtitling,text,video)' ),
			array( 'far fa-closed-captioning' => 'Closed Captioning (cc,deaf,hearing,subtitle,subtitling,text,video)' ),
			array( 'fas fa-deaf' => 'Deaf (ear,hearing,sign language)' ),
			array( 'fas fa-low-vision' => 'Low Vision (blind,eye,sight)' ),
			array( 'fas fa-phone-volume' => 'Phone Volume (call,earphone,number,sound,support,telephone,voice,volume-control-phone)' ),
			array( 'fas fa-question-circle' => 'Question Circle (help,information,support,unknown)' ),
			array( 'far fa-question-circle' => 'Question Circle (help,information,support,unknown)' ),
			array( 'fas fa-sign-language' => 'Sign Language (Translate,asl,deaf,hands)' ),
			array( 'fas fa-tty' => 'TTY (communication,deaf,telephone,teletypewriter,text)' ),
			array( 'fas fa-universal-access' => 'Universal Access (accessibility,hearing,person,seeing,visual impairment)' ),
			array( 'fas fa-wheelchair' => 'Wheelchair (accessible,handicap,person)' ),
		),
		'Alert' => array(
			array( 'fas fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'far fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'fas fa-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'far fa-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'fas fa-exclamation' => 'exclamation (alert,danger,error,important,notice,notification,notify,problem,warning)' ),
			array( 'fas fa-exclamation-circle' => 'Exclamation Circle (alert,danger,error,important,notice,notification,notify,problem,warning)' ),
			array( 'fas fa-exclamation-triangle' => 'Exclamation Triangle (alert,danger,error,important,notice,notification,notify,problem,warning)' ),
			array( 'fas fa-radiation' => 'Radiation (danger,dangerous,deadly,hazard,nuclear,radioactive,warning)' ),
			array( 'fas fa-radiation-alt' => 'Alternate Radiation (danger,dangerous,deadly,hazard,nuclear,radioactive,warning)' ),
			array( 'fas fa-skull-crossbones' => 'Skull & Crossbones (Dungeons & Dragons,alert,bones,d&d,danger,dead,deadly,death,dnd,fantasy,halloween,holiday,jolly-roger,pirate,poison,skeleton,warning)' ),
		),
		'Animals' => array(
			array( 'fas fa-cat' => 'Cat (feline,halloween,holiday,kitten,kitty,meow,pet)' ),
			array( 'fas fa-crow' => 'Crow (bird,bullfrog,fauna,halloween,holiday,toad)' ),
			array( 'fas fa-dog' => 'Dog (animal,canine,fauna,mammal,pet,pooch,puppy,woof)' ),
			array( 'fas fa-dove' => 'Dove (bird,fauna,flying,peace,war)' ),
			array( 'fas fa-dragon' => 'Dragon (Dungeons & Dragons,d&d,dnd,fantasy,fire,lizard,serpent)' ),
			array( 'fas fa-feather' => 'Feather (bird,light,plucked,quill,write)' ),
			array( 'fas fa-feather-alt' => 'Alternate Feather (bird,light,plucked,quill,write)' ),
			array( 'fas fa-fish' => 'Fish (fauna,gold,seafood,swimming)' ),
			array( 'fas fa-frog' => 'Frog (amphibian,bullfrog,fauna,hop,kermit,kiss,prince,ribbit,toad,wart)' ),
			array( 'fas fa-hippo' => 'Hippo (animal,fauna,hippopotamus,hungry,mammal)' ),
			array( 'fas fa-horse' => 'Horse (equus,fauna,mammmal,mare,neigh,pony)' ),
			array( 'fas fa-horse-head' => 'Horse Head (equus,fauna,mammmal,mare,neigh,pony)' ),
			array( 'fas fa-kiwi-bird' => 'Kiwi Bird (bird,fauna,new zealand)' ),
			array( 'fas fa-otter' => 'Otter (animal,badger,fauna,fur,mammal,marten)' ),
			array( 'fas fa-paw' => 'Paw (animal,cat,dog,pet,print)' ),
			array( 'fas fa-spider' => 'Spider (arachnid,bug,charlotte,crawl,eight,halloween)' ),
		),
		'Arrows' => array(
			array( 'fas fa-angle-double-down' => 'Angle Double Down (arrows,caret,download,expand)' ),
			array( 'fas fa-angle-double-left' => 'Angle Double Left (arrows,back,caret,laquo,previous,quote)' ),
			array( 'fas fa-angle-double-right' => 'Angle Double Right (arrows,caret,forward,more,next,quote,raquo)' ),
			array( 'fas fa-angle-double-up' => 'Angle Double Up (arrows,caret,collapse,upload)' ),
			array( 'fas fa-angle-down' => 'angle-down (arrow,caret,download,expand)' ),
			array( 'fas fa-angle-left' => 'angle-left (arrow,back,caret,less,previous)' ),
			array( 'fas fa-angle-right' => 'angle-right (arrow,care,forward,more,next)' ),
			array( 'fas fa-angle-up' => 'angle-up (arrow,caret,collapse,upload)' ),
			array( 'fas fa-arrow-alt-circle-down' => 'Alternate Arrow Circle Down (arrow-circle-o-down,download)' ),
			array( 'far fa-arrow-alt-circle-down' => 'Alternate Arrow Circle Down (arrow-circle-o-down,download)' ),
			array( 'fas fa-arrow-alt-circle-left' => 'Alternate Arrow Circle Left (arrow-circle-o-left,back,previous)' ),
			array( 'far fa-arrow-alt-circle-left' => 'Alternate Arrow Circle Left (arrow-circle-o-left,back,previous)' ),
			array( 'fas fa-arrow-alt-circle-right' => 'Alternate Arrow Circle Right (arrow-circle-o-right,forward,next)' ),
			array( 'far fa-arrow-alt-circle-right' => 'Alternate Arrow Circle Right (arrow-circle-o-right,forward,next)' ),
			array( 'fas fa-arrow-alt-circle-up' => 'Alternate Arrow Circle Up (arrow-circle-o-up)' ),
			array( 'far fa-arrow-alt-circle-up' => 'Alternate Arrow Circle Up (arrow-circle-o-up)' ),
			array( 'fas fa-arrow-circle-down' => 'Arrow Circle Down (download)' ),
			array( 'fas fa-arrow-circle-left' => 'Arrow Circle Left (back,previous)' ),
			array( 'fas fa-arrow-circle-right' => 'Arrow Circle Right (forward,next)' ),
			array( 'fas fa-arrow-circle-up' => 'Arrow Circle Up (upload)' ),
			array( 'fas fa-arrow-down' => 'arrow-down (download)' ),
			array( 'fas fa-arrow-left' => 'arrow-left (back,previous)' ),
			array( 'fas fa-arrow-right' => 'arrow-right (forward,next)' ),
			array( 'fas fa-arrow-up' => 'arrow-up (forward,upload)' ),
			array( 'fas fa-arrows-alt' => 'Alternate Arrows (arrow,arrows,bigger,enlarge,expand,fullscreen,move,position,reorder,resize)' ),
			array( 'fas fa-arrows-alt-h' => 'Alternate Arrows Horizontal (arrows-h,expand,horizontal,landscape,resize,wide)' ),
			array( 'fas fa-arrows-alt-v' => 'Alternate Arrows Vertical (arrows-v,expand,portrait,resize,tall,vertical)' ),
			array( 'fas fa-caret-down' => 'Caret Down (arrow,dropdown,expand,menu,more,triangle)' ),
			array( 'fas fa-caret-left' => 'Caret Left (arrow,back,previous,triangle)' ),
			array( 'fas fa-caret-right' => 'Caret Right (arrow,forward,next,triangle)' ),
			array( 'fas fa-caret-square-down' => 'Caret Square Down (arrow,caret-square-o-down,dropdown,expand,menu,more,triangle)' ),
			array( 'far fa-caret-square-down' => 'Caret Square Down (arrow,caret-square-o-down,dropdown,expand,menu,more,triangle)' ),
			array( 'fas fa-caret-square-left' => 'Caret Square Left (arrow,back,caret-square-o-left,previous,triangle)' ),
			array( 'far fa-caret-square-left' => 'Caret Square Left (arrow,back,caret-square-o-left,previous,triangle)' ),
			array( 'fas fa-caret-square-right' => 'Caret Square Right (arrow,caret-square-o-right,forward,next,triangle)' ),
			array( 'far fa-caret-square-right' => 'Caret Square Right (arrow,caret-square-o-right,forward,next,triangle)' ),
			array( 'fas fa-caret-square-up' => 'Caret Square Up (arrow,caret-square-o-up,collapse,triangle,upload)' ),
			array( 'far fa-caret-square-up' => 'Caret Square Up (arrow,caret-square-o-up,collapse,triangle,upload)' ),
			array( 'fas fa-caret-up' => 'Caret Up (arrow,collapse,triangle)' ),
			array( 'fas fa-cart-arrow-down' => 'Shopping Cart Arrow Down (download,save,shopping)' ),
			array( 'fas fa-chart-line' => 'Line Chart (activity,analytics,chart,dashboard,gain,graph,increase,line)' ),
			array( 'fas fa-chevron-circle-down' => 'Chevron Circle Down (arrow,download,dropdown,menu,more)' ),
			array( 'fas fa-chevron-circle-left' => 'Chevron Circle Left (arrow,back,previous)' ),
			array( 'fas fa-chevron-circle-right' => 'Chevron Circle Right (arrow,forward,next)' ),
			array( 'fas fa-chevron-circle-up' => 'Chevron Circle Up (arrow,collapse,upload)' ),
			array( 'fas fa-chevron-down' => 'chevron-down (arrow,download,expand)' ),
			array( 'fas fa-chevron-left' => 'chevron-left (arrow,back,bracket,previous)' ),
			array( 'fas fa-chevron-right' => 'chevron-right (arrow,bracket,forward,next)' ),
			array( 'fas fa-chevron-up' => 'chevron-up (arrow,collapse,upload)' ),
			array( 'fas fa-cloud-download-alt' => 'Alternate Cloud Download (download,export,save)' ),
			array( 'fas fa-cloud-upload-alt' => 'Alternate Cloud Upload (cloud-upload,import,save,upload)' ),
			array( 'fas fa-compress-arrows-alt' => 'Alternate Compress Arrows (collapse,fullscreen,minimize,move,resize,shrink,smaller)' ),
			array( 'fas fa-download' => 'Download (export,hard drive,save,transfer)' ),
			array( 'fas fa-exchange-alt' => 'Alternate Exchange (arrow,arrows,exchange,reciprocate,return,swap,transfer)' ),
			array( 'fas fa-expand-arrows-alt' => 'Alternate Expand Arrows (arrows-alt,bigger,enlarge,move,resize)' ),
			array( 'fas fa-external-link-alt' => 'Alternate External Link (external-link,new,open,share)' ),
			array( 'fas fa-external-link-square-alt' => 'Alternate External Link Square (external-link-square,new,open,share)' ),
			array( 'fas fa-hand-point-down' => 'Hand Pointing Down (finger,hand-o-down,point)' ),
			array( 'far fa-hand-point-down' => 'Hand Pointing Down (finger,hand-o-down,point)' ),
			array( 'fas fa-hand-point-left' => 'Hand Pointing Left (back,finger,hand-o-left,left,point,previous)' ),
			array( 'far fa-hand-point-left' => 'Hand Pointing Left (back,finger,hand-o-left,left,point,previous)' ),
			array( 'fas fa-hand-point-right' => 'Hand Pointing Right (finger,forward,hand-o-right,next,point,right)' ),
			array( 'far fa-hand-point-right' => 'Hand Pointing Right (finger,forward,hand-o-right,next,point,right)' ),
			array( 'fas fa-hand-point-up' => 'Hand Pointing Up (finger,hand-o-up,point)' ),
			array( 'far fa-hand-point-up' => 'Hand Pointing Up (finger,hand-o-up,point)' ),
			array( 'fas fa-hand-pointer' => 'Pointer (Hand) (arrow,cursor,select)' ),
			array( 'far fa-hand-pointer' => 'Pointer (Hand) (arrow,cursor,select)' ),
			array( 'fas fa-history' => 'History (Rewind,clock,reverse,time,time machine)' ),
			array( 'fas fa-level-down-alt' => 'Alternate Level Down (arrow,level-down)' ),
			array( 'fas fa-level-up-alt' => 'Alternate Level Up (arrow,level-up)' ),
			array( 'fas fa-location-arrow' => 'location-arrow (address,compass,coordinate,direction,gps,map,navigation,place)' ),
			array( 'fas fa-long-arrow-alt-down' => 'Alternate Long Arrow Down (download,long-arrow-down)' ),
			array( 'fas fa-long-arrow-alt-left' => 'Alternate Long Arrow Left (back,long-arrow-left,previous)' ),
			array( 'fas fa-long-arrow-alt-right' => 'Alternate Long Arrow Right (forward,long-arrow-right,next)' ),
			array( 'fas fa-long-arrow-alt-up' => 'Alternate Long Arrow Up (long-arrow-up,upload)' ),
			array( 'fas fa-mouse-pointer' => 'Mouse Pointer (arrow,cursor,select)' ),
			array( 'fas fa-play' => 'play (audio,music,playing,sound,start,video)' ),
			array( 'fas fa-random' => 'random (arrows,shuffle,sort,swap,switch,transfer)' ),
			array( 'fas fa-recycle' => 'Recycle (Waste,compost,garbage,reuse,trash)' ),
			array( 'fas fa-redo' => 'Redo (forward,refresh,reload,repeat)' ),
			array( 'fas fa-redo-alt' => 'Alternate Redo (forward,refresh,reload,repeat)' ),
			array( 'fas fa-reply' => 'Reply (mail,message,respond)' ),
			array( 'fas fa-reply-all' => 'reply-all (mail,message,respond)' ),
			array( 'fas fa-retweet' => 'Retweet (refresh,reload,share,swap)' ),
			array( 'fas fa-share' => 'Share (forward,save,send,social)' ),
			array( 'fas fa-share-square' => 'Share Square (forward,save,send,social)' ),
			array( 'far fa-share-square' => 'Share Square (forward,save,send,social)' ),
			array( 'fas fa-sign-in-alt' => 'Alternate Sign In (arrow,enter,join,log in,login,sign in,sign up,sign-in,signin,signup)' ),
			array( 'fas fa-sign-out-alt' => 'Alternate Sign Out (arrow,exit,leave,log out,logout,sign-out)' ),
			array( 'fas fa-sort' => 'Sort (filter,order)' ),
			array( 'fas fa-sort-alpha-down' => 'Sort Alphabetical Down (alphabetical,arrange,filter,order,sort-alpha-asc)' ),
			array( 'fas fa-sort-alpha-down-alt' => 'Alternate Sort Alphabetical Down (alphabetical,arrange,filter,order,sort-alpha-asc)' ),
			array( 'fas fa-sort-alpha-up' => 'Sort Alphabetical Up (alphabetical,arrange,filter,order,sort-alpha-desc)' ),
			array( 'fas fa-sort-alpha-up-alt' => 'Alternate Sort Alphabetical Up (alphabetical,arrange,filter,order,sort-alpha-desc)' ),
			array( 'fas fa-sort-amount-down' => 'Sort Amount Down (arrange,filter,number,order,sort-amount-asc)' ),
			array( 'fas fa-sort-amount-down-alt' => 'Alternate Sort Amount Down (arrange,filter,order,sort-amount-asc)' ),
			array( 'fas fa-sort-amount-up' => 'Sort Amount Up (arrange,filter,order,sort-amount-desc)' ),
			array( 'fas fa-sort-amount-up-alt' => 'Alternate Sort Amount Up (arrange,filter,order,sort-amount-desc)' ),
			array( 'fas fa-sort-down' => 'Sort Down (Descending) (arrow,descending,filter,order,sort-desc)' ),
			array( 'fas fa-sort-numeric-down' => 'Sort Numeric Down (arrange,filter,numbers,order,sort-numeric-asc)' ),
			array( 'fas fa-sort-numeric-down-alt' => 'Alternate Sort Numeric Down (arrange,filter,numbers,order,sort-numeric-asc)' ),
			array( 'fas fa-sort-numeric-up' => 'Sort Numeric Up (arrange,filter,numbers,order,sort-numeric-desc)' ),
			array( 'fas fa-sort-numeric-up-alt' => 'Alternate Sort Numeric Up (arrange,filter,numbers,order,sort-numeric-desc)' ),
			array( 'fas fa-sort-up' => 'Sort Up (Ascending) (arrow,ascending,filter,order,sort-asc)' ),
			array( 'fas fa-sync' => 'Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'fas fa-sync-alt' => 'Alternate Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'fas fa-text-height' => 'text-height (edit,font,format,text,type)' ),
			array( 'fas fa-text-width' => 'Text Width (edit,font,format,text,type)' ),
			array( 'fas fa-undo' => 'Undo (back,control z,exchange,oops,return,rotate,swap)' ),
			array( 'fas fa-undo-alt' => 'Alternate Undo (back,control z,exchange,oops,return,swap)' ),
			array( 'fas fa-upload' => 'Upload (hard drive,import,publish)' ),
		),
		'Audio & Video' => array(
			array( 'fas fa-audio-description' => 'Audio Description (blind,narration,video,visual)' ),
			array( 'fas fa-backward' => 'backward (previous,rewind)' ),
			array( 'fas fa-broadcast-tower' => 'Broadcast Tower (airwaves,antenna,radio,reception,waves)' ),
			array( 'fas fa-circle' => 'Circle (circle-thin,diameter,dot,ellipse,notification,round)' ),
			array( 'far fa-circle' => 'Circle (circle-thin,diameter,dot,ellipse,notification,round)' ),
			array( 'fas fa-closed-captioning' => 'Closed Captioning (cc,deaf,hearing,subtitle,subtitling,text,video)' ),
			array( 'far fa-closed-captioning' => 'Closed Captioning (cc,deaf,hearing,subtitle,subtitling,text,video)' ),
			array( 'fas fa-compress' => 'Compress (collapse,fullscreen,minimize,move,resize,shrink,smaller)' ),
			array( 'fas fa-compress-arrows-alt' => 'Alternate Compress Arrows (collapse,fullscreen,minimize,move,resize,shrink,smaller)' ),
			array( 'fas fa-eject' => 'eject (abort,cancel,cd,discharge)' ),
			array( 'fas fa-expand' => 'Expand (arrow,bigger,enlarge,resize)' ),
			array( 'fas fa-expand-arrows-alt' => 'Alternate Expand Arrows (arrows-alt,bigger,enlarge,move,resize)' ),
			array( 'fas fa-fast-backward' => 'fast-backward (beginning,first,previous,rewind,start)' ),
			array( 'fas fa-fast-forward' => 'fast-forward (end,last,next)' ),
			array( 'fas fa-file-audio' => 'Audio File (document,mp3,music,page,play,sound)' ),
			array( 'far fa-file-audio' => 'Audio File (document,mp3,music,page,play,sound)' ),
			array( 'fas fa-file-video' => 'Video File (document,m4v,movie,mp4,play)' ),
			array( 'far fa-file-video' => 'Video File (document,m4v,movie,mp4,play)' ),
			array( 'fas fa-film' => 'Film (cinema,movie,strip,video)' ),
			array( 'fas fa-forward' => 'forward (forward,next,skip)' ),
			array( 'fas fa-headphones' => 'headphones (audio,listen,music,sound,speaker)' ),
			array( 'fas fa-microphone' => 'microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-microphone-alt' => 'Alternate Microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-microphone-alt-slash' => 'Alternate Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-microphone-slash' => 'Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-music' => 'Music (lyrics,melody,note,sing,sound)' ),
			array( 'fas fa-pause' => 'pause (hold,wait)' ),
			array( 'fas fa-pause-circle' => 'Pause Circle (hold,wait)' ),
			array( 'far fa-pause-circle' => 'Pause Circle (hold,wait)' ),
			array( 'fas fa-phone-volume' => 'Phone Volume (call,earphone,number,sound,support,telephone,voice,volume-control-phone)' ),
			array( 'fas fa-photo-video' => 'Photo Video (av,film,image,library,media)' ),
			array( 'fas fa-play' => 'play (audio,music,playing,sound,start,video)' ),
			array( 'fas fa-play-circle' => 'Play Circle (audio,music,playing,sound,start,video)' ),
			array( 'far fa-play-circle' => 'Play Circle (audio,music,playing,sound,start,video)' ),
			array( 'fas fa-podcast' => 'Podcast (audio,broadcast,music,sound)' ),
			array( 'fas fa-random' => 'random (arrows,shuffle,sort,swap,switch,transfer)' ),
			array( 'fas fa-redo' => 'Redo (forward,refresh,reload,repeat)' ),
			array( 'fas fa-redo-alt' => 'Alternate Redo (forward,refresh,reload,repeat)' ),
			array( 'fas fa-rss' => 'rss (blog,feed,journal,news,writing)' ),
			array( 'fas fa-rss-square' => 'RSS Square (blog,feed,journal,news,writing)' ),
			array( 'fas fa-step-backward' => 'step-backward (beginning,first,previous,rewind,start)' ),
			array( 'fas fa-step-forward' => 'step-forward (end,last,next)' ),
			array( 'fas fa-stop' => 'stop (block,box,square)' ),
			array( 'fas fa-stop-circle' => 'Stop Circle (block,box,circle,square)' ),
			array( 'far fa-stop-circle' => 'Stop Circle (block,box,circle,square)' ),
			array( 'fas fa-sync' => 'Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'fas fa-sync-alt' => 'Alternate Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'fas fa-tv' => 'Television (computer,display,monitor,television)' ),
			array( 'fas fa-undo' => 'Undo (back,control z,exchange,oops,return,rotate,swap)' ),
			array( 'fas fa-undo-alt' => 'Alternate Undo (back,control z,exchange,oops,return,swap)' ),
			array( 'fas fa-video' => 'Video (camera,film,movie,record,video-camera)' ),
			array( 'fas fa-volume-down' => 'Volume Down (audio,lower,music,quieter,sound,speaker)' ),
			array( 'fas fa-volume-mute' => 'Volume Mute (audio,music,quiet,sound,speaker)' ),
			array( 'fas fa-volume-off' => 'Volume Off (audio,ban,music,mute,quiet,silent,sound)' ),
			array( 'fas fa-volume-up' => 'Volume Up (audio,higher,louder,music,sound,speaker)' ),
			array( 'fab fa-youtube' => 'YouTube (film,video,youtube-play,youtube-square)' ),
		),
		'Automotive' => array(
			array( 'fas fa-air-freshener' => 'Air Freshener (car,deodorize,fresh,pine,scent)' ),
			array( 'fas fa-ambulance' => 'ambulance (emergency,emt,er,help,hospital,support,vehicle)' ),
			array( 'fas fa-bus' => 'Bus (public transportation,transportation,travel,vehicle)' ),
			array( 'fas fa-bus-alt' => 'Bus Alt (mta,public transportation,transportation,travel,vehicle)' ),
			array( 'fas fa-car' => 'Car (auto,automobile,sedan,transportation,travel,vehicle)' ),
			array( 'fas fa-car-alt' => 'Alternate Car (auto,automobile,sedan,transportation,travel,vehicle)' ),
			array( 'fas fa-car-battery' => 'Car Battery (auto,electric,mechanic,power)' ),
			array( 'fas fa-car-crash' => 'Car Crash (accident,auto,automobile,insurance,sedan,transportation,vehicle,wreck)' ),
			array( 'fas fa-car-side' => 'Car Side (auto,automobile,sedan,transportation,travel,vehicle)' ),
			array( 'fas fa-charging-station' => 'Charging Station (electric,ev,tesla,vehicle)' ),
			array( 'fas fa-gas-pump' => 'Gas Pump (car,fuel,gasoline,petrol)' ),
			array( 'fas fa-motorcycle' => 'Motorcycle (bike,machine,transportation,vehicle)' ),
			array( 'fas fa-oil-can' => 'Oil Can (auto,crude,gasoline,grease,lubricate,petroleum)' ),
			array( 'fas fa-shuttle-van' => 'Shuttle Van (airport,machine,public-transportation,transportation,travel,vehicle)' ),
			array( 'fas fa-tachometer-alt' => 'Alternate Tachometer (dashboard,fast,odometer,speed,speedometer)' ),
			array( 'fas fa-taxi' => 'Taxi (cab,cabbie,car,car service,lyft,machine,transportation,travel,uber,vehicle)' ),
			array( 'fas fa-truck' => 'truck (cargo,delivery,shipping,vehicle)' ),
			array( 'fas fa-truck-monster' => 'Truck Monster (offroad,vehicle,wheel)' ),
			array( 'fas fa-truck-pickup' => 'Truck Side (cargo,vehicle)' ),
		),
		'Autumn' => array(
			array( 'fas fa-apple-alt' => 'Fruit Apple (fall,fruit,fuji,macintosh,orchard,seasonal,vegan)' ),
			array( 'fas fa-campground' => 'Campground (camping,fall,outdoors,teepee,tent,tipi)' ),
			array( 'fas fa-cloud-sun' => 'Cloud with Sun (clear,day,daytime,fall,outdoors,overcast,partly cloudy)' ),
			array( 'fas fa-drumstick-bite' => 'Drumstick with Bite Taken Out (bone,chicken,leg,meat,poultry,turkey)' ),
			array( 'fas fa-football-ball' => 'Football Ball (ball,fall,nfl,pigskin,seasonal)' ),
			array( 'fas fa-hiking' => 'Hiking (activity,backpack,fall,fitness,outdoors,person,seasonal,walking)' ),
			array( 'fas fa-mountain' => 'Mountain (glacier,hiking,hill,landscape,travel,view)' ),
			array( 'fas fa-tractor' => 'Tractor (agriculture,farm,vehicle)' ),
			array( 'fas fa-tree' => 'Tree (bark,fall,flora,forest,nature,plant,seasonal)' ),
			array( 'fas fa-wind' => 'Wind (air,blow,breeze,fall,seasonal,weather)' ),
			array( 'fas fa-wine-bottle' => 'Wine Bottle (alcohol,beverage,cabernet,drink,glass,grapes,merlot,sauvignon)' ),
		),
		'Beverage' => array(
			array( 'fas fa-beer' => 'beer (alcohol,ale,bar,beverage,brewery,drink,lager,liquor,mug,stein)' ),
			array( 'fas fa-blender' => 'Blender (cocktail,milkshake,mixer,puree,smoothie)' ),
			array( 'fas fa-cocktail' => 'Cocktail (alcohol,beverage,drink,gin,glass,margarita,martini,vodka)' ),
			array( 'fas fa-coffee' => 'Coffee (beverage,breakfast,cafe,drink,fall,morning,mug,seasonal,tea)' ),
			array( 'fas fa-flask' => 'Flask (beaker,experimental,labs,science)' ),
			array( 'fas fa-glass-cheers' => 'Glass Cheers (alcohol,bar,beverage,celebration,champagne,clink,drink,holiday,new year\'s eve,party,toast)' ),
			array( 'fas fa-glass-martini' => 'Martini Glass (alcohol,bar,beverage,drink,liquor)' ),
			array( 'fas fa-glass-martini-alt' => 'Alternate Glass Martini (alcohol,bar,beverage,drink,liquor)' ),
			array( 'fas fa-glass-whiskey' => 'Glass Whiskey (alcohol,bar,beverage,bourbon,drink,liquor,neat,rye,scotch,whisky)' ),
			array( 'fas fa-mug-hot' => 'Mug Hot (caliente,cocoa,coffee,cup,drink,holiday,hot chocolate,steam,tea,warmth)' ),
			array( 'fas fa-wine-bottle' => 'Wine Bottle (alcohol,beverage,cabernet,drink,glass,grapes,merlot,sauvignon)' ),
			array( 'fas fa-wine-glass' => 'Wine Glass (alcohol,beverage,cabernet,drink,grapes,merlot,sauvignon)' ),
			array( 'fas fa-wine-glass-alt' => 'Alternate Wine Glas (alcohol,beverage,cabernet,drink,grapes,merlot,sauvignon)' ),
		),
		'Brands' => array(
			array('fab fa-500px' => '500px'),
			array('fab fa-accessible-icon' => 'accessible-icon'),
			array('fab fa-accusoft' => 'accusoft'),
			array('fab fa-adn' => 'adn'),
			array('fab fa-adversal' => 'adversal'),
			array('fab fa-affiliatetheme' => 'affiliatetheme'),
			array('fab fa-algolia' => 'algolia'),
			array('fab fa-amazon' => 'amazon'),
			array('fab fa-amazon-pay' => 'amazon-pay'),
			array('fab fa-amilia' => 'amilia'),
			array('fab fa-android' => 'android'),
			array('fab fa-angellist' => 'angellist'),
			array('fab fa-angrycreative' => 'angrycreative'),
			array('fab fa-angular' => 'angular'),
			array('fab fa-app-store' => 'app-store'),
			array('fab fa-app-store-ios' => 'app-store-ios'),
			array('fab fa-apper' => 'apper'),
			array('fab fa-apple' => 'apple'),
			array('fab fa-apple-pay' => 'apple-pay'),
			array('fab fa-asymmetrik' => 'asymmetrik'),
			array('fab fa-audible' => 'audible'),
			array('fab fa-autoprefixer' => 'autoprefixer'),
			array('fab fa-avianex' => 'avianex'),
			array('fab fa-aviato' => 'aviato'),
			array('fab fa-aws' => 'aws'),
			array('fab fa-bandcamp' => 'bandcamp'),
			array('fab fa-behance' => 'behance'),
			array('fab fa-behance-square' => 'behance-square'),
			array('fab fa-bimobject' => 'bimobject'),
			array('fab fa-bitbucket' => 'bitbucket'),
			array('fab fa-bitcoin' => 'bitcoin'),
			array('fab fa-bity' => 'bity'),
			array('fab fa-black-tie' => 'black-tie'),
			array('fab fa-blackberry' => 'blackberry'),
			array('fab fa-blogger' => 'blogger'),
			array('fab fa-blogger-b' => 'blogger-b'),
			array('fab fa-bluetooth' => 'bluetooth'),
			array('fab fa-bluetooth-b' => 'bluetooth-b'),
			array('fab fa-btc' => 'btc'),
			array('fab fa-buromobelexperte' => 'buromobelexperte'),
			array('fab fa-buysellads' => 'buysellads'),
			array('fab fa-cc-amazon-pay' => 'cc-amazon-pay'),
			array('fab fa-cc-amex' => 'cc-amex'),
			array('fab fa-cc-apple-pay' => 'cc-apple-pay'),
			array('fab fa-cc-diners-club' => 'cc-diners-club'),
			array('fab fa-cc-discover' => 'cc-discover'),
			array('fab fa-cc-jcb' => 'cc-jcb'),
			array('fab fa-cc-mastercard' => 'cc-mastercard'),
			array('fab fa-cc-paypal' => 'cc-paypal'),
			array('fab fa-cc-stripe' => 'cc-stripe'),
			array('fab fa-cc-visa' => 'cc-visa'),
			array('fab fa-centercode' => 'centercode'),
			array('fab fa-chrome' => 'chrome'),
			array('fab fa-cloudscale' => 'cloudscale'),
			array('fab fa-cloudsmith' => 'cloudsmith'),
			array('fab fa-cloudversify' => 'cloudversify'),
			array('fab fa-codepen' => 'codepen'),
			array('fab fa-codiepie' => 'codiepie'),
			array('fab fa-connectdevelop' => 'connectdevelop'),
			array('fab fa-contao' => 'contao'),
			array('fab fa-cpanel' => 'cpanel'),
			array('fab fa-creative-commons' => 'creative-commons'),
			array('fab fa-css3' => 'css3'),
			array('fab fa-css3-alt' => 'css3-alt'),
			array('fab fa-cuttlefish' => 'cuttlefish'),
			array('fab fa-d-and-d' => 'd-and-d'),
			array('fab fa-dashcube' => 'dashcube'),
			array('fab fa-delicious' => 'delicious'),
			array('fab fa-deploydog' => 'deploydog'),
			array('fab fa-deskpro' => 'deskpro'),
			array('fab fa-deviantart' => 'deviantart'),
			array('fab fa-digg' => 'digg'),
			array('fab fa-digital-ocean' => 'digital-ocean'),
			array('fab fa-discord' => 'discord'),
			array('fab fa-discourse' => 'discourse'),
			array('fab fa-dochub' => 'dochub'),
			array('fab fa-docker' => 'docker'),
			array('fab fa-draft2digital' => 'draft2digital'),
			array('fab fa-dribbble' => 'dribbble'),
			array('fab fa-dribbble-square' => 'dribbble-square'),
			array('fab fa-dropbox' => 'dropbox'),
			array('fab fa-drupal' => 'drupal'),
			array('fab fa-dyalog' => 'dyalog'),
			array('fab fa-earlybirds' => 'earlybirds'),
			array('fab fa-edge' => 'edge'),
			array('fab fa-elementor' => 'elementor'),
			array('fab fa-ember' => 'ember'),
			array('fab fa-empire' => 'empire'),
			array('fab fa-envira' => 'envira'),
			array('fab fa-erlang' => 'erlang'),
			array('fab fa-ethereum' => 'ethereum'),
			array('fab fa-etsy' => 'etsy'),
			array('fab fa-expeditedssl' => 'expeditedssl'),
			array('fab fa-facebook' => 'facebook'),
			array('fab fa-facebook-f' => 'facebook-f'),
			array('fab fa-facebook-messenger' => 'facebook-messenger'),
			array('fab fa-facebook-square' => 'facebook-square'),
			array('fab fa-firefox' => 'firefox'),
			array('fab fa-first-order' => 'first-order'),
			array('fab fa-firstdraft' => 'firstdraft'),
			array('fab fa-flickr' => 'flickr'),
			array('fab fa-flipboard' => 'flipboard'),
			array('fab fa-fly' => 'fly'),
			array('fab fa-font-awesome' => 'font-awesome'),
			array('fab fa-font-awesome-alt' => 'font-awesome-alt'),
			array('fab fa-font-awesome-flag' => 'font-awesome-flag'),
			array('fab fa-fonticons' => 'fonticons'),
			array('fab fa-fonticons-fi' => 'fonticons-fi'),
			array('fab fa-fort-awesome' => 'fort-awesome'),
			array('fab fa-fort-awesome-alt' => 'fort-awesome-alt'),
			array('fab fa-forumbee' => 'forumbee'),
			array('fab fa-foursquare' => 'foursquare'),
			array('fab fa-free-code-camp' => 'free-code-camp'),
			array('fab fa-freebsd' => 'freebsd'),
			array('fab fa-get-pocket' => 'get-pocket'),
			array('fab fa-gg' => 'gg'),
			array('fab fa-gg-circle' => 'gg-circle'),
			array('fab fa-git' => 'git'),
			array('fab fa-git-square' => 'git-square'),
			array('fab fa-github' => 'github'),
			array('fab fa-github-alt' => 'github-alt'),
			array('fab fa-github-square' => 'github-square'),
			array('fab fa-gitkraken' => 'gitkraken'),
			array('fab fa-gitlab' => 'gitlab'),
			array('fab fa-gitter' => 'gitter'),
			array('fab fa-glide' => 'glide'),
			array('fab fa-glide-g' => 'glide-g'),
			array('fab fa-gofore' => 'gofore'),
			array('fab fa-goodreads' => 'goodreads'),
			array('fab fa-goodreads-g' => 'goodreads-g'),
			array('fab fa-google' => 'google'),
			array('fab fa-google-drive' => 'google-drive'),
			array('fab fa-google-play' => 'google-play'),
			array('fab fa-google-plus' => 'google-plus'),
			array('fab fa-google-plus-g' => 'google-plus-g'),
			array('fab fa-google-plus-square' => 'google-plus-square'),
			array('fab fa-google-wallet' => 'google-wallet'),
			array('fab fa-gratipay' => 'gratipay'),
			array('fab fa-grav' => 'grav'),
			array('fab fa-gripfire' => 'gripfire'),
			array('fab fa-grunt' => 'grunt'),
			array('fab fa-gulp' => 'gulp'),
			array('fab fa-hacker-news' => 'hacker-news'),
			array('fab fa-hacker-news-square' => 'hacker-news-square'),
			array('fab fa-hips' => 'hips'),
			array('fab fa-hire-a-helper' => 'hire-a-helper'),
			array('fab fa-hooli' => 'hooli'),
			array('fab fa-hotjar' => 'hotjar'),
			array('fab fa-houzz' => 'houzz'),
			array('fab fa-html5' => 'html5'),
			array('fab fa-hubspot' => 'hubspot'),
			array('fab fa-imdb' => 'imdb'),
			array('fab fa-instagram' => 'instagram'),
			array('fab fa-internet-explorer' => 'internet-explorer'),
			array('fab fa-ioxhost' => 'ioxhost'),
			array('fab fa-itunes' => 'itunes'),
			array('fab fa-itunes-note' => 'itunes-note'),
			array('fab fa-jenkins' => 'jenkins'),
			array('fab fa-joget' => 'joget'),
			array('fab fa-joomla' => 'joomla'),
			array('fab fa-js' => 'js'),
			array('fab fa-js-square' => 'js-square'),
			array('fab fa-jsfiddle' => 'jsfiddle'),
			array('fab fa-keycdn' => 'keycdn'),
			array('fab fa-kickstarter' => 'kickstarter'),
			array('fab fa-kickstarter-k' => 'kickstarter-k'),
			array('fab fa-korvue' => 'korvue'),
			array('fab fa-laravel' => 'laravel'),
			array('fab fa-lastfm' => 'lastfm'),
			array('fab fa-lastfm-square' => 'lastfm-square'),
			array('fab fa-leanpub' => 'leanpub'),
			array('fab fa-less' => 'less'),
			array('fab fa-line' => 'line'),
			array('fab fa-linkedin' => 'linkedin'),
			array('fab fa-linkedin-in' => 'linkedin-in'),
			array('fab fa-linode' => 'linode'),
			array('fab fa-linux' => 'linux'),
			array('fab fa-lyft' => 'lyft'),
			array('fab fa-magento' => 'magento'),
			array('fab fa-maxcdn' => 'maxcdn'),
			array('fab fa-medapps' => 'medapps'),
			array('fab fa-medium' => 'medium'),
			array('fab fa-medium-m' => 'medium-m'),
			array('fab fa-medrt' => 'medrt'),
			array('fab fa-meetup' => 'meetup'),
			array('fab fa-microsoft' => 'microsoft'),
			array('fab fa-mix' => 'mix'),
			array('fab fa-mixcloud' => 'mixcloud'),
			array('fab fa-mizuni' => 'mizuni'),
			array('fab fa-modx' => 'modx'),
			array('fab fa-monero' => 'monero'),
			array('fab fa-napster' => 'napster'),
			array('fab fa-nintendo-switch' => 'nintendo-switch'),
			array('fab fa-node' => 'node'),
			array('fab fa-node-js' => 'node-js'),
			array('fab fa-npm' => 'npm'),
			array('fab fa-ns8' => 'ns8'),
			array('fab fa-nutritionix' => 'nutritionix'),
			array('fab fa-odnoklassniki' => 'odnoklassniki'),
			array('fab fa-odnoklassniki-square' => 'odnoklassniki-square'),
			array('fab fa-opencart' => 'opencart'),
			array('fab fa-openid' => 'openid'),
			array('fab fa-opera' => 'opera'),
			array('fab fa-optin-monster' => 'optin-monster'),
			array('fab fa-osi' => 'osi'),
			array('fab fa-page4' => 'page4'),
			array('fab fa-pagelines' => 'pagelines'),
			array('fab fa-palfed' => 'palfed'),
			array('fab fa-patreon' => 'patreon'),
			array('fab fa-paypal' => 'paypal'),
			array('fab fa-periscope' => 'periscope'),
			array('fab fa-phabricator' => 'phabricator'),
			array('fab fa-phoenix-framework' => 'phoenix-framework'),
			array('fab fa-php' => 'php'),
			array('fab fa-pied-piper' => 'pied-piper'),
			array('fab fa-pied-piper-alt' => 'pied-piper-alt'),
			array('fab fa-pied-piper-pp' => 'pied-piper-pp'),
			array('fab fa-pinterest' => 'pinterest'),
			array('fab fa-pinterest-p' => 'pinterest-p'),
			array('fab fa-pinterest-square' => 'pinterest-square'),
			array('fab fa-playstation' => 'playstation'),
			array('fab fa-product-hunt' => 'product-hunt'),
			array('fab fa-pushed' => 'pushed'),
			array('fab fa-python' => 'python'),
			array('fab fa-qq' => 'qq'),
			array('fab fa-quinscape' => 'quinscape'),
			array('fab fa-quora' => 'quora'),
			array('fab fa-ravelry' => 'ravelry'),
			array('fab fa-react' => 'react'),
			array('fab fa-rebel' => 'rebel'),
			array('fab fa-red-river' => 'red-river'),
			array('fab fa-reddit' => 'reddit'),
			array('fab fa-reddit-alien' => 'reddit-alien'),
			array('fab fa-reddit-square' => 'reddit-square'),
			array('fab fa-rendact' => 'rendact'),
			array('fab fa-renren' => 'renren'),
			array('fab fa-replyd' => 'replyd'),
			array('fab fa-resolving' => 'resolving'),
			array('fab fa-rocketchat' => 'rocketchat'),
			array('fab fa-rockrms' => 'rockrms'),
			array('fab fa-safari' => 'safari'),
			array('fab fa-sass' => 'sass'),
			array('fab fa-schlix' => 'schlix'),
			array('fab fa-scribd' => 'scribd'),
			array('fab fa-searchengin' => 'searchengin'),
			array('fab fa-sellcast' => 'sellcast'),
			array('fab fa-sellsy' => 'sellsy'),
			array('fab fa-servicestack' => 'servicestack'),
			array('fab fa-shirtsinbulk' => 'shirtsinbulk'),
			array('fab fa-simplybuilt' => 'simplybuilt'),
			array('fab fa-sistrix' => 'sistrix'),
			array('fab fa-skyatlas' => 'skyatlas'),
			array('fab fa-skype' => 'skype'),
			array('fab fa-slack' => 'slack'),
			array('fab fa-slack-hash' => 'slack-hash'),
			array('fab fa-slideshare' => 'slideshare'),
			array('fab fa-snapchat' => 'snapchat'),
			array('fab fa-snapchat-ghost' => 'snapchat-ghost'),
			array('fab fa-snapchat-square' => 'snapchat-square'),
			array('fab fa-soundcloud' => 'soundcloud'),
			array('fab fa-speakap' => 'speakap'),
			array('fab fa-spotify' => 'spotify'),
			array('fab fa-stack-exchange' => 'stack-exchange'),
			array('fab fa-stack-overflow' => 'stack-overflow'),
			array('fab fa-staylinked' => 'staylinked'),
			array('fab fa-steam' => 'steam'),
			array('fab fa-steam-square' => 'steam-square'),
			array('fab fa-steam-symbol' => 'steam-symbol'),
			array('fab fa-sticker-mule' => 'sticker-mule'),
			array('fab fa-strava' => 'strava'),
			array('fab fa-stripe' => 'stripe'),
			array('fab fa-stripe-s' => 'stripe-s'),
			array('fab fa-studiovinari' => 'studiovinari'),
			array('fab fa-stumbleupon' => 'stumbleupon'),
			array('fab fa-stumbleupon-circle' => 'stumbleupon-circle'),
			array('fab fa-superpowers' => 'superpowers'),
			array('fab fa-supple' => 'supple'),
			array('fab fa-telegram' => 'telegram'),
			array('fab fa-telegram-plane' => 'telegram-plane'),
			array('fab fa-tencent-weibo' => 'tencent-weibo'),
			array('fab fa-themeisle' => 'themeisle'),
			array('fab fa-trello' => 'trello'),
			array('fab fa-tripadvisor' => 'tripadvisor'),
			array('fab fa-tumblr' => 'tumblr'),
			array('fab fa-tumblr-square' => 'tumblr-square'),
			array('fab fa-twitch' => 'twitch'),
			array('fab fa-twitter' => 'twitter'),
			array('fab fa-twitter-square' => 'twitter-square'),
			array('fab fa-typo3' => 'typo3'),
			array('fab fa-uber' => 'uber'),
			array('fab fa-uikit' => 'uikit'),
			array('fab fa-uniregistry' => 'uniregistry'),
			array('fab fa-untappd' => 'untappd'),
			array('fab fa-usb' => 'usb'),
			array('fab fa-ussunnah' => 'ussunnah'),
			array('fab fa-vaadin' => 'vaadin'),
			array('fab fa-viacoin' => 'viacoin'),
			array('fab fa-viadeo' => 'viadeo'),
			array('fab fa-viadeo-square' => 'viadeo-square'),
			array('fab fa-viber' => 'viber'),
			array('fab fa-vimeo' => 'vimeo'),
			array('fab fa-vimeo-square' => 'vimeo-square'),
			array('fab fa-vimeo-v' => 'vimeo-v'),
			array('fab fa-vine' => 'vine'),
			array('fab fa-vk' => 'vk'),
			array('fab fa-vnv' => 'vnv'),
			array('fab fa-vuejs' => 'vuejs'),
			array('fab fa-weibo' => 'weibo'),
			array('fab fa-weixin' => 'weixin'),
			array('fab fa-whatsapp' => 'whatsapp'),
			array('fab fa-whatsapp-square' => 'whatsapp-square'),
			array('fab fa-whmcs' => 'whmcs'),
			array('fab fa-wikipedia-w' => 'wikipedia-w'),
			array('fab fa-windows' => 'windows'),
			array('fab fa-wordpress' => 'wordpress'),
			array('fab fa-wordpress-simple' => 'wordpress-simple'),
			array('fab fa-wpbeginner' => 'wpbeginner'),
			array('fab fa-wpexplorer' => 'wpexplorer'),
			array('fab fa-wpforms' => 'wpforms'),
			array('fab fa-xbox' => 'xbox'),
			array('fab fa-xing' => 'xing'),
			array('fab fa-xing-square' => 'xing-square'),
			array('fab fa-y-combinator' => 'y-combinator'),
			array('fab fa-yahoo' => 'yahoo'),
			array('fab fa-yandex' => 'yandex'),
			array('fab fa-yandex-international' => 'yandex-international'),
			array('fab fa-yelp' => 'yelp'),
			array('fab fa-yoast' => 'yoast'),
			array('fab fa-youtube' => 'youtube'),
			array('fab fa-youtube-square' => 'youtube-square'),
		),
		'Buildings' => array(
			array( 'fas fa-archway' => 'Archway (arc,monument,road,street,tunnel)' ),
			array( 'fas fa-building' => 'Building (apartment,business,city,company,office,work)' ),
			array( 'far fa-building' => 'Building (apartment,business,city,company,office,work)' ),
			array( 'fas fa-campground' => 'Campground (camping,fall,outdoors,teepee,tent,tipi)' ),
			array( 'fas fa-church' => 'Church (building,cathedral,chapel,community,religion)' ),
			array( 'fas fa-city' => 'City (buildings,busy,skyscrapers,urban,windows)' ),
			array( 'fas fa-clinic-medical' => 'Medical Clinic (doctor,general practitioner,hospital,infirmary,medicine,office,outpatient)' ),
			array( 'fas fa-dungeon' => 'Dungeon (Dungeons & Dragons,building,d&d,dnd,door,entrance,fantasy,gate)' ),
			array( 'fas fa-gopuram' => 'Gopuram (building,entrance,hinduism,temple,tower)' ),
			array( 'fas fa-home' => 'home (abode,building,house,main)' ),
			array( 'fas fa-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'far fa-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'fas fa-hospital-alt' => 'Alternate Hospital (building,emergency room,medical center)' ),
			array( 'fas fa-hotel' => 'Hotel (building,inn,lodging,motel,resort,travel)' ),
			array( 'fas fa-house-damage' => 'Damaged House (building,devastation,disaster,home,insurance)' ),
			array( 'fas fa-igloo' => 'Igloo (dome,dwelling,eskimo,home,house,ice,snow)' ),
			array( 'fas fa-industry' => 'Industry (building,factory,industrial,manufacturing,mill,warehouse)' ),
			array( 'fas fa-kaaba' => 'Kaaba (building,cube,islam,muslim)' ),
			array( 'fas fa-landmark' => 'Landmark (building,historic,memorable,monument,politics)' ),
			array( 'fas fa-monument' => 'Monument (building,historic,landmark,memorable)' ),
			array( 'fas fa-mosque' => 'Mosque (building,islam,landmark,muslim)' ),
			array( 'fas fa-place-of-worship' => 'Place of Worship (building,church,holy,mosque,synagogue)' ),
			array( 'fas fa-school' => 'School (building,education,learn,student,teacher)' ),
			array( 'fas fa-store' => 'Store (building,buy,purchase,shopping)' ),
			array( 'fas fa-store-alt' => 'Alternate Store (building,buy,purchase,shopping)' ),
			array( 'fas fa-synagogue' => 'Synagogue (building,jewish,judaism,religion,star of david,temple)' ),
			array( 'fas fa-torii-gate' => 'Torii Gate (building,shintoism)' ),
			array( 'fas fa-university' => 'University (bank,building,college,higher education - students,institution)' ),
			array( 'fas fa-vihara' => 'Vihara (buddhism,buddhist,building,monastery)' ),
			array( 'fas fa-warehouse' => 'Warehouse (building,capacity,garage,inventory,storage)' ),
		),
		'Business' => array(
			array( 'fas fa-address-book' => 'Address Book (contact,directory,index,little black book,rolodex)' ),
			array( 'far fa-address-book' => 'Address Book (contact,directory,index,little black book,rolodex)' ),
			array( 'fas fa-address-card' => 'Address Card (about,contact,id,identification,postcard,profile)' ),
			array( 'far fa-address-card' => 'Address Card (about,contact,id,identification,postcard,profile)' ),
			array( 'fas fa-archive' => 'Archive (box,package,save,storage)' ),
			array( 'fas fa-balance-scale' => 'Balance Scale (balanced,justice,legal,measure,weight)' ),
			array( 'fas fa-balance-scale-left' => 'Balance Scale (Left-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'fas fa-balance-scale-right' => 'Balance Scale (Right-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'fas fa-birthday-cake' => 'Birthday Cake (anniversary,bakery,candles,celebration,dessert,frosting,holiday,party,pastry)' ),
			array( 'fas fa-book' => 'book (diary,documentation,journal,library,read)' ),
			array( 'fas fa-briefcase' => 'Briefcase (bag,business,luggage,office,work)' ),
			array( 'fas fa-building' => 'Building (apartment,business,city,company,office,work)' ),
			array( 'far fa-building' => 'Building (apartment,business,city,company,office,work)' ),
			array( 'fas fa-bullhorn' => 'bullhorn (announcement,broadcast,louder,megaphone,share)' ),
			array( 'fas fa-bullseye' => 'Bullseye (archery,goal,objective,target)' ),
			array( 'fas fa-business-time' => 'Business Time (alarm,briefcase,business socks,clock,flight of the conchords,reminder,wednesday)' ),
			array( 'fas fa-calculator' => 'Calculator (abacus,addition,arithmetic,counting,math,multiplication,subtraction)' ),
			array( 'fas fa-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'far fa-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'fas fa-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'far fa-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'fas fa-certificate' => 'certificate (badge,star,verified)' ),
			array( 'fas fa-chart-area' => 'Area Chart (analytics,area,chart,graph)' ),
			array( 'fas fa-chart-bar' => 'Bar Chart (analytics,bar,chart,graph)' ),
			array( 'far fa-chart-bar' => 'Bar Chart (analytics,bar,chart,graph)' ),
			array( 'fas fa-chart-line' => 'Line Chart (activity,analytics,chart,dashboard,gain,graph,increase,line)' ),
			array( 'fas fa-chart-pie' => 'Pie Chart (analytics,chart,diagram,graph,pie)' ),
			array( 'fas fa-city' => 'City (buildings,busy,skyscrapers,urban,windows)' ),
			array( 'fas fa-clipboard' => 'Clipboard (copy,notes,paste,record)' ),
			array( 'far fa-clipboard' => 'Clipboard (copy,notes,paste,record)' ),
			array( 'fas fa-coffee' => 'Coffee (beverage,breakfast,cafe,drink,fall,morning,mug,seasonal,tea)' ),
			array( 'fas fa-columns' => 'Columns (browser,dashboard,organize,panes,split)' ),
			array( 'fas fa-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'far fa-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'fas fa-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'far fa-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'fas fa-copyright' => 'Copyright (brand,mark,register,trademark)' ),
			array( 'far fa-copyright' => 'Copyright (brand,mark,register,trademark)' ),
			array( 'fas fa-cut' => 'Cut (clip,scissors,snip)' ),
			array( 'fas fa-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'far fa-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'fas fa-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'far fa-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'fas fa-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'far fa-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'fas fa-envelope-square' => 'Envelope Square (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'fas fa-eraser' => 'eraser (art,delete,remove,rubber)' ),
			array( 'fas fa-fax' => 'Fax (business,communicate,copy,facsimile,send)' ),
			array( 'fas fa-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'far fa-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'fas fa-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'far fa-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'fas fa-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'far fa-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'fas fa-folder-minus' => 'Folder Minus (archive,delete,directory,document,file,negative,remove)' ),
			array( 'fas fa-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'far fa-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'fas fa-folder-plus' => 'Folder Plus (add,archive,create,directory,document,file,new,positive)' ),
			array( 'fas fa-glasses' => 'Glasses (hipster,nerd,reading,sight,spectacles,vision)' ),
			array( 'fas fa-globe' => 'Globe (all,coordinates,country,earth,global,gps,language,localize,location,map,online,place,planet,translate,travel,world)' ),
			array( 'fas fa-highlighter' => 'Highlighter (edit,marker,sharpie,update,write)' ),
			array( 'fas fa-industry' => 'Industry (building,factory,industrial,manufacturing,mill,warehouse)' ),
			array( 'fas fa-landmark' => 'Landmark (building,historic,memorable,monument,politics)' ),
			array( 'fas fa-marker' => 'Marker (design,edit,sharpie,update,write)' ),
			array( 'fas fa-paperclip' => 'Paperclip (attach,attachment,connect,link)' ),
			array( 'fas fa-paste' => 'Paste (clipboard,copy,document,paper)' ),
			array( 'fas fa-pen' => 'Pen (design,edit,update,write)' ),
			array( 'fas fa-pen-alt' => 'Alternate Pen (design,edit,update,write)' ),
			array( 'fas fa-pen-fancy' => 'Pen Fancy (design,edit,fountain pen,update,write)' ),
			array( 'fas fa-pen-nib' => 'Pen Nib (design,edit,fountain pen,update,write)' ),
			array( 'fas fa-pen-square' => 'Pen Square (edit,pencil-square,update,write)' ),
			array( 'fas fa-pencil-alt' => 'Alternate Pencil (design,edit,pencil,update,write)' ),
			array( 'fas fa-percent' => 'Percent (discount,fraction,proportion,rate,ratio)' ),
			array( 'fas fa-phone' => 'Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'fas fa-phone-alt' => 'Alternate Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'fas fa-phone-slash' => 'Phone Slash (call,cancel,earphone,mute,number,support,telephone,voice)' ),
			array( 'fas fa-phone-square' => 'Phone Square (call,earphone,number,support,telephone,voice)' ),
			array( 'fas fa-phone-square-alt' => 'Alternate Phone Square (call,earphone,number,support,telephone,voice)' ),
			array( 'fas fa-phone-volume' => 'Phone Volume (call,earphone,number,sound,support,telephone,voice,volume-control-phone)' ),
			array( 'fas fa-print' => 'print (business,copy,document,office,paper)' ),
			array( 'fas fa-project-diagram' => 'Project Diagram (chart,graph,network,pert)' ),
			array( 'fas fa-registered' => 'Registered Trademark (copyright,mark,trademark)' ),
			array( 'far fa-registered' => 'Registered Trademark (copyright,mark,trademark)' ),
			array( 'fas fa-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'far fa-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'fas fa-sitemap' => 'Sitemap (directory,hierarchy,ia,information architecture,organization)' ),
			array( 'fas fa-socks' => 'Socks (business socks,business time,clothing,feet,flight of the conchords,wednesday)' ),
			array( 'fas fa-sticky-note' => 'Sticky Note (message,note,paper,reminder,sticker)' ),
			array( 'far fa-sticky-note' => 'Sticky Note (message,note,paper,reminder,sticker)' ),
			array( 'fas fa-stream' => 'Stream (flow,list,timeline)' ),
			array( 'fas fa-table' => 'table (data,excel,spreadsheet)' ),
			array( 'fas fa-tag' => 'tag (discount,label,price,shopping)' ),
			array( 'fas fa-tags' => 'tags (discount,label,price,shopping)' ),
			array( 'fas fa-tasks' => 'Tasks (checklist,downloading,downloads,loading,progress,project management,settings,to do)' ),
			array( 'fas fa-thumbtack' => 'Thumbtack (coordinates,location,marker,pin,thumb-tack)' ),
			array( 'fas fa-trademark' => 'Trademark (copyright,register,symbol)' ),
			array( 'fas fa-wallet' => 'Wallet (billfold,cash,currency,money)' ),
		),
		'Camping' => array(
			array( 'fas fa-binoculars' => 'Binoculars (glasses,magnify,scenic,spyglass,view)' ),
			array( 'fas fa-campground' => 'Campground (camping,fall,outdoors,teepee,tent,tipi)' ),
			array( 'fas fa-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'far fa-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'fas fa-fire' => 'fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'fas fa-fire-alt' => 'Alternate Fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'fas fa-first-aid' => 'First Aid (emergency,emt,health,medical,rescue)' ),
			array( 'fas fa-frog' => 'Frog (amphibian,bullfrog,fauna,hop,kermit,kiss,prince,ribbit,toad,wart)' ),
			array( 'fas fa-hiking' => 'Hiking (activity,backpack,fall,fitness,outdoors,person,seasonal,walking)' ),
			array( 'fas fa-map' => 'Map (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'far fa-map' => 'Map (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'fas fa-map-marked' => 'Map Marked (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'fas fa-map-marked-alt' => 'Alternate Map Marked (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'fas fa-map-signs' => 'Map Signs (directions,directory,map,signage,wayfinding)' ),
			array( 'fas fa-mountain' => 'Mountain (glacier,hiking,hill,landscape,travel,view)' ),
			array( 'fas fa-route' => 'Route (directions,navigation,travel)' ),
			array( 'fas fa-toilet-paper' => 'Toilet Paper (bathroom,halloween,holiday,lavatory,prank,restroom,roll)' ),
			array( 'fas fa-tree' => 'Tree (bark,fall,flora,forest,nature,plant,seasonal)' ),
		),
		'Charity' => array(
			array( 'fas fa-dollar-sign' => 'Dollar Sign ($,cost,dollar-sign,money,price,usd)' ),
			array( 'fas fa-donate' => 'Donate (contribute,generosity,gift,give)' ),
			array( 'fas fa-dove' => 'Dove (bird,fauna,flying,peace,war)' ),
			array( 'fas fa-gift' => 'gift (christmas,generosity,giving,holiday,party,present,wrapped,xmas)' ),
			array( 'fas fa-globe' => 'Globe (all,coordinates,country,earth,global,gps,language,localize,location,map,online,place,planet,translate,travel,world)' ),
			array( 'fas fa-hand-holding-heart' => 'Hand Holding Heart (carry,charity,gift,lift,package)' ),
			array( 'fas fa-hand-holding-usd' => 'Hand Holding US Dollar ($,carry,dollar sign,donation,giving,lift,money,price)' ),
			array( 'fas fa-hands-helping' => 'Helping Hands (aid,assistance,handshake,partnership,volunteering)' ),
			array( 'fas fa-handshake' => 'Handshake (agreement,greeting,meeting,partnership)' ),
			array( 'far fa-handshake' => 'Handshake (agreement,greeting,meeting,partnership)' ),
			array( 'fas fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'far fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'fas fa-leaf' => 'leaf (eco,flora,nature,plant,vegan)' ),
			array( 'fas fa-parachute-box' => 'Parachute Box (aid,assistance,rescue,supplies)' ),
			array( 'fas fa-piggy-bank' => 'Piggy Bank (bank,save,savings)' ),
			array( 'fas fa-ribbon' => 'Ribbon (badge,cause,lapel,pin)' ),
			array( 'fas fa-seedling' => 'Seedling (flora,grow,plant,vegan)' ),
		),
		'Chat' => array(
			array( 'fas fa-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'far fa-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'fas fa-comment-alt' => 'Alternate Comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'far fa-comment-alt' => 'Alternate Comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'fas fa-comment-dots' => 'Comment Dots (bubble,chat,commenting,conversation,feedback,message,more,note,notification,reply,sms,speech,texting)' ),
			array( 'far fa-comment-dots' => 'Comment Dots (bubble,chat,commenting,conversation,feedback,message,more,note,notification,reply,sms,speech,texting)' ),
			array( 'fas fa-comment-medical' => 'Alternate Medical Chat (advice,bubble,chat,commenting,conversation,diagnose,feedback,message,note,notification,prescription,sms,speech,texting)' ),
			array( 'fas fa-comment-slash' => 'Comment Slash (bubble,cancel,chat,commenting,conversation,feedback,message,mute,note,notification,quiet,sms,speech,texting)' ),
			array( 'fas fa-comments' => 'comments (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'far fa-comments' => 'comments (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'fas fa-frown' => 'Frowning Face (disapprove,emoticon,face,rating,sad)' ),
			array( 'far fa-frown' => 'Frowning Face (disapprove,emoticon,face,rating,sad)' ),
			array( 'fas fa-icons' => 'Icons (bolt,emoji,heart,image,music,photo,symbols)' ),
			array( 'fas fa-meh' => 'Neutral Face (emoticon,face,neutral,rating)' ),
			array( 'far fa-meh' => 'Neutral Face (emoticon,face,neutral,rating)' ),
			array( 'fas fa-phone' => 'Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'fas fa-phone-alt' => 'Alternate Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'fas fa-phone-slash' => 'Phone Slash (call,cancel,earphone,mute,number,support,telephone,voice)' ),
			array( 'fas fa-poo' => 'Poo (crap,poop,shit,smile,turd)' ),
			array( 'fas fa-quote-left' => 'quote-left (mention,note,phrase,text,type)' ),
			array( 'fas fa-quote-right' => 'quote-right (mention,note,phrase,text,type)' ),
			array( 'fas fa-smile' => 'Smiling Face (approve,emoticon,face,happy,rating,satisfied)' ),
			array( 'far fa-smile' => 'Smiling Face (approve,emoticon,face,happy,rating,satisfied)' ),
			array( 'fas fa-sms' => 'SMS (chat,conversation,message,mobile,notification,phone,sms,texting)' ),
			array( 'fas fa-video' => 'Video (camera,film,movie,record,video-camera)' ),
			array( 'fas fa-video-slash' => 'Video Slash (add,create,film,new,positive,record,video)' ),
		),
		'Chess' => array(
			array( 'fas fa-chess' => 'Chess (board,castle,checkmate,game,king,rook,strategy,tournament)' ),
			array( 'fas fa-chess-bishop' => 'Chess Bishop (board,checkmate,game,strategy)' ),
			array( 'fas fa-chess-board' => 'Chess Board (board,checkmate,game,strategy)' ),
			array( 'fas fa-chess-king' => 'Chess King (board,checkmate,game,strategy)' ),
			array( 'fas fa-chess-knight' => 'Chess Knight (board,checkmate,game,horse,strategy)' ),
			array( 'fas fa-chess-pawn' => 'Chess Pawn (board,checkmate,game,strategy)' ),
			array( 'fas fa-chess-queen' => 'Chess Queen (board,checkmate,game,strategy)' ),
			array( 'fas fa-chess-rook' => 'Chess Rook (board,castle,checkmate,game,strategy)' ),
			array( 'fas fa-square-full' => 'Square Full (block,box,shape)' ),
		),
		'Childhood' => array(
			array( 'fas fa-apple-alt' => 'Fruit Apple (fall,fruit,fuji,macintosh,orchard,seasonal,vegan)' ),
			array( 'fas fa-baby' => 'Baby (child,diaper,doll,human,infant,kid,offspring,person,sprout)' ),
			array( 'fas fa-baby-carriage' => 'Baby Carriage (buggy,carrier,infant,push,stroller,transportation,walk,wheels)' ),
			array( 'fas fa-bath' => 'Bath (clean,shower,tub,wash)' ),
			array( 'fas fa-biking' => 'Biking (bicycle,bike,cycle,cycling,ride,wheel)' ),
			array( 'fas fa-birthday-cake' => 'Birthday Cake (anniversary,bakery,candles,celebration,dessert,frosting,holiday,party,pastry)' ),
			array( 'fas fa-cookie' => 'Cookie (baked good,chips,chocolate,eat,snack,sweet,treat)' ),
			array( 'fas fa-cookie-bite' => 'Cookie Bite (baked good,bitten,chips,chocolate,eat,snack,sweet,treat)' ),
			array( 'fas fa-gamepad' => 'Gamepad (arcade,controller,d-pad,joystick,video,video game)' ),
			array( 'fas fa-ice-cream' => 'Ice Cream (chocolate,cone,dessert,frozen,scoop,sorbet,vanilla,yogurt)' ),
			array( 'fas fa-mitten' => 'Mitten (clothing,cold,glove,hands,knitted,seasonal,warmth)' ),
			array( 'fas fa-robot' => 'Robot (android,automate,computer,cyborg)' ),
			array( 'fas fa-school' => 'School (building,education,learn,student,teacher)' ),
			array( 'fas fa-shapes' => 'Shapes (blocks,build,circle,square,triangle)' ),
			array( 'fas fa-snowman' => 'Snowman (decoration,frost,frosty,holiday)' ),
		),
		'Clothing' => array(
			array( 'fas fa-graduation-cap' => 'Graduation Cap (ceremony,college,graduate,learning,school,student)' ),
			array( 'fas fa-hat-cowboy' => 'Cowboy Hat (buckaroo,horse,jackeroo,john b.,old west,pardner,ranch,rancher,rodeo,western,wrangler)' ),
			array( 'fas fa-hat-cowboy-side' => 'Cowboy Hat Side (buckaroo,horse,jackeroo,john b.,old west,pardner,ranch,rancher,rodeo,western,wrangler)' ),
			array( 'fas fa-hat-wizard' => 'Wizard\'s Hat (Dungeons & Dragons,accessory,buckle,clothing,d&d,dnd,fantasy,halloween,head,holiday,mage,magic,pointy,witch)' ),
			array( 'fas fa-mitten' => 'Mitten (clothing,cold,glove,hands,knitted,seasonal,warmth)' ),
			array( 'fas fa-shoe-prints' => 'Shoe Prints (feet,footprints,steps,walk)' ),
			array( 'fas fa-socks' => 'Socks (business socks,business time,clothing,feet,flight of the conchords,wednesday)' ),
			array( 'fas fa-tshirt' => 'T-Shirt (clothing,fashion,garment,shirt)' ),
			array( 'fas fa-user-tie' => 'User Tie (avatar,business,clothing,formal,professional,suit)' ),
		),
		'Code' => array(
			array( 'fas fa-archive' => 'Archive (box,package,save,storage)' ),
			array( 'fas fa-barcode' => 'barcode (info,laser,price,scan,upc)' ),
			array( 'fas fa-bath' => 'Bath (clean,shower,tub,wash)' ),
			array( 'fas fa-bug' => 'Bug (beetle,error,insect,report)' ),
			array( 'fas fa-code' => 'Code (brackets,code,development,html)' ),
			array( 'fas fa-code-branch' => 'Code Branch (branch,code-fork,fork,git,github,rebase,svn,vcs,version)' ),
			array( 'fas fa-coffee' => 'Coffee (beverage,breakfast,cafe,drink,fall,morning,mug,seasonal,tea)' ),
			array( 'fas fa-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'far fa-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'fas fa-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'far fa-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'fas fa-file-code' => 'Code File (css,development,document,html)' ),
			array( 'far fa-file-code' => 'Code File (css,development,document,html)' ),
			array( 'fas fa-filter' => 'Filter (funnel,options,separate,sort)' ),
			array( 'fas fa-fire-extinguisher' => 'fire-extinguisher (burn,caliente,fire fighter,flame,heat,hot,rescue)' ),
			array( 'fas fa-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'far fa-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'fas fa-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'far fa-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'fas fa-keyboard' => 'Keyboard (accessory,edit,input,text,type,write)' ),
			array( 'far fa-keyboard' => 'Keyboard (accessory,edit,input,text,type,write)' ),
			array( 'fas fa-laptop-code' => 'Laptop Code (computer,cpu,dell,demo,develop,device,mac,macbook,machine,pc)' ),
			array( 'fas fa-microchip' => 'Microchip (cpu,hardware,processor,technology)' ),
			array( 'fas fa-project-diagram' => 'Project Diagram (chart,graph,network,pert)' ),
			array( 'fas fa-qrcode' => 'qrcode (barcode,info,information,scan)' ),
			array( 'fas fa-shield-alt' => 'Alternate Shield (achievement,award,block,defend,security,winner)' ),
			array( 'fas fa-sitemap' => 'Sitemap (directory,hierarchy,ia,information architecture,organization)' ),
			array( 'fas fa-stream' => 'Stream (flow,list,timeline)' ),
			array( 'fas fa-terminal' => 'Terminal (code,command,console,development,prompt)' ),
			array( 'fas fa-user-secret' => 'User Secret (clothing,coat,hat,incognito,person,privacy,spy,whisper)' ),
			array( 'fas fa-window-close' => 'Window Close (browser,cancel,computer,development)' ),
			array( 'far fa-window-close' => 'Window Close (browser,cancel,computer,development)' ),
			array( 'fas fa-window-maximize' => 'Window Maximize (browser,computer,development,expand)' ),
			array( 'far fa-window-maximize' => 'Window Maximize (browser,computer,development,expand)' ),
			array( 'fas fa-window-minimize' => 'Window Minimize (browser,collapse,computer,development)' ),
			array( 'far fa-window-minimize' => 'Window Minimize (browser,collapse,computer,development)' ),
			array( 'fas fa-window-restore' => 'Window Restore (browser,computer,development)' ),
			array( 'far fa-window-restore' => 'Window Restore (browser,computer,development)' ),
		),
		'Communication' => array(
			array( 'fas fa-address-book' => 'Address Book (contact,directory,index,little black book,rolodex)' ),
			array( 'far fa-address-book' => 'Address Book (contact,directory,index,little black book,rolodex)' ),
			array( 'fas fa-address-card' => 'Address Card (about,contact,id,identification,postcard,profile)' ),
			array( 'far fa-address-card' => 'Address Card (about,contact,id,identification,postcard,profile)' ),
			array( 'fas fa-american-sign-language-interpreting' => 'American Sign Language Interpreting (asl,deaf,finger,hand,interpret,speak)' ),
			array( 'fas fa-assistive-listening-systems' => 'Assistive Listening Systems (amplify,audio,deaf,ear,headset,hearing,sound)' ),
			array( 'fas fa-at' => 'At (address,author,e-mail,email,handle)' ),
			array( 'fas fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'far fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'fas fa-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'far fa-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'fab fa-bluetooth' => 'Bluetooth' ),
			array( 'fab fa-bluetooth-b' => 'Bluetooth' ),
			array( 'fas fa-broadcast-tower' => 'Broadcast Tower (airwaves,antenna,radio,reception,waves)' ),
			array( 'fas fa-bullhorn' => 'bullhorn (announcement,broadcast,louder,megaphone,share)' ),
			array( 'fas fa-chalkboard' => 'Chalkboard (blackboard,learning,school,teaching,whiteboard,writing)' ),
			array( 'fas fa-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'far fa-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'fas fa-comment-alt' => 'Alternate Comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'far fa-comment-alt' => 'Alternate Comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'fas fa-comments' => 'comments (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'far fa-comments' => 'comments (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'fas fa-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'far fa-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'fas fa-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'far fa-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'fas fa-envelope-square' => 'Envelope Square (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'fas fa-fax' => 'Fax (business,communicate,copy,facsimile,send)' ),
			array( 'fas fa-inbox' => 'inbox (archive,desk,email,mail,message)' ),
			array( 'fas fa-language' => 'Language (dialect,idiom,localize,speech,translate,vernacular)' ),
			array( 'fas fa-microphone' => 'microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-microphone-alt' => 'Alternate Microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-microphone-alt-slash' => 'Alternate Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-microphone-slash' => 'Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-mobile' => 'Mobile Phone (apple,call,cell phone,cellphone,device,iphone,number,screen,telephone)' ),
			array( 'fas fa-mobile-alt' => 'Alternate Mobile (apple,call,cell phone,cellphone,device,iphone,number,screen,telephone)' ),
			array( 'fas fa-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'far fa-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'fas fa-phone' => 'Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'fas fa-phone-alt' => 'Alternate Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'fas fa-phone-slash' => 'Phone Slash (call,cancel,earphone,mute,number,support,telephone,voice)' ),
			array( 'fas fa-phone-square' => 'Phone Square (call,earphone,number,support,telephone,voice)' ),
			array( 'fas fa-phone-square-alt' => 'Alternate Phone Square (call,earphone,number,support,telephone,voice)' ),
			array( 'fas fa-phone-volume' => 'Phone Volume (call,earphone,number,sound,support,telephone,voice,volume-control-phone)' ),
			array( 'fas fa-rss' => 'rss (blog,feed,journal,news,writing)' ),
			array( 'fas fa-rss-square' => 'RSS Square (blog,feed,journal,news,writing)' ),
			array( 'fas fa-tty' => 'TTY (communication,deaf,telephone,teletypewriter,text)' ),
			array( 'fas fa-voicemail' => 'Voicemail (answer,inbox,message,phone)' ),
			array( 'fas fa-wifi' => 'WiFi (connection,hotspot,internet,network,wireless)' ),
		),
		'Computers' => array(
			array( 'fas fa-database' => 'Database (computer,development,directory,memory,storage)' ),
			array( 'fas fa-desktop' => 'Desktop (computer,cpu,demo,desktop,device,imac,machine,monitor,pc,screen)' ),
			array( 'fas fa-download' => 'Download (export,hard drive,save,transfer)' ),
			array( 'fas fa-ethernet' => 'Ethernet (cable,cat 5,cat 6,connection,hardware,internet,network,wired)' ),
			array( 'fas fa-hdd' => 'HDD (cpu,hard drive,harddrive,machine,save,storage)' ),
			array( 'far fa-hdd' => 'HDD (cpu,hard drive,harddrive,machine,save,storage)' ),
			array( 'fas fa-headphones' => 'headphones (audio,listen,music,sound,speaker)' ),
			array( 'fas fa-keyboard' => 'Keyboard (accessory,edit,input,text,type,write)' ),
			array( 'far fa-keyboard' => 'Keyboard (accessory,edit,input,text,type,write)' ),
			array( 'fas fa-laptop' => 'Laptop (computer,cpu,dell,demo,device,mac,macbook,machine,pc)' ),
			array( 'fas fa-memory' => 'Memory (DIMM,RAM,hardware,storage,technology)' ),
			array( 'fas fa-microchip' => 'Microchip (cpu,hardware,processor,technology)' ),
			array( 'fas fa-mobile' => 'Mobile Phone (apple,call,cell phone,cellphone,device,iphone,number,screen,telephone)' ),
			array( 'fas fa-mobile-alt' => 'Alternate Mobile (apple,call,cell phone,cellphone,device,iphone,number,screen,telephone)' ),
			array( 'fas fa-mouse' => 'Mouse (click,computer,cursor,input,peripheral)' ),
			array( 'fas fa-plug' => 'Plug (connect,electric,online,power)' ),
			array( 'fas fa-power-off' => 'Power Off (cancel,computer,on,reboot,restart)' ),
			array( 'fas fa-print' => 'print (business,copy,document,office,paper)' ),
			array( 'fas fa-satellite' => 'Satellite (communications,hardware,orbit,space)' ),
			array( 'fas fa-satellite-dish' => 'Satellite Dish (SETI,communications,hardware,receiver,saucer,signal)' ),
			array( 'fas fa-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'far fa-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'fas fa-sd-card' => 'Sd Card (image,memory,photo,save)' ),
			array( 'fas fa-server' => 'Server (computer,cpu,database,hardware,network)' ),
			array( 'fas fa-sim-card' => 'SIM Card (hard drive,hardware,portable,storage,technology,tiny)' ),
			array( 'fas fa-stream' => 'Stream (flow,list,timeline)' ),
			array( 'fas fa-tablet' => 'tablet (apple,device,ipad,kindle,screen)' ),
			array( 'fas fa-tablet-alt' => 'Alternate Tablet (apple,device,ipad,kindle,screen)' ),
			array( 'fas fa-tv' => 'Television (computer,display,monitor,television)' ),
			array( 'fas fa-upload' => 'Upload (hard drive,import,publish)' ),
		),
		'Construction' => array(
			array( 'fas fa-brush' => 'Brush (art,bristles,color,handle,paint)' ),
			array( 'fas fa-drafting-compass' => 'Drafting Compass (design,map,mechanical drawing,plot,plotting)' ),
			array( 'fas fa-dumpster' => 'Dumpster (alley,bin,commercial,trash,waste)' ),
			array( 'fas fa-hammer' => 'Hammer (admin,fix,repair,settings,tool)' ),
			array( 'fas fa-hard-hat' => 'Hard Hat (construction,hardhat,helmet,safety)' ),
			array( 'fas fa-paint-roller' => 'Paint Roller (acrylic,art,brush,color,fill,paint,pigment,watercolor)' ),
			array( 'fas fa-pencil-alt' => 'Alternate Pencil (design,edit,pencil,update,write)' ),
			array( 'fas fa-pencil-ruler' => 'Pencil Ruler (design,draft,draw,pencil)' ),
			array( 'fas fa-ruler' => 'Ruler (design,draft,length,measure,planning)' ),
			array( 'fas fa-ruler-combined' => 'Ruler Combined (design,draft,length,measure,planning)' ),
			array( 'fas fa-ruler-horizontal' => 'Ruler Horizontal (design,draft,length,measure,planning)' ),
			array( 'fas fa-ruler-vertical' => 'Ruler Vertical (design,draft,length,measure,planning)' ),
			array( 'fas fa-screwdriver' => 'Screwdriver (admin,fix,mechanic,repair,settings,tool)' ),
			array( 'fas fa-toolbox' => 'Toolbox (admin,container,fix,repair,settings,tools)' ),
			array( 'fas fa-tools' => 'Tools (admin,fix,repair,screwdriver,settings,tools,wrench)' ),
			array( 'fas fa-truck-pickup' => 'Truck Side (cargo,vehicle)' ),
			array( 'fas fa-wrench' => 'Wrench (construction,fix,mechanic,plumbing,settings,spanner,tool,update)' ),
		),
		'Currency' => array(
			array( 'fab fa-bitcoin' => 'Bitcoin' ),
			array( 'fab fa-btc' => 'BTC' ),
			array( 'fas fa-dollar-sign' => 'Dollar Sign ($,cost,dollar-sign,money,price,usd)' ),
			array( 'fab fa-ethereum' => 'Ethereum' ),
			array( 'fas fa-euro-sign' => 'Euro Sign (currency,dollar,exchange,money)' ),
			array( 'fab fa-gg' => 'GG Currency' ),
			array( 'fab fa-gg-circle' => 'GG Currency Circle' ),
			array( 'fas fa-hryvnia' => 'Hryvnia (currency,money,ukraine,ukrainian)' ),
			array( 'fas fa-lira-sign' => 'Turkish Lira Sign (currency,money,try,turkish)' ),
			array( 'fas fa-money-bill' => 'Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'fas fa-money-bill-alt' => 'Alternate Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'far fa-money-bill-alt' => 'Alternate Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'fas fa-money-bill-wave' => 'Wavy Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'fas fa-money-bill-wave-alt' => 'Alternate Wavy Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'fas fa-money-check' => 'Money Check (bank check,buy,checkout,cheque,money,payment,price,purchase)' ),
			array( 'fas fa-money-check-alt' => 'Alternate Money Check (bank check,buy,checkout,cheque,money,payment,price,purchase)' ),
			array( 'fas fa-pound-sign' => 'Pound Sign (currency,gbp,money)' ),
			array( 'fas fa-ruble-sign' => 'Ruble Sign (currency,money,rub)' ),
			array( 'fas fa-rupee-sign' => 'Indian Rupee Sign (currency,indian,inr,money)' ),
			array( 'fas fa-shekel-sign' => 'Shekel Sign (currency,ils,money)' ),
			array( 'fas fa-tenge' => 'Tenge (currency,kazakhstan,money,price)' ),
			array( 'fas fa-won-sign' => 'Won Sign (currency,krw,money)' ),
			array( 'fas fa-yen-sign' => 'Yen Sign (currency,jpy,money)' ),
		),
		'Date & Time' => array(
			array( 'fas fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'far fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'fas fa-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'far fa-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'fas fa-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'far fa-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'fas fa-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'far fa-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'fas fa-calendar-check' => 'Calendar Check (accept,agree,appointment,confirm,correct,date,done,event,ok,schedule,select,success,tick,time,todo,when)' ),
			array( 'far fa-calendar-check' => 'Calendar Check (accept,agree,appointment,confirm,correct,date,done,event,ok,schedule,select,success,tick,time,todo,when)' ),
			array( 'fas fa-calendar-minus' => 'Calendar Minus (calendar,date,delete,event,negative,remove,schedule,time,when)' ),
			array( 'far fa-calendar-minus' => 'Calendar Minus (calendar,date,delete,event,negative,remove,schedule,time,when)' ),
			array( 'fas fa-calendar-plus' => 'Calendar Plus (add,calendar,create,date,event,new,positive,schedule,time,when)' ),
			array( 'far fa-calendar-plus' => 'Calendar Plus (add,calendar,create,date,event,new,positive,schedule,time,when)' ),
			array( 'fas fa-calendar-times' => 'Calendar Times (archive,calendar,date,delete,event,remove,schedule,time,when,x)' ),
			array( 'far fa-calendar-times' => 'Calendar Times (archive,calendar,date,delete,event,remove,schedule,time,when,x)' ),
			array( 'fas fa-clock' => 'Clock (date,late,schedule,time,timer,timestamp,watch)' ),
			array( 'far fa-clock' => 'Clock (date,late,schedule,time,timer,timestamp,watch)' ),
			array( 'fas fa-hourglass' => 'Hourglass (hour,minute,sand,stopwatch,time)' ),
			array( 'far fa-hourglass' => 'Hourglass (hour,minute,sand,stopwatch,time)' ),
			array( 'fas fa-hourglass-end' => 'Hourglass End (hour,minute,sand,stopwatch,time)' ),
			array( 'fas fa-hourglass-half' => 'Hourglass Half (hour,minute,sand,stopwatch,time)' ),
			array( 'fas fa-hourglass-start' => 'Hourglass Start (hour,minute,sand,stopwatch,time)' ),
			array( 'fas fa-stopwatch' => 'Stopwatch (clock,reminder,time)' ),
		),
		'Design' => array(
			array( 'fas fa-adjust' => 'adjust (contrast,dark,light,saturation)' ),
			array( 'fas fa-bezier-curve' => 'Bezier Curve (curves,illustrator,lines,path,vector)' ),
			array( 'fas fa-brush' => 'Brush (art,bristles,color,handle,paint)' ),
			array( 'fas fa-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'far fa-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'fas fa-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'far fa-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'fas fa-crop' => 'crop (design,frame,mask,resize,shrink)' ),
			array( 'fas fa-crop-alt' => 'Alternate Crop (design,frame,mask,resize,shrink)' ),
			array( 'fas fa-crosshairs' => 'Crosshairs (aim,bullseye,gpd,picker,position)' ),
			array( 'fas fa-cut' => 'Cut (clip,scissors,snip)' ),
			array( 'fas fa-drafting-compass' => 'Drafting Compass (design,map,mechanical drawing,plot,plotting)' ),
			array( 'fas fa-draw-polygon' => 'Draw Polygon (anchors,lines,object,render,shape)' ),
			array( 'fas fa-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'far fa-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'fas fa-eraser' => 'eraser (art,delete,remove,rubber)' ),
			array( 'fas fa-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'far fa-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'fas fa-eye-dropper' => 'Eye Dropper (beaker,clone,color,copy,eyedropper,pipette)' ),
			array( 'fas fa-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'far fa-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'fas fa-fill' => 'Fill (bucket,color,paint,paint bucket)' ),
			array( 'fas fa-fill-drip' => 'Fill Drip (bucket,color,drop,paint,paint bucket,spill)' ),
			array( 'fas fa-highlighter' => 'Highlighter (edit,marker,sharpie,update,write)' ),
			array( 'fas fa-icons' => 'Icons (bolt,emoji,heart,image,music,photo,symbols)' ),
			array( 'fas fa-layer-group' => 'Layer Group (arrange,develop,layers,map,stack)' ),
			array( 'fas fa-magic' => 'magic (autocomplete,automatic,mage,magic,spell,wand,witch,wizard)' ),
			array( 'fas fa-marker' => 'Marker (design,edit,sharpie,update,write)' ),
			array( 'fas fa-object-group' => 'Object Group (combine,copy,design,merge,select)' ),
			array( 'far fa-object-group' => 'Object Group (combine,copy,design,merge,select)' ),
			array( 'fas fa-object-ungroup' => 'Object Ungroup (copy,design,merge,select,separate)' ),
			array( 'far fa-object-ungroup' => 'Object Ungroup (copy,design,merge,select,separate)' ),
			array( 'fas fa-paint-brush' => 'Paint Brush (acrylic,art,brush,color,fill,paint,pigment,watercolor)' ),
			array( 'fas fa-paint-roller' => 'Paint Roller (acrylic,art,brush,color,fill,paint,pigment,watercolor)' ),
			array( 'fas fa-palette' => 'Palette (acrylic,art,brush,color,fill,paint,pigment,watercolor)' ),
			array( 'fas fa-paste' => 'Paste (clipboard,copy,document,paper)' ),
			array( 'fas fa-pen' => 'Pen (design,edit,update,write)' ),
			array( 'fas fa-pen-alt' => 'Alternate Pen (design,edit,update,write)' ),
			array( 'fas fa-pen-fancy' => 'Pen Fancy (design,edit,fountain pen,update,write)' ),
			array( 'fas fa-pen-nib' => 'Pen Nib (design,edit,fountain pen,update,write)' ),
			array( 'fas fa-pencil-alt' => 'Alternate Pencil (design,edit,pencil,update,write)' ),
			array( 'fas fa-pencil-ruler' => 'Pencil Ruler (design,draft,draw,pencil)' ),
			array( 'fas fa-ruler-combined' => 'Ruler Combined (design,draft,length,measure,planning)' ),
			array( 'fas fa-ruler-horizontal' => 'Ruler Horizontal (design,draft,length,measure,planning)' ),
			array( 'fas fa-ruler-vertical' => 'Ruler Vertical (design,draft,length,measure,planning)' ),
			array( 'fas fa-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'far fa-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'fas fa-splotch' => 'Splotch (Ink,blob,blotch,glob,stain)' ),
			array( 'fas fa-spray-can' => 'Spray Can (Paint,aerosol,design,graffiti,tag)' ),
			array( 'fas fa-stamp' => 'Stamp (art,certificate,imprint,rubber,seal)' ),
			array( 'fas fa-swatchbook' => 'Swatchbook (Pantone,color,design,hue,palette)' ),
			array( 'fas fa-tint' => 'tint (color,drop,droplet,raindrop,waterdrop)' ),
			array( 'fas fa-tint-slash' => 'Tint Slash (color,drop,droplet,raindrop,waterdrop)' ),
			array( 'fas fa-vector-square' => 'Vector Square (anchors,lines,object,render,shape)' ),
		),
		'Editors' => array(
			array( 'fas fa-align-center' => 'align-center (format,middle,paragraph,text)' ),
			array( 'fas fa-align-justify' => 'align-justify (format,paragraph,text)' ),
			array( 'fas fa-align-left' => 'align-left (format,paragraph,text)' ),
			array( 'fas fa-align-right' => 'align-right (format,paragraph,text)' ),
			array( 'fas fa-bold' => 'bold (emphasis,format,text)' ),
			array( 'fas fa-border-all' => 'Border All (cell,grid,outline,stroke,table)' ),
			array( 'fas fa-border-none' => 'Border None (cell,grid,outline,stroke,table)' ),
			array( 'fas fa-border-style' => 'Border Style' ),
			array( 'fas fa-clipboard' => 'Clipboard (copy,notes,paste,record)' ),
			array( 'far fa-clipboard' => 'Clipboard (copy,notes,paste,record)' ),
			array( 'fas fa-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'far fa-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'fas fa-columns' => 'Columns (browser,dashboard,organize,panes,split)' ),
			array( 'fas fa-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'far fa-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'fas fa-cut' => 'Cut (clip,scissors,snip)' ),
			array( 'fas fa-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'far fa-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'fas fa-eraser' => 'eraser (art,delete,remove,rubber)' ),
			array( 'fas fa-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'far fa-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'fas fa-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'far fa-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'fas fa-font' => 'font (alphabet,glyph,text,type,typeface)' ),
			array( 'fas fa-glasses' => 'Glasses (hipster,nerd,reading,sight,spectacles,vision)' ),
			array( 'fas fa-heading' => 'heading (format,header,text,title)' ),
			array( 'fas fa-highlighter' => 'Highlighter (edit,marker,sharpie,update,write)' ),
			array( 'fas fa-i-cursor' => 'I Beam Cursor (editing,i-beam,type,writing)' ),
			array( 'fas fa-icons' => 'Icons (bolt,emoji,heart,image,music,photo,symbols)' ),
			array( 'fas fa-indent' => 'Indent (align,justify,paragraph,tab)' ),
			array( 'fas fa-italic' => 'italic (edit,emphasis,font,format,text,type)' ),
			array( 'fas fa-link' => 'Link (attach,attachment,chain,connect)' ),
			array( 'fas fa-list' => 'List (checklist,completed,done,finished,ol,todo,ul)' ),
			array( 'fas fa-list-alt' => 'Alternate List (checklist,completed,done,finished,ol,todo,ul)' ),
			array( 'far fa-list-alt' => 'Alternate List (checklist,completed,done,finished,ol,todo,ul)' ),
			array( 'fas fa-list-ol' => 'list-ol (checklist,completed,done,finished,numbers,ol,todo,ul)' ),
			array( 'fas fa-list-ul' => 'list-ul (checklist,completed,done,finished,ol,todo,ul)' ),
			array( 'fas fa-marker' => 'Marker (design,edit,sharpie,update,write)' ),
			array( 'fas fa-outdent' => 'Outdent (align,justify,paragraph,tab)' ),
			array( 'fas fa-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'far fa-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'fas fa-paperclip' => 'Paperclip (attach,attachment,connect,link)' ),
			array( 'fas fa-paragraph' => 'paragraph (edit,format,text,writing)' ),
			array( 'fas fa-paste' => 'Paste (clipboard,copy,document,paper)' ),
			array( 'fas fa-pen' => 'Pen (design,edit,update,write)' ),
			array( 'fas fa-pen-alt' => 'Alternate Pen (design,edit,update,write)' ),
			array( 'fas fa-pen-fancy' => 'Pen Fancy (design,edit,fountain pen,update,write)' ),
			array( 'fas fa-pen-nib' => 'Pen Nib (design,edit,fountain pen,update,write)' ),
			array( 'fas fa-pencil-alt' => 'Alternate Pencil (design,edit,pencil,update,write)' ),
			array( 'fas fa-print' => 'print (business,copy,document,office,paper)' ),
			array( 'fas fa-quote-left' => 'quote-left (mention,note,phrase,text,type)' ),
			array( 'fas fa-quote-right' => 'quote-right (mention,note,phrase,text,type)' ),
			array( 'fas fa-redo' => 'Redo (forward,refresh,reload,repeat)' ),
			array( 'fas fa-redo-alt' => 'Alternate Redo (forward,refresh,reload,repeat)' ),
			array( 'fas fa-remove-format' => 'Remove Format (cancel,font,format,remove,style,text)' ),
			array( 'fas fa-reply' => 'Reply (mail,message,respond)' ),
			array( 'fas fa-reply-all' => 'reply-all (mail,message,respond)' ),
			array( 'fas fa-screwdriver' => 'Screwdriver (admin,fix,mechanic,repair,settings,tool)' ),
			array( 'fas fa-share' => 'Share (forward,save,send,social)' ),
			array( 'fas fa-spell-check' => 'Spell Check (dictionary,edit,editor,grammar,text)' ),
			array( 'fas fa-strikethrough' => 'Strikethrough (cancel,edit,font,format,text,type)' ),
			array( 'fas fa-subscript' => 'subscript (edit,font,format,text,type)' ),
			array( 'fas fa-superscript' => 'superscript (edit,exponential,font,format,text,type)' ),
			array( 'fas fa-sync' => 'Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'fas fa-sync-alt' => 'Alternate Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'fas fa-table' => 'table (data,excel,spreadsheet)' ),
			array( 'fas fa-tasks' => 'Tasks (checklist,downloading,downloads,loading,progress,project management,settings,to do)' ),
			array( 'fas fa-text-height' => 'text-height (edit,font,format,text,type)' ),
			array( 'fas fa-text-width' => 'Text Width (edit,font,format,text,type)' ),
			array( 'fas fa-th' => 'th (blocks,boxes,grid,squares)' ),
			array( 'fas fa-th-large' => 'th-large (blocks,boxes,grid,squares)' ),
			array( 'fas fa-th-list' => 'th-list (checklist,completed,done,finished,ol,todo,ul)' ),
			array( 'fas fa-tools' => 'Tools (admin,fix,repair,screwdriver,settings,tools,wrench)' ),
			array( 'fas fa-trash' => 'Trash (delete,garbage,hide,remove)' ),
			array( 'fas fa-trash-alt' => 'Alternate Trash (delete,garbage,hide,remove,trash-o)' ),
			array( 'far fa-trash-alt' => 'Alternate Trash (delete,garbage,hide,remove,trash-o)' ),
			array( 'fas fa-trash-restore' => 'Trash Restore (back,control z,oops,undo)' ),
			array( 'fas fa-trash-restore-alt' => 'Alternative Trash Restore (back,control z,oops,undo)' ),
			array( 'fas fa-underline' => 'Underline (edit,emphasis,format,text,writing)' ),
			array( 'fas fa-undo' => 'Undo (back,control z,exchange,oops,return,rotate,swap)' ),
			array( 'fas fa-undo-alt' => 'Alternate Undo (back,control z,exchange,oops,return,swap)' ),
			array( 'fas fa-unlink' => 'unlink (attachment,chain,chain-broken,remove)' ),
			array( 'fas fa-wrench' => 'Wrench (construction,fix,mechanic,plumbing,settings,spanner,tool,update)' ),
		),
		'Education' => array(
			array( 'fas fa-apple-alt' => 'Fruit Apple (fall,fruit,fuji,macintosh,orchard,seasonal,vegan)' ),
			array( 'fas fa-atom' => 'Atom (atheism,chemistry,ion,nuclear,science)' ),
			array( 'fas fa-award' => 'Award (honor,praise,prize,recognition,ribbon,trophy)' ),
			array( 'fas fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'far fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'fas fa-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'far fa-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'fas fa-book-open' => 'Book Open (flyer,library,notebook,open book,pamphlet,reading)' ),
			array( 'fas fa-book-reader' => 'Book Reader (flyer,library,notebook,open book,pamphlet,reading)' ),
			array( 'fas fa-chalkboard' => 'Chalkboard (blackboard,learning,school,teaching,whiteboard,writing)' ),
			array( 'fas fa-chalkboard-teacher' => 'Chalkboard Teacher (blackboard,instructor,learning,professor,school,whiteboard,writing)' ),
			array( 'fas fa-graduation-cap' => 'Graduation Cap (ceremony,college,graduate,learning,school,student)' ),
			array( 'fas fa-laptop-code' => 'Laptop Code (computer,cpu,dell,demo,develop,device,mac,macbook,machine,pc)' ),
			array( 'fas fa-microscope' => 'Microscope (electron,lens,optics,science,shrink)' ),
			array( 'fas fa-music' => 'Music (lyrics,melody,note,sing,sound)' ),
			array( 'fas fa-school' => 'School (building,education,learn,student,teacher)' ),
			array( 'fas fa-shapes' => 'Shapes (blocks,build,circle,square,triangle)' ),
			array( 'fas fa-theater-masks' => 'Theater Masks (comedy,perform,theatre,tragedy)' ),
			array( 'fas fa-user-graduate' => 'User Graduate (cap,clothing,commencement,gown,graduation,person,student)' ),
		),
		'Emoji' => array(
			array( 'fas fa-angry' => 'Angry Face (disapprove,emoticon,face,mad,upset)' ),
			array( 'far fa-angry' => 'Angry Face (disapprove,emoticon,face,mad,upset)' ),
			array( 'fas fa-dizzy' => 'Dizzy Face (dazed,dead,disapprove,emoticon,face)' ),
			array( 'far fa-dizzy' => 'Dizzy Face (dazed,dead,disapprove,emoticon,face)' ),
			array( 'fas fa-flushed' => 'Flushed Face (embarrassed,emoticon,face)' ),
			array( 'far fa-flushed' => 'Flushed Face (embarrassed,emoticon,face)' ),
			array( 'fas fa-frown' => 'Frowning Face (disapprove,emoticon,face,rating,sad)' ),
			array( 'far fa-frown' => 'Frowning Face (disapprove,emoticon,face,rating,sad)' ),
			array( 'fas fa-frown-open' => 'Frowning Face With Open Mouth (disapprove,emoticon,face,rating,sad)' ),
			array( 'far fa-frown-open' => 'Frowning Face With Open Mouth (disapprove,emoticon,face,rating,sad)' ),
			array( 'fas fa-grimace' => 'Grimacing Face (cringe,emoticon,face,teeth)' ),
			array( 'far fa-grimace' => 'Grimacing Face (cringe,emoticon,face,teeth)' ),
			array( 'fas fa-grin' => 'Grinning Face (emoticon,face,laugh,smile)' ),
			array( 'far fa-grin' => 'Grinning Face (emoticon,face,laugh,smile)' ),
			array( 'fas fa-grin-alt' => 'Alternate Grinning Face (emoticon,face,laugh,smile)' ),
			array( 'far fa-grin-alt' => 'Alternate Grinning Face (emoticon,face,laugh,smile)' ),
			array( 'fas fa-grin-beam' => 'Grinning Face With Smiling Eyes (emoticon,face,laugh,smile)' ),
			array( 'far fa-grin-beam' => 'Grinning Face With Smiling Eyes (emoticon,face,laugh,smile)' ),
			array( 'fas fa-grin-beam-sweat' => 'Grinning Face With Sweat (embarass,emoticon,face,smile)' ),
			array( 'far fa-grin-beam-sweat' => 'Grinning Face With Sweat (embarass,emoticon,face,smile)' ),
			array( 'fas fa-grin-hearts' => 'Smiling Face With Heart-Eyes (emoticon,face,love,smile)' ),
			array( 'far fa-grin-hearts' => 'Smiling Face With Heart-Eyes (emoticon,face,love,smile)' ),
			array( 'fas fa-grin-squint' => 'Grinning Squinting Face (emoticon,face,laugh,smile)' ),
			array( 'far fa-grin-squint' => 'Grinning Squinting Face (emoticon,face,laugh,smile)' ),
			array( 'fas fa-grin-squint-tears' => 'Rolling on the Floor Laughing (emoticon,face,happy,smile)' ),
			array( 'far fa-grin-squint-tears' => 'Rolling on the Floor Laughing (emoticon,face,happy,smile)' ),
			array( 'fas fa-grin-stars' => 'Star-Struck (emoticon,face,star-struck)' ),
			array( 'far fa-grin-stars' => 'Star-Struck (emoticon,face,star-struck)' ),
			array( 'fas fa-grin-tears' => 'Face With Tears of Joy (LOL,emoticon,face)' ),
			array( 'far fa-grin-tears' => 'Face With Tears of Joy (LOL,emoticon,face)' ),
			array( 'fas fa-grin-tongue' => 'Face With Tongue (LOL,emoticon,face)' ),
			array( 'far fa-grin-tongue' => 'Face With Tongue (LOL,emoticon,face)' ),
			array( 'fas fa-grin-tongue-squint' => 'Squinting Face With Tongue (LOL,emoticon,face)' ),
			array( 'far fa-grin-tongue-squint' => 'Squinting Face With Tongue (LOL,emoticon,face)' ),
			array( 'fas fa-grin-tongue-wink' => 'Winking Face With Tongue (LOL,emoticon,face)' ),
			array( 'far fa-grin-tongue-wink' => 'Winking Face With Tongue (LOL,emoticon,face)' ),
			array( 'fas fa-grin-wink' => 'Grinning Winking Face (emoticon,face,flirt,laugh,smile)' ),
			array( 'far fa-grin-wink' => 'Grinning Winking Face (emoticon,face,flirt,laugh,smile)' ),
			array( 'fas fa-kiss' => 'Kissing Face (beso,emoticon,face,love,smooch)' ),
			array( 'far fa-kiss' => 'Kissing Face (beso,emoticon,face,love,smooch)' ),
			array( 'fas fa-kiss-beam' => 'Kissing Face With Smiling Eyes (beso,emoticon,face,love,smooch)' ),
			array( 'far fa-kiss-beam' => 'Kissing Face With Smiling Eyes (beso,emoticon,face,love,smooch)' ),
			array( 'fas fa-kiss-wink-heart' => 'Face Blowing a Kiss (beso,emoticon,face,love,smooch)' ),
			array( 'far fa-kiss-wink-heart' => 'Face Blowing a Kiss (beso,emoticon,face,love,smooch)' ),
			array( 'fas fa-laugh' => 'Grinning Face With Big Eyes (LOL,emoticon,face,laugh,smile)' ),
			array( 'far fa-laugh' => 'Grinning Face With Big Eyes (LOL,emoticon,face,laugh,smile)' ),
			array( 'fas fa-laugh-beam' => 'Laugh Face with Beaming Eyes (LOL,emoticon,face,happy,smile)' ),
			array( 'far fa-laugh-beam' => 'Laugh Face with Beaming Eyes (LOL,emoticon,face,happy,smile)' ),
			array( 'fas fa-laugh-squint' => 'Laughing Squinting Face (LOL,emoticon,face,happy,smile)' ),
			array( 'far fa-laugh-squint' => 'Laughing Squinting Face (LOL,emoticon,face,happy,smile)' ),
			array( 'fas fa-laugh-wink' => 'Laughing Winking Face (LOL,emoticon,face,happy,smile)' ),
			array( 'far fa-laugh-wink' => 'Laughing Winking Face (LOL,emoticon,face,happy,smile)' ),
			array( 'fas fa-meh' => 'Neutral Face (emoticon,face,neutral,rating)' ),
			array( 'far fa-meh' => 'Neutral Face (emoticon,face,neutral,rating)' ),
			array( 'fas fa-meh-blank' => 'Face Without Mouth (emoticon,face,neutral,rating)' ),
			array( 'far fa-meh-blank' => 'Face Without Mouth (emoticon,face,neutral,rating)' ),
			array( 'fas fa-meh-rolling-eyes' => 'Face With Rolling Eyes (emoticon,face,neutral,rating)' ),
			array( 'far fa-meh-rolling-eyes' => 'Face With Rolling Eyes (emoticon,face,neutral,rating)' ),
			array( 'fas fa-sad-cry' => 'Crying Face (emoticon,face,tear,tears)' ),
			array( 'far fa-sad-cry' => 'Crying Face (emoticon,face,tear,tears)' ),
			array( 'fas fa-sad-tear' => 'Loudly Crying Face (emoticon,face,tear,tears)' ),
			array( 'far fa-sad-tear' => 'Loudly Crying Face (emoticon,face,tear,tears)' ),
			array( 'fas fa-smile' => 'Smiling Face (approve,emoticon,face,happy,rating,satisfied)' ),
			array( 'far fa-smile' => 'Smiling Face (approve,emoticon,face,happy,rating,satisfied)' ),
			array( 'fas fa-smile-beam' => 'Beaming Face With Smiling Eyes (emoticon,face,happy,positive)' ),
			array( 'far fa-smile-beam' => 'Beaming Face With Smiling Eyes (emoticon,face,happy,positive)' ),
			array( 'fas fa-smile-wink' => 'Winking Face (emoticon,face,happy,hint,joke)' ),
			array( 'far fa-smile-wink' => 'Winking Face (emoticon,face,happy,hint,joke)' ),
			array( 'fas fa-surprise' => 'Hushed Face (emoticon,face,shocked)' ),
			array( 'far fa-surprise' => 'Hushed Face (emoticon,face,shocked)' ),
			array( 'fas fa-tired' => 'Tired Face (angry,emoticon,face,grumpy,upset)' ),
			array( 'far fa-tired' => 'Tired Face (angry,emoticon,face,grumpy,upset)' ),
		),
		'Energy' => array(
			array( 'fas fa-atom' => 'Atom (atheism,chemistry,ion,nuclear,science)' ),
			array( 'fas fa-battery-empty' => 'Battery Empty (charge,dead,power,status)' ),
			array( 'fas fa-battery-full' => 'Battery Full (charge,power,status)' ),
			array( 'fas fa-battery-half' => 'Battery 1/2 Full (charge,power,status)' ),
			array( 'fas fa-battery-quarter' => 'Battery 1/4 Full (charge,low,power,status)' ),
			array( 'fas fa-battery-three-quarters' => 'Battery 3/4 Full (charge,power,status)' ),
			array( 'fas fa-broadcast-tower' => 'Broadcast Tower (airwaves,antenna,radio,reception,waves)' ),
			array( 'fas fa-burn' => 'Burn (caliente,energy,fire,flame,gas,heat,hot)' ),
			array( 'fas fa-charging-station' => 'Charging Station (electric,ev,tesla,vehicle)' ),
			array( 'fas fa-fire' => 'fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'fas fa-fire-alt' => 'Alternate Fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'fas fa-gas-pump' => 'Gas Pump (car,fuel,gasoline,petrol)' ),
			array( 'fas fa-industry' => 'Industry (building,factory,industrial,manufacturing,mill,warehouse)' ),
			array( 'fas fa-leaf' => 'leaf (eco,flora,nature,plant,vegan)' ),
			array( 'fas fa-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'far fa-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'fas fa-plug' => 'Plug (connect,electric,online,power)' ),
			array( 'fas fa-poop' => 'Poop (crap,poop,shit,smile,turd)' ),
			array( 'fas fa-power-off' => 'Power Off (cancel,computer,on,reboot,restart)' ),
			array( 'fas fa-radiation' => 'Radiation (danger,dangerous,deadly,hazard,nuclear,radioactive,warning)' ),
			array( 'fas fa-radiation-alt' => 'Alternate Radiation (danger,dangerous,deadly,hazard,nuclear,radioactive,warning)' ),
			array( 'fas fa-seedling' => 'Seedling (flora,grow,plant,vegan)' ),
			array( 'fas fa-solar-panel' => 'Solar Panel (clean,eco-friendly,energy,green,sun)' ),
			array( 'fas fa-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'far fa-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'fas fa-water' => 'Water (lake,liquid,ocean,sea,swim,wet)' ),
			array( 'fas fa-wind' => 'Wind (air,blow,breeze,fall,seasonal,weather)' ),
		),
		'Files' => array(
			array( 'fas fa-archive' => 'Archive (box,package,save,storage)' ),
			array( 'fas fa-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'far fa-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'fas fa-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'far fa-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'fas fa-cut' => 'Cut (clip,scissors,snip)' ),
			array( 'fas fa-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'far fa-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'fas fa-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'far fa-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'fas fa-file-archive' => 'Archive File (.zip,bundle,compress,compression,download,zip)' ),
			array( 'far fa-file-archive' => 'Archive File (.zip,bundle,compress,compression,download,zip)' ),
			array( 'fas fa-file-audio' => 'Audio File (document,mp3,music,page,play,sound)' ),
			array( 'far fa-file-audio' => 'Audio File (document,mp3,music,page,play,sound)' ),
			array( 'fas fa-file-code' => 'Code File (css,development,document,html)' ),
			array( 'far fa-file-code' => 'Code File (css,development,document,html)' ),
			array( 'fas fa-file-excel' => 'Excel File (csv,document,numbers,spreadsheets,table)' ),
			array( 'far fa-file-excel' => 'Excel File (csv,document,numbers,spreadsheets,table)' ),
			array( 'fas fa-file-image' => 'Image File (document,image,jpg,photo,png)' ),
			array( 'far fa-file-image' => 'Image File (document,image,jpg,photo,png)' ),
			array( 'fas fa-file-pdf' => 'PDF File (acrobat,document,preview,save)' ),
			array( 'far fa-file-pdf' => 'PDF File (acrobat,document,preview,save)' ),
			array( 'fas fa-file-powerpoint' => 'Powerpoint File (display,document,keynote,presentation)' ),
			array( 'far fa-file-powerpoint' => 'Powerpoint File (display,document,keynote,presentation)' ),
			array( 'fas fa-file-video' => 'Video File (document,m4v,movie,mp4,play)' ),
			array( 'far fa-file-video' => 'Video File (document,m4v,movie,mp4,play)' ),
			array( 'fas fa-file-word' => 'Word File (document,edit,page,text,writing)' ),
			array( 'far fa-file-word' => 'Word File (document,edit,page,text,writing)' ),
			array( 'fas fa-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'far fa-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'fas fa-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'far fa-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'fas fa-paste' => 'Paste (clipboard,copy,document,paper)' ),
			array( 'fas fa-photo-video' => 'Photo Video (av,film,image,library,media)' ),
			array( 'fas fa-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'far fa-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'fas fa-sticky-note' => 'Sticky Note (message,note,paper,reminder,sticker)' ),
			array( 'far fa-sticky-note' => 'Sticky Note (message,note,paper,reminder,sticker)' ),
		),
		'Finance' => array(
			array( 'fas fa-balance-scale' => 'Balance Scale (balanced,justice,legal,measure,weight)' ),
			array( 'fas fa-balance-scale-left' => 'Balance Scale (Left-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'fas fa-balance-scale-right' => 'Balance Scale (Right-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'fas fa-book' => 'book (diary,documentation,journal,library,read)' ),
			array( 'fas fa-cash-register' => 'Cash Register (buy,cha-ching,change,checkout,commerce,leaerboard,machine,pay,payment,purchase,store)' ),
			array( 'fas fa-chart-line' => 'Line Chart (activity,analytics,chart,dashboard,gain,graph,increase,line)' ),
			array( 'fas fa-chart-pie' => 'Pie Chart (analytics,chart,diagram,graph,pie)' ),
			array( 'fas fa-coins' => 'Coins (currency,dime,financial,gold,money,penny)' ),
			array( 'fas fa-comment-dollar' => 'Comment Dollar (bubble,chat,commenting,conversation,feedback,message,money,note,notification,pay,sms,speech,spend,texting,transfer)' ),
			array( 'fas fa-comments-dollar' => 'Comments Dollar (bubble,chat,commenting,conversation,feedback,message,money,note,notification,pay,sms,speech,spend,texting,transfer)' ),
			array( 'fas fa-credit-card' => 'Credit Card (buy,checkout,credit-card-alt,debit,money,payment,purchase)' ),
			array( 'far fa-credit-card' => 'Credit Card (buy,checkout,credit-card-alt,debit,money,payment,purchase)' ),
			array( 'fas fa-donate' => 'Donate (contribute,generosity,gift,give)' ),
			array( 'fas fa-file-invoice' => 'File Invoice (account,bill,charge,document,payment,receipt)' ),
			array( 'fas fa-file-invoice-dollar' => 'File Invoice with US Dollar ($,account,bill,charge,document,dollar-sign,money,payment,receipt,usd)' ),
			array( 'fas fa-hand-holding-usd' => 'Hand Holding US Dollar ($,carry,dollar sign,donation,giving,lift,money,price)' ),
			array( 'fas fa-landmark' => 'Landmark (building,historic,memorable,monument,politics)' ),
			array( 'fas fa-money-bill' => 'Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'fas fa-money-bill-alt' => 'Alternate Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'far fa-money-bill-alt' => 'Alternate Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'fas fa-money-bill-wave' => 'Wavy Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'fas fa-money-bill-wave-alt' => 'Alternate Wavy Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'fas fa-money-check' => 'Money Check (bank check,buy,checkout,cheque,money,payment,price,purchase)' ),
			array( 'fas fa-money-check-alt' => 'Alternate Money Check (bank check,buy,checkout,cheque,money,payment,price,purchase)' ),
			array( 'fas fa-percentage' => 'Percentage (discount,fraction,proportion,rate,ratio)' ),
			array( 'fas fa-piggy-bank' => 'Piggy Bank (bank,save,savings)' ),
			array( 'fas fa-receipt' => 'Receipt (check,invoice,money,pay,table)' ),
			array( 'fas fa-stamp' => 'Stamp (art,certificate,imprint,rubber,seal)' ),
			array( 'fas fa-wallet' => 'Wallet (billfold,cash,currency,money)' ),
		),
		'Fitness' => array(
			array( 'fas fa-bicycle' => 'Bicycle (bike,gears,pedal,transportation,vehicle)' ),
			array( 'fas fa-biking' => 'Biking (bicycle,bike,cycle,cycling,ride,wheel)' ),
			array( 'fas fa-burn' => 'Burn (caliente,energy,fire,flame,gas,heat,hot)' ),
			array( 'fas fa-fire-alt' => 'Alternate Fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'fas fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'far fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'fas fa-heartbeat' => 'Heartbeat (ekg,electrocardiogram,health,lifeline,vital signs)' ),
			array( 'fas fa-hiking' => 'Hiking (activity,backpack,fall,fitness,outdoors,person,seasonal,walking)' ),
			array( 'fas fa-running' => 'Running (exercise,health,jog,person,run,sport,sprint)' ),
			array( 'fas fa-shoe-prints' => 'Shoe Prints (feet,footprints,steps,walk)' ),
			array( 'fas fa-skating' => 'Skating (activity,figure skating,fitness,ice,person,winter)' ),
			array( 'fas fa-skiing' => 'Skiing (activity,downhill,fast,fitness,olympics,outdoors,person,seasonal,slalom)' ),
			array( 'fas fa-skiing-nordic' => 'Skiing Nordic (activity,cross country,fitness,outdoors,person,seasonal)' ),
			array( 'fas fa-snowboarding' => 'Snowboarding (activity,fitness,olympics,outdoors,person)' ),
			array( 'fas fa-spa' => 'Spa (flora,massage,mindfulness,plant,wellness)' ),
			array( 'fas fa-swimmer' => 'Swimmer (athlete,head,man,olympics,person,pool,water)' ),
			array( 'fas fa-walking' => 'Walking (exercise,health,pedometer,person,steps)' ),
		),
		'Food' => array(
			array( 'fas fa-apple-alt' => 'Fruit Apple (fall,fruit,fuji,macintosh,orchard,seasonal,vegan)' ),
			array( 'fas fa-bacon' => 'Bacon (blt,breakfast,ham,lard,meat,pancetta,pork,rasher)' ),
			array( 'fas fa-bone' => 'Bone (calcium,dog,skeletal,skeleton,tibia)' ),
			array( 'fas fa-bread-slice' => 'Bread Slice (bake,bakery,baking,dough,flour,gluten,grain,sandwich,sourdough,toast,wheat,yeast)' ),
			array( 'fas fa-candy-cane' => 'Candy Cane (candy,christmas,holiday,mint,peppermint,striped,xmas)' ),
			array( 'fas fa-carrot' => 'Carrot (bugs bunny,orange,vegan,vegetable)' ),
			array( 'fas fa-cheese' => 'Cheese (cheddar,curd,gouda,melt,parmesan,sandwich,swiss,wedge)' ),
			array( 'fas fa-cloud-meatball' => 'Cloud with (a chance of) Meatball (FLDSMDFR,food,spaghetti,storm)' ),
			array( 'fas fa-cookie' => 'Cookie (baked good,chips,chocolate,eat,snack,sweet,treat)' ),
			array( 'fas fa-drumstick-bite' => 'Drumstick with Bite Taken Out (bone,chicken,leg,meat,poultry,turkey)' ),
			array( 'fas fa-egg' => 'Egg (breakfast,chicken,easter,shell,yolk)' ),
			array( 'fas fa-fish' => 'Fish (fauna,gold,seafood,swimming)' ),
			array( 'fas fa-hamburger' => 'Hamburger (bacon,beef,burger,burger king,cheeseburger,fast food,grill,ground beef,mcdonalds,sandwich)' ),
			array( 'fas fa-hotdog' => 'Hot Dog (bun,chili,frankfurt,frankfurter,kosher,polish,sandwich,sausage,vienna,weiner)' ),
			array( 'fas fa-ice-cream' => 'Ice Cream (chocolate,cone,dessert,frozen,scoop,sorbet,vanilla,yogurt)' ),
			array( 'fas fa-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'far fa-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'fas fa-pepper-hot' => 'Hot Pepper (buffalo wings,capsicum,chili,chilli,habanero,jalapeno,mexican,spicy,tabasco,vegetable)' ),
			array( 'fas fa-pizza-slice' => 'Pizza Slice (cheese,chicago,italian,mozzarella,new york,pepperoni,pie,slice,teenage mutant ninja turtles,tomato)' ),
			array( 'fas fa-seedling' => 'Seedling (flora,grow,plant,vegan)' ),
			array( 'fas fa-stroopwafel' => 'Stroopwafel (caramel,cookie,dessert,sweets,waffle)' ),
		),
		'Fruits & Vegetables' => array(
			array( 'fas fa-apple-alt' => 'Fruit Apple (fall,fruit,fuji,macintosh,orchard,seasonal,vegan)' ),
			array( 'fas fa-carrot' => 'Carrot (bugs bunny,orange,vegan,vegetable)' ),
			array( 'fas fa-leaf' => 'leaf (eco,flora,nature,plant,vegan)' ),
			array( 'fas fa-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'far fa-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'fas fa-pepper-hot' => 'Hot Pepper (buffalo wings,capsicum,chili,chilli,habanero,jalapeno,mexican,spicy,tabasco,vegetable)' ),
			array( 'fas fa-seedling' => 'Seedling (flora,grow,plant,vegan)' ),
		),
		'Games' => array(
			array( 'fas fa-chess' => 'Chess (board,castle,checkmate,game,king,rook,strategy,tournament)' ),
			array( 'fas fa-chess-bishop' => 'Chess Bishop (board,checkmate,game,strategy)' ),
			array( 'fas fa-chess-board' => 'Chess Board (board,checkmate,game,strategy)' ),
			array( 'fas fa-chess-king' => 'Chess King (board,checkmate,game,strategy)' ),
			array( 'fas fa-chess-knight' => 'Chess Knight (board,checkmate,game,horse,strategy)' ),
			array( 'fas fa-chess-pawn' => 'Chess Pawn (board,checkmate,game,strategy)' ),
			array( 'fas fa-chess-queen' => 'Chess Queen (board,checkmate,game,strategy)' ),
			array( 'fas fa-chess-rook' => 'Chess Rook (board,castle,checkmate,game,strategy)' ),
			array( 'fas fa-dice' => 'Dice (chance,gambling,game,roll)' ),
			array( 'fas fa-dice-d20' => 'Dice D20 (Dungeons & Dragons,chance,d&d,dnd,fantasy,gambling,game,roll)' ),
			array( 'fas fa-dice-d6' => 'Dice D6 (Dungeons & Dragons,chance,d&d,dnd,fantasy,gambling,game,roll)' ),
			array( 'fas fa-dice-five' => 'Dice Five (chance,gambling,game,roll)' ),
			array( 'fas fa-dice-four' => 'Dice Four (chance,gambling,game,roll)' ),
			array( 'fas fa-dice-one' => 'Dice One (chance,gambling,game,roll)' ),
			array( 'fas fa-dice-six' => 'Dice Six (chance,gambling,game,roll)' ),
			array( 'fas fa-dice-three' => 'Dice Three (chance,gambling,game,roll)' ),
			array( 'fas fa-dice-two' => 'Dice Two (chance,gambling,game,roll)' ),
			array( 'fas fa-gamepad' => 'Gamepad (arcade,controller,d-pad,joystick,video,video game)' ),
			array( 'fas fa-ghost' => 'Ghost (apparition,blinky,clyde,floating,halloween,holiday,inky,pinky,spirit)' ),
			array( 'fas fa-headset' => 'Headset (audio,gamer,gaming,listen,live chat,microphone,shot caller,sound,support,telemarketer)' ),
			array( 'fas fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'far fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'fab fa-playstation' => 'PlayStation' ),
			array( 'fas fa-puzzle-piece' => 'Puzzle Piece (add-on,addon,game,section)' ),
			array( 'fab fa-steam' => 'Steam' ),
			array( 'fab fa-steam-square' => 'Steam Square' ),
			array( 'fab fa-steam-symbol' => 'Steam Symbol' ),
			array( 'fab fa-twitch' => 'Twitch' ),
			array( 'fab fa-xbox' => 'Xbox' ),
		),
		'Genders' => array(
			array( 'fas fa-genderless' => 'Genderless (androgynous,asexual,sexless)' ),
			array( 'fas fa-mars' => 'Mars (male)' ),
			array( 'fas fa-mars-double' => 'Mars Double' ),
			array( 'fas fa-mars-stroke' => 'Mars Stroke' ),
			array( 'fas fa-mars-stroke-h' => 'Mars Stroke Horizontal' ),
			array( 'fas fa-mars-stroke-v' => 'Mars Stroke Vertical' ),
			array( 'fas fa-mercury' => 'Mercury (transgender)' ),
			array( 'fas fa-neuter' => 'Neuter' ),
			array( 'fas fa-transgender' => 'Transgender (intersex)' ),
			array( 'fas fa-transgender-alt' => 'Alternate Transgender (intersex)' ),
			array( 'fas fa-venus' => 'Venus (female)' ),
			array( 'fas fa-venus-double' => 'Venus Double (female)' ),
			array( 'fas fa-venus-mars' => 'Venus Mars (Gender)' ),
		),
		'Halloween' => array(
			array( 'fas fa-book-dead' => 'Book of the Dead (Dungeons & Dragons,crossbones,d&d,dark arts,death,dnd,documentation,evil,fantasy,halloween,holiday,necronomicon,read,skull,spell)' ),
			array( 'fas fa-broom' => 'Broom (clean,firebolt,fly,halloween,nimbus 2000,quidditch,sweep,witch)' ),
			array( 'fas fa-cat' => 'Cat (feline,halloween,holiday,kitten,kitty,meow,pet)' ),
			array( 'fas fa-cloud-moon' => 'Cloud with Moon (crescent,evening,lunar,night,partly cloudy,sky)' ),
			array( 'fas fa-crow' => 'Crow (bird,bullfrog,fauna,halloween,holiday,toad)' ),
			array( 'fas fa-ghost' => 'Ghost (apparition,blinky,clyde,floating,halloween,holiday,inky,pinky,spirit)' ),
			array( 'fas fa-hat-wizard' => 'Wizard\'s Hat (Dungeons & Dragons,accessory,buckle,clothing,d&d,dnd,fantasy,halloween,head,holiday,mage,magic,pointy,witch)' ),
			array( 'fas fa-mask' => 'Mask (carnivale,costume,disguise,halloween,secret,super hero)' ),
			array( 'fas fa-skull-crossbones' => 'Skull & Crossbones (Dungeons & Dragons,alert,bones,d&d,danger,dead,deadly,death,dnd,fantasy,halloween,holiday,jolly-roger,pirate,poison,skeleton,warning)' ),
			array( 'fas fa-spider' => 'Spider (arachnid,bug,charlotte,crawl,eight,halloween)' ),
			array( 'fas fa-toilet-paper' => 'Toilet Paper (bathroom,halloween,holiday,lavatory,prank,restroom,roll)' ),
		),
		'Hands' => array(
			array( 'fas fa-allergies' => 'Allergies (allergy,freckles,hand,hives,pox,skin,spots)' ),
			array( 'fas fa-fist-raised' => 'Raised Fist (Dungeons & Dragons,d&d,dnd,fantasy,hand,ki,monk,resist,strength,unarmed combat)' ),
			array( 'fas fa-hand-holding' => 'Hand Holding (carry,lift)' ),
			array( 'fas fa-hand-holding-heart' => 'Hand Holding Heart (carry,charity,gift,lift,package)' ),
			array( 'fas fa-hand-holding-usd' => 'Hand Holding US Dollar ($,carry,dollar sign,donation,giving,lift,money,price)' ),
			array( 'fas fa-hand-lizard' => 'Lizard (Hand) (game,roshambo)' ),
			array( 'far fa-hand-lizard' => 'Lizard (Hand) (game,roshambo)' ),
			array( 'fas fa-hand-middle-finger' => 'Hand with Middle Finger Raised (flip the bird,gesture,hate,rude)' ),
			array( 'fas fa-hand-paper' => 'Paper (Hand) (game,halt,roshambo,stop)' ),
			array( 'far fa-hand-paper' => 'Paper (Hand) (game,halt,roshambo,stop)' ),
			array( 'fas fa-hand-peace' => 'Peace (Hand) (rest,truce)' ),
			array( 'far fa-hand-peace' => 'Peace (Hand) (rest,truce)' ),
			array( 'fas fa-hand-point-down' => 'Hand Pointing Down (finger,hand-o-down,point)' ),
			array( 'far fa-hand-point-down' => 'Hand Pointing Down (finger,hand-o-down,point)' ),
			array( 'fas fa-hand-point-left' => 'Hand Pointing Left (back,finger,hand-o-left,left,point,previous)' ),
			array( 'far fa-hand-point-left' => 'Hand Pointing Left (back,finger,hand-o-left,left,point,previous)' ),
			array( 'fas fa-hand-point-right' => 'Hand Pointing Right (finger,forward,hand-o-right,next,point,right)' ),
			array( 'far fa-hand-point-right' => 'Hand Pointing Right (finger,forward,hand-o-right,next,point,right)' ),
			array( 'fas fa-hand-point-up' => 'Hand Pointing Up (finger,hand-o-up,point)' ),
			array( 'far fa-hand-point-up' => 'Hand Pointing Up (finger,hand-o-up,point)' ),
			array( 'fas fa-hand-pointer' => 'Pointer (Hand) (arrow,cursor,select)' ),
			array( 'far fa-hand-pointer' => 'Pointer (Hand) (arrow,cursor,select)' ),
			array( 'fas fa-hand-rock' => 'Rock (Hand) (fist,game,roshambo)' ),
			array( 'far fa-hand-rock' => 'Rock (Hand) (fist,game,roshambo)' ),
			array( 'fas fa-hand-scissors' => 'Scissors (Hand) (cut,game,roshambo)' ),
			array( 'far fa-hand-scissors' => 'Scissors (Hand) (cut,game,roshambo)' ),
			array( 'fas fa-hand-spock' => 'Spock (Hand) (live long,prosper,salute,star trek,vulcan)' ),
			array( 'far fa-hand-spock' => 'Spock (Hand) (live long,prosper,salute,star trek,vulcan)' ),
			array( 'fas fa-hands' => 'Hands (carry,hold,lift)' ),
			array( 'fas fa-hands-helping' => 'Helping Hands (aid,assistance,handshake,partnership,volunteering)' ),
			array( 'fas fa-handshake' => 'Handshake (agreement,greeting,meeting,partnership)' ),
			array( 'far fa-handshake' => 'Handshake (agreement,greeting,meeting,partnership)' ),
			array( 'fas fa-praying-hands' => 'Praying Hands (kneel,preach,religion,worship)' ),
			array( 'fas fa-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'far fa-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'fas fa-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
			array( 'far fa-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
		),
		'Health' => array(
			array( 'fab fa-accessible-icon' => 'Accessible Icon (accessibility,handicap,person,wheelchair,wheelchair-alt)' ),
			array( 'fas fa-ambulance' => 'ambulance (emergency,emt,er,help,hospital,support,vehicle)' ),
			array( 'fas fa-h-square' => 'H Square (directions,emergency,hospital,hotel,map)' ),
			array( 'fas fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'far fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'fas fa-heartbeat' => 'Heartbeat (ekg,electrocardiogram,health,lifeline,vital signs)' ),
			array( 'fas fa-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'far fa-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'fas fa-medkit' => 'medkit (first aid,firstaid,health,help,support)' ),
			array( 'fas fa-plus-square' => 'Plus Square (add,create,expand,new,positive,shape)' ),
			array( 'far fa-plus-square' => 'Plus Square (add,create,expand,new,positive,shape)' ),
			array( 'fas fa-prescription' => 'Prescription (drugs,medical,medicine,pharmacy,rx)' ),
			array( 'fas fa-stethoscope' => 'Stethoscope (diagnosis,doctor,general practitioner,hospital,infirmary,medicine,office,outpatient)' ),
			array( 'fas fa-user-md' => 'Doctor (job,medical,nurse,occupation,physician,profile,surgeon)' ),
			array( 'fas fa-wheelchair' => 'Wheelchair (accessible,handicap,person)' ),
		),
		'Holiday' => array(
			array( 'fas fa-candy-cane' => 'Candy Cane (candy,christmas,holiday,mint,peppermint,striped,xmas)' ),
			array( 'fas fa-carrot' => 'Carrot (bugs bunny,orange,vegan,vegetable)' ),
			array( 'fas fa-cookie-bite' => 'Cookie Bite (baked good,bitten,chips,chocolate,eat,snack,sweet,treat)' ),
			array( 'fas fa-gift' => 'gift (christmas,generosity,giving,holiday,party,present,wrapped,xmas)' ),
			array( 'fas fa-gifts' => 'Gifts (christmas,generosity,giving,holiday,party,present,wrapped,xmas)' ),
			array( 'fas fa-glass-cheers' => 'Glass Cheers (alcohol,bar,beverage,celebration,champagne,clink,drink,holiday,new year\'s eve,party,toast)' ),
			array( 'fas fa-holly-berry' => 'Holly Berry (catwoman,christmas,decoration,flora,halle,holiday,ororo munroe,plant,storm,xmas)' ),
			array( 'fas fa-mug-hot' => 'Mug Hot (caliente,cocoa,coffee,cup,drink,holiday,hot chocolate,steam,tea,warmth)' ),
			array( 'fas fa-sleigh' => 'Sleigh (christmas,claus,fly,holiday,santa,sled,snow,xmas)' ),
			array( 'fas fa-snowman' => 'Snowman (decoration,frost,frosty,holiday)' ),
		),
		'Hotel' => array(
			array( 'fas fa-baby-carriage' => 'Baby Carriage (buggy,carrier,infant,push,stroller,transportation,walk,wheels)' ),
			array( 'fas fa-bath' => 'Bath (clean,shower,tub,wash)' ),
			array( 'fas fa-bed' => 'Bed (lodging,rest,sleep,travel)' ),
			array( 'fas fa-briefcase' => 'Briefcase (bag,business,luggage,office,work)' ),
			array( 'fas fa-car' => 'Car (auto,automobile,sedan,transportation,travel,vehicle)' ),
			array( 'fas fa-cocktail' => 'Cocktail (alcohol,beverage,drink,gin,glass,margarita,martini,vodka)' ),
			array( 'fas fa-coffee' => 'Coffee (beverage,breakfast,cafe,drink,fall,morning,mug,seasonal,tea)' ),
			array( 'fas fa-concierge-bell' => 'Concierge Bell (attention,hotel,receptionist,service,support)' ),
			array( 'fas fa-dice' => 'Dice (chance,gambling,game,roll)' ),
			array( 'fas fa-dice-five' => 'Dice Five (chance,gambling,game,roll)' ),
			array( 'fas fa-door-closed' => 'Door Closed (enter,exit,locked)' ),
			array( 'fas fa-door-open' => 'Door Open (enter,exit,welcome)' ),
			array( 'fas fa-dumbbell' => 'Dumbbell (exercise,gym,strength,weight,weight-lifting)' ),
			array( 'fas fa-glass-martini' => 'Martini Glass (alcohol,bar,beverage,drink,liquor)' ),
			array( 'fas fa-glass-martini-alt' => 'Alternate Glass Martini (alcohol,bar,beverage,drink,liquor)' ),
			array( 'fas fa-hot-tub' => 'Hot Tub (bath,jacuzzi,massage,sauna,spa)' ),
			array( 'fas fa-hotel' => 'Hotel (building,inn,lodging,motel,resort,travel)' ),
			array( 'fas fa-infinity' => 'Infinity (eternity,forever,math)' ),
			array( 'fas fa-key' => 'key (lock,password,private,secret,unlock)' ),
			array( 'fas fa-luggage-cart' => 'Luggage Cart (bag,baggage,suitcase,travel)' ),
			array( 'fas fa-shower' => 'Shower (bath,clean,faucet,water)' ),
			array( 'fas fa-shuttle-van' => 'Shuttle Van (airport,machine,public-transportation,transportation,travel,vehicle)' ),
			array( 'fas fa-smoking' => 'Smoking (cancer,cigarette,nicotine,smoking status,tobacco)' ),
			array( 'fas fa-smoking-ban' => 'Smoking Ban (ban,cancel,no smoking,non-smoking)' ),
			array( 'fas fa-snowflake' => 'Snowflake (precipitation,rain,winter)' ),
			array( 'far fa-snowflake' => 'Snowflake (precipitation,rain,winter)' ),
			array( 'fas fa-spa' => 'Spa (flora,massage,mindfulness,plant,wellness)' ),
			array( 'fas fa-suitcase' => 'Suitcase (baggage,luggage,move,suitcase,travel,trip)' ),
			array( 'fas fa-suitcase-rolling' => 'Suitcase Rolling (baggage,luggage,move,suitcase,travel,trip)' ),
			array( 'fas fa-swimmer' => 'Swimmer (athlete,head,man,olympics,person,pool,water)' ),
			array( 'fas fa-swimming-pool' => 'Swimming Pool (ladder,recreation,swim,water)' ),
			array( 'fas fa-tv' => 'Television (computer,display,monitor,television)' ),
			array( 'fas fa-umbrella-beach' => 'Umbrella Beach (protection,recreation,sand,shade,summer,sun)' ),
			array( 'fas fa-utensils' => 'Utensils (cutlery,dining,dinner,eat,food,fork,knife,restaurant)' ),
			array( 'fas fa-wheelchair' => 'Wheelchair (accessible,handicap,person)' ),
			array( 'fas fa-wifi' => 'WiFi (connection,hotspot,internet,network,wireless)' ),
		),
		'Household' => array(
			array( 'fas fa-bath' => 'Bath (clean,shower,tub,wash)' ),
			array( 'fas fa-bed' => 'Bed (lodging,rest,sleep,travel)' ),
			array( 'fas fa-blender' => 'Blender (cocktail,milkshake,mixer,puree,smoothie)' ),
			array( 'fas fa-chair' => 'Chair (furniture,seat,sit)' ),
			array( 'fas fa-couch' => 'Couch (chair,cushion,furniture,relax,sofa)' ),
			array( 'fas fa-door-closed' => 'Door Closed (enter,exit,locked)' ),
			array( 'fas fa-door-open' => 'Door Open (enter,exit,welcome)' ),
			array( 'fas fa-dungeon' => 'Dungeon (Dungeons & Dragons,building,d&d,dnd,door,entrance,fantasy,gate)' ),
			array( 'fas fa-fan' => 'Fan (ac,air conditioning,blade,blower,cool,hot)' ),
			array( 'fas fa-shower' => 'Shower (bath,clean,faucet,water)' ),
			array( 'fas fa-toilet-paper' => 'Toilet Paper (bathroom,halloween,holiday,lavatory,prank,restroom,roll)' ),
			array( 'fas fa-tv' => 'Television (computer,display,monitor,television)' ),
		),
		'Images' => array(
			array( 'fas fa-adjust' => 'adjust (contrast,dark,light,saturation)' ),
			array( 'fas fa-bolt' => 'Lightning Bolt (electricity,lightning,weather,zap)' ),
			array( 'fas fa-camera' => 'camera (image,lens,photo,picture,record,shutter,video)' ),
			array( 'fas fa-camera-retro' => 'Retro Camera (image,lens,photo,picture,record,shutter,video)' ),
			array( 'fas fa-chalkboard' => 'Chalkboard (blackboard,learning,school,teaching,whiteboard,writing)' ),
			array( 'fas fa-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'far fa-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'fas fa-compress' => 'Compress (collapse,fullscreen,minimize,move,resize,shrink,smaller)' ),
			array( 'fas fa-compress-arrows-alt' => 'Alternate Compress Arrows (collapse,fullscreen,minimize,move,resize,shrink,smaller)' ),
			array( 'fas fa-expand' => 'Expand (arrow,bigger,enlarge,resize)' ),
			array( 'fas fa-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'far fa-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'fas fa-eye-dropper' => 'Eye Dropper (beaker,clone,color,copy,eyedropper,pipette)' ),
			array( 'fas fa-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'far fa-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'fas fa-file-image' => 'Image File (document,image,jpg,photo,png)' ),
			array( 'far fa-file-image' => 'Image File (document,image,jpg,photo,png)' ),
			array( 'fas fa-film' => 'Film (cinema,movie,strip,video)' ),
			array( 'fas fa-id-badge' => 'Identification Badge (address,contact,identification,license,profile)' ),
			array( 'far fa-id-badge' => 'Identification Badge (address,contact,identification,license,profile)' ),
			array( 'fas fa-id-card' => 'Identification Card (contact,demographics,document,identification,issued,profile)' ),
			array( 'far fa-id-card' => 'Identification Card (contact,demographics,document,identification,issued,profile)' ),
			array( 'fas fa-image' => 'Image (album,landscape,photo,picture)' ),
			array( 'far fa-image' => 'Image (album,landscape,photo,picture)' ),
			array( 'fas fa-images' => 'Images (album,landscape,photo,picture)' ),
			array( 'far fa-images' => 'Images (album,landscape,photo,picture)' ),
			array( 'fas fa-photo-video' => 'Photo Video (av,film,image,library,media)' ),
			array( 'fas fa-portrait' => 'Portrait (id,image,photo,picture,selfie)' ),
			array( 'fas fa-sliders-h' => 'Horizontal Sliders (adjust,settings,sliders,toggle)' ),
			array( 'fas fa-tint' => 'tint (color,drop,droplet,raindrop,waterdrop)' ),
		),
		'Interfaces' => array(
			array( 'fas fa-award' => 'Award (honor,praise,prize,recognition,ribbon,trophy)' ),
			array( 'fas fa-ban' => 'ban (abort,ban,block,cancel,delete,hide,prohibit,remove,stop,trash)' ),
			array( 'fas fa-barcode' => 'barcode (info,laser,price,scan,upc)' ),
			array( 'fas fa-bars' => 'Bars (checklist,drag,hamburger,list,menu,nav,navigation,ol,reorder,settings,todo,ul)' ),
			array( 'fas fa-beer' => 'beer (alcohol,ale,bar,beverage,brewery,drink,lager,liquor,mug,stein)' ),
			array( 'fas fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'far fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'fas fa-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'far fa-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'fas fa-blog' => 'Blog (journal,log,online,personal,post,web 2.0,wordpress,writing)' ),
			array( 'fas fa-bug' => 'Bug (beetle,error,insect,report)' ),
			array( 'fas fa-bullhorn' => 'bullhorn (announcement,broadcast,louder,megaphone,share)' ),
			array( 'fas fa-bullseye' => 'Bullseye (archery,goal,objective,target)' ),
			array( 'fas fa-calculator' => 'Calculator (abacus,addition,arithmetic,counting,math,multiplication,subtraction)' ),
			array( 'fas fa-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'far fa-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'fas fa-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'far fa-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'fas fa-calendar-check' => 'Calendar Check (accept,agree,appointment,confirm,correct,date,done,event,ok,schedule,select,success,tick,time,todo,when)' ),
			array( 'far fa-calendar-check' => 'Calendar Check (accept,agree,appointment,confirm,correct,date,done,event,ok,schedule,select,success,tick,time,todo,when)' ),
			array( 'fas fa-calendar-minus' => 'Calendar Minus (calendar,date,delete,event,negative,remove,schedule,time,when)' ),
			array( 'far fa-calendar-minus' => 'Calendar Minus (calendar,date,delete,event,negative,remove,schedule,time,when)' ),
			array( 'fas fa-calendar-plus' => 'Calendar Plus (add,calendar,create,date,event,new,positive,schedule,time,when)' ),
			array( 'far fa-calendar-plus' => 'Calendar Plus (add,calendar,create,date,event,new,positive,schedule,time,when)' ),
			array( 'fas fa-calendar-times' => 'Calendar Times (archive,calendar,date,delete,event,remove,schedule,time,when,x)' ),
			array( 'far fa-calendar-times' => 'Calendar Times (archive,calendar,date,delete,event,remove,schedule,time,when,x)' ),
			array( 'fas fa-certificate' => 'certificate (badge,star,verified)' ),
			array( 'fas fa-check' => 'Check (accept,agree,checkmark,confirm,correct,done,notice,notification,notify,ok,select,success,tick,todo,yes)' ),
			array( 'fas fa-check-circle' => 'Check Circle (accept,agree,confirm,correct,done,ok,select,success,tick,todo,yes)' ),
			array( 'far fa-check-circle' => 'Check Circle (accept,agree,confirm,correct,done,ok,select,success,tick,todo,yes)' ),
			array( 'fas fa-check-double' => 'Double Check (accept,agree,checkmark,confirm,correct,done,notice,notification,notify,ok,select,success,tick,todo)' ),
			array( 'fas fa-check-square' => 'Check Square (accept,agree,checkmark,confirm,correct,done,ok,select,success,tick,todo,yes)' ),
			array( 'far fa-check-square' => 'Check Square (accept,agree,checkmark,confirm,correct,done,ok,select,success,tick,todo,yes)' ),
			array( 'fas fa-circle' => 'Circle (circle-thin,diameter,dot,ellipse,notification,round)' ),
			array( 'far fa-circle' => 'Circle (circle-thin,diameter,dot,ellipse,notification,round)' ),
			array( 'fas fa-clipboard' => 'Clipboard (copy,notes,paste,record)' ),
			array( 'far fa-clipboard' => 'Clipboard (copy,notes,paste,record)' ),
			array( 'fas fa-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'far fa-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'fas fa-cloud' => 'Cloud (atmosphere,fog,overcast,save,upload,weather)' ),
			array( 'fas fa-cloud-download-alt' => 'Alternate Cloud Download (download,export,save)' ),
			array( 'fas fa-cloud-upload-alt' => 'Alternate Cloud Upload (cloud-upload,import,save,upload)' ),
			array( 'fas fa-coffee' => 'Coffee (beverage,breakfast,cafe,drink,fall,morning,mug,seasonal,tea)' ),
			array( 'fas fa-cog' => 'cog (gear,mechanical,settings,sprocket,wheel)' ),
			array( 'fas fa-cogs' => 'cogs (gears,mechanical,settings,sprocket,wheel)' ),
			array( 'fas fa-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'far fa-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'fas fa-cut' => 'Cut (clip,scissors,snip)' ),
			array( 'fas fa-database' => 'Database (computer,development,directory,memory,storage)' ),
			array( 'fas fa-dot-circle' => 'Dot Circle (bullseye,notification,target)' ),
			array( 'far fa-dot-circle' => 'Dot Circle (bullseye,notification,target)' ),
			array( 'fas fa-download' => 'Download (export,hard drive,save,transfer)' ),
			array( 'fas fa-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'far fa-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'fas fa-ellipsis-h' => 'Horizontal Ellipsis (dots,drag,kebab,list,menu,nav,navigation,ol,reorder,settings,ul)' ),
			array( 'fas fa-ellipsis-v' => 'Vertical Ellipsis (dots,drag,kebab,list,menu,nav,navigation,ol,reorder,settings,ul)' ),
			array( 'fas fa-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'far fa-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'fas fa-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'far fa-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'fas fa-eraser' => 'eraser (art,delete,remove,rubber)' ),
			array( 'fas fa-exclamation' => 'exclamation (alert,danger,error,important,notice,notification,notify,problem,warning)' ),
			array( 'fas fa-exclamation-circle' => 'Exclamation Circle (alert,danger,error,important,notice,notification,notify,problem,warning)' ),
			array( 'fas fa-exclamation-triangle' => 'Exclamation Triangle (alert,danger,error,important,notice,notification,notify,problem,warning)' ),
			array( 'fas fa-external-link-alt' => 'Alternate External Link (external-link,new,open,share)' ),
			array( 'fas fa-external-link-square-alt' => 'Alternate External Link Square (external-link-square,new,open,share)' ),
			array( 'fas fa-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'far fa-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'fas fa-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'far fa-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'fas fa-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'far fa-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'fas fa-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'far fa-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'fas fa-file-download' => 'File Download (document,export,save)' ),
			array( 'fas fa-file-export' => 'File Export (download,save)' ),
			array( 'fas fa-file-import' => 'File Import (copy,document,send,upload)' ),
			array( 'fas fa-file-upload' => 'File Upload (document,import,page,save)' ),
			array( 'fas fa-filter' => 'Filter (funnel,options,separate,sort)' ),
			array( 'fas fa-fingerprint' => 'Fingerprint (human,id,identification,lock,smudge,touch,unique,unlock)' ),
			array( 'fas fa-flag' => 'flag (country,notice,notification,notify,pole,report,symbol)' ),
			array( 'far fa-flag' => 'flag (country,notice,notification,notify,pole,report,symbol)' ),
			array( 'fas fa-flag-checkered' => 'flag-checkered (notice,notification,notify,pole,racing,report,symbol)' ),
			array( 'fas fa-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'far fa-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'fas fa-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'far fa-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'fas fa-frown' => 'Frowning Face (disapprove,emoticon,face,rating,sad)' ),
			array( 'far fa-frown' => 'Frowning Face (disapprove,emoticon,face,rating,sad)' ),
			array( 'fas fa-glasses' => 'Glasses (hipster,nerd,reading,sight,spectacles,vision)' ),
			array( 'fas fa-grip-horizontal' => 'Grip Horizontal (affordance,drag,drop,grab,handle)' ),
			array( 'fas fa-grip-lines' => 'Grip Lines (affordance,drag,drop,grab,handle)' ),
			array( 'fas fa-grip-lines-vertical' => 'Grip Lines Vertical (affordance,drag,drop,grab,handle)' ),
			array( 'fas fa-grip-vertical' => 'Grip Vertical (affordance,drag,drop,grab,handle)' ),
			array( 'fas fa-hashtag' => 'Hashtag (Twitter,instagram,pound,social media,tag)' ),
			array( 'fas fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'far fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'fas fa-history' => 'History (Rewind,clock,reverse,time,time machine)' ),
			array( 'fas fa-home' => 'home (abode,building,house,main)' ),
			array( 'fas fa-i-cursor' => 'I Beam Cursor (editing,i-beam,type,writing)' ),
			array( 'fas fa-info' => 'Info (details,help,information,more,support)' ),
			array( 'fas fa-info-circle' => 'Info Circle (details,help,information,more,support)' ),
			array( 'fas fa-language' => 'Language (dialect,idiom,localize,speech,translate,vernacular)' ),
			array( 'fas fa-magic' => 'magic (autocomplete,automatic,mage,magic,spell,wand,witch,wizard)' ),
			array( 'fas fa-marker' => 'Marker (design,edit,sharpie,update,write)' ),
			array( 'fas fa-medal' => 'Medal (award,ribbon,star,trophy)' ),
			array( 'fas fa-meh' => 'Neutral Face (emoticon,face,neutral,rating)' ),
			array( 'far fa-meh' => 'Neutral Face (emoticon,face,neutral,rating)' ),
			array( 'fas fa-microphone' => 'microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-microphone-alt' => 'Alternate Microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-microphone-slash' => 'Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-minus' => 'minus (collapse,delete,hide,minify,negative,remove,trash)' ),
			array( 'fas fa-minus-circle' => 'Minus Circle (delete,hide,negative,remove,shape,trash)' ),
			array( 'fas fa-minus-square' => 'Minus Square (collapse,delete,hide,minify,negative,remove,shape,trash)' ),
			array( 'far fa-minus-square' => 'Minus Square (collapse,delete,hide,minify,negative,remove,shape,trash)' ),
			array( 'fas fa-paste' => 'Paste (clipboard,copy,document,paper)' ),
			array( 'fas fa-pen' => 'Pen (design,edit,update,write)' ),
			array( 'fas fa-pen-alt' => 'Alternate Pen (design,edit,update,write)' ),
			array( 'fas fa-pen-fancy' => 'Pen Fancy (design,edit,fountain pen,update,write)' ),
			array( 'fas fa-pencil-alt' => 'Alternate Pencil (design,edit,pencil,update,write)' ),
			array( 'fas fa-plus' => 'plus (add,create,expand,new,positive,shape)' ),
			array( 'fas fa-plus-circle' => 'Plus Circle (add,create,expand,new,positive,shape)' ),
			array( 'fas fa-plus-square' => 'Plus Square (add,create,expand,new,positive,shape)' ),
			array( 'far fa-plus-square' => 'Plus Square (add,create,expand,new,positive,shape)' ),
			array( 'fas fa-poo' => 'Poo (crap,poop,shit,smile,turd)' ),
			array( 'fas fa-qrcode' => 'qrcode (barcode,info,information,scan)' ),
			array( 'fas fa-question' => 'Question (help,information,support,unknown)' ),
			array( 'fas fa-question-circle' => 'Question Circle (help,information,support,unknown)' ),
			array( 'far fa-question-circle' => 'Question Circle (help,information,support,unknown)' ),
			array( 'fas fa-quote-left' => 'quote-left (mention,note,phrase,text,type)' ),
			array( 'fas fa-quote-right' => 'quote-right (mention,note,phrase,text,type)' ),
			array( 'fas fa-redo' => 'Redo (forward,refresh,reload,repeat)' ),
			array( 'fas fa-redo-alt' => 'Alternate Redo (forward,refresh,reload,repeat)' ),
			array( 'fas fa-reply' => 'Reply (mail,message,respond)' ),
			array( 'fas fa-reply-all' => 'reply-all (mail,message,respond)' ),
			array( 'fas fa-rss' => 'rss (blog,feed,journal,news,writing)' ),
			array( 'fas fa-rss-square' => 'RSS Square (blog,feed,journal,news,writing)' ),
			array( 'fas fa-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'far fa-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'fas fa-screwdriver' => 'Screwdriver (admin,fix,mechanic,repair,settings,tool)' ),
			array( 'fas fa-search' => 'Search (bigger,enlarge,find,magnify,preview,zoom)' ),
			array( 'fas fa-search-minus' => 'Search Minus (minify,negative,smaller,zoom,zoom out)' ),
			array( 'fas fa-search-plus' => 'Search Plus (bigger,enlarge,magnify,positive,zoom,zoom in)' ),
			array( 'fas fa-share' => 'Share (forward,save,send,social)' ),
			array( 'fas fa-share-alt' => 'Alternate Share (forward,save,send,social)' ),
			array( 'fas fa-share-alt-square' => 'Alternate Share Square (forward,save,send,social)' ),
			array( 'fas fa-share-square' => 'Share Square (forward,save,send,social)' ),
			array( 'far fa-share-square' => 'Share Square (forward,save,send,social)' ),
			array( 'fas fa-shield-alt' => 'Alternate Shield (achievement,award,block,defend,security,winner)' ),
			array( 'fas fa-sign-in-alt' => 'Alternate Sign In (arrow,enter,join,log in,login,sign in,sign up,sign-in,signin,signup)' ),
			array( 'fas fa-sign-out-alt' => 'Alternate Sign Out (arrow,exit,leave,log out,logout,sign-out)' ),
			array( 'fas fa-signal' => 'signal (bars,graph,online,reception,status)' ),
			array( 'fas fa-sitemap' => 'Sitemap (directory,hierarchy,ia,information architecture,organization)' ),
			array( 'fas fa-sliders-h' => 'Horizontal Sliders (adjust,settings,sliders,toggle)' ),
			array( 'fas fa-smile' => 'Smiling Face (approve,emoticon,face,happy,rating,satisfied)' ),
			array( 'far fa-smile' => 'Smiling Face (approve,emoticon,face,happy,rating,satisfied)' ),
			array( 'fas fa-sort' => 'Sort (filter,order)' ),
			array( 'fas fa-sort-alpha-down' => 'Sort Alphabetical Down (alphabetical,arrange,filter,order,sort-alpha-asc)' ),
			array( 'fas fa-sort-alpha-down-alt' => 'Alternate Sort Alphabetical Down (alphabetical,arrange,filter,order,sort-alpha-asc)' ),
			array( 'fas fa-sort-alpha-up' => 'Sort Alphabetical Up (alphabetical,arrange,filter,order,sort-alpha-desc)' ),
			array( 'fas fa-sort-alpha-up-alt' => 'Alternate Sort Alphabetical Up (alphabetical,arrange,filter,order,sort-alpha-desc)' ),
			array( 'fas fa-sort-amount-down' => 'Sort Amount Down (arrange,filter,number,order,sort-amount-asc)' ),
			array( 'fas fa-sort-amount-down-alt' => 'Alternate Sort Amount Down (arrange,filter,order,sort-amount-asc)' ),
			array( 'fas fa-sort-amount-up' => 'Sort Amount Up (arrange,filter,order,sort-amount-desc)' ),
			array( 'fas fa-sort-amount-up-alt' => 'Alternate Sort Amount Up (arrange,filter,order,sort-amount-desc)' ),
			array( 'fas fa-sort-down' => 'Sort Down (Descending) (arrow,descending,filter,order,sort-desc)' ),
			array( 'fas fa-sort-numeric-down' => 'Sort Numeric Down (arrange,filter,numbers,order,sort-numeric-asc)' ),
			array( 'fas fa-sort-numeric-down-alt' => 'Alternate Sort Numeric Down (arrange,filter,numbers,order,sort-numeric-asc)' ),
			array( 'fas fa-sort-numeric-up' => 'Sort Numeric Up (arrange,filter,numbers,order,sort-numeric-desc)' ),
			array( 'fas fa-sort-numeric-up-alt' => 'Alternate Sort Numeric Up (arrange,filter,numbers,order,sort-numeric-desc)' ),
			array( 'fas fa-sort-up' => 'Sort Up (Ascending) (arrow,ascending,filter,order,sort-asc)' ),
			array( 'fas fa-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'far fa-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'fas fa-star-half' => 'star-half (achievement,award,rating,score,star-half-empty,star-half-full)' ),
			array( 'far fa-star-half' => 'star-half (achievement,award,rating,score,star-half-empty,star-half-full)' ),
			array( 'fas fa-sync' => 'Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'fas fa-sync-alt' => 'Alternate Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'fas fa-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'far fa-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'fas fa-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
			array( 'far fa-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
			array( 'fas fa-times' => 'Times (close,cross,error,exit,incorrect,notice,notification,notify,problem,wrong,x)' ),
			array( 'fas fa-times-circle' => 'Times Circle (close,cross,exit,incorrect,notice,notification,notify,problem,wrong,x)' ),
			array( 'far fa-times-circle' => 'Times Circle (close,cross,exit,incorrect,notice,notification,notify,problem,wrong,x)' ),
			array( 'fas fa-toggle-off' => 'Toggle Off (switch)' ),
			array( 'fas fa-toggle-on' => 'Toggle On (switch)' ),
			array( 'fas fa-tools' => 'Tools (admin,fix,repair,screwdriver,settings,tools,wrench)' ),
			array( 'fas fa-trash' => 'Trash (delete,garbage,hide,remove)' ),
			array( 'fas fa-trash-alt' => 'Alternate Trash (delete,garbage,hide,remove,trash-o)' ),
			array( 'far fa-trash-alt' => 'Alternate Trash (delete,garbage,hide,remove,trash-o)' ),
			array( 'fas fa-trash-restore' => 'Trash Restore (back,control z,oops,undo)' ),
			array( 'fas fa-trash-restore-alt' => 'Alternative Trash Restore (back,control z,oops,undo)' ),
			array( 'fas fa-trophy' => 'trophy (achievement,award,cup,game,winner)' ),
			array( 'fas fa-undo' => 'Undo (back,control z,exchange,oops,return,rotate,swap)' ),
			array( 'fas fa-undo-alt' => 'Alternate Undo (back,control z,exchange,oops,return,swap)' ),
			array( 'fas fa-upload' => 'Upload (hard drive,import,publish)' ),
			array( 'fas fa-user' => 'User (account,avatar,head,human,man,person,profile)' ),
			array( 'far fa-user' => 'User (account,avatar,head,human,man,person,profile)' ),
			array( 'fas fa-user-alt' => 'Alternate User (account,avatar,head,human,man,person,profile)' ),
			array( 'fas fa-user-circle' => 'User Circle (account,avatar,head,human,man,person,profile)' ),
			array( 'far fa-user-circle' => 'User Circle (account,avatar,head,human,man,person,profile)' ),
			array( 'fas fa-volume-down' => 'Volume Down (audio,lower,music,quieter,sound,speaker)' ),
			array( 'fas fa-volume-mute' => 'Volume Mute (audio,music,quiet,sound,speaker)' ),
			array( 'fas fa-volume-off' => 'Volume Off (audio,ban,music,mute,quiet,silent,sound)' ),
			array( 'fas fa-volume-up' => 'Volume Up (audio,higher,louder,music,sound,speaker)' ),
			array( 'fas fa-wifi' => 'WiFi (connection,hotspot,internet,network,wireless)' ),
			array( 'fas fa-wrench' => 'Wrench (construction,fix,mechanic,plumbing,settings,spanner,tool,update)' ),
		),
		'Logistics' => array(
			array( 'fas fa-box' => 'Box (archive,container,package,storage)' ),
			array( 'fas fa-boxes' => 'Boxes (archives,inventory,storage,warehouse)' ),
			array( 'fas fa-clipboard-check' => 'Clipboard with Check (accept,agree,confirm,done,ok,select,success,tick,todo,yes)' ),
			array( 'fas fa-clipboard-list' => 'Clipboard List (checklist,completed,done,finished,intinerary,ol,schedule,tick,todo,ul)' ),
			array( 'fas fa-dolly' => 'Dolly (carry,shipping,transport)' ),
			array( 'fas fa-dolly-flatbed' => 'Dolly Flatbed (carry,inventory,shipping,transport)' ),
			array( 'fas fa-hard-hat' => 'Hard Hat (construction,hardhat,helmet,safety)' ),
			array( 'fas fa-pallet' => 'Pallet (archive,box,inventory,shipping,warehouse)' ),
			array( 'fas fa-shipping-fast' => 'Shipping Fast (express,fedex,mail,overnight,package,ups)' ),
			array( 'fas fa-truck' => 'truck (cargo,delivery,shipping,vehicle)' ),
			array( 'fas fa-warehouse' => 'Warehouse (building,capacity,garage,inventory,storage)' ),
		),
		'Maps' => array(
			array( 'fas fa-ambulance' => 'ambulance (emergency,emt,er,help,hospital,support,vehicle)' ),
			array( 'fas fa-anchor' => 'Anchor (berth,boat,dock,embed,link,maritime,moor,secure)' ),
			array( 'fas fa-balance-scale' => 'Balance Scale (balanced,justice,legal,measure,weight)' ),
			array( 'fas fa-balance-scale-left' => 'Balance Scale (Left-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'fas fa-balance-scale-right' => 'Balance Scale (Right-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'fas fa-bath' => 'Bath (clean,shower,tub,wash)' ),
			array( 'fas fa-bed' => 'Bed (lodging,rest,sleep,travel)' ),
			array( 'fas fa-beer' => 'beer (alcohol,ale,bar,beverage,brewery,drink,lager,liquor,mug,stein)' ),
			array( 'fas fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'far fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'fas fa-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'far fa-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'fas fa-bicycle' => 'Bicycle (bike,gears,pedal,transportation,vehicle)' ),
			array( 'fas fa-binoculars' => 'Binoculars (glasses,magnify,scenic,spyglass,view)' ),
			array( 'fas fa-birthday-cake' => 'Birthday Cake (anniversary,bakery,candles,celebration,dessert,frosting,holiday,party,pastry)' ),
			array( 'fas fa-blind' => 'Blind (cane,disability,person,sight)' ),
			array( 'fas fa-bomb' => 'Bomb (error,explode,fuse,grenade,warning)' ),
			array( 'fas fa-book' => 'book (diary,documentation,journal,library,read)' ),
			array( 'fas fa-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'far fa-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'fas fa-briefcase' => 'Briefcase (bag,business,luggage,office,work)' ),
			array( 'fas fa-building' => 'Building (apartment,business,city,company,office,work)' ),
			array( 'far fa-building' => 'Building (apartment,business,city,company,office,work)' ),
			array( 'fas fa-car' => 'Car (auto,automobile,sedan,transportation,travel,vehicle)' ),
			array( 'fas fa-coffee' => 'Coffee (beverage,breakfast,cafe,drink,fall,morning,mug,seasonal,tea)' ),
			array( 'fas fa-crosshairs' => 'Crosshairs (aim,bullseye,gpd,picker,position)' ),
			array( 'fas fa-directions' => 'Directions (map,navigation,sign,turn)' ),
			array( 'fas fa-dollar-sign' => 'Dollar Sign ($,cost,dollar-sign,money,price,usd)' ),
			array( 'fas fa-draw-polygon' => 'Draw Polygon (anchors,lines,object,render,shape)' ),
			array( 'fas fa-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'far fa-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'fas fa-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'far fa-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'fas fa-fighter-jet' => 'fighter-jet (airplane,fast,fly,goose,maverick,plane,quick,top gun,transportation,travel)' ),
			array( 'fas fa-fire' => 'fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'fas fa-fire-alt' => 'Alternate Fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'fas fa-fire-extinguisher' => 'fire-extinguisher (burn,caliente,fire fighter,flame,heat,hot,rescue)' ),
			array( 'fas fa-flag' => 'flag (country,notice,notification,notify,pole,report,symbol)' ),
			array( 'far fa-flag' => 'flag (country,notice,notification,notify,pole,report,symbol)' ),
			array( 'fas fa-flag-checkered' => 'flag-checkered (notice,notification,notify,pole,racing,report,symbol)' ),
			array( 'fas fa-flask' => 'Flask (beaker,experimental,labs,science)' ),
			array( 'fas fa-gamepad' => 'Gamepad (arcade,controller,d-pad,joystick,video,video game)' ),
			array( 'fas fa-gavel' => 'Gavel (hammer,judge,law,lawyer,opinion)' ),
			array( 'fas fa-gift' => 'gift (christmas,generosity,giving,holiday,party,present,wrapped,xmas)' ),
			array( 'fas fa-glass-martini' => 'Martini Glass (alcohol,bar,beverage,drink,liquor)' ),
			array( 'fas fa-globe' => 'Globe (all,coordinates,country,earth,global,gps,language,localize,location,map,online,place,planet,translate,travel,world)' ),
			array( 'fas fa-graduation-cap' => 'Graduation Cap (ceremony,college,graduate,learning,school,student)' ),
			array( 'fas fa-h-square' => 'H Square (directions,emergency,hospital,hotel,map)' ),
			array( 'fas fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'far fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'fas fa-heartbeat' => 'Heartbeat (ekg,electrocardiogram,health,lifeline,vital signs)' ),
			array( 'fas fa-helicopter' => 'Helicopter (airwolf,apache,chopper,flight,fly,travel)' ),
			array( 'fas fa-home' => 'home (abode,building,house,main)' ),
			array( 'fas fa-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'far fa-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'fas fa-image' => 'Image (album,landscape,photo,picture)' ),
			array( 'far fa-image' => 'Image (album,landscape,photo,picture)' ),
			array( 'fas fa-images' => 'Images (album,landscape,photo,picture)' ),
			array( 'far fa-images' => 'Images (album,landscape,photo,picture)' ),
			array( 'fas fa-industry' => 'Industry (building,factory,industrial,manufacturing,mill,warehouse)' ),
			array( 'fas fa-info' => 'Info (details,help,information,more,support)' ),
			array( 'fas fa-info-circle' => 'Info Circle (details,help,information,more,support)' ),
			array( 'fas fa-key' => 'key (lock,password,private,secret,unlock)' ),
			array( 'fas fa-landmark' => 'Landmark (building,historic,memorable,monument,politics)' ),
			array( 'fas fa-layer-group' => 'Layer Group (arrange,develop,layers,map,stack)' ),
			array( 'fas fa-leaf' => 'leaf (eco,flora,nature,plant,vegan)' ),
			array( 'fas fa-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'far fa-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'fas fa-life-ring' => 'Life Ring (coast guard,help,overboard,save,support)' ),
			array( 'far fa-life-ring' => 'Life Ring (coast guard,help,overboard,save,support)' ),
			array( 'fas fa-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'far fa-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'fas fa-location-arrow' => 'location-arrow (address,compass,coordinate,direction,gps,map,navigation,place)' ),
			array( 'fas fa-low-vision' => 'Low Vision (blind,eye,sight)' ),
			array( 'fas fa-magnet' => 'magnet (Attract,lodestone,tool)' ),
			array( 'fas fa-male' => 'Male (human,man,person,profile,user)' ),
			array( 'fas fa-map' => 'Map (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'far fa-map' => 'Map (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'fas fa-map-marker' => 'map-marker (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'fas fa-map-marker-alt' => 'Alternate Map Marker (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'fas fa-map-pin' => 'Map Pin (address,agree,coordinates,destination,gps,localize,location,map,marker,navigation,pin,place,position,travel)' ),
			array( 'fas fa-map-signs' => 'Map Signs (directions,directory,map,signage,wayfinding)' ),
			array( 'fas fa-medkit' => 'medkit (first aid,firstaid,health,help,support)' ),
			array( 'fas fa-money-bill' => 'Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'fas fa-money-bill-alt' => 'Alternate Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'far fa-money-bill-alt' => 'Alternate Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'fas fa-motorcycle' => 'Motorcycle (bike,machine,transportation,vehicle)' ),
			array( 'fas fa-music' => 'Music (lyrics,melody,note,sing,sound)' ),
			array( 'fas fa-newspaper' => 'Newspaper (article,editorial,headline,journal,journalism,news,press)' ),
			array( 'far fa-newspaper' => 'Newspaper (article,editorial,headline,journal,journalism,news,press)' ),
			array( 'fas fa-parking' => 'Parking (auto,car,garage,meter)' ),
			array( 'fas fa-paw' => 'Paw (animal,cat,dog,pet,print)' ),
			array( 'fas fa-phone' => 'Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'fas fa-phone-alt' => 'Alternate Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'fas fa-phone-square' => 'Phone Square (call,earphone,number,support,telephone,voice)' ),
			array( 'fas fa-phone-square-alt' => 'Alternate Phone Square (call,earphone,number,support,telephone,voice)' ),
			array( 'fas fa-phone-volume' => 'Phone Volume (call,earphone,number,sound,support,telephone,voice,volume-control-phone)' ),
			array( 'fas fa-plane' => 'plane (airplane,destination,fly,location,mode,travel,trip)' ),
			array( 'fas fa-plug' => 'Plug (connect,electric,online,power)' ),
			array( 'fas fa-plus' => 'plus (add,create,expand,new,positive,shape)' ),
			array( 'fas fa-plus-square' => 'Plus Square (add,create,expand,new,positive,shape)' ),
			array( 'far fa-plus-square' => 'Plus Square (add,create,expand,new,positive,shape)' ),
			array( 'fas fa-print' => 'print (business,copy,document,office,paper)' ),
			array( 'fas fa-recycle' => 'Recycle (Waste,compost,garbage,reuse,trash)' ),
			array( 'fas fa-restroom' => 'Restroom (bathroom,john,loo,potty,washroom,waste,wc)' ),
			array( 'fas fa-road' => 'road (highway,map,pavement,route,street,travel)' ),
			array( 'fas fa-rocket' => 'rocket (aircraft,app,jet,launch,nasa,space)' ),
			array( 'fas fa-route' => 'Route (directions,navigation,travel)' ),
			array( 'fas fa-search' => 'Search (bigger,enlarge,find,magnify,preview,zoom)' ),
			array( 'fas fa-search-minus' => 'Search Minus (minify,negative,smaller,zoom,zoom out)' ),
			array( 'fas fa-search-plus' => 'Search Plus (bigger,enlarge,magnify,positive,zoom,zoom in)' ),
			array( 'fas fa-ship' => 'Ship (boat,sea,water)' ),
			array( 'fas fa-shoe-prints' => 'Shoe Prints (feet,footprints,steps,walk)' ),
			array( 'fas fa-shopping-bag' => 'Shopping Bag (buy,checkout,grocery,payment,purchase)' ),
			array( 'fas fa-shopping-basket' => 'Shopping Basket (buy,checkout,grocery,payment,purchase)' ),
			array( 'fas fa-shopping-cart' => 'shopping-cart (buy,checkout,grocery,payment,purchase)' ),
			array( 'fas fa-shower' => 'Shower (bath,clean,faucet,water)' ),
			array( 'fas fa-snowplow' => 'Snowplow (clean up,cold,road,storm,winter)' ),
			array( 'fas fa-street-view' => 'Street View (directions,location,map,navigation)' ),
			array( 'fas fa-subway' => 'Subway (machine,railway,train,transportation,vehicle)' ),
			array( 'fas fa-suitcase' => 'Suitcase (baggage,luggage,move,suitcase,travel,trip)' ),
			array( 'fas fa-tag' => 'tag (discount,label,price,shopping)' ),
			array( 'fas fa-tags' => 'tags (discount,label,price,shopping)' ),
			array( 'fas fa-taxi' => 'Taxi (cab,cabbie,car,car service,lyft,machine,transportation,travel,uber,vehicle)' ),
			array( 'fas fa-thumbtack' => 'Thumbtack (coordinates,location,marker,pin,thumb-tack)' ),
			array( 'fas fa-ticket-alt' => 'Alternate Ticket (movie,pass,support,ticket)' ),
			array( 'fas fa-tint' => 'tint (color,drop,droplet,raindrop,waterdrop)' ),
			array( 'fas fa-traffic-light' => 'Traffic Light (direction,road,signal,travel)' ),
			array( 'fas fa-train' => 'Train (bullet,commute,locomotive,railway,subway)' ),
			array( 'fas fa-tram' => 'Tram (crossing,machine,mountains,seasonal,transportation)' ),
			array( 'fas fa-tree' => 'Tree (bark,fall,flora,forest,nature,plant,seasonal)' ),
			array( 'fas fa-trophy' => 'trophy (achievement,award,cup,game,winner)' ),
			array( 'fas fa-truck' => 'truck (cargo,delivery,shipping,vehicle)' ),
			array( 'fas fa-tty' => 'TTY (communication,deaf,telephone,teletypewriter,text)' ),
			array( 'fas fa-umbrella' => 'Umbrella (protection,rain,storm,wet)' ),
			array( 'fas fa-university' => 'University (bank,building,college,higher education - students,institution)' ),
			array( 'fas fa-utensil-spoon' => 'Utensil Spoon (cutlery,dining,scoop,silverware,spoon)' ),
			array( 'fas fa-utensils' => 'Utensils (cutlery,dining,dinner,eat,food,fork,knife,restaurant)' ),
			array( 'fas fa-wheelchair' => 'Wheelchair (accessible,handicap,person)' ),
			array( 'fas fa-wifi' => 'WiFi (connection,hotspot,internet,network,wireless)' ),
			array( 'fas fa-wine-glass' => 'Wine Glass (alcohol,beverage,cabernet,drink,grapes,merlot,sauvignon)' ),
			array( 'fas fa-wrench' => 'Wrench (construction,fix,mechanic,plumbing,settings,spanner,tool,update)' ),
		),
		'Maritime' => array(
			array( 'fas fa-anchor' => 'Anchor (berth,boat,dock,embed,link,maritime,moor,secure)' ),
			array( 'fas fa-binoculars' => 'Binoculars (glasses,magnify,scenic,spyglass,view)' ),
			array( 'fas fa-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'far fa-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'fas fa-dharmachakra' => 'Dharmachakra (buddhism,buddhist,wheel of dharma)' ),
			array( 'fas fa-frog' => 'Frog (amphibian,bullfrog,fauna,hop,kermit,kiss,prince,ribbit,toad,wart)' ),
			array( 'fas fa-ship' => 'Ship (boat,sea,water)' ),
			array( 'fas fa-skull-crossbones' => 'Skull & Crossbones (Dungeons & Dragons,alert,bones,d&d,danger,dead,deadly,death,dnd,fantasy,halloween,holiday,jolly-roger,pirate,poison,skeleton,warning)' ),
			array( 'fas fa-swimmer' => 'Swimmer (athlete,head,man,olympics,person,pool,water)' ),
			array( 'fas fa-water' => 'Water (lake,liquid,ocean,sea,swim,wet)' ),
			array( 'fas fa-wind' => 'Wind (air,blow,breeze,fall,seasonal,weather)' ),
		),
		'Marketing' => array(
			array( 'fas fa-ad' => 'Ad (advertisement,media,newspaper,promotion,publicity)' ),
			array( 'fas fa-bullhorn' => 'bullhorn (announcement,broadcast,louder,megaphone,share)' ),
			array( 'fas fa-bullseye' => 'Bullseye (archery,goal,objective,target)' ),
			array( 'fas fa-comment-dollar' => 'Comment Dollar (bubble,chat,commenting,conversation,feedback,message,money,note,notification,pay,sms,speech,spend,texting,transfer)' ),
			array( 'fas fa-comments-dollar' => 'Comments Dollar (bubble,chat,commenting,conversation,feedback,message,money,note,notification,pay,sms,speech,spend,texting,transfer)' ),
			array( 'fas fa-envelope-open-text' => 'Envelope Open-text (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'fas fa-funnel-dollar' => 'Funnel Dollar (filter,money,options,separate,sort)' ),
			array( 'fas fa-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'far fa-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'fas fa-mail-bulk' => 'Mail Bulk (archive,envelope,letter,post office,postal,postcard,send,stamp,usps)' ),
			array( 'fas fa-poll' => 'Poll (results,survey,trend,vote,voting)' ),
			array( 'fas fa-poll-h' => 'Poll H (results,survey,trend,vote,voting)' ),
			array( 'fas fa-search-dollar' => 'Search Dollar (bigger,enlarge,find,magnify,money,preview,zoom)' ),
			array( 'fas fa-search-location' => 'Search Location (bigger,enlarge,find,magnify,preview,zoom)' ),
		),
		'Mathematics' => array(
			array( 'fas fa-calculator' => 'Calculator (abacus,addition,arithmetic,counting,math,multiplication,subtraction)' ),
			array( 'fas fa-divide' => 'Divide (arithmetic,calculus,division,math)' ),
			array( 'fas fa-equals' => 'Equals (arithmetic,even,match,math)' ),
			array( 'fas fa-greater-than' => 'Greater Than (arithmetic,compare,math)' ),
			array( 'fas fa-greater-than-equal' => 'Greater Than Equal To (arithmetic,compare,math)' ),
			array( 'fas fa-infinity' => 'Infinity (eternity,forever,math)' ),
			array( 'fas fa-less-than' => 'Less Than (arithmetic,compare,math)' ),
			array( 'fas fa-less-than-equal' => 'Less Than Equal To (arithmetic,compare,math)' ),
			array( 'fas fa-minus' => 'minus (collapse,delete,hide,minify,negative,remove,trash)' ),
			array( 'fas fa-not-equal' => 'Not Equal (arithmetic,compare,math)' ),
			array( 'fas fa-percentage' => 'Percentage (discount,fraction,proportion,rate,ratio)' ),
			array( 'fas fa-plus' => 'plus (add,create,expand,new,positive,shape)' ),
			array( 'fas fa-square-root-alt' => 'Alternate Square Root (arithmetic,calculus,division,math)' ),
			array( 'fas fa-subscript' => 'subscript (edit,font,format,text,type)' ),
			array( 'fas fa-superscript' => 'superscript (edit,exponential,font,format,text,type)' ),
			array( 'fas fa-times' => 'Times (close,cross,error,exit,incorrect,notice,notification,notify,problem,wrong,x)' ),
			array( 'fas fa-wave-square' => 'Square Wave (frequency,pulse,signal)' ),
		),
		'Medical' => array(
			array( 'fas fa-allergies' => 'Allergies (allergy,freckles,hand,hives,pox,skin,spots)' ),
			array( 'fas fa-ambulance' => 'ambulance (emergency,emt,er,help,hospital,support,vehicle)' ),
			array( 'fas fa-band-aid' => 'Band-Aid (bandage,boo boo,first aid,ouch)' ),
			array( 'fas fa-biohazard' => 'Biohazard (danger,dangerous,hazmat,medical,radioactive,toxic,waste,zombie)' ),
			array( 'fas fa-bone' => 'Bone (calcium,dog,skeletal,skeleton,tibia)' ),
			array( 'fas fa-bong' => 'Bong (aparatus,cannabis,marijuana,pipe,smoke,smoking)' ),
			array( 'fas fa-book-medical' => 'Medical Book (diary,documentation,health,history,journal,library,read,record)' ),
			array( 'fas fa-brain' => 'Brain (cerebellum,gray matter,intellect,medulla oblongata,mind,noodle,wit)' ),
			array( 'fas fa-briefcase-medical' => 'Medical Briefcase (doctor,emt,first aid,health)' ),
			array( 'fas fa-burn' => 'Burn (caliente,energy,fire,flame,gas,heat,hot)' ),
			array( 'fas fa-cannabis' => 'Cannabis (bud,chronic,drugs,endica,endo,ganja,marijuana,mary jane,pot,reefer,sativa,spliff,weed,whacky-tabacky)' ),
			array( 'fas fa-capsules' => 'Capsules (drugs,medicine,pills,prescription)' ),
			array( 'fas fa-clinic-medical' => 'Medical Clinic (doctor,general practitioner,hospital,infirmary,medicine,office,outpatient)' ),
			array( 'fas fa-comment-medical' => 'Alternate Medical Chat (advice,bubble,chat,commenting,conversation,diagnose,feedback,message,note,notification,prescription,sms,speech,texting)' ),
			array( 'fas fa-crutch' => 'Crutch (cane,injury,mobility,wheelchair)' ),
			array( 'fas fa-diagnoses' => 'Diagnoses (analyze,detect,diagnosis,examine,medicine)' ),
			array( 'fas fa-dna' => 'DNA (double helix,genetic,helix,molecule,protein)' ),
			array( 'fas fa-file-medical' => 'Medical File (document,health,history,prescription,record)' ),
			array( 'fas fa-file-medical-alt' => 'Alternate Medical File (document,health,history,prescription,record)' ),
			array( 'fas fa-file-prescription' => 'File Prescription (document,drugs,medical,medicine,rx)' ),
			array( 'fas fa-first-aid' => 'First Aid (emergency,emt,health,medical,rescue)' ),
			array( 'fas fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'far fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'fas fa-heartbeat' => 'Heartbeat (ekg,electrocardiogram,health,lifeline,vital signs)' ),
			array( 'fas fa-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'far fa-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'fas fa-hospital-alt' => 'Alternate Hospital (building,emergency room,medical center)' ),
			array( 'fas fa-hospital-symbol' => 'Hospital Symbol (clinic,emergency,map)' ),
			array( 'fas fa-id-card-alt' => 'Alternate Identification Card (contact,demographics,document,identification,issued,profile)' ),
			array( 'fas fa-joint' => 'Joint (blunt,cannabis,doobie,drugs,marijuana,roach,smoke,smoking,spliff)' ),
			array( 'fas fa-laptop-medical' => 'Laptop Medical (computer,device,ehr,electronic health records,history)' ),
			array( 'fas fa-microscope' => 'Microscope (electron,lens,optics,science,shrink)' ),
			array( 'fas fa-mortar-pestle' => 'Mortar Pestle (crush,culinary,grind,medical,mix,pharmacy,prescription,spices)' ),
			array( 'fas fa-notes-medical' => 'Medical Notes (clipboard,doctor,ehr,health,history,records)' ),
			array( 'fas fa-pager' => 'Pager (beeper,cellphone,communication)' ),
			array( 'fas fa-pills' => 'Pills (drugs,medicine,prescription,tablets)' ),
			array( 'fas fa-plus' => 'plus (add,create,expand,new,positive,shape)' ),
			array( 'fas fa-poop' => 'Poop (crap,poop,shit,smile,turd)' ),
			array( 'fas fa-prescription' => 'Prescription (drugs,medical,medicine,pharmacy,rx)' ),
			array( 'fas fa-prescription-bottle' => 'Prescription Bottle (drugs,medical,medicine,pharmacy,rx)' ),
			array( 'fas fa-prescription-bottle-alt' => 'Alternate Prescription Bottle (drugs,medical,medicine,pharmacy,rx)' ),
			array( 'fas fa-procedures' => 'Procedures (EKG,bed,electrocardiogram,health,hospital,life,patient,vital)' ),
			array( 'fas fa-radiation' => 'Radiation (danger,dangerous,deadly,hazard,nuclear,radioactive,warning)' ),
			array( 'fas fa-radiation-alt' => 'Alternate Radiation (danger,dangerous,deadly,hazard,nuclear,radioactive,warning)' ),
			array( 'fas fa-smoking' => 'Smoking (cancer,cigarette,nicotine,smoking status,tobacco)' ),
			array( 'fas fa-smoking-ban' => 'Smoking Ban (ban,cancel,no smoking,non-smoking)' ),
			array( 'fas fa-star-of-life' => 'Star of Life (doctor,emt,first aid,health,medical)' ),
			array( 'fas fa-stethoscope' => 'Stethoscope (diagnosis,doctor,general practitioner,hospital,infirmary,medicine,office,outpatient)' ),
			array( 'fas fa-syringe' => 'Syringe (doctor,immunizations,medical,needle)' ),
			array( 'fas fa-tablets' => 'Tablets (drugs,medicine,pills,prescription)' ),
			array( 'fas fa-teeth' => 'Teeth (bite,dental,dentist,gums,mouth,smile,tooth)' ),
			array( 'fas fa-teeth-open' => 'Teeth Open (dental,dentist,gums bite,mouth,smile,tooth)' ),
			array( 'fas fa-thermometer' => 'Thermometer (mercury,status,temperature)' ),
			array( 'fas fa-tooth' => 'Tooth (bicuspid,dental,dentist,molar,mouth,teeth)' ),
			array( 'fas fa-user-md' => 'Doctor (job,medical,nurse,occupation,physician,profile,surgeon)' ),
			array( 'fas fa-user-nurse' => 'Nurse (doctor,midwife,practitioner,surgeon)' ),
			array( 'fas fa-vial' => 'Vial (experiment,lab,sample,science,test,test tube)' ),
			array( 'fas fa-vials' => 'Vials (experiment,lab,sample,science,test,test tube)' ),
			array( 'fas fa-weight' => 'Weight (health,measurement,scale,weight)' ),
			array( 'fas fa-x-ray' => 'X-Ray (health,medical,radiological images,radiology,skeleton)' ),
		),
		'Moving' => array(
			array( 'fas fa-archive' => 'Archive (box,package,save,storage)' ),
			array( 'fas fa-box-open' => 'Box Open (archive,container,package,storage,unpack)' ),
			array( 'fas fa-couch' => 'Couch (chair,cushion,furniture,relax,sofa)' ),
			array( 'fas fa-dolly' => 'Dolly (carry,shipping,transport)' ),
			array( 'fas fa-people-carry' => 'People Carry (box,carry,fragile,help,movers,package)' ),
			array( 'fas fa-route' => 'Route (directions,navigation,travel)' ),
			array( 'fas fa-sign' => 'Sign (directions,real estate,signage,wayfinding)' ),
			array( 'fas fa-suitcase' => 'Suitcase (baggage,luggage,move,suitcase,travel,trip)' ),
			array( 'fas fa-tape' => 'Tape (design,package,sticky)' ),
			array( 'fas fa-truck-loading' => 'Truck Loading (box,cargo,delivery,inventory,moving,rental,vehicle)' ),
			array( 'fas fa-truck-moving' => 'Truck Moving (cargo,inventory,rental,vehicle)' ),
			array( 'fas fa-wine-glass' => 'Wine Glass (alcohol,beverage,cabernet,drink,grapes,merlot,sauvignon)' ),
		),
		'Music' => array(
			array( 'fas fa-drum' => 'Drum (instrument,music,percussion,snare,sound)' ),
			array( 'fas fa-drum-steelpan' => 'Drum Steelpan (calypso,instrument,music,percussion,reggae,snare,sound,steel,tropical)' ),
			array( 'fas fa-file-audio' => 'Audio File (document,mp3,music,page,play,sound)' ),
			array( 'far fa-file-audio' => 'Audio File (document,mp3,music,page,play,sound)' ),
			array( 'fas fa-guitar' => 'Guitar (acoustic,instrument,music,rock,rock and roll,song,strings)' ),
			array( 'fas fa-headphones' => 'headphones (audio,listen,music,sound,speaker)' ),
			array( 'fas fa-headphones-alt' => 'Alternate Headphones (audio,listen,music,sound,speaker)' ),
			array( 'fas fa-microphone' => 'microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-microphone-alt' => 'Alternate Microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-microphone-alt-slash' => 'Alternate Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-microphone-slash' => 'Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-music' => 'Music (lyrics,melody,note,sing,sound)' ),
			array( 'fab fa-napster' => 'Napster' ),
			array( 'fas fa-play' => 'play (audio,music,playing,sound,start,video)' ),
			array( 'fas fa-record-vinyl' => 'Record Vinyl (LP,album,analog,music,phonograph,sound)' ),
			array( 'fas fa-sliders-h' => 'Horizontal Sliders (adjust,settings,sliders,toggle)' ),
			array( 'fab fa-soundcloud' => 'SoundCloud' ),
			array( 'fab fa-spotify' => 'Spotify' ),
			array( 'fas fa-volume-down' => 'Volume Down (audio,lower,music,quieter,sound,speaker)' ),
			array( 'fas fa-volume-mute' => 'Volume Mute (audio,music,quiet,sound,speaker)' ),
			array( 'fas fa-volume-off' => 'Volume Off (audio,ban,music,mute,quiet,silent,sound)' ),
			array( 'fas fa-volume-up' => 'Volume Up (audio,higher,louder,music,sound,speaker)' ),
		),
		'Objects' => array(
			array( 'fas fa-ambulance' => 'ambulance (emergency,emt,er,help,hospital,support,vehicle)' ),
			array( 'fas fa-anchor' => 'Anchor (berth,boat,dock,embed,link,maritime,moor,secure)' ),
			array( 'fas fa-archive' => 'Archive (box,package,save,storage)' ),
			array( 'fas fa-award' => 'Award (honor,praise,prize,recognition,ribbon,trophy)' ),
			array( 'fas fa-baby-carriage' => 'Baby Carriage (buggy,carrier,infant,push,stroller,transportation,walk,wheels)' ),
			array( 'fas fa-balance-scale' => 'Balance Scale (balanced,justice,legal,measure,weight)' ),
			array( 'fas fa-balance-scale-left' => 'Balance Scale (Left-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'fas fa-balance-scale-right' => 'Balance Scale (Right-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'fas fa-bath' => 'Bath (clean,shower,tub,wash)' ),
			array( 'fas fa-bed' => 'Bed (lodging,rest,sleep,travel)' ),
			array( 'fas fa-beer' => 'beer (alcohol,ale,bar,beverage,brewery,drink,lager,liquor,mug,stein)' ),
			array( 'fas fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'far fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'fas fa-bicycle' => 'Bicycle (bike,gears,pedal,transportation,vehicle)' ),
			array( 'fas fa-binoculars' => 'Binoculars (glasses,magnify,scenic,spyglass,view)' ),
			array( 'fas fa-birthday-cake' => 'Birthday Cake (anniversary,bakery,candles,celebration,dessert,frosting,holiday,party,pastry)' ),
			array( 'fas fa-blender' => 'Blender (cocktail,milkshake,mixer,puree,smoothie)' ),
			array( 'fas fa-bomb' => 'Bomb (error,explode,fuse,grenade,warning)' ),
			array( 'fas fa-book' => 'book (diary,documentation,journal,library,read)' ),
			array( 'fas fa-book-dead' => 'Book of the Dead (Dungeons & Dragons,crossbones,d&d,dark arts,death,dnd,documentation,evil,fantasy,halloween,holiday,necronomicon,read,skull,spell)' ),
			array( 'fas fa-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'far fa-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'fas fa-briefcase' => 'Briefcase (bag,business,luggage,office,work)' ),
			array( 'fas fa-broadcast-tower' => 'Broadcast Tower (airwaves,antenna,radio,reception,waves)' ),
			array( 'fas fa-bug' => 'Bug (beetle,error,insect,report)' ),
			array( 'fas fa-building' => 'Building (apartment,business,city,company,office,work)' ),
			array( 'far fa-building' => 'Building (apartment,business,city,company,office,work)' ),
			array( 'fas fa-bullhorn' => 'bullhorn (announcement,broadcast,louder,megaphone,share)' ),
			array( 'fas fa-bullseye' => 'Bullseye (archery,goal,objective,target)' ),
			array( 'fas fa-bus' => 'Bus (public transportation,transportation,travel,vehicle)' ),
			array( 'fas fa-calculator' => 'Calculator (abacus,addition,arithmetic,counting,math,multiplication,subtraction)' ),
			array( 'fas fa-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'far fa-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'fas fa-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'far fa-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'fas fa-camera' => 'camera (image,lens,photo,picture,record,shutter,video)' ),
			array( 'fas fa-camera-retro' => 'Retro Camera (image,lens,photo,picture,record,shutter,video)' ),
			array( 'fas fa-candy-cane' => 'Candy Cane (candy,christmas,holiday,mint,peppermint,striped,xmas)' ),
			array( 'fas fa-car' => 'Car (auto,automobile,sedan,transportation,travel,vehicle)' ),
			array( 'fas fa-carrot' => 'Carrot (bugs bunny,orange,vegan,vegetable)' ),
			array( 'fas fa-church' => 'Church (building,cathedral,chapel,community,religion)' ),
			array( 'fas fa-clipboard' => 'Clipboard (copy,notes,paste,record)' ),
			array( 'far fa-clipboard' => 'Clipboard (copy,notes,paste,record)' ),
			array( 'fas fa-cloud' => 'Cloud (atmosphere,fog,overcast,save,upload,weather)' ),
			array( 'fas fa-coffee' => 'Coffee (beverage,breakfast,cafe,drink,fall,morning,mug,seasonal,tea)' ),
			array( 'fas fa-cog' => 'cog (gear,mechanical,settings,sprocket,wheel)' ),
			array( 'fas fa-cogs' => 'cogs (gears,mechanical,settings,sprocket,wheel)' ),
			array( 'fas fa-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'far fa-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'fas fa-cookie' => 'Cookie (baked good,chips,chocolate,eat,snack,sweet,treat)' ),
			array( 'fas fa-cookie-bite' => 'Cookie Bite (baked good,bitten,chips,chocolate,eat,snack,sweet,treat)' ),
			array( 'fas fa-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'far fa-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'fas fa-cube' => 'Cube (3d,block,dice,package,square,tesseract)' ),
			array( 'fas fa-cubes' => 'Cubes (3d,block,dice,package,pyramid,square,stack,tesseract)' ),
			array( 'fas fa-cut' => 'Cut (clip,scissors,snip)' ),
			array( 'fas fa-dice' => 'Dice (chance,gambling,game,roll)' ),
			array( 'fas fa-dice-d20' => 'Dice D20 (Dungeons & Dragons,chance,d&d,dnd,fantasy,gambling,game,roll)' ),
			array( 'fas fa-dice-d6' => 'Dice D6 (Dungeons & Dragons,chance,d&d,dnd,fantasy,gambling,game,roll)' ),
			array( 'fas fa-dice-five' => 'Dice Five (chance,gambling,game,roll)' ),
			array( 'fas fa-dice-four' => 'Dice Four (chance,gambling,game,roll)' ),
			array( 'fas fa-dice-one' => 'Dice One (chance,gambling,game,roll)' ),
			array( 'fas fa-dice-six' => 'Dice Six (chance,gambling,game,roll)' ),
			array( 'fas fa-dice-three' => 'Dice Three (chance,gambling,game,roll)' ),
			array( 'fas fa-dice-two' => 'Dice Two (chance,gambling,game,roll)' ),
			array( 'fas fa-digital-tachograph' => 'Digital Tachograph (data,distance,speed,tachometer)' ),
			array( 'fas fa-door-closed' => 'Door Closed (enter,exit,locked)' ),
			array( 'fas fa-door-open' => 'Door Open (enter,exit,welcome)' ),
			array( 'fas fa-drum' => 'Drum (instrument,music,percussion,snare,sound)' ),
			array( 'fas fa-drum-steelpan' => 'Drum Steelpan (calypso,instrument,music,percussion,reggae,snare,sound,steel,tropical)' ),
			array( 'fas fa-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'far fa-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'fas fa-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'far fa-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'fas fa-eraser' => 'eraser (art,delete,remove,rubber)' ),
			array( 'fas fa-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'far fa-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'fas fa-eye-dropper' => 'Eye Dropper (beaker,clone,color,copy,eyedropper,pipette)' ),
			array( 'fas fa-fax' => 'Fax (business,communicate,copy,facsimile,send)' ),
			array( 'fas fa-feather' => 'Feather (bird,light,plucked,quill,write)' ),
			array( 'fas fa-feather-alt' => 'Alternate Feather (bird,light,plucked,quill,write)' ),
			array( 'fas fa-fighter-jet' => 'fighter-jet (airplane,fast,fly,goose,maverick,plane,quick,top gun,transportation,travel)' ),
			array( 'fas fa-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'far fa-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'fas fa-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'far fa-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'fas fa-file-prescription' => 'File Prescription (document,drugs,medical,medicine,rx)' ),
			array( 'fas fa-film' => 'Film (cinema,movie,strip,video)' ),
			array( 'fas fa-fire' => 'fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'fas fa-fire-alt' => 'Alternate Fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'fas fa-fire-extinguisher' => 'fire-extinguisher (burn,caliente,fire fighter,flame,heat,hot,rescue)' ),
			array( 'fas fa-flag' => 'flag (country,notice,notification,notify,pole,report,symbol)' ),
			array( 'far fa-flag' => 'flag (country,notice,notification,notify,pole,report,symbol)' ),
			array( 'fas fa-flag-checkered' => 'flag-checkered (notice,notification,notify,pole,racing,report,symbol)' ),
			array( 'fas fa-flask' => 'Flask (beaker,experimental,labs,science)' ),
			array( 'fas fa-futbol' => 'Futbol (ball,football,mls,soccer)' ),
			array( 'far fa-futbol' => 'Futbol (ball,football,mls,soccer)' ),
			array( 'fas fa-gamepad' => 'Gamepad (arcade,controller,d-pad,joystick,video,video game)' ),
			array( 'fas fa-gavel' => 'Gavel (hammer,judge,law,lawyer,opinion)' ),
			array( 'fas fa-gem' => 'Gem (diamond,jewelry,sapphire,stone,treasure)' ),
			array( 'far fa-gem' => 'Gem (diamond,jewelry,sapphire,stone,treasure)' ),
			array( 'fas fa-gift' => 'gift (christmas,generosity,giving,holiday,party,present,wrapped,xmas)' ),
			array( 'fas fa-gifts' => 'Gifts (christmas,generosity,giving,holiday,party,present,wrapped,xmas)' ),
			array( 'fas fa-glass-cheers' => 'Glass Cheers (alcohol,bar,beverage,celebration,champagne,clink,drink,holiday,new year\'s eve,party,toast)' ),
			array( 'fas fa-glass-martini' => 'Martini Glass (alcohol,bar,beverage,drink,liquor)' ),
			array( 'fas fa-glass-whiskey' => 'Glass Whiskey (alcohol,bar,beverage,bourbon,drink,liquor,neat,rye,scotch,whisky)' ),
			array( 'fas fa-glasses' => 'Glasses (hipster,nerd,reading,sight,spectacles,vision)' ),
			array( 'fas fa-globe' => 'Globe (all,coordinates,country,earth,global,gps,language,localize,location,map,online,place,planet,translate,travel,world)' ),
			array( 'fas fa-graduation-cap' => 'Graduation Cap (ceremony,college,graduate,learning,school,student)' ),
			array( 'fas fa-guitar' => 'Guitar (acoustic,instrument,music,rock,rock and roll,song,strings)' ),
			array( 'fas fa-hat-wizard' => 'Wizard\'s Hat (Dungeons & Dragons,accessory,buckle,clothing,d&d,dnd,fantasy,halloween,head,holiday,mage,magic,pointy,witch)' ),
			array( 'fas fa-hdd' => 'HDD (cpu,hard drive,harddrive,machine,save,storage)' ),
			array( 'far fa-hdd' => 'HDD (cpu,hard drive,harddrive,machine,save,storage)' ),
			array( 'fas fa-headphones' => 'headphones (audio,listen,music,sound,speaker)' ),
			array( 'fas fa-headphones-alt' => 'Alternate Headphones (audio,listen,music,sound,speaker)' ),
			array( 'fas fa-headset' => 'Headset (audio,gamer,gaming,listen,live chat,microphone,shot caller,sound,support,telemarketer)' ),
			array( 'fas fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'far fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'fas fa-heart-broken' => 'Heart Broken (breakup,crushed,dislike,dumped,grief,love,lovesick,relationship,sad)' ),
			array( 'fas fa-helicopter' => 'Helicopter (airwolf,apache,chopper,flight,fly,travel)' ),
			array( 'fas fa-highlighter' => 'Highlighter (edit,marker,sharpie,update,write)' ),
			array( 'fas fa-holly-berry' => 'Holly Berry (catwoman,christmas,decoration,flora,halle,holiday,ororo munroe,plant,storm,xmas)' ),
			array( 'fas fa-home' => 'home (abode,building,house,main)' ),
			array( 'fas fa-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'far fa-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'fas fa-hourglass' => 'Hourglass (hour,minute,sand,stopwatch,time)' ),
			array( 'far fa-hourglass' => 'Hourglass (hour,minute,sand,stopwatch,time)' ),
			array( 'fas fa-igloo' => 'Igloo (dome,dwelling,eskimo,home,house,ice,snow)' ),
			array( 'fas fa-image' => 'Image (album,landscape,photo,picture)' ),
			array( 'far fa-image' => 'Image (album,landscape,photo,picture)' ),
			array( 'fas fa-images' => 'Images (album,landscape,photo,picture)' ),
			array( 'far fa-images' => 'Images (album,landscape,photo,picture)' ),
			array( 'fas fa-industry' => 'Industry (building,factory,industrial,manufacturing,mill,warehouse)' ),
			array( 'fas fa-key' => 'key (lock,password,private,secret,unlock)' ),
			array( 'fas fa-keyboard' => 'Keyboard (accessory,edit,input,text,type,write)' ),
			array( 'far fa-keyboard' => 'Keyboard (accessory,edit,input,text,type,write)' ),
			array( 'fas fa-laptop' => 'Laptop (computer,cpu,dell,demo,device,mac,macbook,machine,pc)' ),
			array( 'fas fa-leaf' => 'leaf (eco,flora,nature,plant,vegan)' ),
			array( 'fas fa-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'far fa-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'fas fa-life-ring' => 'Life Ring (coast guard,help,overboard,save,support)' ),
			array( 'far fa-life-ring' => 'Life Ring (coast guard,help,overboard,save,support)' ),
			array( 'fas fa-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'far fa-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'fas fa-lock' => 'lock (admin,lock,open,password,private,protect,security)' ),
			array( 'fas fa-lock-open' => 'Lock Open (admin,lock,open,password,private,protect,security)' ),
			array( 'fas fa-magic' => 'magic (autocomplete,automatic,mage,magic,spell,wand,witch,wizard)' ),
			array( 'fas fa-magnet' => 'magnet (Attract,lodestone,tool)' ),
			array( 'fas fa-map' => 'Map (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'far fa-map' => 'Map (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'fas fa-map-marker' => 'map-marker (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'fas fa-map-marker-alt' => 'Alternate Map Marker (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'fas fa-map-pin' => 'Map Pin (address,agree,coordinates,destination,gps,localize,location,map,marker,navigation,pin,place,position,travel)' ),
			array( 'fas fa-map-signs' => 'Map Signs (directions,directory,map,signage,wayfinding)' ),
			array( 'fas fa-marker' => 'Marker (design,edit,sharpie,update,write)' ),
			array( 'fas fa-medal' => 'Medal (award,ribbon,star,trophy)' ),
			array( 'fas fa-medkit' => 'medkit (first aid,firstaid,health,help,support)' ),
			array( 'fas fa-memory' => 'Memory (DIMM,RAM,hardware,storage,technology)' ),
			array( 'fas fa-microchip' => 'Microchip (cpu,hardware,processor,technology)' ),
			array( 'fas fa-microphone' => 'microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-microphone-alt' => 'Alternate Microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-mitten' => 'Mitten (clothing,cold,glove,hands,knitted,seasonal,warmth)' ),
			array( 'fas fa-mobile' => 'Mobile Phone (apple,call,cell phone,cellphone,device,iphone,number,screen,telephone)' ),
			array( 'fas fa-mobile-alt' => 'Alternate Mobile (apple,call,cell phone,cellphone,device,iphone,number,screen,telephone)' ),
			array( 'fas fa-money-bill' => 'Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'fas fa-money-bill-alt' => 'Alternate Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'far fa-money-bill-alt' => 'Alternate Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'fas fa-money-check' => 'Money Check (bank check,buy,checkout,cheque,money,payment,price,purchase)' ),
			array( 'fas fa-money-check-alt' => 'Alternate Money Check (bank check,buy,checkout,cheque,money,payment,price,purchase)' ),
			array( 'fas fa-moon' => 'Moon (contrast,crescent,dark,lunar,night)' ),
			array( 'far fa-moon' => 'Moon (contrast,crescent,dark,lunar,night)' ),
			array( 'fas fa-motorcycle' => 'Motorcycle (bike,machine,transportation,vehicle)' ),
			array( 'fas fa-mug-hot' => 'Mug Hot (caliente,cocoa,coffee,cup,drink,holiday,hot chocolate,steam,tea,warmth)' ),
			array( 'fas fa-newspaper' => 'Newspaper (article,editorial,headline,journal,journalism,news,press)' ),
			array( 'far fa-newspaper' => 'Newspaper (article,editorial,headline,journal,journalism,news,press)' ),
			array( 'fas fa-paint-brush' => 'Paint Brush (acrylic,art,brush,color,fill,paint,pigment,watercolor)' ),
			array( 'fas fa-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'far fa-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'fas fa-paperclip' => 'Paperclip (attach,attachment,connect,link)' ),
			array( 'fas fa-paste' => 'Paste (clipboard,copy,document,paper)' ),
			array( 'fas fa-paw' => 'Paw (animal,cat,dog,pet,print)' ),
			array( 'fas fa-pen' => 'Pen (design,edit,update,write)' ),
			array( 'fas fa-pen-alt' => 'Alternate Pen (design,edit,update,write)' ),
			array( 'fas fa-pen-fancy' => 'Pen Fancy (design,edit,fountain pen,update,write)' ),
			array( 'fas fa-pen-nib' => 'Pen Nib (design,edit,fountain pen,update,write)' ),
			array( 'fas fa-pencil-alt' => 'Alternate Pencil (design,edit,pencil,update,write)' ),
			array( 'fas fa-phone' => 'Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'fas fa-phone-alt' => 'Alternate Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'fas fa-plane' => 'plane (airplane,destination,fly,location,mode,travel,trip)' ),
			array( 'fas fa-plug' => 'Plug (connect,electric,online,power)' ),
			array( 'fas fa-print' => 'print (business,copy,document,office,paper)' ),
			array( 'fas fa-puzzle-piece' => 'Puzzle Piece (add-on,addon,game,section)' ),
			array( 'fas fa-ring' => 'Ring (Dungeons & Dragons,Gollum,band,binding,d&d,dnd,engagement,fantasy,gold,jewelry,marriage,precious)' ),
			array( 'fas fa-road' => 'road (highway,map,pavement,route,street,travel)' ),
			array( 'fas fa-rocket' => 'rocket (aircraft,app,jet,launch,nasa,space)' ),
			array( 'fas fa-ruler-combined' => 'Ruler Combined (design,draft,length,measure,planning)' ),
			array( 'fas fa-ruler-horizontal' => 'Ruler Horizontal (design,draft,length,measure,planning)' ),
			array( 'fas fa-ruler-vertical' => 'Ruler Vertical (design,draft,length,measure,planning)' ),
			array( 'fas fa-satellite' => 'Satellite (communications,hardware,orbit,space)' ),
			array( 'fas fa-satellite-dish' => 'Satellite Dish (SETI,communications,hardware,receiver,saucer,signal)' ),
			array( 'fas fa-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'far fa-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'fas fa-school' => 'School (building,education,learn,student,teacher)' ),
			array( 'fas fa-screwdriver' => 'Screwdriver (admin,fix,mechanic,repair,settings,tool)' ),
			array( 'fas fa-scroll' => 'Scroll (Dungeons & Dragons,announcement,d&d,dnd,fantasy,paper,script)' ),
			array( 'fas fa-sd-card' => 'Sd Card (image,memory,photo,save)' ),
			array( 'fas fa-search' => 'Search (bigger,enlarge,find,magnify,preview,zoom)' ),
			array( 'fas fa-shield-alt' => 'Alternate Shield (achievement,award,block,defend,security,winner)' ),
			array( 'fas fa-shopping-bag' => 'Shopping Bag (buy,checkout,grocery,payment,purchase)' ),
			array( 'fas fa-shopping-basket' => 'Shopping Basket (buy,checkout,grocery,payment,purchase)' ),
			array( 'fas fa-shopping-cart' => 'shopping-cart (buy,checkout,grocery,payment,purchase)' ),
			array( 'fas fa-shower' => 'Shower (bath,clean,faucet,water)' ),
			array( 'fas fa-sim-card' => 'SIM Card (hard drive,hardware,portable,storage,technology,tiny)' ),
			array( 'fas fa-skull-crossbones' => 'Skull & Crossbones (Dungeons & Dragons,alert,bones,d&d,danger,dead,deadly,death,dnd,fantasy,halloween,holiday,jolly-roger,pirate,poison,skeleton,warning)' ),
			array( 'fas fa-sleigh' => 'Sleigh (christmas,claus,fly,holiday,santa,sled,snow,xmas)' ),
			array( 'fas fa-snowflake' => 'Snowflake (precipitation,rain,winter)' ),
			array( 'far fa-snowflake' => 'Snowflake (precipitation,rain,winter)' ),
			array( 'fas fa-snowplow' => 'Snowplow (clean up,cold,road,storm,winter)' ),
			array( 'fas fa-space-shuttle' => 'Space Shuttle (astronaut,machine,nasa,rocket,transportation)' ),
			array( 'fas fa-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'far fa-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'fas fa-sticky-note' => 'Sticky Note (message,note,paper,reminder,sticker)' ),
			array( 'far fa-sticky-note' => 'Sticky Note (message,note,paper,reminder,sticker)' ),
			array( 'fas fa-stopwatch' => 'Stopwatch (clock,reminder,time)' ),
			array( 'fas fa-stroopwafel' => 'Stroopwafel (caramel,cookie,dessert,sweets,waffle)' ),
			array( 'fas fa-subway' => 'Subway (machine,railway,train,transportation,vehicle)' ),
			array( 'fas fa-suitcase' => 'Suitcase (baggage,luggage,move,suitcase,travel,trip)' ),
			array( 'fas fa-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'far fa-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'fas fa-tablet' => 'tablet (apple,device,ipad,kindle,screen)' ),
			array( 'fas fa-tablet-alt' => 'Alternate Tablet (apple,device,ipad,kindle,screen)' ),
			array( 'fas fa-tachometer-alt' => 'Alternate Tachometer (dashboard,fast,odometer,speed,speedometer)' ),
			array( 'fas fa-tag' => 'tag (discount,label,price,shopping)' ),
			array( 'fas fa-tags' => 'tags (discount,label,price,shopping)' ),
			array( 'fas fa-taxi' => 'Taxi (cab,cabbie,car,car service,lyft,machine,transportation,travel,uber,vehicle)' ),
			array( 'fas fa-thumbtack' => 'Thumbtack (coordinates,location,marker,pin,thumb-tack)' ),
			array( 'fas fa-ticket-alt' => 'Alternate Ticket (movie,pass,support,ticket)' ),
			array( 'fas fa-toilet' => 'Toilet (bathroom,flush,john,loo,pee,plumbing,poop,porcelain,potty,restroom,throne,washroom,waste,wc)' ),
			array( 'fas fa-toolbox' => 'Toolbox (admin,container,fix,repair,settings,tools)' ),
			array( 'fas fa-tools' => 'Tools (admin,fix,repair,screwdriver,settings,tools,wrench)' ),
			array( 'fas fa-train' => 'Train (bullet,commute,locomotive,railway,subway)' ),
			array( 'fas fa-tram' => 'Tram (crossing,machine,mountains,seasonal,transportation)' ),
			array( 'fas fa-trash' => 'Trash (delete,garbage,hide,remove)' ),
			array( 'fas fa-trash-alt' => 'Alternate Trash (delete,garbage,hide,remove,trash-o)' ),
			array( 'far fa-trash-alt' => 'Alternate Trash (delete,garbage,hide,remove,trash-o)' ),
			array( 'fas fa-tree' => 'Tree (bark,fall,flora,forest,nature,plant,seasonal)' ),
			array( 'fas fa-trophy' => 'trophy (achievement,award,cup,game,winner)' ),
			array( 'fas fa-truck' => 'truck (cargo,delivery,shipping,vehicle)' ),
			array( 'fas fa-tv' => 'Television (computer,display,monitor,television)' ),
			array( 'fas fa-umbrella' => 'Umbrella (protection,rain,storm,wet)' ),
			array( 'fas fa-university' => 'University (bank,building,college,higher education - students,institution)' ),
			array( 'fas fa-unlock' => 'unlock (admin,lock,password,private,protect)' ),
			array( 'fas fa-unlock-alt' => 'Alternate Unlock (admin,lock,password,private,protect)' ),
			array( 'fas fa-utensil-spoon' => 'Utensil Spoon (cutlery,dining,scoop,silverware,spoon)' ),
			array( 'fas fa-utensils' => 'Utensils (cutlery,dining,dinner,eat,food,fork,knife,restaurant)' ),
			array( 'fas fa-wallet' => 'Wallet (billfold,cash,currency,money)' ),
			array( 'fas fa-weight' => 'Weight (health,measurement,scale,weight)' ),
			array( 'fas fa-wheelchair' => 'Wheelchair (accessible,handicap,person)' ),
			array( 'fas fa-wine-glass' => 'Wine Glass (alcohol,beverage,cabernet,drink,grapes,merlot,sauvignon)' ),
			array( 'fas fa-wrench' => 'Wrench (construction,fix,mechanic,plumbing,settings,spanner,tool,update)' ),
		),
		'Payments & Shopping' => array(
			array( 'fab fa-alipay' => 'Alipay' ),
			array( 'fab fa-amazon-pay' => 'Amazon Pay' ),
			array( 'fab fa-apple-pay' => 'Apple Pay' ),
			array( 'fas fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'far fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'fab fa-bitcoin' => 'Bitcoin' ),
			array( 'fas fa-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'far fa-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'fab fa-btc' => 'BTC' ),
			array( 'fas fa-bullhorn' => 'bullhorn (announcement,broadcast,louder,megaphone,share)' ),
			array( 'fas fa-camera' => 'camera (image,lens,photo,picture,record,shutter,video)' ),
			array( 'fas fa-camera-retro' => 'Retro Camera (image,lens,photo,picture,record,shutter,video)' ),
			array( 'fas fa-cart-arrow-down' => 'Shopping Cart Arrow Down (download,save,shopping)' ),
			array( 'fas fa-cart-plus' => 'Add to Shopping Cart (add,create,new,positive,shopping)' ),
			array( 'fab fa-cc-amazon-pay' => 'Amazon Pay Credit Card' ),
			array( 'fab fa-cc-amex' => 'American Express Credit Card (amex)' ),
			array( 'fab fa-cc-apple-pay' => 'Apple Pay Credit Card' ),
			array( 'fab fa-cc-diners-club' => 'Diner\'s Club Credit Card' ),
			array( 'fab fa-cc-discover' => 'Discover Credit Card' ),
			array( 'fab fa-cc-jcb' => 'JCB Credit Card' ),
			array( 'fab fa-cc-mastercard' => 'MasterCard Credit Card' ),
			array( 'fab fa-cc-paypal' => 'Paypal Credit Card' ),
			array( 'fab fa-cc-stripe' => 'Stripe Credit Card' ),
			array( 'fab fa-cc-visa' => 'Visa Credit Card' ),
			array( 'fas fa-certificate' => 'certificate (badge,star,verified)' ),
			array( 'fas fa-credit-card' => 'Credit Card (buy,checkout,credit-card-alt,debit,money,payment,purchase)' ),
			array( 'far fa-credit-card' => 'Credit Card (buy,checkout,credit-card-alt,debit,money,payment,purchase)' ),
			array( 'fab fa-ethereum' => 'Ethereum' ),
			array( 'fas fa-gem' => 'Gem (diamond,jewelry,sapphire,stone,treasure)' ),
			array( 'far fa-gem' => 'Gem (diamond,jewelry,sapphire,stone,treasure)' ),
			array( 'fas fa-gift' => 'gift (christmas,generosity,giving,holiday,party,present,wrapped,xmas)' ),
			array( 'fab fa-google-wallet' => 'Google Wallet' ),
			array( 'fas fa-handshake' => 'Handshake (agreement,greeting,meeting,partnership)' ),
			array( 'far fa-handshake' => 'Handshake (agreement,greeting,meeting,partnership)' ),
			array( 'fas fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'far fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'fas fa-key' => 'key (lock,password,private,secret,unlock)' ),
			array( 'fas fa-money-check' => 'Money Check (bank check,buy,checkout,cheque,money,payment,price,purchase)' ),
			array( 'fas fa-money-check-alt' => 'Alternate Money Check (bank check,buy,checkout,cheque,money,payment,price,purchase)' ),
			array( 'fab fa-paypal' => 'Paypal' ),
			array( 'fas fa-receipt' => 'Receipt (check,invoice,money,pay,table)' ),
			array( 'fas fa-shopping-bag' => 'Shopping Bag (buy,checkout,grocery,payment,purchase)' ),
			array( 'fas fa-shopping-basket' => 'Shopping Basket (buy,checkout,grocery,payment,purchase)' ),
			array( 'fas fa-shopping-cart' => 'shopping-cart (buy,checkout,grocery,payment,purchase)' ),
			array( 'fas fa-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'far fa-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'fab fa-stripe' => 'Stripe' ),
			array( 'fab fa-stripe-s' => 'Stripe S' ),
			array( 'fas fa-tag' => 'tag (discount,label,price,shopping)' ),
			array( 'fas fa-tags' => 'tags (discount,label,price,shopping)' ),
			array( 'fas fa-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'far fa-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'fas fa-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
			array( 'far fa-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
			array( 'fas fa-trophy' => 'trophy (achievement,award,cup,game,winner)' ),
		),
		'Pharmacy' => array(
			array( 'fas fa-band-aid' => 'Band-Aid (bandage,boo boo,first aid,ouch)' ),
			array( 'fas fa-book-medical' => 'Medical Book (diary,documentation,health,history,journal,library,read,record)' ),
			array( 'fas fa-cannabis' => 'Cannabis (bud,chronic,drugs,endica,endo,ganja,marijuana,mary jane,pot,reefer,sativa,spliff,weed,whacky-tabacky)' ),
			array( 'fas fa-capsules' => 'Capsules (drugs,medicine,pills,prescription)' ),
			array( 'fas fa-clinic-medical' => 'Medical Clinic (doctor,general practitioner,hospital,infirmary,medicine,office,outpatient)' ),
			array( 'fas fa-eye-dropper' => 'Eye Dropper (beaker,clone,color,copy,eyedropper,pipette)' ),
			array( 'fas fa-file-medical' => 'Medical File (document,health,history,prescription,record)' ),
			array( 'fas fa-file-prescription' => 'File Prescription (document,drugs,medical,medicine,rx)' ),
			array( 'fas fa-first-aid' => 'First Aid (emergency,emt,health,medical,rescue)' ),
			array( 'fas fa-flask' => 'Flask (beaker,experimental,labs,science)' ),
			array( 'fas fa-history' => 'History (Rewind,clock,reverse,time,time machine)' ),
			array( 'fas fa-joint' => 'Joint (blunt,cannabis,doobie,drugs,marijuana,roach,smoke,smoking,spliff)' ),
			array( 'fas fa-laptop-medical' => 'Laptop Medical (computer,device,ehr,electronic health records,history)' ),
			array( 'fas fa-mortar-pestle' => 'Mortar Pestle (crush,culinary,grind,medical,mix,pharmacy,prescription,spices)' ),
			array( 'fas fa-notes-medical' => 'Medical Notes (clipboard,doctor,ehr,health,history,records)' ),
			array( 'fas fa-pills' => 'Pills (drugs,medicine,prescription,tablets)' ),
			array( 'fas fa-prescription' => 'Prescription (drugs,medical,medicine,pharmacy,rx)' ),
			array( 'fas fa-prescription-bottle' => 'Prescription Bottle (drugs,medical,medicine,pharmacy,rx)' ),
			array( 'fas fa-prescription-bottle-alt' => 'Alternate Prescription Bottle (drugs,medical,medicine,pharmacy,rx)' ),
			array( 'fas fa-receipt' => 'Receipt (check,invoice,money,pay,table)' ),
			array( 'fas fa-skull-crossbones' => 'Skull & Crossbones (Dungeons & Dragons,alert,bones,d&d,danger,dead,deadly,death,dnd,fantasy,halloween,holiday,jolly-roger,pirate,poison,skeleton,warning)' ),
			array( 'fas fa-syringe' => 'Syringe (doctor,immunizations,medical,needle)' ),
			array( 'fas fa-tablets' => 'Tablets (drugs,medicine,pills,prescription)' ),
			array( 'fas fa-thermometer' => 'Thermometer (mercury,status,temperature)' ),
			array( 'fas fa-vial' => 'Vial (experiment,lab,sample,science,test,test tube)' ),
			array( 'fas fa-vials' => 'Vials (experiment,lab,sample,science,test,test tube)' ),
		),
		'Political' => array(
			array( 'fas fa-award' => 'Award (honor,praise,prize,recognition,ribbon,trophy)' ),
			array( 'fas fa-balance-scale' => 'Balance Scale (balanced,justice,legal,measure,weight)' ),
			array( 'fas fa-balance-scale-left' => 'Balance Scale (Left-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'fas fa-balance-scale-right' => 'Balance Scale (Right-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'fas fa-bullhorn' => 'bullhorn (announcement,broadcast,louder,megaphone,share)' ),
			array( 'fas fa-check-double' => 'Double Check (accept,agree,checkmark,confirm,correct,done,notice,notification,notify,ok,select,success,tick,todo)' ),
			array( 'fas fa-democrat' => 'Democrat (american,democratic party,donkey,election,left,left-wing,liberal,politics,usa)' ),
			array( 'fas fa-donate' => 'Donate (contribute,generosity,gift,give)' ),
			array( 'fas fa-dove' => 'Dove (bird,fauna,flying,peace,war)' ),
			array( 'fas fa-fist-raised' => 'Raised Fist (Dungeons & Dragons,d&d,dnd,fantasy,hand,ki,monk,resist,strength,unarmed combat)' ),
			array( 'fas fa-flag-usa' => 'United States of America Flag (betsy ross,country,old glory,stars,stripes,symbol)' ),
			array( 'fas fa-handshake' => 'Handshake (agreement,greeting,meeting,partnership)' ),
			array( 'far fa-handshake' => 'Handshake (agreement,greeting,meeting,partnership)' ),
			array( 'fas fa-person-booth' => 'Person Entering Booth (changing,changing room,election,human,person,vote,voting)' ),
			array( 'fas fa-piggy-bank' => 'Piggy Bank (bank,save,savings)' ),
			array( 'fas fa-republican' => 'Republican (american,conservative,election,elephant,politics,republican party,right,right-wing,usa)' ),
			array( 'fas fa-vote-yea' => 'Vote Yea (accept,cast,election,politics,positive,yes)' ),
		),
		'Religion' => array(
			array( 'fas fa-ankh' => 'Ankh (amulet,copper,coptic christianity,copts,crux ansata,egypt,venus)' ),
			array( 'fas fa-atom' => 'Atom (atheism,chemistry,ion,nuclear,science)' ),
			array( 'fas fa-bible' => 'Bible (book,catholicism,christianity,god,holy)' ),
			array( 'fas fa-church' => 'Church (building,cathedral,chapel,community,religion)' ),
			array( 'fas fa-cross' => 'Cross (catholicism,christianity,church,jesus)' ),
			array( 'fas fa-dharmachakra' => 'Dharmachakra (buddhism,buddhist,wheel of dharma)' ),
			array( 'fas fa-dove' => 'Dove (bird,fauna,flying,peace,war)' ),
			array( 'fas fa-gopuram' => 'Gopuram (building,entrance,hinduism,temple,tower)' ),
			array( 'fas fa-hamsa' => 'Hamsa (amulet,christianity,islam,jewish,judaism,muslim,protection)' ),
			array( 'fas fa-hanukiah' => 'Hanukiah (candle,hanukkah,jewish,judaism,light)' ),
			array( 'fas fa-haykal' => 'Haykal (bahai,bahá\'í,star)' ),
			array( 'fas fa-jedi' => 'Jedi (crest,force,sith,skywalker,star wars,yoda)' ),
			array( 'fas fa-journal-whills' => 'Journal of the Whills (book,force,jedi,sith,star wars,yoda)' ),
			array( 'fas fa-kaaba' => 'Kaaba (building,cube,islam,muslim)' ),
			array( 'fas fa-khanda' => 'Khanda (chakkar,sikh,sikhism,sword)' ),
			array( 'fas fa-menorah' => 'Menorah (candle,hanukkah,jewish,judaism,light)' ),
			array( 'fas fa-mosque' => 'Mosque (building,islam,landmark,muslim)' ),
			array( 'fas fa-om' => 'Om (buddhism,hinduism,jainism,mantra)' ),
			array( 'fas fa-pastafarianism' => 'Pastafarianism (agnosticism,atheism,flying spaghetti monster,fsm)' ),
			array( 'fas fa-peace' => 'Peace (serenity,tranquility,truce,war)' ),
			array( 'fas fa-place-of-worship' => 'Place of Worship (building,church,holy,mosque,synagogue)' ),
			array( 'fas fa-pray' => 'Pray (kneel,preach,religion,worship)' ),
			array( 'fas fa-praying-hands' => 'Praying Hands (kneel,preach,religion,worship)' ),
			array( 'fas fa-quran' => 'Quran (book,islam,muslim,religion)' ),
			array( 'fas fa-star-and-crescent' => 'Star and Crescent (islam,muslim,religion)' ),
			array( 'fas fa-star-of-david' => 'Star of David (jewish,judaism,religion)' ),
			array( 'fas fa-synagogue' => 'Synagogue (building,jewish,judaism,religion,star of david,temple)' ),
			array( 'fas fa-torah' => 'Torah (book,jewish,judaism,religion,scroll)' ),
			array( 'fas fa-torii-gate' => 'Torii Gate (building,shintoism)' ),
			array( 'fas fa-vihara' => 'Vihara (buddhism,buddhist,building,monastery)' ),
			array( 'fas fa-yin-yang' => 'Yin Yang (daoism,opposites,taoism)' ),
		),
		'Science' => array(
			array( 'fas fa-atom' => 'Atom (atheism,chemistry,ion,nuclear,science)' ),
			array( 'fas fa-biohazard' => 'Biohazard (danger,dangerous,hazmat,medical,radioactive,toxic,waste,zombie)' ),
			array( 'fas fa-brain' => 'Brain (cerebellum,gray matter,intellect,medulla oblongata,mind,noodle,wit)' ),
			array( 'fas fa-burn' => 'Burn (caliente,energy,fire,flame,gas,heat,hot)' ),
			array( 'fas fa-capsules' => 'Capsules (drugs,medicine,pills,prescription)' ),
			array( 'fas fa-clipboard-check' => 'Clipboard with Check (accept,agree,confirm,done,ok,select,success,tick,todo,yes)' ),
			array( 'fas fa-dna' => 'DNA (double helix,genetic,helix,molecule,protein)' ),
			array( 'fas fa-eye-dropper' => 'Eye Dropper (beaker,clone,color,copy,eyedropper,pipette)' ),
			array( 'fas fa-filter' => 'Filter (funnel,options,separate,sort)' ),
			array( 'fas fa-fire' => 'fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'fas fa-fire-alt' => 'Alternate Fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'fas fa-flask' => 'Flask (beaker,experimental,labs,science)' ),
			array( 'fas fa-frog' => 'Frog (amphibian,bullfrog,fauna,hop,kermit,kiss,prince,ribbit,toad,wart)' ),
			array( 'fas fa-magnet' => 'magnet (Attract,lodestone,tool)' ),
			array( 'fas fa-microscope' => 'Microscope (electron,lens,optics,science,shrink)' ),
			array( 'fas fa-mortar-pestle' => 'Mortar Pestle (crush,culinary,grind,medical,mix,pharmacy,prescription,spices)' ),
			array( 'fas fa-pills' => 'Pills (drugs,medicine,prescription,tablets)' ),
			array( 'fas fa-prescription-bottle' => 'Prescription Bottle (drugs,medical,medicine,pharmacy,rx)' ),
			array( 'fas fa-radiation' => 'Radiation (danger,dangerous,deadly,hazard,nuclear,radioactive,warning)' ),
			array( 'fas fa-radiation-alt' => 'Alternate Radiation (danger,dangerous,deadly,hazard,nuclear,radioactive,warning)' ),
			array( 'fas fa-seedling' => 'Seedling (flora,grow,plant,vegan)' ),
			array( 'fas fa-skull-crossbones' => 'Skull & Crossbones (Dungeons & Dragons,alert,bones,d&d,danger,dead,deadly,death,dnd,fantasy,halloween,holiday,jolly-roger,pirate,poison,skeleton,warning)' ),
			array( 'fas fa-syringe' => 'Syringe (doctor,immunizations,medical,needle)' ),
			array( 'fas fa-tablets' => 'Tablets (drugs,medicine,pills,prescription)' ),
			array( 'fas fa-temperature-high' => 'High Temperature (cook,mercury,summer,thermometer,warm)' ),
			array( 'fas fa-temperature-low' => 'Low Temperature (cold,cool,mercury,thermometer,winter)' ),
			array( 'fas fa-vial' => 'Vial (experiment,lab,sample,science,test,test tube)' ),
			array( 'fas fa-vials' => 'Vials (experiment,lab,sample,science,test,test tube)' ),
		),
		'Science Fiction' => array(
			array( 'fab fa-galactic-republic' => 'Galactic Republic (politics,star wars)' ),
			array( 'fab fa-galactic-senate' => 'Galactic Senate (star wars)' ),
			array( 'fas fa-globe' => 'Globe (all,coordinates,country,earth,global,gps,language,localize,location,map,online,place,planet,translate,travel,world)' ),
			array( 'fas fa-jedi' => 'Jedi (crest,force,sith,skywalker,star wars,yoda)' ),
			array( 'fab fa-jedi-order' => 'Jedi Order (star wars)' ),
			array( 'fas fa-journal-whills' => 'Journal of the Whills (book,force,jedi,sith,star wars,yoda)' ),
			array( 'fas fa-meteor' => 'Meteor (armageddon,asteroid,comet,shooting star,space)' ),
			array( 'fas fa-moon' => 'Moon (contrast,crescent,dark,lunar,night)' ),
			array( 'far fa-moon' => 'Moon (contrast,crescent,dark,lunar,night)' ),
			array( 'fab fa-old-republic' => 'Old Republic (politics,star wars)' ),
			array( 'fas fa-robot' => 'Robot (android,automate,computer,cyborg)' ),
			array( 'fas fa-rocket' => 'rocket (aircraft,app,jet,launch,nasa,space)' ),
			array( 'fas fa-satellite' => 'Satellite (communications,hardware,orbit,space)' ),
			array( 'fas fa-satellite-dish' => 'Satellite Dish (SETI,communications,hardware,receiver,saucer,signal)' ),
			array( 'fas fa-space-shuttle' => 'Space Shuttle (astronaut,machine,nasa,rocket,transportation)' ),
			array( 'fas fa-user-astronaut' => 'User Astronaut (avatar,clothing,cosmonaut,nasa,space,suit)' ),
		),
		'Security' => array(
			array( 'fas fa-ban' => 'ban (abort,ban,block,cancel,delete,hide,prohibit,remove,stop,trash)' ),
			array( 'fas fa-bug' => 'Bug (beetle,error,insect,report)' ),
			array( 'fas fa-door-closed' => 'Door Closed (enter,exit,locked)' ),
			array( 'fas fa-door-open' => 'Door Open (enter,exit,welcome)' ),
			array( 'fas fa-dungeon' => 'Dungeon (Dungeons & Dragons,building,d&d,dnd,door,entrance,fantasy,gate)' ),
			array( 'fas fa-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'far fa-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'fas fa-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'far fa-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'fas fa-file-contract' => 'File Contract (agreement,binding,document,legal,signature)' ),
			array( 'fas fa-file-signature' => 'File Signature (John Hancock,contract,document,name)' ),
			array( 'fas fa-fingerprint' => 'Fingerprint (human,id,identification,lock,smudge,touch,unique,unlock)' ),
			array( 'fas fa-id-badge' => 'Identification Badge (address,contact,identification,license,profile)' ),
			array( 'far fa-id-badge' => 'Identification Badge (address,contact,identification,license,profile)' ),
			array( 'fas fa-id-card' => 'Identification Card (contact,demographics,document,identification,issued,profile)' ),
			array( 'far fa-id-card' => 'Identification Card (contact,demographics,document,identification,issued,profile)' ),
			array( 'fas fa-id-card-alt' => 'Alternate Identification Card (contact,demographics,document,identification,issued,profile)' ),
			array( 'fas fa-key' => 'key (lock,password,private,secret,unlock)' ),
			array( 'fas fa-lock' => 'lock (admin,lock,open,password,private,protect,security)' ),
			array( 'fas fa-lock-open' => 'Lock Open (admin,lock,open,password,private,protect,security)' ),
			array( 'fas fa-mask' => 'Mask (carnivale,costume,disguise,halloween,secret,super hero)' ),
			array( 'fas fa-passport' => 'Passport (document,id,identification,issued,travel)' ),
			array( 'fas fa-shield-alt' => 'Alternate Shield (achievement,award,block,defend,security,winner)' ),
			array( 'fas fa-unlock' => 'unlock (admin,lock,password,private,protect)' ),
			array( 'fas fa-unlock-alt' => 'Alternate Unlock (admin,lock,password,private,protect)' ),
			array( 'fas fa-user-lock' => 'User Lock (admin,lock,person,private,unlock)' ),
			array( 'fas fa-user-secret' => 'User Secret (clothing,coat,hat,incognito,person,privacy,spy,whisper)' ),
			array( 'fas fa-user-shield' => 'User Shield (admin,person,private,protect,safe)' ),
		),
		'Shapes' => array(
			array( 'fas fa-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'far fa-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'fas fa-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'far fa-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'fas fa-certificate' => 'certificate (badge,star,verified)' ),
			array( 'fas fa-circle' => 'Circle (circle-thin,diameter,dot,ellipse,notification,round)' ),
			array( 'far fa-circle' => 'Circle (circle-thin,diameter,dot,ellipse,notification,round)' ),
			array( 'fas fa-cloud' => 'Cloud (atmosphere,fog,overcast,save,upload,weather)' ),
			array( 'fas fa-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'far fa-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'fas fa-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'far fa-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'fas fa-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'far fa-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'fas fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'far fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'fas fa-heart-broken' => 'Heart Broken (breakup,crushed,dislike,dumped,grief,love,lovesick,relationship,sad)' ),
			array( 'fas fa-map-marker' => 'map-marker (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'fas fa-play' => 'play (audio,music,playing,sound,start,video)' ),
			array( 'fas fa-shapes' => 'Shapes (blocks,build,circle,square,triangle)' ),
			array( 'fas fa-square' => 'Square (block,box,shape)' ),
			array( 'far fa-square' => 'Square (block,box,shape)' ),
			array( 'fas fa-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'far fa-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
		),
		'Shopping' => array(
			array( 'fas fa-barcode' => 'barcode (info,laser,price,scan,upc)' ),
			array( 'fas fa-cart-arrow-down' => 'Shopping Cart Arrow Down (download,save,shopping)' ),
			array( 'fas fa-cart-plus' => 'Add to Shopping Cart (add,create,new,positive,shopping)' ),
			array( 'fas fa-cash-register' => 'Cash Register (buy,cha-ching,change,checkout,commerce,leaerboard,machine,pay,payment,purchase,store)' ),
			array( 'fas fa-gift' => 'gift (christmas,generosity,giving,holiday,party,present,wrapped,xmas)' ),
			array( 'fas fa-gifts' => 'Gifts (christmas,generosity,giving,holiday,party,present,wrapped,xmas)' ),
			array( 'fas fa-person-booth' => 'Person Entering Booth (changing,changing room,election,human,person,vote,voting)' ),
			array( 'fas fa-receipt' => 'Receipt (check,invoice,money,pay,table)' ),
			array( 'fas fa-shipping-fast' => 'Shipping Fast (express,fedex,mail,overnight,package,ups)' ),
			array( 'fas fa-shopping-bag' => 'Shopping Bag (buy,checkout,grocery,payment,purchase)' ),
			array( 'fas fa-shopping-basket' => 'Shopping Basket (buy,checkout,grocery,payment,purchase)' ),
			array( 'fas fa-shopping-cart' => 'shopping-cart (buy,checkout,grocery,payment,purchase)' ),
			array( 'fas fa-store' => 'Store (building,buy,purchase,shopping)' ),
			array( 'fas fa-store-alt' => 'Alternate Store (building,buy,purchase,shopping)' ),
			array( 'fas fa-truck' => 'truck (cargo,delivery,shipping,vehicle)' ),
			array( 'fas fa-tshirt' => 'T-Shirt (clothing,fashion,garment,shirt)' ),
		),
		'Social' => array(
			array( 'fas fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'far fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'fas fa-birthday-cake' => 'Birthday Cake (anniversary,bakery,candles,celebration,dessert,frosting,holiday,party,pastry)' ),
			array( 'fas fa-camera' => 'camera (image,lens,photo,picture,record,shutter,video)' ),
			array( 'fas fa-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'far fa-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'fas fa-comment-alt' => 'Alternate Comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'far fa-comment-alt' => 'Alternate Comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'fas fa-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'far fa-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'fas fa-hashtag' => 'Hashtag (Twitter,instagram,pound,social media,tag)' ),
			array( 'fas fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'far fa-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'fas fa-icons' => 'Icons (bolt,emoji,heart,image,music,photo,symbols)' ),
			array( 'fas fa-image' => 'Image (album,landscape,photo,picture)' ),
			array( 'far fa-image' => 'Image (album,landscape,photo,picture)' ),
			array( 'fas fa-images' => 'Images (album,landscape,photo,picture)' ),
			array( 'far fa-images' => 'Images (album,landscape,photo,picture)' ),
			array( 'fas fa-map-marker' => 'map-marker (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'fas fa-map-marker-alt' => 'Alternate Map Marker (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'fas fa-photo-video' => 'Photo Video (av,film,image,library,media)' ),
			array( 'fas fa-poll' => 'Poll (results,survey,trend,vote,voting)' ),
			array( 'fas fa-poll-h' => 'Poll H (results,survey,trend,vote,voting)' ),
			array( 'fas fa-retweet' => 'Retweet (refresh,reload,share,swap)' ),
			array( 'fas fa-share' => 'Share (forward,save,send,social)' ),
			array( 'fas fa-share-alt' => 'Alternate Share (forward,save,send,social)' ),
			array( 'fas fa-share-square' => 'Share Square (forward,save,send,social)' ),
			array( 'far fa-share-square' => 'Share Square (forward,save,send,social)' ),
			array( 'fas fa-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'far fa-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'fas fa-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'far fa-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'fas fa-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
			array( 'far fa-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
			array( 'fas fa-thumbtack' => 'Thumbtack (coordinates,location,marker,pin,thumb-tack)' ),
			array( 'fas fa-user' => 'User (account,avatar,head,human,man,person,profile)' ),
			array( 'far fa-user' => 'User (account,avatar,head,human,man,person,profile)' ),
			array( 'fas fa-user-circle' => 'User Circle (account,avatar,head,human,man,person,profile)' ),
			array( 'far fa-user-circle' => 'User Circle (account,avatar,head,human,man,person,profile)' ),
			array( 'fas fa-user-friends' => 'User Friends (group,people,person,team,users)' ),
			array( 'fas fa-user-plus' => 'User Plus (add,avatar,positive,sign up,signup,team)' ),
			array( 'fas fa-users' => 'Users (friends,group,people,persons,profiles,team)' ),
			array( 'fas fa-video' => 'Video (camera,film,movie,record,video-camera)' ),
		),
		'Spinners' => array(
			array( 'fas fa-asterisk' => 'asterisk (annotation,details,reference,star)' ),
			array( 'fas fa-atom' => 'Atom (atheism,chemistry,ion,nuclear,science)' ),
			array( 'fas fa-certificate' => 'certificate (badge,star,verified)' ),
			array( 'fas fa-circle-notch' => 'Circle Notched (circle-o-notch,diameter,dot,ellipse,round,spinner)' ),
			array( 'fas fa-cog' => 'cog (gear,mechanical,settings,sprocket,wheel)' ),
			array( 'fas fa-compact-disc' => 'Compact Disc (album,bluray,cd,disc,dvd,media,movie,music,record,video,vinyl)' ),
			array( 'fas fa-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'far fa-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'fas fa-crosshairs' => 'Crosshairs (aim,bullseye,gpd,picker,position)' ),
			array( 'fas fa-dharmachakra' => 'Dharmachakra (buddhism,buddhist,wheel of dharma)' ),
			array( 'fas fa-fan' => 'Fan (ac,air conditioning,blade,blower,cool,hot)' ),
			array( 'fas fa-haykal' => 'Haykal (bahai,bahá\'í,star)' ),
			array( 'fas fa-life-ring' => 'Life Ring (coast guard,help,overboard,save,support)' ),
			array( 'far fa-life-ring' => 'Life Ring (coast guard,help,overboard,save,support)' ),
			array( 'fas fa-palette' => 'Palette (acrylic,art,brush,color,fill,paint,pigment,watercolor)' ),
			array( 'fas fa-ring' => 'Ring (Dungeons & Dragons,Gollum,band,binding,d&d,dnd,engagement,fantasy,gold,jewelry,marriage,precious)' ),
			array( 'fas fa-slash' => 'Slash (cancel,close,mute,off,stop,x)' ),
			array( 'fas fa-snowflake' => 'Snowflake (precipitation,rain,winter)' ),
			array( 'far fa-snowflake' => 'Snowflake (precipitation,rain,winter)' ),
			array( 'fas fa-spinner' => 'Spinner (circle,loading,progress)' ),
			array( 'fas fa-stroopwafel' => 'Stroopwafel (caramel,cookie,dessert,sweets,waffle)' ),
			array( 'fas fa-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'far fa-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'fas fa-sync' => 'Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'fas fa-sync-alt' => 'Alternate Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'fas fa-yin-yang' => 'Yin Yang (daoism,opposites,taoism)' ),
		),
		'Sports' => array(
			array( 'fas fa-baseball-ball' => 'Baseball Ball (foul,hardball,league,leather,mlb,softball,sport)' ),
			array( 'fas fa-basketball-ball' => 'Basketball Ball (dribble,dunk,hoop,nba)' ),
			array( 'fas fa-biking' => 'Biking (bicycle,bike,cycle,cycling,ride,wheel)' ),
			array( 'fas fa-bowling-ball' => 'Bowling Ball (alley,candlepin,gutter,lane,strike,tenpin)' ),
			array( 'fas fa-dumbbell' => 'Dumbbell (exercise,gym,strength,weight,weight-lifting)' ),
			array( 'fas fa-football-ball' => 'Football Ball (ball,fall,nfl,pigskin,seasonal)' ),
			array( 'fas fa-futbol' => 'Futbol (ball,football,mls,soccer)' ),
			array( 'far fa-futbol' => 'Futbol (ball,football,mls,soccer)' ),
			array( 'fas fa-golf-ball' => 'Golf Ball (caddy,eagle,putt,tee)' ),
			array( 'fas fa-hockey-puck' => 'Hockey Puck (ice,nhl,sport)' ),
			array( 'fas fa-quidditch' => 'Quidditch (ball,bludger,broom,golden snitch,harry potter,hogwarts,quaffle,sport,wizard)' ),
			array( 'fas fa-running' => 'Running (exercise,health,jog,person,run,sport,sprint)' ),
			array( 'fas fa-skating' => 'Skating (activity,figure skating,fitness,ice,person,winter)' ),
			array( 'fas fa-skiing' => 'Skiing (activity,downhill,fast,fitness,olympics,outdoors,person,seasonal,slalom)' ),
			array( 'fas fa-skiing-nordic' => 'Skiing Nordic (activity,cross country,fitness,outdoors,person,seasonal)' ),
			array( 'fas fa-snowboarding' => 'Snowboarding (activity,fitness,olympics,outdoors,person)' ),
			array( 'fas fa-swimmer' => 'Swimmer (athlete,head,man,olympics,person,pool,water)' ),
			array( 'fas fa-table-tennis' => 'Table Tennis (ball,paddle,ping pong)' ),
			array( 'fas fa-volleyball-ball' => 'Volleyball Ball (beach,olympics,sport)' ),
		),
		'Spring' => array(
			array( 'fas fa-allergies' => 'Allergies (allergy,freckles,hand,hives,pox,skin,spots)' ),
			array( 'fas fa-broom' => 'Broom (clean,firebolt,fly,halloween,nimbus 2000,quidditch,sweep,witch)' ),
			array( 'fas fa-cloud-sun' => 'Cloud with Sun (clear,day,daytime,fall,outdoors,overcast,partly cloudy)' ),
			array( 'fas fa-cloud-sun-rain' => 'Cloud with Sun and Rain (day,overcast,precipitation,storm,summer,sunshower)' ),
			array( 'fas fa-frog' => 'Frog (amphibian,bullfrog,fauna,hop,kermit,kiss,prince,ribbit,toad,wart)' ),
			array( 'fas fa-rainbow' => 'Rainbow (gold,leprechaun,prism,rain,sky)' ),
			array( 'fas fa-seedling' => 'Seedling (flora,grow,plant,vegan)' ),
			array( 'fas fa-umbrella' => 'Umbrella (protection,rain,storm,wet)' ),
		),
		'Status' => array(
			array( 'fas fa-ban' => 'ban (abort,ban,block,cancel,delete,hide,prohibit,remove,stop,trash)' ),
			array( 'fas fa-battery-empty' => 'Battery Empty (charge,dead,power,status)' ),
			array( 'fas fa-battery-full' => 'Battery Full (charge,power,status)' ),
			array( 'fas fa-battery-half' => 'Battery 1/2 Full (charge,power,status)' ),
			array( 'fas fa-battery-quarter' => 'Battery 1/4 Full (charge,low,power,status)' ),
			array( 'fas fa-battery-three-quarters' => 'Battery 3/4 Full (charge,power,status)' ),
			array( 'fas fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'far fa-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'fas fa-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'far fa-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'fas fa-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'far fa-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'fas fa-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'far fa-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'fas fa-calendar-check' => 'Calendar Check (accept,agree,appointment,confirm,correct,date,done,event,ok,schedule,select,success,tick,time,todo,when)' ),
			array( 'far fa-calendar-check' => 'Calendar Check (accept,agree,appointment,confirm,correct,date,done,event,ok,schedule,select,success,tick,time,todo,when)' ),
			array( 'fas fa-calendar-day' => 'Calendar with Day Focus (date,detail,event,focus,schedule,single day,time,today,when)' ),
			array( 'fas fa-calendar-minus' => 'Calendar Minus (calendar,date,delete,event,negative,remove,schedule,time,when)' ),
			array( 'far fa-calendar-minus' => 'Calendar Minus (calendar,date,delete,event,negative,remove,schedule,time,when)' ),
			array( 'fas fa-calendar-plus' => 'Calendar Plus (add,calendar,create,date,event,new,positive,schedule,time,when)' ),
			array( 'far fa-calendar-plus' => 'Calendar Plus (add,calendar,create,date,event,new,positive,schedule,time,when)' ),
			array( 'fas fa-calendar-times' => 'Calendar Times (archive,calendar,date,delete,event,remove,schedule,time,when,x)' ),
			array( 'far fa-calendar-times' => 'Calendar Times (archive,calendar,date,delete,event,remove,schedule,time,when,x)' ),
			array( 'fas fa-calendar-week' => 'Calendar with Week Focus (date,detail,event,focus,schedule,single week,time,today,when)' ),
			array( 'fas fa-cart-arrow-down' => 'Shopping Cart Arrow Down (download,save,shopping)' ),
			array( 'fas fa-cart-plus' => 'Add to Shopping Cart (add,create,new,positive,shopping)' ),
			array( 'fas fa-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'far fa-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'fas fa-comment-alt' => 'Alternate Comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'far fa-comment-alt' => 'Alternate Comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'fas fa-comment-slash' => 'Comment Slash (bubble,cancel,chat,commenting,conversation,feedback,message,mute,note,notification,quiet,sms,speech,texting)' ),
			array( 'fas fa-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'far fa-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'fas fa-door-closed' => 'Door Closed (enter,exit,locked)' ),
			array( 'fas fa-door-open' => 'Door Open (enter,exit,welcome)' ),
			array( 'fas fa-exclamation' => 'exclamation (alert,danger,error,important,notice,notification,notify,problem,warning)' ),
			array( 'fas fa-exclamation-circle' => 'Exclamation Circle (alert,danger,error,important,notice,notification,notify,problem,warning)' ),
			array( 'fas fa-exclamation-triangle' => 'Exclamation Triangle (alert,danger,error,important,notice,notification,notify,problem,warning)' ),
			array( 'fas fa-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'far fa-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'fas fa-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'far fa-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'fas fa-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'far fa-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'fas fa-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'far fa-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'fas fa-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'far fa-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'fas fa-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'far fa-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'fas fa-gas-pump' => 'Gas Pump (car,fuel,gasoline,petrol)' ),
			array( 'fas fa-info' => 'Info (details,help,information,more,support)' ),
			array( 'fas fa-info-circle' => 'Info Circle (details,help,information,more,support)' ),
			array( 'fas fa-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'far fa-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'fas fa-lock' => 'lock (admin,lock,open,password,private,protect,security)' ),
			array( 'fas fa-lock-open' => 'Lock Open (admin,lock,open,password,private,protect,security)' ),
			array( 'fas fa-map-marker' => 'map-marker (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'fas fa-map-marker-alt' => 'Alternate Map Marker (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'fas fa-microphone' => 'microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-microphone-alt' => 'Alternate Microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-microphone-alt-slash' => 'Alternate Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-microphone-slash' => 'Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-minus' => 'minus (collapse,delete,hide,minify,negative,remove,trash)' ),
			array( 'fas fa-minus-circle' => 'Minus Circle (delete,hide,negative,remove,shape,trash)' ),
			array( 'fas fa-minus-square' => 'Minus Square (collapse,delete,hide,minify,negative,remove,shape,trash)' ),
			array( 'far fa-minus-square' => 'Minus Square (collapse,delete,hide,minify,negative,remove,shape,trash)' ),
			array( 'fas fa-parking' => 'Parking (auto,car,garage,meter)' ),
			array( 'fas fa-phone' => 'Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'fas fa-phone-alt' => 'Alternate Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'fas fa-phone-slash' => 'Phone Slash (call,cancel,earphone,mute,number,support,telephone,voice)' ),
			array( 'fas fa-plus' => 'plus (add,create,expand,new,positive,shape)' ),
			array( 'fas fa-plus-circle' => 'Plus Circle (add,create,expand,new,positive,shape)' ),
			array( 'fas fa-plus-square' => 'Plus Square (add,create,expand,new,positive,shape)' ),
			array( 'far fa-plus-square' => 'Plus Square (add,create,expand,new,positive,shape)' ),
			array( 'fas fa-print' => 'print (business,copy,document,office,paper)' ),
			array( 'fas fa-question' => 'Question (help,information,support,unknown)' ),
			array( 'fas fa-question-circle' => 'Question Circle (help,information,support,unknown)' ),
			array( 'far fa-question-circle' => 'Question Circle (help,information,support,unknown)' ),
			array( 'fas fa-shield-alt' => 'Alternate Shield (achievement,award,block,defend,security,winner)' ),
			array( 'fas fa-shopping-cart' => 'shopping-cart (buy,checkout,grocery,payment,purchase)' ),
			array( 'fas fa-sign-in-alt' => 'Alternate Sign In (arrow,enter,join,log in,login,sign in,sign up,sign-in,signin,signup)' ),
			array( 'fas fa-sign-out-alt' => 'Alternate Sign Out (arrow,exit,leave,log out,logout,sign-out)' ),
			array( 'fas fa-signal' => 'signal (bars,graph,online,reception,status)' ),
			array( 'fas fa-smoking-ban' => 'Smoking Ban (ban,cancel,no smoking,non-smoking)' ),
			array( 'fas fa-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'far fa-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'fas fa-star-half' => 'star-half (achievement,award,rating,score,star-half-empty,star-half-full)' ),
			array( 'far fa-star-half' => 'star-half (achievement,award,rating,score,star-half-empty,star-half-full)' ),
			array( 'fas fa-star-half-alt' => 'Alternate Star Half (achievement,award,rating,score,star-half-empty,star-half-full)' ),
			array( 'fas fa-stream' => 'Stream (flow,list,timeline)' ),
			array( 'fas fa-thermometer-empty' => 'Thermometer Empty (cold,mercury,status,temperature)' ),
			array( 'fas fa-thermometer-full' => 'Thermometer Full (fever,hot,mercury,status,temperature)' ),
			array( 'fas fa-thermometer-half' => 'Thermometer 1/2 Full (mercury,status,temperature)' ),
			array( 'fas fa-thermometer-quarter' => 'Thermometer 1/4 Full (mercury,status,temperature)' ),
			array( 'fas fa-thermometer-three-quarters' => 'Thermometer 3/4 Full (mercury,status,temperature)' ),
			array( 'fas fa-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'far fa-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'fas fa-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
			array( 'far fa-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
			array( 'fas fa-tint' => 'tint (color,drop,droplet,raindrop,waterdrop)' ),
			array( 'fas fa-tint-slash' => 'Tint Slash (color,drop,droplet,raindrop,waterdrop)' ),
			array( 'fas fa-toggle-off' => 'Toggle Off (switch)' ),
			array( 'fas fa-toggle-on' => 'Toggle On (switch)' ),
			array( 'fas fa-unlock' => 'unlock (admin,lock,password,private,protect)' ),
			array( 'fas fa-unlock-alt' => 'Alternate Unlock (admin,lock,password,private,protect)' ),
			array( 'fas fa-user' => 'User (account,avatar,head,human,man,person,profile)' ),
			array( 'far fa-user' => 'User (account,avatar,head,human,man,person,profile)' ),
			array( 'fas fa-user-alt' => 'Alternate User (account,avatar,head,human,man,person,profile)' ),
			array( 'fas fa-user-alt-slash' => 'Alternate User Slash (account,avatar,head,human,man,person,profile)' ),
			array( 'fas fa-user-slash' => 'User Slash (ban,delete,remove)' ),
			array( 'fas fa-video' => 'Video (camera,film,movie,record,video-camera)' ),
			array( 'fas fa-video-slash' => 'Video Slash (add,create,film,new,positive,record,video)' ),
			array( 'fas fa-volume-down' => 'Volume Down (audio,lower,music,quieter,sound,speaker)' ),
			array( 'fas fa-volume-mute' => 'Volume Mute (audio,music,quiet,sound,speaker)' ),
			array( 'fas fa-volume-off' => 'Volume Off (audio,ban,music,mute,quiet,silent,sound)' ),
			array( 'fas fa-volume-up' => 'Volume Up (audio,higher,louder,music,sound,speaker)' ),
			array( 'fas fa-wifi' => 'WiFi (connection,hotspot,internet,network,wireless)' ),
		),
		'Summer' => array(
			array( 'fas fa-anchor' => 'Anchor (berth,boat,dock,embed,link,maritime,moor,secure)' ),
			array( 'fas fa-biking' => 'Biking (bicycle,bike,cycle,cycling,ride,wheel)' ),
			array( 'fas fa-fish' => 'Fish (fauna,gold,seafood,swimming)' ),
			array( 'fas fa-hotdog' => 'Hot Dog (bun,chili,frankfurt,frankfurter,kosher,polish,sandwich,sausage,vienna,weiner)' ),
			array( 'fas fa-ice-cream' => 'Ice Cream (chocolate,cone,dessert,frozen,scoop,sorbet,vanilla,yogurt)' ),
			array( 'fas fa-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'far fa-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'fas fa-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'far fa-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'fas fa-swimmer' => 'Swimmer (athlete,head,man,olympics,person,pool,water)' ),
			array( 'fas fa-swimming-pool' => 'Swimming Pool (ladder,recreation,swim,water)' ),
			array( 'fas fa-umbrella-beach' => 'Umbrella Beach (protection,recreation,sand,shade,summer,sun)' ),
			array( 'fas fa-volleyball-ball' => 'Volleyball Ball (beach,olympics,sport)' ),
			array( 'fas fa-water' => 'Water (lake,liquid,ocean,sea,swim,wet)' ),
		),
		'Tabletop Gaming' => array(
			array( 'fab fa-acquisitions-incorporated' => 'Acquisitions Incorporated (Dungeons & Dragons,d&d,dnd,fantasy,game,gaming,tabletop)' ),
			array( 'fas fa-book-dead' => 'Book of the Dead (Dungeons & Dragons,crossbones,d&d,dark arts,death,dnd,documentation,evil,fantasy,halloween,holiday,necronomicon,read,skull,spell)' ),
			array( 'fab fa-critical-role' => 'Critical Role (Dungeons & Dragons,d&d,dnd,fantasy,game,gaming,tabletop)' ),
			array( 'fab fa-d-and-d' => 'Dungeons & Dragons' ),
			array( 'fab fa-d-and-d-beyond' => 'D&D Beyond (Dungeons & Dragons,d&d,dnd,fantasy,gaming,tabletop)' ),
			array( 'fas fa-dice-d20' => 'Dice D20 (Dungeons & Dragons,chance,d&d,dnd,fantasy,gambling,game,roll)' ),
			array( 'fas fa-dice-d6' => 'Dice D6 (Dungeons & Dragons,chance,d&d,dnd,fantasy,gambling,game,roll)' ),
			array( 'fas fa-dragon' => 'Dragon (Dungeons & Dragons,d&d,dnd,fantasy,fire,lizard,serpent)' ),
			array( 'fas fa-dungeon' => 'Dungeon (Dungeons & Dragons,building,d&d,dnd,door,entrance,fantasy,gate)' ),
			array( 'fab fa-fantasy-flight-games' => 'Fantasy Flight-games (Dungeons & Dragons,d&d,dnd,fantasy,game,gaming,tabletop)' ),
			array( 'fas fa-fist-raised' => 'Raised Fist (Dungeons & Dragons,d&d,dnd,fantasy,hand,ki,monk,resist,strength,unarmed combat)' ),
			array( 'fas fa-hat-wizard' => 'Wizard\'s Hat (Dungeons & Dragons,accessory,buckle,clothing,d&d,dnd,fantasy,halloween,head,holiday,mage,magic,pointy,witch)' ),
			array( 'fab fa-penny-arcade' => 'Penny Arcade (Dungeons & Dragons,d&d,dnd,fantasy,game,gaming,pax,tabletop)' ),
			array( 'fas fa-ring' => 'Ring (Dungeons & Dragons,Gollum,band,binding,d&d,dnd,engagement,fantasy,gold,jewelry,marriage,precious)' ),
			array( 'fas fa-scroll' => 'Scroll (Dungeons & Dragons,announcement,d&d,dnd,fantasy,paper,script)' ),
			array( 'fas fa-skull-crossbones' => 'Skull & Crossbones (Dungeons & Dragons,alert,bones,d&d,danger,dead,deadly,death,dnd,fantasy,halloween,holiday,jolly-roger,pirate,poison,skeleton,warning)' ),
			array( 'fab fa-wizards-of-the-coast' => 'Wizards of the Coast (Dungeons & Dragons,d&d,dnd,fantasy,game,gaming,tabletop)' ),
		),
		'Toggle' => array(
			array( 'fas fa-bullseye' => 'Bullseye (archery,goal,objective,target)' ),
			array( 'fas fa-check-circle' => 'Check Circle (accept,agree,confirm,correct,done,ok,select,success,tick,todo,yes)' ),
			array( 'far fa-check-circle' => 'Check Circle (accept,agree,confirm,correct,done,ok,select,success,tick,todo,yes)' ),
			array( 'fas fa-circle' => 'Circle (circle-thin,diameter,dot,ellipse,notification,round)' ),
			array( 'far fa-circle' => 'Circle (circle-thin,diameter,dot,ellipse,notification,round)' ),
			array( 'fas fa-dot-circle' => 'Dot Circle (bullseye,notification,target)' ),
			array( 'far fa-dot-circle' => 'Dot Circle (bullseye,notification,target)' ),
			array( 'fas fa-microphone' => 'microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-microphone-slash' => 'Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'fas fa-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'far fa-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'fas fa-star-half' => 'star-half (achievement,award,rating,score,star-half-empty,star-half-full)' ),
			array( 'far fa-star-half' => 'star-half (achievement,award,rating,score,star-half-empty,star-half-full)' ),
			array( 'fas fa-star-half-alt' => 'Alternate Star Half (achievement,award,rating,score,star-half-empty,star-half-full)' ),
			array( 'fas fa-toggle-off' => 'Toggle Off (switch)' ),
			array( 'fas fa-toggle-on' => 'Toggle On (switch)' ),
			array( 'fas fa-wifi' => 'WiFi (connection,hotspot,internet,network,wireless)' ),
		),
		'Travel' => array(
			array( 'fas fa-archway' => 'Archway (arc,monument,road,street,tunnel)' ),
			array( 'fas fa-atlas' => 'Atlas (book,directions,geography,globe,map,travel,wayfinding)' ),
			array( 'fas fa-bed' => 'Bed (lodging,rest,sleep,travel)' ),
			array( 'fas fa-bus' => 'Bus (public transportation,transportation,travel,vehicle)' ),
			array( 'fas fa-bus-alt' => 'Bus Alt (mta,public transportation,transportation,travel,vehicle)' ),
			array( 'fas fa-cocktail' => 'Cocktail (alcohol,beverage,drink,gin,glass,margarita,martini,vodka)' ),
			array( 'fas fa-concierge-bell' => 'Concierge Bell (attention,hotel,receptionist,service,support)' ),
			array( 'fas fa-dumbbell' => 'Dumbbell (exercise,gym,strength,weight,weight-lifting)' ),
			array( 'fas fa-glass-martini' => 'Martini Glass (alcohol,bar,beverage,drink,liquor)' ),
			array( 'fas fa-glass-martini-alt' => 'Alternate Glass Martini (alcohol,bar,beverage,drink,liquor)' ),
			array( 'fas fa-globe-africa' => 'Globe with Africa shown (all,country,earth,global,gps,language,localize,location,map,online,place,planet,translate,travel,world)' ),
			array( 'fas fa-globe-americas' => 'Globe with Americas shown (all,country,earth,global,gps,language,localize,location,map,online,place,planet,translate,travel,world)' ),
			array( 'fas fa-globe-asia' => 'Globe with Asia shown (all,country,earth,global,gps,language,localize,location,map,online,place,planet,translate,travel,world)' ),
			array( 'fas fa-globe-europe' => 'Globe with Europe shown (all,country,earth,global,gps,language,localize,location,map,online,place,planet,translate,travel,world)' ),
			array( 'fas fa-hot-tub' => 'Hot Tub (bath,jacuzzi,massage,sauna,spa)' ),
			array( 'fas fa-hotel' => 'Hotel (building,inn,lodging,motel,resort,travel)' ),
			array( 'fas fa-luggage-cart' => 'Luggage Cart (bag,baggage,suitcase,travel)' ),
			array( 'fas fa-map' => 'Map (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'far fa-map' => 'Map (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'fas fa-map-marked' => 'Map Marked (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'fas fa-map-marked-alt' => 'Alternate Map Marked (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'fas fa-monument' => 'Monument (building,historic,landmark,memorable)' ),
			array( 'fas fa-passport' => 'Passport (document,id,identification,issued,travel)' ),
			array( 'fas fa-plane' => 'plane (airplane,destination,fly,location,mode,travel,trip)' ),
			array( 'fas fa-plane-arrival' => 'Plane Arrival (airplane,arriving,destination,fly,land,landing,location,mode,travel,trip)' ),
			array( 'fas fa-plane-departure' => 'Plane Departure (airplane,departing,destination,fly,location,mode,take off,taking off,travel,trip)' ),
			array( 'fas fa-shuttle-van' => 'Shuttle Van (airport,machine,public-transportation,transportation,travel,vehicle)' ),
			array( 'fas fa-spa' => 'Spa (flora,massage,mindfulness,plant,wellness)' ),
			array( 'fas fa-suitcase' => 'Suitcase (baggage,luggage,move,suitcase,travel,trip)' ),
			array( 'fas fa-suitcase-rolling' => 'Suitcase Rolling (baggage,luggage,move,suitcase,travel,trip)' ),
			array( 'fas fa-swimmer' => 'Swimmer (athlete,head,man,olympics,person,pool,water)' ),
			array( 'fas fa-swimming-pool' => 'Swimming Pool (ladder,recreation,swim,water)' ),
			array( 'fas fa-taxi' => 'Taxi (cab,cabbie,car,car service,lyft,machine,transportation,travel,uber,vehicle)' ),
			array( 'fas fa-tram' => 'Tram (crossing,machine,mountains,seasonal,transportation)' ),
			array( 'fas fa-tv' => 'Television (computer,display,monitor,television)' ),
			array( 'fas fa-umbrella-beach' => 'Umbrella Beach (protection,recreation,sand,shade,summer,sun)' ),
			array( 'fas fa-wine-glass' => 'Wine Glass (alcohol,beverage,cabernet,drink,grapes,merlot,sauvignon)' ),
			array( 'fas fa-wine-glass-alt' => 'Alternate Wine Glas (alcohol,beverage,cabernet,drink,grapes,merlot,sauvignon)' ),
		),
		'Users & People' => array(
			array( 'fab fa-accessible-icon' => 'Accessible Icon (accessibility,handicap,person,wheelchair,wheelchair-alt)' ),
			array( 'fas fa-address-book' => 'Address Book (contact,directory,index,little black book,rolodex)' ),
			array( 'far fa-address-book' => 'Address Book (contact,directory,index,little black book,rolodex)' ),
			array( 'fas fa-address-card' => 'Address Card (about,contact,id,identification,postcard,profile)' ),
			array( 'far fa-address-card' => 'Address Card (about,contact,id,identification,postcard,profile)' ),
			array( 'fas fa-baby' => 'Baby (child,diaper,doll,human,infant,kid,offspring,person,sprout)' ),
			array( 'fas fa-bed' => 'Bed (lodging,rest,sleep,travel)' ),
			array( 'fas fa-biking' => 'Biking (bicycle,bike,cycle,cycling,ride,wheel)' ),
			array( 'fas fa-blind' => 'Blind (cane,disability,person,sight)' ),
			array( 'fas fa-chalkboard-teacher' => 'Chalkboard Teacher (blackboard,instructor,learning,professor,school,whiteboard,writing)' ),
			array( 'fas fa-child' => 'Child (boy,girl,kid,toddler,young)' ),
			array( 'fas fa-female' => 'Female (human,person,profile,user,woman)' ),
			array( 'fas fa-frown' => 'Frowning Face (disapprove,emoticon,face,rating,sad)' ),
			array( 'far fa-frown' => 'Frowning Face (disapprove,emoticon,face,rating,sad)' ),
			array( 'fas fa-hiking' => 'Hiking (activity,backpack,fall,fitness,outdoors,person,seasonal,walking)' ),
			array( 'fas fa-id-badge' => 'Identification Badge (address,contact,identification,license,profile)' ),
			array( 'far fa-id-badge' => 'Identification Badge (address,contact,identification,license,profile)' ),
			array( 'fas fa-id-card' => 'Identification Card (contact,demographics,document,identification,issued,profile)' ),
			array( 'far fa-id-card' => 'Identification Card (contact,demographics,document,identification,issued,profile)' ),
			array( 'fas fa-id-card-alt' => 'Alternate Identification Card (contact,demographics,document,identification,issued,profile)' ),
			array( 'fas fa-male' => 'Male (human,man,person,profile,user)' ),
			array( 'fas fa-meh' => 'Neutral Face (emoticon,face,neutral,rating)' ),
			array( 'far fa-meh' => 'Neutral Face (emoticon,face,neutral,rating)' ),
			array( 'fas fa-people-carry' => 'People Carry (box,carry,fragile,help,movers,package)' ),
			array( 'fas fa-person-booth' => 'Person Entering Booth (changing,changing room,election,human,person,vote,voting)' ),
			array( 'fas fa-poo' => 'Poo (crap,poop,shit,smile,turd)' ),
			array( 'fas fa-portrait' => 'Portrait (id,image,photo,picture,selfie)' ),
			array( 'fas fa-power-off' => 'Power Off (cancel,computer,on,reboot,restart)' ),
			array( 'fas fa-pray' => 'Pray (kneel,preach,religion,worship)' ),
			array( 'fas fa-restroom' => 'Restroom (bathroom,john,loo,potty,washroom,waste,wc)' ),
			array( 'fas fa-running' => 'Running (exercise,health,jog,person,run,sport,sprint)' ),
			array( 'fas fa-skating' => 'Skating (activity,figure skating,fitness,ice,person,winter)' ),
			array( 'fas fa-skiing' => 'Skiing (activity,downhill,fast,fitness,olympics,outdoors,person,seasonal,slalom)' ),
			array( 'fas fa-skiing-nordic' => 'Skiing Nordic (activity,cross country,fitness,outdoors,person,seasonal)' ),
			array( 'fas fa-smile' => 'Smiling Face (approve,emoticon,face,happy,rating,satisfied)' ),
			array( 'far fa-smile' => 'Smiling Face (approve,emoticon,face,happy,rating,satisfied)' ),
			array( 'fas fa-snowboarding' => 'Snowboarding (activity,fitness,olympics,outdoors,person)' ),
			array( 'fas fa-street-view' => 'Street View (directions,location,map,navigation)' ),
			array( 'fas fa-swimmer' => 'Swimmer (athlete,head,man,olympics,person,pool,water)' ),
			array( 'fas fa-user' => 'User (account,avatar,head,human,man,person,profile)' ),
			array( 'far fa-user' => 'User (account,avatar,head,human,man,person,profile)' ),
			array( 'fas fa-user-alt' => 'Alternate User (account,avatar,head,human,man,person,profile)' ),
			array( 'fas fa-user-alt-slash' => 'Alternate User Slash (account,avatar,head,human,man,person,profile)' ),
			array( 'fas fa-user-astronaut' => 'User Astronaut (avatar,clothing,cosmonaut,nasa,space,suit)' ),
			array( 'fas fa-user-check' => 'User Check (accept,check,person,verified)' ),
			array( 'fas fa-user-circle' => 'User Circle (account,avatar,head,human,man,person,profile)' ),
			array( 'far fa-user-circle' => 'User Circle (account,avatar,head,human,man,person,profile)' ),
			array( 'fas fa-user-clock' => 'User Clock (alert,person,remind,time)' ),
			array( 'fas fa-user-cog' => 'User Cog (admin,cog,person,settings)' ),
			array( 'fas fa-user-edit' => 'User Edit (edit,pen,pencil,person,update,write)' ),
			array( 'fas fa-user-friends' => 'User Friends (group,people,person,team,users)' ),
			array( 'fas fa-user-graduate' => 'User Graduate (cap,clothing,commencement,gown,graduation,person,student)' ),
			array( 'fas fa-user-injured' => 'User Injured (cast,injury,ouch,patient,person,sling)' ),
			array( 'fas fa-user-lock' => 'User Lock (admin,lock,person,private,unlock)' ),
			array( 'fas fa-user-md' => 'Doctor (job,medical,nurse,occupation,physician,profile,surgeon)' ),
			array( 'fas fa-user-minus' => 'User Minus (delete,negative,remove)' ),
			array( 'fas fa-user-ninja' => 'User Ninja (assassin,avatar,dangerous,deadly,sneaky)' ),
			array( 'fas fa-user-nurse' => 'Nurse (doctor,midwife,practitioner,surgeon)' ),
			array( 'fas fa-user-plus' => 'User Plus (add,avatar,positive,sign up,signup,team)' ),
			array( 'fas fa-user-secret' => 'User Secret (clothing,coat,hat,incognito,person,privacy,spy,whisper)' ),
			array( 'fas fa-user-shield' => 'User Shield (admin,person,private,protect,safe)' ),
			array( 'fas fa-user-slash' => 'User Slash (ban,delete,remove)' ),
			array( 'fas fa-user-tag' => 'User Tag (avatar,discount,label,person,role,special)' ),
			array( 'fas fa-user-tie' => 'User Tie (avatar,business,clothing,formal,professional,suit)' ),
			array( 'fas fa-user-times' => 'Remove User (archive,delete,remove,x)' ),
			array( 'fas fa-users' => 'Users (friends,group,people,persons,profiles,team)' ),
			array( 'fas fa-users-cog' => 'Users Cog (admin,cog,group,person,settings,team)' ),
			array( 'fas fa-walking' => 'Walking (exercise,health,pedometer,person,steps)' ),
			array( 'fas fa-wheelchair' => 'Wheelchair (accessible,handicap,person)' ),
		),
		'Vehicles' => array(
			array( 'fab fa-accessible-icon' => 'Accessible Icon (accessibility,handicap,person,wheelchair,wheelchair-alt)' ),
			array( 'fas fa-ambulance' => 'ambulance (emergency,emt,er,help,hospital,support,vehicle)' ),
			array( 'fas fa-baby-carriage' => 'Baby Carriage (buggy,carrier,infant,push,stroller,transportation,walk,wheels)' ),
			array( 'fas fa-bicycle' => 'Bicycle (bike,gears,pedal,transportation,vehicle)' ),
			array( 'fas fa-bus' => 'Bus (public transportation,transportation,travel,vehicle)' ),
			array( 'fas fa-bus-alt' => 'Bus Alt (mta,public transportation,transportation,travel,vehicle)' ),
			array( 'fas fa-car' => 'Car (auto,automobile,sedan,transportation,travel,vehicle)' ),
			array( 'fas fa-car-alt' => 'Alternate Car (auto,automobile,sedan,transportation,travel,vehicle)' ),
			array( 'fas fa-car-crash' => 'Car Crash (accident,auto,automobile,insurance,sedan,transportation,vehicle,wreck)' ),
			array( 'fas fa-car-side' => 'Car Side (auto,automobile,sedan,transportation,travel,vehicle)' ),
			array( 'fas fa-fighter-jet' => 'fighter-jet (airplane,fast,fly,goose,maverick,plane,quick,top gun,transportation,travel)' ),
			array( 'fas fa-helicopter' => 'Helicopter (airwolf,apache,chopper,flight,fly,travel)' ),
			array( 'fas fa-horse' => 'Horse (equus,fauna,mammmal,mare,neigh,pony)' ),
			array( 'fas fa-motorcycle' => 'Motorcycle (bike,machine,transportation,vehicle)' ),
			array( 'fas fa-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'far fa-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'fas fa-plane' => 'plane (airplane,destination,fly,location,mode,travel,trip)' ),
			array( 'fas fa-rocket' => 'rocket (aircraft,app,jet,launch,nasa,space)' ),
			array( 'fas fa-ship' => 'Ship (boat,sea,water)' ),
			array( 'fas fa-shopping-cart' => 'shopping-cart (buy,checkout,grocery,payment,purchase)' ),
			array( 'fas fa-shuttle-van' => 'Shuttle Van (airport,machine,public-transportation,transportation,travel,vehicle)' ),
			array( 'fas fa-sleigh' => 'Sleigh (christmas,claus,fly,holiday,santa,sled,snow,xmas)' ),
			array( 'fas fa-snowplow' => 'Snowplow (clean up,cold,road,storm,winter)' ),
			array( 'fas fa-space-shuttle' => 'Space Shuttle (astronaut,machine,nasa,rocket,transportation)' ),
			array( 'fas fa-subway' => 'Subway (machine,railway,train,transportation,vehicle)' ),
			array( 'fas fa-taxi' => 'Taxi (cab,cabbie,car,car service,lyft,machine,transportation,travel,uber,vehicle)' ),
			array( 'fas fa-tractor' => 'Tractor (agriculture,farm,vehicle)' ),
			array( 'fas fa-train' => 'Train (bullet,commute,locomotive,railway,subway)' ),
			array( 'fas fa-tram' => 'Tram (crossing,machine,mountains,seasonal,transportation)' ),
			array( 'fas fa-truck' => 'truck (cargo,delivery,shipping,vehicle)' ),
			array( 'fas fa-truck-monster' => 'Truck Monster (offroad,vehicle,wheel)' ),
			array( 'fas fa-truck-pickup' => 'Truck Side (cargo,vehicle)' ),
			array( 'fas fa-wheelchair' => 'Wheelchair (accessible,handicap,person)' ),
		),
		'Weather' => array(
			array( 'fas fa-bolt' => 'Lightning Bolt (electricity,lightning,weather,zap)' ),
			array( 'fas fa-cloud' => 'Cloud (atmosphere,fog,overcast,save,upload,weather)' ),
			array( 'fas fa-cloud-meatball' => 'Cloud with (a chance of) Meatball (FLDSMDFR,food,spaghetti,storm)' ),
			array( 'fas fa-cloud-moon' => 'Cloud with Moon (crescent,evening,lunar,night,partly cloudy,sky)' ),
			array( 'fas fa-cloud-moon-rain' => 'Cloud with Moon and Rain (crescent,evening,lunar,night,partly cloudy,precipitation,rain,sky,storm)' ),
			array( 'fas fa-cloud-rain' => 'Cloud with Rain (precipitation,rain,sky,storm)' ),
			array( 'fas fa-cloud-showers-heavy' => 'Cloud with Heavy Showers (precipitation,rain,sky,storm)' ),
			array( 'fas fa-cloud-sun' => 'Cloud with Sun (clear,day,daytime,fall,outdoors,overcast,partly cloudy)' ),
			array( 'fas fa-cloud-sun-rain' => 'Cloud with Sun and Rain (day,overcast,precipitation,storm,summer,sunshower)' ),
			array( 'fas fa-meteor' => 'Meteor (armageddon,asteroid,comet,shooting star,space)' ),
			array( 'fas fa-moon' => 'Moon (contrast,crescent,dark,lunar,night)' ),
			array( 'far fa-moon' => 'Moon (contrast,crescent,dark,lunar,night)' ),
			array( 'fas fa-poo-storm' => 'Poo Storm (bolt,cloud,euphemism,lightning,mess,poop,shit,turd)' ),
			array( 'fas fa-rainbow' => 'Rainbow (gold,leprechaun,prism,rain,sky)' ),
			array( 'fas fa-smog' => 'Smog (dragon,fog,haze,pollution,smoke,weather)' ),
			array( 'fas fa-snowflake' => 'Snowflake (precipitation,rain,winter)' ),
			array( 'far fa-snowflake' => 'Snowflake (precipitation,rain,winter)' ),
			array( 'fas fa-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'far fa-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'fas fa-temperature-high' => 'High Temperature (cook,mercury,summer,thermometer,warm)' ),
			array( 'fas fa-temperature-low' => 'Low Temperature (cold,cool,mercury,thermometer,winter)' ),
			array( 'fas fa-umbrella' => 'Umbrella (protection,rain,storm,wet)' ),
			array( 'fas fa-water' => 'Water (lake,liquid,ocean,sea,swim,wet)' ),
			array( 'fas fa-wind' => 'Wind (air,blow,breeze,fall,seasonal,weather)' ),
		),
		'Winter' => array(
			array( 'fas fa-glass-whiskey' => 'Glass Whiskey (alcohol,bar,beverage,bourbon,drink,liquor,neat,rye,scotch,whisky)' ),
			array( 'fas fa-icicles' => 'Icicles (cold,frozen,hanging,ice,seasonal,sharp)' ),
			array( 'fas fa-igloo' => 'Igloo (dome,dwelling,eskimo,home,house,ice,snow)' ),
			array( 'fas fa-mitten' => 'Mitten (clothing,cold,glove,hands,knitted,seasonal,warmth)' ),
			array( 'fas fa-skating' => 'Skating (activity,figure skating,fitness,ice,person,winter)' ),
			array( 'fas fa-skiing' => 'Skiing (activity,downhill,fast,fitness,olympics,outdoors,person,seasonal,slalom)' ),
			array( 'fas fa-skiing-nordic' => 'Skiing Nordic (activity,cross country,fitness,outdoors,person,seasonal)' ),
			array( 'fas fa-snowboarding' => 'Snowboarding (activity,fitness,olympics,outdoors,person)' ),
			array( 'fas fa-snowplow' => 'Snowplow (clean up,cold,road,storm,winter)' ),
			array( 'fas fa-tram' => 'Tram (crossing,machine,mountains,seasonal,transportation)' ),
		),
		'Writing' => array(
			array( 'fas fa-archive' => 'Archive (box,package,save,storage)' ),
			array( 'fas fa-blog' => 'Blog (journal,log,online,personal,post,web 2.0,wordpress,writing)' ),
			array( 'fas fa-book' => 'book (diary,documentation,journal,library,read)' ),
			array( 'fas fa-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'far fa-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'fas fa-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'far fa-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'fas fa-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'far fa-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'fas fa-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'far fa-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'fas fa-eraser' => 'eraser (art,delete,remove,rubber)' ),
			array( 'fas fa-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'far fa-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'fas fa-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'far fa-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'fas fa-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'far fa-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'fas fa-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'far fa-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'fas fa-keyboard' => 'Keyboard (accessory,edit,input,text,type,write)' ),
			array( 'far fa-keyboard' => 'Keyboard (accessory,edit,input,text,type,write)' ),
			array( 'fas fa-newspaper' => 'Newspaper (article,editorial,headline,journal,journalism,news,press)' ),
			array( 'far fa-newspaper' => 'Newspaper (article,editorial,headline,journal,journalism,news,press)' ),
			array( 'fas fa-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'far fa-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'fas fa-paperclip' => 'Paperclip (attach,attachment,connect,link)' ),
			array( 'fas fa-paragraph' => 'paragraph (edit,format,text,writing)' ),
			array( 'fas fa-pen' => 'Pen (design,edit,update,write)' ),
			array( 'fas fa-pen-alt' => 'Alternate Pen (design,edit,update,write)' ),
			array( 'fas fa-pen-square' => 'Pen Square (edit,pencil-square,update,write)' ),
			array( 'fas fa-pencil-alt' => 'Alternate Pencil (design,edit,pencil,update,write)' ),
			array( 'fas fa-quote-left' => 'quote-left (mention,note,phrase,text,type)' ),
			array( 'fas fa-quote-right' => 'quote-right (mention,note,phrase,text,type)' ),
			array( 'fas fa-sticky-note' => 'Sticky Note (message,note,paper,reminder,sticker)' ),
			array( 'far fa-sticky-note' => 'Sticky Note (message,note,paper,reminder,sticker)' ),
			array( 'fas fa-thumbtack' => 'Thumbtack (coordinates,location,marker,pin,thumb-tack)' ),
		),
		'Other' => array(
			array( 'fas fa-backspace' => 'Backspace (command,delete,erase,keyboard,undo)' ),
			array( 'fas fa-blender-phone' => 'Blender Phone (appliance,cocktail,communication,fantasy,milkshake,mixer,puree,silly,smoothie)' ),
			array( 'fas fa-crown' => 'Crown (award,favorite,king,queen,royal,tiara)' ),
			array( 'fas fa-dumpster-fire' => 'Dumpster Fire (alley,bin,commercial,danger,dangerous,euphemism,flame,heat,hot,trash,waste)' ),
			array( 'fas fa-file-csv' => 'File CSV (document,excel,numbers,spreadsheets,table)' ),
			array( 'fas fa-network-wired' => 'Wired Network (computer,connect,ethernet,internet,intranet)' ),
			array( 'fas fa-signature' => 'Signature (John Hancock,cursive,name,writing)' ),
			array( 'fas fa-skull' => 'Skull (bones,skeleton,x-ray,yorick)' ),
			array( 'fas fa-vr-cardboard' => 'Cardboard VR (3d,augment,google,reality,virtual)' ),
			array( 'fas fa-weight-hanging' => 'Hanging Weight (anvil,heavy,measurement)' ),
		),
	);

	return array_merge( $icons, $fontawesome_icons );
}

add_filter( 'vc_iconpicker-type-material', 'vc_iconpicker_type_material' );
/**
 * Material icon set from Google
 * @param $icons
 *
 * @return array
 * @since 5.0
 *
 */
function vc_iconpicker_type_material( $icons ) {
	$material = array(
		array( 'vc-material vc-material-3d_rotation' => '3d rotation' ),
		array( 'vc-material vc-material-ac_unit' => 'ac unit' ),
		array( 'vc-material vc-material-alarm' => 'alarm' ),
		array( 'vc-material vc-material-access_alarms' => 'access alarms' ),
		array( 'vc-material vc-material-schedule' => 'schedule' ),
		array( 'vc-material vc-material-accessibility' => 'accessibility' ),
		array( 'vc-material vc-material-accessible' => 'accessible' ),
		array( 'vc-material vc-material-account_balance' => 'account balance' ),
		array( 'vc-material vc-material-account_balance_wallet' => 'account balance wallet' ),
		array( 'vc-material vc-material-account_box' => 'account box' ),
		array( 'vc-material vc-material-account_circle' => 'account circle' ),
		array( 'vc-material vc-material-adb' => 'adb' ),
		array( 'vc-material vc-material-add' => 'add' ),
		array( 'vc-material vc-material-add_a_photo' => 'add a photo' ),
		array( 'vc-material vc-material-alarm_add' => 'alarm add' ),
		array( 'vc-material vc-material-add_alert' => 'add alert' ),
		array( 'vc-material vc-material-add_box' => 'add box' ),
		array( 'vc-material vc-material-add_circle' => 'add circle' ),
		array( 'vc-material vc-material-control_point' => 'control point' ),
		array( 'vc-material vc-material-add_location' => 'add location' ),
		array( 'vc-material vc-material-add_shopping_cart' => 'add shopping cart' ),
		array( 'vc-material vc-material-queue' => 'queue' ),
		array( 'vc-material vc-material-add_to_queue' => 'add to queue' ),
		array( 'vc-material vc-material-adjust' => 'adjust' ),
		array( 'vc-material vc-material-airline_seat_flat' => 'airline seat flat' ),
		array( 'vc-material vc-material-airline_seat_flat_angled' => 'airline seat flat angled' ),
		array( 'vc-material vc-material-airline_seat_individual_suite' => 'airline seat individual suite' ),
		array( 'vc-material vc-material-airline_seat_legroom_extra' => 'airline seat legroom extra' ),
		array( 'vc-material vc-material-airline_seat_legroom_normal' => 'airline seat legroom normal' ),
		array( 'vc-material vc-material-airline_seat_legroom_reduced' => 'airline seat legroom reduced' ),
		array( 'vc-material vc-material-airline_seat_recline_extra' => 'airline seat recline extra' ),
		array( 'vc-material vc-material-airline_seat_recline_normal' => 'airline seat recline normal' ),
		array( 'vc-material vc-material-flight' => 'flight' ),
		array( 'vc-material vc-material-airplanemode_inactive' => 'airplanemode inactive' ),
		array( 'vc-material vc-material-airplay' => 'airplay' ),
		array( 'vc-material vc-material-airport_shuttle' => 'airport shuttle' ),
		array( 'vc-material vc-material-alarm_off' => 'alarm off' ),
		array( 'vc-material vc-material-alarm_on' => 'alarm on' ),
		array( 'vc-material vc-material-album' => 'album' ),
		array( 'vc-material vc-material-all_inclusive' => 'all inclusive' ),
		array( 'vc-material vc-material-all_out' => 'all out' ),
		array( 'vc-material vc-material-android' => 'android' ),
		array( 'vc-material vc-material-announcement' => 'announcement' ),
		array( 'vc-material vc-material-apps' => 'apps' ),
		array( 'vc-material vc-material-archive' => 'archive' ),
		array( 'vc-material vc-material-arrow_back' => 'arrow back' ),
		array( 'vc-material vc-material-arrow_downward' => 'arrow downward' ),
		array( 'vc-material vc-material-arrow_drop_down' => 'arrow drop down' ),
		array( 'vc-material vc-material-arrow_drop_down_circle' => 'arrow drop down circle' ),
		array( 'vc-material vc-material-arrow_drop_up' => 'arrow drop up' ),
		array( 'vc-material vc-material-arrow_forward' => 'arrow forward' ),
		array( 'vc-material vc-material-arrow_upward' => 'arrow upward' ),
		array( 'vc-material vc-material-art_track' => 'art track' ),
		array( 'vc-material vc-material-aspect_ratio' => 'aspect ratio' ),
		array( 'vc-material vc-material-poll' => 'poll' ),
		array( 'vc-material vc-material-assignment' => 'assignment' ),
		array( 'vc-material vc-material-assignment_ind' => 'assignment ind' ),
		array( 'vc-material vc-material-assignment_late' => 'assignment late' ),
		array( 'vc-material vc-material-assignment_return' => 'assignment return' ),
		array( 'vc-material vc-material-assignment_returned' => 'assignment returned' ),
		array( 'vc-material vc-material-assignment_turned_in' => 'assignment turned in' ),
		array( 'vc-material vc-material-assistant' => 'assistant' ),
		array( 'vc-material vc-material-flag' => 'flag' ),
		array( 'vc-material vc-material-attach_file' => 'attach file' ),
		array( 'vc-material vc-material-attach_money' => 'attach money' ),
		array( 'vc-material vc-material-attachment' => 'attachment' ),
		array( 'vc-material vc-material-audiotrack' => 'audiotrack' ),
		array( 'vc-material vc-material-autorenew' => 'autorenew' ),
		array( 'vc-material vc-material-av_timer' => 'av timer' ),
		array( 'vc-material vc-material-backspace' => 'backspace' ),
		array( 'vc-material vc-material-cloud_upload' => 'cloud upload' ),
		array( 'vc-material vc-material-battery_alert' => 'battery alert' ),
		array( 'vc-material vc-material-battery_charging_full' => 'battery charging full' ),
		array( 'vc-material vc-material-battery_std' => 'battery std' ),
		array( 'vc-material vc-material-battery_unknown' => 'battery unknown' ),
		array( 'vc-material vc-material-beach_access' => 'beach access' ),
		array( 'vc-material vc-material-beenhere' => 'beenhere' ),
		array( 'vc-material vc-material-block' => 'block' ),
		array( 'vc-material vc-material-bluetooth' => 'bluetooth' ),
		array( 'vc-material vc-material-bluetooth_searching' => 'bluetooth searching' ),
		array( 'vc-material vc-material-bluetooth_connected' => 'bluetooth connected' ),
		array( 'vc-material vc-material-bluetooth_disabled' => 'bluetooth disabled' ),
		array( 'vc-material vc-material-blur_circular' => 'blur circular' ),
		array( 'vc-material vc-material-blur_linear' => 'blur linear' ),
		array( 'vc-material vc-material-blur_off' => 'blur off' ),
		array( 'vc-material vc-material-blur_on' => 'blur on' ),
		array( 'vc-material vc-material-class' => 'class' ),
		array( 'vc-material vc-material-turned_in' => 'turned in' ),
		array( 'vc-material vc-material-turned_in_not' => 'turned in not' ),
		array( 'vc-material vc-material-border_all' => 'border all' ),
		array( 'vc-material vc-material-border_bottom' => 'border bottom' ),
		array( 'vc-material vc-material-border_clear' => 'border clear' ),
		array( 'vc-material vc-material-border_color' => 'border color' ),
		array( 'vc-material vc-material-border_horizontal' => 'border horizontal' ),
		array( 'vc-material vc-material-border_inner' => 'border inner' ),
		array( 'vc-material vc-material-border_left' => 'border left' ),
		array( 'vc-material vc-material-border_outer' => 'border outer' ),
		array( 'vc-material vc-material-border_right' => 'border right' ),
		array( 'vc-material vc-material-border_style' => 'border style' ),
		array( 'vc-material vc-material-border_top' => 'border top' ),
		array( 'vc-material vc-material-border_vertical' => 'border vertical' ),
		array( 'vc-material vc-material-branding_watermark' => 'branding watermark' ),
		array( 'vc-material vc-material-brightness_1' => 'brightness 1' ),
		array( 'vc-material vc-material-brightness_2' => 'brightness 2' ),
		array( 'vc-material vc-material-brightness_3' => 'brightness 3' ),
		array( 'vc-material vc-material-brightness_4' => 'brightness 4' ),
		array( 'vc-material vc-material-brightness_low' => 'brightness low' ),
		array( 'vc-material vc-material-brightness_medium' => 'brightness medium' ),
		array( 'vc-material vc-material-brightness_high' => 'brightness high' ),
		array( 'vc-material vc-material-brightness_auto' => 'brightness auto' ),
		array( 'vc-material vc-material-broken_image' => 'broken image' ),
		array( 'vc-material vc-material-brush' => 'brush' ),
		array( 'vc-material vc-material-bubble_chart' => 'bubble chart' ),
		array( 'vc-material vc-material-bug_report' => 'bug report' ),
		array( 'vc-material vc-material-build' => 'build' ),
		array( 'vc-material vc-material-burst_mode' => 'burst mode' ),
		array( 'vc-material vc-material-domain' => 'domain' ),
		array( 'vc-material vc-material-business_center' => 'business center' ),
		array( 'vc-material vc-material-cached' => 'cached' ),
		array( 'vc-material vc-material-cake' => 'cake' ),
		array( 'vc-material vc-material-phone' => 'phone' ),
		array( 'vc-material vc-material-call_end' => 'call end' ),
		array( 'vc-material vc-material-call_made' => 'call made' ),
		array( 'vc-material vc-material-merge_type' => 'merge type' ),
		array( 'vc-material vc-material-call_missed' => 'call missed' ),
		array( 'vc-material vc-material-call_missed_outgoing' => 'call missed outgoing' ),
		array( 'vc-material vc-material-call_received' => 'call received' ),
		array( 'vc-material vc-material-call_split' => 'call split' ),
		array( 'vc-material vc-material-call_to_action' => 'call to action' ),
		array( 'vc-material vc-material-camera' => 'camera' ),
		array( 'vc-material vc-material-photo_camera' => 'photo camera' ),
		array( 'vc-material vc-material-camera_enhance' => 'camera enhance' ),
		array( 'vc-material vc-material-camera_front' => 'camera front' ),
		array( 'vc-material vc-material-camera_rear' => 'camera rear' ),
		array( 'vc-material vc-material-camera_roll' => 'camera roll' ),
		array( 'vc-material vc-material-cancel' => 'cancel' ),
		array( 'vc-material vc-material-redeem' => 'redeem' ),
		array( 'vc-material vc-material-card_membership' => 'card membership' ),
		array( 'vc-material vc-material-card_travel' => 'card travel' ),
		array( 'vc-material vc-material-casino' => 'casino' ),
		array( 'vc-material vc-material-cast' => 'cast' ),
		array( 'vc-material vc-material-cast_connected' => 'cast connected' ),
		array( 'vc-material vc-material-center_focus_strong' => 'center focus strong' ),
		array( 'vc-material vc-material-center_focus_weak' => 'center focus weak' ),
		array( 'vc-material vc-material-change_history' => 'change history' ),
		array( 'vc-material vc-material-chat' => 'chat' ),
		array( 'vc-material vc-material-chat_bubble' => 'chat bubble' ),
		array( 'vc-material vc-material-chat_bubble_outline' => 'chat bubble outline' ),
		array( 'vc-material vc-material-check' => 'check' ),
		array( 'vc-material vc-material-check_box' => 'check box' ),
		array( 'vc-material vc-material-check_box_outline_blank' => 'check box outline blank' ),
		array( 'vc-material vc-material-check_circle' => 'check circle' ),
		array( 'vc-material vc-material-navigate_before' => 'navigate before' ),
		array( 'vc-material vc-material-navigate_next' => 'navigate next' ),
		array( 'vc-material vc-material-child_care' => 'child care' ),
		array( 'vc-material vc-material-child_friendly' => 'child friendly' ),
		array( 'vc-material vc-material-chrome_reader_mode' => 'chrome reader mode' ),
		array( 'vc-material vc-material-close' => 'close' ),
		array( 'vc-material vc-material-clear_all' => 'clear all' ),
		array( 'vc-material vc-material-closed_caption' => 'closed caption' ),
		array( 'vc-material vc-material-wb_cloudy' => 'wb cloudy' ),
		array( 'vc-material vc-material-cloud_circle' => 'cloud circle' ),
		array( 'vc-material vc-material-cloud_done' => 'cloud done' ),
		array( 'vc-material vc-material-cloud_download' => 'cloud download' ),
		array( 'vc-material vc-material-cloud_off' => 'cloud off' ),
		array( 'vc-material vc-material-cloud_queue' => 'cloud queue' ),
		array( 'vc-material vc-material-code' => 'code' ),
		array( 'vc-material vc-material-photo_library' => 'photo library' ),
		array( 'vc-material vc-material-collections_bookmark' => 'collections bookmark' ),
		array( 'vc-material vc-material-palette' => 'palette' ),
		array( 'vc-material vc-material-colorize' => 'colorize' ),
		array( 'vc-material vc-material-comment' => 'comment' ),
		array( 'vc-material vc-material-compare' => 'compare' ),
		array( 'vc-material vc-material-compare_arrows' => 'compare arrows' ),
		array( 'vc-material vc-material-laptop' => 'laptop' ),
		array( 'vc-material vc-material-confirmation_number' => 'confirmation number' ),
		array( 'vc-material vc-material-contact_mail' => 'contact mail' ),
		array( 'vc-material vc-material-contact_phone' => 'contact phone' ),
		array( 'vc-material vc-material-contacts' => 'contacts' ),
		array( 'vc-material vc-material-content_copy' => 'content copy' ),
		array( 'vc-material vc-material-content_cut' => 'content cut' ),
		array( 'vc-material vc-material-content_paste' => 'content paste' ),
		array( 'vc-material vc-material-control_point_duplicate' => 'control point duplicate' ),
		array( 'vc-material vc-material-copyright' => 'copyright' ),
		array( 'vc-material vc-material-mode_edit' => 'mode edit' ),
		array( 'vc-material vc-material-create_new_folder' => 'create new folder' ),
		array( 'vc-material vc-material-payment' => 'payment' ),
		array( 'vc-material vc-material-crop' => 'crop' ),
		array( 'vc-material vc-material-crop_16_9' => 'crop 16 9' ),
		array( 'vc-material vc-material-crop_3_2' => 'crop 3 2' ),
		array( 'vc-material vc-material-crop_landscape' => 'crop landscape' ),
		array( 'vc-material vc-material-crop_7_5' => 'crop 7 5' ),
		array( 'vc-material vc-material-crop_din' => 'crop din' ),
		array( 'vc-material vc-material-crop_free' => 'crop free' ),
		array( 'vc-material vc-material-crop_original' => 'crop original' ),
		array( 'vc-material vc-material-crop_portrait' => 'crop portrait' ),
		array( 'vc-material vc-material-crop_rotate' => 'crop rotate' ),
		array( 'vc-material vc-material-crop_square' => 'crop square' ),
		array( 'vc-material vc-material-dashboard' => 'dashboard' ),
		array( 'vc-material vc-material-data_usage' => 'data usage' ),
		array( 'vc-material vc-material-date_range' => 'date range' ),
		array( 'vc-material vc-material-dehaze' => 'dehaze' ),
		array( 'vc-material vc-material-delete' => 'delete' ),
		array( 'vc-material vc-material-delete_forever' => 'delete forever' ),
		array( 'vc-material vc-material-delete_sweep' => 'delete sweep' ),
		array( 'vc-material vc-material-description' => 'description' ),
		array( 'vc-material vc-material-desktop_mac' => 'desktop mac' ),
		array( 'vc-material vc-material-desktop_windows' => 'desktop windows' ),
		array( 'vc-material vc-material-details' => 'details' ),
		array( 'vc-material vc-material-developer_board' => 'developer board' ),
		array( 'vc-material vc-material-developer_mode' => 'developer mode' ),
		array( 'vc-material vc-material-device_hub' => 'device hub' ),
		array( 'vc-material vc-material-phonelink' => 'phonelink' ),
		array( 'vc-material vc-material-devices_other' => 'devices other' ),
		array( 'vc-material vc-material-dialer_sip' => 'dialer sip' ),
		array( 'vc-material vc-material-dialpad' => 'dialpad' ),
		array( 'vc-material vc-material-directions' => 'directions' ),
		array( 'vc-material vc-material-directions_bike' => 'directions bike' ),
		array( 'vc-material vc-material-directions_boat' => 'directions boat' ),
		array( 'vc-material vc-material-directions_bus' => 'directions bus' ),
		array( 'vc-material vc-material-directions_car' => 'directions car' ),
		array( 'vc-material vc-material-directions_railway' => 'directions railway' ),
		array( 'vc-material vc-material-directions_run' => 'directions run' ),
		array( 'vc-material vc-material-directions_transit' => 'directions transit' ),
		array( 'vc-material vc-material-directions_walk' => 'directions walk' ),
		array( 'vc-material vc-material-disc_full' => 'disc full' ),
		array( 'vc-material vc-material-dns' => 'dns' ),
		array( 'vc-material vc-material-not_interested' => 'not interested' ),
		array( 'vc-material vc-material-do_not_disturb_alt' => 'do not disturb alt' ),
		array( 'vc-material vc-material-do_not_disturb_off' => 'do not disturb off' ),
		array( 'vc-material vc-material-remove_circle' => 'remove circle' ),
		array( 'vc-material vc-material-dock' => 'dock' ),
		array( 'vc-material vc-material-done' => 'done' ),
		array( 'vc-material vc-material-done_all' => 'done all' ),
		array( 'vc-material vc-material-donut_large' => 'donut large' ),
		array( 'vc-material vc-material-donut_small' => 'donut small' ),
		array( 'vc-material vc-material-drafts' => 'drafts' ),
		array( 'vc-material vc-material-drag_handle' => 'drag handle' ),
		array( 'vc-material vc-material-time_to_leave' => 'time to leave' ),
		array( 'vc-material vc-material-dvr' => 'dvr' ),
		array( 'vc-material vc-material-edit_location' => 'edit location' ),
		array( 'vc-material vc-material-eject' => 'eject' ),
		array( 'vc-material vc-material-markunread' => 'markunread' ),
		array( 'vc-material vc-material-enhanced_encryption' => 'enhanced encryption' ),
		array( 'vc-material vc-material-equalizer' => 'equalizer' ),
		array( 'vc-material vc-material-error' => 'error' ),
		array( 'vc-material vc-material-error_outline' => 'error outline' ),
		array( 'vc-material vc-material-euro_symbol' => 'euro symbol' ),
		array( 'vc-material vc-material-ev_station' => 'ev station' ),
		array( 'vc-material vc-material-insert_invitation' => 'insert invitation' ),
		array( 'vc-material vc-material-event_available' => 'event available' ),
		array( 'vc-material vc-material-event_busy' => 'event busy' ),
		array( 'vc-material vc-material-event_note' => 'event note' ),
		array( 'vc-material vc-material-event_seat' => 'event seat' ),
		array( 'vc-material vc-material-exit_to_app' => 'exit to app' ),
		array( 'vc-material vc-material-expand_less' => 'expand less' ),
		array( 'vc-material vc-material-expand_more' => 'expand more' ),
		array( 'vc-material vc-material-explicit' => 'explicit' ),
		array( 'vc-material vc-material-explore' => 'explore' ),
		array( 'vc-material vc-material-exposure' => 'exposure' ),
		array( 'vc-material vc-material-exposure_neg_1' => 'exposure neg 1' ),
		array( 'vc-material vc-material-exposure_neg_2' => 'exposure neg 2' ),
		array( 'vc-material vc-material-exposure_plus_1' => 'exposure plus 1' ),
		array( 'vc-material vc-material-exposure_plus_2' => 'exposure plus 2' ),
		array( 'vc-material vc-material-exposure_zero' => 'exposure zero' ),
		array( 'vc-material vc-material-extension' => 'extension' ),
		array( 'vc-material vc-material-face' => 'face' ),
		array( 'vc-material vc-material-fast_forward' => 'fast forward' ),
		array( 'vc-material vc-material-fast_rewind' => 'fast rewind' ),
		array( 'vc-material vc-material-favorite' => 'favorite' ),
		array( 'vc-material vc-material-favorite_border' => 'favorite border' ),
		array( 'vc-material vc-material-featured_play_list' => 'featured play list' ),
		array( 'vc-material vc-material-featured_video' => 'featured video' ),
		array( 'vc-material vc-material-sms_failed' => 'sms failed' ),
		array( 'vc-material vc-material-fiber_dvr' => 'fiber dvr' ),
		array( 'vc-material vc-material-fiber_manual_record' => 'fiber manual record' ),
		array( 'vc-material vc-material-fiber_new' => 'fiber new' ),
		array( 'vc-material vc-material-fiber_pin' => 'fiber pin' ),
		array( 'vc-material vc-material-fiber_smart_record' => 'fiber smart record' ),
		array( 'vc-material vc-material-get_app' => 'get app' ),
		array( 'vc-material vc-material-file_upload' => 'file upload' ),
		array( 'vc-material vc-material-filter' => 'filter' ),
		array( 'vc-material vc-material-filter_1' => 'filter 1' ),
		array( 'vc-material vc-material-filter_2' => 'filter 2' ),
		array( 'vc-material vc-material-filter_3' => 'filter 3' ),
		array( 'vc-material vc-material-filter_4' => 'filter 4' ),
		array( 'vc-material vc-material-filter_5' => 'filter 5' ),
		array( 'vc-material vc-material-filter_6' => 'filter 6' ),
		array( 'vc-material vc-material-filter_7' => 'filter 7' ),
		array( 'vc-material vc-material-filter_8' => 'filter 8' ),
		array( 'vc-material vc-material-filter_9' => 'filter 9' ),
		array( 'vc-material vc-material-filter_9_plus' => 'filter 9 plus' ),
		array( 'vc-material vc-material-filter_b_and_w' => 'filter b and w' ),
		array( 'vc-material vc-material-filter_center_focus' => 'filter center focus' ),
		array( 'vc-material vc-material-filter_drama' => 'filter drama' ),
		array( 'vc-material vc-material-filter_frames' => 'filter frames' ),
		array( 'vc-material vc-material-terrain' => 'terrain' ),
		array( 'vc-material vc-material-filter_list' => 'filter list' ),
		array( 'vc-material vc-material-filter_none' => 'filter none' ),
		array( 'vc-material vc-material-filter_tilt_shift' => 'filter tilt shift' ),
		array( 'vc-material vc-material-filter_vintage' => 'filter vintage' ),
		array( 'vc-material vc-material-find_in_page' => 'find in page' ),
		array( 'vc-material vc-material-find_replace' => 'find replace' ),
		array( 'vc-material vc-material-fingerprint' => 'fingerprint' ),
		array( 'vc-material vc-material-first_page' => 'first page' ),
		array( 'vc-material vc-material-fitness_center' => 'fitness center' ),
		array( 'vc-material vc-material-flare' => 'flare' ),
		array( 'vc-material vc-material-flash_auto' => 'flash auto' ),
		array( 'vc-material vc-material-flash_off' => 'flash off' ),
		array( 'vc-material vc-material-flash_on' => 'flash on' ),
		array( 'vc-material vc-material-flight_land' => 'flight land' ),
		array( 'vc-material vc-material-flight_takeoff' => 'flight takeoff' ),
		array( 'vc-material vc-material-flip' => 'flip' ),
		array( 'vc-material vc-material-flip_to_back' => 'flip to back' ),
		array( 'vc-material vc-material-flip_to_front' => 'flip to front' ),
		array( 'vc-material vc-material-folder' => 'folder' ),
		array( 'vc-material vc-material-folder_open' => 'folder open' ),
		array( 'vc-material vc-material-folder_shared' => 'folder shared' ),
		array( 'vc-material vc-material-folder_special' => 'folder special' ),
		array( 'vc-material vc-material-font_download' => 'font download' ),
		array( 'vc-material vc-material-format_align_center' => 'format align center' ),
		array( 'vc-material vc-material-format_align_justify' => 'format align justify' ),
		array( 'vc-material vc-material-format_align_left' => 'format align left' ),
		array( 'vc-material vc-material-format_align_right' => 'format align right' ),
		array( 'vc-material vc-material-format_bold' => 'format bold' ),
		array( 'vc-material vc-material-format_clear' => 'format clear' ),
		array( 'vc-material vc-material-format_color_fill' => 'format color fill' ),
		array( 'vc-material vc-material-format_color_reset' => 'format color reset' ),
		array( 'vc-material vc-material-format_color_text' => 'format color text' ),
		array( 'vc-material vc-material-format_indent_decrease' => 'format indent decrease' ),
		array( 'vc-material vc-material-format_indent_increase' => 'format indent increase' ),
		array( 'vc-material vc-material-format_italic' => 'format italic' ),
		array( 'vc-material vc-material-format_line_spacing' => 'format line spacing' ),
		array( 'vc-material vc-material-format_list_bulleted' => 'format list bulleted' ),
		array( 'vc-material vc-material-format_list_numbered' => 'format list numbered' ),
		array( 'vc-material vc-material-format_paint' => 'format paint' ),
		array( 'vc-material vc-material-format_quote' => 'format quote' ),
		array( 'vc-material vc-material-format_shapes' => 'format shapes' ),
		array( 'vc-material vc-material-format_size' => 'format size' ),
		array( 'vc-material vc-material-format_strikethrough' => 'format strikethrough' ),
		array( 'vc-material vc-material-format_textdirection_l_to_r' => 'format textdirection l to r' ),
		array( 'vc-material vc-material-format_textdirection_r_to_l' => 'format textdirection r to l' ),
		array( 'vc-material vc-material-format_underlined' => 'format underlined' ),
		array( 'vc-material vc-material-question_answer' => 'question answer' ),
		array( 'vc-material vc-material-forward' => 'forward' ),
		array( 'vc-material vc-material-forward_10' => 'forward 10' ),
		array( 'vc-material vc-material-forward_30' => 'forward 30' ),
		array( 'vc-material vc-material-forward_5' => 'forward 5' ),
		array( 'vc-material vc-material-free_breakfast' => 'free breakfast' ),
		array( 'vc-material vc-material-fullscreen' => 'fullscreen' ),
		array( 'vc-material vc-material-fullscreen_exit' => 'fullscreen exit' ),
		array( 'vc-material vc-material-functions' => 'functions' ),
		array( 'vc-material vc-material-g_translate' => 'g translate' ),
		array( 'vc-material vc-material-games' => 'games' ),
		array( 'vc-material vc-material-gavel' => 'gavel' ),
		array( 'vc-material vc-material-gesture' => 'gesture' ),
		array( 'vc-material vc-material-gif' => 'gif' ),
		array( 'vc-material vc-material-goat' => 'goat' ),
		array( 'vc-material vc-material-golf_course' => 'golf course' ),
		array( 'vc-material vc-material-my_location' => 'my location' ),
		array( 'vc-material vc-material-location_searching' => 'location searching' ),
		array( 'vc-material vc-material-location_disabled' => 'location disabled' ),
		array( 'vc-material vc-material-star' => 'star' ),
		array( 'vc-material vc-material-gradient' => 'gradient' ),
		array( 'vc-material vc-material-grain' => 'grain' ),
		array( 'vc-material vc-material-graphic_eq' => 'graphic eq' ),
		array( 'vc-material vc-material-grid_off' => 'grid off' ),
		array( 'vc-material vc-material-grid_on' => 'grid on' ),
		array( 'vc-material vc-material-people' => 'people' ),
		array( 'vc-material vc-material-group_add' => 'group add' ),
		array( 'vc-material vc-material-group_work' => 'group work' ),
		array( 'vc-material vc-material-hd' => 'hd' ),
		array( 'vc-material vc-material-hdr_off' => 'hdr off' ),
		array( 'vc-material vc-material-hdr_on' => 'hdr on' ),
		array( 'vc-material vc-material-hdr_strong' => 'hdr strong' ),
		array( 'vc-material vc-material-hdr_weak' => 'hdr weak' ),
		array( 'vc-material vc-material-headset' => 'headset' ),
		array( 'vc-material vc-material-headset_mic' => 'headset mic' ),
		array( 'vc-material vc-material-healing' => 'healing' ),
		array( 'vc-material vc-material-hearing' => 'hearing' ),
		array( 'vc-material vc-material-help' => 'help' ),
		array( 'vc-material vc-material-help_outline' => 'help outline' ),
		array( 'vc-material vc-material-high_quality' => 'high quality' ),
		array( 'vc-material vc-material-highlight' => 'highlight' ),
		array( 'vc-material vc-material-highlight_off' => 'highlight off' ),
		array( 'vc-material vc-material-restore' => 'restore' ),
		array( 'vc-material vc-material-home' => 'home' ),
		array( 'vc-material vc-material-hot_tub' => 'hot tub' ),
		array( 'vc-material vc-material-local_hotel' => 'local hotel' ),
		array( 'vc-material vc-material-hourglass_empty' => 'hourglass empty' ),
		array( 'vc-material vc-material-hourglass_full' => 'hourglass full' ),
		array( 'vc-material vc-material-http' => 'http' ),
		array( 'vc-material vc-material-lock' => 'lock' ),
		array( 'vc-material vc-material-photo' => 'photo' ),
		array( 'vc-material vc-material-image_aspect_ratio' => 'image aspect ratio' ),
		array( 'vc-material vc-material-import_contacts' => 'import contacts' ),
		array( 'vc-material vc-material-import_export' => 'import export' ),
		array( 'vc-material vc-material-important_devices' => 'important devices' ),
		array( 'vc-material vc-material-inbox' => 'inbox' ),
		array( 'vc-material vc-material-indeterminate_check_box' => 'indeterminate check box' ),
		array( 'vc-material vc-material-info' => 'info' ),
		array( 'vc-material vc-material-info_outline' => 'info outline' ),
		array( 'vc-material vc-material-input' => 'input' ),
		array( 'vc-material vc-material-insert_comment' => 'insert comment' ),
		array( 'vc-material vc-material-insert_drive_file' => 'insert drive file' ),
		array( 'vc-material vc-material-tag_faces' => 'tag faces' ),
		array( 'vc-material vc-material-link' => 'link' ),
		array( 'vc-material vc-material-invert_colors' => 'invert colors' ),
		array( 'vc-material vc-material-invert_colors_off' => 'invert colors off' ),
		array( 'vc-material vc-material-iso' => 'iso' ),
		array( 'vc-material vc-material-keyboard' => 'keyboard' ),
		array( 'vc-material vc-material-keyboard_arrow_down' => 'keyboard arrow down' ),
		array( 'vc-material vc-material-keyboard_arrow_left' => 'keyboard arrow left' ),
		array( 'vc-material vc-material-keyboard_arrow_right' => 'keyboard arrow right' ),
		array( 'vc-material vc-material-keyboard_arrow_up' => 'keyboard arrow up' ),
		array( 'vc-material vc-material-keyboard_backspace' => 'keyboard backspace' ),
		array( 'vc-material vc-material-keyboard_capslock' => 'keyboard capslock' ),
		array( 'vc-material vc-material-keyboard_hide' => 'keyboard hide' ),
		array( 'vc-material vc-material-keyboard_return' => 'keyboard return' ),
		array( 'vc-material vc-material-keyboard_tab' => 'keyboard tab' ),
		array( 'vc-material vc-material-keyboard_voice' => 'keyboard voice' ),
		array( 'vc-material vc-material-kitchen' => 'kitchen' ),
		array( 'vc-material vc-material-label' => 'label' ),
		array( 'vc-material vc-material-label_outline' => 'label outline' ),
		array( 'vc-material vc-material-language' => 'language' ),
		array( 'vc-material vc-material-laptop_chromebook' => 'laptop chromebook' ),
		array( 'vc-material vc-material-laptop_mac' => 'laptop mac' ),
		array( 'vc-material vc-material-laptop_windows' => 'laptop windows' ),
		array( 'vc-material vc-material-last_page' => 'last page' ),
		array( 'vc-material vc-material-open_in_new' => 'open in new' ),
		array( 'vc-material vc-material-layers' => 'layers' ),
		array( 'vc-material vc-material-layers_clear' => 'layers clear' ),
		array( 'vc-material vc-material-leak_add' => 'leak add' ),
		array( 'vc-material vc-material-leak_remove' => 'leak remove' ),
		array( 'vc-material vc-material-lens' => 'lens' ),
		array( 'vc-material vc-material-library_books' => 'library books' ),
		array( 'vc-material vc-material-library_music' => 'library music' ),
		array( 'vc-material vc-material-lightbulb_outline' => 'lightbulb outline' ),
		array( 'vc-material vc-material-line_style' => 'line style' ),
		array( 'vc-material vc-material-line_weight' => 'line weight' ),
		array( 'vc-material vc-material-linear_scale' => 'linear scale' ),
		array( 'vc-material vc-material-linked_camera' => 'linked camera' ),
		array( 'vc-material vc-material-list' => 'list' ),
		array( 'vc-material vc-material-live_help' => 'live help' ),
		array( 'vc-material vc-material-live_tv' => 'live tv' ),
		array( 'vc-material vc-material-local_play' => 'local play' ),
		array( 'vc-material vc-material-local_airport' => 'local airport' ),
		array( 'vc-material vc-material-local_atm' => 'local atm' ),
		array( 'vc-material vc-material-local_bar' => 'local bar' ),
		array( 'vc-material vc-material-local_cafe' => 'local cafe' ),
		array( 'vc-material vc-material-local_car_wash' => 'local car wash' ),
		array( 'vc-material vc-material-local_convenience_store' => 'local convenience store' ),
		array( 'vc-material vc-material-restaurant_menu' => 'restaurant menu' ),
		array( 'vc-material vc-material-local_drink' => 'local drink' ),
		array( 'vc-material vc-material-local_florist' => 'local florist' ),
		array( 'vc-material vc-material-local_gas_station' => 'local gas station' ),
		array( 'vc-material vc-material-shopping_cart' => 'shopping cart' ),
		array( 'vc-material vc-material-local_hospital' => 'local hospital' ),
		array( 'vc-material vc-material-local_laundry_service' => 'local laundry service' ),
		array( 'vc-material vc-material-local_library' => 'local library' ),
		array( 'vc-material vc-material-local_mall' => 'local mall' ),
		array( 'vc-material vc-material-theaters' => 'theaters' ),
		array( 'vc-material vc-material-local_offer' => 'local offer' ),
		array( 'vc-material vc-material-local_parking' => 'local parking' ),
		array( 'vc-material vc-material-local_pharmacy' => 'local pharmacy' ),
		array( 'vc-material vc-material-local_pizza' => 'local pizza' ),
		array( 'vc-material vc-material-print' => 'print' ),
		array( 'vc-material vc-material-local_shipping' => 'local shipping' ),
		array( 'vc-material vc-material-local_taxi' => 'local taxi' ),
		array( 'vc-material vc-material-location_city' => 'location city' ),
		array( 'vc-material vc-material-location_off' => 'location off' ),
		array( 'vc-material vc-material-room' => 'room' ),
		array( 'vc-material vc-material-lock_open' => 'lock open' ),
		array( 'vc-material vc-material-lock_outline' => 'lock outline' ),
		array( 'vc-material vc-material-looks' => 'looks' ),
		array( 'vc-material vc-material-looks_3' => 'looks 3' ),
		array( 'vc-material vc-material-looks_4' => 'looks 4' ),
		array( 'vc-material vc-material-looks_5' => 'looks 5' ),
		array( 'vc-material vc-material-looks_6' => 'looks 6' ),
		array( 'vc-material vc-material-looks_one' => 'looks one' ),
		array( 'vc-material vc-material-looks_two' => 'looks two' ),
		array( 'vc-material vc-material-sync' => 'sync' ),
		array( 'vc-material vc-material-loupe' => 'loupe' ),
		array( 'vc-material vc-material-low_priority' => 'low priority' ),
		array( 'vc-material vc-material-loyalty' => 'loyalty' ),
		array( 'vc-material vc-material-mail_outline' => 'mail outline' ),
		array( 'vc-material vc-material-map' => 'map' ),
		array( 'vc-material vc-material-markunread_mailbox' => 'markunread mailbox' ),
		array( 'vc-material vc-material-memory' => 'memory' ),
		array( 'vc-material vc-material-menu' => 'menu' ),
		array( 'vc-material vc-material-message' => 'message' ),
		array( 'vc-material vc-material-mic' => 'mic' ),
		array( 'vc-material vc-material-mic_none' => 'mic none' ),
		array( 'vc-material vc-material-mic_off' => 'mic off' ),
		array( 'vc-material vc-material-mms' => 'mms' ),
		array( 'vc-material vc-material-mode_comment' => 'mode comment' ),
		array( 'vc-material vc-material-monetization_on' => 'monetization on' ),
		array( 'vc-material vc-material-money_off' => 'money off' ),
		array( 'vc-material vc-material-monochrome_photos' => 'monochrome photos' ),
		array( 'vc-material vc-material-mood_bad' => 'mood bad' ),
		array( 'vc-material vc-material-more' => 'more' ),
		array( 'vc-material vc-material-more_horiz' => 'more horiz' ),
		array( 'vc-material vc-material-more_vert' => 'more vert' ),
		array( 'vc-material vc-material-motorcycle' => 'motorcycle' ),
		array( 'vc-material vc-material-mouse' => 'mouse' ),
		array( 'vc-material vc-material-move_to_inbox' => 'move to inbox' ),
		array( 'vc-material vc-material-movie_creation' => 'movie creation' ),
		array( 'vc-material vc-material-movie_filter' => 'movie filter' ),
		array( 'vc-material vc-material-multiline_chart' => 'multiline chart' ),
		array( 'vc-material vc-material-music_note' => 'music note' ),
		array( 'vc-material vc-material-music_video' => 'music video' ),
		array( 'vc-material vc-material-nature' => 'nature' ),
		array( 'vc-material vc-material-nature_people' => 'nature people' ),
		array( 'vc-material vc-material-navigation' => 'navigation' ),
		array( 'vc-material vc-material-near_me' => 'near me' ),
		array( 'vc-material vc-material-network_cell' => 'network cell' ),
		array( 'vc-material vc-material-network_check' => 'network check' ),
		array( 'vc-material vc-material-network_locked' => 'network locked' ),
		array( 'vc-material vc-material-network_wifi' => 'network wifi' ),
		array( 'vc-material vc-material-new_releases' => 'new releases' ),
		array( 'vc-material vc-material-next_week' => 'next week' ),
		array( 'vc-material vc-material-nfc' => 'nfc' ),
		array( 'vc-material vc-material-no_encryption' => 'no encryption' ),
		array( 'vc-material vc-material-signal_cellular_no_sim' => 'signal cellular no sim' ),
		array( 'vc-material vc-material-note' => 'note' ),
		array( 'vc-material vc-material-note_add' => 'note add' ),
		array( 'vc-material vc-material-notifications' => 'notifications' ),
		array( 'vc-material vc-material-notifications_active' => 'notifications active' ),
		array( 'vc-material vc-material-notifications_none' => 'notifications none' ),
		array( 'vc-material vc-material-notifications_off' => 'notifications off' ),
		array( 'vc-material vc-material-notifications_paused' => 'notifications paused' ),
		array( 'vc-material vc-material-offline_pin' => 'offline pin' ),
		array( 'vc-material vc-material-ondemand_video' => 'ondemand video' ),
		array( 'vc-material vc-material-opacity' => 'opacity' ),
		array( 'vc-material vc-material-open_in_browser' => 'open in browser' ),
		array( 'vc-material vc-material-open_with' => 'open with' ),
		array( 'vc-material vc-material-pages' => 'pages' ),
		array( 'vc-material vc-material-pageview' => 'pageview' ),
		array( 'vc-material vc-material-pan_tool' => 'pan tool' ),
		array( 'vc-material vc-material-panorama' => 'panorama' ),
		array( 'vc-material vc-material-radio_button_unchecked' => 'radio button unchecked' ),
		array( 'vc-material vc-material-panorama_horizontal' => 'panorama horizontal' ),
		array( 'vc-material vc-material-panorama_vertical' => 'panorama vertical' ),
		array( 'vc-material vc-material-panorama_wide_angle' => 'panorama wide angle' ),
		array( 'vc-material vc-material-party_mode' => 'party mode' ),
		array( 'vc-material vc-material-pause' => 'pause' ),
		array( 'vc-material vc-material-pause_circle_filled' => 'pause circle filled' ),
		array( 'vc-material vc-material-pause_circle_outline' => 'pause circle outline' ),
		array( 'vc-material vc-material-people_outline' => 'people outline' ),
		array( 'vc-material vc-material-perm_camera_mic' => 'perm camera mic' ),
		array( 'vc-material vc-material-perm_contact_calendar' => 'perm contact calendar' ),
		array( 'vc-material vc-material-perm_data_setting' => 'perm data setting' ),
		array( 'vc-material vc-material-perm_device_information' => 'perm device information' ),
		array( 'vc-material vc-material-person_outline' => 'person outline' ),
		array( 'vc-material vc-material-perm_media' => 'perm media' ),
		array( 'vc-material vc-material-perm_phone_msg' => 'perm phone msg' ),
		array( 'vc-material vc-material-perm_scan_wifi' => 'perm scan wifi' ),
		array( 'vc-material vc-material-person' => 'person' ),
		array( 'vc-material vc-material-person_add' => 'person add' ),
		array( 'vc-material vc-material-person_pin' => 'person pin' ),
		array( 'vc-material vc-material-person_pin_circle' => 'person pin circle' ),
		array( 'vc-material vc-material-personal_video' => 'personal video' ),
		array( 'vc-material vc-material-pets' => 'pets' ),
		array( 'vc-material vc-material-phone_android' => 'phone android' ),
		array( 'vc-material vc-material-phone_bluetooth_speaker' => 'phone bluetooth speaker' ),
		array( 'vc-material vc-material-phone_forwarded' => 'phone forwarded' ),
		array( 'vc-material vc-material-phone_in_talk' => 'phone in talk' ),
		array( 'vc-material vc-material-phone_iphone' => 'phone iphone' ),
		array( 'vc-material vc-material-phone_locked' => 'phone locked' ),
		array( 'vc-material vc-material-phone_missed' => 'phone missed' ),
		array( 'vc-material vc-material-phone_paused' => 'phone paused' ),
		array( 'vc-material vc-material-phonelink_erase' => 'phonelink erase' ),
		array( 'vc-material vc-material-phonelink_lock' => 'phonelink lock' ),
		array( 'vc-material vc-material-phonelink_off' => 'phonelink off' ),
		array( 'vc-material vc-material-phonelink_ring' => 'phonelink ring' ),
		array( 'vc-material vc-material-phonelink_setup' => 'phonelink setup' ),
		array( 'vc-material vc-material-photo_album' => 'photo album' ),
		array( 'vc-material vc-material-photo_filter' => 'photo filter' ),
		array( 'vc-material vc-material-photo_size_select_actual' => 'photo size select actual' ),
		array( 'vc-material vc-material-photo_size_select_large' => 'photo size select large' ),
		array( 'vc-material vc-material-photo_size_select_small' => 'photo size select small' ),
		array( 'vc-material vc-material-picture_as_pdf' => 'picture as pdf' ),
		array( 'vc-material vc-material-picture_in_picture' => 'picture in picture' ),
		array( 'vc-material vc-material-picture_in_picture_alt' => 'picture in picture alt' ),
		array( 'vc-material vc-material-pie_chart' => 'pie chart' ),
		array( 'vc-material vc-material-pie_chart_outlined' => 'pie chart outlined' ),
		array( 'vc-material vc-material-pin_drop' => 'pin drop' ),
		array( 'vc-material vc-material-play_arrow' => 'play arrow' ),
		array( 'vc-material vc-material-play_circle_filled' => 'play circle filled' ),
		array( 'vc-material vc-material-play_circle_outline' => 'play circle outline' ),
		array( 'vc-material vc-material-play_for_work' => 'play for work' ),
		array( 'vc-material vc-material-playlist_add' => 'playlist add' ),
		array( 'vc-material vc-material-playlist_add_check' => 'playlist add check' ),
		array( 'vc-material vc-material-playlist_play' => 'playlist play' ),
		array( 'vc-material vc-material-plus_one' => 'plus one' ),
		array( 'vc-material vc-material-polymer' => 'polymer' ),
		array( 'vc-material vc-material-pool' => 'pool' ),
		array( 'vc-material vc-material-portable_wifi_off' => 'portable wifi off' ),
		array( 'vc-material vc-material-portrait' => 'portrait' ),
		array( 'vc-material vc-material-power' => 'power' ),
		array( 'vc-material vc-material-power_input' => 'power input' ),
		array( 'vc-material vc-material-power_settings_new' => 'power settings new' ),
		array( 'vc-material vc-material-pregnant_woman' => 'pregnant woman' ),
		array( 'vc-material vc-material-present_to_all' => 'present to all' ),
		array( 'vc-material vc-material-priority_high' => 'priority high' ),
		array( 'vc-material vc-material-public' => 'public' ),
		array( 'vc-material vc-material-publish' => 'publish' ),
		array( 'vc-material vc-material-queue_music' => 'queue music' ),
		array( 'vc-material vc-material-queue_play_next' => 'queue play next' ),
		array( 'vc-material vc-material-radio' => 'radio' ),
		array( 'vc-material vc-material-radio_button_checked' => 'radio button checked' ),
		array( 'vc-material vc-material-rate_review' => 'rate review' ),
		array( 'vc-material vc-material-receipt' => 'receipt' ),
		array( 'vc-material vc-material-recent_actors' => 'recent actors' ),
		array( 'vc-material vc-material-record_voice_over' => 'record voice over' ),
		array( 'vc-material vc-material-redo' => 'redo' ),
		array( 'vc-material vc-material-refresh' => 'refresh' ),
		array( 'vc-material vc-material-remove' => 'remove' ),
		array( 'vc-material vc-material-remove_circle_outline' => 'remove circle outline' ),
		array( 'vc-material vc-material-remove_from_queue' => 'remove from queue' ),
		array( 'vc-material vc-material-visibility' => 'visibility' ),
		array( 'vc-material vc-material-remove_shopping_cart' => 'remove shopping cart' ),
		array( 'vc-material vc-material-reorder' => 'reorder' ),
		array( 'vc-material vc-material-repeat' => 'repeat' ),
		array( 'vc-material vc-material-repeat_one' => 'repeat one' ),
		array( 'vc-material vc-material-replay' => 'replay' ),
		array( 'vc-material vc-material-replay_10' => 'replay 10' ),
		array( 'vc-material vc-material-replay_30' => 'replay 30' ),
		array( 'vc-material vc-material-replay_5' => 'replay 5' ),
		array( 'vc-material vc-material-reply' => 'reply' ),
		array( 'vc-material vc-material-reply_all' => 'reply all' ),
		array( 'vc-material vc-material-report' => 'report' ),
		array( 'vc-material vc-material-warning' => 'warning' ),
		array( 'vc-material vc-material-restaurant' => 'restaurant' ),
		array( 'vc-material vc-material-restore_page' => 'restore page' ),
		array( 'vc-material vc-material-ring_volume' => 'ring volume' ),
		array( 'vc-material vc-material-room_service' => 'room service' ),
		array( 'vc-material vc-material-rotate_90_degrees_ccw' => 'rotate 90 degrees ccw' ),
		array( 'vc-material vc-material-rotate_left' => 'rotate left' ),
		array( 'vc-material vc-material-rotate_right' => 'rotate right' ),
		array( 'vc-material vc-material-rounded_corner' => 'rounded corner' ),
		array( 'vc-material vc-material-router' => 'router' ),
		array( 'vc-material vc-material-rowing' => 'rowing' ),
		array( 'vc-material vc-material-rss_feed' => 'rss feed' ),
		array( 'vc-material vc-material-rv_hookup' => 'rv hookup' ),
		array( 'vc-material vc-material-satellite' => 'satellite' ),
		array( 'vc-material vc-material-save' => 'save' ),
		array( 'vc-material vc-material-scanner' => 'scanner' ),
		array( 'vc-material vc-material-school' => 'school' ),
		array( 'vc-material vc-material-screen_lock_landscape' => 'screen lock landscape' ),
		array( 'vc-material vc-material-screen_lock_portrait' => 'screen lock portrait' ),
		array( 'vc-material vc-material-screen_lock_rotation' => 'screen lock rotation' ),
		array( 'vc-material vc-material-screen_rotation' => 'screen rotation' ),
		array( 'vc-material vc-material-screen_share' => 'screen share' ),
		array( 'vc-material vc-material-sd_storage' => 'sd storage' ),
		array( 'vc-material vc-material-search' => 'search' ),
		array( 'vc-material vc-material-security' => 'security' ),
		array( 'vc-material vc-material-select_all' => 'select all' ),
		array( 'vc-material vc-material-send' => 'send' ),
		array( 'vc-material vc-material-sentiment_dissatisfied' => 'sentiment dissatisfied' ),
		array( 'vc-material vc-material-sentiment_neutral' => 'sentiment neutral' ),
		array( 'vc-material vc-material-sentiment_satisfied' => 'sentiment satisfied' ),
		array( 'vc-material vc-material-sentiment_very_dissatisfied' => 'sentiment very dissatisfied' ),
		array( 'vc-material vc-material-sentiment_very_satisfied' => 'sentiment very satisfied' ),
		array( 'vc-material vc-material-settings' => 'settings' ),
		array( 'vc-material vc-material-settings_applications' => 'settings applications' ),
		array( 'vc-material vc-material-settings_backup_restore' => 'settings backup restore' ),
		array( 'vc-material vc-material-settings_bluetooth' => 'settings bluetooth' ),
		array( 'vc-material vc-material-settings_brightness' => 'settings brightness' ),
		array( 'vc-material vc-material-settings_cell' => 'settings cell' ),
		array( 'vc-material vc-material-settings_ethernet' => 'settings ethernet' ),
		array( 'vc-material vc-material-settings_input_antenna' => 'settings input antenna' ),
		array( 'vc-material vc-material-settings_input_composite' => 'settings input composite' ),
		array( 'vc-material vc-material-settings_input_hdmi' => 'settings input hdmi' ),
		array( 'vc-material vc-material-settings_input_svideo' => 'settings input svideo' ),
		array( 'vc-material vc-material-settings_overscan' => 'settings overscan' ),
		array( 'vc-material vc-material-settings_phone' => 'settings phone' ),
		array( 'vc-material vc-material-settings_power' => 'settings power' ),
		array( 'vc-material vc-material-settings_remote' => 'settings remote' ),
		array( 'vc-material vc-material-settings_system_daydream' => 'settings system daydream' ),
		array( 'vc-material vc-material-settings_voice' => 'settings voice' ),
		array( 'vc-material vc-material-share' => 'share' ),
		array( 'vc-material vc-material-shop' => 'shop' ),
		array( 'vc-material vc-material-shop_two' => 'shop two' ),
		array( 'vc-material vc-material-shopping_basket' => 'shopping basket' ),
		array( 'vc-material vc-material-short_text' => 'short text' ),
		array( 'vc-material vc-material-show_chart' => 'show chart' ),
		array( 'vc-material vc-material-shuffle' => 'shuffle' ),
		array( 'vc-material vc-material-signal_cellular_4_bar' => 'signal cellular 4 bar' ),
		array( 'vc-material vc-material-signal_cellular_connected_no_internet_4_bar' => 'signal_cellular_connected_no internet 4 bar' ),
		array( 'vc-material vc-material-signal_cellular_null' => 'signal cellular null' ),
		array( 'vc-material vc-material-signal_cellular_off' => 'signal cellular off' ),
		array( 'vc-material vc-material-signal_wifi_4_bar' => 'signal wifi 4 bar' ),
		array( 'vc-material vc-material-signal_wifi_4_bar_lock' => 'signal wifi 4 bar lock' ),
		array( 'vc-material vc-material-signal_wifi_off' => 'signal wifi off' ),
		array( 'vc-material vc-material-sim_card' => 'sim card' ),
		array( 'vc-material vc-material-sim_card_alert' => 'sim card alert' ),
		array( 'vc-material vc-material-skip_next' => 'skip next' ),
		array( 'vc-material vc-material-skip_previous' => 'skip previous' ),
		array( 'vc-material vc-material-slideshow' => 'slideshow' ),
		array( 'vc-material vc-material-slow_motion_video' => 'slow motion video' ),
		array( 'vc-material vc-material-stay_primary_portrait' => 'stay primary portrait' ),
		array( 'vc-material vc-material-smoke_free' => 'smoke free' ),
		array( 'vc-material vc-material-smoking_rooms' => 'smoking rooms' ),
		array( 'vc-material vc-material-textsms' => 'textsms' ),
		array( 'vc-material vc-material-snooze' => 'snooze' ),
		array( 'vc-material vc-material-sort' => 'sort' ),
		array( 'vc-material vc-material-sort_by_alpha' => 'sort by alpha' ),
		array( 'vc-material vc-material-spa' => 'spa' ),
		array( 'vc-material vc-material-space_bar' => 'space bar' ),
		array( 'vc-material vc-material-speaker' => 'speaker' ),
		array( 'vc-material vc-material-speaker_group' => 'speaker group' ),
		array( 'vc-material vc-material-speaker_notes' => 'speaker notes' ),
		array( 'vc-material vc-material-speaker_notes_off' => 'speaker notes off' ),
		array( 'vc-material vc-material-speaker_phone' => 'speaker phone' ),
		array( 'vc-material vc-material-spellcheck' => 'spellcheck' ),
		array( 'vc-material vc-material-star_border' => 'star border' ),
		array( 'vc-material vc-material-star_half' => 'star half' ),
		array( 'vc-material vc-material-stars' => 'stars' ),
		array( 'vc-material vc-material-stay_primary_landscape' => 'stay primary landscape' ),
		array( 'vc-material vc-material-stop' => 'stop' ),
		array( 'vc-material vc-material-stop_screen_share' => 'stop screen share' ),
		array( 'vc-material vc-material-storage' => 'storage' ),
		array( 'vc-material vc-material-store_mall_directory' => 'store mall directory' ),
		array( 'vc-material vc-material-straighten' => 'straighten' ),
		array( 'vc-material vc-material-streetview' => 'streetview' ),
		array( 'vc-material vc-material-strikethrough_s' => 'strikethrough s' ),
		array( 'vc-material vc-material-style' => 'style' ),
		array( 'vc-material vc-material-subdirectory_arrow_left' => 'subdirectory arrow left' ),
		array( 'vc-material vc-material-subdirectory_arrow_right' => 'subdirectory arrow right' ),
		array( 'vc-material vc-material-subject' => 'subject' ),
		array( 'vc-material vc-material-subscriptions' => 'subscriptions' ),
		array( 'vc-material vc-material-subtitles' => 'subtitles' ),
		array( 'vc-material vc-material-subway' => 'subway' ),
		array( 'vc-material vc-material-supervisor_account' => 'supervisor account' ),
		array( 'vc-material vc-material-surround_sound' => 'surround sound' ),
		array( 'vc-material vc-material-swap_calls' => 'swap calls' ),
		array( 'vc-material vc-material-swap_horiz' => 'swap horiz' ),
		array( 'vc-material vc-material-swap_vert' => 'swap vert' ),
		array( 'vc-material vc-material-swap_vertical_circle' => 'swap vertical circle' ),
		array( 'vc-material vc-material-switch_camera' => 'switch camera' ),
		array( 'vc-material vc-material-switch_video' => 'switch video' ),
		array( 'vc-material vc-material-sync_disabled' => 'sync disabled' ),
		array( 'vc-material vc-material-sync_problem' => 'sync problem' ),
		array( 'vc-material vc-material-system_update' => 'system update' ),
		array( 'vc-material vc-material-system_update_alt' => 'system update alt' ),
		array( 'vc-material vc-material-tab' => 'tab' ),
		array( 'vc-material vc-material-tab_unselected' => 'tab unselected' ),
		array( 'vc-material vc-material-tablet' => 'tablet' ),
		array( 'vc-material vc-material-tablet_android' => 'tablet android' ),
		array( 'vc-material vc-material-tablet_mac' => 'tablet mac' ),
		array( 'vc-material vc-material-tap_and_play' => 'tap and play' ),
		array( 'vc-material vc-material-text_fields' => 'text fields' ),
		array( 'vc-material vc-material-text_format' => 'text format' ),
		array( 'vc-material vc-material-texture' => 'texture' ),
		array( 'vc-material vc-material-thumb_down' => 'thumb down' ),
		array( 'vc-material vc-material-thumb_up' => 'thumb up' ),
		array( 'vc-material vc-material-thumbs_up_down' => 'thumbs up down' ),
		array( 'vc-material vc-material-timelapse' => 'timelapse' ),
		array( 'vc-material vc-material-timeline' => 'timeline' ),
		array( 'vc-material vc-material-timer' => 'timer' ),
		array( 'vc-material vc-material-timer_10' => 'timer 10' ),
		array( 'vc-material vc-material-timer_3' => 'timer 3' ),
		array( 'vc-material vc-material-timer_off' => 'timer off' ),
		array( 'vc-material vc-material-title' => 'title' ),
		array( 'vc-material vc-material-toc' => 'toc' ),
		array( 'vc-material vc-material-today' => 'today' ),
		array( 'vc-material vc-material-toll' => 'toll' ),
		array( 'vc-material vc-material-tonality' => 'tonality' ),
		array( 'vc-material vc-material-touch_app' => 'touch app' ),
		array( 'vc-material vc-material-toys' => 'toys' ),
		array( 'vc-material vc-material-track_changes' => 'track changes' ),
		array( 'vc-material vc-material-traffic' => 'traffic' ),
		array( 'vc-material vc-material-train' => 'train' ),
		array( 'vc-material vc-material-tram' => 'tram' ),
		array( 'vc-material vc-material-transfer_within_a_station' => 'transfer within a station' ),
		array( 'vc-material vc-material-transform' => 'transform' ),
		array( 'vc-material vc-material-translate' => 'translate' ),
		array( 'vc-material vc-material-trending_down' => 'trending down' ),
		array( 'vc-material vc-material-trending_flat' => 'trending flat' ),
		array( 'vc-material vc-material-trending_up' => 'trending up' ),
		array( 'vc-material vc-material-tune' => 'tune' ),
		array( 'vc-material vc-material-tv' => 'tv' ),
		array( 'vc-material vc-material-unarchive' => 'unarchive' ),
		array( 'vc-material vc-material-undo' => 'undo' ),
		array( 'vc-material vc-material-unfold_less' => 'unfold less' ),
		array( 'vc-material vc-material-unfold_more' => 'unfold more' ),
		array( 'vc-material vc-material-update' => 'update' ),
		array( 'vc-material vc-material-usb' => 'usb' ),
		array( 'vc-material vc-material-verified_user' => 'verified user' ),
		array( 'vc-material vc-material-vertical_align_bottom' => 'vertical align bottom' ),
		array( 'vc-material vc-material-vertical_align_center' => 'vertical align center' ),
		array( 'vc-material vc-material-vertical_align_top' => 'vertical align top' ),
		array( 'vc-material vc-material-vibration' => 'vibration' ),
		array( 'vc-material vc-material-video_call' => 'video call' ),
		array( 'vc-material vc-material-video_label' => 'video label' ),
		array( 'vc-material vc-material-video_library' => 'video library' ),
		array( 'vc-material vc-material-videocam' => 'videocam' ),
		array( 'vc-material vc-material-videocam_off' => 'videocam off' ),
		array( 'vc-material vc-material-videogame_asset' => 'videogame asset' ),
		array( 'vc-material vc-material-view_agenda' => 'view agenda' ),
		array( 'vc-material vc-material-view_array' => 'view array' ),
		array( 'vc-material vc-material-view_carousel' => 'view carousel' ),
		array( 'vc-material vc-material-view_column' => 'view column' ),
		array( 'vc-material vc-material-view_comfy' => 'view comfy' ),
		array( 'vc-material vc-material-view_compact' => 'view compact' ),
		array( 'vc-material vc-material-view_day' => 'view day' ),
		array( 'vc-material vc-material-view_headline' => 'view headline' ),
		array( 'vc-material vc-material-view_list' => 'view list' ),
		array( 'vc-material vc-material-view_module' => 'view module' ),
		array( 'vc-material vc-material-view_quilt' => 'view quilt' ),
		array( 'vc-material vc-material-view_stream' => 'view stream' ),
		array( 'vc-material vc-material-view_week' => 'view week' ),
		array( 'vc-material vc-material-vignette' => 'vignette' ),
		array( 'vc-material vc-material-visibility_off' => 'visibility off' ),
		array( 'vc-material vc-material-voice_chat' => 'voice chat' ),
		array( 'vc-material vc-material-voicemail' => 'voicemail' ),
		array( 'vc-material vc-material-volume_down' => 'volume down' ),
		array( 'vc-material vc-material-volume_mute' => 'volume mute' ),
		array( 'vc-material vc-material-volume_off' => 'volume off' ),
		array( 'vc-material vc-material-volume_up' => 'volume up' ),
		array( 'vc-material vc-material-vpn_key' => 'vpn key' ),
		array( 'vc-material vc-material-vpn_lock' => 'vpn lock' ),
		array( 'vc-material vc-material-wallpaper' => 'wallpaper' ),
		array( 'vc-material vc-material-watch' => 'watch' ),
		array( 'vc-material vc-material-watch_later' => 'watch later' ),
		array( 'vc-material vc-material-wb_auto' => 'wb auto' ),
		array( 'vc-material vc-material-wb_incandescent' => 'wb incandescent' ),
		array( 'vc-material vc-material-wb_iridescent' => 'wb iridescent' ),
		array( 'vc-material vc-material-wb_sunny' => 'wb sunny' ),
		array( 'vc-material vc-material-wc' => 'wc' ),
		array( 'vc-material vc-material-web' => 'web' ),
		array( 'vc-material vc-material-web_asset' => 'web asset' ),
		array( 'vc-material vc-material-weekend' => 'weekend' ),
		array( 'vc-material vc-material-whatshot' => 'whatshot' ),
		array( 'vc-material vc-material-widgets' => 'widgets' ),
		array( 'vc-material vc-material-wifi' => 'wifi' ),
		array( 'vc-material vc-material-wifi_lock' => 'wifi lock' ),
		array( 'vc-material vc-material-wifi_tethering' => 'wifi tethering' ),
		array( 'vc-material vc-material-work' => 'work' ),
		array( 'vc-material vc-material-wrap_text' => 'wrap text' ),
		array( 'vc-material vc-material-youtube_searched_for' => 'youtube searched for' ),
		array( 'vc-material vc-material-zoom_in' => 'zoom in' ),
		array( 'vc-material vc-material-zoom_out' => 'zoom out' ),
		array( 'vc-material vc-material-zoom_out_map' => 'zoom out map' ),
	);

	return array_merge( $icons, $material );
}

add_filter( 'vc_iconpicker-type-lineawesome', 'vc_iconpicker_type_lineawesome' );

/**
 * Line Awesome https://icons8.com/line-awesome
 *
 * @return array - of icons for iconpicker, can be categorized, or not.
 * @since 6.2
 */
function vc_iconpicker_type_lineawesome( $icons ) {
	$lineawesome_icons = array(
		'Accessibility' => array(
			array( 'lab la-accessible-icon' => 'Accessible Icon (accessibility,handicap,person,wheelchair,wheelchair-alt)' ),
			array( 'las la-american-sign-language-interpreting' => 'American Sign Language Interpreting (asl,deaf,finger,hand,interpret,speak)' ),
			array( 'las la-assistive-listening-systems' => 'Assistive Listening Systems (amplify,audio,deaf,ear,headset,hearing,sound)' ),
			array( 'las la-audio-description' => 'Audio Description (blind,narration,video,visual)' ),
			array( 'las la-blind' => 'Blind (cane,disability,person,sight)' ),
			array( 'las la-braille' => 'Braille (alphabet,blind,dots,raised,vision)' ),
			array( 'las la-closed-captioning' => 'Closed Captioning (cc,deaf,hearing,subtitle,subtitling,text,video)' ),
			array( 'lar la-closed-captioning' => 'Closed Captioning (cc,deaf,hearing,subtitle,subtitling,text,video)' ),
			array( 'las la-deaf' => 'Deaf (ear,hearing,sign language)' ),
			array( 'las la-low-vision' => 'Low Vision (blind,eye,sight)' ),
			array( 'las la-phone-volume' => 'Phone Volume (call,earphone,number,sound,support,telephone,voice,volume-control-phone)' ),
			array( 'las la-question-circle' => 'Question Circle (help,information,support,unknown)' ),
			array( 'lar la-question-circle' => 'Question Circle (help,information,support,unknown)' ),
			array( 'las la-sign-language' => 'Sign Language (Translate,asl,deaf,hands)' ),
			array( 'las la-tty' => 'TTY (communication,deaf,telephone,teletypewriter,text)' ),
			array( 'las la-universal-access' => 'Universal Access (accessibility,hearing,person,seeing,visual impairment)' ),
			array( 'las la-wheelchair' => 'Wheelchair (accessible,handicap,person)' ),
		),
		'Alert' => array(
			array( 'las la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'lar la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'las la-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'lar la-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'las la-exclamation' => 'exclamation (alert,danger,error,important,notice,notification,notify,problem,warning)' ),
			array( 'las la-exclamation-circle' => 'Exclamation Circle (alert,danger,error,important,notice,notification,notify,problem,warning)' ),
			array( 'las la-exclamation-triangle' => 'Exclamation Triangle (alert,danger,error,important,notice,notification,notify,problem,warning)' ),
			array( 'las la-radiation' => 'Radiation (danger,dangerous,deadly,hazard,nuclear,radioactive,warning)' ),
			array( 'las la-radiation-alt' => 'Alternate Radiation (danger,dangerous,deadly,hazard,nuclear,radioactive,warning)' ),
			array( 'las la-skull-crossbones' => 'Skull & Crossbones (Dungeons & Dragons,alert,bones,d&d,danger,dead,deadly,death,dnd,fantasy,halloween,holiday,jolly-roger,pirate,poison,skeleton,warning)' ),
		),
		'Animals' => array(
			array( 'las la-cat' => 'Cat (feline,halloween,holiday,kitten,kitty,meow,pet)' ),
			array( 'las la-crow' => 'Crow (bird,bullfrog,fauna,halloween,holiday,toad)' ),
			array( 'las la-dog' => 'Dog (animal,canine,fauna,mammal,pet,pooch,puppy,woof)' ),
			array( 'las la-dove' => 'Dove (bird,fauna,flying,peace,war)' ),
			array( 'las la-dragon' => 'Dragon (Dungeons & Dragons,d&d,dnd,fantasy,fire,lizard,serpent)' ),
			array( 'las la-feather' => 'Feather (bird,light,plucked,quill,write)' ),
			array( 'las la-feather-alt' => 'Alternate Feather (bird,light,plucked,quill,write)' ),
			array( 'las la-fish' => 'Fish (fauna,gold,seafood,swimming)' ),
			array( 'las la-frog' => 'Frog (amphibian,bullfrog,fauna,hop,kermit,kiss,prince,ribbit,toad,wart)' ),
			array( 'las la-hippo' => 'Hippo (animal,fauna,hippopotamus,hungry,mammal)' ),
			array( 'las la-horse' => 'Horse (equus,fauna,mammmal,mare,neigh,pony)' ),
			array( 'las la-horse-head' => 'Horse Head (equus,fauna,mammmal,mare,neigh,pony)' ),
			array( 'las la-kiwi-bird' => 'Kiwi Bird (bird,fauna,new zealand)' ),
			array( 'las la-otter' => 'Otter (animal,badger,fauna,fur,mammal,marten)' ),
			array( 'las la-paw' => 'Paw (animal,cat,dog,pet,print)' ),
			array( 'las la-spider' => 'Spider (arachnid,bug,charlotte,crawl,eight,halloween)' ),
		),
		'Arrows' => array(
			array( 'las la-angle-double-down' => 'Angle Double Down (arrows,caret,download,expand)' ),
			array( 'las la-angle-double-left' => 'Angle Double Left (arrows,back,caret,laquo,previous,quote)' ),
			array( 'las la-angle-double-right' => 'Angle Double Right (arrows,caret,forward,more,next,quote,raquo)' ),
			array( 'las la-angle-double-up' => 'Angle Double Up (arrows,caret,collapse,upload)' ),
			array( 'las la-angle-down' => 'angle-down (arrow,caret,download,expand)' ),
			array( 'las la-angle-left' => 'angle-left (arrow,back,caret,less,previous)' ),
			array( 'las la-angle-right' => 'angle-right (arrow,care,forward,more,next)' ),
			array( 'las la-angle-up' => 'angle-up (arrow,caret,collapse,upload)' ),
			array( 'las la-arrow-alt-circle-down' => 'Alternate Arrow Circle Down (arrow-circle-o-down,download)' ),
			array( 'lar la-arrow-alt-circle-down' => 'Alternate Arrow Circle Down (arrow-circle-o-down,download)' ),
			array( 'las la-arrow-alt-circle-left' => 'Alternate Arrow Circle Left (arrow-circle-o-left,back,previous)' ),
			array( 'lar la-arrow-alt-circle-left' => 'Alternate Arrow Circle Left (arrow-circle-o-left,back,previous)' ),
			array( 'las la-arrow-alt-circle-right' => 'Alternate Arrow Circle Right (arrow-circle-o-right,forward,next)' ),
			array( 'lar la-arrow-alt-circle-right' => 'Alternate Arrow Circle Right (arrow-circle-o-right,forward,next)' ),
			array( 'las la-arrow-alt-circle-up' => 'Alternate Arrow Circle Up (arrow-circle-o-up)' ),
			array( 'lar la-arrow-alt-circle-up' => 'Alternate Arrow Circle Up (arrow-circle-o-up)' ),
			array( 'las la-arrow-circle-down' => 'Arrow Circle Down (download)' ),
			array( 'las la-arrow-circle-left' => 'Arrow Circle Left (back,previous)' ),
			array( 'las la-arrow-circle-right' => 'Arrow Circle Right (forward,next)' ),
			array( 'las la-arrow-circle-up' => 'Arrow Circle Up (upload)' ),
			array( 'las la-arrow-down' => 'arrow-down (download)' ),
			array( 'las la-arrow-left' => 'arrow-left (back,previous)' ),
			array( 'las la-arrow-right' => 'arrow-right (forward,next)' ),
			array( 'las la-arrow-up' => 'arrow-up (forward,upload)' ),
			array( 'las la-arrows-alt' => 'Alternate Arrows (arrow,arrows,bigger,enlarge,expand,fullscreen,move,position,reorder,resize)' ),
			array( 'las la-arrows-alt-h' => 'Alternate Arrows Horizontal (arrows-h,expand,horizontal,landscape,resize,wide)' ),
			array( 'las la-arrows-alt-v' => 'Alternate Arrows Vertical (arrows-v,expand,portrait,resize,tall,vertical)' ),
			array( 'las la-caret-down' => 'Caret Down (arrow,dropdown,expand,menu,more,triangle)' ),
			array( 'las la-caret-left' => 'Caret Left (arrow,back,previous,triangle)' ),
			array( 'las la-caret-right' => 'Caret Right (arrow,forward,next,triangle)' ),
			array( 'las la-caret-square-down' => 'Caret Square Down (arrow,caret-square-o-down,dropdown,expand,menu,more,triangle)' ),
			array( 'lar la-caret-square-down' => 'Caret Square Down (arrow,caret-square-o-down,dropdown,expand,menu,more,triangle)' ),
			array( 'las la-caret-square-left' => 'Caret Square Left (arrow,back,caret-square-o-left,previous,triangle)' ),
			array( 'lar la-caret-square-left' => 'Caret Square Left (arrow,back,caret-square-o-left,previous,triangle)' ),
			array( 'las la-caret-square-right' => 'Caret Square Right (arrow,caret-square-o-right,forward,next,triangle)' ),
			array( 'lar la-caret-square-right' => 'Caret Square Right (arrow,caret-square-o-right,forward,next,triangle)' ),
			array( 'las la-caret-square-up' => 'Caret Square Up (arrow,caret-square-o-up,collapse,triangle,upload)' ),
			array( 'lar la-caret-square-up' => 'Caret Square Up (arrow,caret-square-o-up,collapse,triangle,upload)' ),
			array( 'las la-caret-up' => 'Caret Up (arrow,collapse,triangle)' ),
			array( 'las la-cart-arrow-down' => 'Shopping Cart Arrow Down (download,save,shopping)' ),
			array( 'las la-chart-line' => 'Line Chart (activity,analytics,chart,dashboard,gain,graph,increase,line)' ),
			array( 'las la-chevron-circle-down' => 'Chevron Circle Down (arrow,download,dropdown,menu,more)' ),
			array( 'las la-chevron-circle-left' => 'Chevron Circle Left (arrow,back,previous)' ),
			array( 'las la-chevron-circle-right' => 'Chevron Circle Right (arrow,forward,next)' ),
			array( 'las la-chevron-circle-up' => 'Chevron Circle Up (arrow,collapse,upload)' ),
			array( 'las la-chevron-down' => 'chevron-down (arrow,download,expand)' ),
			array( 'las la-chevron-left' => 'chevron-left (arrow,back,bracket,previous)' ),
			array( 'las la-chevron-right' => 'chevron-right (arrow,bracket,forward,next)' ),
			array( 'las la-chevron-up' => 'chevron-up (arrow,collapse,upload)' ),
			array( 'las la-cloud-download-alt' => 'Alternate Cloud Download (download,export,save)' ),
			array( 'las la-cloud-upload-alt' => 'Alternate Cloud Upload (cloud-upload,import,save,upload)' ),
			array( 'las la-compress-arrows-alt' => 'Alternate Compress Arrows (collapse,fullscreen,minimize,move,resize,shrink,smaller)' ),
			array( 'las la-download' => 'Download (export,hard drive,save,transfer)' ),
			array( 'las la-exchange-alt' => 'Alternate Exchange (arrow,arrows,exchange,reciprocate,return,swap,transfer)' ),
			array( 'las la-expand-arrows-alt' => 'Alternate Expand Arrows (arrows-alt,bigger,enlarge,move,resize)' ),
			array( 'las la-external-link-alt' => 'Alternate External Link (external-link,new,open,share)' ),
			array( 'las la-external-link-square-alt' => 'Alternate External Link Square (external-link-square,new,open,share)' ),
			array( 'las la-hand-point-down' => 'Hand Pointing Down (finger,hand-o-down,point)' ),
			array( 'lar la-hand-point-down' => 'Hand Pointing Down (finger,hand-o-down,point)' ),
			array( 'las la-hand-point-left' => 'Hand Pointing Left (back,finger,hand-o-left,left,point,previous)' ),
			array( 'lar la-hand-point-left' => 'Hand Pointing Left (back,finger,hand-o-left,left,point,previous)' ),
			array( 'las la-hand-point-right' => 'Hand Pointing Right (finger,forward,hand-o-right,next,point,right)' ),
			array( 'lar la-hand-point-right' => 'Hand Pointing Right (finger,forward,hand-o-right,next,point,right)' ),
			array( 'las la-hand-point-up' => 'Hand Pointing Up (finger,hand-o-up,point)' ),
			array( 'lar la-hand-point-up' => 'Hand Pointing Up (finger,hand-o-up,point)' ),
			array( 'las la-hand-pointer' => 'Pointer (Hand) (arrow,cursor,select)' ),
			array( 'lar la-hand-pointer' => 'Pointer (Hand) (arrow,cursor,select)' ),
			array( 'las la-history' => 'History (Rewind,clock,reverse,time,time machine)' ),
			array( 'las la-level-down-alt' => 'Alternate Level Down (arrow,level-down)' ),
			array( 'las la-level-up-alt' => 'Alternate Level Up (arrow,level-up)' ),
			array( 'las la-location-arrow' => 'location-arrow (address,compass,coordinate,direction,gps,map,navigation,place)' ),
			array( 'las la-long-arrow-alt-down' => 'Alternate Long Arrow Down (download,long-arrow-down)' ),
			array( 'las la-long-arrow-alt-left' => 'Alternate Long Arrow Left (back,long-arrow-left,previous)' ),
			array( 'las la-long-arrow-alt-right' => 'Alternate Long Arrow Right (forward,long-arrow-right,next)' ),
			array( 'las la-long-arrow-alt-up' => 'Alternate Long Arrow Up (long-arrow-up,upload)' ),
			array( 'las la-mouse-pointer' => 'Mouse Pointer (arrow,cursor,select)' ),
			array( 'las la-play' => 'play (audio,music,playing,sound,start,video)' ),
			array( 'las la-random' => 'random (arrows,shuffle,sort,swap,switch,transfer)' ),
			array( 'las la-recycle' => 'Recycle (Waste,compost,garbage,reuse,trash)' ),
			array( 'las la-redo' => 'Redo (forward,refresh,reload,repeat)' ),
			array( 'las la-redo-alt' => 'Alternate Redo (forward,refresh,reload,repeat)' ),
			array( 'las la-reply' => 'Reply (mail,message,respond)' ),
			array( 'las la-reply-all' => 'reply-all (mail,message,respond)' ),
			array( 'las la-retweet' => 'Retweet (refresh,reload,share,swap)' ),
			array( 'las la-share' => 'Share (forward,save,send,social)' ),
			array( 'las la-share-square' => 'Share Square (forward,save,send,social)' ),
			array( 'lar la-share-square' => 'Share Square (forward,save,send,social)' ),
			array( 'las la-sign-in-alt' => 'Alternate Sign In (arrow,enter,join,log in,login,sign in,sign up,sign-in,signin,signup)' ),
			array( 'las la-sign-out-alt' => 'Alternate Sign Out (arrow,exit,leave,log out,logout,sign-out)' ),
			array( 'las la-sort' => 'Sort (filter,order)' ),
			array( 'las la-sort-alpha-down' => 'Sort Alphabetical Down (alphabetical,arrange,filter,order,sort-alpha-asc)' ),
			array( 'las la-sort-alpha-down-alt' => 'Alternate Sort Alphabetical Down (alphabetical,arrange,filter,order,sort-alpha-asc)' ),
			array( 'las la-sort-alpha-up' => 'Sort Alphabetical Up (alphabetical,arrange,filter,order,sort-alpha-desc)' ),
			array( 'las la-sort-alpha-up-alt' => 'Alternate Sort Alphabetical Up (alphabetical,arrange,filter,order,sort-alpha-desc)' ),
			array( 'las la-sort-amount-down' => 'Sort Amount Down (arrange,filter,number,order,sort-amount-asc)' ),
			array( 'las la-sort-amount-down-alt' => 'Alternate Sort Amount Down (arrange,filter,order,sort-amount-asc)' ),
			array( 'las la-sort-amount-up' => 'Sort Amount Up (arrange,filter,order,sort-amount-desc)' ),
			array( 'las la-sort-amount-up-alt' => 'Alternate Sort Amount Up (arrange,filter,order,sort-amount-desc)' ),
			array( 'las la-sort-down' => 'Sort Down (Descending) (arrow,descending,filter,order,sort-desc)' ),
			array( 'las la-sort-numeric-down' => 'Sort Numeric Down (arrange,filter,numbers,order,sort-numeric-asc)' ),
			array( 'las la-sort-numeric-down-alt' => 'Alternate Sort Numeric Down (arrange,filter,numbers,order,sort-numeric-asc)' ),
			array( 'las la-sort-numeric-up' => 'Sort Numeric Up (arrange,filter,numbers,order,sort-numeric-desc)' ),
			array( 'las la-sort-numeric-up-alt' => 'Alternate Sort Numeric Up (arrange,filter,numbers,order,sort-numeric-desc)' ),
			array( 'las la-sort-up' => 'Sort Up (Ascending) (arrow,ascending,filter,order,sort-asc)' ),
			array( 'las la-sync' => 'Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'las la-sync-alt' => 'Alternate Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'las la-text-height' => 'text-height (edit,font,format,text,type)' ),
			array( 'las la-text-width' => 'Text Width (edit,font,format,text,type)' ),
			array( 'las la-undo' => 'Undo (back,control z,exchange,oops,return,rotate,swap)' ),
			array( 'las la-undo-alt' => 'Alternate Undo (back,control z,exchange,oops,return,swap)' ),
			array( 'las la-upload' => 'Upload (hard drive,import,publish)' ),
		),
		'Audio & Video' => array(
			array( 'las la-audio-description' => 'Audio Description (blind,narration,video,visual)' ),
			array( 'las la-backward' => 'backward (previous,rewind)' ),
			array( 'las la-broadcast-tower' => 'Broadcast Tower (airwaves,antenna,radio,reception,waves)' ),
			array( 'las la-circle' => 'Circle (circle-thin,diameter,dot,ellipse,notification,round)' ),
			array( 'lar la-circle' => 'Circle (circle-thin,diameter,dot,ellipse,notification,round)' ),
			array( 'las la-closed-captioning' => 'Closed Captioning (cc,deaf,hearing,subtitle,subtitling,text,video)' ),
			array( 'lar la-closed-captioning' => 'Closed Captioning (cc,deaf,hearing,subtitle,subtitling,text,video)' ),
			array( 'las la-compress' => 'Compress (collapse,fullscreen,minimize,move,resize,shrink,smaller)' ),
			array( 'las la-compress-arrows-alt' => 'Alternate Compress Arrows (collapse,fullscreen,minimize,move,resize,shrink,smaller)' ),
			array( 'las la-eject' => 'eject (abort,cancel,cd,discharge)' ),
			array( 'las la-expand' => 'Expand (arrow,bigger,enlarge,resize)' ),
			array( 'las la-expand-arrows-alt' => 'Alternate Expand Arrows (arrows-alt,bigger,enlarge,move,resize)' ),
			array( 'las la-fast-backward' => 'fast-backward (beginning,first,previous,rewind,start)' ),
			array( 'las la-fast-forward' => 'fast-forward (end,last,next)' ),
			array( 'las la-file-audio' => 'Audio File (document,mp3,music,page,play,sound)' ),
			array( 'lar la-file-audio' => 'Audio File (document,mp3,music,page,play,sound)' ),
			array( 'las la-file-video' => 'Video File (document,m4v,movie,mp4,play)' ),
			array( 'lar la-file-video' => 'Video File (document,m4v,movie,mp4,play)' ),
			array( 'las la-film' => 'Film (cinema,movie,strip,video)' ),
			array( 'las la-forward' => 'forward (forward,next,skip)' ),
			array( 'las la-headphones' => 'headphones (audio,listen,music,sound,speaker)' ),
			array( 'las la-microphone' => 'microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'las la-microphone-alt' => 'Alternate Microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'las la-microphone-alt-slash' => 'Alternate Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'las la-microphone-slash' => 'Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'las la-music' => 'Music (lyrics,melody,note,sing,sound)' ),
			array( 'las la-pause' => 'pause (hold,wait)' ),
			array( 'las la-pause-circle' => 'Pause Circle (hold,wait)' ),
			array( 'lar la-pause-circle' => 'Pause Circle (hold,wait)' ),
			array( 'las la-phone-volume' => 'Phone Volume (call,earphone,number,sound,support,telephone,voice,volume-control-phone)' ),
			array( 'las la-photo-video' => 'Photo Video (av,film,image,library,media)' ),
			array( 'las la-play' => 'play (audio,music,playing,sound,start,video)' ),
			array( 'las la-play-circle' => 'Play Circle (audio,music,playing,sound,start,video)' ),
			array( 'lar la-play-circle' => 'Play Circle (audio,music,playing,sound,start,video)' ),
			array( 'las la-podcast' => 'Podcast (audio,broadcast,music,sound)' ),
			array( 'las la-random' => 'random (arrows,shuffle,sort,swap,switch,transfer)' ),
			array( 'las la-redo' => 'Redo (forward,refresh,reload,repeat)' ),
			array( 'las la-redo-alt' => 'Alternate Redo (forward,refresh,reload,repeat)' ),
			array( 'las la-rss' => 'rss (blog,feed,journal,news,writing)' ),
			array( 'las la-rss-square' => 'RSS Square (blog,feed,journal,news,writing)' ),
			array( 'las la-step-backward' => 'step-backward (beginning,first,previous,rewind,start)' ),
			array( 'las la-step-forward' => 'step-forward (end,last,next)' ),
			array( 'las la-stop' => 'stop (block,box,square)' ),
			array( 'las la-stop-circle' => 'Stop Circle (block,box,circle,square)' ),
			array( 'lar la-stop-circle' => 'Stop Circle (block,box,circle,square)' ),
			array( 'las la-sync' => 'Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'las la-sync-alt' => 'Alternate Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'las la-tv' => 'Television (computer,display,monitor,television)' ),
			array( 'las la-undo' => 'Undo (back,control z,exchange,oops,return,rotate,swap)' ),
			array( 'las la-undo-alt' => 'Alternate Undo (back,control z,exchange,oops,return,swap)' ),
			array( 'las la-video' => 'Video (camera,film,movie,record,video-camera)' ),
			array( 'las la-volume-down' => 'Volume Down (audio,lower,music,quieter,sound,speaker)' ),
			array( 'las la-volume-mute' => 'Volume Mute (audio,music,quiet,sound,speaker)' ),
			array( 'las la-volume-off' => 'Volume Off (audio,ban,music,mute,quiet,silent,sound)' ),
			array( 'las la-volume-up' => 'Volume Up (audio,higher,louder,music,sound,speaker)' ),
			array( 'lab la-youtube' => 'YouTube (film,video,youtube-play,youtube-square)' ),
		),
		'Automotive' => array(
			array( 'las la-air-freshener' => 'Air Freshener (car,deodorize,fresh,pine,scent)' ),
			array( 'las la-ambulance' => 'ambulance (emergency,emt,er,help,hospital,support,vehicle)' ),
			array( 'las la-bus' => 'Bus (public transportation,transportation,travel,vehicle)' ),
			array( 'las la-bus-alt' => 'Bus Alt (mta,public transportation,transportation,travel,vehicle)' ),
			array( 'las la-car' => 'Car (auto,automobile,sedan,transportation,travel,vehicle)' ),
			array( 'las la-car-alt' => 'Alternate Car (auto,automobile,sedan,transportation,travel,vehicle)' ),
			array( 'las la-car-battery' => 'Car Battery (auto,electric,mechanic,power)' ),
			array( 'las la-car-crash' => 'Car Crash (accident,auto,automobile,insurance,sedan,transportation,vehicle,wreck)' ),
			array( 'las la-car-side' => 'Car Side (auto,automobile,sedan,transportation,travel,vehicle)' ),
			array( 'las la-charging-station' => 'Charging Station (electric,ev,tesla,vehicle)' ),
			array( 'las la-gas-pump' => 'Gas Pump (car,fuel,gasoline,petrol)' ),
			array( 'las la-motorcycle' => 'Motorcycle (bike,machine,transportation,vehicle)' ),
			array( 'las la-oil-can' => 'Oil Can (auto,crude,gasoline,grease,lubricate,petroleum)' ),
			array( 'las la-shuttle-van' => 'Shuttle Van (airport,machine,public-transportation,transportation,travel,vehicle)' ),
			array( 'las la-tachometer-alt' => 'Alternate Tachometer (dashboard,fast,odometer,speed,speedometer)' ),
			array( 'las la-taxi' => 'Taxi (cab,cabbie,car,car service,lyft,machine,transportation,travel,uber,vehicle)' ),
			array( 'las la-truck' => 'truck (cargo,delivery,shipping,vehicle)' ),
			array( 'las la-truck-monster' => 'Truck Monster (offroad,vehicle,wheel)' ),
			array( 'las la-truck-pickup' => 'Truck Side (cargo,vehicle)' ),
		),
		'Autumn' => array(
			array( 'las la-apple-alt' => 'Fruit Apple (fall,fruit,fuji,macintosh,orchard,seasonal,vegan)' ),
			array( 'las la-campground' => 'Campground (camping,fall,outdoors,teepee,tent,tipi)' ),
			array( 'las la-cloud-sun' => 'Cloud with Sun (clear,day,daytime,fall,outdoors,overcast,partly cloudy)' ),
			array( 'las la-drumstick-bite' => 'Drumstick with Bite Taken Out (bone,chicken,leg,meat,poultry,turkey)' ),
			array( 'las la-football-ball' => 'Football Ball (ball,fall,nfl,pigskin,seasonal)' ),
			array( 'las la-hiking' => 'Hiking (activity,backpack,fall,fitness,outdoors,person,seasonal,walking)' ),
			array( 'las la-mountain' => 'Mountain (glacier,hiking,hill,landscape,travel,view)' ),
			array( 'las la-tractor' => 'Tractor (agriculture,farm,vehicle)' ),
			array( 'las la-tree' => 'Tree (bark,fall,flora,forest,nature,plant,seasonal)' ),
			array( 'las la-wind' => 'Wind (air,blow,breeze,fall,seasonal,weather)' ),
			array( 'las la-wine-bottle' => 'Wine Bottle (alcohol,beverage,cabernet,drink,glass,grapes,merlot,sauvignon)' ),
		),
		'Beverage' => array(
			array( 'las la-beer' => 'beer (alcohol,ale,bar,beverage,brewery,drink,lager,liquor,mug,stein)' ),
			array( 'las la-blender' => 'Blender (cocktail,milkshake,mixer,puree,smoothie)' ),
			array( 'las la-cocktail' => 'Cocktail (alcohol,beverage,drink,gin,glass,margarita,martini,vodka)' ),
			array( 'las la-coffee' => 'Coffee (beverage,breakfast,cafe,drink,fall,morning,mug,seasonal,tea)' ),
			array( 'las la-flask' => 'Flask (beaker,experimental,labs,science)' ),
			array( 'las la-glass-cheers' => 'Glass Cheers (alcohol,bar,beverage,celebration,champagne,clink,drink,holiday,new year\'s eve,party,toast)' ),
			array( 'las la-glass-martini' => 'Martini Glass (alcohol,bar,beverage,drink,liquor)' ),
			array( 'las la-glass-martini-alt' => 'Alternate Glass Martini (alcohol,bar,beverage,drink,liquor)' ),
			array( 'las la-glass-whiskey' => 'Glass Whiskey (alcohol,bar,beverage,bourbon,drink,liquor,neat,rye,scotch,whisky)' ),
			array( 'las la-mug-hot' => 'Mug Hot (caliente,cocoa,coffee,cup,drink,holiday,hot chocolate,steam,tea,warmth)' ),
			array( 'las la-wine-bottle' => 'Wine Bottle (alcohol,beverage,cabernet,drink,glass,grapes,merlot,sauvignon)' ),
			array( 'las la-wine-glass' => 'Wine Glass (alcohol,beverage,cabernet,drink,grapes,merlot,sauvignon)' ),
			array( 'las la-wine-glass-alt' => 'Alternate Wine Glas (alcohol,beverage,cabernet,drink,grapes,merlot,sauvignon)' ),
		),
		'Brands' => array(
			array( 'lab la-creative-commons' => 'Creative Commons' ),
			array( 'lab la-twitter-square' => 'Twitter Square (social network,tweet)' ),
			array( 'lab la-facebook-square' => 'Facebook Square (social network)' ),
			array( 'lab la-linkedin' => 'LinkedIn (linkedin-square)' ),
			array( 'lab la-github-square' => 'GitHub Square (octocat)' ),
			array( 'lab la-twitter' => 'Twitter (social network,tweet)' ),
			array( 'lab la-facebook-f' => 'Facebook F (facebook)' ),
			array( 'lab la-github' => 'GitHub (octocat)' ),
			array( 'lab la-pinterest' => 'Pinterest' ),
			array( 'lab la-pinterest-square' => 'Pinterest Square' ),
			array( 'lab la-google-plus-square' => 'Google Plus Square (social network)' ),
			array( 'lab la-google-plus-g' => 'Google Plus G (google-plus,social network)' ),
			array( 'lab la-linkedin-in' => 'LinkedIn In (linkedin)' ),
			array( 'lab la-github-alt' => 'Alternate GitHub (octocat)' ),
			array( 'lab la-maxcdn' => 'MaxCDN' ),
			array( 'lab la-html5' => 'HTML 5 Logo' ),
			array( 'lab la-css3' => 'CSS 3 Logo (code)' ),
			array( 'lab la-youtube-square' => 'YouTube Square' ),
			array( 'lab la-xing' => 'Xing' ),
			array( 'lab la-xing-square' => 'Xing Square' ),
			array( 'lab la-dropbox' => 'Dropbox' ),
			array( 'lab la-stack-overflow' => 'Stack Overflow' ),
			array( 'lab la-instagram' => 'Instagram' ),
			array( 'lab la-flickr' => 'Flickr' ),
			array( 'lab la-adn' => 'App.net' ),
			array( 'lab la-bitbucket' => 'Bitbucket (atlassian,bitbucket-square,git)' ),
			array( 'lab la-tumblr' => 'Tumblr' ),
			array( 'lab la-tumblr-square' => 'Tumblr Square' ),
			array( 'lab la-apple' => 'Apple (fruit,ios,mac,operating system,os,osx)' ),
			array( 'lab la-windows' => 'Windows (microsoft,operating system,os)' ),
			array( 'lab la-android' => 'Android (robot)' ),
			array( 'lab la-linux' => 'Linux (tux)' ),
			array( 'lab la-dribbble' => 'Dribbble' ),
			array( 'lab la-skype' => 'Skype' ),
			array( 'lab la-foursquare' => 'Foursquare' ),
			array( 'lab la-trello' => 'Trello (atlassian)' ),
			array( 'lab la-gratipay' => 'Gratipay (Gittip) (favorite,heart,like,love)' ),
			array( 'lab la-vk' => 'VK' ),
			array( 'lab la-weibo' => 'Weibo' ),
			array( 'lab la-renren' => 'Renren' ),
			array( 'lab la-pagelines' => 'Pagelines (eco,flora,leaf,leaves,nature,plant,tree)' ),
			array( 'lab la-stack-exchange' => 'Stack Exchange' ),
			array( 'lab la-vimeo-square' => 'Vimeo Square' ),
			array( 'lab la-slack' => 'Slack Logo (anchor,hash,hashtag)' ),
			array( 'lab la-wordpress' => 'WordPress Logo' ),
			array( 'lab la-openid' => 'OpenID' ),
			array( 'lab la-yahoo' => 'Yahoo Logo' ),
			array( 'lab la-google' => 'Google Logo' ),
			array( 'lab la-reddit' => 'reddit Logo' ),
			array( 'lab la-reddit-square' => 'reddit Square' ),
			array( 'lab la-stumbleupon-circle' => 'StumbleUpon Circle' ),
			array( 'lab la-stumbleupon' => 'StumbleUpon Logo' ),
			array( 'lab la-delicious' => 'Delicious' ),
			array( 'lab la-digg' => 'Digg Logo' ),
			array( 'lab la-pied-piper-pp' => 'Pied Piper PP Logo (Old)' ),
			array( 'lab la-pied-piper-alt' => 'Alternate Pied Piper Logo' ),
			array( 'lab la-drupal' => 'Drupal Logo' ),
			array( 'lab la-joomla' => 'Joomla Logo' ),
			array( 'lab la-behance' => 'Behance' ),
			array( 'lab la-behance-square' => 'Behance Square' ),
			array( 'lab la-deviantart' => 'deviantART' ),
			array( 'lab la-vine' => 'Vine' ),
			array( 'lab la-codepen' => 'Codepen' ),
			array( 'lab la-jsfiddle' => 'jsFiddle' ),
			array( 'lab la-rebel' => 'Rebel Alliance' ),
			array( 'lab la-empire' => 'Galactic Empire' ),
			array( 'lab la-git-square' => 'Git Square' ),
			array( 'lab la-git' => 'Git' ),
			array( 'lab la-hacker-news' => 'Hacker News' ),
			array( 'lab la-tencent-weibo' => 'Tencent Weibo' ),
			array( 'lab la-qq' => 'QQ' ),
			array( 'lab la-weixin' => 'Weixin (WeChat)' ),
			array( 'lab la-slideshare' => 'Slideshare' ),
			array( 'lab la-yelp' => 'Yelp' ),
			array( 'lab la-lastfm' => 'last.fm' ),
			array( 'lab la-lastfm-square' => 'last.fm Square' ),
			array( 'lab la-ioxhost' => 'ioxhost' ),
			array( 'lab la-angellist' => 'AngelList' ),
			array( 'lab la-font-awesome' => 'Font Awesome (meanpath)' ),
			array( 'lab la-buysellads' => 'BuySellAds' ),
			array( 'lab la-connectdevelop' => 'Connect Develop' ),
			array( 'lab la-dashcube' => 'DashCube' ),
			array( 'lab la-forumbee' => 'Forumbee' ),
			array( 'lab la-leanpub' => 'Leanpub' ),
			array( 'lab la-sellsy' => 'Sellsy' ),
			array( 'lab la-shirtsinbulk' => 'Shirts in Bulk' ),
			array( 'lab la-simplybuilt' => 'SimplyBuilt' ),
			array( 'lab la-skyatlas' => 'skyatlas' ),
			array( 'lab la-facebook' => 'Facebook (facebook-official,social network)' ),
			array( 'lab la-pinterest-p' => 'Pinterest P' ),
			array( 'lab la-whatsapp' => 'What\'s App' ),
			array( 'lab la-viacoin' => 'Viacoin' ),
			array( 'lab la-medium' => 'Medium' ),
			array( 'lab la-y-combinator' => 'Y Combinator' ),
			array( 'lab la-optin-monster' => 'Optin Monster' ),
			array( 'lab la-opencart' => 'OpenCart' ),
			array( 'lab la-expeditedssl' => 'ExpeditedSSL' ),
			array( 'lab la-tripadvisor' => 'TripAdvisor' ),
			array( 'lab la-odnoklassniki' => 'Odnoklassniki' ),
			array( 'lab la-odnoklassniki-square' => 'Odnoklassniki Square' ),
			array( 'lab la-get-pocket' => 'Get Pocket' ),
			array( 'lab la-wikipedia-w' => 'Wikipedia W' ),
			array( 'lab la-safari' => 'Safari (browser)' ),
			array( 'lab la-chrome' => 'Chrome (browser)' ),
			array( 'lab la-firefox' => 'Firefox (browser)' ),
			array( 'lab la-opera' => 'Opera' ),
			array( 'lab la-internet-explorer' => 'Internet-explorer (browser,ie)' ),
			array( 'lab la-contao' => 'Contao' ),
			array( 'lab la-500px' => '500px' ),
			array( 'lab la-amazon' => 'Amazon' ),
			array( 'lab la-houzz' => 'Houzz' ),
			array( 'lab la-vimeo-v' => 'Vimeo (vimeo)' ),
			array( 'lab la-black-tie' => 'Font Awesome Black Tie' ),
			array( 'lab la-fonticons' => 'Fonticons' ),
			array( 'lab la-reddit-alien' => 'reddit Alien' ),
			array( 'lab la-edge' => 'Edge Browser (browser,ie)' ),
			array( 'lab la-codiepie' => 'Codie Pie' ),
			array( 'lab la-modx' => 'MODX' ),
			array( 'lab la-fort-awesome' => 'Fort Awesome (castle)' ),
			array( 'lab la-usb' => 'USB' ),
			array( 'lab la-product-hunt' => 'Product Hunt' ),
			array( 'lab la-mixcloud' => 'Mixcloud' ),
			array( 'lab la-scribd' => 'Scribd' ),
			array( 'lab la-gitlab' => 'GitLab (Axosoft)' ),
			array( 'lab la-wpbeginner' => 'WPBeginner' ),
			array( 'lab la-wpforms' => 'WPForms' ),
			array( 'lab la-envira' => 'Envira Gallery (leaf)' ),
			array( 'lab la-glide' => 'Glide' ),
			array( 'lab la-glide-g' => 'Glide G' ),
			array( 'lab la-viadeo' => 'Video' ),
			array( 'lab la-viadeo-square' => 'Video Square' ),
			array( 'lab la-snapchat' => 'Snapchat' ),
			array( 'lab la-snapchat-ghost' => 'Snapchat Ghost' ),
			array( 'lab la-snapchat-square' => 'Snapchat Square' ),
			array( 'lab la-pied-piper' => 'Pied Piper Logo' ),
			array( 'lab la-first-order' => 'First Order' ),
			array( 'lab la-yoast' => 'Yoast' ),
			array( 'lab la-themeisle' => 'ThemeIsle' ),
			array( 'lab la-google-plus' => 'Google Plus (google-plus-circle,google-plus-official)' ),
			array( 'lab la-linode' => 'Linode' ),
			array( 'lab la-quora' => 'Quora' ),
			array( 'lab la-free-code-camp' => 'Free Code Camp' ),
			array( 'lab la-telegram' => 'Telegram' ),
			array( 'lab la-bandcamp' => 'Bandcamp' ),
			array( 'lab la-grav' => 'Grav' ),
			array( 'lab la-etsy' => 'Etsy' ),
			array( 'lab la-imdb' => 'IMDB' ),
			array( 'lab la-ravelry' => 'Ravelry' ),
			array( 'lab la-sellcast' => 'Sellcast (eercast)' ),
			array( 'lab la-superpowers' => 'Superpowers' ),
			array( 'lab la-wpexplorer' => 'WPExplorer' ),
			array( 'lab la-meetup' => 'Meetup' ),
		),
		'Buildings' => array(
			array( 'las la-archway' => 'Archway (arc,monument,road,street,tunnel)' ),
			array( 'las la-building' => 'Building (apartment,business,city,company,office,work)' ),
			array( 'lar la-building' => 'Building (apartment,business,city,company,office,work)' ),
			array( 'las la-campground' => 'Campground (camping,fall,outdoors,teepee,tent,tipi)' ),
			array( 'las la-church' => 'Church (building,cathedral,chapel,community,religion)' ),
			array( 'las la-city' => 'City (buildings,busy,skyscrapers,urban,windows)' ),
			array( 'las la-clinic-medical' => 'Medical Clinic (doctor,general practitioner,hospital,infirmary,medicine,office,outpatient)' ),
			array( 'las la-dungeon' => 'Dungeon (Dungeons & Dragons,building,d&d,dnd,door,entrance,fantasy,gate)' ),
			array( 'las la-gopuram' => 'Gopuram (building,entrance,hinduism,temple,tower)' ),
			array( 'las la-home' => 'home (abode,building,house,main)' ),
			array( 'las la-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'lar la-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'las la-hospital-alt' => 'Alternate Hospital (building,emergency room,medical center)' ),
			array( 'las la-hotel' => 'Hotel (building,inn,lodging,motel,resort,travel)' ),
			array( 'las la-house-damage' => 'Damaged House (building,devastation,disaster,home,insurance)' ),
			array( 'las la-igloo' => 'Igloo (dome,dwelling,eskimo,home,house,ice,snow)' ),
			array( 'las la-industry' => 'Industry (building,factory,industrial,manufacturing,mill,warehouse)' ),
			array( 'las la-kaaba' => 'Kaaba (building,cube,islam,muslim)' ),
			array( 'las la-landmark' => 'Landmark (building,historic,memorable,monument,politics)' ),
			array( 'las la-monument' => 'Monument (building,historic,landmark,memorable)' ),
			array( 'las la-mosque' => 'Mosque (building,islam,landmark,muslim)' ),
			array( 'las la-place-of-worship' => 'Place of Worship (building,church,holy,mosque,synagogue)' ),
			array( 'las la-school' => 'School (building,education,learn,student,teacher)' ),
			array( 'las la-store' => 'Store (building,buy,purchase,shopping)' ),
			array( 'las la-store-alt' => 'Alternate Store (building,buy,purchase,shopping)' ),
			array( 'las la-synagogue' => 'Synagogue (building,jewish,judaism,religion,star of david,temple)' ),
			array( 'las la-torii-gate' => 'Torii Gate (building,shintoism)' ),
			array( 'las la-university' => 'University (bank,building,college,higher education - students,institution)' ),
			array( 'las la-vihara' => 'Vihara (buddhism,buddhist,building,monastery)' ),
			array( 'las la-warehouse' => 'Warehouse (building,capacity,garage,inventory,storage)' ),
		),
		'Business' => array(
			array( 'las la-address-book' => 'Address Book (contact,directory,index,little black book,rolodex)' ),
			array( 'lar la-address-book' => 'Address Book (contact,directory,index,little black book,rolodex)' ),
			array( 'las la-address-card' => 'Address Card (about,contact,id,identification,postcard,profile)' ),
			array( 'lar la-address-card' => 'Address Card (about,contact,id,identification,postcard,profile)' ),
			array( 'las la-archive' => 'Archive (box,package,save,storage)' ),
			array( 'las la-balance-scale' => 'Balance Scale (balanced,justice,legal,measure,weight)' ),
			array( 'las la-balance-scale-left' => 'Balance Scale (Left-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'las la-balance-scale-right' => 'Balance Scale (Right-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'las la-birthday-cake' => 'Birthday Cake (anniversary,bakery,candles,celebration,dessert,frosting,holiday,party,pastry)' ),
			array( 'las la-book' => 'book (diary,documentation,journal,library,read)' ),
			array( 'las la-briefcase' => 'Briefcase (bag,business,luggage,office,work)' ),
			array( 'las la-building' => 'Building (apartment,business,city,company,office,work)' ),
			array( 'lar la-building' => 'Building (apartment,business,city,company,office,work)' ),
			array( 'las la-bullhorn' => 'bullhorn (announcement,broadcast,louder,megaphone,share)' ),
			array( 'las la-bullseye' => 'Bullseye (archery,goal,objective,target)' ),
			array( 'las la-business-time' => 'Business Time (alarm,briefcase,business socks,clock,flight of the conchords,reminder,wednesday)' ),
			array( 'las la-calculator' => 'Calculator (abacus,addition,arithmetic,counting,math,multiplication,subtraction)' ),
			array( 'las la-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'lar la-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'las la-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'lar la-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'las la-certificate' => 'certificate (badge,star,verified)' ),
			array( 'las la-chart-area' => 'Area Chart (analytics,area,chart,graph)' ),
			array( 'las la-chart-bar' => 'Bar Chart (analytics,bar,chart,graph)' ),
			array( 'lar la-chart-bar' => 'Bar Chart (analytics,bar,chart,graph)' ),
			array( 'las la-chart-line' => 'Line Chart (activity,analytics,chart,dashboard,gain,graph,increase,line)' ),
			array( 'las la-chart-pie' => 'Pie Chart (analytics,chart,diagram,graph,pie)' ),
			array( 'las la-city' => 'City (buildings,busy,skyscrapers,urban,windows)' ),
			array( 'las la-clipboard' => 'Clipboard (copy,notes,paste,record)' ),
			array( 'lar la-clipboard' => 'Clipboard (copy,notes,paste,record)' ),
			array( 'las la-coffee' => 'Coffee (beverage,breakfast,cafe,drink,fall,morning,mug,seasonal,tea)' ),
			array( 'las la-columns' => 'Columns (browser,dashboard,organize,panes,split)' ),
			array( 'las la-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'lar la-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'las la-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'lar la-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'las la-copyright' => 'Copyright (brand,mark,register,trademark)' ),
			array( 'lar la-copyright' => 'Copyright (brand,mark,register,trademark)' ),
			array( 'las la-cut' => 'Cut (clip,scissors,snip)' ),
			array( 'las la-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'lar la-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'las la-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'lar la-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'las la-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'lar la-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'las la-envelope-square' => 'Envelope Square (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'las la-eraser' => 'eraser (art,delete,remove,rubber)' ),
			array( 'las la-fax' => 'Fax (business,communicate,copy,facsimile,send)' ),
			array( 'las la-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'lar la-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'las la-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'lar la-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'las la-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'lar la-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'las la-folder-minus' => 'Folder Minus (archive,delete,directory,document,file,negative,remove)' ),
			array( 'las la-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'lar la-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'las la-folder-plus' => 'Folder Plus (add,archive,create,directory,document,file,new,positive)' ),
			array( 'las la-glasses' => 'Glasses (hipster,nerd,reading,sight,spectacles,vision)' ),
			array( 'las la-globe' => 'Globe (all,coordinates,country,earth,global,gps,language,localize,location,map,online,place,planet,translate,travel,world)' ),
			array( 'las la-highlighter' => 'Highlighter (edit,marker,sharpie,update,write)' ),
			array( 'las la-industry' => 'Industry (building,factory,industrial,manufacturing,mill,warehouse)' ),
			array( 'las la-landmark' => 'Landmark (building,historic,memorable,monument,politics)' ),
			array( 'las la-marker' => 'Marker (design,edit,sharpie,update,write)' ),
			array( 'las la-paperclip' => 'Paperclip (attach,attachment,connect,link)' ),
			array( 'las la-paste' => 'Paste (clipboard,copy,document,paper)' ),
			array( 'las la-pen' => 'Pen (design,edit,update,write)' ),
			array( 'las la-pen-alt' => 'Alternate Pen (design,edit,update,write)' ),
			array( 'las la-pen-fancy' => 'Pen Fancy (design,edit,fountain pen,update,write)' ),
			array( 'las la-pen-nib' => 'Pen Nib (design,edit,fountain pen,update,write)' ),
			array( 'las la-pen-square' => 'Pen Square (edit,pencil-square,update,write)' ),
			array( 'las la-pencil-alt' => 'Alternate Pencil (design,edit,pencil,update,write)' ),
			array( 'las la-percent' => 'Percent (discount,fraction,proportion,rate,ratio)' ),
			array( 'las la-phone' => 'Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'las la-phone-alt' => 'Alternate Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'las la-phone-slash' => 'Phone Slash (call,cancel,earphone,mute,number,support,telephone,voice)' ),
			array( 'las la-phone-square' => 'Phone Square (call,earphone,number,support,telephone,voice)' ),
			array( 'las la-phone-square-alt' => 'Alternate Phone Square (call,earphone,number,support,telephone,voice)' ),
			array( 'las la-phone-volume' => 'Phone Volume (call,earphone,number,sound,support,telephone,voice,volume-control-phone)' ),
			array( 'las la-print' => 'print (business,copy,document,office,paper)' ),
			array( 'las la-project-diagram' => 'Project Diagram (chart,graph,network,pert)' ),
			array( 'las la-registered' => 'Registered Trademark (copyright,mark,trademark)' ),
			array( 'lar la-registered' => 'Registered Trademark (copyright,mark,trademark)' ),
			array( 'las la-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'lar la-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'las la-sitemap' => 'Sitemap (directory,hierarchy,ia,information architecture,organization)' ),
			array( 'las la-socks' => 'Socks (business socks,business time,clothing,feet,flight of the conchords,wednesday)' ),
			array( 'las la-sticky-note' => 'Sticky Note (message,note,paper,reminder,sticker)' ),
			array( 'lar la-sticky-note' => 'Sticky Note (message,note,paper,reminder,sticker)' ),
			array( 'las la-stream' => 'Stream (flow,list,timeline)' ),
			array( 'las la-table' => 'table (data,excel,spreadsheet)' ),
			array( 'las la-tag' => 'tag (discount,label,price,shopping)' ),
			array( 'las la-tags' => 'tags (discount,label,price,shopping)' ),
			array( 'las la-tasks' => 'Tasks (checklist,downloading,downloads,loading,progress,project management,settings,to do)' ),
			array( 'las la-thumbtack' => 'Thumbtack (coordinates,location,marker,pin,thumb-tack)' ),
			array( 'las la-trademark' => 'Trademark (copyright,register,symbol)' ),
			array( 'las la-wallet' => 'Wallet (billfold,cash,currency,money)' ),
		),
		'Camping' => array(
			array( 'las la-binoculars' => 'Binoculars (glasses,magnify,scenic,spyglass,view)' ),
			array( 'las la-campground' => 'Campground (camping,fall,outdoors,teepee,tent,tipi)' ),
			array( 'las la-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'lar la-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'las la-fire' => 'fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'las la-fire-alt' => 'Alternate Fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'las la-first-aid' => 'First Aid (emergency,emt,health,medical,rescue)' ),
			array( 'las la-frog' => 'Frog (amphibian,bullfrog,fauna,hop,kermit,kiss,prince,ribbit,toad,wart)' ),
			array( 'las la-hiking' => 'Hiking (activity,backpack,fall,fitness,outdoors,person,seasonal,walking)' ),
			array( 'las la-map' => 'Map (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'lar la-map' => 'Map (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'las la-map-marked' => 'Map Marked (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'las la-map-marked-alt' => 'Alternate Map Marked (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'las la-map-signs' => 'Map Signs (directions,directory,map,signage,wayfinding)' ),
			array( 'las la-mountain' => 'Mountain (glacier,hiking,hill,landscape,travel,view)' ),
			array( 'las la-route' => 'Route (directions,navigation,travel)' ),
			array( 'las la-toilet-paper' => 'Toilet Paper (bathroom,halloween,holiday,lavatory,prank,restroom,roll)' ),
			array( 'las la-tree' => 'Tree (bark,fall,flora,forest,nature,plant,seasonal)' ),
		),
		'Charity' => array(
			array( 'las la-dollar-sign' => 'Dollar Sign ($,cost,dollar-sign,money,price,usd)' ),
			array( 'las la-donate' => 'Donate (contribute,generosity,gift,give)' ),
			array( 'las la-dove' => 'Dove (bird,fauna,flying,peace,war)' ),
			array( 'las la-gift' => 'gift (christmas,generosity,giving,holiday,party,present,wrapped,xmas)' ),
			array( 'las la-globe' => 'Globe (all,coordinates,country,earth,global,gps,language,localize,location,map,online,place,planet,translate,travel,world)' ),
			array( 'las la-hand-holding-heart' => 'Hand Holding Heart (carry,charity,gift,lift,package)' ),
			array( 'las la-hand-holding-usd' => 'Hand Holding US Dollar ($,carry,dollar sign,donation,giving,lift,money,price)' ),
			array( 'las la-hands-helping' => 'Helping Hands (aid,assistance,handshake,partnership,volunteering)' ),
			array( 'las la-handshake' => 'Handshake (agreement,greeting,meeting,partnership)' ),
			array( 'lar la-handshake' => 'Handshake (agreement,greeting,meeting,partnership)' ),
			array( 'las la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'lar la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'las la-leaf' => 'leaf (eco,flora,nature,plant,vegan)' ),
			array( 'las la-parachute-box' => 'Parachute Box (aid,assistance,rescue,supplies)' ),
			array( 'las la-piggy-bank' => 'Piggy Bank (bank,save,savings)' ),
			array( 'las la-ribbon' => 'Ribbon (badge,cause,lapel,pin)' ),
			array( 'las la-seedling' => 'Seedling (flora,grow,plant,vegan)' ),
		),
		'Chat' => array(
			array( 'las la-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'lar la-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'las la-comment-alt' => 'Alternate Comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'lar la-comment-alt' => 'Alternate Comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'las la-comment-dots' => 'Comment Dots (bubble,chat,commenting,conversation,feedback,message,more,note,notification,reply,sms,speech,texting)' ),
			array( 'lar la-comment-dots' => 'Comment Dots (bubble,chat,commenting,conversation,feedback,message,more,note,notification,reply,sms,speech,texting)' ),
			array( 'las la-comment-medical' => 'Alternate Medical Chat (advice,bubble,chat,commenting,conversation,diagnose,feedback,message,note,notification,prescription,sms,speech,texting)' ),
			array( 'las la-comment-slash' => 'Comment Slash (bubble,cancel,chat,commenting,conversation,feedback,message,mute,note,notification,quiet,sms,speech,texting)' ),
			array( 'las la-comments' => 'comments (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'lar la-comments' => 'comments (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'las la-frown' => 'Frowning Face (disapprove,emoticon,face,rating,sad)' ),
			array( 'lar la-frown' => 'Frowning Face (disapprove,emoticon,face,rating,sad)' ),
			array( 'las la-icons' => 'Icons (bolt,emoji,heart,image,music,photo,symbols)' ),
			array( 'las la-meh' => 'Neutral Face (emoticon,face,neutral,rating)' ),
			array( 'lar la-meh' => 'Neutral Face (emoticon,face,neutral,rating)' ),
			array( 'las la-phone' => 'Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'las la-phone-alt' => 'Alternate Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'las la-phone-slash' => 'Phone Slash (call,cancel,earphone,mute,number,support,telephone,voice)' ),
			array( 'las la-poo' => 'Poo (crap,poop,shit,smile,turd)' ),
			array( 'las la-quote-left' => 'quote-left (mention,note,phrase,text,type)' ),
			array( 'las la-quote-right' => 'quote-right (mention,note,phrase,text,type)' ),
			array( 'las la-smile' => 'Smiling Face (approve,emoticon,face,happy,rating,satisfied)' ),
			array( 'lar la-smile' => 'Smiling Face (approve,emoticon,face,happy,rating,satisfied)' ),
			array( 'las la-sms' => 'SMS (chat,conversation,message,mobile,notification,phone,sms,texting)' ),
			array( 'las la-video' => 'Video (camera,film,movie,record,video-camera)' ),
			array( 'las la-video-slash' => 'Video Slash (add,create,film,new,positive,record,video)' ),
		),
		'Chess' => array(
			array( 'las la-chess' => 'Chess (board,castle,checkmate,game,king,rook,strategy,tournament)' ),
			array( 'las la-chess-bishop' => 'Chess Bishop (board,checkmate,game,strategy)' ),
			array( 'las la-chess-board' => 'Chess Board (board,checkmate,game,strategy)' ),
			array( 'las la-chess-king' => 'Chess King (board,checkmate,game,strategy)' ),
			array( 'las la-chess-knight' => 'Chess Knight (board,checkmate,game,horse,strategy)' ),
			array( 'las la-chess-pawn' => 'Chess Pawn (board,checkmate,game,strategy)' ),
			array( 'las la-chess-queen' => 'Chess Queen (board,checkmate,game,strategy)' ),
			array( 'las la-chess-rook' => 'Chess Rook (board,castle,checkmate,game,strategy)' ),
			array( 'las la-square-full' => 'Square Full (block,box,shape)' ),
		),
		'Childhood' => array(
			array( 'las la-apple-alt' => 'Fruit Apple (fall,fruit,fuji,macintosh,orchard,seasonal,vegan)' ),
			array( 'las la-baby' => 'Baby (child,diaper,doll,human,infant,kid,offspring,person,sprout)' ),
			array( 'las la-baby-carriage' => 'Baby Carriage (buggy,carrier,infant,push,stroller,transportation,walk,wheels)' ),
			array( 'las la-bath' => 'Bath (clean,shower,tub,wash)' ),
			array( 'las la-biking' => 'Biking (bicycle,bike,cycle,cycling,ride,wheel)' ),
			array( 'las la-birthday-cake' => 'Birthday Cake (anniversary,bakery,candles,celebration,dessert,frosting,holiday,party,pastry)' ),
			array( 'las la-cookie' => 'Cookie (baked good,chips,chocolate,eat,snack,sweet,treat)' ),
			array( 'las la-cookie-bite' => 'Cookie Bite (baked good,bitten,chips,chocolate,eat,snack,sweet,treat)' ),
			array( 'las la-gamepad' => 'Gamepad (arcade,controller,d-pad,joystick,video,video game)' ),
			array( 'las la-ice-cream' => 'Ice Cream (chocolate,cone,dessert,frozen,scoop,sorbet,vanilla,yogurt)' ),
			array( 'las la-mitten' => 'Mitten (clothing,cold,glove,hands,knitted,seasonal,warmth)' ),
			array( 'las la-robot' => 'Robot (android,automate,computer,cyborg)' ),
			array( 'las la-school' => 'School (building,education,learn,student,teacher)' ),
			array( 'las la-shapes' => 'Shapes (blocks,build,circle,square,triangle)' ),
			array( 'las la-snowman' => 'Snowman (decoration,frost,frosty,holiday)' ),
		),
		'Clothing' => array(
			array( 'las la-graduation-cap' => 'Graduation Cap (ceremony,college,graduate,learning,school,student)' ),
			array( 'las la-hat-cowboy' => 'Cowboy Hat (buckaroo,horse,jackeroo,john b.,old west,pardner,ranch,rancher,rodeo,western,wrangler)' ),
			array( 'las la-hat-cowboy-side' => 'Cowboy Hat Side (buckaroo,horse,jackeroo,john b.,old west,pardner,ranch,rancher,rodeo,western,wrangler)' ),
			array( 'las la-hat-wizard' => 'Wizard\'s Hat (Dungeons & Dragons,accessory,buckle,clothing,d&d,dnd,fantasy,halloween,head,holiday,mage,magic,pointy,witch)' ),
			array( 'las la-mitten' => 'Mitten (clothing,cold,glove,hands,knitted,seasonal,warmth)' ),
			array( 'las la-shoe-prints' => 'Shoe Prints (feet,footprints,steps,walk)' ),
			array( 'las la-socks' => 'Socks (business socks,business time,clothing,feet,flight of the conchords,wednesday)' ),
			array( 'las la-tshirt' => 'T-Shirt (clothing,fashion,garment,shirt)' ),
			array( 'las la-user-tie' => 'User Tie (avatar,business,clothing,formal,professional,suit)' ),
		),
		'Code' => array(
			array( 'las la-archive' => 'Archive (box,package,save,storage)' ),
			array( 'las la-barcode' => 'barcode (info,laser,price,scan,upc)' ),
			array( 'las la-bath' => 'Bath (clean,shower,tub,wash)' ),
			array( 'las la-bug' => 'Bug (beetle,error,insect,report)' ),
			array( 'las la-code' => 'Code (brackets,code,development,html)' ),
			array( 'las la-code-branch' => 'Code Branch (branch,code-fork,fork,git,github,rebase,svn,vcs,version)' ),
			array( 'las la-coffee' => 'Coffee (beverage,breakfast,cafe,drink,fall,morning,mug,seasonal,tea)' ),
			array( 'las la-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'lar la-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'las la-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'lar la-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'las la-file-code' => 'Code File (css,development,document,html)' ),
			array( 'lar la-file-code' => 'Code File (css,development,document,html)' ),
			array( 'las la-filter' => 'Filter (funnel,options,separate,sort)' ),
			array( 'las la-fire-extinguisher' => 'fire-extinguisher (burn,caliente,fire fighter,flame,heat,hot,rescue)' ),
			array( 'las la-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'lar la-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'las la-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'lar la-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'las la-keyboard' => 'Keyboard (accessory,edit,input,text,type,write)' ),
			array( 'lar la-keyboard' => 'Keyboard (accessory,edit,input,text,type,write)' ),
			array( 'las la-laptop-code' => 'Laptop Code (computer,cpu,dell,demo,develop,device,mac,macbook,machine,pc)' ),
			array( 'las la-microchip' => 'Microchip (cpu,hardware,processor,technology)' ),
			array( 'las la-project-diagram' => 'Project Diagram (chart,graph,network,pert)' ),
			array( 'las la-qrcode' => 'qrcode (barcode,info,information,scan)' ),
			array( 'las la-shield-alt' => 'Alternate Shield (achievement,award,block,defend,security,winner)' ),
			array( 'las la-sitemap' => 'Sitemap (directory,hierarchy,ia,information architecture,organization)' ),
			array( 'las la-stream' => 'Stream (flow,list,timeline)' ),
			array( 'las la-terminal' => 'Terminal (code,command,console,development,prompt)' ),
			array( 'las la-user-secret' => 'User Secret (clothing,coat,hat,incognito,person,privacy,spy,whisper)' ),
			array( 'las la-window-close' => 'Window Close (browser,cancel,computer,development)' ),
			array( 'lar la-window-close' => 'Window Close (browser,cancel,computer,development)' ),
			array( 'las la-window-maximize' => 'Window Maximize (browser,computer,development,expand)' ),
			array( 'lar la-window-maximize' => 'Window Maximize (browser,computer,development,expand)' ),
			array( 'las la-window-minimize' => 'Window Minimize (browser,collapse,computer,development)' ),
			array( 'lar la-window-minimize' => 'Window Minimize (browser,collapse,computer,development)' ),
			array( 'las la-window-restore' => 'Window Restore (browser,computer,development)' ),
			array( 'lar la-window-restore' => 'Window Restore (browser,computer,development)' ),
		),
		'Communication' => array(
			array( 'las la-address-book' => 'Address Book (contact,directory,index,little black book,rolodex)' ),
			array( 'lar la-address-book' => 'Address Book (contact,directory,index,little black book,rolodex)' ),
			array( 'las la-address-card' => 'Address Card (about,contact,id,identification,postcard,profile)' ),
			array( 'lar la-address-card' => 'Address Card (about,contact,id,identification,postcard,profile)' ),
			array( 'las la-american-sign-language-interpreting' => 'American Sign Language Interpreting (asl,deaf,finger,hand,interpret,speak)' ),
			array( 'las la-assistive-listening-systems' => 'Assistive Listening Systems (amplify,audio,deaf,ear,headset,hearing,sound)' ),
			array( 'las la-at' => 'At (address,author,e-mail,email,handle)' ),
			array( 'las la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'lar la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'las la-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'lar la-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'lab la-bluetooth' => 'Bluetooth' ),
			array( 'lab la-bluetooth-b' => 'Bluetooth' ),
			array( 'las la-broadcast-tower' => 'Broadcast Tower (airwaves,antenna,radio,reception,waves)' ),
			array( 'las la-bullhorn' => 'bullhorn (announcement,broadcast,louder,megaphone,share)' ),
			array( 'las la-chalkboard' => 'Chalkboard (blackboard,learning,school,teaching,whiteboard,writing)' ),
			array( 'las la-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'lar la-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'las la-comment-alt' => 'Alternate Comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'lar la-comment-alt' => 'Alternate Comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'las la-comments' => 'comments (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'lar la-comments' => 'comments (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'las la-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'lar la-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'las la-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'lar la-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'las la-envelope-square' => 'Envelope Square (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'las la-fax' => 'Fax (business,communicate,copy,facsimile,send)' ),
			array( 'las la-inbox' => 'inbox (archive,desk,email,mail,message)' ),
			array( 'las la-language' => 'Language (dialect,idiom,localize,speech,translate,vernacular)' ),
			array( 'las la-microphone' => 'microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'las la-microphone-alt' => 'Alternate Microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'las la-microphone-alt-slash' => 'Alternate Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'las la-microphone-slash' => 'Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'las la-mobile' => 'Mobile Phone (apple,call,cell phone,cellphone,device,iphone,number,screen,telephone)' ),
			array( 'las la-mobile-alt' => 'Alternate Mobile (apple,call,cell phone,cellphone,device,iphone,number,screen,telephone)' ),
			array( 'las la-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'lar la-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'las la-phone' => 'Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'las la-phone-alt' => 'Alternate Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'las la-phone-slash' => 'Phone Slash (call,cancel,earphone,mute,number,support,telephone,voice)' ),
			array( 'las la-phone-square' => 'Phone Square (call,earphone,number,support,telephone,voice)' ),
			array( 'las la-phone-square-alt' => 'Alternate Phone Square (call,earphone,number,support,telephone,voice)' ),
			array( 'las la-phone-volume' => 'Phone Volume (call,earphone,number,sound,support,telephone,voice,volume-control-phone)' ),
			array( 'las la-rss' => 'rss (blog,feed,journal,news,writing)' ),
			array( 'las la-rss-square' => 'RSS Square (blog,feed,journal,news,writing)' ),
			array( 'las la-tty' => 'TTY (communication,deaf,telephone,teletypewriter,text)' ),
			array( 'las la-voicemail' => 'Voicemail (answer,inbox,message,phone)' ),
			array( 'las la-wifi' => 'WiFi (connection,hotspot,internet,network,wireless)' ),
		),
		'Computers' => array(
			array( 'las la-database' => 'Database (computer,development,directory,memory,storage)' ),
			array( 'las la-desktop' => 'Desktop (computer,cpu,demo,desktop,device,imac,machine,monitor,pc,screen)' ),
			array( 'las la-download' => 'Download (export,hard drive,save,transfer)' ),
			array( 'las la-ethernet' => 'Ethernet (cable,cat 5,cat 6,connection,hardware,internet,network,wired)' ),
			array( 'las la-hdd' => 'HDD (cpu,hard drive,harddrive,machine,save,storage)' ),
			array( 'lar la-hdd' => 'HDD (cpu,hard drive,harddrive,machine,save,storage)' ),
			array( 'las la-headphones' => 'headphones (audio,listen,music,sound,speaker)' ),
			array( 'las la-keyboard' => 'Keyboard (accessory,edit,input,text,type,write)' ),
			array( 'lar la-keyboard' => 'Keyboard (accessory,edit,input,text,type,write)' ),
			array( 'las la-laptop' => 'Laptop (computer,cpu,dell,demo,device,mac,macbook,machine,pc)' ),
			array( 'las la-memory' => 'Memory (DIMM,RAM,hardware,storage,technology)' ),
			array( 'las la-microchip' => 'Microchip (cpu,hardware,processor,technology)' ),
			array( 'las la-mobile' => 'Mobile Phone (apple,call,cell phone,cellphone,device,iphone,number,screen,telephone)' ),
			array( 'las la-mobile-alt' => 'Alternate Mobile (apple,call,cell phone,cellphone,device,iphone,number,screen,telephone)' ),
			array( 'las la-mouse' => 'Mouse (click,computer,cursor,input,peripheral)' ),
			array( 'las la-plug' => 'Plug (connect,electric,online,power)' ),
			array( 'las la-power-off' => 'Power Off (cancel,computer,on,reboot,restart)' ),
			array( 'las la-print' => 'print (business,copy,document,office,paper)' ),
			array( 'las la-satellite' => 'Satellite (communications,hardware,orbit,space)' ),
			array( 'las la-satellite-dish' => 'Satellite Dish (SETI,communications,hardware,receiver,saucer,signal)' ),
			array( 'las la-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'lar la-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'las la-sd-card' => 'Sd Card (image,memory,photo,save)' ),
			array( 'las la-server' => 'Server (computer,cpu,database,hardware,network)' ),
			array( 'las la-sim-card' => 'SIM Card (hard drive,hardware,portable,storage,technology,tiny)' ),
			array( 'las la-stream' => 'Stream (flow,list,timeline)' ),
			array( 'las la-tablet' => 'tablet (apple,device,ipad,kindle,screen)' ),
			array( 'las la-tablet-alt' => 'Alternate Tablet (apple,device,ipad,kindle,screen)' ),
			array( 'las la-tv' => 'Television (computer,display,monitor,television)' ),
			array( 'las la-upload' => 'Upload (hard drive,import,publish)' ),
		),
		'Construction' => array(
			array( 'las la-brush' => 'Brush (art,bristles,color,handle,paint)' ),
			array( 'las la-drafting-compass' => 'Drafting Compass (design,map,mechanical drawing,plot,plotting)' ),
			array( 'las la-dumpster' => 'Dumpster (alley,bin,commercial,trash,waste)' ),
			array( 'las la-hammer' => 'Hammer (admin,fix,repair,settings,tool)' ),
			array( 'las la-hard-hat' => 'Hard Hat (construction,hardhat,helmet,safety)' ),
			array( 'las la-paint-roller' => 'Paint Roller (acrylic,art,brush,color,fill,paint,pigment,watercolor)' ),
			array( 'las la-pencil-alt' => 'Alternate Pencil (design,edit,pencil,update,write)' ),
			array( 'las la-pencil-ruler' => 'Pencil Ruler (design,draft,draw,pencil)' ),
			array( 'las la-ruler' => 'Ruler (design,draft,length,measure,planning)' ),
			array( 'las la-ruler-combined' => 'Ruler Combined (design,draft,length,measure,planning)' ),
			array( 'las la-ruler-horizontal' => 'Ruler Horizontal (design,draft,length,measure,planning)' ),
			array( 'las la-ruler-vertical' => 'Ruler Vertical (design,draft,length,measure,planning)' ),
			array( 'las la-screwdriver' => 'Screwdriver (admin,fix,mechanic,repair,settings,tool)' ),
			array( 'las la-toolbox' => 'Toolbox (admin,container,fix,repair,settings,tools)' ),
			array( 'las la-tools' => 'Tools (admin,fix,repair,screwdriver,settings,tools,wrench)' ),
			array( 'las la-truck-pickup' => 'Truck Side (cargo,vehicle)' ),
			array( 'las la-wrench' => 'Wrench (construction,fix,mechanic,plumbing,settings,spanner,tool,update)' ),
		),
		'Currency' => array(
			array( 'lab la-bitcoin' => 'Bitcoin' ),
			array( 'lab la-btc' => 'BTC' ),
			array( 'las la-dollar-sign' => 'Dollar Sign ($,cost,dollar-sign,money,price,usd)' ),
			array( 'lab la-ethereum' => 'Ethereum' ),
			array( 'las la-euro-sign' => 'Euro Sign (currency,dollar,exchange,money)' ),
			array( 'lab la-gg' => 'GG Currency' ),
			array( 'lab la-gg-circle' => 'GG Currency Circle' ),
			array( 'las la-hryvnia' => 'Hryvnia (currency,money,ukraine,ukrainian)' ),
			array( 'las la-lira-sign' => 'Turkish Lira Sign (currency,money,try,turkish)' ),
			array( 'las la-money-bill' => 'Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'las la-money-bill-alt' => 'Alternate Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'lar la-money-bill-alt' => 'Alternate Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'las la-money-bill-wave' => 'Wavy Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'las la-money-bill-wave-alt' => 'Alternate Wavy Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'las la-money-check' => 'Money Check (bank check,buy,checkout,cheque,money,payment,price,purchase)' ),
			array( 'las la-money-check-alt' => 'Alternate Money Check (bank check,buy,checkout,cheque,money,payment,price,purchase)' ),
			array( 'las la-pound-sign' => 'Pound Sign (currency,gbp,money)' ),
			array( 'las la-ruble-sign' => 'Ruble Sign (currency,money,rub)' ),
			array( 'las la-rupee-sign' => 'Indian Rupee Sign (currency,indian,inr,money)' ),
			array( 'las la-shekel-sign' => 'Shekel Sign (currency,ils,money)' ),
			array( 'las la-tenge' => 'Tenge (currency,kazakhstan,money,price)' ),
			array( 'las la-won-sign' => 'Won Sign (currency,krw,money)' ),
			array( 'las la-yen-sign' => 'Yen Sign (currency,jpy,money)' ),
		),
		'Date & Time' => array(
			array( 'las la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'lar la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'las la-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'lar la-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'las la-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'lar la-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'las la-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'lar la-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'las la-calendar-check' => 'Calendar Check (accept,agree,appointment,confirm,correct,date,done,event,ok,schedule,select,success,tick,time,todo,when)' ),
			array( 'lar la-calendar-check' => 'Calendar Check (accept,agree,appointment,confirm,correct,date,done,event,ok,schedule,select,success,tick,time,todo,when)' ),
			array( 'las la-calendar-minus' => 'Calendar Minus (calendar,date,delete,event,negative,remove,schedule,time,when)' ),
			array( 'lar la-calendar-minus' => 'Calendar Minus (calendar,date,delete,event,negative,remove,schedule,time,when)' ),
			array( 'las la-calendar-plus' => 'Calendar Plus (add,calendar,create,date,event,new,positive,schedule,time,when)' ),
			array( 'lar la-calendar-plus' => 'Calendar Plus (add,calendar,create,date,event,new,positive,schedule,time,when)' ),
			array( 'las la-calendar-times' => 'Calendar Times (archive,calendar,date,delete,event,remove,schedule,time,when,x)' ),
			array( 'lar la-calendar-times' => 'Calendar Times (archive,calendar,date,delete,event,remove,schedule,time,when,x)' ),
			array( 'las la-clock' => 'Clock (date,late,schedule,time,timer,timestamp,watch)' ),
			array( 'lar la-clock' => 'Clock (date,late,schedule,time,timer,timestamp,watch)' ),
			array( 'las la-hourglass' => 'Hourglass (hour,minute,sand,stopwatch,time)' ),
			array( 'lar la-hourglass' => 'Hourglass (hour,minute,sand,stopwatch,time)' ),
			array( 'las la-hourglass-end' => 'Hourglass End (hour,minute,sand,stopwatch,time)' ),
			array( 'las la-hourglass-half' => 'Hourglass Half (hour,minute,sand,stopwatch,time)' ),
			array( 'las la-hourglass-start' => 'Hourglass Start (hour,minute,sand,stopwatch,time)' ),
			array( 'las la-stopwatch' => 'Stopwatch (clock,reminder,time)' ),
		),
		'Design' => array(
			array( 'las la-adjust' => 'adjust (contrast,dark,light,saturation)' ),
			array( 'las la-bezier-curve' => 'Bezier Curve (curves,illustrator,lines,path,vector)' ),
			array( 'las la-brush' => 'Brush (art,bristles,color,handle,paint)' ),
			array( 'las la-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'lar la-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'las la-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'lar la-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'las la-crop' => 'crop (design,frame,mask,resize,shrink)' ),
			array( 'las la-crop-alt' => 'Alternate Crop (design,frame,mask,resize,shrink)' ),
			array( 'las la-crosshairs' => 'Crosshairs (aim,bullseye,gpd,picker,position)' ),
			array( 'las la-cut' => 'Cut (clip,scissors,snip)' ),
			array( 'las la-drafting-compass' => 'Drafting Compass (design,map,mechanical drawing,plot,plotting)' ),
			array( 'las la-draw-polygon' => 'Draw Polygon (anchors,lines,object,render,shape)' ),
			array( 'las la-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'lar la-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'las la-eraser' => 'eraser (art,delete,remove,rubber)' ),
			array( 'las la-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'lar la-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'las la-eye-dropper' => 'Eye Dropper (beaker,clone,color,copy,eyedropper,pipette)' ),
			array( 'las la-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'lar la-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'las la-fill' => 'Fill (bucket,color,paint,paint bucket)' ),
			array( 'las la-fill-drip' => 'Fill Drip (bucket,color,drop,paint,paint bucket,spill)' ),
			array( 'las la-highlighter' => 'Highlighter (edit,marker,sharpie,update,write)' ),
			array( 'las la-icons' => 'Icons (bolt,emoji,heart,image,music,photo,symbols)' ),
			array( 'las la-layer-group' => 'Layer Group (arrange,develop,layers,map,stack)' ),
			array( 'las la-magic' => 'magic (autocomplete,automatic,mage,magic,spell,wand,witch,wizard)' ),
			array( 'las la-marker' => 'Marker (design,edit,sharpie,update,write)' ),
			array( 'las la-object-group' => 'Object Group (combine,copy,design,merge,select)' ),
			array( 'lar la-object-group' => 'Object Group (combine,copy,design,merge,select)' ),
			array( 'las la-object-ungroup' => 'Object Ungroup (copy,design,merge,select,separate)' ),
			array( 'lar la-object-ungroup' => 'Object Ungroup (copy,design,merge,select,separate)' ),
			array( 'las la-paint-brush' => 'Paint Brush (acrylic,art,brush,color,fill,paint,pigment,watercolor)' ),
			array( 'las la-paint-roller' => 'Paint Roller (acrylic,art,brush,color,fill,paint,pigment,watercolor)' ),
			array( 'las la-palette' => 'Palette (acrylic,art,brush,color,fill,paint,pigment,watercolor)' ),
			array( 'las la-paste' => 'Paste (clipboard,copy,document,paper)' ),
			array( 'las la-pen' => 'Pen (design,edit,update,write)' ),
			array( 'las la-pen-alt' => 'Alternate Pen (design,edit,update,write)' ),
			array( 'las la-pen-fancy' => 'Pen Fancy (design,edit,fountain pen,update,write)' ),
			array( 'las la-pen-nib' => 'Pen Nib (design,edit,fountain pen,update,write)' ),
			array( 'las la-pencil-alt' => 'Alternate Pencil (design,edit,pencil,update,write)' ),
			array( 'las la-pencil-ruler' => 'Pencil Ruler (design,draft,draw,pencil)' ),
			array( 'las la-ruler-combined' => 'Ruler Combined (design,draft,length,measure,planning)' ),
			array( 'las la-ruler-horizontal' => 'Ruler Horizontal (design,draft,length,measure,planning)' ),
			array( 'las la-ruler-vertical' => 'Ruler Vertical (design,draft,length,measure,planning)' ),
			array( 'las la-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'lar la-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'las la-splotch' => 'Splotch (Ink,blob,blotch,glob,stain)' ),
			array( 'las la-spray-can' => 'Spray Can (Paint,aerosol,design,graffiti,tag)' ),
			array( 'las la-stamp' => 'Stamp (art,certificate,imprint,rubber,seal)' ),
			array( 'las la-swatchbook' => 'Swatchbook (Pantone,color,design,hue,palette)' ),
			array( 'las la-tint' => 'tint (color,drop,droplet,raindrop,waterdrop)' ),
			array( 'las la-tint-slash' => 'Tint Slash (color,drop,droplet,raindrop,waterdrop)' ),
			array( 'las la-vector-square' => 'Vector Square (anchors,lines,object,render,shape)' ),
		),
		'Editors' => array(
			array( 'las la-align-center' => 'align-center (format,middle,paragraph,text)' ),
			array( 'las la-align-justify' => 'align-justify (format,paragraph,text)' ),
			array( 'las la-align-left' => 'align-left (format,paragraph,text)' ),
			array( 'las la-align-right' => 'align-right (format,paragraph,text)' ),
			array( 'las la-bold' => 'bold (emphasis,format,text)' ),
			array( 'las la-border-all' => 'Border All (cell,grid,outline,stroke,table)' ),
			array( 'las la-border-none' => 'Border None (cell,grid,outline,stroke,table)' ),
			array( 'las la-border-style' => 'Border Style' ),
			array( 'las la-clipboard' => 'Clipboard (copy,notes,paste,record)' ),
			array( 'lar la-clipboard' => 'Clipboard (copy,notes,paste,record)' ),
			array( 'las la-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'lar la-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'las la-columns' => 'Columns (browser,dashboard,organize,panes,split)' ),
			array( 'las la-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'lar la-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'las la-cut' => 'Cut (clip,scissors,snip)' ),
			array( 'las la-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'lar la-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'las la-eraser' => 'eraser (art,delete,remove,rubber)' ),
			array( 'las la-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'lar la-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'las la-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'lar la-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'las la-font' => 'font (alphabet,glyph,text,type,typeface)' ),
			array( 'las la-glasses' => 'Glasses (hipster,nerd,reading,sight,spectacles,vision)' ),
			array( 'las la-heading' => 'heading (format,header,text,title)' ),
			array( 'las la-highlighter' => 'Highlighter (edit,marker,sharpie,update,write)' ),
			array( 'las la-i-cursor' => 'I Beam Cursor (editing,i-beam,type,writing)' ),
			array( 'las la-icons' => 'Icons (bolt,emoji,heart,image,music,photo,symbols)' ),
			array( 'las la-indent' => 'Indent (align,justify,paragraph,tab)' ),
			array( 'las la-italic' => 'italic (edit,emphasis,font,format,text,type)' ),
			array( 'las la-link' => 'Link (attach,attachment,chain,connect)' ),
			array( 'las la-list' => 'List (checklist,completed,done,finished,ol,todo,ul)' ),
			array( 'las la-list-alt' => 'Alternate List (checklist,completed,done,finished,ol,todo,ul)' ),
			array( 'lar la-list-alt' => 'Alternate List (checklist,completed,done,finished,ol,todo,ul)' ),
			array( 'las la-list-ol' => 'list-ol (checklist,completed,done,finished,numbers,ol,todo,ul)' ),
			array( 'las la-list-ul' => 'list-ul (checklist,completed,done,finished,ol,todo,ul)' ),
			array( 'las la-marker' => 'Marker (design,edit,sharpie,update,write)' ),
			array( 'las la-outdent' => 'Outdent (align,justify,paragraph,tab)' ),
			array( 'las la-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'lar la-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'las la-paperclip' => 'Paperclip (attach,attachment,connect,link)' ),
			array( 'las la-paragraph' => 'paragraph (edit,format,text,writing)' ),
			array( 'las la-paste' => 'Paste (clipboard,copy,document,paper)' ),
			array( 'las la-pen' => 'Pen (design,edit,update,write)' ),
			array( 'las la-pen-alt' => 'Alternate Pen (design,edit,update,write)' ),
			array( 'las la-pen-fancy' => 'Pen Fancy (design,edit,fountain pen,update,write)' ),
			array( 'las la-pen-nib' => 'Pen Nib (design,edit,fountain pen,update,write)' ),
			array( 'las la-pencil-alt' => 'Alternate Pencil (design,edit,pencil,update,write)' ),
			array( 'las la-print' => 'print (business,copy,document,office,paper)' ),
			array( 'las la-quote-left' => 'quote-left (mention,note,phrase,text,type)' ),
			array( 'las la-quote-right' => 'quote-right (mention,note,phrase,text,type)' ),
			array( 'las la-redo' => 'Redo (forward,refresh,reload,repeat)' ),
			array( 'las la-redo-alt' => 'Alternate Redo (forward,refresh,reload,repeat)' ),
			array( 'las la-remove-format' => 'Remove Format (cancel,font,format,remove,style,text)' ),
			array( 'las la-reply' => 'Reply (mail,message,respond)' ),
			array( 'las la-reply-all' => 'reply-all (mail,message,respond)' ),
			array( 'las la-screwdriver' => 'Screwdriver (admin,fix,mechanic,repair,settings,tool)' ),
			array( 'las la-share' => 'Share (forward,save,send,social)' ),
			array( 'las la-spell-check' => 'Spell Check (dictionary,edit,editor,grammar,text)' ),
			array( 'las la-strikethrough' => 'Strikethrough (cancel,edit,font,format,text,type)' ),
			array( 'las la-subscript' => 'subscript (edit,font,format,text,type)' ),
			array( 'las la-superscript' => 'superscript (edit,exponential,font,format,text,type)' ),
			array( 'las la-sync' => 'Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'las la-sync-alt' => 'Alternate Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'las la-table' => 'table (data,excel,spreadsheet)' ),
			array( 'las la-tasks' => 'Tasks (checklist,downloading,downloads,loading,progress,project management,settings,to do)' ),
			array( 'las la-text-height' => 'text-height (edit,font,format,text,type)' ),
			array( 'las la-text-width' => 'Text Width (edit,font,format,text,type)' ),
			array( 'las la-th' => 'th (blocks,boxes,grid,squares)' ),
			array( 'las la-th-large' => 'th-large (blocks,boxes,grid,squares)' ),
			array( 'las la-th-list' => 'th-list (checklist,completed,done,finished,ol,todo,ul)' ),
			array( 'las la-tools' => 'Tools (admin,fix,repair,screwdriver,settings,tools,wrench)' ),
			array( 'las la-trash' => 'Trash (delete,garbage,hide,remove)' ),
			array( 'las la-trash-alt' => 'Alternate Trash (delete,garbage,hide,remove,trash-o)' ),
			array( 'lar la-trash-alt' => 'Alternate Trash (delete,garbage,hide,remove,trash-o)' ),
			array( 'las la-trash-restore' => 'Trash Restore (back,control z,oops,undo)' ),
			array( 'las la-trash-restore-alt' => 'Alternative Trash Restore (back,control z,oops,undo)' ),
			array( 'las la-underline' => 'Underline (edit,emphasis,format,text,writing)' ),
			array( 'las la-undo' => 'Undo (back,control z,exchange,oops,return,rotate,swap)' ),
			array( 'las la-undo-alt' => 'Alternate Undo (back,control z,exchange,oops,return,swap)' ),
			array( 'las la-unlink' => 'unlink (attachment,chain,chain-broken,remove)' ),
			array( 'las la-wrench' => 'Wrench (construction,fix,mechanic,plumbing,settings,spanner,tool,update)' ),
		),
		'Education' => array(
			array( 'las la-apple-alt' => 'Fruit Apple (fall,fruit,fuji,macintosh,orchard,seasonal,vegan)' ),
			array( 'las la-atom' => 'Atom (atheism,chemistry,ion,nuclear,science)' ),
			array( 'las la-award' => 'Award (honor,praise,prize,recognition,ribbon,trophy)' ),
			array( 'las la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'lar la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'las la-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'lar la-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'las la-book-open' => 'Book Open (flyer,library,notebook,open book,pamphlet,reading)' ),
			array( 'las la-book-reader' => 'Book Reader (flyer,library,notebook,open book,pamphlet,reading)' ),
			array( 'las la-chalkboard' => 'Chalkboard (blackboard,learning,school,teaching,whiteboard,writing)' ),
			array( 'las la-chalkboard-teacher' => 'Chalkboard Teacher (blackboard,instructor,learning,professor,school,whiteboard,writing)' ),
			array( 'las la-graduation-cap' => 'Graduation Cap (ceremony,college,graduate,learning,school,student)' ),
			array( 'las la-laptop-code' => 'Laptop Code (computer,cpu,dell,demo,develop,device,mac,macbook,machine,pc)' ),
			array( 'las la-microscope' => 'Microscope (electron,lens,optics,science,shrink)' ),
			array( 'las la-music' => 'Music (lyrics,melody,note,sing,sound)' ),
			array( 'las la-school' => 'School (building,education,learn,student,teacher)' ),
			array( 'las la-shapes' => 'Shapes (blocks,build,circle,square,triangle)' ),
			array( 'las la-theater-masks' => 'Theater Masks (comedy,perform,theatre,tragedy)' ),
			array( 'las la-user-graduate' => 'User Graduate (cap,clothing,commencement,gown,graduation,person,student)' ),
		),
		'Emoji' => array(
			array( 'las la-angry' => 'Angry Face (disapprove,emoticon,face,mad,upset)' ),
			array( 'lar la-angry' => 'Angry Face (disapprove,emoticon,face,mad,upset)' ),
			array( 'las la-dizzy' => 'Dizzy Face (dazed,dead,disapprove,emoticon,face)' ),
			array( 'lar la-dizzy' => 'Dizzy Face (dazed,dead,disapprove,emoticon,face)' ),
			array( 'las la-flushed' => 'Flushed Face (embarrassed,emoticon,face)' ),
			array( 'lar la-flushed' => 'Flushed Face (embarrassed,emoticon,face)' ),
			array( 'las la-frown' => 'Frowning Face (disapprove,emoticon,face,rating,sad)' ),
			array( 'lar la-frown' => 'Frowning Face (disapprove,emoticon,face,rating,sad)' ),
			array( 'las la-frown-open' => 'Frowning Face With Open Mouth (disapprove,emoticon,face,rating,sad)' ),
			array( 'lar la-frown-open' => 'Frowning Face With Open Mouth (disapprove,emoticon,face,rating,sad)' ),
			array( 'las la-grimace' => 'Grimacing Face (cringe,emoticon,face,teeth)' ),
			array( 'lar la-grimace' => 'Grimacing Face (cringe,emoticon,face,teeth)' ),
			array( 'las la-grin' => 'Grinning Face (emoticon,face,laugh,smile)' ),
			array( 'lar la-grin' => 'Grinning Face (emoticon,face,laugh,smile)' ),
			array( 'las la-grin-alt' => 'Alternate Grinning Face (emoticon,face,laugh,smile)' ),
			array( 'lar la-grin-alt' => 'Alternate Grinning Face (emoticon,face,laugh,smile)' ),
			array( 'las la-grin-beam' => 'Grinning Face With Smiling Eyes (emoticon,face,laugh,smile)' ),
			array( 'lar la-grin-beam' => 'Grinning Face With Smiling Eyes (emoticon,face,laugh,smile)' ),
			array( 'las la-grin-beam-sweat' => 'Grinning Face With Sweat (embarass,emoticon,face,smile)' ),
			array( 'lar la-grin-beam-sweat' => 'Grinning Face With Sweat (embarass,emoticon,face,smile)' ),
			array( 'las la-grin-hearts' => 'Smiling Face With Heart-Eyes (emoticon,face,love,smile)' ),
			array( 'lar la-grin-hearts' => 'Smiling Face With Heart-Eyes (emoticon,face,love,smile)' ),
			array( 'las la-grin-squint' => 'Grinning Squinting Face (emoticon,face,laugh,smile)' ),
			array( 'lar la-grin-squint' => 'Grinning Squinting Face (emoticon,face,laugh,smile)' ),
			array( 'las la-grin-squint-tears' => 'Rolling on the Floor Laughing (emoticon,face,happy,smile)' ),
			array( 'lar la-grin-squint-tears' => 'Rolling on the Floor Laughing (emoticon,face,happy,smile)' ),
			array( 'las la-grin-stars' => 'Star-Struck (emoticon,face,star-struck)' ),
			array( 'lar la-grin-stars' => 'Star-Struck (emoticon,face,star-struck)' ),
			array( 'las la-grin-tears' => 'Face With Tears of Joy (LOL,emoticon,face)' ),
			array( 'lar la-grin-tears' => 'Face With Tears of Joy (LOL,emoticon,face)' ),
			array( 'las la-grin-tongue' => 'Face With Tongue (LOL,emoticon,face)' ),
			array( 'lar la-grin-tongue' => 'Face With Tongue (LOL,emoticon,face)' ),
			array( 'las la-grin-tongue-squint' => 'Squinting Face With Tongue (LOL,emoticon,face)' ),
			array( 'lar la-grin-tongue-squint' => 'Squinting Face With Tongue (LOL,emoticon,face)' ),
			array( 'las la-grin-tongue-wink' => 'Winking Face With Tongue (LOL,emoticon,face)' ),
			array( 'lar la-grin-tongue-wink' => 'Winking Face With Tongue (LOL,emoticon,face)' ),
			array( 'las la-grin-wink' => 'Grinning Winking Face (emoticon,face,flirt,laugh,smile)' ),
			array( 'lar la-grin-wink' => 'Grinning Winking Face (emoticon,face,flirt,laugh,smile)' ),
			array( 'las la-kiss' => 'Kissing Face (beso,emoticon,face,love,smooch)' ),
			array( 'lar la-kiss' => 'Kissing Face (beso,emoticon,face,love,smooch)' ),
			array( 'las la-kiss-beam' => 'Kissing Face With Smiling Eyes (beso,emoticon,face,love,smooch)' ),
			array( 'lar la-kiss-beam' => 'Kissing Face With Smiling Eyes (beso,emoticon,face,love,smooch)' ),
			array( 'las la-kiss-wink-heart' => 'Face Blowing a Kiss (beso,emoticon,face,love,smooch)' ),
			array( 'lar la-kiss-wink-heart' => 'Face Blowing a Kiss (beso,emoticon,face,love,smooch)' ),
			array( 'las la-laugh' => 'Grinning Face With Big Eyes (LOL,emoticon,face,laugh,smile)' ),
			array( 'lar la-laugh' => 'Grinning Face With Big Eyes (LOL,emoticon,face,laugh,smile)' ),
			array( 'las la-laugh-beam' => 'Laugh Face with Beaming Eyes (LOL,emoticon,face,happy,smile)' ),
			array( 'lar la-laugh-beam' => 'Laugh Face with Beaming Eyes (LOL,emoticon,face,happy,smile)' ),
			array( 'las la-laugh-squint' => 'Laughing Squinting Face (LOL,emoticon,face,happy,smile)' ),
			array( 'lar la-laugh-squint' => 'Laughing Squinting Face (LOL,emoticon,face,happy,smile)' ),
			array( 'las la-laugh-wink' => 'Laughing Winking Face (LOL,emoticon,face,happy,smile)' ),
			array( 'lar la-laugh-wink' => 'Laughing Winking Face (LOL,emoticon,face,happy,smile)' ),
			array( 'las la-meh' => 'Neutral Face (emoticon,face,neutral,rating)' ),
			array( 'lar la-meh' => 'Neutral Face (emoticon,face,neutral,rating)' ),
			array( 'las la-meh-blank' => 'Face Without Mouth (emoticon,face,neutral,rating)' ),
			array( 'lar la-meh-blank' => 'Face Without Mouth (emoticon,face,neutral,rating)' ),
			array( 'las la-meh-rolling-eyes' => 'Face With Rolling Eyes (emoticon,face,neutral,rating)' ),
			array( 'lar la-meh-rolling-eyes' => 'Face With Rolling Eyes (emoticon,face,neutral,rating)' ),
			array( 'las la-sad-cry' => 'Crying Face (emoticon,face,tear,tears)' ),
			array( 'lar la-sad-cry' => 'Crying Face (emoticon,face,tear,tears)' ),
			array( 'las la-sad-tear' => 'Loudly Crying Face (emoticon,face,tear,tears)' ),
			array( 'lar la-sad-tear' => 'Loudly Crying Face (emoticon,face,tear,tears)' ),
			array( 'las la-smile' => 'Smiling Face (approve,emoticon,face,happy,rating,satisfied)' ),
			array( 'lar la-smile' => 'Smiling Face (approve,emoticon,face,happy,rating,satisfied)' ),
			array( 'las la-smile-beam' => 'Beaming Face With Smiling Eyes (emoticon,face,happy,positive)' ),
			array( 'lar la-smile-beam' => 'Beaming Face With Smiling Eyes (emoticon,face,happy,positive)' ),
			array( 'las la-smile-wink' => 'Winking Face (emoticon,face,happy,hint,joke)' ),
			array( 'lar la-smile-wink' => 'Winking Face (emoticon,face,happy,hint,joke)' ),
			array( 'las la-surprise' => 'Hushed Face (emoticon,face,shocked)' ),
			array( 'lar la-surprise' => 'Hushed Face (emoticon,face,shocked)' ),
			array( 'las la-tired' => 'Tired Face (angry,emoticon,face,grumpy,upset)' ),
			array( 'lar la-tired' => 'Tired Face (angry,emoticon,face,grumpy,upset)' ),
		),
		'Energy' => array(
			array( 'las la-atom' => 'Atom (atheism,chemistry,ion,nuclear,science)' ),
			array( 'las la-battery-empty' => 'Battery Empty (charge,dead,power,status)' ),
			array( 'las la-battery-full' => 'Battery Full (charge,power,status)' ),
			array( 'las la-battery-half' => 'Battery 1/2 Full (charge,power,status)' ),
			array( 'las la-battery-quarter' => 'Battery 1/4 Full (charge,low,power,status)' ),
			array( 'las la-battery-three-quarters' => 'Battery 3/4 Full (charge,power,status)' ),
			array( 'las la-broadcast-tower' => 'Broadcast Tower (airwaves,antenna,radio,reception,waves)' ),
			array( 'las la-burn' => 'Burn (caliente,energy,fire,flame,gas,heat,hot)' ),
			array( 'las la-charging-station' => 'Charging Station (electric,ev,tesla,vehicle)' ),
			array( 'las la-fire' => 'fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'las la-fire-alt' => 'Alternate Fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'las la-gas-pump' => 'Gas Pump (car,fuel,gasoline,petrol)' ),
			array( 'las la-industry' => 'Industry (building,factory,industrial,manufacturing,mill,warehouse)' ),
			array( 'las la-leaf' => 'leaf (eco,flora,nature,plant,vegan)' ),
			array( 'las la-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'lar la-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'las la-plug' => 'Plug (connect,electric,online,power)' ),
			array( 'las la-poop' => 'Poop (crap,poop,shit,smile,turd)' ),
			array( 'las la-power-off' => 'Power Off (cancel,computer,on,reboot,restart)' ),
			array( 'las la-radiation' => 'Radiation (danger,dangerous,deadly,hazard,nuclear,radioactive,warning)' ),
			array( 'las la-radiation-alt' => 'Alternate Radiation (danger,dangerous,deadly,hazard,nuclear,radioactive,warning)' ),
			array( 'las la-seedling' => 'Seedling (flora,grow,plant,vegan)' ),
			array( 'las la-solar-panel' => 'Solar Panel (clean,eco-friendly,energy,green,sun)' ),
			array( 'las la-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'lar la-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'las la-water' => 'Water (lake,liquid,ocean,sea,swim,wet)' ),
			array( 'las la-wind' => 'Wind (air,blow,breeze,fall,seasonal,weather)' ),
		),
		'Files' => array(
			array( 'las la-archive' => 'Archive (box,package,save,storage)' ),
			array( 'las la-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'lar la-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'las la-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'lar la-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'las la-cut' => 'Cut (clip,scissors,snip)' ),
			array( 'las la-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'lar la-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'las la-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'lar la-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'las la-file-archive' => 'Archive File (.zip,bundle,compress,compression,download,zip)' ),
			array( 'lar la-file-archive' => 'Archive File (.zip,bundle,compress,compression,download,zip)' ),
			array( 'las la-file-audio' => 'Audio File (document,mp3,music,page,play,sound)' ),
			array( 'lar la-file-audio' => 'Audio File (document,mp3,music,page,play,sound)' ),
			array( 'las la-file-code' => 'Code File (css,development,document,html)' ),
			array( 'lar la-file-code' => 'Code File (css,development,document,html)' ),
			array( 'las la-file-excel' => 'Excel File (csv,document,numbers,spreadsheets,table)' ),
			array( 'lar la-file-excel' => 'Excel File (csv,document,numbers,spreadsheets,table)' ),
			array( 'las la-file-image' => 'Image File (document,image,jpg,photo,png)' ),
			array( 'lar la-file-image' => 'Image File (document,image,jpg,photo,png)' ),
			array( 'las la-file-pdf' => 'PDF File (acrobat,document,preview,save)' ),
			array( 'lar la-file-pdf' => 'PDF File (acrobat,document,preview,save)' ),
			array( 'las la-file-powerpoint' => 'Powerpoint File (display,document,keynote,presentation)' ),
			array( 'lar la-file-powerpoint' => 'Powerpoint File (display,document,keynote,presentation)' ),
			array( 'las la-file-video' => 'Video File (document,m4v,movie,mp4,play)' ),
			array( 'lar la-file-video' => 'Video File (document,m4v,movie,mp4,play)' ),
			array( 'las la-file-word' => 'Word File (document,edit,page,text,writing)' ),
			array( 'lar la-file-word' => 'Word File (document,edit,page,text,writing)' ),
			array( 'las la-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'lar la-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'las la-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'lar la-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'las la-paste' => 'Paste (clipboard,copy,document,paper)' ),
			array( 'las la-photo-video' => 'Photo Video (av,film,image,library,media)' ),
			array( 'las la-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'lar la-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'las la-sticky-note' => 'Sticky Note (message,note,paper,reminder,sticker)' ),
			array( 'lar la-sticky-note' => 'Sticky Note (message,note,paper,reminder,sticker)' ),
		),
		'Finance' => array(
			array( 'las la-balance-scale' => 'Balance Scale (balanced,justice,legal,measure,weight)' ),
			array( 'las la-balance-scale-left' => 'Balance Scale (Left-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'las la-balance-scale-right' => 'Balance Scale (Right-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'las la-book' => 'book (diary,documentation,journal,library,read)' ),
			array( 'las la-cash-register' => 'Cash Register (buy,cha-ching,change,checkout,commerce,leaerboard,machine,pay,payment,purchase,store)' ),
			array( 'las la-chart-line' => 'Line Chart (activity,analytics,chart,dashboard,gain,graph,increase,line)' ),
			array( 'las la-chart-pie' => 'Pie Chart (analytics,chart,diagram,graph,pie)' ),
			array( 'las la-coins' => 'Coins (currency,dime,financial,gold,money,penny)' ),
			array( 'las la-comment-dollar' => 'Comment Dollar (bubble,chat,commenting,conversation,feedback,message,money,note,notification,pay,sms,speech,spend,texting,transfer)' ),
			array( 'las la-comments-dollar' => 'Comments Dollar (bubble,chat,commenting,conversation,feedback,message,money,note,notification,pay,sms,speech,spend,texting,transfer)' ),
			array( 'las la-credit-card' => 'Credit Card (buy,checkout,credit-card-alt,debit,money,payment,purchase)' ),
			array( 'lar la-credit-card' => 'Credit Card (buy,checkout,credit-card-alt,debit,money,payment,purchase)' ),
			array( 'las la-donate' => 'Donate (contribute,generosity,gift,give)' ),
			array( 'las la-file-invoice' => 'File Invoice (account,bill,charge,document,payment,receipt)' ),
			array( 'las la-file-invoice-dollar' => 'File Invoice with US Dollar ($,account,bill,charge,document,dollar-sign,money,payment,receipt,usd)' ),
			array( 'las la-hand-holding-usd' => 'Hand Holding US Dollar ($,carry,dollar sign,donation,giving,lift,money,price)' ),
			array( 'las la-landmark' => 'Landmark (building,historic,memorable,monument,politics)' ),
			array( 'las la-money-bill' => 'Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'las la-money-bill-alt' => 'Alternate Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'lar la-money-bill-alt' => 'Alternate Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'las la-money-bill-wave' => 'Wavy Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'las la-money-bill-wave-alt' => 'Alternate Wavy Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'las la-money-check' => 'Money Check (bank check,buy,checkout,cheque,money,payment,price,purchase)' ),
			array( 'las la-money-check-alt' => 'Alternate Money Check (bank check,buy,checkout,cheque,money,payment,price,purchase)' ),
			array( 'las la-percentage' => 'Percentage (discount,fraction,proportion,rate,ratio)' ),
			array( 'las la-piggy-bank' => 'Piggy Bank (bank,save,savings)' ),
			array( 'las la-receipt' => 'Receipt (check,invoice,money,pay,table)' ),
			array( 'las la-stamp' => 'Stamp (art,certificate,imprint,rubber,seal)' ),
			array( 'las la-wallet' => 'Wallet (billfold,cash,currency,money)' ),
		),
		'Fitness' => array(
			array( 'las la-bicycle' => 'Bicycle (bike,gears,pedal,transportation,vehicle)' ),
			array( 'las la-biking' => 'Biking (bicycle,bike,cycle,cycling,ride,wheel)' ),
			array( 'las la-burn' => 'Burn (caliente,energy,fire,flame,gas,heat,hot)' ),
			array( 'las la-fire-alt' => 'Alternate Fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'las la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'lar la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'las la-heartbeat' => 'Heartbeat (ekg,electrocardiogram,health,lifeline,vital signs)' ),
			array( 'las la-hiking' => 'Hiking (activity,backpack,fall,fitness,outdoors,person,seasonal,walking)' ),
			array( 'las la-running' => 'Running (exercise,health,jog,person,run,sport,sprint)' ),
			array( 'las la-shoe-prints' => 'Shoe Prints (feet,footprints,steps,walk)' ),
			array( 'las la-skating' => 'Skating (activity,figure skating,fitness,ice,person,winter)' ),
			array( 'las la-skiing' => 'Skiing (activity,downhill,fast,fitness,olympics,outdoors,person,seasonal,slalom)' ),
			array( 'las la-skiing-nordic' => 'Skiing Nordic (activity,cross country,fitness,outdoors,person,seasonal)' ),
			array( 'las la-snowboarding' => 'Snowboarding (activity,fitness,olympics,outdoors,person)' ),
			array( 'las la-spa' => 'Spa (flora,massage,mindfulness,plant,wellness)' ),
			array( 'las la-swimmer' => 'Swimmer (athlete,head,man,olympics,person,pool,water)' ),
			array( 'las la-walking' => 'Walking (exercise,health,pedometer,person,steps)' ),
		),
		'Food' => array(
			array( 'las la-apple-alt' => 'Fruit Apple (fall,fruit,fuji,macintosh,orchard,seasonal,vegan)' ),
			array( 'las la-bacon' => 'Bacon (blt,breakfast,ham,lard,meat,pancetta,pork,rasher)' ),
			array( 'las la-bone' => 'Bone (calcium,dog,skeletal,skeleton,tibia)' ),
			array( 'las la-bread-slice' => 'Bread Slice (bake,bakery,baking,dough,flour,gluten,grain,sandwich,sourdough,toast,wheat,yeast)' ),
			array( 'las la-candy-cane' => 'Candy Cane (candy,christmas,holiday,mint,peppermint,striped,xmas)' ),
			array( 'las la-carrot' => 'Carrot (bugs bunny,orange,vegan,vegetable)' ),
			array( 'las la-cheese' => 'Cheese (cheddar,curd,gouda,melt,parmesan,sandwich,swiss,wedge)' ),
			array( 'las la-cloud-meatball' => 'Cloud with (a chance of) Meatball (FLDSMDFR,food,spaghetti,storm)' ),
			array( 'las la-cookie' => 'Cookie (baked good,chips,chocolate,eat,snack,sweet,treat)' ),
			array( 'las la-drumstick-bite' => 'Drumstick with Bite Taken Out (bone,chicken,leg,meat,poultry,turkey)' ),
			array( 'las la-egg' => 'Egg (breakfast,chicken,easter,shell,yolk)' ),
			array( 'las la-fish' => 'Fish (fauna,gold,seafood,swimming)' ),
			array( 'las la-hamburger' => 'Hamburger (bacon,beef,burger,burger king,cheeseburger,fast food,grill,ground beef,mcdonalds,sandwich)' ),
			array( 'las la-hotdog' => 'Hot Dog (bun,chili,frankfurt,frankfurter,kosher,polish,sandwich,sausage,vienna,weiner)' ),
			array( 'las la-ice-cream' => 'Ice Cream (chocolate,cone,dessert,frozen,scoop,sorbet,vanilla,yogurt)' ),
			array( 'las la-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'lar la-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'las la-pepper-hot' => 'Hot Pepper (buffalo wings,capsicum,chili,chilli,habanero,jalapeno,mexican,spicy,tabasco,vegetable)' ),
			array( 'las la-pizza-slice' => 'Pizza Slice (cheese,chicago,italian,mozzarella,new york,pepperoni,pie,slice,teenage mutant ninja turtles,tomato)' ),
			array( 'las la-seedling' => 'Seedling (flora,grow,plant,vegan)' ),
			array( 'las la-stroopwafel' => 'Stroopwafel (caramel,cookie,dessert,sweets,waffle)' ),
		),
		'Fruits & Vegetables' => array(
			array( 'las la-apple-alt' => 'Fruit Apple (fall,fruit,fuji,macintosh,orchard,seasonal,vegan)' ),
			array( 'las la-carrot' => 'Carrot (bugs bunny,orange,vegan,vegetable)' ),
			array( 'las la-leaf' => 'leaf (eco,flora,nature,plant,vegan)' ),
			array( 'las la-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'lar la-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'las la-pepper-hot' => 'Hot Pepper (buffalo wings,capsicum,chili,chilli,habanero,jalapeno,mexican,spicy,tabasco,vegetable)' ),
			array( 'las la-seedling' => 'Seedling (flora,grow,plant,vegan)' ),
		),
		'Games' => array(
			array( 'las la-chess' => 'Chess (board,castle,checkmate,game,king,rook,strategy,tournament)' ),
			array( 'las la-chess-bishop' => 'Chess Bishop (board,checkmate,game,strategy)' ),
			array( 'las la-chess-board' => 'Chess Board (board,checkmate,game,strategy)' ),
			array( 'las la-chess-king' => 'Chess King (board,checkmate,game,strategy)' ),
			array( 'las la-chess-knight' => 'Chess Knight (board,checkmate,game,horse,strategy)' ),
			array( 'las la-chess-pawn' => 'Chess Pawn (board,checkmate,game,strategy)' ),
			array( 'las la-chess-queen' => 'Chess Queen (board,checkmate,game,strategy)' ),
			array( 'las la-chess-rook' => 'Chess Rook (board,castle,checkmate,game,strategy)' ),
			array( 'las la-dice' => 'Dice (chance,gambling,game,roll)' ),
			array( 'las la-dice-d20' => 'Dice D20 (Dungeons & Dragons,chance,d&d,dnd,fantasy,gambling,game,roll)' ),
			array( 'las la-dice-d6' => 'Dice D6 (Dungeons & Dragons,chance,d&d,dnd,fantasy,gambling,game,roll)' ),
			array( 'las la-dice-five' => 'Dice Five (chance,gambling,game,roll)' ),
			array( 'las la-dice-four' => 'Dice Four (chance,gambling,game,roll)' ),
			array( 'las la-dice-one' => 'Dice One (chance,gambling,game,roll)' ),
			array( 'las la-dice-six' => 'Dice Six (chance,gambling,game,roll)' ),
			array( 'las la-dice-three' => 'Dice Three (chance,gambling,game,roll)' ),
			array( 'las la-dice-two' => 'Dice Two (chance,gambling,game,roll)' ),
			array( 'las la-gamepad' => 'Gamepad (arcade,controller,d-pad,joystick,video,video game)' ),
			array( 'las la-ghost' => 'Ghost (apparition,blinky,clyde,floating,halloween,holiday,inky,pinky,spirit)' ),
			array( 'las la-headset' => 'Headset (audio,gamer,gaming,listen,live chat,microphone,shot caller,sound,support,telemarketer)' ),
			array( 'las la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'lar la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'lab la-playstation' => 'PlayStation' ),
			array( 'las la-puzzle-piece' => 'Puzzle Piece (add-on,addon,game,section)' ),
			array( 'lab la-steam' => 'Steam' ),
			array( 'lab la-steam-square' => 'Steam Square' ),
			array( 'lab la-steam-symbol' => 'Steam Symbol' ),
			array( 'lab la-twitch' => 'Twitch' ),
			array( 'lab la-xbox' => 'Xbox' ),
		),
		'Genders' => array(
			array( 'las la-genderless' => 'Genderless (androgynous,asexual,sexless)' ),
			array( 'las la-mars' => 'Mars (male)' ),
			array( 'las la-mars-double' => 'Mars Double' ),
			array( 'las la-mars-stroke' => 'Mars Stroke' ),
			array( 'las la-mars-stroke-h' => 'Mars Stroke Horizontal' ),
			array( 'las la-mars-stroke-v' => 'Mars Stroke Vertical' ),
			array( 'las la-mercury' => 'Mercury (transgender)' ),
			array( 'las la-neuter' => 'Neuter' ),
			array( 'las la-transgender' => 'Transgender (intersex)' ),
			array( 'las la-transgender-alt' => 'Alternate Transgender (intersex)' ),
			array( 'las la-venus' => 'Venus (female)' ),
			array( 'las la-venus-double' => 'Venus Double (female)' ),
			array( 'las la-venus-mars' => 'Venus Mars (Gender)' ),
		),
		'Halloween' => array(
			array( 'las la-book-dead' => 'Book of the Dead (Dungeons & Dragons,crossbones,d&d,dark arts,death,dnd,documentation,evil,fantasy,halloween,holiday,necronomicon,read,skull,spell)' ),
			array( 'las la-broom' => 'Broom (clean,firebolt,fly,halloween,nimbus 2000,quidditch,sweep,witch)' ),
			array( 'las la-cat' => 'Cat (feline,halloween,holiday,kitten,kitty,meow,pet)' ),
			array( 'las la-cloud-moon' => 'Cloud with Moon (crescent,evening,lunar,night,partly cloudy,sky)' ),
			array( 'las la-crow' => 'Crow (bird,bullfrog,fauna,halloween,holiday,toad)' ),
			array( 'las la-ghost' => 'Ghost (apparition,blinky,clyde,floating,halloween,holiday,inky,pinky,spirit)' ),
			array( 'las la-hat-wizard' => 'Wizard\'s Hat (Dungeons & Dragons,accessory,buckle,clothing,d&d,dnd,fantasy,halloween,head,holiday,mage,magic,pointy,witch)' ),
			array( 'las la-mask' => 'Mask (carnivale,costume,disguise,halloween,secret,super hero)' ),
			array( 'las la-skull-crossbones' => 'Skull & Crossbones (Dungeons & Dragons,alert,bones,d&d,danger,dead,deadly,death,dnd,fantasy,halloween,holiday,jolly-roger,pirate,poison,skeleton,warning)' ),
			array( 'las la-spider' => 'Spider (arachnid,bug,charlotte,crawl,eight,halloween)' ),
			array( 'las la-toilet-paper' => 'Toilet Paper (bathroom,halloween,holiday,lavatory,prank,restroom,roll)' ),
		),
		'Hands' => array(
			array( 'las la-allergies' => 'Allergies (allergy,freckles,hand,hives,pox,skin,spots)' ),
			array( 'las la-fist-raised' => 'Raised Fist (Dungeons & Dragons,d&d,dnd,fantasy,hand,ki,monk,resist,strength,unarmed combat)' ),
			array( 'las la-hand-holding' => 'Hand Holding (carry,lift)' ),
			array( 'las la-hand-holding-heart' => 'Hand Holding Heart (carry,charity,gift,lift,package)' ),
			array( 'las la-hand-holding-usd' => 'Hand Holding US Dollar ($,carry,dollar sign,donation,giving,lift,money,price)' ),
			array( 'las la-hand-lizard' => 'Lizard (Hand) (game,roshambo)' ),
			array( 'lar la-hand-lizard' => 'Lizard (Hand) (game,roshambo)' ),
			array( 'las la-hand-middle-finger' => 'Hand with Middle Finger Raised (flip the bird,gesture,hate,rude)' ),
			array( 'las la-hand-paper' => 'Paper (Hand) (game,halt,roshambo,stop)' ),
			array( 'lar la-hand-paper' => 'Paper (Hand) (game,halt,roshambo,stop)' ),
			array( 'las la-hand-peace' => 'Peace (Hand) (rest,truce)' ),
			array( 'lar la-hand-peace' => 'Peace (Hand) (rest,truce)' ),
			array( 'las la-hand-point-down' => 'Hand Pointing Down (finger,hand-o-down,point)' ),
			array( 'lar la-hand-point-down' => 'Hand Pointing Down (finger,hand-o-down,point)' ),
			array( 'las la-hand-point-left' => 'Hand Pointing Left (back,finger,hand-o-left,left,point,previous)' ),
			array( 'lar la-hand-point-left' => 'Hand Pointing Left (back,finger,hand-o-left,left,point,previous)' ),
			array( 'las la-hand-point-right' => 'Hand Pointing Right (finger,forward,hand-o-right,next,point,right)' ),
			array( 'lar la-hand-point-right' => 'Hand Pointing Right (finger,forward,hand-o-right,next,point,right)' ),
			array( 'las la-hand-point-up' => 'Hand Pointing Up (finger,hand-o-up,point)' ),
			array( 'lar la-hand-point-up' => 'Hand Pointing Up (finger,hand-o-up,point)' ),
			array( 'las la-hand-pointer' => 'Pointer (Hand) (arrow,cursor,select)' ),
			array( 'lar la-hand-pointer' => 'Pointer (Hand) (arrow,cursor,select)' ),
			array( 'las la-hand-rock' => 'Rock (Hand) (fist,game,roshambo)' ),
			array( 'lar la-hand-rock' => 'Rock (Hand) (fist,game,roshambo)' ),
			array( 'las la-hand-scissors' => 'Scissors (Hand) (cut,game,roshambo)' ),
			array( 'lar la-hand-scissors' => 'Scissors (Hand) (cut,game,roshambo)' ),
			array( 'las la-hand-spock' => 'Spock (Hand) (live long,prosper,salute,star trek,vulcan)' ),
			array( 'lar la-hand-spock' => 'Spock (Hand) (live long,prosper,salute,star trek,vulcan)' ),
			array( 'las la-hands' => 'Hands (carry,hold,lift)' ),
			array( 'las la-hands-helping' => 'Helping Hands (aid,assistance,handshake,partnership,volunteering)' ),
			array( 'las la-handshake' => 'Handshake (agreement,greeting,meeting,partnership)' ),
			array( 'lar la-handshake' => 'Handshake (agreement,greeting,meeting,partnership)' ),
			array( 'las la-praying-hands' => 'Praying Hands (kneel,preach,religion,worship)' ),
			array( 'las la-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'lar la-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'las la-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
			array( 'lar la-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
		),
		'Health' => array(
			array( 'lab la-accessible-icon' => 'Accessible Icon (accessibility,handicap,person,wheelchair,wheelchair-alt)' ),
			array( 'las la-ambulance' => 'ambulance (emergency,emt,er,help,hospital,support,vehicle)' ),
			array( 'las la-h-square' => 'H Square (directions,emergency,hospital,hotel,map)' ),
			array( 'las la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'lar la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'las la-heartbeat' => 'Heartbeat (ekg,electrocardiogram,health,lifeline,vital signs)' ),
			array( 'las la-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'lar la-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'las la-medkit' => 'medkit (first aid,firstaid,health,help,support)' ),
			array( 'las la-plus-square' => 'Plus Square (add,create,expand,new,positive,shape)' ),
			array( 'lar la-plus-square' => 'Plus Square (add,create,expand,new,positive,shape)' ),
			array( 'las la-prescription' => 'Prescription (drugs,medical,medicine,pharmacy,rx)' ),
			array( 'las la-stethoscope' => 'Stethoscope (diagnosis,doctor,general practitioner,hospital,infirmary,medicine,office,outpatient)' ),
			array( 'las la-user-md' => 'Doctor (job,medical,nurse,occupation,physician,profile,surgeon)' ),
			array( 'las la-wheelchair' => 'Wheelchair (accessible,handicap,person)' ),
		),
		'Holiday' => array(
			array( 'las la-candy-cane' => 'Candy Cane (candy,christmas,holiday,mint,peppermint,striped,xmas)' ),
			array( 'las la-carrot' => 'Carrot (bugs bunny,orange,vegan,vegetable)' ),
			array( 'las la-cookie-bite' => 'Cookie Bite (baked good,bitten,chips,chocolate,eat,snack,sweet,treat)' ),
			array( 'las la-gift' => 'gift (christmas,generosity,giving,holiday,party,present,wrapped,xmas)' ),
			array( 'las la-gifts' => 'Gifts (christmas,generosity,giving,holiday,party,present,wrapped,xmas)' ),
			array( 'las la-glass-cheers' => 'Glass Cheers (alcohol,bar,beverage,celebration,champagne,clink,drink,holiday,new year\'s eve,party,toast)' ),
			array( 'las la-holly-berry' => 'Holly Berry (catwoman,christmas,decoration,flora,halle,holiday,ororo munroe,plant,storm,xmas)' ),
			array( 'las la-mug-hot' => 'Mug Hot (caliente,cocoa,coffee,cup,drink,holiday,hot chocolate,steam,tea,warmth)' ),
			array( 'las la-sleigh' => 'Sleigh (christmas,claus,fly,holiday,santa,sled,snow,xmas)' ),
			array( 'las la-snowman' => 'Snowman (decoration,frost,frosty,holiday)' ),
		),
		'Hotel' => array(
			array( 'las la-baby-carriage' => 'Baby Carriage (buggy,carrier,infant,push,stroller,transportation,walk,wheels)' ),
			array( 'las la-bath' => 'Bath (clean,shower,tub,wash)' ),
			array( 'las la-bed' => 'Bed (lodging,rest,sleep,travel)' ),
			array( 'las la-briefcase' => 'Briefcase (bag,business,luggage,office,work)' ),
			array( 'las la-car' => 'Car (auto,automobile,sedan,transportation,travel,vehicle)' ),
			array( 'las la-cocktail' => 'Cocktail (alcohol,beverage,drink,gin,glass,margarita,martini,vodka)' ),
			array( 'las la-coffee' => 'Coffee (beverage,breakfast,cafe,drink,fall,morning,mug,seasonal,tea)' ),
			array( 'las la-concierge-bell' => 'Concierge Bell (attention,hotel,receptionist,service,support)' ),
			array( 'las la-dice' => 'Dice (chance,gambling,game,roll)' ),
			array( 'las la-dice-five' => 'Dice Five (chance,gambling,game,roll)' ),
			array( 'las la-door-closed' => 'Door Closed (enter,exit,locked)' ),
			array( 'las la-door-open' => 'Door Open (enter,exit,welcome)' ),
			array( 'las la-dumbbell' => 'Dumbbell (exercise,gym,strength,weight,weight-lifting)' ),
			array( 'las la-glass-martini' => 'Martini Glass (alcohol,bar,beverage,drink,liquor)' ),
			array( 'las la-glass-martini-alt' => 'Alternate Glass Martini (alcohol,bar,beverage,drink,liquor)' ),
			array( 'las la-hot-tub' => 'Hot Tub (bath,jacuzzi,massage,sauna,spa)' ),
			array( 'las la-hotel' => 'Hotel (building,inn,lodging,motel,resort,travel)' ),
			array( 'las la-infinity' => 'Infinity (eternity,forever,math)' ),
			array( 'las la-key' => 'key (lock,password,private,secret,unlock)' ),
			array( 'las la-luggage-cart' => 'Luggage Cart (bag,baggage,suitcase,travel)' ),
			array( 'las la-shower' => 'Shower (bath,clean,faucet,water)' ),
			array( 'las la-shuttle-van' => 'Shuttle Van (airport,machine,public-transportation,transportation,travel,vehicle)' ),
			array( 'las la-smoking' => 'Smoking (cancer,cigarette,nicotine,smoking status,tobacco)' ),
			array( 'las la-smoking-ban' => 'Smoking Ban (ban,cancel,no smoking,non-smoking)' ),
			array( 'las la-snowflake' => 'Snowflake (precipitation,rain,winter)' ),
			array( 'lar la-snowflake' => 'Snowflake (precipitation,rain,winter)' ),
			array( 'las la-spa' => 'Spa (flora,massage,mindfulness,plant,wellness)' ),
			array( 'las la-suitcase' => 'Suitcase (baggage,luggage,move,suitcase,travel,trip)' ),
			array( 'las la-suitcase-rolling' => 'Suitcase Rolling (baggage,luggage,move,suitcase,travel,trip)' ),
			array( 'las la-swimmer' => 'Swimmer (athlete,head,man,olympics,person,pool,water)' ),
			array( 'las la-swimming-pool' => 'Swimming Pool (ladder,recreation,swim,water)' ),
			array( 'las la-tv' => 'Television (computer,display,monitor,television)' ),
			array( 'las la-umbrella-beach' => 'Umbrella Beach (protection,recreation,sand,shade,summer,sun)' ),
			array( 'las la-utensils' => 'Utensils (cutlery,dining,dinner,eat,food,fork,knife,restaurant)' ),
			array( 'las la-wheelchair' => 'Wheelchair (accessible,handicap,person)' ),
			array( 'las la-wifi' => 'WiFi (connection,hotspot,internet,network,wireless)' ),
		),
		'Household' => array(
			array( 'las la-bath' => 'Bath (clean,shower,tub,wash)' ),
			array( 'las la-bed' => 'Bed (lodging,rest,sleep,travel)' ),
			array( 'las la-blender' => 'Blender (cocktail,milkshake,mixer,puree,smoothie)' ),
			array( 'las la-chair' => 'Chair (furniture,seat,sit)' ),
			array( 'las la-couch' => 'Couch (chair,cushion,furniture,relax,sofa)' ),
			array( 'las la-door-closed' => 'Door Closed (enter,exit,locked)' ),
			array( 'las la-door-open' => 'Door Open (enter,exit,welcome)' ),
			array( 'las la-dungeon' => 'Dungeon (Dungeons & Dragons,building,d&d,dnd,door,entrance,fantasy,gate)' ),
			array( 'las la-fan' => 'Fan (ac,air conditioning,blade,blower,cool,hot)' ),
			array( 'las la-shower' => 'Shower (bath,clean,faucet,water)' ),
			array( 'las la-toilet-paper' => 'Toilet Paper (bathroom,halloween,holiday,lavatory,prank,restroom,roll)' ),
			array( 'las la-tv' => 'Television (computer,display,monitor,television)' ),
		),
		'Images' => array(
			array( 'las la-adjust' => 'adjust (contrast,dark,light,saturation)' ),
			array( 'las la-bolt' => 'Lightning Bolt (electricity,lightning,weather,zap)' ),
			array( 'las la-camera' => 'camera (image,lens,photo,picture,record,shutter,video)' ),
			array( 'las la-camera-retro' => 'Retro Camera (image,lens,photo,picture,record,shutter,video)' ),
			array( 'las la-chalkboard' => 'Chalkboard (blackboard,learning,school,teaching,whiteboard,writing)' ),
			array( 'las la-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'lar la-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'las la-compress' => 'Compress (collapse,fullscreen,minimize,move,resize,shrink,smaller)' ),
			array( 'las la-compress-arrows-alt' => 'Alternate Compress Arrows (collapse,fullscreen,minimize,move,resize,shrink,smaller)' ),
			array( 'las la-expand' => 'Expand (arrow,bigger,enlarge,resize)' ),
			array( 'las la-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'lar la-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'las la-eye-dropper' => 'Eye Dropper (beaker,clone,color,copy,eyedropper,pipette)' ),
			array( 'las la-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'lar la-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'las la-file-image' => 'Image File (document,image,jpg,photo,png)' ),
			array( 'lar la-file-image' => 'Image File (document,image,jpg,photo,png)' ),
			array( 'las la-film' => 'Film (cinema,movie,strip,video)' ),
			array( 'las la-id-badge' => 'Identification Badge (address,contact,identification,license,profile)' ),
			array( 'lar la-id-badge' => 'Identification Badge (address,contact,identification,license,profile)' ),
			array( 'las la-id-card' => 'Identification Card (contact,demographics,document,identification,issued,profile)' ),
			array( 'lar la-id-card' => 'Identification Card (contact,demographics,document,identification,issued,profile)' ),
			array( 'las la-image' => 'Image (album,landscape,photo,picture)' ),
			array( 'lar la-image' => 'Image (album,landscape,photo,picture)' ),
			array( 'las la-images' => 'Images (album,landscape,photo,picture)' ),
			array( 'lar la-images' => 'Images (album,landscape,photo,picture)' ),
			array( 'las la-photo-video' => 'Photo Video (av,film,image,library,media)' ),
			array( 'las la-portrait' => 'Portrait (id,image,photo,picture,selfie)' ),
			array( 'las la-sliders-h' => 'Horizontal Sliders (adjust,settings,sliders,toggle)' ),
			array( 'las la-tint' => 'tint (color,drop,droplet,raindrop,waterdrop)' ),
		),
		'Interfaces' => array(
			array( 'las la-award' => 'Award (honor,praise,prize,recognition,ribbon,trophy)' ),
			array( 'las la-ban' => 'ban (abort,ban,block,cancel,delete,hide,prohibit,remove,stop,trash)' ),
			array( 'las la-barcode' => 'barcode (info,laser,price,scan,upc)' ),
			array( 'las la-bars' => 'Bars (checklist,drag,hamburger,list,menu,nav,navigation,ol,reorder,settings,todo,ul)' ),
			array( 'las la-beer' => 'beer (alcohol,ale,bar,beverage,brewery,drink,lager,liquor,mug,stein)' ),
			array( 'las la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'lar la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'las la-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'lar la-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'las la-blog' => 'Blog (journal,log,online,personal,post,web 2.0,wordpress,writing)' ),
			array( 'las la-bug' => 'Bug (beetle,error,insect,report)' ),
			array( 'las la-bullhorn' => 'bullhorn (announcement,broadcast,louder,megaphone,share)' ),
			array( 'las la-bullseye' => 'Bullseye (archery,goal,objective,target)' ),
			array( 'las la-calculator' => 'Calculator (abacus,addition,arithmetic,counting,math,multiplication,subtraction)' ),
			array( 'las la-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'lar la-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'las la-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'lar la-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'las la-calendar-check' => 'Calendar Check (accept,agree,appointment,confirm,correct,date,done,event,ok,schedule,select,success,tick,time,todo,when)' ),
			array( 'lar la-calendar-check' => 'Calendar Check (accept,agree,appointment,confirm,correct,date,done,event,ok,schedule,select,success,tick,time,todo,when)' ),
			array( 'las la-calendar-minus' => 'Calendar Minus (calendar,date,delete,event,negative,remove,schedule,time,when)' ),
			array( 'lar la-calendar-minus' => 'Calendar Minus (calendar,date,delete,event,negative,remove,schedule,time,when)' ),
			array( 'las la-calendar-plus' => 'Calendar Plus (add,calendar,create,date,event,new,positive,schedule,time,when)' ),
			array( 'lar la-calendar-plus' => 'Calendar Plus (add,calendar,create,date,event,new,positive,schedule,time,when)' ),
			array( 'las la-calendar-times' => 'Calendar Times (archive,calendar,date,delete,event,remove,schedule,time,when,x)' ),
			array( 'lar la-calendar-times' => 'Calendar Times (archive,calendar,date,delete,event,remove,schedule,time,when,x)' ),
			array( 'las la-certificate' => 'certificate (badge,star,verified)' ),
			array( 'las la-check' => 'Check (accept,agree,checkmark,confirm,correct,done,notice,notification,notify,ok,select,success,tick,todo,yes)' ),
			array( 'las la-check-circle' => 'Check Circle (accept,agree,confirm,correct,done,ok,select,success,tick,todo,yes)' ),
			array( 'lar la-check-circle' => 'Check Circle (accept,agree,confirm,correct,done,ok,select,success,tick,todo,yes)' ),
			array( 'las la-check-double' => 'Double Check (accept,agree,checkmark,confirm,correct,done,notice,notification,notify,ok,select,success,tick,todo)' ),
			array( 'las la-check-square' => 'Check Square (accept,agree,checkmark,confirm,correct,done,ok,select,success,tick,todo,yes)' ),
			array( 'lar la-check-square' => 'Check Square (accept,agree,checkmark,confirm,correct,done,ok,select,success,tick,todo,yes)' ),
			array( 'las la-circle' => 'Circle (circle-thin,diameter,dot,ellipse,notification,round)' ),
			array( 'lar la-circle' => 'Circle (circle-thin,diameter,dot,ellipse,notification,round)' ),
			array( 'las la-clipboard' => 'Clipboard (copy,notes,paste,record)' ),
			array( 'lar la-clipboard' => 'Clipboard (copy,notes,paste,record)' ),
			array( 'las la-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'lar la-clone' => 'Clone (arrange,copy,duplicate,paste)' ),
			array( 'las la-cloud' => 'Cloud (atmosphere,fog,overcast,save,upload,weather)' ),
			array( 'las la-cloud-download-alt' => 'Alternate Cloud Download (download,export,save)' ),
			array( 'las la-cloud-upload-alt' => 'Alternate Cloud Upload (cloud-upload,import,save,upload)' ),
			array( 'las la-coffee' => 'Coffee (beverage,breakfast,cafe,drink,fall,morning,mug,seasonal,tea)' ),
			array( 'las la-cog' => 'cog (gear,mechanical,settings,sprocket,wheel)' ),
			array( 'las la-cogs' => 'cogs (gears,mechanical,settings,sprocket,wheel)' ),
			array( 'las la-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'lar la-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'las la-cut' => 'Cut (clip,scissors,snip)' ),
			array( 'las la-database' => 'Database (computer,development,directory,memory,storage)' ),
			array( 'las la-dot-circle' => 'Dot Circle (bullseye,notification,target)' ),
			array( 'lar la-dot-circle' => 'Dot Circle (bullseye,notification,target)' ),
			array( 'las la-download' => 'Download (export,hard drive,save,transfer)' ),
			array( 'las la-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'lar la-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'las la-ellipsis-h' => 'Horizontal Ellipsis (dots,drag,kebab,list,menu,nav,navigation,ol,reorder,settings,ul)' ),
			array( 'las la-ellipsis-v' => 'Vertical Ellipsis (dots,drag,kebab,list,menu,nav,navigation,ol,reorder,settings,ul)' ),
			array( 'las la-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'lar la-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'las la-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'lar la-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'las la-eraser' => 'eraser (art,delete,remove,rubber)' ),
			array( 'las la-exclamation' => 'exclamation (alert,danger,error,important,notice,notification,notify,problem,warning)' ),
			array( 'las la-exclamation-circle' => 'Exclamation Circle (alert,danger,error,important,notice,notification,notify,problem,warning)' ),
			array( 'las la-exclamation-triangle' => 'Exclamation Triangle (alert,danger,error,important,notice,notification,notify,problem,warning)' ),
			array( 'las la-external-link-alt' => 'Alternate External Link (external-link,new,open,share)' ),
			array( 'las la-external-link-square-alt' => 'Alternate External Link Square (external-link-square,new,open,share)' ),
			array( 'las la-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'lar la-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'las la-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'lar la-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'las la-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'lar la-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'las la-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'lar la-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'las la-file-download' => 'File Download (document,export,save)' ),
			array( 'las la-file-export' => 'File Export (download,save)' ),
			array( 'las la-file-import' => 'File Import (copy,document,send,upload)' ),
			array( 'las la-file-upload' => 'File Upload (document,import,page,save)' ),
			array( 'las la-filter' => 'Filter (funnel,options,separate,sort)' ),
			array( 'las la-fingerprint' => 'Fingerprint (human,id,identification,lock,smudge,touch,unique,unlock)' ),
			array( 'las la-flag' => 'flag (country,notice,notification,notify,pole,report,symbol)' ),
			array( 'lar la-flag' => 'flag (country,notice,notification,notify,pole,report,symbol)' ),
			array( 'las la-flag-checkered' => 'flag-checkered (notice,notification,notify,pole,racing,report,symbol)' ),
			array( 'las la-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'lar la-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'las la-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'lar la-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'las la-frown' => 'Frowning Face (disapprove,emoticon,face,rating,sad)' ),
			array( 'lar la-frown' => 'Frowning Face (disapprove,emoticon,face,rating,sad)' ),
			array( 'las la-glasses' => 'Glasses (hipster,nerd,reading,sight,spectacles,vision)' ),
			array( 'las la-grip-horizontal' => 'Grip Horizontal (affordance,drag,drop,grab,handle)' ),
			array( 'las la-grip-lines' => 'Grip Lines (affordance,drag,drop,grab,handle)' ),
			array( 'las la-grip-lines-vertical' => 'Grip Lines Vertical (affordance,drag,drop,grab,handle)' ),
			array( 'las la-grip-vertical' => 'Grip Vertical (affordance,drag,drop,grab,handle)' ),
			array( 'las la-hashtag' => 'Hashtag (Twitter,instagram,pound,social media,tag)' ),
			array( 'las la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'lar la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'las la-history' => 'History (Rewind,clock,reverse,time,time machine)' ),
			array( 'las la-home' => 'home (abode,building,house,main)' ),
			array( 'las la-i-cursor' => 'I Beam Cursor (editing,i-beam,type,writing)' ),
			array( 'las la-info' => 'Info (details,help,information,more,support)' ),
			array( 'las la-info-circle' => 'Info Circle (details,help,information,more,support)' ),
			array( 'las la-language' => 'Language (dialect,idiom,localize,speech,translate,vernacular)' ),
			array( 'las la-magic' => 'magic (autocomplete,automatic,mage,magic,spell,wand,witch,wizard)' ),
			array( 'las la-marker' => 'Marker (design,edit,sharpie,update,write)' ),
			array( 'las la-medal' => 'Medal (award,ribbon,star,trophy)' ),
			array( 'las la-meh' => 'Neutral Face (emoticon,face,neutral,rating)' ),
			array( 'lar la-meh' => 'Neutral Face (emoticon,face,neutral,rating)' ),
			array( 'las la-microphone' => 'microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'las la-microphone-alt' => 'Alternate Microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'las la-microphone-slash' => 'Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'las la-minus' => 'minus (collapse,delete,hide,minify,negative,remove,trash)' ),
			array( 'las la-minus-circle' => 'Minus Circle (delete,hide,negative,remove,shape,trash)' ),
			array( 'las la-minus-square' => 'Minus Square (collapse,delete,hide,minify,negative,remove,shape,trash)' ),
			array( 'lar la-minus-square' => 'Minus Square (collapse,delete,hide,minify,negative,remove,shape,trash)' ),
			array( 'las la-paste' => 'Paste (clipboard,copy,document,paper)' ),
			array( 'las la-pen' => 'Pen (design,edit,update,write)' ),
			array( 'las la-pen-alt' => 'Alternate Pen (design,edit,update,write)' ),
			array( 'las la-pen-fancy' => 'Pen Fancy (design,edit,fountain pen,update,write)' ),
			array( 'las la-pencil-alt' => 'Alternate Pencil (design,edit,pencil,update,write)' ),
			array( 'las la-plus' => 'plus (add,create,expand,new,positive,shape)' ),
			array( 'las la-plus-circle' => 'Plus Circle (add,create,expand,new,positive,shape)' ),
			array( 'las la-plus-square' => 'Plus Square (add,create,expand,new,positive,shape)' ),
			array( 'lar la-plus-square' => 'Plus Square (add,create,expand,new,positive,shape)' ),
			array( 'las la-poo' => 'Poo (crap,poop,shit,smile,turd)' ),
			array( 'las la-qrcode' => 'qrcode (barcode,info,information,scan)' ),
			array( 'las la-question' => 'Question (help,information,support,unknown)' ),
			array( 'las la-question-circle' => 'Question Circle (help,information,support,unknown)' ),
			array( 'lar la-question-circle' => 'Question Circle (help,information,support,unknown)' ),
			array( 'las la-quote-left' => 'quote-left (mention,note,phrase,text,type)' ),
			array( 'las la-quote-right' => 'quote-right (mention,note,phrase,text,type)' ),
			array( 'las la-redo' => 'Redo (forward,refresh,reload,repeat)' ),
			array( 'las la-redo-alt' => 'Alternate Redo (forward,refresh,reload,repeat)' ),
			array( 'las la-reply' => 'Reply (mail,message,respond)' ),
			array( 'las la-reply-all' => 'reply-all (mail,message,respond)' ),
			array( 'las la-rss' => 'rss (blog,feed,journal,news,writing)' ),
			array( 'las la-rss-square' => 'RSS Square (blog,feed,journal,news,writing)' ),
			array( 'las la-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'lar la-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'las la-screwdriver' => 'Screwdriver (admin,fix,mechanic,repair,settings,tool)' ),
			array( 'las la-search' => 'Search (bigger,enlarge,find,magnify,preview,zoom)' ),
			array( 'las la-search-minus' => 'Search Minus (minify,negative,smaller,zoom,zoom out)' ),
			array( 'las la-search-plus' => 'Search Plus (bigger,enlarge,magnify,positive,zoom,zoom in)' ),
			array( 'las la-share' => 'Share (forward,save,send,social)' ),
			array( 'las la-share-alt' => 'Alternate Share (forward,save,send,social)' ),
			array( 'las la-share-alt-square' => 'Alternate Share Square (forward,save,send,social)' ),
			array( 'las la-share-square' => 'Share Square (forward,save,send,social)' ),
			array( 'lar la-share-square' => 'Share Square (forward,save,send,social)' ),
			array( 'las la-shield-alt' => 'Alternate Shield (achievement,award,block,defend,security,winner)' ),
			array( 'las la-sign-in-alt' => 'Alternate Sign In (arrow,enter,join,log in,login,sign in,sign up,sign-in,signin,signup)' ),
			array( 'las la-sign-out-alt' => 'Alternate Sign Out (arrow,exit,leave,log out,logout,sign-out)' ),
			array( 'las la-signal' => 'signal (bars,graph,online,reception,status)' ),
			array( 'las la-sitemap' => 'Sitemap (directory,hierarchy,ia,information architecture,organization)' ),
			array( 'las la-sliders-h' => 'Horizontal Sliders (adjust,settings,sliders,toggle)' ),
			array( 'las la-smile' => 'Smiling Face (approve,emoticon,face,happy,rating,satisfied)' ),
			array( 'lar la-smile' => 'Smiling Face (approve,emoticon,face,happy,rating,satisfied)' ),
			array( 'las la-sort' => 'Sort (filter,order)' ),
			array( 'las la-sort-alpha-down' => 'Sort Alphabetical Down (alphabetical,arrange,filter,order,sort-alpha-asc)' ),
			array( 'las la-sort-alpha-down-alt' => 'Alternate Sort Alphabetical Down (alphabetical,arrange,filter,order,sort-alpha-asc)' ),
			array( 'las la-sort-alpha-up' => 'Sort Alphabetical Up (alphabetical,arrange,filter,order,sort-alpha-desc)' ),
			array( 'las la-sort-alpha-up-alt' => 'Alternate Sort Alphabetical Up (alphabetical,arrange,filter,order,sort-alpha-desc)' ),
			array( 'las la-sort-amount-down' => 'Sort Amount Down (arrange,filter,number,order,sort-amount-asc)' ),
			array( 'las la-sort-amount-down-alt' => 'Alternate Sort Amount Down (arrange,filter,order,sort-amount-asc)' ),
			array( 'las la-sort-amount-up' => 'Sort Amount Up (arrange,filter,order,sort-amount-desc)' ),
			array( 'las la-sort-amount-up-alt' => 'Alternate Sort Amount Up (arrange,filter,order,sort-amount-desc)' ),
			array( 'las la-sort-down' => 'Sort Down (Descending) (arrow,descending,filter,order,sort-desc)' ),
			array( 'las la-sort-numeric-down' => 'Sort Numeric Down (arrange,filter,numbers,order,sort-numeric-asc)' ),
			array( 'las la-sort-numeric-down-alt' => 'Alternate Sort Numeric Down (arrange,filter,numbers,order,sort-numeric-asc)' ),
			array( 'las la-sort-numeric-up' => 'Sort Numeric Up (arrange,filter,numbers,order,sort-numeric-desc)' ),
			array( 'las la-sort-numeric-up-alt' => 'Alternate Sort Numeric Up (arrange,filter,numbers,order,sort-numeric-desc)' ),
			array( 'las la-sort-up' => 'Sort Up (Ascending) (arrow,ascending,filter,order,sort-asc)' ),
			array( 'las la-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'lar la-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'las la-star-half' => 'star-half (achievement,award,rating,score,star-half-empty,star-half-full)' ),
			array( 'lar la-star-half' => 'star-half (achievement,award,rating,score,star-half-empty,star-half-full)' ),
			array( 'las la-sync' => 'Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'las la-sync-alt' => 'Alternate Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'las la-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'lar la-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'las la-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
			array( 'lar la-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
			array( 'las la-times' => 'Times (close,cross,error,exit,incorrect,notice,notification,notify,problem,wrong,x)' ),
			array( 'las la-times-circle' => 'Times Circle (close,cross,exit,incorrect,notice,notification,notify,problem,wrong,x)' ),
			array( 'lar la-times-circle' => 'Times Circle (close,cross,exit,incorrect,notice,notification,notify,problem,wrong,x)' ),
			array( 'las la-toggle-off' => 'Toggle Off (switch)' ),
			array( 'las la-toggle-on' => 'Toggle On (switch)' ),
			array( 'las la-tools' => 'Tools (admin,fix,repair,screwdriver,settings,tools,wrench)' ),
			array( 'las la-trash' => 'Trash (delete,garbage,hide,remove)' ),
			array( 'las la-trash-alt' => 'Alternate Trash (delete,garbage,hide,remove,trash-o)' ),
			array( 'lar la-trash-alt' => 'Alternate Trash (delete,garbage,hide,remove,trash-o)' ),
			array( 'las la-trash-restore' => 'Trash Restore (back,control z,oops,undo)' ),
			array( 'las la-trash-restore-alt' => 'Alternative Trash Restore (back,control z,oops,undo)' ),
			array( 'las la-trophy' => 'trophy (achievement,award,cup,game,winner)' ),
			array( 'las la-undo' => 'Undo (back,control z,exchange,oops,return,rotate,swap)' ),
			array( 'las la-undo-alt' => 'Alternate Undo (back,control z,exchange,oops,return,swap)' ),
			array( 'las la-upload' => 'Upload (hard drive,import,publish)' ),
			array( 'las la-user' => 'User (account,avatar,head,human,man,person,profile)' ),
			array( 'lar la-user' => 'User (account,avatar,head,human,man,person,profile)' ),
			array( 'las la-user-alt' => 'Alternate User (account,avatar,head,human,man,person,profile)' ),
			array( 'las la-user-circle' => 'User Circle (account,avatar,head,human,man,person,profile)' ),
			array( 'lar la-user-circle' => 'User Circle (account,avatar,head,human,man,person,profile)' ),
			array( 'las la-volume-down' => 'Volume Down (audio,lower,music,quieter,sound,speaker)' ),
			array( 'las la-volume-mute' => 'Volume Mute (audio,music,quiet,sound,speaker)' ),
			array( 'las la-volume-off' => 'Volume Off (audio,ban,music,mute,quiet,silent,sound)' ),
			array( 'las la-volume-up' => 'Volume Up (audio,higher,louder,music,sound,speaker)' ),
			array( 'las la-wifi' => 'WiFi (connection,hotspot,internet,network,wireless)' ),
			array( 'las la-wrench' => 'Wrench (construction,fix,mechanic,plumbing,settings,spanner,tool,update)' ),
		),
		'Logistics' => array(
			array( 'las la-box' => 'Box (archive,container,package,storage)' ),
			array( 'las la-boxes' => 'Boxes (archives,inventory,storage,warehouse)' ),
			array( 'las la-clipboard-check' => 'Clipboard with Check (accept,agree,confirm,done,ok,select,success,tick,todo,yes)' ),
			array( 'las la-clipboard-list' => 'Clipboard List (checklist,completed,done,finished,intinerary,ol,schedule,tick,todo,ul)' ),
			array( 'las la-dolly' => 'Dolly (carry,shipping,transport)' ),
			array( 'las la-dolly-flatbed' => 'Dolly Flatbed (carry,inventory,shipping,transport)' ),
			array( 'las la-hard-hat' => 'Hard Hat (construction,hardhat,helmet,safety)' ),
			array( 'las la-pallet' => 'Pallet (archive,box,inventory,shipping,warehouse)' ),
			array( 'las la-shipping-fast' => 'Shipping Fast (express,fedex,mail,overnight,package,ups)' ),
			array( 'las la-truck' => 'truck (cargo,delivery,shipping,vehicle)' ),
			array( 'las la-warehouse' => 'Warehouse (building,capacity,garage,inventory,storage)' ),
		),
		'Maps' => array(
			array( 'las la-ambulance' => 'ambulance (emergency,emt,er,help,hospital,support,vehicle)' ),
			array( 'las la-anchor' => 'Anchor (berth,boat,dock,embed,link,maritime,moor,secure)' ),
			array( 'las la-balance-scale' => 'Balance Scale (balanced,justice,legal,measure,weight)' ),
			array( 'las la-balance-scale-left' => 'Balance Scale (Left-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'las la-balance-scale-right' => 'Balance Scale (Right-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'las la-bath' => 'Bath (clean,shower,tub,wash)' ),
			array( 'las la-bed' => 'Bed (lodging,rest,sleep,travel)' ),
			array( 'las la-beer' => 'beer (alcohol,ale,bar,beverage,brewery,drink,lager,liquor,mug,stein)' ),
			array( 'las la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'lar la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'las la-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'lar la-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'las la-bicycle' => 'Bicycle (bike,gears,pedal,transportation,vehicle)' ),
			array( 'las la-binoculars' => 'Binoculars (glasses,magnify,scenic,spyglass,view)' ),
			array( 'las la-birthday-cake' => 'Birthday Cake (anniversary,bakery,candles,celebration,dessert,frosting,holiday,party,pastry)' ),
			array( 'las la-blind' => 'Blind (cane,disability,person,sight)' ),
			array( 'las la-bomb' => 'Bomb (error,explode,fuse,grenade,warning)' ),
			array( 'las la-book' => 'book (diary,documentation,journal,library,read)' ),
			array( 'las la-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'lar la-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'las la-briefcase' => 'Briefcase (bag,business,luggage,office,work)' ),
			array( 'las la-building' => 'Building (apartment,business,city,company,office,work)' ),
			array( 'lar la-building' => 'Building (apartment,business,city,company,office,work)' ),
			array( 'las la-car' => 'Car (auto,automobile,sedan,transportation,travel,vehicle)' ),
			array( 'las la-coffee' => 'Coffee (beverage,breakfast,cafe,drink,fall,morning,mug,seasonal,tea)' ),
			array( 'las la-crosshairs' => 'Crosshairs (aim,bullseye,gpd,picker,position)' ),
			array( 'las la-directions' => 'Directions (map,navigation,sign,turn)' ),
			array( 'las la-dollar-sign' => 'Dollar Sign ($,cost,dollar-sign,money,price,usd)' ),
			array( 'las la-draw-polygon' => 'Draw Polygon (anchors,lines,object,render,shape)' ),
			array( 'las la-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'lar la-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'las la-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'lar la-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'las la-fighter-jet' => 'fighter-jet (airplane,fast,fly,goose,maverick,plane,quick,top gun,transportation,travel)' ),
			array( 'las la-fire' => 'fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'las la-fire-alt' => 'Alternate Fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'las la-fire-extinguisher' => 'fire-extinguisher (burn,caliente,fire fighter,flame,heat,hot,rescue)' ),
			array( 'las la-flag' => 'flag (country,notice,notification,notify,pole,report,symbol)' ),
			array( 'lar la-flag' => 'flag (country,notice,notification,notify,pole,report,symbol)' ),
			array( 'las la-flag-checkered' => 'flag-checkered (notice,notification,notify,pole,racing,report,symbol)' ),
			array( 'las la-flask' => 'Flask (beaker,experimental,labs,science)' ),
			array( 'las la-gamepad' => 'Gamepad (arcade,controller,d-pad,joystick,video,video game)' ),
			array( 'las la-gavel' => 'Gavel (hammer,judge,law,lawyer,opinion)' ),
			array( 'las la-gift' => 'gift (christmas,generosity,giving,holiday,party,present,wrapped,xmas)' ),
			array( 'las la-glass-martini' => 'Martini Glass (alcohol,bar,beverage,drink,liquor)' ),
			array( 'las la-globe' => 'Globe (all,coordinates,country,earth,global,gps,language,localize,location,map,online,place,planet,translate,travel,world)' ),
			array( 'las la-graduation-cap' => 'Graduation Cap (ceremony,college,graduate,learning,school,student)' ),
			array( 'las la-h-square' => 'H Square (directions,emergency,hospital,hotel,map)' ),
			array( 'las la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'lar la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'las la-heartbeat' => 'Heartbeat (ekg,electrocardiogram,health,lifeline,vital signs)' ),
			array( 'las la-helicopter' => 'Helicopter (airwolf,apache,chopper,flight,fly,travel)' ),
			array( 'las la-home' => 'home (abode,building,house,main)' ),
			array( 'las la-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'lar la-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'las la-image' => 'Image (album,landscape,photo,picture)' ),
			array( 'lar la-image' => 'Image (album,landscape,photo,picture)' ),
			array( 'las la-images' => 'Images (album,landscape,photo,picture)' ),
			array( 'lar la-images' => 'Images (album,landscape,photo,picture)' ),
			array( 'las la-industry' => 'Industry (building,factory,industrial,manufacturing,mill,warehouse)' ),
			array( 'las la-info' => 'Info (details,help,information,more,support)' ),
			array( 'las la-info-circle' => 'Info Circle (details,help,information,more,support)' ),
			array( 'las la-key' => 'key (lock,password,private,secret,unlock)' ),
			array( 'las la-landmark' => 'Landmark (building,historic,memorable,monument,politics)' ),
			array( 'las la-layer-group' => 'Layer Group (arrange,develop,layers,map,stack)' ),
			array( 'las la-leaf' => 'leaf (eco,flora,nature,plant,vegan)' ),
			array( 'las la-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'lar la-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'las la-life-ring' => 'Life Ring (coast guard,help,overboard,save,support)' ),
			array( 'lar la-life-ring' => 'Life Ring (coast guard,help,overboard,save,support)' ),
			array( 'las la-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'lar la-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'las la-location-arrow' => 'location-arrow (address,compass,coordinate,direction,gps,map,navigation,place)' ),
			array( 'las la-low-vision' => 'Low Vision (blind,eye,sight)' ),
			array( 'las la-magnet' => 'magnet (Attract,lodestone,tool)' ),
			array( 'las la-male' => 'Male (human,man,person,profile,user)' ),
			array( 'las la-map' => 'Map (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'lar la-map' => 'Map (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'las la-map-marker' => 'map-marker (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'las la-map-marker-alt' => 'Alternate Map Marker (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'las la-map-pin' => 'Map Pin (address,agree,coordinates,destination,gps,localize,location,map,marker,navigation,pin,place,position,travel)' ),
			array( 'las la-map-signs' => 'Map Signs (directions,directory,map,signage,wayfinding)' ),
			array( 'las la-medkit' => 'medkit (first aid,firstaid,health,help,support)' ),
			array( 'las la-money-bill' => 'Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'las la-money-bill-alt' => 'Alternate Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'lar la-money-bill-alt' => 'Alternate Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'las la-motorcycle' => 'Motorcycle (bike,machine,transportation,vehicle)' ),
			array( 'las la-music' => 'Music (lyrics,melody,note,sing,sound)' ),
			array( 'las la-newspaper' => 'Newspaper (article,editorial,headline,journal,journalism,news,press)' ),
			array( 'lar la-newspaper' => 'Newspaper (article,editorial,headline,journal,journalism,news,press)' ),
			array( 'las la-parking' => 'Parking (auto,car,garage,meter)' ),
			array( 'las la-paw' => 'Paw (animal,cat,dog,pet,print)' ),
			array( 'las la-phone' => 'Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'las la-phone-alt' => 'Alternate Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'las la-phone-square' => 'Phone Square (call,earphone,number,support,telephone,voice)' ),
			array( 'las la-phone-square-alt' => 'Alternate Phone Square (call,earphone,number,support,telephone,voice)' ),
			array( 'las la-phone-volume' => 'Phone Volume (call,earphone,number,sound,support,telephone,voice,volume-control-phone)' ),
			array( 'las la-plane' => 'plane (airplane,destination,fly,location,mode,travel,trip)' ),
			array( 'las la-plug' => 'Plug (connect,electric,online,power)' ),
			array( 'las la-plus' => 'plus (add,create,expand,new,positive,shape)' ),
			array( 'las la-plus-square' => 'Plus Square (add,create,expand,new,positive,shape)' ),
			array( 'lar la-plus-square' => 'Plus Square (add,create,expand,new,positive,shape)' ),
			array( 'las la-print' => 'print (business,copy,document,office,paper)' ),
			array( 'las la-recycle' => 'Recycle (Waste,compost,garbage,reuse,trash)' ),
			array( 'las la-restroom' => 'Restroom (bathroom,john,loo,potty,washroom,waste,wc)' ),
			array( 'las la-road' => 'road (highway,map,pavement,route,street,travel)' ),
			array( 'las la-rocket' => 'rocket (aircraft,app,jet,launch,nasa,space)' ),
			array( 'las la-route' => 'Route (directions,navigation,travel)' ),
			array( 'las la-search' => 'Search (bigger,enlarge,find,magnify,preview,zoom)' ),
			array( 'las la-search-minus' => 'Search Minus (minify,negative,smaller,zoom,zoom out)' ),
			array( 'las la-search-plus' => 'Search Plus (bigger,enlarge,magnify,positive,zoom,zoom in)' ),
			array( 'las la-ship' => 'Ship (boat,sea,water)' ),
			array( 'las la-shoe-prints' => 'Shoe Prints (feet,footprints,steps,walk)' ),
			array( 'las la-shopping-bag' => 'Shopping Bag (buy,checkout,grocery,payment,purchase)' ),
			array( 'las la-shopping-basket' => 'Shopping Basket (buy,checkout,grocery,payment,purchase)' ),
			array( 'las la-shopping-cart' => 'shopping-cart (buy,checkout,grocery,payment,purchase)' ),
			array( 'las la-shower' => 'Shower (bath,clean,faucet,water)' ),
			array( 'las la-snowplow' => 'Snowplow (clean up,cold,road,storm,winter)' ),
			array( 'las la-street-view' => 'Street View (directions,location,map,navigation)' ),
			array( 'las la-subway' => 'Subway (machine,railway,train,transportation,vehicle)' ),
			array( 'las la-suitcase' => 'Suitcase (baggage,luggage,move,suitcase,travel,trip)' ),
			array( 'las la-tag' => 'tag (discount,label,price,shopping)' ),
			array( 'las la-tags' => 'tags (discount,label,price,shopping)' ),
			array( 'las la-taxi' => 'Taxi (cab,cabbie,car,car service,lyft,machine,transportation,travel,uber,vehicle)' ),
			array( 'las la-thumbtack' => 'Thumbtack (coordinates,location,marker,pin,thumb-tack)' ),
			array( 'las la-ticket-alt' => 'Alternate Ticket (movie,pass,support,ticket)' ),
			array( 'las la-tint' => 'tint (color,drop,droplet,raindrop,waterdrop)' ),
			array( 'las la-traffic-light' => 'Traffic Light (direction,road,signal,travel)' ),
			array( 'las la-train' => 'Train (bullet,commute,locomotive,railway,subway)' ),
			array( 'las la-tram' => 'Tram (crossing,machine,mountains,seasonal,transportation)' ),
			array( 'las la-tree' => 'Tree (bark,fall,flora,forest,nature,plant,seasonal)' ),
			array( 'las la-trophy' => 'trophy (achievement,award,cup,game,winner)' ),
			array( 'las la-truck' => 'truck (cargo,delivery,shipping,vehicle)' ),
			array( 'las la-tty' => 'TTY (communication,deaf,telephone,teletypewriter,text)' ),
			array( 'las la-umbrella' => 'Umbrella (protection,rain,storm,wet)' ),
			array( 'las la-university' => 'University (bank,building,college,higher education - students,institution)' ),
			array( 'las la-utensil-spoon' => 'Utensil Spoon (cutlery,dining,scoop,silverware,spoon)' ),
			array( 'las la-utensils' => 'Utensils (cutlery,dining,dinner,eat,food,fork,knife,restaurant)' ),
			array( 'las la-wheelchair' => 'Wheelchair (accessible,handicap,person)' ),
			array( 'las la-wifi' => 'WiFi (connection,hotspot,internet,network,wireless)' ),
			array( 'las la-wine-glass' => 'Wine Glass (alcohol,beverage,cabernet,drink,grapes,merlot,sauvignon)' ),
			array( 'las la-wrench' => 'Wrench (construction,fix,mechanic,plumbing,settings,spanner,tool,update)' ),
		),
		'Maritime' => array(
			array( 'las la-anchor' => 'Anchor (berth,boat,dock,embed,link,maritime,moor,secure)' ),
			array( 'las la-binoculars' => 'Binoculars (glasses,magnify,scenic,spyglass,view)' ),
			array( 'las la-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'lar la-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'las la-dharmachakra' => 'Dharmachakra (buddhism,buddhist,wheel of dharma)' ),
			array( 'las la-frog' => 'Frog (amphibian,bullfrog,fauna,hop,kermit,kiss,prince,ribbit,toad,wart)' ),
			array( 'las la-ship' => 'Ship (boat,sea,water)' ),
			array( 'las la-skull-crossbones' => 'Skull & Crossbones (Dungeons & Dragons,alert,bones,d&d,danger,dead,deadly,death,dnd,fantasy,halloween,holiday,jolly-roger,pirate,poison,skeleton,warning)' ),
			array( 'las la-swimmer' => 'Swimmer (athlete,head,man,olympics,person,pool,water)' ),
			array( 'las la-water' => 'Water (lake,liquid,ocean,sea,swim,wet)' ),
			array( 'las la-wind' => 'Wind (air,blow,breeze,fall,seasonal,weather)' ),
		),
		'Marketing' => array(
			array( 'las la-ad' => 'Ad (advertisement,media,newspaper,promotion,publicity)' ),
			array( 'las la-bullhorn' => 'bullhorn (announcement,broadcast,louder,megaphone,share)' ),
			array( 'las la-bullseye' => 'Bullseye (archery,goal,objective,target)' ),
			array( 'las la-comment-dollar' => 'Comment Dollar (bubble,chat,commenting,conversation,feedback,message,money,note,notification,pay,sms,speech,spend,texting,transfer)' ),
			array( 'las la-comments-dollar' => 'Comments Dollar (bubble,chat,commenting,conversation,feedback,message,money,note,notification,pay,sms,speech,spend,texting,transfer)' ),
			array( 'las la-envelope-open-text' => 'Envelope Open-text (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'las la-funnel-dollar' => 'Funnel Dollar (filter,money,options,separate,sort)' ),
			array( 'las la-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'lar la-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'las la-mail-bulk' => 'Mail Bulk (archive,envelope,letter,post office,postal,postcard,send,stamp,usps)' ),
			array( 'las la-poll' => 'Poll (results,survey,trend,vote,voting)' ),
			array( 'las la-poll-h' => 'Poll H (results,survey,trend,vote,voting)' ),
			array( 'las la-search-dollar' => 'Search Dollar (bigger,enlarge,find,magnify,money,preview,zoom)' ),
			array( 'las la-search-location' => 'Search Location (bigger,enlarge,find,magnify,preview,zoom)' ),
		),
		'Mathematics' => array(
			array( 'las la-calculator' => 'Calculator (abacus,addition,arithmetic,counting,math,multiplication,subtraction)' ),
			array( 'las la-divide' => 'Divide (arithmetic,calculus,division,math)' ),
			array( 'las la-equals' => 'Equals (arithmetic,even,match,math)' ),
			array( 'las la-greater-than' => 'Greater Than (arithmetic,compare,math)' ),
			array( 'las la-greater-than-equal' => 'Greater Than Equal To (arithmetic,compare,math)' ),
			array( 'las la-infinity' => 'Infinity (eternity,forever,math)' ),
			array( 'las la-less-than' => 'Less Than (arithmetic,compare,math)' ),
			array( 'las la-less-than-equal' => 'Less Than Equal To (arithmetic,compare,math)' ),
			array( 'las la-minus' => 'minus (collapse,delete,hide,minify,negative,remove,trash)' ),
			array( 'las la-not-equal' => 'Not Equal (arithmetic,compare,math)' ),
			array( 'las la-percentage' => 'Percentage (discount,fraction,proportion,rate,ratio)' ),
			array( 'las la-plus' => 'plus (add,create,expand,new,positive,shape)' ),
			array( 'las la-square-root-alt' => 'Alternate Square Root (arithmetic,calculus,division,math)' ),
			array( 'las la-subscript' => 'subscript (edit,font,format,text,type)' ),
			array( 'las la-superscript' => 'superscript (edit,exponential,font,format,text,type)' ),
			array( 'las la-times' => 'Times (close,cross,error,exit,incorrect,notice,notification,notify,problem,wrong,x)' ),
			array( 'las la-wave-square' => 'Square Wave (frequency,pulse,signal)' ),
		),
		'Medical' => array(
			array( 'las la-allergies' => 'Allergies (allergy,freckles,hand,hives,pox,skin,spots)' ),
			array( 'las la-ambulance' => 'ambulance (emergency,emt,er,help,hospital,support,vehicle)' ),
			array( 'las la-band-aid' => 'Band-Aid (bandage,boo boo,first aid,ouch)' ),
			array( 'las la-biohazard' => 'Biohazard (danger,dangerous,hazmat,medical,radioactive,toxic,waste,zombie)' ),
			array( 'las la-bone' => 'Bone (calcium,dog,skeletal,skeleton,tibia)' ),
			array( 'las la-bong' => 'Bong (aparatus,cannabis,marijuana,pipe,smoke,smoking)' ),
			array( 'las la-book-medical' => 'Medical Book (diary,documentation,health,history,journal,library,read,record)' ),
			array( 'las la-brain' => 'Brain (cerebellum,gray matter,intellect,medulla oblongata,mind,noodle,wit)' ),
			array( 'las la-briefcase-medical' => 'Medical Briefcase (doctor,emt,first aid,health)' ),
			array( 'las la-burn' => 'Burn (caliente,energy,fire,flame,gas,heat,hot)' ),
			array( 'las la-cannabis' => 'Cannabis (bud,chronic,drugs,endica,endo,ganja,marijuana,mary jane,pot,reefer,sativa,spliff,weed,whacky-tabacky)' ),
			array( 'las la-capsules' => 'Capsules (drugs,medicine,pills,prescription)' ),
			array( 'las la-clinic-medical' => 'Medical Clinic (doctor,general practitioner,hospital,infirmary,medicine,office,outpatient)' ),
			array( 'las la-comment-medical' => 'Alternate Medical Chat (advice,bubble,chat,commenting,conversation,diagnose,feedback,message,note,notification,prescription,sms,speech,texting)' ),
			array( 'las la-crutch' => 'Crutch (cane,injury,mobility,wheelchair)' ),
			array( 'las la-diagnoses' => 'Diagnoses (analyze,detect,diagnosis,examine,medicine)' ),
			array( 'las la-dna' => 'DNA (double helix,genetic,helix,molecule,protein)' ),
			array( 'las la-file-medical' => 'Medical File (document,health,history,prescription,record)' ),
			array( 'las la-file-medical-alt' => 'Alternate Medical File (document,health,history,prescription,record)' ),
			array( 'las la-file-prescription' => 'File Prescription (document,drugs,medical,medicine,rx)' ),
			array( 'las la-first-aid' => 'First Aid (emergency,emt,health,medical,rescue)' ),
			array( 'las la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'lar la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'las la-heartbeat' => 'Heartbeat (ekg,electrocardiogram,health,lifeline,vital signs)' ),
			array( 'las la-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'lar la-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'las la-hospital-alt' => 'Alternate Hospital (building,emergency room,medical center)' ),
			array( 'las la-hospital-symbol' => 'Hospital Symbol (clinic,emergency,map)' ),
			array( 'las la-id-card-alt' => 'Alternate Identification Card (contact,demographics,document,identification,issued,profile)' ),
			array( 'las la-joint' => 'Joint (blunt,cannabis,doobie,drugs,marijuana,roach,smoke,smoking,spliff)' ),
			array( 'las la-laptop-medical' => 'Laptop Medical (computer,device,ehr,electronic health records,history)' ),
			array( 'las la-microscope' => 'Microscope (electron,lens,optics,science,shrink)' ),
			array( 'las la-mortar-pestle' => 'Mortar Pestle (crush,culinary,grind,medical,mix,pharmacy,prescription,spices)' ),
			array( 'las la-notes-medical' => 'Medical Notes (clipboard,doctor,ehr,health,history,records)' ),
			array( 'las la-pager' => 'Pager (beeper,cellphone,communication)' ),
			array( 'las la-pills' => 'Pills (drugs,medicine,prescription,tablets)' ),
			array( 'las la-plus' => 'plus (add,create,expand,new,positive,shape)' ),
			array( 'las la-poop' => 'Poop (crap,poop,shit,smile,turd)' ),
			array( 'las la-prescription' => 'Prescription (drugs,medical,medicine,pharmacy,rx)' ),
			array( 'las la-prescription-bottle' => 'Prescription Bottle (drugs,medical,medicine,pharmacy,rx)' ),
			array( 'las la-prescription-bottle-alt' => 'Alternate Prescription Bottle (drugs,medical,medicine,pharmacy,rx)' ),
			array( 'las la-procedures' => 'Procedures (EKG,bed,electrocardiogram,health,hospital,life,patient,vital)' ),
			array( 'las la-radiation' => 'Radiation (danger,dangerous,deadly,hazard,nuclear,radioactive,warning)' ),
			array( 'las la-radiation-alt' => 'Alternate Radiation (danger,dangerous,deadly,hazard,nuclear,radioactive,warning)' ),
			array( 'las la-smoking' => 'Smoking (cancer,cigarette,nicotine,smoking status,tobacco)' ),
			array( 'las la-smoking-ban' => 'Smoking Ban (ban,cancel,no smoking,non-smoking)' ),
			array( 'las la-star-of-life' => 'Star of Life (doctor,emt,first aid,health,medical)' ),
			array( 'las la-stethoscope' => 'Stethoscope (diagnosis,doctor,general practitioner,hospital,infirmary,medicine,office,outpatient)' ),
			array( 'las la-syringe' => 'Syringe (doctor,immunizations,medical,needle)' ),
			array( 'las la-tablets' => 'Tablets (drugs,medicine,pills,prescription)' ),
			array( 'las la-teeth' => 'Teeth (bite,dental,dentist,gums,mouth,smile,tooth)' ),
			array( 'las la-teeth-open' => 'Teeth Open (dental,dentist,gums bite,mouth,smile,tooth)' ),
			array( 'las la-thermometer' => 'Thermometer (mercury,status,temperature)' ),
			array( 'las la-tooth' => 'Tooth (bicuspid,dental,dentist,molar,mouth,teeth)' ),
			array( 'las la-user-md' => 'Doctor (job,medical,nurse,occupation,physician,profile,surgeon)' ),
			array( 'las la-user-nurse' => 'Nurse (doctor,midwife,practitioner,surgeon)' ),
			array( 'las la-vial' => 'Vial (experiment,lab,sample,science,test,test tube)' ),
			array( 'las la-vials' => 'Vials (experiment,lab,sample,science,test,test tube)' ),
			array( 'las la-weight' => 'Weight (health,measurement,scale,weight)' ),
			array( 'las la-x-ray' => 'X-Ray (health,medical,radiological images,radiology,skeleton)' ),
		),
		'Moving' => array(
			array( 'las la-archive' => 'Archive (box,package,save,storage)' ),
			array( 'las la-box-open' => 'Box Open (archive,container,package,storage,unpack)' ),
			array( 'las la-couch' => 'Couch (chair,cushion,furniture,relax,sofa)' ),
			array( 'las la-dolly' => 'Dolly (carry,shipping,transport)' ),
			array( 'las la-people-carry' => 'People Carry (box,carry,fragile,help,movers,package)' ),
			array( 'las la-route' => 'Route (directions,navigation,travel)' ),
			array( 'las la-sign' => 'Sign (directions,real estate,signage,wayfinding)' ),
			array( 'las la-suitcase' => 'Suitcase (baggage,luggage,move,suitcase,travel,trip)' ),
			array( 'las la-tape' => 'Tape (design,package,sticky)' ),
			array( 'las la-truck-loading' => 'Truck Loading (box,cargo,delivery,inventory,moving,rental,vehicle)' ),
			array( 'las la-truck-moving' => 'Truck Moving (cargo,inventory,rental,vehicle)' ),
			array( 'las la-wine-glass' => 'Wine Glass (alcohol,beverage,cabernet,drink,grapes,merlot,sauvignon)' ),
		),
		'Music' => array(
			array( 'las la-drum' => 'Drum (instrument,music,percussion,snare,sound)' ),
			array( 'las la-drum-steelpan' => 'Drum Steelpan (calypso,instrument,music,percussion,reggae,snare,sound,steel,tropical)' ),
			array( 'las la-file-audio' => 'Audio File (document,mp3,music,page,play,sound)' ),
			array( 'lar la-file-audio' => 'Audio File (document,mp3,music,page,play,sound)' ),
			array( 'las la-guitar' => 'Guitar (acoustic,instrument,music,rock,rock and roll,song,strings)' ),
			array( 'las la-headphones' => 'headphones (audio,listen,music,sound,speaker)' ),
			array( 'las la-headphones-alt' => 'Alternate Headphones (audio,listen,music,sound,speaker)' ),
			array( 'las la-microphone' => 'microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'las la-microphone-alt' => 'Alternate Microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'las la-microphone-alt-slash' => 'Alternate Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'las la-microphone-slash' => 'Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'las la-music' => 'Music (lyrics,melody,note,sing,sound)' ),
			array( 'lab la-napster' => 'Napster' ),
			array( 'las la-play' => 'play (audio,music,playing,sound,start,video)' ),
			array( 'las la-record-vinyl' => 'Record Vinyl (LP,album,analog,music,phonograph,sound)' ),
			array( 'las la-sliders-h' => 'Horizontal Sliders (adjust,settings,sliders,toggle)' ),
			array( 'lab la-soundcloud' => 'SoundCloud' ),
			array( 'lab la-spotify' => 'Spotify' ),
			array( 'las la-volume-down' => 'Volume Down (audio,lower,music,quieter,sound,speaker)' ),
			array( 'las la-volume-mute' => 'Volume Mute (audio,music,quiet,sound,speaker)' ),
			array( 'las la-volume-off' => 'Volume Off (audio,ban,music,mute,quiet,silent,sound)' ),
			array( 'las la-volume-up' => 'Volume Up (audio,higher,louder,music,sound,speaker)' ),
		),
		'Objects' => array(
			array( 'las la-ambulance' => 'ambulance (emergency,emt,er,help,hospital,support,vehicle)' ),
			array( 'las la-anchor' => 'Anchor (berth,boat,dock,embed,link,maritime,moor,secure)' ),
			array( 'las la-archive' => 'Archive (box,package,save,storage)' ),
			array( 'las la-award' => 'Award (honor,praise,prize,recognition,ribbon,trophy)' ),
			array( 'las la-baby-carriage' => 'Baby Carriage (buggy,carrier,infant,push,stroller,transportation,walk,wheels)' ),
			array( 'las la-balance-scale' => 'Balance Scale (balanced,justice,legal,measure,weight)' ),
			array( 'las la-balance-scale-left' => 'Balance Scale (Left-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'las la-balance-scale-right' => 'Balance Scale (Right-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'las la-bath' => 'Bath (clean,shower,tub,wash)' ),
			array( 'las la-bed' => 'Bed (lodging,rest,sleep,travel)' ),
			array( 'las la-beer' => 'beer (alcohol,ale,bar,beverage,brewery,drink,lager,liquor,mug,stein)' ),
			array( 'las la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'lar la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'las la-bicycle' => 'Bicycle (bike,gears,pedal,transportation,vehicle)' ),
			array( 'las la-binoculars' => 'Binoculars (glasses,magnify,scenic,spyglass,view)' ),
			array( 'las la-birthday-cake' => 'Birthday Cake (anniversary,bakery,candles,celebration,dessert,frosting,holiday,party,pastry)' ),
			array( 'las la-blender' => 'Blender (cocktail,milkshake,mixer,puree,smoothie)' ),
			array( 'las la-bomb' => 'Bomb (error,explode,fuse,grenade,warning)' ),
			array( 'las la-book' => 'book (diary,documentation,journal,library,read)' ),
			array( 'las la-book-dead' => 'Book of the Dead (Dungeons & Dragons,crossbones,d&d,dark arts,death,dnd,documentation,evil,fantasy,halloween,holiday,necronomicon,read,skull,spell)' ),
			array( 'las la-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'lar la-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'las la-briefcase' => 'Briefcase (bag,business,luggage,office,work)' ),
			array( 'las la-broadcast-tower' => 'Broadcast Tower (airwaves,antenna,radio,reception,waves)' ),
			array( 'las la-bug' => 'Bug (beetle,error,insect,report)' ),
			array( 'las la-building' => 'Building (apartment,business,city,company,office,work)' ),
			array( 'lar la-building' => 'Building (apartment,business,city,company,office,work)' ),
			array( 'las la-bullhorn' => 'bullhorn (announcement,broadcast,louder,megaphone,share)' ),
			array( 'las la-bullseye' => 'Bullseye (archery,goal,objective,target)' ),
			array( 'las la-bus' => 'Bus (public transportation,transportation,travel,vehicle)' ),
			array( 'las la-calculator' => 'Calculator (abacus,addition,arithmetic,counting,math,multiplication,subtraction)' ),
			array( 'las la-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'lar la-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'las la-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'lar la-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'las la-camera' => 'camera (image,lens,photo,picture,record,shutter,video)' ),
			array( 'las la-camera-retro' => 'Retro Camera (image,lens,photo,picture,record,shutter,video)' ),
			array( 'las la-candy-cane' => 'Candy Cane (candy,christmas,holiday,mint,peppermint,striped,xmas)' ),
			array( 'las la-car' => 'Car (auto,automobile,sedan,transportation,travel,vehicle)' ),
			array( 'las la-carrot' => 'Carrot (bugs bunny,orange,vegan,vegetable)' ),
			array( 'las la-church' => 'Church (building,cathedral,chapel,community,religion)' ),
			array( 'las la-clipboard' => 'Clipboard (copy,notes,paste,record)' ),
			array( 'lar la-clipboard' => 'Clipboard (copy,notes,paste,record)' ),
			array( 'las la-cloud' => 'Cloud (atmosphere,fog,overcast,save,upload,weather)' ),
			array( 'las la-coffee' => 'Coffee (beverage,breakfast,cafe,drink,fall,morning,mug,seasonal,tea)' ),
			array( 'las la-cog' => 'cog (gear,mechanical,settings,sprocket,wheel)' ),
			array( 'las la-cogs' => 'cogs (gears,mechanical,settings,sprocket,wheel)' ),
			array( 'las la-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'lar la-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'las la-cookie' => 'Cookie (baked good,chips,chocolate,eat,snack,sweet,treat)' ),
			array( 'las la-cookie-bite' => 'Cookie Bite (baked good,bitten,chips,chocolate,eat,snack,sweet,treat)' ),
			array( 'las la-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'lar la-copy' => 'Copy (clone,duplicate,file,files-o,paper,paste)' ),
			array( 'las la-cube' => 'Cube (3d,block,dice,package,square,tesseract)' ),
			array( 'las la-cubes' => 'Cubes (3d,block,dice,package,pyramid,square,stack,tesseract)' ),
			array( 'las la-cut' => 'Cut (clip,scissors,snip)' ),
			array( 'las la-dice' => 'Dice (chance,gambling,game,roll)' ),
			array( 'las la-dice-d20' => 'Dice D20 (Dungeons & Dragons,chance,d&d,dnd,fantasy,gambling,game,roll)' ),
			array( 'las la-dice-d6' => 'Dice D6 (Dungeons & Dragons,chance,d&d,dnd,fantasy,gambling,game,roll)' ),
			array( 'las la-dice-five' => 'Dice Five (chance,gambling,game,roll)' ),
			array( 'las la-dice-four' => 'Dice Four (chance,gambling,game,roll)' ),
			array( 'las la-dice-one' => 'Dice One (chance,gambling,game,roll)' ),
			array( 'las la-dice-six' => 'Dice Six (chance,gambling,game,roll)' ),
			array( 'las la-dice-three' => 'Dice Three (chance,gambling,game,roll)' ),
			array( 'las la-dice-two' => 'Dice Two (chance,gambling,game,roll)' ),
			array( 'las la-digital-tachograph' => 'Digital Tachograph (data,distance,speed,tachometer)' ),
			array( 'las la-door-closed' => 'Door Closed (enter,exit,locked)' ),
			array( 'las la-door-open' => 'Door Open (enter,exit,welcome)' ),
			array( 'las la-drum' => 'Drum (instrument,music,percussion,snare,sound)' ),
			array( 'las la-drum-steelpan' => 'Drum Steelpan (calypso,instrument,music,percussion,reggae,snare,sound,steel,tropical)' ),
			array( 'las la-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'lar la-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'las la-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'lar la-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'las la-eraser' => 'eraser (art,delete,remove,rubber)' ),
			array( 'las la-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'lar la-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'las la-eye-dropper' => 'Eye Dropper (beaker,clone,color,copy,eyedropper,pipette)' ),
			array( 'las la-fax' => 'Fax (business,communicate,copy,facsimile,send)' ),
			array( 'las la-feather' => 'Feather (bird,light,plucked,quill,write)' ),
			array( 'las la-feather-alt' => 'Alternate Feather (bird,light,plucked,quill,write)' ),
			array( 'las la-fighter-jet' => 'fighter-jet (airplane,fast,fly,goose,maverick,plane,quick,top gun,transportation,travel)' ),
			array( 'las la-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'lar la-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'las la-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'lar la-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'las la-file-prescription' => 'File Prescription (document,drugs,medical,medicine,rx)' ),
			array( 'las la-film' => 'Film (cinema,movie,strip,video)' ),
			array( 'las la-fire' => 'fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'las la-fire-alt' => 'Alternate Fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'las la-fire-extinguisher' => 'fire-extinguisher (burn,caliente,fire fighter,flame,heat,hot,rescue)' ),
			array( 'las la-flag' => 'flag (country,notice,notification,notify,pole,report,symbol)' ),
			array( 'lar la-flag' => 'flag (country,notice,notification,notify,pole,report,symbol)' ),
			array( 'las la-flag-checkered' => 'flag-checkered (notice,notification,notify,pole,racing,report,symbol)' ),
			array( 'las la-flask' => 'Flask (beaker,experimental,labs,science)' ),
			array( 'las la-futbol' => 'Futbol (ball,football,mls,soccer)' ),
			array( 'lar la-futbol' => 'Futbol (ball,football,mls,soccer)' ),
			array( 'las la-gamepad' => 'Gamepad (arcade,controller,d-pad,joystick,video,video game)' ),
			array( 'las la-gavel' => 'Gavel (hammer,judge,law,lawyer,opinion)' ),
			array( 'las la-gem' => 'Gem (diamond,jewelry,sapphire,stone,treasure)' ),
			array( 'lar la-gem' => 'Gem (diamond,jewelry,sapphire,stone,treasure)' ),
			array( 'las la-gift' => 'gift (christmas,generosity,giving,holiday,party,present,wrapped,xmas)' ),
			array( 'las la-gifts' => 'Gifts (christmas,generosity,giving,holiday,party,present,wrapped,xmas)' ),
			array( 'las la-glass-cheers' => 'Glass Cheers (alcohol,bar,beverage,celebration,champagne,clink,drink,holiday,new year\'s eve,party,toast)' ),
			array( 'las la-glass-martini' => 'Martini Glass (alcohol,bar,beverage,drink,liquor)' ),
			array( 'las la-glass-whiskey' => 'Glass Whiskey (alcohol,bar,beverage,bourbon,drink,liquor,neat,rye,scotch,whisky)' ),
			array( 'las la-glasses' => 'Glasses (hipster,nerd,reading,sight,spectacles,vision)' ),
			array( 'las la-globe' => 'Globe (all,coordinates,country,earth,global,gps,language,localize,location,map,online,place,planet,translate,travel,world)' ),
			array( 'las la-graduation-cap' => 'Graduation Cap (ceremony,college,graduate,learning,school,student)' ),
			array( 'las la-guitar' => 'Guitar (acoustic,instrument,music,rock,rock and roll,song,strings)' ),
			array( 'las la-hat-wizard' => 'Wizard\'s Hat (Dungeons & Dragons,accessory,buckle,clothing,d&d,dnd,fantasy,halloween,head,holiday,mage,magic,pointy,witch)' ),
			array( 'las la-hdd' => 'HDD (cpu,hard drive,harddrive,machine,save,storage)' ),
			array( 'lar la-hdd' => 'HDD (cpu,hard drive,harddrive,machine,save,storage)' ),
			array( 'las la-headphones' => 'headphones (audio,listen,music,sound,speaker)' ),
			array( 'las la-headphones-alt' => 'Alternate Headphones (audio,listen,music,sound,speaker)' ),
			array( 'las la-headset' => 'Headset (audio,gamer,gaming,listen,live chat,microphone,shot caller,sound,support,telemarketer)' ),
			array( 'las la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'lar la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'las la-heart-broken' => 'Heart Broken (breakup,crushed,dislike,dumped,grief,love,lovesick,relationship,sad)' ),
			array( 'las la-helicopter' => 'Helicopter (airwolf,apache,chopper,flight,fly,travel)' ),
			array( 'las la-highlighter' => 'Highlighter (edit,marker,sharpie,update,write)' ),
			array( 'las la-holly-berry' => 'Holly Berry (catwoman,christmas,decoration,flora,halle,holiday,ororo munroe,plant,storm,xmas)' ),
			array( 'las la-home' => 'home (abode,building,house,main)' ),
			array( 'las la-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'lar la-hospital' => 'hospital (building,emergency room,medical center)' ),
			array( 'las la-hourglass' => 'Hourglass (hour,minute,sand,stopwatch,time)' ),
			array( 'lar la-hourglass' => 'Hourglass (hour,minute,sand,stopwatch,time)' ),
			array( 'las la-igloo' => 'Igloo (dome,dwelling,eskimo,home,house,ice,snow)' ),
			array( 'las la-image' => 'Image (album,landscape,photo,picture)' ),
			array( 'lar la-image' => 'Image (album,landscape,photo,picture)' ),
			array( 'las la-images' => 'Images (album,landscape,photo,picture)' ),
			array( 'lar la-images' => 'Images (album,landscape,photo,picture)' ),
			array( 'las la-industry' => 'Industry (building,factory,industrial,manufacturing,mill,warehouse)' ),
			array( 'las la-key' => 'key (lock,password,private,secret,unlock)' ),
			array( 'las la-keyboard' => 'Keyboard (accessory,edit,input,text,type,write)' ),
			array( 'lar la-keyboard' => 'Keyboard (accessory,edit,input,text,type,write)' ),
			array( 'las la-laptop' => 'Laptop (computer,cpu,dell,demo,device,mac,macbook,machine,pc)' ),
			array( 'las la-leaf' => 'leaf (eco,flora,nature,plant,vegan)' ),
			array( 'las la-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'lar la-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'las la-life-ring' => 'Life Ring (coast guard,help,overboard,save,support)' ),
			array( 'lar la-life-ring' => 'Life Ring (coast guard,help,overboard,save,support)' ),
			array( 'las la-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'lar la-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'las la-lock' => 'lock (admin,lock,open,password,private,protect,security)' ),
			array( 'las la-lock-open' => 'Lock Open (admin,lock,open,password,private,protect,security)' ),
			array( 'las la-magic' => 'magic (autocomplete,automatic,mage,magic,spell,wand,witch,wizard)' ),
			array( 'las la-magnet' => 'magnet (Attract,lodestone,tool)' ),
			array( 'las la-map' => 'Map (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'lar la-map' => 'Map (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'las la-map-marker' => 'map-marker (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'las la-map-marker-alt' => 'Alternate Map Marker (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'las la-map-pin' => 'Map Pin (address,agree,coordinates,destination,gps,localize,location,map,marker,navigation,pin,place,position,travel)' ),
			array( 'las la-map-signs' => 'Map Signs (directions,directory,map,signage,wayfinding)' ),
			array( 'las la-marker' => 'Marker (design,edit,sharpie,update,write)' ),
			array( 'las la-medal' => 'Medal (award,ribbon,star,trophy)' ),
			array( 'las la-medkit' => 'medkit (first aid,firstaid,health,help,support)' ),
			array( 'las la-memory' => 'Memory (DIMM,RAM,hardware,storage,technology)' ),
			array( 'las la-microchip' => 'Microchip (cpu,hardware,processor,technology)' ),
			array( 'las la-microphone' => 'microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'las la-microphone-alt' => 'Alternate Microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'las la-mitten' => 'Mitten (clothing,cold,glove,hands,knitted,seasonal,warmth)' ),
			array( 'las la-mobile' => 'Mobile Phone (apple,call,cell phone,cellphone,device,iphone,number,screen,telephone)' ),
			array( 'las la-mobile-alt' => 'Alternate Mobile (apple,call,cell phone,cellphone,device,iphone,number,screen,telephone)' ),
			array( 'las la-money-bill' => 'Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'las la-money-bill-alt' => 'Alternate Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'lar la-money-bill-alt' => 'Alternate Money Bill (buy,cash,checkout,money,payment,price,purchase)' ),
			array( 'las la-money-check' => 'Money Check (bank check,buy,checkout,cheque,money,payment,price,purchase)' ),
			array( 'las la-money-check-alt' => 'Alternate Money Check (bank check,buy,checkout,cheque,money,payment,price,purchase)' ),
			array( 'las la-moon' => 'Moon (contrast,crescent,dark,lunar,night)' ),
			array( 'lar la-moon' => 'Moon (contrast,crescent,dark,lunar,night)' ),
			array( 'las la-motorcycle' => 'Motorcycle (bike,machine,transportation,vehicle)' ),
			array( 'las la-mug-hot' => 'Mug Hot (caliente,cocoa,coffee,cup,drink,holiday,hot chocolate,steam,tea,warmth)' ),
			array( 'las la-newspaper' => 'Newspaper (article,editorial,headline,journal,journalism,news,press)' ),
			array( 'lar la-newspaper' => 'Newspaper (article,editorial,headline,journal,journalism,news,press)' ),
			array( 'las la-paint-brush' => 'Paint Brush (acrylic,art,brush,color,fill,paint,pigment,watercolor)' ),
			array( 'las la-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'lar la-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'las la-paperclip' => 'Paperclip (attach,attachment,connect,link)' ),
			array( 'las la-paste' => 'Paste (clipboard,copy,document,paper)' ),
			array( 'las la-paw' => 'Paw (animal,cat,dog,pet,print)' ),
			array( 'las la-pen' => 'Pen (design,edit,update,write)' ),
			array( 'las la-pen-alt' => 'Alternate Pen (design,edit,update,write)' ),
			array( 'las la-pen-fancy' => 'Pen Fancy (design,edit,fountain pen,update,write)' ),
			array( 'las la-pen-nib' => 'Pen Nib (design,edit,fountain pen,update,write)' ),
			array( 'las la-pencil-alt' => 'Alternate Pencil (design,edit,pencil,update,write)' ),
			array( 'las la-phone' => 'Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'las la-phone-alt' => 'Alternate Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'las la-plane' => 'plane (airplane,destination,fly,location,mode,travel,trip)' ),
			array( 'las la-plug' => 'Plug (connect,electric,online,power)' ),
			array( 'las la-print' => 'print (business,copy,document,office,paper)' ),
			array( 'las la-puzzle-piece' => 'Puzzle Piece (add-on,addon,game,section)' ),
			array( 'las la-ring' => 'Ring (Dungeons & Dragons,Gollum,band,binding,d&d,dnd,engagement,fantasy,gold,jewelry,marriage,precious)' ),
			array( 'las la-road' => 'road (highway,map,pavement,route,street,travel)' ),
			array( 'las la-rocket' => 'rocket (aircraft,app,jet,launch,nasa,space)' ),
			array( 'las la-ruler-combined' => 'Ruler Combined (design,draft,length,measure,planning)' ),
			array( 'las la-ruler-horizontal' => 'Ruler Horizontal (design,draft,length,measure,planning)' ),
			array( 'las la-ruler-vertical' => 'Ruler Vertical (design,draft,length,measure,planning)' ),
			array( 'las la-satellite' => 'Satellite (communications,hardware,orbit,space)' ),
			array( 'las la-satellite-dish' => 'Satellite Dish (SETI,communications,hardware,receiver,saucer,signal)' ),
			array( 'las la-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'lar la-save' => 'Save (disk,download,floppy,floppy-o)' ),
			array( 'las la-school' => 'School (building,education,learn,student,teacher)' ),
			array( 'las la-screwdriver' => 'Screwdriver (admin,fix,mechanic,repair,settings,tool)' ),
			array( 'las la-scroll' => 'Scroll (Dungeons & Dragons,announcement,d&d,dnd,fantasy,paper,script)' ),
			array( 'las la-sd-card' => 'Sd Card (image,memory,photo,save)' ),
			array( 'las la-search' => 'Search (bigger,enlarge,find,magnify,preview,zoom)' ),
			array( 'las la-shield-alt' => 'Alternate Shield (achievement,award,block,defend,security,winner)' ),
			array( 'las la-shopping-bag' => 'Shopping Bag (buy,checkout,grocery,payment,purchase)' ),
			array( 'las la-shopping-basket' => 'Shopping Basket (buy,checkout,grocery,payment,purchase)' ),
			array( 'las la-shopping-cart' => 'shopping-cart (buy,checkout,grocery,payment,purchase)' ),
			array( 'las la-shower' => 'Shower (bath,clean,faucet,water)' ),
			array( 'las la-sim-card' => 'SIM Card (hard drive,hardware,portable,storage,technology,tiny)' ),
			array( 'las la-skull-crossbones' => 'Skull & Crossbones (Dungeons & Dragons,alert,bones,d&d,danger,dead,deadly,death,dnd,fantasy,halloween,holiday,jolly-roger,pirate,poison,skeleton,warning)' ),
			array( 'las la-sleigh' => 'Sleigh (christmas,claus,fly,holiday,santa,sled,snow,xmas)' ),
			array( 'las la-snowflake' => 'Snowflake (precipitation,rain,winter)' ),
			array( 'lar la-snowflake' => 'Snowflake (precipitation,rain,winter)' ),
			array( 'las la-snowplow' => 'Snowplow (clean up,cold,road,storm,winter)' ),
			array( 'las la-space-shuttle' => 'Space Shuttle (astronaut,machine,nasa,rocket,transportation)' ),
			array( 'las la-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'lar la-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'las la-sticky-note' => 'Sticky Note (message,note,paper,reminder,sticker)' ),
			array( 'lar la-sticky-note' => 'Sticky Note (message,note,paper,reminder,sticker)' ),
			array( 'las la-stopwatch' => 'Stopwatch (clock,reminder,time)' ),
			array( 'las la-stroopwafel' => 'Stroopwafel (caramel,cookie,dessert,sweets,waffle)' ),
			array( 'las la-subway' => 'Subway (machine,railway,train,transportation,vehicle)' ),
			array( 'las la-suitcase' => 'Suitcase (baggage,luggage,move,suitcase,travel,trip)' ),
			array( 'las la-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'lar la-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'las la-tablet' => 'tablet (apple,device,ipad,kindle,screen)' ),
			array( 'las la-tablet-alt' => 'Alternate Tablet (apple,device,ipad,kindle,screen)' ),
			array( 'las la-tachometer-alt' => 'Alternate Tachometer (dashboard,fast,odometer,speed,speedometer)' ),
			array( 'las la-tag' => 'tag (discount,label,price,shopping)' ),
			array( 'las la-tags' => 'tags (discount,label,price,shopping)' ),
			array( 'las la-taxi' => 'Taxi (cab,cabbie,car,car service,lyft,machine,transportation,travel,uber,vehicle)' ),
			array( 'las la-thumbtack' => 'Thumbtack (coordinates,location,marker,pin,thumb-tack)' ),
			array( 'las la-ticket-alt' => 'Alternate Ticket (movie,pass,support,ticket)' ),
			array( 'las la-toilet' => 'Toilet (bathroom,flush,john,loo,pee,plumbing,poop,porcelain,potty,restroom,throne,washroom,waste,wc)' ),
			array( 'las la-toolbox' => 'Toolbox (admin,container,fix,repair,settings,tools)' ),
			array( 'las la-tools' => 'Tools (admin,fix,repair,screwdriver,settings,tools,wrench)' ),
			array( 'las la-train' => 'Train (bullet,commute,locomotive,railway,subway)' ),
			array( 'las la-tram' => 'Tram (crossing,machine,mountains,seasonal,transportation)' ),
			array( 'las la-trash' => 'Trash (delete,garbage,hide,remove)' ),
			array( 'las la-trash-alt' => 'Alternate Trash (delete,garbage,hide,remove,trash-o)' ),
			array( 'lar la-trash-alt' => 'Alternate Trash (delete,garbage,hide,remove,trash-o)' ),
			array( 'las la-tree' => 'Tree (bark,fall,flora,forest,nature,plant,seasonal)' ),
			array( 'las la-trophy' => 'trophy (achievement,award,cup,game,winner)' ),
			array( 'las la-truck' => 'truck (cargo,delivery,shipping,vehicle)' ),
			array( 'las la-tv' => 'Television (computer,display,monitor,television)' ),
			array( 'las la-umbrella' => 'Umbrella (protection,rain,storm,wet)' ),
			array( 'las la-university' => 'University (bank,building,college,higher education - students,institution)' ),
			array( 'las la-unlock' => 'unlock (admin,lock,password,private,protect)' ),
			array( 'las la-unlock-alt' => 'Alternate Unlock (admin,lock,password,private,protect)' ),
			array( 'las la-utensil-spoon' => 'Utensil Spoon (cutlery,dining,scoop,silverware,spoon)' ),
			array( 'las la-utensils' => 'Utensils (cutlery,dining,dinner,eat,food,fork,knife,restaurant)' ),
			array( 'las la-wallet' => 'Wallet (billfold,cash,currency,money)' ),
			array( 'las la-weight' => 'Weight (health,measurement,scale,weight)' ),
			array( 'las la-wheelchair' => 'Wheelchair (accessible,handicap,person)' ),
			array( 'las la-wine-glass' => 'Wine Glass (alcohol,beverage,cabernet,drink,grapes,merlot,sauvignon)' ),
			array( 'las la-wrench' => 'Wrench (construction,fix,mechanic,plumbing,settings,spanner,tool,update)' ),
		),
		'Payments & Shopping' => array(
			array( 'lab la-alipay' => 'Alipay' ),
			array( 'lab la-amazon-pay' => 'Amazon Pay' ),
			array( 'lab la-apple-pay' => 'Apple Pay' ),
			array( 'las la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'lar la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'lab la-bitcoin' => 'Bitcoin' ),
			array( 'las la-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'lar la-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'lab la-btc' => 'BTC' ),
			array( 'las la-bullhorn' => 'bullhorn (announcement,broadcast,louder,megaphone,share)' ),
			array( 'las la-camera' => 'camera (image,lens,photo,picture,record,shutter,video)' ),
			array( 'las la-camera-retro' => 'Retro Camera (image,lens,photo,picture,record,shutter,video)' ),
			array( 'las la-cart-arrow-down' => 'Shopping Cart Arrow Down (download,save,shopping)' ),
			array( 'las la-cart-plus' => 'Add to Shopping Cart (add,create,new,positive,shopping)' ),
			array( 'lab la-cc-amazon-pay' => 'Amazon Pay Credit Card' ),
			array( 'lab la-cc-amex' => 'American Express Credit Card (amex)' ),
			array( 'lab la-cc-apple-pay' => 'Apple Pay Credit Card' ),
			array( 'lab la-cc-diners-club' => 'Diner\'s Club Credit Card' ),
			array( 'lab la-cc-discover' => 'Discover Credit Card' ),
			array( 'lab la-cc-jcb' => 'JCB Credit Card' ),
			array( 'lab la-cc-mastercard' => 'MasterCard Credit Card' ),
			array( 'lab la-cc-paypal' => 'Paypal Credit Card' ),
			array( 'lab la-cc-stripe' => 'Stripe Credit Card' ),
			array( 'lab la-cc-visa' => 'Visa Credit Card' ),
			array( 'las la-certificate' => 'certificate (badge,star,verified)' ),
			array( 'las la-credit-card' => 'Credit Card (buy,checkout,credit-card-alt,debit,money,payment,purchase)' ),
			array( 'lar la-credit-card' => 'Credit Card (buy,checkout,credit-card-alt,debit,money,payment,purchase)' ),
			array( 'lab la-ethereum' => 'Ethereum' ),
			array( 'las la-gem' => 'Gem (diamond,jewelry,sapphire,stone,treasure)' ),
			array( 'lar la-gem' => 'Gem (diamond,jewelry,sapphire,stone,treasure)' ),
			array( 'las la-gift' => 'gift (christmas,generosity,giving,holiday,party,present,wrapped,xmas)' ),
			array( 'lab la-google-wallet' => 'Google Wallet' ),
			array( 'las la-handshake' => 'Handshake (agreement,greeting,meeting,partnership)' ),
			array( 'lar la-handshake' => 'Handshake (agreement,greeting,meeting,partnership)' ),
			array( 'las la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'lar la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'las la-key' => 'key (lock,password,private,secret,unlock)' ),
			array( 'las la-money-check' => 'Money Check (bank check,buy,checkout,cheque,money,payment,price,purchase)' ),
			array( 'las la-money-check-alt' => 'Alternate Money Check (bank check,buy,checkout,cheque,money,payment,price,purchase)' ),
			array( 'lab la-paypal' => 'Paypal' ),
			array( 'las la-receipt' => 'Receipt (check,invoice,money,pay,table)' ),
			array( 'las la-shopping-bag' => 'Shopping Bag (buy,checkout,grocery,payment,purchase)' ),
			array( 'las la-shopping-basket' => 'Shopping Basket (buy,checkout,grocery,payment,purchase)' ),
			array( 'las la-shopping-cart' => 'shopping-cart (buy,checkout,grocery,payment,purchase)' ),
			array( 'las la-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'lar la-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'lab la-stripe' => 'Stripe' ),
			array( 'lab la-stripe-s' => 'Stripe S' ),
			array( 'las la-tag' => 'tag (discount,label,price,shopping)' ),
			array( 'las la-tags' => 'tags (discount,label,price,shopping)' ),
			array( 'las la-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'lar la-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'las la-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
			array( 'lar la-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
			array( 'las la-trophy' => 'trophy (achievement,award,cup,game,winner)' ),
		),
		'Pharmacy' => array(
			array( 'las la-band-aid' => 'Band-Aid (bandage,boo boo,first aid,ouch)' ),
			array( 'las la-book-medical' => 'Medical Book (diary,documentation,health,history,journal,library,read,record)' ),
			array( 'las la-cannabis' => 'Cannabis (bud,chronic,drugs,endica,endo,ganja,marijuana,mary jane,pot,reefer,sativa,spliff,weed,whacky-tabacky)' ),
			array( 'las la-capsules' => 'Capsules (drugs,medicine,pills,prescription)' ),
			array( 'las la-clinic-medical' => 'Medical Clinic (doctor,general practitioner,hospital,infirmary,medicine,office,outpatient)' ),
			array( 'las la-eye-dropper' => 'Eye Dropper (beaker,clone,color,copy,eyedropper,pipette)' ),
			array( 'las la-file-medical' => 'Medical File (document,health,history,prescription,record)' ),
			array( 'las la-file-prescription' => 'File Prescription (document,drugs,medical,medicine,rx)' ),
			array( 'las la-first-aid' => 'First Aid (emergency,emt,health,medical,rescue)' ),
			array( 'las la-flask' => 'Flask (beaker,experimental,labs,science)' ),
			array( 'las la-history' => 'History (Rewind,clock,reverse,time,time machine)' ),
			array( 'las la-joint' => 'Joint (blunt,cannabis,doobie,drugs,marijuana,roach,smoke,smoking,spliff)' ),
			array( 'las la-laptop-medical' => 'Laptop Medical (computer,device,ehr,electronic health records,history)' ),
			array( 'las la-mortar-pestle' => 'Mortar Pestle (crush,culinary,grind,medical,mix,pharmacy,prescription,spices)' ),
			array( 'las la-notes-medical' => 'Medical Notes (clipboard,doctor,ehr,health,history,records)' ),
			array( 'las la-pills' => 'Pills (drugs,medicine,prescription,tablets)' ),
			array( 'las la-prescription' => 'Prescription (drugs,medical,medicine,pharmacy,rx)' ),
			array( 'las la-prescription-bottle' => 'Prescription Bottle (drugs,medical,medicine,pharmacy,rx)' ),
			array( 'las la-prescription-bottle-alt' => 'Alternate Prescription Bottle (drugs,medical,medicine,pharmacy,rx)' ),
			array( 'las la-receipt' => 'Receipt (check,invoice,money,pay,table)' ),
			array( 'las la-skull-crossbones' => 'Skull & Crossbones (Dungeons & Dragons,alert,bones,d&d,danger,dead,deadly,death,dnd,fantasy,halloween,holiday,jolly-roger,pirate,poison,skeleton,warning)' ),
			array( 'las la-syringe' => 'Syringe (doctor,immunizations,medical,needle)' ),
			array( 'las la-tablets' => 'Tablets (drugs,medicine,pills,prescription)' ),
			array( 'las la-thermometer' => 'Thermometer (mercury,status,temperature)' ),
			array( 'las la-vial' => 'Vial (experiment,lab,sample,science,test,test tube)' ),
			array( 'las la-vials' => 'Vials (experiment,lab,sample,science,test,test tube)' ),
		),
		'Political' => array(
			array( 'las la-award' => 'Award (honor,praise,prize,recognition,ribbon,trophy)' ),
			array( 'las la-balance-scale' => 'Balance Scale (balanced,justice,legal,measure,weight)' ),
			array( 'las la-balance-scale-left' => 'Balance Scale (Left-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'las la-balance-scale-right' => 'Balance Scale (Right-Weighted) (justice,legal,measure,unbalanced,weight)' ),
			array( 'las la-bullhorn' => 'bullhorn (announcement,broadcast,louder,megaphone,share)' ),
			array( 'las la-check-double' => 'Double Check (accept,agree,checkmark,confirm,correct,done,notice,notification,notify,ok,select,success,tick,todo)' ),
			array( 'las la-democrat' => 'Democrat (american,democratic party,donkey,election,left,left-wing,liberal,politics,usa)' ),
			array( 'las la-donate' => 'Donate (contribute,generosity,gift,give)' ),
			array( 'las la-dove' => 'Dove (bird,fauna,flying,peace,war)' ),
			array( 'las la-fist-raised' => 'Raised Fist (Dungeons & Dragons,d&d,dnd,fantasy,hand,ki,monk,resist,strength,unarmed combat)' ),
			array( 'las la-flag-usa' => 'United States of America Flag (betsy ross,country,old glory,stars,stripes,symbol)' ),
			array( 'las la-handshake' => 'Handshake (agreement,greeting,meeting,partnership)' ),
			array( 'lar la-handshake' => 'Handshake (agreement,greeting,meeting,partnership)' ),
			array( 'las la-person-booth' => 'Person Entering Booth (changing,changing room,election,human,person,vote,voting)' ),
			array( 'las la-piggy-bank' => 'Piggy Bank (bank,save,savings)' ),
			array( 'las la-republican' => 'Republican (american,conservative,election,elephant,politics,republican party,right,right-wing,usa)' ),
			array( 'las la-vote-yea' => 'Vote Yea (accept,cast,election,politics,positive,yes)' ),
		),
		'Religion' => array(
			array( 'las la-ankh' => 'Ankh (amulet,copper,coptic christianity,copts,crux ansata,egypt,venus)' ),
			array( 'las la-atom' => 'Atom (atheism,chemistry,ion,nuclear,science)' ),
			array( 'las la-bible' => 'Bible (book,catholicism,christianity,god,holy)' ),
			array( 'las la-church' => 'Church (building,cathedral,chapel,community,religion)' ),
			array( 'las la-cross' => 'Cross (catholicism,christianity,church,jesus)' ),
			array( 'las la-dharmachakra' => 'Dharmachakra (buddhism,buddhist,wheel of dharma)' ),
			array( 'las la-dove' => 'Dove (bird,fauna,flying,peace,war)' ),
			array( 'las la-gopuram' => 'Gopuram (building,entrance,hinduism,temple,tower)' ),
			array( 'las la-hamsa' => 'Hamsa (amulet,christianity,islam,jewish,judaism,muslim,protection)' ),
			array( 'las la-hanukiah' => 'Hanukiah (candle,hanukkah,jewish,judaism,light)' ),
			array( 'las la-haykal' => 'Haykal (bahai,bahá\'í,star)' ),
			array( 'las la-jedi' => 'Jedi (crest,force,sith,skywalker,star wars,yoda)' ),
			array( 'las la-journal-whills' => 'Journal of the Whills (book,force,jedi,sith,star wars,yoda)' ),
			array( 'las la-kaaba' => 'Kaaba (building,cube,islam,muslim)' ),
			array( 'las la-khanda' => 'Khanda (chakkar,sikh,sikhism,sword)' ),
			array( 'las la-menorah' => 'Menorah (candle,hanukkah,jewish,judaism,light)' ),
			array( 'las la-mosque' => 'Mosque (building,islam,landmark,muslim)' ),
			array( 'las la-om' => 'Om (buddhism,hinduism,jainism,mantra)' ),
			array( 'las la-pastafarianism' => 'Pastafarianism (agnosticism,atheism,flying spaghetti monster,fsm)' ),
			array( 'las la-peace' => 'Peace (serenity,tranquility,truce,war)' ),
			array( 'las la-place-of-worship' => 'Place of Worship (building,church,holy,mosque,synagogue)' ),
			array( 'las la-pray' => 'Pray (kneel,preach,religion,worship)' ),
			array( 'las la-praying-hands' => 'Praying Hands (kneel,preach,religion,worship)' ),
			array( 'las la-quran' => 'Quran (book,islam,muslim,religion)' ),
			array( 'las la-star-and-crescent' => 'Star and Crescent (islam,muslim,religion)' ),
			array( 'las la-star-of-david' => 'Star of David (jewish,judaism,religion)' ),
			array( 'las la-synagogue' => 'Synagogue (building,jewish,judaism,religion,star of david,temple)' ),
			array( 'las la-torah' => 'Torah (book,jewish,judaism,religion,scroll)' ),
			array( 'las la-torii-gate' => 'Torii Gate (building,shintoism)' ),
			array( 'las la-vihara' => 'Vihara (buddhism,buddhist,building,monastery)' ),
			array( 'las la-yin-yang' => 'Yin Yang (daoism,opposites,taoism)' ),
		),
		'Science' => array(
			array( 'las la-atom' => 'Atom (atheism,chemistry,ion,nuclear,science)' ),
			array( 'las la-biohazard' => 'Biohazard (danger,dangerous,hazmat,medical,radioactive,toxic,waste,zombie)' ),
			array( 'las la-brain' => 'Brain (cerebellum,gray matter,intellect,medulla oblongata,mind,noodle,wit)' ),
			array( 'las la-burn' => 'Burn (caliente,energy,fire,flame,gas,heat,hot)' ),
			array( 'las la-capsules' => 'Capsules (drugs,medicine,pills,prescription)' ),
			array( 'las la-clipboard-check' => 'Clipboard with Check (accept,agree,confirm,done,ok,select,success,tick,todo,yes)' ),
			array( 'las la-dna' => 'DNA (double helix,genetic,helix,molecule,protein)' ),
			array( 'las la-eye-dropper' => 'Eye Dropper (beaker,clone,color,copy,eyedropper,pipette)' ),
			array( 'las la-filter' => 'Filter (funnel,options,separate,sort)' ),
			array( 'las la-fire' => 'fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'las la-fire-alt' => 'Alternate Fire (burn,caliente,flame,heat,hot,popular)' ),
			array( 'las la-flask' => 'Flask (beaker,experimental,labs,science)' ),
			array( 'las la-frog' => 'Frog (amphibian,bullfrog,fauna,hop,kermit,kiss,prince,ribbit,toad,wart)' ),
			array( 'las la-magnet' => 'magnet (Attract,lodestone,tool)' ),
			array( 'las la-microscope' => 'Microscope (electron,lens,optics,science,shrink)' ),
			array( 'las la-mortar-pestle' => 'Mortar Pestle (crush,culinary,grind,medical,mix,pharmacy,prescription,spices)' ),
			array( 'las la-pills' => 'Pills (drugs,medicine,prescription,tablets)' ),
			array( 'las la-prescription-bottle' => 'Prescription Bottle (drugs,medical,medicine,pharmacy,rx)' ),
			array( 'las la-radiation' => 'Radiation (danger,dangerous,deadly,hazard,nuclear,radioactive,warning)' ),
			array( 'las la-radiation-alt' => 'Alternate Radiation (danger,dangerous,deadly,hazard,nuclear,radioactive,warning)' ),
			array( 'las la-seedling' => 'Seedling (flora,grow,plant,vegan)' ),
			array( 'las la-skull-crossbones' => 'Skull & Crossbones (Dungeons & Dragons,alert,bones,d&d,danger,dead,deadly,death,dnd,fantasy,halloween,holiday,jolly-roger,pirate,poison,skeleton,warning)' ),
			array( 'las la-syringe' => 'Syringe (doctor,immunizations,medical,needle)' ),
			array( 'las la-tablets' => 'Tablets (drugs,medicine,pills,prescription)' ),
			array( 'las la-temperature-high' => 'High Temperature (cook,mercury,summer,thermometer,warm)' ),
			array( 'las la-temperature-low' => 'Low Temperature (cold,cool,mercury,thermometer,winter)' ),
			array( 'las la-vial' => 'Vial (experiment,lab,sample,science,test,test tube)' ),
			array( 'las la-vials' => 'Vials (experiment,lab,sample,science,test,test tube)' ),
		),
		'Science Fiction' => array(
			array( 'lab la-galactic-republic' => 'Galactic Republic (politics,star wars)' ),
			array( 'lab la-galactic-senate' => 'Galactic Senate (star wars)' ),
			array( 'las la-globe' => 'Globe (all,coordinates,country,earth,global,gps,language,localize,location,map,online,place,planet,translate,travel,world)' ),
			array( 'las la-jedi' => 'Jedi (crest,force,sith,skywalker,star wars,yoda)' ),
			array( 'lab la-jedi-order' => 'Jedi Order (star wars)' ),
			array( 'las la-journal-whills' => 'Journal of the Whills (book,force,jedi,sith,star wars,yoda)' ),
			array( 'las la-meteor' => 'Meteor (armageddon,asteroid,comet,shooting star,space)' ),
			array( 'las la-moon' => 'Moon (contrast,crescent,dark,lunar,night)' ),
			array( 'lar la-moon' => 'Moon (contrast,crescent,dark,lunar,night)' ),
			array( 'lab la-old-republic' => 'Old Republic (politics,star wars)' ),
			array( 'las la-robot' => 'Robot (android,automate,computer,cyborg)' ),
			array( 'las la-rocket' => 'rocket (aircraft,app,jet,launch,nasa,space)' ),
			array( 'las la-satellite' => 'Satellite (communications,hardware,orbit,space)' ),
			array( 'las la-satellite-dish' => 'Satellite Dish (SETI,communications,hardware,receiver,saucer,signal)' ),
			array( 'las la-space-shuttle' => 'Space Shuttle (astronaut,machine,nasa,rocket,transportation)' ),
			array( 'las la-user-astronaut' => 'User Astronaut (avatar,clothing,cosmonaut,nasa,space,suit)' ),
		),
		'Security' => array(
			array( 'las la-ban' => 'ban (abort,ban,block,cancel,delete,hide,prohibit,remove,stop,trash)' ),
			array( 'las la-bug' => 'Bug (beetle,error,insect,report)' ),
			array( 'las la-door-closed' => 'Door Closed (enter,exit,locked)' ),
			array( 'las la-door-open' => 'Door Open (enter,exit,welcome)' ),
			array( 'las la-dungeon' => 'Dungeon (Dungeons & Dragons,building,d&d,dnd,door,entrance,fantasy,gate)' ),
			array( 'las la-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'lar la-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'las la-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'lar la-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'las la-file-contract' => 'File Contract (agreement,binding,document,legal,signature)' ),
			array( 'las la-file-signature' => 'File Signature (John Hancock,contract,document,name)' ),
			array( 'las la-fingerprint' => 'Fingerprint (human,id,identification,lock,smudge,touch,unique,unlock)' ),
			array( 'las la-id-badge' => 'Identification Badge (address,contact,identification,license,profile)' ),
			array( 'lar la-id-badge' => 'Identification Badge (address,contact,identification,license,profile)' ),
			array( 'las la-id-card' => 'Identification Card (contact,demographics,document,identification,issued,profile)' ),
			array( 'lar la-id-card' => 'Identification Card (contact,demographics,document,identification,issued,profile)' ),
			array( 'las la-id-card-alt' => 'Alternate Identification Card (contact,demographics,document,identification,issued,profile)' ),
			array( 'las la-key' => 'key (lock,password,private,secret,unlock)' ),
			array( 'las la-lock' => 'lock (admin,lock,open,password,private,protect,security)' ),
			array( 'las la-lock-open' => 'Lock Open (admin,lock,open,password,private,protect,security)' ),
			array( 'las la-mask' => 'Mask (carnivale,costume,disguise,halloween,secret,super hero)' ),
			array( 'las la-passport' => 'Passport (document,id,identification,issued,travel)' ),
			array( 'las la-shield-alt' => 'Alternate Shield (achievement,award,block,defend,security,winner)' ),
			array( 'las la-unlock' => 'unlock (admin,lock,password,private,protect)' ),
			array( 'las la-unlock-alt' => 'Alternate Unlock (admin,lock,password,private,protect)' ),
			array( 'las la-user-lock' => 'User Lock (admin,lock,person,private,unlock)' ),
			array( 'las la-user-secret' => 'User Secret (clothing,coat,hat,incognito,person,privacy,spy,whisper)' ),
			array( 'las la-user-shield' => 'User Shield (admin,person,private,protect,safe)' ),
		),
		'Shapes' => array(
			array( 'las la-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'lar la-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'las la-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'lar la-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'las la-certificate' => 'certificate (badge,star,verified)' ),
			array( 'las la-circle' => 'Circle (circle-thin,diameter,dot,ellipse,notification,round)' ),
			array( 'lar la-circle' => 'Circle (circle-thin,diameter,dot,ellipse,notification,round)' ),
			array( 'las la-cloud' => 'Cloud (atmosphere,fog,overcast,save,upload,weather)' ),
			array( 'las la-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'lar la-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'las la-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'lar la-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'las la-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'lar la-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'las la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'lar la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'las la-heart-broken' => 'Heart Broken (breakup,crushed,dislike,dumped,grief,love,lovesick,relationship,sad)' ),
			array( 'las la-map-marker' => 'map-marker (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'las la-play' => 'play (audio,music,playing,sound,start,video)' ),
			array( 'las la-shapes' => 'Shapes (blocks,build,circle,square,triangle)' ),
			array( 'las la-square' => 'Square (block,box,shape)' ),
			array( 'lar la-square' => 'Square (block,box,shape)' ),
			array( 'las la-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'lar la-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
		),
		'Shopping' => array(
			array( 'las la-barcode' => 'barcode (info,laser,price,scan,upc)' ),
			array( 'las la-cart-arrow-down' => 'Shopping Cart Arrow Down (download,save,shopping)' ),
			array( 'las la-cart-plus' => 'Add to Shopping Cart (add,create,new,positive,shopping)' ),
			array( 'las la-cash-register' => 'Cash Register (buy,cha-ching,change,checkout,commerce,leaerboard,machine,pay,payment,purchase,store)' ),
			array( 'las la-gift' => 'gift (christmas,generosity,giving,holiday,party,present,wrapped,xmas)' ),
			array( 'las la-gifts' => 'Gifts (christmas,generosity,giving,holiday,party,present,wrapped,xmas)' ),
			array( 'las la-person-booth' => 'Person Entering Booth (changing,changing room,election,human,person,vote,voting)' ),
			array( 'las la-receipt' => 'Receipt (check,invoice,money,pay,table)' ),
			array( 'las la-shipping-fast' => 'Shipping Fast (express,fedex,mail,overnight,package,ups)' ),
			array( 'las la-shopping-bag' => 'Shopping Bag (buy,checkout,grocery,payment,purchase)' ),
			array( 'las la-shopping-basket' => 'Shopping Basket (buy,checkout,grocery,payment,purchase)' ),
			array( 'las la-shopping-cart' => 'shopping-cart (buy,checkout,grocery,payment,purchase)' ),
			array( 'las la-store' => 'Store (building,buy,purchase,shopping)' ),
			array( 'las la-store-alt' => 'Alternate Store (building,buy,purchase,shopping)' ),
			array( 'las la-truck' => 'truck (cargo,delivery,shipping,vehicle)' ),
			array( 'las la-tshirt' => 'T-Shirt (clothing,fashion,garment,shirt)' ),
		),
		'Social' => array(
			array( 'las la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'lar la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'las la-birthday-cake' => 'Birthday Cake (anniversary,bakery,candles,celebration,dessert,frosting,holiday,party,pastry)' ),
			array( 'las la-camera' => 'camera (image,lens,photo,picture,record,shutter,video)' ),
			array( 'las la-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'lar la-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'las la-comment-alt' => 'Alternate Comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'lar la-comment-alt' => 'Alternate Comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'las la-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'lar la-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'las la-hashtag' => 'Hashtag (Twitter,instagram,pound,social media,tag)' ),
			array( 'las la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'lar la-heart' => 'Heart (favorite,like,love,relationship,valentine)' ),
			array( 'las la-icons' => 'Icons (bolt,emoji,heart,image,music,photo,symbols)' ),
			array( 'las la-image' => 'Image (album,landscape,photo,picture)' ),
			array( 'lar la-image' => 'Image (album,landscape,photo,picture)' ),
			array( 'las la-images' => 'Images (album,landscape,photo,picture)' ),
			array( 'lar la-images' => 'Images (album,landscape,photo,picture)' ),
			array( 'las la-map-marker' => 'map-marker (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'las la-map-marker-alt' => 'Alternate Map Marker (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'las la-photo-video' => 'Photo Video (av,film,image,library,media)' ),
			array( 'las la-poll' => 'Poll (results,survey,trend,vote,voting)' ),
			array( 'las la-poll-h' => 'Poll H (results,survey,trend,vote,voting)' ),
			array( 'las la-retweet' => 'Retweet (refresh,reload,share,swap)' ),
			array( 'las la-share' => 'Share (forward,save,send,social)' ),
			array( 'las la-share-alt' => 'Alternate Share (forward,save,send,social)' ),
			array( 'las la-share-square' => 'Share Square (forward,save,send,social)' ),
			array( 'lar la-share-square' => 'Share Square (forward,save,send,social)' ),
			array( 'las la-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'lar la-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'las la-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'lar la-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'las la-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
			array( 'lar la-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
			array( 'las la-thumbtack' => 'Thumbtack (coordinates,location,marker,pin,thumb-tack)' ),
			array( 'las la-user' => 'User (account,avatar,head,human,man,person,profile)' ),
			array( 'lar la-user' => 'User (account,avatar,head,human,man,person,profile)' ),
			array( 'las la-user-circle' => 'User Circle (account,avatar,head,human,man,person,profile)' ),
			array( 'lar la-user-circle' => 'User Circle (account,avatar,head,human,man,person,profile)' ),
			array( 'las la-user-friends' => 'User Friends (group,people,person,team,users)' ),
			array( 'las la-user-plus' => 'User Plus (add,avatar,positive,sign up,signup,team)' ),
			array( 'las la-users' => 'Users (friends,group,people,persons,profiles,team)' ),
			array( 'las la-video' => 'Video (camera,film,movie,record,video-camera)' ),
		),
		'Spinners' => array(
			array( 'las la-asterisk' => 'asterisk (annotation,details,reference,star)' ),
			array( 'las la-atom' => 'Atom (atheism,chemistry,ion,nuclear,science)' ),
			array( 'las la-certificate' => 'certificate (badge,star,verified)' ),
			array( 'las la-circle-notch' => 'Circle Notched (circle-o-notch,diameter,dot,ellipse,round,spinner)' ),
			array( 'las la-cog' => 'cog (gear,mechanical,settings,sprocket,wheel)' ),
			array( 'las la-compact-disc' => 'Compact Disc (album,bluray,cd,disc,dvd,media,movie,music,record,video,vinyl)' ),
			array( 'las la-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'lar la-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'las la-crosshairs' => 'Crosshairs (aim,bullseye,gpd,picker,position)' ),
			array( 'las la-dharmachakra' => 'Dharmachakra (buddhism,buddhist,wheel of dharma)' ),
			array( 'las la-fan' => 'Fan (ac,air conditioning,blade,blower,cool,hot)' ),
			array( 'las la-haykal' => 'Haykal (bahai,bahá\'í,star)' ),
			array( 'las la-life-ring' => 'Life Ring (coast guard,help,overboard,save,support)' ),
			array( 'lar la-life-ring' => 'Life Ring (coast guard,help,overboard,save,support)' ),
			array( 'las la-palette' => 'Palette (acrylic,art,brush,color,fill,paint,pigment,watercolor)' ),
			array( 'las la-ring' => 'Ring (Dungeons & Dragons,Gollum,band,binding,d&d,dnd,engagement,fantasy,gold,jewelry,marriage,precious)' ),
			array( 'las la-slash' => 'Slash (cancel,close,mute,off,stop,x)' ),
			array( 'las la-snowflake' => 'Snowflake (precipitation,rain,winter)' ),
			array( 'lar la-snowflake' => 'Snowflake (precipitation,rain,winter)' ),
			array( 'las la-spinner' => 'Spinner (circle,loading,progress)' ),
			array( 'las la-stroopwafel' => 'Stroopwafel (caramel,cookie,dessert,sweets,waffle)' ),
			array( 'las la-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'lar la-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'las la-sync' => 'Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'las la-sync-alt' => 'Alternate Sync (exchange,refresh,reload,rotate,swap)' ),
			array( 'las la-yin-yang' => 'Yin Yang (daoism,opposites,taoism)' ),
		),
		'Sports' => array(
			array( 'las la-baseball-ball' => 'Baseball Ball (foul,hardball,league,leather,mlb,softball,sport)' ),
			array( 'las la-basketball-ball' => 'Basketball Ball (dribble,dunk,hoop,nba)' ),
			array( 'las la-biking' => 'Biking (bicycle,bike,cycle,cycling,ride,wheel)' ),
			array( 'las la-bowling-ball' => 'Bowling Ball (alley,candlepin,gutter,lane,strike,tenpin)' ),
			array( 'las la-dumbbell' => 'Dumbbell (exercise,gym,strength,weight,weight-lifting)' ),
			array( 'las la-football-ball' => 'Football Ball (ball,fall,nfl,pigskin,seasonal)' ),
			array( 'las la-futbol' => 'Futbol (ball,football,mls,soccer)' ),
			array( 'lar la-futbol' => 'Futbol (ball,football,mls,soccer)' ),
			array( 'las la-golf-ball' => 'Golf Ball (caddy,eagle,putt,tee)' ),
			array( 'las la-hockey-puck' => 'Hockey Puck (ice,nhl,sport)' ),
			array( 'las la-quidditch' => 'Quidditch (ball,bludger,broom,golden snitch,harry potter,hogwarts,quaffle,sport,wizard)' ),
			array( 'las la-running' => 'Running (exercise,health,jog,person,run,sport,sprint)' ),
			array( 'las la-skating' => 'Skating (activity,figure skating,fitness,ice,person,winter)' ),
			array( 'las la-skiing' => 'Skiing (activity,downhill,fast,fitness,olympics,outdoors,person,seasonal,slalom)' ),
			array( 'las la-skiing-nordic' => 'Skiing Nordic (activity,cross country,fitness,outdoors,person,seasonal)' ),
			array( 'las la-snowboarding' => 'Snowboarding (activity,fitness,olympics,outdoors,person)' ),
			array( 'las la-swimmer' => 'Swimmer (athlete,head,man,olympics,person,pool,water)' ),
			array( 'las la-table-tennis' => 'Table Tennis (ball,paddle,ping pong)' ),
			array( 'las la-volleyball-ball' => 'Volleyball Ball (beach,olympics,sport)' ),
		),
		'Spring' => array(
			array( 'las la-allergies' => 'Allergies (allergy,freckles,hand,hives,pox,skin,spots)' ),
			array( 'las la-broom' => 'Broom (clean,firebolt,fly,halloween,nimbus 2000,quidditch,sweep,witch)' ),
			array( 'las la-cloud-sun' => 'Cloud with Sun (clear,day,daytime,fall,outdoors,overcast,partly cloudy)' ),
			array( 'las la-cloud-sun-rain' => 'Cloud with Sun and Rain (day,overcast,precipitation,storm,summer,sunshower)' ),
			array( 'las la-frog' => 'Frog (amphibian,bullfrog,fauna,hop,kermit,kiss,prince,ribbit,toad,wart)' ),
			array( 'las la-rainbow' => 'Rainbow (gold,leprechaun,prism,rain,sky)' ),
			array( 'las la-seedling' => 'Seedling (flora,grow,plant,vegan)' ),
			array( 'las la-umbrella' => 'Umbrella (protection,rain,storm,wet)' ),
		),
		'Status' => array(
			array( 'las la-ban' => 'ban (abort,ban,block,cancel,delete,hide,prohibit,remove,stop,trash)' ),
			array( 'las la-battery-empty' => 'Battery Empty (charge,dead,power,status)' ),
			array( 'las la-battery-full' => 'Battery Full (charge,power,status)' ),
			array( 'las la-battery-half' => 'Battery 1/2 Full (charge,power,status)' ),
			array( 'las la-battery-quarter' => 'Battery 1/4 Full (charge,low,power,status)' ),
			array( 'las la-battery-three-quarters' => 'Battery 3/4 Full (charge,power,status)' ),
			array( 'las la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'lar la-bell' => 'bell (alarm,alert,chime,notification,reminder)' ),
			array( 'las la-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'lar la-bell-slash' => 'Bell Slash (alert,cancel,disabled,notification,off,reminder)' ),
			array( 'las la-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'lar la-calendar' => 'Calendar (calendar-o,date,event,schedule,time,when)' ),
			array( 'las la-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'lar la-calendar-alt' => 'Alternate Calendar (calendar,date,event,schedule,time,when)' ),
			array( 'las la-calendar-check' => 'Calendar Check (accept,agree,appointment,confirm,correct,date,done,event,ok,schedule,select,success,tick,time,todo,when)' ),
			array( 'lar la-calendar-check' => 'Calendar Check (accept,agree,appointment,confirm,correct,date,done,event,ok,schedule,select,success,tick,time,todo,when)' ),
			array( 'las la-calendar-day' => 'Calendar with Day Focus (date,detail,event,focus,schedule,single day,time,today,when)' ),
			array( 'las la-calendar-minus' => 'Calendar Minus (calendar,date,delete,event,negative,remove,schedule,time,when)' ),
			array( 'lar la-calendar-minus' => 'Calendar Minus (calendar,date,delete,event,negative,remove,schedule,time,when)' ),
			array( 'las la-calendar-plus' => 'Calendar Plus (add,calendar,create,date,event,new,positive,schedule,time,when)' ),
			array( 'lar la-calendar-plus' => 'Calendar Plus (add,calendar,create,date,event,new,positive,schedule,time,when)' ),
			array( 'las la-calendar-times' => 'Calendar Times (archive,calendar,date,delete,event,remove,schedule,time,when,x)' ),
			array( 'lar la-calendar-times' => 'Calendar Times (archive,calendar,date,delete,event,remove,schedule,time,when,x)' ),
			array( 'las la-calendar-week' => 'Calendar with Week Focus (date,detail,event,focus,schedule,single week,time,today,when)' ),
			array( 'las la-cart-arrow-down' => 'Shopping Cart Arrow Down (download,save,shopping)' ),
			array( 'las la-cart-plus' => 'Add to Shopping Cart (add,create,new,positive,shopping)' ),
			array( 'las la-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'lar la-comment' => 'comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'las la-comment-alt' => 'Alternate Comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'lar la-comment-alt' => 'Alternate Comment (bubble,chat,commenting,conversation,feedback,message,note,notification,sms,speech,texting)' ),
			array( 'las la-comment-slash' => 'Comment Slash (bubble,cancel,chat,commenting,conversation,feedback,message,mute,note,notification,quiet,sms,speech,texting)' ),
			array( 'las la-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'lar la-compass' => 'Compass (directions,directory,location,menu,navigation,safari,travel)' ),
			array( 'las la-door-closed' => 'Door Closed (enter,exit,locked)' ),
			array( 'las la-door-open' => 'Door Open (enter,exit,welcome)' ),
			array( 'las la-exclamation' => 'exclamation (alert,danger,error,important,notice,notification,notify,problem,warning)' ),
			array( 'las la-exclamation-circle' => 'Exclamation Circle (alert,danger,error,important,notice,notification,notify,problem,warning)' ),
			array( 'las la-exclamation-triangle' => 'Exclamation Triangle (alert,danger,error,important,notice,notification,notify,problem,warning)' ),
			array( 'las la-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'lar la-eye' => 'Eye (look,optic,see,seen,show,sight,views,visible)' ),
			array( 'las la-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'lar la-eye-slash' => 'Eye Slash (blind,hide,show,toggle,unseen,views,visible,visiblity)' ),
			array( 'las la-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'lar la-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'las la-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'lar la-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'las la-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'lar la-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'las la-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'lar la-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'las la-gas-pump' => 'Gas Pump (car,fuel,gasoline,petrol)' ),
			array( 'las la-info' => 'Info (details,help,information,more,support)' ),
			array( 'las la-info-circle' => 'Info Circle (details,help,information,more,support)' ),
			array( 'las la-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'lar la-lightbulb' => 'Lightbulb (energy,idea,inspiration,light)' ),
			array( 'las la-lock' => 'lock (admin,lock,open,password,private,protect,security)' ),
			array( 'las la-lock-open' => 'Lock Open (admin,lock,open,password,private,protect,security)' ),
			array( 'las la-map-marker' => 'map-marker (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'las la-map-marker-alt' => 'Alternate Map Marker (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'las la-microphone' => 'microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'las la-microphone-alt' => 'Alternate Microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'las la-microphone-alt-slash' => 'Alternate Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'las la-microphone-slash' => 'Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'las la-minus' => 'minus (collapse,delete,hide,minify,negative,remove,trash)' ),
			array( 'las la-minus-circle' => 'Minus Circle (delete,hide,negative,remove,shape,trash)' ),
			array( 'las la-minus-square' => 'Minus Square (collapse,delete,hide,minify,negative,remove,shape,trash)' ),
			array( 'lar la-minus-square' => 'Minus Square (collapse,delete,hide,minify,negative,remove,shape,trash)' ),
			array( 'las la-parking' => 'Parking (auto,car,garage,meter)' ),
			array( 'las la-phone' => 'Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'las la-phone-alt' => 'Alternate Phone (call,earphone,number,support,telephone,voice)' ),
			array( 'las la-phone-slash' => 'Phone Slash (call,cancel,earphone,mute,number,support,telephone,voice)' ),
			array( 'las la-plus' => 'plus (add,create,expand,new,positive,shape)' ),
			array( 'las la-plus-circle' => 'Plus Circle (add,create,expand,new,positive,shape)' ),
			array( 'las la-plus-square' => 'Plus Square (add,create,expand,new,positive,shape)' ),
			array( 'lar la-plus-square' => 'Plus Square (add,create,expand,new,positive,shape)' ),
			array( 'las la-print' => 'print (business,copy,document,office,paper)' ),
			array( 'las la-question' => 'Question (help,information,support,unknown)' ),
			array( 'las la-question-circle' => 'Question Circle (help,information,support,unknown)' ),
			array( 'lar la-question-circle' => 'Question Circle (help,information,support,unknown)' ),
			array( 'las la-shield-alt' => 'Alternate Shield (achievement,award,block,defend,security,winner)' ),
			array( 'las la-shopping-cart' => 'shopping-cart (buy,checkout,grocery,payment,purchase)' ),
			array( 'las la-sign-in-alt' => 'Alternate Sign In (arrow,enter,join,log in,login,sign in,sign up,sign-in,signin,signup)' ),
			array( 'las la-sign-out-alt' => 'Alternate Sign Out (arrow,exit,leave,log out,logout,sign-out)' ),
			array( 'las la-signal' => 'signal (bars,graph,online,reception,status)' ),
			array( 'las la-smoking-ban' => 'Smoking Ban (ban,cancel,no smoking,non-smoking)' ),
			array( 'las la-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'lar la-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'las la-star-half' => 'star-half (achievement,award,rating,score,star-half-empty,star-half-full)' ),
			array( 'lar la-star-half' => 'star-half (achievement,award,rating,score,star-half-empty,star-half-full)' ),
			array( 'las la-star-half-alt' => 'Alternate Star Half (achievement,award,rating,score,star-half-empty,star-half-full)' ),
			array( 'las la-stream' => 'Stream (flow,list,timeline)' ),
			array( 'las la-thermometer-empty' => 'Thermometer Empty (cold,mercury,status,temperature)' ),
			array( 'las la-thermometer-full' => 'Thermometer Full (fever,hot,mercury,status,temperature)' ),
			array( 'las la-thermometer-half' => 'Thermometer 1/2 Full (mercury,status,temperature)' ),
			array( 'las la-thermometer-quarter' => 'Thermometer 1/4 Full (mercury,status,temperature)' ),
			array( 'las la-thermometer-three-quarters' => 'Thermometer 3/4 Full (mercury,status,temperature)' ),
			array( 'las la-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'lar la-thumbs-down' => 'thumbs-down (disagree,disapprove,dislike,hand,social,thumbs-o-down)' ),
			array( 'las la-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
			array( 'lar la-thumbs-up' => 'thumbs-up (agree,approve,favorite,hand,like,ok,okay,social,success,thumbs-o-up,yes,you got it dude)' ),
			array( 'las la-tint' => 'tint (color,drop,droplet,raindrop,waterdrop)' ),
			array( 'las la-tint-slash' => 'Tint Slash (color,drop,droplet,raindrop,waterdrop)' ),
			array( 'las la-toggle-off' => 'Toggle Off (switch)' ),
			array( 'las la-toggle-on' => 'Toggle On (switch)' ),
			array( 'las la-unlock' => 'unlock (admin,lock,password,private,protect)' ),
			array( 'las la-unlock-alt' => 'Alternate Unlock (admin,lock,password,private,protect)' ),
			array( 'las la-user' => 'User (account,avatar,head,human,man,person,profile)' ),
			array( 'lar la-user' => 'User (account,avatar,head,human,man,person,profile)' ),
			array( 'las la-user-alt' => 'Alternate User (account,avatar,head,human,man,person,profile)' ),
			array( 'las la-user-alt-slash' => 'Alternate User Slash (account,avatar,head,human,man,person,profile)' ),
			array( 'las la-user-slash' => 'User Slash (ban,delete,remove)' ),
			array( 'las la-video' => 'Video (camera,film,movie,record,video-camera)' ),
			array( 'las la-video-slash' => 'Video Slash (add,create,film,new,positive,record,video)' ),
			array( 'las la-volume-down' => 'Volume Down (audio,lower,music,quieter,sound,speaker)' ),
			array( 'las la-volume-mute' => 'Volume Mute (audio,music,quiet,sound,speaker)' ),
			array( 'las la-volume-off' => 'Volume Off (audio,ban,music,mute,quiet,silent,sound)' ),
			array( 'las la-volume-up' => 'Volume Up (audio,higher,louder,music,sound,speaker)' ),
			array( 'las la-wifi' => 'WiFi (connection,hotspot,internet,network,wireless)' ),
		),
		'Summer' => array(
			array( 'las la-anchor' => 'Anchor (berth,boat,dock,embed,link,maritime,moor,secure)' ),
			array( 'las la-biking' => 'Biking (bicycle,bike,cycle,cycling,ride,wheel)' ),
			array( 'las la-fish' => 'Fish (fauna,gold,seafood,swimming)' ),
			array( 'las la-hotdog' => 'Hot Dog (bun,chili,frankfurt,frankfurter,kosher,polish,sandwich,sausage,vienna,weiner)' ),
			array( 'las la-ice-cream' => 'Ice Cream (chocolate,cone,dessert,frozen,scoop,sorbet,vanilla,yogurt)' ),
			array( 'las la-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'lar la-lemon' => 'Lemon (citrus,lemonade,lime,tart)' ),
			array( 'las la-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'lar la-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'las la-swimmer' => 'Swimmer (athlete,head,man,olympics,person,pool,water)' ),
			array( 'las la-swimming-pool' => 'Swimming Pool (ladder,recreation,swim,water)' ),
			array( 'las la-umbrella-beach' => 'Umbrella Beach (protection,recreation,sand,shade,summer,sun)' ),
			array( 'las la-volleyball-ball' => 'Volleyball Ball (beach,olympics,sport)' ),
			array( 'las la-water' => 'Water (lake,liquid,ocean,sea,swim,wet)' ),
		),
		'Tabletop Gaming' => array(
			array( 'lab la-acquisitions-incorporated' => 'Acquisitions Incorporated (Dungeons & Dragons,d&d,dnd,fantasy,game,gaming,tabletop)' ),
			array( 'las la-book-dead' => 'Book of the Dead (Dungeons & Dragons,crossbones,d&d,dark arts,death,dnd,documentation,evil,fantasy,halloween,holiday,necronomicon,read,skull,spell)' ),
			array( 'lab la-critical-role' => 'Critical Role (Dungeons & Dragons,d&d,dnd,fantasy,game,gaming,tabletop)' ),
			array( 'lab la-d-and-d' => 'Dungeons & Dragons' ),
			array( 'lab la-d-and-d-beyond' => 'D&D Beyond (Dungeons & Dragons,d&d,dnd,fantasy,gaming,tabletop)' ),
			array( 'las la-dice-d20' => 'Dice D20 (Dungeons & Dragons,chance,d&d,dnd,fantasy,gambling,game,roll)' ),
			array( 'las la-dice-d6' => 'Dice D6 (Dungeons & Dragons,chance,d&d,dnd,fantasy,gambling,game,roll)' ),
			array( 'las la-dragon' => 'Dragon (Dungeons & Dragons,d&d,dnd,fantasy,fire,lizard,serpent)' ),
			array( 'las la-dungeon' => 'Dungeon (Dungeons & Dragons,building,d&d,dnd,door,entrance,fantasy,gate)' ),
			array( 'lab la-fantasy-flight-games' => 'Fantasy Flight-games (Dungeons & Dragons,d&d,dnd,fantasy,game,gaming,tabletop)' ),
			array( 'las la-fist-raised' => 'Raised Fist (Dungeons & Dragons,d&d,dnd,fantasy,hand,ki,monk,resist,strength,unarmed combat)' ),
			array( 'las la-hat-wizard' => 'Wizard\'s Hat (Dungeons & Dragons,accessory,buckle,clothing,d&d,dnd,fantasy,halloween,head,holiday,mage,magic,pointy,witch)' ),
			array( 'lab la-penny-arcade' => 'Penny Arcade (Dungeons & Dragons,d&d,dnd,fantasy,game,gaming,pax,tabletop)' ),
			array( 'las la-ring' => 'Ring (Dungeons & Dragons,Gollum,band,binding,d&d,dnd,engagement,fantasy,gold,jewelry,marriage,precious)' ),
			array( 'las la-scroll' => 'Scroll (Dungeons & Dragons,announcement,d&d,dnd,fantasy,paper,script)' ),
			array( 'las la-skull-crossbones' => 'Skull & Crossbones (Dungeons & Dragons,alert,bones,d&d,danger,dead,deadly,death,dnd,fantasy,halloween,holiday,jolly-roger,pirate,poison,skeleton,warning)' ),
			array( 'lab la-wizards-of-the-coast' => 'Wizards of the Coast (Dungeons & Dragons,d&d,dnd,fantasy,game,gaming,tabletop)' ),
		),
		'Toggle' => array(
			array( 'las la-bullseye' => 'Bullseye (archery,goal,objective,target)' ),
			array( 'las la-check-circle' => 'Check Circle (accept,agree,confirm,correct,done,ok,select,success,tick,todo,yes)' ),
			array( 'lar la-check-circle' => 'Check Circle (accept,agree,confirm,correct,done,ok,select,success,tick,todo,yes)' ),
			array( 'las la-circle' => 'Circle (circle-thin,diameter,dot,ellipse,notification,round)' ),
			array( 'lar la-circle' => 'Circle (circle-thin,diameter,dot,ellipse,notification,round)' ),
			array( 'las la-dot-circle' => 'Dot Circle (bullseye,notification,target)' ),
			array( 'lar la-dot-circle' => 'Dot Circle (bullseye,notification,target)' ),
			array( 'las la-microphone' => 'microphone (audio,podcast,record,sing,sound,voice)' ),
			array( 'las la-microphone-slash' => 'Microphone Slash (audio,disable,mute,podcast,record,sing,sound,voice)' ),
			array( 'las la-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'lar la-star' => 'Star (achievement,award,favorite,important,night,rating,score)' ),
			array( 'las la-star-half' => 'star-half (achievement,award,rating,score,star-half-empty,star-half-full)' ),
			array( 'lar la-star-half' => 'star-half (achievement,award,rating,score,star-half-empty,star-half-full)' ),
			array( 'las la-star-half-alt' => 'Alternate Star Half (achievement,award,rating,score,star-half-empty,star-half-full)' ),
			array( 'las la-toggle-off' => 'Toggle Off (switch)' ),
			array( 'las la-toggle-on' => 'Toggle On (switch)' ),
			array( 'las la-wifi' => 'WiFi (connection,hotspot,internet,network,wireless)' ),
		),
		'Travel' => array(
			array( 'las la-archway' => 'Archway (arc,monument,road,street,tunnel)' ),
			array( 'las la-atlas' => 'Atlas (book,directions,geography,globe,map,travel,wayfinding)' ),
			array( 'las la-bed' => 'Bed (lodging,rest,sleep,travel)' ),
			array( 'las la-bus' => 'Bus (public transportation,transportation,travel,vehicle)' ),
			array( 'las la-bus-alt' => 'Bus Alt (mta,public transportation,transportation,travel,vehicle)' ),
			array( 'las la-cocktail' => 'Cocktail (alcohol,beverage,drink,gin,glass,margarita,martini,vodka)' ),
			array( 'las la-concierge-bell' => 'Concierge Bell (attention,hotel,receptionist,service,support)' ),
			array( 'las la-dumbbell' => 'Dumbbell (exercise,gym,strength,weight,weight-lifting)' ),
			array( 'las la-glass-martini' => 'Martini Glass (alcohol,bar,beverage,drink,liquor)' ),
			array( 'las la-glass-martini-alt' => 'Alternate Glass Martini (alcohol,bar,beverage,drink,liquor)' ),
			array( 'las la-globe-africa' => 'Globe with Africa shown (all,country,earth,global,gps,language,localize,location,map,online,place,planet,translate,travel,world)' ),
			array( 'las la-globe-americas' => 'Globe with Americas shown (all,country,earth,global,gps,language,localize,location,map,online,place,planet,translate,travel,world)' ),
			array( 'las la-globe-asia' => 'Globe with Asia shown (all,country,earth,global,gps,language,localize,location,map,online,place,planet,translate,travel,world)' ),
			array( 'las la-globe-europe' => 'Globe with Europe shown (all,country,earth,global,gps,language,localize,location,map,online,place,planet,translate,travel,world)' ),
			array( 'las la-hot-tub' => 'Hot Tub (bath,jacuzzi,massage,sauna,spa)' ),
			array( 'las la-hotel' => 'Hotel (building,inn,lodging,motel,resort,travel)' ),
			array( 'las la-luggage-cart' => 'Luggage Cart (bag,baggage,suitcase,travel)' ),
			array( 'las la-map' => 'Map (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'lar la-map' => 'Map (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'las la-map-marked' => 'Map Marked (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'las la-map-marked-alt' => 'Alternate Map Marked (address,coordinates,destination,gps,localize,location,map,navigation,paper,pin,place,point of interest,position,route,travel)' ),
			array( 'las la-monument' => 'Monument (building,historic,landmark,memorable)' ),
			array( 'las la-passport' => 'Passport (document,id,identification,issued,travel)' ),
			array( 'las la-plane' => 'plane (airplane,destination,fly,location,mode,travel,trip)' ),
			array( 'las la-plane-arrival' => 'Plane Arrival (airplane,arriving,destination,fly,land,landing,location,mode,travel,trip)' ),
			array( 'las la-plane-departure' => 'Plane Departure (airplane,departing,destination,fly,location,mode,take off,taking off,travel,trip)' ),
			array( 'las la-shuttle-van' => 'Shuttle Van (airport,machine,public-transportation,transportation,travel,vehicle)' ),
			array( 'las la-spa' => 'Spa (flora,massage,mindfulness,plant,wellness)' ),
			array( 'las la-suitcase' => 'Suitcase (baggage,luggage,move,suitcase,travel,trip)' ),
			array( 'las la-suitcase-rolling' => 'Suitcase Rolling (baggage,luggage,move,suitcase,travel,trip)' ),
			array( 'las la-swimmer' => 'Swimmer (athlete,head,man,olympics,person,pool,water)' ),
			array( 'las la-swimming-pool' => 'Swimming Pool (ladder,recreation,swim,water)' ),
			array( 'las la-taxi' => 'Taxi (cab,cabbie,car,car service,lyft,machine,transportation,travel,uber,vehicle)' ),
			array( 'las la-tram' => 'Tram (crossing,machine,mountains,seasonal,transportation)' ),
			array( 'las la-tv' => 'Television (computer,display,monitor,television)' ),
			array( 'las la-umbrella-beach' => 'Umbrella Beach (protection,recreation,sand,shade,summer,sun)' ),
			array( 'las la-wine-glass' => 'Wine Glass (alcohol,beverage,cabernet,drink,grapes,merlot,sauvignon)' ),
			array( 'las la-wine-glass-alt' => 'Alternate Wine Glas (alcohol,beverage,cabernet,drink,grapes,merlot,sauvignon)' ),
		),
		'Users & People' => array(
			array( 'lab la-accessible-icon' => 'Accessible Icon (accessibility,handicap,person,wheelchair,wheelchair-alt)' ),
			array( 'las la-address-book' => 'Address Book (contact,directory,index,little black book,rolodex)' ),
			array( 'lar la-address-book' => 'Address Book (contact,directory,index,little black book,rolodex)' ),
			array( 'las la-address-card' => 'Address Card (about,contact,id,identification,postcard,profile)' ),
			array( 'lar la-address-card' => 'Address Card (about,contact,id,identification,postcard,profile)' ),
			array( 'las la-baby' => 'Baby (child,diaper,doll,human,infant,kid,offspring,person,sprout)' ),
			array( 'las la-bed' => 'Bed (lodging,rest,sleep,travel)' ),
			array( 'las la-biking' => 'Biking (bicycle,bike,cycle,cycling,ride,wheel)' ),
			array( 'las la-blind' => 'Blind (cane,disability,person,sight)' ),
			array( 'las la-chalkboard-teacher' => 'Chalkboard Teacher (blackboard,instructor,learning,professor,school,whiteboard,writing)' ),
			array( 'las la-child' => 'Child (boy,girl,kid,toddler,young)' ),
			array( 'las la-female' => 'Female (human,person,profile,user,woman)' ),
			array( 'las la-frown' => 'Frowning Face (disapprove,emoticon,face,rating,sad)' ),
			array( 'lar la-frown' => 'Frowning Face (disapprove,emoticon,face,rating,sad)' ),
			array( 'las la-hiking' => 'Hiking (activity,backpack,fall,fitness,outdoors,person,seasonal,walking)' ),
			array( 'las la-id-badge' => 'Identification Badge (address,contact,identification,license,profile)' ),
			array( 'lar la-id-badge' => 'Identification Badge (address,contact,identification,license,profile)' ),
			array( 'las la-id-card' => 'Identification Card (contact,demographics,document,identification,issued,profile)' ),
			array( 'lar la-id-card' => 'Identification Card (contact,demographics,document,identification,issued,profile)' ),
			array( 'las la-id-card-alt' => 'Alternate Identification Card (contact,demographics,document,identification,issued,profile)' ),
			array( 'las la-male' => 'Male (human,man,person,profile,user)' ),
			array( 'las la-meh' => 'Neutral Face (emoticon,face,neutral,rating)' ),
			array( 'lar la-meh' => 'Neutral Face (emoticon,face,neutral,rating)' ),
			array( 'las la-people-carry' => 'People Carry (box,carry,fragile,help,movers,package)' ),
			array( 'las la-person-booth' => 'Person Entering Booth (changing,changing room,election,human,person,vote,voting)' ),
			array( 'las la-poo' => 'Poo (crap,poop,shit,smile,turd)' ),
			array( 'las la-portrait' => 'Portrait (id,image,photo,picture,selfie)' ),
			array( 'las la-power-off' => 'Power Off (cancel,computer,on,reboot,restart)' ),
			array( 'las la-pray' => 'Pray (kneel,preach,religion,worship)' ),
			array( 'las la-restroom' => 'Restroom (bathroom,john,loo,potty,washroom,waste,wc)' ),
			array( 'las la-running' => 'Running (exercise,health,jog,person,run,sport,sprint)' ),
			array( 'las la-skating' => 'Skating (activity,figure skating,fitness,ice,person,winter)' ),
			array( 'las la-skiing' => 'Skiing (activity,downhill,fast,fitness,olympics,outdoors,person,seasonal,slalom)' ),
			array( 'las la-skiing-nordic' => 'Skiing Nordic (activity,cross country,fitness,outdoors,person,seasonal)' ),
			array( 'las la-smile' => 'Smiling Face (approve,emoticon,face,happy,rating,satisfied)' ),
			array( 'lar la-smile' => 'Smiling Face (approve,emoticon,face,happy,rating,satisfied)' ),
			array( 'las la-snowboarding' => 'Snowboarding (activity,fitness,olympics,outdoors,person)' ),
			array( 'las la-street-view' => 'Street View (directions,location,map,navigation)' ),
			array( 'las la-swimmer' => 'Swimmer (athlete,head,man,olympics,person,pool,water)' ),
			array( 'las la-user' => 'User (account,avatar,head,human,man,person,profile)' ),
			array( 'lar la-user' => 'User (account,avatar,head,human,man,person,profile)' ),
			array( 'las la-user-alt' => 'Alternate User (account,avatar,head,human,man,person,profile)' ),
			array( 'las la-user-alt-slash' => 'Alternate User Slash (account,avatar,head,human,man,person,profile)' ),
			array( 'las la-user-astronaut' => 'User Astronaut (avatar,clothing,cosmonaut,nasa,space,suit)' ),
			array( 'las la-user-check' => 'User Check (accept,check,person,verified)' ),
			array( 'las la-user-circle' => 'User Circle (account,avatar,head,human,man,person,profile)' ),
			array( 'lar la-user-circle' => 'User Circle (account,avatar,head,human,man,person,profile)' ),
			array( 'las la-user-clock' => 'User Clock (alert,person,remind,time)' ),
			array( 'las la-user-cog' => 'User Cog (admin,cog,person,settings)' ),
			array( 'las la-user-edit' => 'User Edit (edit,pen,pencil,person,update,write)' ),
			array( 'las la-user-friends' => 'User Friends (group,people,person,team,users)' ),
			array( 'las la-user-graduate' => 'User Graduate (cap,clothing,commencement,gown,graduation,person,student)' ),
			array( 'las la-user-injured' => 'User Injured (cast,injury,ouch,patient,person,sling)' ),
			array( 'las la-user-lock' => 'User Lock (admin,lock,person,private,unlock)' ),
			array( 'las la-user-md' => 'Doctor (job,medical,nurse,occupation,physician,profile,surgeon)' ),
			array( 'las la-user-minus' => 'User Minus (delete,negative,remove)' ),
			array( 'las la-user-ninja' => 'User Ninja (assassin,avatar,dangerous,deadly,sneaky)' ),
			array( 'las la-user-nurse' => 'Nurse (doctor,midwife,practitioner,surgeon)' ),
			array( 'las la-user-plus' => 'User Plus (add,avatar,positive,sign up,signup,team)' ),
			array( 'las la-user-secret' => 'User Secret (clothing,coat,hat,incognito,person,privacy,spy,whisper)' ),
			array( 'las la-user-shield' => 'User Shield (admin,person,private,protect,safe)' ),
			array( 'las la-user-slash' => 'User Slash (ban,delete,remove)' ),
			array( 'las la-user-tag' => 'User Tag (avatar,discount,label,person,role,special)' ),
			array( 'las la-user-tie' => 'User Tie (avatar,business,clothing,formal,professional,suit)' ),
			array( 'las la-user-times' => 'Remove User (archive,delete,remove,x)' ),
			array( 'las la-users' => 'Users (friends,group,people,persons,profiles,team)' ),
			array( 'las la-users-cog' => 'Users Cog (admin,cog,group,person,settings,team)' ),
			array( 'las la-walking' => 'Walking (exercise,health,pedometer,person,steps)' ),
			array( 'las la-wheelchair' => 'Wheelchair (accessible,handicap,person)' ),
		),
		'Vehicles' => array(
			array( 'lab la-accessible-icon' => 'Accessible Icon (accessibility,handicap,person,wheelchair,wheelchair-alt)' ),
			array( 'las la-ambulance' => 'ambulance (emergency,emt,er,help,hospital,support,vehicle)' ),
			array( 'las la-baby-carriage' => 'Baby Carriage (buggy,carrier,infant,push,stroller,transportation,walk,wheels)' ),
			array( 'las la-bicycle' => 'Bicycle (bike,gears,pedal,transportation,vehicle)' ),
			array( 'las la-bus' => 'Bus (public transportation,transportation,travel,vehicle)' ),
			array( 'las la-bus-alt' => 'Bus Alt (mta,public transportation,transportation,travel,vehicle)' ),
			array( 'las la-car' => 'Car (auto,automobile,sedan,transportation,travel,vehicle)' ),
			array( 'las la-car-alt' => 'Alternate Car (auto,automobile,sedan,transportation,travel,vehicle)' ),
			array( 'las la-car-crash' => 'Car Crash (accident,auto,automobile,insurance,sedan,transportation,vehicle,wreck)' ),
			array( 'las la-car-side' => 'Car Side (auto,automobile,sedan,transportation,travel,vehicle)' ),
			array( 'las la-fighter-jet' => 'fighter-jet (airplane,fast,fly,goose,maverick,plane,quick,top gun,transportation,travel)' ),
			array( 'las la-helicopter' => 'Helicopter (airwolf,apache,chopper,flight,fly,travel)' ),
			array( 'las la-horse' => 'Horse (equus,fauna,mammmal,mare,neigh,pony)' ),
			array( 'las la-motorcycle' => 'Motorcycle (bike,machine,transportation,vehicle)' ),
			array( 'las la-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'lar la-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'las la-plane' => 'plane (airplane,destination,fly,location,mode,travel,trip)' ),
			array( 'las la-rocket' => 'rocket (aircraft,app,jet,launch,nasa,space)' ),
			array( 'las la-ship' => 'Ship (boat,sea,water)' ),
			array( 'las la-shopping-cart' => 'shopping-cart (buy,checkout,grocery,payment,purchase)' ),
			array( 'las la-shuttle-van' => 'Shuttle Van (airport,machine,public-transportation,transportation,travel,vehicle)' ),
			array( 'las la-sleigh' => 'Sleigh (christmas,claus,fly,holiday,santa,sled,snow,xmas)' ),
			array( 'las la-snowplow' => 'Snowplow (clean up,cold,road,storm,winter)' ),
			array( 'las la-space-shuttle' => 'Space Shuttle (astronaut,machine,nasa,rocket,transportation)' ),
			array( 'las la-subway' => 'Subway (machine,railway,train,transportation,vehicle)' ),
			array( 'las la-taxi' => 'Taxi (cab,cabbie,car,car service,lyft,machine,transportation,travel,uber,vehicle)' ),
			array( 'las la-tractor' => 'Tractor (agriculture,farm,vehicle)' ),
			array( 'las la-train' => 'Train (bullet,commute,locomotive,railway,subway)' ),
			array( 'las la-tram' => 'Tram (crossing,machine,mountains,seasonal,transportation)' ),
			array( 'las la-truck' => 'truck (cargo,delivery,shipping,vehicle)' ),
			array( 'las la-truck-monster' => 'Truck Monster (offroad,vehicle,wheel)' ),
			array( 'las la-truck-pickup' => 'Truck Side (cargo,vehicle)' ),
			array( 'las la-wheelchair' => 'Wheelchair (accessible,handicap,person)' ),
		),
		'Weather' => array(
			array( 'las la-bolt' => 'Lightning Bolt (electricity,lightning,weather,zap)' ),
			array( 'las la-cloud' => 'Cloud (atmosphere,fog,overcast,save,upload,weather)' ),
			array( 'las la-cloud-meatball' => 'Cloud with (a chance of) Meatball (FLDSMDFR,food,spaghetti,storm)' ),
			array( 'las la-cloud-moon' => 'Cloud with Moon (crescent,evening,lunar,night,partly cloudy,sky)' ),
			array( 'las la-cloud-moon-rain' => 'Cloud with Moon and Rain (crescent,evening,lunar,night,partly cloudy,precipitation,rain,sky,storm)' ),
			array( 'las la-cloud-rain' => 'Cloud with Rain (precipitation,rain,sky,storm)' ),
			array( 'las la-cloud-showers-heavy' => 'Cloud with Heavy Showers (precipitation,rain,sky,storm)' ),
			array( 'las la-cloud-sun' => 'Cloud with Sun (clear,day,daytime,fall,outdoors,overcast,partly cloudy)' ),
			array( 'las la-cloud-sun-rain' => 'Cloud with Sun and Rain (day,overcast,precipitation,storm,summer,sunshower)' ),
			array( 'las la-meteor' => 'Meteor (armageddon,asteroid,comet,shooting star,space)' ),
			array( 'las la-moon' => 'Moon (contrast,crescent,dark,lunar,night)' ),
			array( 'lar la-moon' => 'Moon (contrast,crescent,dark,lunar,night)' ),
			array( 'las la-poo-storm' => 'Poo Storm (bolt,cloud,euphemism,lightning,mess,poop,shit,turd)' ),
			array( 'las la-rainbow' => 'Rainbow (gold,leprechaun,prism,rain,sky)' ),
			array( 'las la-smog' => 'Smog (dragon,fog,haze,pollution,smoke,weather)' ),
			array( 'las la-snowflake' => 'Snowflake (precipitation,rain,winter)' ),
			array( 'lar la-snowflake' => 'Snowflake (precipitation,rain,winter)' ),
			array( 'las la-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'lar la-sun' => 'Sun (brighten,contrast,day,lighter,sol,solar,star,weather)' ),
			array( 'las la-temperature-high' => 'High Temperature (cook,mercury,summer,thermometer,warm)' ),
			array( 'las la-temperature-low' => 'Low Temperature (cold,cool,mercury,thermometer,winter)' ),
			array( 'las la-umbrella' => 'Umbrella (protection,rain,storm,wet)' ),
			array( 'las la-water' => 'Water (lake,liquid,ocean,sea,swim,wet)' ),
			array( 'las la-wind' => 'Wind (air,blow,breeze,fall,seasonal,weather)' ),
		),
		'Winter' => array(
			array( 'las la-glass-whiskey' => 'Glass Whiskey (alcohol,bar,beverage,bourbon,drink,liquor,neat,rye,scotch,whisky)' ),
			array( 'las la-icicles' => 'Icicles (cold,frozen,hanging,ice,seasonal,sharp)' ),
			array( 'las la-igloo' => 'Igloo (dome,dwelling,eskimo,home,house,ice,snow)' ),
			array( 'las la-mitten' => 'Mitten (clothing,cold,glove,hands,knitted,seasonal,warmth)' ),
			array( 'las la-skating' => 'Skating (activity,figure skating,fitness,ice,person,winter)' ),
			array( 'las la-skiing' => 'Skiing (activity,downhill,fast,fitness,olympics,outdoors,person,seasonal,slalom)' ),
			array( 'las la-skiing-nordic' => 'Skiing Nordic (activity,cross country,fitness,outdoors,person,seasonal)' ),
			array( 'las la-snowboarding' => 'Snowboarding (activity,fitness,olympics,outdoors,person)' ),
			array( 'las la-snowplow' => 'Snowplow (clean up,cold,road,storm,winter)' ),
			array( 'las la-tram' => 'Tram (crossing,machine,mountains,seasonal,transportation)' ),
		),
		'Writing' => array(
			array( 'las la-archive' => 'Archive (box,package,save,storage)' ),
			array( 'las la-blog' => 'Blog (journal,log,online,personal,post,web 2.0,wordpress,writing)' ),
			array( 'las la-book' => 'book (diary,documentation,journal,library,read)' ),
			array( 'las la-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'lar la-bookmark' => 'bookmark (favorite,marker,read,remember,save)' ),
			array( 'las la-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'lar la-edit' => 'Edit (edit,pen,pencil,update,write)' ),
			array( 'las la-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'lar la-envelope' => 'Envelope (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'las la-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'lar la-envelope-open' => 'Envelope Open (e-mail,email,letter,mail,message,notification,support)' ),
			array( 'las la-eraser' => 'eraser (art,delete,remove,rubber)' ),
			array( 'las la-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'lar la-file' => 'File (document,new,page,pdf,resume)' ),
			array( 'las la-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'lar la-file-alt' => 'Alternate File (document,file-text,invoice,new,page,pdf)' ),
			array( 'las la-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'lar la-folder' => 'Folder (archive,directory,document,file)' ),
			array( 'las la-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'lar la-folder-open' => 'Folder Open (archive,directory,document,empty,file,new)' ),
			array( 'las la-keyboard' => 'Keyboard (accessory,edit,input,text,type,write)' ),
			array( 'lar la-keyboard' => 'Keyboard (accessory,edit,input,text,type,write)' ),
			array( 'las la-newspaper' => 'Newspaper (article,editorial,headline,journal,journalism,news,press)' ),
			array( 'lar la-newspaper' => 'Newspaper (article,editorial,headline,journal,journalism,news,press)' ),
			array( 'las la-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'lar la-paper-plane' => 'Paper Plane (air,float,fold,mail,paper,send)' ),
			array( 'las la-paperclip' => 'Paperclip (attach,attachment,connect,link)' ),
			array( 'las la-paragraph' => 'paragraph (edit,format,text,writing)' ),
			array( 'las la-pen' => 'Pen (design,edit,update,write)' ),
			array( 'las la-pen-alt' => 'Alternate Pen (design,edit,update,write)' ),
			array( 'las la-pen-square' => 'Pen Square (edit,pencil-square,update,write)' ),
			array( 'las la-pencil-alt' => 'Alternate Pencil (design,edit,pencil,update,write)' ),
			array( 'las la-quote-left' => 'quote-left (mention,note,phrase,text,type)' ),
			array( 'las la-quote-right' => 'quote-right (mention,note,phrase,text,type)' ),
			array( 'las la-sticky-note' => 'Sticky Note (message,note,paper,reminder,sticker)' ),
			array( 'lar la-sticky-note' => 'Sticky Note (message,note,paper,reminder,sticker)' ),
			array( 'las la-thumbtack' => 'Thumbtack (coordinates,location,marker,pin,thumb-tack)' ),
		),
		'Other' => array(
			array( 'las la-backspace' => 'Backspace (command,delete,erase,keyboard,undo)' ),
			array( 'las la-blender-phone' => 'Blender Phone (appliance,cocktail,communication,fantasy,milkshake,mixer,puree,silly,smoothie)' ),
			array( 'las la-crown' => 'Crown (award,favorite,king,queen,royal,tiara)' ),
			array( 'las la-dumpster-fire' => 'Dumpster Fire (alley,bin,commercial,danger,dangerous,euphemism,flame,heat,hot,trash,waste)' ),
			array( 'las la-file-csv' => 'File CSV (document,excel,numbers,spreadsheets,table)' ),
			array( 'las la-network-wired' => 'Wired Network (computer,connect,ethernet,internet,intranet)' ),
			array( 'las la-signature' => 'Signature (John Hancock,cursive,name,writing)' ),
			array( 'las la-skull' => 'Skull (bones,skeleton,x-ray,yorick)' ),
			array( 'las la-vr-cardboard' => 'Cardboard VR (3d,augment,google,reality,virtual)' ),
			array( 'las la-weight-hanging' => 'Hanging Weight (anvil,heavy,measurement)' ),
		),
	);

	return array_merge( $icons, $lineawesome_icons );
}