<?php
/**
 * Build the Plugin settings
 *
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}


if( ! class_exists( 'WMVP_Settings' ) ) {

	class WMVP_Settings {

		public $item_id;
		public $current_value;
		public $option_type;
		public $settings;
		public $option_name;
		public $disabled;


		/**
		 * __construct
		 */
		function __construct( $settings, $option_name, $data ) {

			$this->prepare_data( $settings, $option_name, $data );

			// Class
			$class = ! empty( $this->settings['class'] ) ?  ' '. $this->settings['class'] .'-options' : '';

			if( ! empty( $this->settings['is_pro'] ) && ! wmvp_is_pro() ){
				$class .= ' wmpv-pro-feature';
			}

			echo '<tr id="'. esc_attr( $this->item_id ) .'-item" valign="top" class="'. esc_attr( $class ) .'">';

			if ( ! empty( $this->settings['name'] ) ) {
				echo '<th scope="row">';
				echo '<label for="setting-' . esc_attr( $this->settings['name'] ) . '">'. $this->settings['name'] .'</label>';
				if( ! empty( $this->settings['is_pro'] ) && ! wmvp_is_pro() ){
					wmvp_pro_label();
				}
				echo '</th><td>';
			}
			else {
				echo '<td colspan="2">';
			}


			$method_name = 'input_' . $this->option_type;
			if ( method_exists( $this, $method_name ) ) {
				$this->$method_name();
			}
			else {
				/**
				 * Allows for custom fields in admin setting panes.
				 */
				do_action( 'WMVP/Settings/admin_field_' . $this->option_type, $settings, $option_name, $data );
			}


			if( ! empty( $this->settings['after'] ) ){

				if( $this->settings['after'] == 'self_value' ){
					
					echo '<div class="wmvp-self-value">';

						if( ! empty( $this->settings['options'][ $this->current_value ] ) ){
							echo $this->settings['options'][ $this->current_value ];
						}
						else{
							echo $this->current_value;
						}
					echo '</div>';
				}
				else{
					echo $this->settings['after'];
				}
			}

			if( ! empty( $this->settings['hint'] ) ){
				echo '<p class="wmvp-description">'. $this->settings['hint'] .'</p>';
			}

			echo '</td></tr>';
		}

		/**
		 * Text
		 */
		private function input_text(){
			?>
				<input id="<?php echo esc_attr( $this->item_id ) ?>" name="<?php echo esc_attr( $this->option_name ) ?>" type="text" value="<?php echo esc_attr( $this->current_value ) ?>" placeholder="<?php echo esc_attr( $this->settings['placeholder'] ) ?>" <?php echo $this->disabled ?>>
			<?php
		}


		/**
		 * Checkbox
		 */
		private function input_checkbox(){
			?>
				<input id="<?php echo esc_attr( $this->item_id ) ?>" name="<?php echo esc_attr( $this->option_name ) ?>" type="checkbox" value="true" <?php checked( $this->current_value, 'true' ) ?> <?php echo $this->disabled ?>>
			<?php
		}


		/**
		 * Radio
		 */
		private function input_radio(){

			$i = 0;
			foreach ( $this->settings['options'] as $option_key => $option ){
				$i++;

				$checked = false;
				if ( ( ! empty( $this->current_value ) && $this->current_value == $option_key ) || ( empty( $this->current_value ) && $i==1 ) ){
					$checked = true;
				}

				?>
					<label class="wmvp-option-label">
						<input name="<?php echo esc_attr( $this->option_name ) ?>" <?php checked( $checked ) ?> type="radio" value="<?php echo esc_attr( $option_key ) ?>" <?php echo $this->disabled ?>> <?php echo $option ?>
					</label>
				<?php
			}
		}


		/**
		 * Multiple Select
		 */
		private function input_multiple_select(){
			?>
			<select id="<?php echo esc_attr( $this->item_id ) ?>" name="<?php echo esc_attr( $this->option_name ) ?>[]" multiple="multiple" <?php echo $this->disabled ?>>

				<?php

					$data = maybe_unserialize( $this->current_value );

					$i = 0;
					foreach ( $this->settings['options'] as $option_key => $option ){
						$selected = false;
						if ( ( ! empty( $data ) && !is_array( $data ) && $data == $option_key ) || ( ! empty( $data) && is_array($data) && in_array( $option_key , $data ) ) || ( empty( $data ) && $i==1 ) ){
							$selected = true;
						}
						?>
							<option value="<?php echo esc_attr( $option_key ) ?>" <?php selected( $selected ) ?>><?php echo esc_attr( $option ) ?></option>
						<?php
					}
				?>
			</select>
			<?php
		}


		/**
		 * Textarea
		 */
		private function input_textarea(){
			?>
				<textarea id="<?php echo esc_attr( $this->item_id ) ?>" name="<?php echo esc_attr( $this->option_name ) ?>" rows="3" <?php echo $this->disabled ?>><?php echo esc_textarea( $this->current_value ) ?></textarea>
			<?php
		}


		/**
		 * Color
		 */
		private function input_color(){
			wp_enqueue_script( 'wp-color-picker' );
			?>
			<div class="wmvp-color-picker-wrapper">
				<input id="<?php echo esc_attr( $this->item_id ) ?>" class="wmvp-color-picker" name="<?php echo esc_attr( $this->option_name ) ?>" type="text" value="<?php echo $this->current_value ?>" data-palette="#9b59b6, #3498db, #2ecc71, #f1c40f, #34495e, #e74c3c" style="width:80px;" <?php echo $this->disabled ?>>
			</div>
			<?php
		}


		/**
		 * Editor
		 */
		private function input_editor(){

			wp_editor(
				$this->current_value,
				$this->item_id,
				array(
					'textarea_name' => $this->option_name,
					'editor_height' => '400px',
					'media_buttons' => false
				)
			);
		}


		/**
		 * Number
		 */
		private function input_number(){
			?>
			<input min="-1000" max="1000000" id="<?php echo esc_attr( $this->item_id ) ?>" name="<?php echo esc_attr( $this->option_name ) ?>" type="number" value="<?php echo esc_attr( $this->current_value ) ?>" placeholder="<?php echo esc_attr( $this->settings['placeholder']) ?>" <?php echo $this->disabled ?>>
			<?php
		}


		/**
		 * Select
		 */
		private function input_select(){
			?>
				<select id="<?php echo esc_attr( $this->item_id ) ?>" name="<?php echo esc_attr( $this->option_name ) ?>">
					<?php
						$i = 0;
						if( ! empty( $this->settings['options'] ) && is_array( $this->settings['options'] ) ){
							foreach ( $this->settings['options'] as $option_key => $option ){
								$i++;

								$selected = false;
								if ( ( ! empty( $this->current_value ) && $this->current_value == $option_key ) || ( empty( $this->current_value ) && $i==1 ) ){
									$selected = true;
								}
								?>
								<option value="<?php echo esc_attr( $option_key ) ?>" <?php selected( $selected ) ?> <?php echo $this->disabled ?>><?php echo esc_attr( $option ) ?></option>
								<?php
							}
						}
					?>
				</select>
			<?php
		}


		/**
		 * Typography
		 */
		private function input_typography(){

			$current_value = wp_parse_args( $this->current_value, array(
				'font'        => '',
				'size'        => '',
				'line_height' => '',
				'weight'      => '',
				'transform' 	=> '',
			));

			?>

			<input name="<?php echo esc_attr( $this->option_name ) ?>[font]" id="<?php echo esc_attr( $this->settings['id'] ) ?>[font]" class="wpvpl-select-font" type="text" value="<?php echo esc_attr( $current_value['font'] ) ?>">

			<div class="wmvp-typography-options">
				<select name="<?php echo esc_attr( $this->option_name ) ?>[size]" id="<?php echo esc_attr( $this->settings['id'] ) ?>[size]">
					<option <?php selected( $current_value['size'], '' ); ?> <?php disabled(1,1); ?>><?php esc_html_e( 'Font Size in Pixels', 'wm-video-playlists' ); ?></option>
					<option value=""><?php esc_html_e( 'Default', 'wm-video-playlists' ); ?></option>
					<?php for( $i=8 ; $i<61 ; $i+=1){ ?>
						<option value="<?php echo esc_attr( $i ) ?>" <?php selected( $current_value['size'], $i ); ?>><?php echo esc_attr( $i ) ?></option>
					<?php } ?>
				</select>

				<select name="<?php echo esc_attr( $this->option_name ) ?>[line_height]" id="<?php echo esc_attr( $this->settings['id'] ) ?>[line_height]">
					<option <?php selected( $current_value['line_height'], '' ); ?> <?php disabled(1,1); ?>><?php esc_html_e( 'Line Height', 'wm-video-playlists' ); ?></option>
					<option value=""><?php esc_html_e( 'Default', 'wm-video-playlists' ); ?></option>

					<?php for( $i=10 ; $i<=60 ; $i+=2.5 ){
						$line_height = $i/10;
						?>
						<option value="<?php echo esc_attr( $line_height ) ?>" <?php selected( $current_value['line_height'], $line_height ); ?>><?php echo esc_attr( $line_height ) ?></option>
					<?php } ?>
				</select>

				<select name="<?php echo esc_attr( $this->option_name ) ?>[weight]" id="<?php echo esc_attr( $this->settings['id'] ) ?>[weight]">
					<option <?php selected( $current_value['weight'], '' ); ?> <?php disabled(1,1); ?>><?php esc_html_e( 'Font Weight', 'wm-video-playlists' ); ?></option>
					<option value=""><?php esc_html_e( 'Default', 'wm-video-playlists' ); ?></option>
					<option value="100" <?php selected( $current_value['weight'], 100 ); ?>><?php esc_html_e( 'Thin 100',        'wm-video-playlists' ); ?></option>
					<option value="200" <?php selected( $current_value['weight'], 200 ); ?>><?php esc_html_e( 'Extra 200 Light', 'wm-video-playlists' ); ?></option>
					<option value="300" <?php selected( $current_value['weight'], 300 ); ?>><?php esc_html_e( 'Light 300',       'wm-video-playlists' ); ?></option>
					<option value="400" <?php selected( $current_value['weight'], 400 ); ?>><?php esc_html_e( 'Regular 400',     'wm-video-playlists' ); ?></option>
					<option value="500" <?php selected( $current_value['weight'], 500 ); ?>><?php esc_html_e( 'Medium 500',      'wm-video-playlists' ); ?></option>
					<option value="600" <?php selected( $current_value['weight'], 600 ); ?>><?php esc_html_e( 'Semi 600 Bold',   'wm-video-playlists' ); ?></option>
					<option value="700" <?php selected( $current_value['weight'], 700 ); ?>><?php esc_html_e( 'Bold 700',        'wm-video-playlists' ); ?></option>
					<option value="800" <?php selected( $current_value['weight'], 800 ); ?>><?php esc_html_e( 'Extra 800 Bold',  'wm-video-playlists' ); ?></option>
					<option value="900" <?php selected( $current_value['weight'], 900 ); ?>><?php esc_html_e( 'Black 900',       'wm-video-playlists' ); ?></option>
				</select>

				<select name="<?php echo esc_attr( $this->option_name ) ?>[transform]" id="<?php echo esc_attr( $this->settings['id'] ) ?>[transform]">
					<option <?php selected( $current_value['transform'], '' ); ?> <?php disabled(1,1); ?>><?php esc_html_e( 'Capitalization', 'wm-video-playlists' ); ?></option>
					<option value=""><?php esc_html_e( 'Default', 'wm-video-playlists' ); ?></option>
					<option value="uppercase"  <?php selected( $current_value['transform'], 'uppercase' ); ?>><?php esc_html_e( 'UPPERCASE',  'wm-video-playlists' ); ?></option>
					<option value="capitalize" <?php selected( $current_value['transform'], 'capitalize' );?>><?php esc_html_e( 'Capitalize', 'wm-video-playlists' ); ?></option>
					<option value="lowercase"  <?php selected( $current_value['transform'], 'lowercase' ); ?>><?php esc_html_e( 'lowercase',  'wm-video-playlists' ); ?></option>
				</select>

			</div>
			<?php
		}


		/**
		 * Prepare Data
		 */
		private function prepare_data( $settings, $option_name, $data ){
			
			// Default Settings
			$settings = wp_parse_args( $settings, array(
				'id' => '',
				'type' => false,
				'placeholder' => '',
			));

			$this->settings    = $settings;
			$this->option_name = $option_name;
			$this->option_type = $settings['type'];

			if( isset( $this->settings['disabled'] ) || ( ! empty( $this->settings['is_pro'] ) && ! wmvp_is_pro() ) ){
				$this->disabled = disabled( 1,1, false );
			}

			// ID
			$this->item_id .= ! empty( $this->settings['prefix'] ) ? $this->settings['prefix'] . '-' : '';
			$this->item_id .= ! empty( $this->settings['id'] )     ? $this->settings['id'] : '';

			if( ! empty( $this->item_id ) && $this->option_type == 'arrayText' ){
				$this->item_id = $this->item_id . '-'. $this->settings['key'];
			}

			// Get the option stored data
			if( ! empty( $data ) ){
				$this->current_value = $data;
			}
			elseif( ! empty( $settings['default'] ) ){
				$this->current_value = $settings['default'];
			}
		}

	}
}
