<?php
/**
 * Video Playlist Class
 *
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}


if( ! class_exists( 'WMVP_API' ) ) {

	class WMVP_API {


		/**
		 * Runs on class initialization. Adds filters and actions.
		 */
		function __construct() {

			// Save Video Plylist
			add_filter( 'WMVP/Videos/Post/List', array( $this, 'save_manual_list' ) );

			// Verify the YouTube API Key
			add_action( 'wp_ajax_wmvp_activate_youtube_key', array( $this, 'activate_youtube_key' ) );
		}


		/**
		 * Save Video Plylist
		 */
		function activate_youtube_key( $videos = false ){
			if( empty( $_REQUEST['key'] ) || ! current_user_can( 'manage_options' ) ){
				return false;
			}

			// Secutity Check
			if ( ! check_ajax_referer( WMVP_PLUGIN_DIR, 'security', false ) ) {
				$message = new WP_Error( 403, esc_html__( 'Invalid security token sent.', 'wm-video-playlists' ) );
				return wp_send_json_error( $message );
				exit;
			}

			// Build the Api request
			$params   = array(
				'q'    => 'WatermelonWP Plugins',
				'part' => 'snippet',
				'type' => 'video',
				'key'  => wp_unslash( $_REQUEST['key'] ),
				'maxResults' => 5,
			);

			$request = $this->youtube_api_request( $params, 'search' );

			if( is_wp_error( $request ) ){
				return wp_send_json_error( $request );
				exit;
			}

			update_option( 'wmvp_youtube_key', wp_unslash( $_REQUEST['key'] ), false );

			wmvp_notice_inline( esc_html__( 'Successfully verified.', 'wm-video-playlists' ), 'success' );
			exit;
		}

		/**
		 * Save Video Plylist
		 */
		function save_manual_list( $videos = false ){
			
			if( empty( $videos ) || ! is_array( $videos ) ){
				return $videos;
			}

			$youtube_videos = array();
			
			foreach( $videos as $key => $single_video ){

				if( empty( $single_video['url'] ) ){
					unset( $videos[ $key ] ); // URL is required
					continue;
				}

				// YouTube
				if( $video_id = $this->is_youtube( $single_video['url'] ) ) {
					$youtube_videos[ $video_id ] = $key;
				}

				// Vimeo
				elseif( $video_id = $this->is_vimeo( $single_video['url'] ) ) {

					$video_info = $this->get_vimeo_video( $video_id );

					if( $video_info ){

						$thumbnails = array(
							'default' => $video_info['thumbnail_small'],
							'medium'  => $video_info['thumbnail_medium'],
							'high'    => $video_info['thumbnail_large'],
						);

						$single_video['id']          = $video_id;
						$single_video['vendor']      = 'vimeo';
						$single_video['thumbnails']  = $thumbnails;
						$single_video['published']   = $video_info['upload_date'];
						$single_video['title']       = $this->remove_emoji( $video_info['title'] );
						$single_video['description'] = $this->remove_emoji( $video_info['description'] );
						$single_video['thumbnail']   = $video_info['thumbnail_medium'];
						$single_video['embed_path']  = "https://player.vimeo.com/video/$video_id?api=1&amp;title=0&amp;byline=0";
		
						// Prepare the Video duration
						if( ! empty( $video_info['duration'] ) ){
							$duration_sec = $video_info['duration'];
							$time_format  = ( $duration_sec >= 3600 ) ? 'H:i:s' : 'i:s';

							$single_video['duration'] = gmdate( $time_format, $duration_sec );
						}

						$videos[ $key ] = $single_video;
					}
				} // Vimeo
			}

			// Make one API Connection
			if( ! empty( $youtube_videos ) ){
				$video_ids = implode( ',', array_keys( $youtube_videos ) );
				$videos_info = $this->get_youtube_video( $video_ids );

				if( ! empty( $videos_info['items'] ) && is_array( $videos_info['items'] ) && ! is_wp_error( $videos_info ) ){
					foreach( $videos_info['items'] as $single_video ){
						if( ! empty( $single_video['id'] ) && isset( $youtube_videos[ $single_video['id'] ] ) ){
							$key = $youtube_videos[ $single_video['id'] ];
							$videos[ $key ] = array_merge( $this->prepare_youtube_data( $single_video ), $videos[ $key ] ) ;
						}
					}
				}
			}

			return $videos;
		}


		/**
		 * Check if the Video is YouTube
		 */
		function is_youtube( $video ) {
			preg_match( "#(?<=v=)[a-zA-Z0-9-]+(?=&)|(?<=v\/)[^&\n]+(?=\?)|(?<=v=)[^&\n]+|(?<=youtu.be/)[^&\n]+#", $video, $matches );

			if( ! empty( $matches[0] ) ){
				return preg_replace( '/\s+/', '', $matches[0] );
			}

			return false;
		}


		/**
		 * Check if the Video is Vimeo
		 */
		function is_vimeo( $video ) {
			preg_match( "/(https?:\/\/)?(www\.)?(player\.)?vimeo\.com\/([a-z]*\/)*([0-9]{6,11})[?]?.*/", $video, $matches );

			if( ! empty( $matches[5] ) ){
				return preg_replace( '/\s+/', '', $matches[5] );
			}

			return false;
		}


		/**
		 * Get YouTube Video data
		 */
		function get_youtube_video( $id ){

			// Build the Api request
			$params = array(
				'part' => 'snippet,contentDetails',
				'id'   => $id,
			);

			$request = $this->youtube_api_request( $params, 'videos' );

			if( is_wp_error( $request ) ){
				return false;
			}

			return $request;
		}


		/**
		 * Get Vimeo Video data
		 */
		function get_vimeo_video( $video ){

			// Build the Api request
			$api_url = "https://vimeo.com/api/v2/video/$video.json";
			$request = wp_remote_get( $api_url );

			// Check if there is no any errors
			if( is_wp_error( $request ) ){
				return null;
			}

			// Prepare the data
			$result = json_decode( wp_remote_retrieve_body( $request ), true );

			// Check if the video title exists
			if( empty( $result[0]['title'] ) ){
				return null;
			}

			return $result[0];
		}


		/**
		 * Remove emojis
		 *
		 * Causes issues https://core.trac.wordpress.org/ticket/21212
		 */
		function remove_emoji( $string ) {

			// Match Emoticons
			$regex_emoticons = '/[\x{1F600}-\x{1F64F}]/u';
			$clear_string    = preg_replace($regex_emoticons, '', $string);

			// Match Miscellaneous Symbols and Pictographs
			$regex_symbols = '/[\x{1F300}-\x{1F5FF}]/u';
			$clear_string  = preg_replace($regex_symbols, '', $clear_string);

			// Match Transport And Map Symbols
			$regex_transport = '/[\x{1F680}-\x{1F6FF}]/u';
			$clear_string = preg_replace($regex_transport, '', $clear_string);

			// Match Miscellaneous Symbols
			$regex_misc   = '/[\x{2600}-\x{26FF}]/u';
			$clear_string = preg_replace($regex_misc, '', $clear_string);

			// Match Dingbats
			$regex_dingbats = '/[\x{2700}-\x{27BF}]/u';
			$clear_string   = preg_replace($regex_dingbats, '', $clear_string);

			return $clear_string;
		}


		/**
		 * Get Youtube Videos via API
		 */
		function youtube_api_request( $params = array(), $endpoint = false ){

			// Add the Api Key to the Params array
			$api_key = ! empty( $params['key'] ) ? $params['key'] : get_option( 'wmvp_youtube_key' );

			if( empty( $api_key ) ) {
				return false;
			}

			$params['key'] = preg_replace( '/\s+/', '', $api_key );

			// Send the Get request
			$response = wp_remote_get( "https://www.googleapis.com/youtube/v3/$endpoint?" . http_build_query( $params ) );

			// Check Response for errors
			$response_code = wp_remote_retrieve_response_code( $response );

			// Return Errors
			if ( is_wp_error( $response ) ){
				return $response;
			}

			elseif ( 200 !== $response_code ){

				if( ! empty( $response['response']['message'] ) ){

					// Get the detailed reason from the YouTube response 
					$response_body = json_decode( wp_remote_retrieve_body( $response ), true );

					// If there is no error in the body
					$response_error = ! empty( $response_body['error']['message'] ) ? $response_body['error']['message'] : $response['response']['message'];

					return new WP_Error( $response_code, $response_error );
				}
				else {
					return new WP_Error( $response_code, 'Connection Error' );
				}
			}

			// All is Good
			return json_decode( wp_remote_retrieve_body( $response ), true );
		}


		/**
		 * 
		 */
		function prepare_youtube_data( $video_info = array()){

			if( empty( $video_info['snippet'] ) ){
				return false;
			}

			$snippet = $video_info['snippet'];

			$thumbnails = array(
				'default'  => ! empty( $snippet['thumbnails']['default']['url'] )  ? $snippet['thumbnails']['default']['url']  : '',  // 120 x 90
				'medium'   => ! empty( $snippet['thumbnails']['medium']['url'] )   ? $snippet['thumbnails']['medium']['url']   : '',  // 320 x 180
				'high'     => ! empty( $snippet['thumbnails']['high']['url'] )     ? $snippet['thumbnails']['high']['url']     : '',  // 480 x 360
				'standard' => ! empty( $snippet['thumbnails']['standard']['url'] ) ? $snippet['thumbnails']['standard']['url'] : '',  // 640 x 480
				'maxres'   => ! empty( $snippet['thumbnails']['maxres']['url'] )   ? $snippet['thumbnails']['maxres']['url']   : '',  // 1280 x 720
			);

			$video['vendor']        = 'youtube';
			$video['id']            = $video_info['id'];
			$video['url']           = 'https://www.youtube.com/watch?v=' . $video_info['id'];
			$video['thumbnails']    = $thumbnails;
			$video['channel_id']    = $snippet['channelId'];
			$video['channel_title'] = $snippet['channelTitle'];
			$video['title']         = $this->remove_emoji( $snippet['title'] );
			$video['description']   = $this->remove_emoji( $snippet['description'] );
			$video['embed_path']    = "https://www.youtube.com/embed/{$video_info['id']}?enablejsapi=1&amp;rel=0&amp;showinfo=0";
			$video['custom_title']  = '';
			$video['disabled']      = '';

			if( ! empty( $snippet['publishedAt'] ) ){
				$date = date_create( $snippet['publishedAt'] );
				$date = date_format( $date, 'U' );
				$video['published'] = $date;
			}

			// Prepare the Video duration
			if( ! empty( $video_info['contentDetails']['duration'] ) ) {
				$interval     = new DateInterval( $video_info['contentDetails']['duration'] );
				$duration_sec = $interval->h * 3600 + $interval->i * 60 + $interval->s;
				$time_format  = ( $duration_sec >= 3600 ) ? 'H:i:s' : 'i:s';

				$video['duration'] = gmdate( $time_format, $duration_sec );
			}

			return $video;
		}


	}

	// Instantiate the class
	new WMVP_API();

}