<?php
/**
 * Kalium WordPress Theme
 *
 * File system class of Kalium
 *
 * @link https://kaliumtheme.com
 */
namespace Kalium\Core;

use PclZip;
use WP_Error;
use WP_Filesystem_Base;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Filesystem {

	/**
	 * Initialization status.
	 *
	 * @var bool
	 */
	private $ok = false;

	/**
	 * Submitted credentials.
	 *
	 * @var array|bool
	 */
	private $credentials;

	/**
	 * Credentials form (HTML).
	 *
	 * @var string
	 */
	private $credentials_form;

	/**
	 * Initialize filesystem.
	 *
	 * @param string $url
	 *
	 * @return bool
	 */
	public function initialize( $url = null ) {
		static $initialized;

		// Initialize only once
		if ( is_null( $initialized ) ) {

			// Load filesystem functions
			if ( ! function_exists( 'WP_Filesystem' ) ) {
				require_once ABSPATH . 'wp-admin/includes/file.php';
			}

			// Use current URL
			if ( ! $url ) {
				$url = add_query_arg( 'kalium_fs_creds_nonce', wp_create_nonce( 'kalium_fs_creds_nonce' ) );
			}

			// Request credentials
			$this->request_credentials( $url, true );

			// Try creating the file system instance
			$this->ok = WP_Filesystem( $this->get_credentials() );

			// Mark as initialized
			$initialized = true;
		}

		return (bool) $this->ok;
	}

	/**
	 * Reference to WP Filesystem instance.
	 *
	 * @return WP_Filesystem_Base|null
	 */
	public function instance() {
		global $wp_filesystem;

		return $wp_filesystem;
	}

	/**
	 * Get errors from WP Filesystem.
	 *
	 * @return WP_Error|null
	 */
	public function get_errors() {
		if ( ! $this->instance() ) {
			return new WP_Error( 'kalium_fs_not_initialized', 'The filesystem instance has not been initialized.' );
		}

		return $this->instance()->errors;
	}

	/**
	 * Check if filesystem reported errors.
	 *
	 * @return bool
	 */
	public function has_errors() {
		return is_wp_error( $this->get_errors() ) && $this->get_errors()->has_errors();
	}

	/**
	 * Get error message from WP Filesystem.
	 *
	 * @return string
	 */
	public function get_error_message() {
		$error_messages = [];

		if ( $this->has_errors() ) {
			$error_messages = $this->get_errors()->get_error_messages();
		}

		return implode( PHP_EOL, $error_messages );
	}

	/**
	 * Request filesystem credentials.
	 *
	 * @param string $url
	 * @param bool   $silent
	 *
	 * @return array|bool
	 */
	public function request_credentials( $url = '', $silent = false ) {
		ob_start();

		// Request credentials
		$this->credentials = request_filesystem_credentials( esc_url_raw( $url ) );

		// Store credentials form
		$this->credentials_form = ob_get_clean();

		if ( ! $silent ) {
			echo $this->get_credentials_form();
		}

		return $this->get_credentials();
	}

	/**
	 * Get filesystem credentials.
	 *
	 * @return array|bool
	 */
	public function get_credentials() {
		return $this->credentials;
	}

	/**
	 * Get credentials form.
	 *
	 * @return string
	 */
	public function get_credentials_form() {
		return $this->credentials_form;
	}

	/**
	 * Absolute path.
	 *
	 * @return string
	 */
	public function abspath() {
		return $this->instance()->abspath();
	}

	/**
	 * The wp-content directory path.
	 *
	 * @return string
	 */
	public function wp_content_dir() {
		return $this->instance()->wp_content_dir();
	}

	/**
	 * Replace abs path with remote file abs path.
	 *
	 * @param string $file
	 *
	 * @return string
	 */
	public function real_abspath( $file ) {
		return str_replace( ABSPATH, $this->abspath(), $file );
	}

	/**
	 * Writes a string to a file.
	 *
	 * @param string    $file
	 * @param string    $contents
	 * @param int|false $mode
	 *
	 * @return bool
	 */
	public function put_contents( $file, $contents, $mode = false ) {
		return $this->instance()->put_contents(
			$this->real_abspath( $file ),
			$contents,
			$mode
		);
	}

	/**
	 * Copies a file.
	 *
	 * @param string    $source
	 * @param string    $destination
	 * @param bool      $overwrite
	 * @param int|false $mode
	 *
	 * @return bool
	 */
	public function copy( $source, $destination, $overwrite = false, $mode = false ) {
		return $this->instance()->copy(
			$this->real_abspath( $source ),
			$this->real_abspath( $destination ),
			$overwrite,
			$mode
		);
	}

	/**
	 * Copies a directory from one location to another via the WordPress Filesystem Abstraction.
	 *
	 * @param string   $from
	 * @param string   $to
	 * @param string[] $skip_list
	 *
	 * @return true|WP_Error
	 */
	public function copy_dir( $from, $to, $skip_list = [] ) {
		return copy_dir(
			$this->real_abspath( $from ),
			$this->real_abspath( $to ),
			$skip_list
		);
	}

	/**
	 * Moves file.
	 *
	 * @param string $source
	 * @param string $destination
	 * @param bool   $overwrite
	 *
	 * @return bool
	 */
	public function move( $source, $destination, $overwrite = false ) {
		return $this->instance()->move(
			$this->real_abspath( $source ),
			$this->real_abspath( $destination ),
			$overwrite
		);
	}

	/**
	 * Deletes a file or directory.
	 *
	 * @param string       $file
	 * @param bool         $recursive
	 * @param string|false $type
	 *
	 * @return bool
	 */
	public function delete( $file, $recursive = 0, $type = 0 ) {
		return $this->instance()->delete( $this->real_abspath( $file ), $recursive, $type );
	}

	/**
	 * Checks file existence.
	 *
	 * @param string $file
	 *
	 * @return bool
	 */
	public function exists( $file ) {
		return $this->instance()->exists( $this->real_abspath( $file ) );
	}

	/**
	 * Gets the file size (in bytes).
	 *
	 * @param string $file
	 *
	 * @return int|false
	 */
	public function size( $file ) {
		return $this->instance()->size( $this->real_abspath( $file ) );
	}

	/**
	 * Sets the access and modification times of a file.
	 *
	 * @param string $file
	 * @param int    $time
	 * @param int    $atime
	 *
	 * @return bool
	 */
	public function touch( $file, $time = 0, $atime = 0 ) {
		return $this->instance()->touch(
			$this->real_abspath( $file ),
			$time,
			$atime
		);
	}

	/**
	 * Unzips a specified ZIP file to a location on the filesystem.
	 *
	 * @param string $file
	 * @param string $to
	 *
	 * @return bool|WP_Error
	 */
	public function unzip_file( $file, $to ) {
		if ( $this->has_errors() ) {
			return $this->get_errors();
		}

		return unzip_file(
			$file,
			$this->real_abspath( $to )
		);
	}

	/**
	 * Compress a file or directory with WordPress PclZip library.
	 *
	 * @param string $source
	 * @param string $destination
	 * @param bool   $remove_path
	 *
	 * @return true|WP_Error
	 */
	public function zip_file( $source, $destination = '', $remove_path = false ) {

		// Set the mbstring internal encoding to a binary safe encoding
		mbstring_binary_safe_encoding();

		// Optional destination path name generate
		if ( ! $destination ) {
			$destination = trailingslashit( dirname( $source ) ) . basename( $source ) . '.zip';
		} elseif ( '.' === dirname( $destination ) ) {
			$destination = trailingslashit( dirname( $source ) ) . $destination;
		}

		// Add zip extension if not present
		if ( ! preg_match( '/\.zip$/i', $destination ) ) {
			$destination .= '.zip';
		}

		// Absolute destination path
		$destination = $this->real_abspath( $destination );

		// Load class file if it's not loaded yet
		if ( ! class_exists( 'PclZip' ) ) {
			require_once ABSPATH . 'wp-admin/includes/class-pclzip.php';
		}

		// Creative archive
		$archive = new PclZip( wp_normalize_path( $destination ) );
		$result  = $archive->add( wp_normalize_path( $source ), PCLZIP_OPT_REMOVE_PATH, $remove_path ? $source : dirname( $source ) );

		// Reset the mbstring internal encoding
		reset_mbstring_encoding();

		// Creating archive failed
		if ( 0 === $result ) {
			return new WP_Error( 'kalium_fs_zip_failed', $archive->error_string );
		}

		return true;
	}
}
