<?php
/**
 * Kalium WordPress Theme
 *
 * Theme plugins class.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Core;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

class Theme_Plugins {

	/**
	 * Theme plugins registered on TGMPA.
	 *
	 * @var array
	 */
	private $theme_plugins = [];

	/**
	 * TGMPA AJAX button labels.
	 *
	 * @var array
	 */
	private $tgmpa_ajax_button_strings = [

		// Install button
		'install'  => [
			'title'      => 'Install',
			'processing' => 'Installing&hellip;',
		],

		// Activate button
		'activate' => [
			'title'      => 'Activate',
			'processing' => 'Activating&hellip;',
		],

		// Update button
		'update'   => [
			'title'      => 'Update',
			'processing' => 'Updating&hellip;',
		],

		// Active state button
		'active'   => [
			'title'      => 'Active',
			'processing' => '',
		],
	];

	/**
	 * Construct.
	 */
	public function __construct() {

		// Register TGMPA Plugins
		add_action( 'tgmpa_register', [ $this, 'register_theme_plugins' ] );

		// Init theme plugins data when needed
		foreach ( [ 'cli_init', 'load-update-core.php' ] as $hook_name ) {
			add_action( $hook_name, [ $this, 'init_theme_plugins_data' ] );
		}

		// Install, update or activate plugins with TGMPA via AJAX
		add_action( 'wp_ajax_kalium-plugins-tgmpa-install', [ $this, 'install_plugin_ajax' ] );

		// Set missing source URL for bundled plugins
		add_filter( 'pre_set_site_transient_update_plugins', [ $this, 'set_source_for_bundled_plugins' ], 100 );

		// Disable redirects to avoid many issues during import
		if ( defined( 'DOING_AJAX' ) && 'kalium-plugins-tgmpa-install' === kalium()->request->request( 'action' ) ) {
			add_filter( 'wp_redirect', '__return_false', 100000 );
		}
	}

	/**
	 * Third party plugins for Kalium.
	 */
	public function register_theme_plugins() {

		// Retrieve plugins data
		$this->init_theme_plugins_data();

		// Plugins list
		$plugins = $this->get_plugins_list();

		// TGMPA Config
		$config = [
			'id'           => 'kalium',
			'menu'         => 'kalium-install-plugins',
			'message'      => '<p>The plugins listed below are optional. You can install and activate only the ones that suit the functionality you need for your website. For optimal performance, we recommend avoiding unnecessary plugins to prevent bloat.</p>',
			'has_notices'  => false,
			'is_automatic' => false,
		];

		// TGMPA menu args
		add_filter(
			'tgmpa_admin_menu_args',
			kalium_hook_array_merge(
				[
					'page_title' => 'Theme Plugins',
					'menu_title' => 'Theme Plugins',
				]
			)
		);

		// Init TGMPA
		tgmpa( $plugins, $config );
	}

	/**
	 * Install, update or activate plugins with TGMPA via AJAX.
	 */
	public function install_plugin_ajax() {
		$plugin_slug        = kalium()->request->query( 'plugin' );
		$is_install_plugin  = 'install-plugin' === kalium()->request->query( 'tgmpa-install' );
		$is_activate_plugin = 'activate-plugin' === kalium()->request->query( 'tgmpa-activate' );
		$is_update_plugin   = 'update-plugin' === kalium()->request->query( 'tgmpa-update' );

		// Make sure it loads in AJAX
		add_filter( 'tgmpa_load', '__return_true' );

		// Init TGMPA
		$GLOBALS['tgmpa']->init();

		// Initialize TGMPA plugins
		$this->register_theme_plugins();

		// Plugin Install hook
		if ( $is_install_plugin ) {

			// Check permissions
			if ( ! current_user_can( 'install_plugins' ) ) {
				$this->ajax_response(
					[
						'success' => false,
						'message' => 'No permissions to install plugins!',
					]
				);
				die();
			}

			// After process completes
			add_action(
				'upgrader_process_complete',
				function () use ( $plugin_slug ) {

					// Avoid any redirect after plugin is activated
					add_filter( 'wp_redirect', '__return_false', 1000 );

					// Activate plugin
					$this->activate_plugin( $plugin_slug );

					// Custom hooks after plugin is activated
					$this->plugins_after_install_ajax_hooks( 'install', $plugin_slug );

					// Button state
					$button_state    = 'active';
					$button_text     = $this->tgmpa_ajax_button_strings[ $button_state ]['title'];
					$button_url      = '';
					$processing_text = '';

					// Activate plugin state
					if ( false === kalium()->is->plugin_active( kalium()->helpers->get_plugin_basename( $plugin_slug ) ) ) {
						$button_state    = 'activate';
						$button_text     = $this->tgmpa_ajax_button_strings[ $button_state ]['title'];
						$processing_text = $this->tgmpa_ajax_button_strings[ $button_state ]['processing'];
						$button_url      = $this->get_install_plugin_ajax_link( $plugin_slug, $button_state );
					}

					// Response
					$response = [
						'success'         => true,
						'action'          => 'install-plugin',
						'button_state'    => $button_state,
						'button_text'     => $button_text,
						'button_url'      => $button_url,
						'processing_text' => $processing_text,
					];

					// Output response
					$this->ajax_response( $response );
				}
			);
		} // Plugin Update hook
		elseif ( $is_update_plugin ) {

			// Check permissions
			if ( ! current_user_can( 'update_plugins' ) ) {
				$this->ajax_response(
					[
						'success' => false,
						'message' => 'No permissions to update plugins!',
					]
				);
				die();
			}

			// Was the plugin update before update
			$plugin_basename   = kalium()->helpers->get_plugin_basename( $plugin_slug );
			$was_plugin_active = kalium()->is->plugin_active( $plugin_basename );

			// After process completes
			add_action(
				'upgrader_process_complete',
				function () use ( $plugin_slug, $was_plugin_active ) {

					// Avoid any redirect after plugin is activated
					add_filter( 'wp_redirect', kalium_hook_return_value( '' ), 1000 );

					// Activate plugin if it was previously active
					if ( $was_plugin_active ) {
						$this->activate_plugin( $plugin_slug );
					}

					// Custom hooks after plugin is activated
					$this->plugins_after_install_ajax_hooks( 'update', $plugin_slug );

					// Button state
					$button_state    = 'activate';
					$button_text     = $this->tgmpa_ajax_button_strings[ $button_state ]['title'];
					$processing_text = $this->tgmpa_ajax_button_strings[ $button_state ]['processing'];
					$button_url      = $this->get_install_plugin_ajax_link( $plugin_slug, $button_state );

					if ( $was_plugin_active ) {
						$button_state    = 'active';
						$button_text     = $this->tgmpa_ajax_button_strings[ $button_state ]['title'];
						$button_url      = '';
						$processing_text = '';
					}

					// Response
					$response = [
						'success'         => true,
						'action'          => 'update-plugin',
						'new_version'     => $this->get_plugin_version( $plugin_slug ),
						'plugin_updated'  => true,
						'button_state'    => $button_state,
						'button_text'     => $button_text,
						'button_url'      => $button_url,
						'processing_text' => $processing_text,
					];

					// Output response
					$this->ajax_response( $response );
				}
			);
		} // Plugin Activate hook
		elseif ( $is_activate_plugin ) {

			// Check permissions
			if ( ! current_user_can( 'activate_plugins' ) ) {
				$this->ajax_response(
					[
						'success' => false,
						'message' => 'No permissions to activate plugins!',
					]
				);
				die();
			}

			// Avoid any redirect after plugin is activated
			add_filter( 'wp_redirect', kalium_hook_return_value( '' ), 1000 );

			// Activate plugin
			$this->activate_plugin( $plugin_slug );

			// Custom hooks after plugin is activated
			$this->plugins_after_install_ajax_hooks( 'activate', $plugin_slug );

			// Button state
			$button_state = 'active';
			$button_text  = $this->tgmpa_ajax_button_strings[ $button_state ]['title'];

			// Response
			$response = [
				'success'        => true,
				'action'         => 'activate-plugin',
				'plugin_updated' => true,
				'button_state'   => $button_state,
				'button_text'    => $button_text,
			];

			// Output response
			$this->ajax_response( $response );
		}

		// Pass request to TGMPA handler
		$GLOBALS['tgmpa']->install_plugins_page();

		die();
	}

	/**
	 * New implementation of sources for premium bundled plugin updates.
	 *
	 * @param object $update_plugins
	 *
	 * @return object
	 */
	public function set_source_for_bundled_plugins( $update_plugins ) {
		$plugin_sources = array_combine( wp_list_pluck( $this->theme_plugins, 'slug' ), $this->theme_plugins );

		$set_sources = static function ( $plugins ) use ( $plugin_sources ) {
			foreach ( $plugins as $plugin_basename => $plugin ) {
				$plugin_slug = dirname( $plugin_basename );

				if ( empty( $plugin->package ) && ! empty( $plugin_sources[ $plugin_slug ] ) ) {
					$new_version     = $plugin->new_version ?? null;
					$bundled_version = $plugin_sources[ $plugin_slug ]['version'] ?? null;

					if ( version_compare( $new_version, $bundled_version, '<=' ) ) {
						$plugin->package = $plugin_sources[ $plugin_slug ]['source'];
					}
				}
			}
		};

		if ( ! empty( $update_plugins->response ) ) {
			$set_sources( $update_plugins->response );
		}

		if ( ! empty( $update_plugins->no_update ) ) {
			$set_sources( $update_plugins->no_update );
		}

		return $update_plugins;
	}

	/**
	 * Get list of plugins that are required or recommended for the theme.
	 *
	 * @return array
	 */
	public function get_plugins_list() {
		$plugins = [];

		// Advanced Custom Fields Pro
		$plugins[] = $this->prepare_plugin_entry(
			[
				'slug'        => 'advanced-custom-fields-pro',
				'native_name' => 'ACF Pro',
				'description' => 'Enables the Parameters and Options area in every page for extra options.',
				'data'        => [
					'tags'       => [ 'required', 'premium-plugin' ],
					'author'     => 'WP Engine',
					'author_uri' => 'https://advancedcustomfields.com/',
					'icon_wp'    => 'https://connect.advancedcustomfields.com/assets/icon-256x256.png',
				],
			]
		);

		// Elementor
		$plugins[] = $this->prepare_plugin_entry(
			[
				'slug'        => 'elementor',
				'native_name' => 'Elementor',
				'description' => 'Elementor is a WordPress plugin for designing websites with a drag-and-drop interface.',
				'data'        => [
					'author'     => 'Elementor',
					'author_uri' => 'https://elementor.com/',
					'icon_wp'    => 'https://ps.w.org/elementor/assets/icon-256x256.gif',
				],
			]
		);

		// WPBakery Page Builder
		$plugins[] = $this->prepare_plugin_entry(
			[
				'slug'        => 'js_composer',
				'native_name' => 'WPB Page Builder',
				'description' => 'Front-end and back-end editor. Build layouts with intuitive drag and drop page builder.',
				'data'        => [
					'tags'       => [ 'premium-plugin' ],
					'author'     => 'WPBakery',
					'author_uri' => 'https://1.envato.market/kvoV0',
				],
			]
		);

		// WooCommerce
		$plugins[] = $this->prepare_plugin_entry(
			[
				'slug'        => 'woocommerce',
				'native_name' => 'WooCommerce',
				'description' => 'WooCommerce is a WordPress plugin for adding eCommerce functionality to websites.',
				'data'        => [
					'author'     => 'Elementor',
					'author_uri' => 'https://elementor.com/',
					'icon_wp'    => 'https://ps.w.org/woocommerce/assets/icon-256x256.gif',
				],
			]
		);

		// Revolution Slider
		$plugins[] = $this->prepare_plugin_entry(
			[
				'slug'        => 'revslider',
				'native_name' => 'Slider Revolution',
				'description' => 'Allows you to create beautiful sliders on your website in an easy way.',
				'data'        => [
					'tags'       => [ 'premium-plugin' ],
					'author'     => 'ThemePunch',
					'author_uri' => 'https://themepunch.pxf.io/J5997',
				],
			]
		);

		// Layer Slider
		$plugins[] = $this->prepare_plugin_entry(
			[
				'slug'        => 'LayerSlider',
				'native_name' => 'Layer Slider',
				'description' => 'A lightweight alternative of Slider Revolution plugin.',
				'data'        => [
					'tags'       => [ 'premium-plugin' ],
					'author'     => 'Kreatura Media',
					'author_uri' => 'https://layerslider.com/',
				],
			]
		);

		return $plugins;
	}

	/**
	 * List theme plugins to install.
	 */
	public function list_theme_plugins() {

		?>
		<ul class="kalium-dashboard__plugins">

			<?php
			foreach ( $this->get_plugins_list() as $plugin ) :
				$plugin_slug = $plugin['slug'];
				$plugin_name = $plugin['native_name'];
				$plugin_data = $plugin['data'];
				$plugin_tags = $plugin_data['tags'];

				// Plugin description
				$description = $plugin['description'];

				// Plugin filters
				$is_required       = in_array( 'required', $plugin_tags );
				$is_recommended    = in_array( 'recommended', $plugin_tags );
				$is_optional       = in_array( 'optional', $plugin_tags );
				$is_premium_plugin = in_array( 'premium-plugin', $plugin_tags );

				// Plugin icon
				$plugin_icon = kalium()->assets_url( sprintf( 'admin/images/plugins/plugin-%1$s.png', $plugin_slug ) );

				if ( $icon_wp = kalium_get_array_key( $plugin_data, 'icon_wp', false ) ) {
					$icon_slug   = is_bool( $icon_wp ) ? $plugin_slug : $icon_wp;
					$plugin_icon = kalium()->is->url( $icon_slug ) ? $icon_slug : sprintf( 'https://ps.w.org/%1$s/assets/icon-256x256.png', esc_attr( $icon_slug ) );
				}

				// Plugin author
				$plugin_author = sprintf( 'By <a href="%2$s" target="_blank" rel="noreferrer noopener">%1$s</a>', esc_html( $plugin_data['author'] ), esc_url( $plugin_data['author_uri'] ) );

				// Plugin badge
				$plugin_badge = $badge_type = null;

				if ( $is_required ) {
					$plugin_badge = 'Required';
					$badge_type   = 'danger';
				} elseif ( $is_premium_plugin ) {
					$plugin_badge = 'Premium';
					$badge_type   = 'info';
				}

				// Filter data
				$filter_data = [
					'required'    => $is_required,
					'recommended' => $is_recommended,
					'optional'    => $is_optional,
					'premium'     => $is_premium_plugin,
				];

				?>
				<li data-filter="<?php echo esc_attr( wp_json_encode( $filter_data ) ); ?>">

					<div class="kalium-dashboard__plugin-item kalium-dashboard__card p-4">
						<div class="kalium-dashboard__plugin-item__heading">
							<img src="<?php echo esc_url( $plugin_icon ); ?>" width="48" height="48"/>
							<h3>
								<?php echo esc_html( $plugin_name ); ?>
								<small>
									<span class="version"><?php echo esc_html( $this->get_plugin_version( $plugin_slug ) ); ?></span>
									|
									<span class="author"><?php echo $plugin_author; ?></span>
								</small>
							</h3>

							<?php
							if ( ! empty( $plugin_badge ) ) :
								printf( '<span class="badge badge-%2$s">%1$s</span>', $plugin_badge, $badge_type );
							endif;
							?>
						</div>

						<?php if ( $description ) : ?>
						<p class="kalium-dashboard__plugin-item__description"><?php echo wp_kses_post( $description ); ?></p>
						<?php endif; ?>

						<div class="kalium-dashboard__plugin-item__footer">
							<?php
							// Plugin install, activate or update link
							$this->install_plugin_link( $plugin_slug );

							// Updates
							if ( $new_version = $this->get_latest_plugin_version( $plugin_slug ) ) :
								?>
								<div class="kalium-dashboard__plugin-item__update-available">
									<i class="kalium-admin-icon-refresh"></i> New version available: <?php echo esc_html( $new_version ); ?>
								</div>
							<?php endif; ?>
						</div>
					</div>
				</li>
				<?php
			endforeach;
			?>
		</ul>
		<?php
	}

	/**
	 * Custom hooks after plugin is installed, activated or updated via TGMPA on AJAX.
	 *
	 * @param string $action
	 * @param string $plugin_slug
	 */
	private function plugins_after_install_ajax_hooks( $action, $plugin_slug ) {

		// Disable WPBakery Page Builder redirect to avoid unexpected AJAX results
		if ( 'js_composer' === $plugin_slug ) {
			delete_transient( '_vc_page_welcome_redirect' );
		} // Disable WooCommerce redirect to avoid unexpected AJAX results
		elseif ( 'woocommerce' === $plugin_slug ) {
			delete_transient( '_wc_activation_redirect' );
		}
	}

	/**
	 * Plugin install link.
	 *
	 * @param string $plugin_slug
	 * @param bool   $echo
	 *
	 * @return void|string
	 */
	private function install_plugin_link( $plugin_slug, $echo = true ) {
		$is_plugin_installed = $GLOBALS['tgmpa']->is_plugin_installed( $plugin_slug );
		$is_plugin_active    = $GLOBALS['tgmpa']->is_plugin_active( $plugin_slug );
		$plugin_has_update   = ! empty( $this->get_latest_plugin_version( $plugin_slug ) );

		// Button classes
		$button_classes = [
			'button',
			'button-tgmpa-ajax',
		];

		// Default button action
		$action_type = 'active';

		// Update plugin
		if ( $plugin_has_update ) {
			$action_type = 'update';
		} // Install plugin
		elseif ( ! $is_plugin_installed ) {
			$action_type = 'install';
		} // Activate plugin
		elseif ( ! $is_plugin_active ) {
			$action_type = 'activate';
		}

		// Button action link
		$action_link = $this->get_install_plugin_ajax_link( $plugin_slug, $action_type );

		// Action type button class
		$button_classes[] = "button--state-{$action_type}";

		$button_state_classes = [
			'update'   => 'button-warning',
			'activate' => 'button-primary',
			'active'   => 'button-disabled',
		];

		$button_classes[] = $button_state_classes[ $action_type ] ?? null;

		// Check permissions
		if ( 'install' === $action_type && ! current_user_can( 'install_plugins' ) ) {
			$button_classes[] = 'button-disabled';
		} elseif ( 'update' === $action_type && ! current_user_can( 'update_plugins' ) ) {
			$button_classes[] = 'button-disabled';
		} elseif ( 'activate' === $action_type && ! current_user_can( 'activate_plugins' ) ) {
			$button_classes[] = 'button-disabled';
		}

		// Display button
		$button = sprintf(
			'<a href="%1$s" class="%4$s" data-title-processing="%3$s">%2$s</a>',
			$action_link,
			$this->tgmpa_ajax_button_strings[ $action_type ]['title'],
			$this->tgmpa_ajax_button_strings[ $action_type ]['processing'],
			kalium_tokenize_list( $button_classes )
		);

		// Echo button
		if ( $echo ) {
			echo $button;
		} else {
			return $button;
		}
	}

	/**
	 * Get plugin install, update or activate link for TGMPA.
	 *
	 * @param string $plugin_slug
	 * @param string $action_type
	 *
	 * @return string|null
	 */
	private function get_install_plugin_ajax_link( $plugin_slug, $action_type ) {
		if ( ! in_array( $action_type, [ 'install', 'update', 'activate' ] ) ) {
			return null;
		}

		return wp_nonce_url(
			sprintf(
				admin_url( 'admin-ajax.php?action=kalium-plugins-tgmpa-install&plugin=%1$s&tgmpa-%2$s=%2$s-plugin' ),
				$plugin_slug,
				$action_type
			),
			'tgmpa-' . $action_type,
			'tgmpa-nonce'
		);
	}

	/**
	 * Initialize theme plugins data.
	 */
	public function init_theme_plugins_data() {
		if ( current_user_can( 'update_plugins' ) && kalium()->request->has( 'force-check' ) ) {
			$clear_cache = true;
		}

		$plugins_data = kalium_api(
			'theme-plugins',
			[
				'cache'         => DAY_IN_SECONDS,
				'cache_refresh' => ! empty( $clear_cache ),
			]
		);

		// Initialize loaded plugins in plugins var
		if ( is_array( $plugins_data ) ) {
			foreach ( $plugins_data as $plugin ) {
				$plugin['source'] = kalium()->license->wrap_license_key( $plugin['source'] );

				$this->theme_plugins[ $plugin['slug'] ] = $plugin;
			}
		}
	}

	/**
	 * Prepare TGMPA plugin entry.
	 *
	 * @param array $plugin_data {
	 *
	 * @type string $slug
	 * @type string $native_name
	 * @type string $name
	 * @type bool   $required
	 * @type array  $data
	 * }
	 *
	 * @return array
	 */
	private function prepare_plugin_entry( $plugin_data ) {
		$plugin = wp_parse_args(
			$plugin_data,
			[
				'slug'        => '',
				'native_name' => '',
				'name'        => '',
				'description' => '',
				'required'    => false,
				'data'        => [],
			]
		);

		// Plugin data
		$plugin['data'] = wp_parse_args(
			$plugin['data'],
			[
				'tags'       => [],
				'author'     => '',
				'author_uri' => '',
			]
		);

		// Insert native name as plugin name
		if ( empty( $plugin['name'] ) ) {
			$plugin['name'] = $plugin['native_name'];
		}

		// Retrieve plugin data
		if ( isset( $this->theme_plugins[ $plugin['slug'] ] ) ) {
			$plugin = array_merge( $plugin, $this->theme_plugins[ $plugin['slug'] ] );
		}

		return $plugin;
	}

	/**
	 * Get current version of plugin.
	 *
	 * @param string $plugin_slug
	 *
	 * @return string
	 */
	private function get_plugin_version( $plugin_slug ) {

		// Installed plugins
		static $plugins = [];

		// Version
		$plugin_version = null;

		// Plugin basename
		$plugin_basename = kalium()->helpers->get_plugin_basename( $plugin_slug );

		// Initialize plugins
		if ( empty( $plugins ) ) {
			$plugins = get_plugins();
		}

		// Get version from TGMPA entries
		if ( isset( $this->theme_plugins[ $plugin_slug ] ) ) {
			$plugin_version = kalium_get_array_key( $this->theme_plugins[ $plugin_slug ], 'version', null );
		}

		// Get current installed version
		if ( $plugin_basename && isset( $plugins[ $plugin_basename ] ) ) {
			$plugin_version = kalium_get_array_key( $plugins[ $plugin_basename ], 'Version', null );
		}

		// Retrieve version from WordPress.org API
		if ( is_null( $plugin_version ) ) {

			// Retrieve version from WordPress.org API
			$remote_plugin_version = $this->get_plugin_version_remote( $plugin_slug );

			if ( $remote_plugin_version ) {
				$plugin_version = $remote_plugin_version;
			}
		}

		return $plugin_version;
	}

	/**
	 * Get plugin version from WordPress API.
	 *
	 * @param string $plugin_slug
	 *
	 * @return string|null
	 */
	private function get_plugin_version_remote( $plugin_slug ) {
		$plugin_version = null;

		// Plugin version transient var
		$plugin_version_var = "kalium_plugin_version_var_{$plugin_slug}";

		// Return transient value
		if ( $transient_value = get_transient( $plugin_version_var ) ) {
			$plugin_version = $transient_value;
		} else {

			// Import plugins_api function
			if ( ! function_exists( 'plugins_api' ) ) {
				require_once ABSPATH . 'wp-admin/includes/plugin-install.php';
			}

			// Request latest version data for the plugin
			$response = plugins_api(
				'plugin_information',
				[
					'slug'   => $plugin_slug,
					'fields' => [
						'sections' => false,
					],
				]
			);

			// When the request is successful
			if ( ! is_wp_error( $response ) ) {

				// Assign latest version of requested plugin
				$plugin_version = $response->version;

				set_transient( $plugin_version_var, $plugin_version, DAY_IN_SECONDS );
			}
		}

		return $plugin_version;
	}

	/**
	 * Get latest version for a plugin (if there is any update of plugin).
	 *
	 * @param string $plugin_slug
	 *
	 * @return string|null
	 */
	private function get_latest_plugin_version( $plugin_slug ) {
		$plugin_updates = get_plugin_updates();
		$latest_version = '';

		// Get latest version from plugin updates array
		foreach ( $plugin_updates as $plugin_file => $plugin ) {
			$slug = dirname( $plugin_file );

			if ( $plugin_slug === $slug && isset( $plugin->update ) ) {
				$latest_version = $plugin->update->new_version;
			}
		}

		// Check if newer version is available in TGMPA
		if ( isset( $this->theme_plugins[ $plugin_slug ], $this->theme_plugins[ $plugin_slug ]['version'] ) ) {
			$current_plugin_version = $this->get_plugin_version( $plugin_slug );
			$tgmpa_plugin_version   = $this->theme_plugins[ $plugin_slug ]['version'];

			if ( version_compare( $tgmpa_plugin_version, $current_plugin_version, '>' ) && version_compare( $tgmpa_plugin_version, $latest_version, '>' ) ) {
				$latest_version = $tgmpa_plugin_version;
			}
		}

		return $latest_version;
	}

	/**
	 * Activate plugin.
	 *
	 * @param string $plugin_slug
	 *
	 * @return bool
	 */
	private function activate_plugin( $plugin_slug ) {
		if ( $plugin_basename = kalium()->helpers->get_plugin_basename( $plugin_slug ) ) {
			activate_plugin( $plugin_basename );

			return true;
		}

		return false;
	}

	/**
	 * Show TGMPA ajax response.
	 */
	private function ajax_response( $response ) {
		printf( '<script type="text/template" class="kalium-tgmpa-ajax-response">%s</script>', wp_json_encode( $response ) );
	}
}
