<?php
/**
 * Image size settings groups.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Customize\Settings_Groups;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

trait Image_Size {

	/**
	 * Image size settings.
	 *
	 * @param string $namespace
	 * @param array  $args
	 */
	public function register_image_size_settings( $namespace, $args = [] ) {
		$settings = $this->get_prefixed_settings(
			$namespace,
			[
				'size',
				'size_custom',
			]
		);

		$args = wp_parse_args(
			$args,
			[
				'custom'         => true,
				'default'        => null,
				'default_custom' => null,
				'heading_title'  => 'Image Size',
				'separator'      => true,
				'dependency'     => [],
			]
		);

		// Image size
		$this->add_control(
			$settings['size'],
			[
				'type'          => 'kalium-select',
				'heading_title' => $args['heading_title'],
				'label'         => 'Image Size',
				'input_style'   => 'width:160px',
				'separator'     => $args['separator'],
				'default'       => $args['default'],
				'choices'       => static function () use ( &$args ) {
					return kalium_get_image_size_names(
						[
							'custom' => $args['custom'],
						]
					);
				},
				'dependency'    => $args['dependency'],
			]
		);

		// Custom image size
		if ( $args['custom'] ) {
			$this->add_control(
				$settings['size_custom'],
				[
					'type'        => 'kalium-dimensions',
					'label'       => 'Custom Size',
					'input_style' => 'width: 160px',
					'default'     => $args['default_custom'],
					'dependency'  => array_merge(
						$args['dependency'],
						[
							$settings['size'] => 'custom',
						]
					),
				]
			);
		}
	}

	/**
	 * Get image size settings.
	 *
	 * @param string $namespace
	 *
	 * @return array
	 */
	public function get_image_size_settings( $namespace ) {
		return array_map(
			'kalium_get_theme_option',
			$this->get_prefixed_settings(
				$namespace,
				[
					'size',
					'size_custom',
				]
			)
		);
	}

	/**
	 * Image size settings.
	 *
	 * @param string $namespace
	 *
	 * @return string|array
	 */
	public function get_image_size_value( $namespace ) {
		$image_size_settings = $this->get_image_size_settings( $namespace );

		$image_size = $image_size_settings['size'];

		// Custom size
		if ( 'custom' === $image_size ) {
			$image_size = kalium_list_dimensions( $image_size_settings['size_custom'] );
		}

		// Default
		if ( ! $image_size ) {
			$image_size = kalium_get_default_theme_option( $namespace . '_size' );
		}

		return $image_size;
	}

	/**
	 * Register aspect ratio settings.
	 *
	 * @param string $namespace
	 * @param array  $args
	 */
	public function register_aspect_ratio_settings( $namespace, $args = [] ) {
		$settings = $this->get_prefixed_settings(
			$namespace,
			[
				'aspect_ratio',
				'aspect_ratio_custom',
			]
		);

		$args = wp_parse_args(
			$args,
			[
				'custom'     => true,
				'default'    => null,
				'dependency' => [],
			]
		);

		// Aspect ratio
		$this->add_control(
			$settings['aspect_ratio'],
			[
				'type'       => 'kalium-select',
				'label'      => 'Aspect Ratio',
				'responsive' => true,
				'choices'    => kalium_get_aspect_ratio_names(
					[
						'custom' => $args['custom'],
					]
				),
				'default'    => $args['default'],
				'setting'    => [
					'transport' => 'postMessage',
				],
				'dependency' => $args['dependency'],
			]
		);

		// Custom aspect ratio size
		if ( $args['custom'] ) {
			$this->add_control(
				$settings['aspect_ratio_custom'],
				[
					'type'        => 'kalium-text',
					'label'       => 'Custom Aspect Ratio',
					'responsive'  => true,
					'input_style' => 'width:100px',
					'placeholder' => 'e.g. 16 / 9',
					'setting'     => [
						'transport' => 'postMessage',
					],
					'dependency'  => array_merge(
						$args['dependency'],
						[
							$settings['aspect_ratio'] => 'custom',
						]
					),
				]
			);
		}
	}

	/**
	 * Get aspect ratio value.
	 *
	 * @param string $namespace
	 *
	 * @return array
	 */
	public function get_aspect_ratio_value( $namespace ) {
		$settings = $this->get_prefixed_settings(
			$namespace,
			[
				'aspect_ratio',
				'aspect_ratio_custom',
			]
		);

		$mapper = static function ( $value, $viewport ) use ( &$settings ) {
			if ( 'custom' === $value ) {
				$custom_ratio = kalium_get_theme_option( $settings['aspect_ratio_custom'] );

				return kalium_get_responsive_value( $custom_ratio, $viewport );
			}

			return 'auto' === $value ? null : $value;
		};

		return kalium_map_responsive_value( kalium_get_theme_option( $settings['aspect_ratio'] ), $mapper, false );
	}
}
