<?php
/**
 * General style settings groups.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
namespace Kalium\Customize\Settings_Groups;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

trait Style {

	/**
	 * Register style settings and controls.
	 *
	 * @param string $namespace
	 * @param array  $args
	 */
	public function register_style_settings( $namespace, $args = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'settings'          => [
					'bg',
					'headings',
					'text',
					'link',
					'padding',
					'border',
					'border_radius',
					'box_shadow',
				],
				'css_selector'      => null,
				'css_var_prefix'    => null,
				'dependency'        => null,
				'section'           => null,

				// Background, border and box shadow props automatically add has-padding class
				'add_padding_class' => false,
			]
		);

		$settings_controls = [
			'text'          => [
				'type'         => 'kalium-color',
				'label'        => 'Text',
				'inline_label' => true,
			],
			'headings'      => [
				'type'         => 'kalium-color',
				'label'        => 'Headings',
				'inline_label' => true,
				'css_var'      => 'heading-color',
			],
			'link'          => [
				'type'         => 'kalium-color',
				'label'        => 'Links',
				'inline_label' => true,
				'tooltip'      => true,
				'choices'      => [
					'normal' => 'Normal',
					'hover'  => 'Hover',
				],
				'css_var'      => [
					'normal' => 'link-color',
					'hover'  => 'link-hover-color',
				],
			],
			'bg'            => [
				'type'         => 'kalium-color',
				'label'        => 'Background',
				'inline_label' => true,
				'allow_clear'  => true,
			],
			'padding'       => [
				'type'       => 'kalium-multi-numeric',
				'label'      => 'Padding',
				'responsive' => true,
				'units'      => kalium_get_default_units( 'box-size' ),
			],
			'vpadding'      => [
				'type'        => 'kalium-multi-numeric',
				'label'       => 'Vertical Padding',
				'responsive'  => true,
				'units'       => kalium_get_default_units( 'box-size' ),
				'value_props' => [
					'top'    => 'Top',
					'bottom' => 'Bottom',
				],
				'default'     => [
					'link' => true,
				],
			],
			'border'        => [
				'type'         => 'kalium-border-box',
				'label'        => 'Border',
				'responsive'   => true,
				'units'        => kalium_get_default_units( 'box-size' ),
				'border_style' => true,
			],
			'border_radius' => [
				'type'       => 'kalium-multi-numeric',
				'label'      => 'Border Radius',
				'responsive' => true,
				'units'      => kalium_get_default_units( 'box-size' ),
			],
			'box_shadow'    => [
				'type'               => 'kalium-box-shadow',
				'label'              => 'Box Shadow',
				'units'              => kalium_get_default_units( 'box-size' ),
				'enable_color_clear' => true,
			],
		];

		// Common control args
		$common_control_args = [
			'setting'      => [
				'transport' => 'postMessage',
			],
			'reset'        => true,
			'css_selector' => $args['css_selector'],
			'dependency'   => $args['dependency'],
		];

		if ( ! is_null( $args['section'] ) ) {
			$common_control_args['section'] = $args['section'];
		}

		// Sort settings by keys ordering
		uksort(
			$settings_controls,
			function ( $a, $b ) use ( &$args ) {
				return array_search( $a, $args['settings'] ) - array_search( $b, $args['settings'] );
			}
		);

		// Filter settings
		$settings_controls = array_filter(
			$settings_controls,
			function ( $control_args, $control_type ) use ( &$args ) {
				return in_array( $control_type, $args['settings'], true );
			},
			ARRAY_FILTER_USE_BOTH
		);

		// Preview content
		$preview_vars = $preview_content = [];

		// Generate preview
		$generate_preview = function ( $control_type, $setting_id, $css_selector, $css_var_name ) {
			$preview = null;

			switch ( $control_type ) {
				// Background, headings and text
				case 'bg':
				case 'headings':
				case 'text':
					$preview = <<<EOD
{{{ data.api.printInlineStyle( {
	selector: '{$css_selector}',
    varName: '{$css_var_name}',
    value: data.{$setting_id},
} ) }}}
EOD;
					break;

				// Link colors
				case 'link':
					$preview = <<<EOD
{{{ data.api.printInlineStyle( {
	selector: '{$css_selector}',
	value: data.{$setting_id},
	parser: function( value ) {
		return {
			[ data.api.cssVarName( '{$css_var_name['normal']}' ) ]: data.api.parseColor( value.normal ),
			[ data.api.cssVarName( '{$css_var_name['hover']}' ) ]: data.api.parseColor( value.hover ),
		};
	},
} ) }}}
EOD;
					break;

				// Padding and Border radius
				case 'padding':
				case 'border_radius':
					$preview = <<<EOD
{{{ data.api.printInlineStyle( {
    selector: '{$css_selector}',
    varName: '{$css_var_name}',
    value: data.{$setting_id},
    parser: 'length',
} ) }}}
EOD;
					break;

				// Vertical padding
				case 'vpadding':
					$preview = <<<EOD
{{{ data.api.printInlineStyle( {
    selector: '{$css_selector}',
    varName: '{$css_var_name}',
    value: data.{$setting_id},
    parser: function( value ) {
        var top = data.api.getMultiNumericProp( value, 'top', 0 ),
            bottom = data.api.getMultiNumericProp( value, 'bottom', 0 ); 
        
        if ( top || bottom ) {
            return top + ' ' + bottom;
        } 
    },
} ) }}}
EOD;
					break;

				// Border
				case 'border':
					$preview = <<<EOD
{{{ data.api.printInlineStyle( {
	selector: '{$css_selector}',
	varName: '{$css_var_name}',
	value: data.{$setting_id},
	parser: 'border',
	resetVars: [
		'{$css_var_name}-style',
		'{$css_var_name}-width',
		'{$css_var_name}-color',
	],
} ) }}}
EOD;
					break;

				// Box shadow
				case 'box_shadow':
					$preview = <<<EOD
{{{ data.api.printInlineStyle( {
	selector: '{$css_selector}',
	varName: '{$css_var_name}',
	value: data.{$setting_id},
	parser: 'box-shadow',
} ) }}}
EOD;
					break;
			}

			return $preview;
		};

		// Register settings and controls
		foreach ( $settings_controls as $control_type => & $control_args ) {
			$setting_id   = $control_args['id'] ?? ( $namespace . '_' . $control_type );
			$control_args = array_merge( $common_control_args, $control_args );

			// Update setting ID
			$control_args['id'] = $setting_id;

			// Extend control args
			if ( ! empty( $args[ 'setting_' . $control_type ] ) ) {
				foreach ( $args[ 'setting_' . $control_type ] as $setting_prop => $setting_value ) {
					if ( ! empty( $control_args[ $setting_prop ] ) && is_array( $control_args[ $setting_prop ] ) ) {
						$control_args[ $setting_prop ] = array_merge( $control_args[ $setting_prop ], $setting_value );
					} else {
						$control_args[ $setting_prop ] = $setting_value;
					}
				}
			}

			// Register control and setting
			$this->add_control( $setting_id, $control_args );

			// Generate preview
			if ( ! empty( $control_args['setting']['transport'] ) && 'postMessage' === $control_args['setting']['transport'] ) {
				$css_selector = $control_args['css_selector'] ?? ':root';
				$css_var_name = $control_args['css_var'] ?? str_replace( '_', '-', ( $args['css_var_prefix'] ?? $namespace ) . '-' . $control_type );

				$preview_vars[]    = $setting_id;
				$preview_content[] = $generate_preview( $control_type, $setting_id, $css_selector, $css_var_name );
			}
		}

		// Generate preview
		if ( ! empty( $preview_vars ) ) {
			$add_padding_vars = [ 'bg', 'border', 'box_shadow' ];

			// Add padding class
			if ( $args['add_padding_class'] && ':root' !== $args['css_selector'] && ! empty( array_intersect( $add_padding_vars, $args['settings'] ) ) ) {
				$add_padding_vars = wp_json_encode(
					array_filter(
						array_map(
							function ( $prop ) use ( $settings_controls ) {
								return $settings_controls[ $prop ]['id'] ?? null;
							},
							$add_padding_vars
						)
					)
				);

				$preview_content[] = <<<EOD
<#
var hasPadding = false,
    paddingEl = data.api.getElement( '{$args['css_selector']}' );
    paddingVars = {$add_padding_vars};
    
if ( paddingEl ) {
    paddingVars.forEach( function( paddingVar ) {
        if ( ! data.api.isEmpty( paddingVar ) ) {
            hasPadding = true;
        }
    } );
    
    paddingEl.classList[ hasPadding ? 'add' : 'remove' ]( 'has-padding' );
}
#>
EOD;
			}

			$this->add_preview_update(
				[
					'type'    => 'css',
					'vars'    => $preview_vars,
					'content' => implode( PHP_EOL, $preview_content ),
				]
			);
		}
	}

	/**
	 * Print style settings.
	 *
	 * @param string $namespace
	 * @param array  $args
	 * @param array  $classes
	 */
	public function print_style_settings( $namespace, $args = [], &$classes = [] ) {
		$args = wp_parse_args(
			$args,
			[
				'settings'          => [
					'bg',
					'headings',
					'text',
					'link',
					'padding',
					'vpadding',
					'border',
					'border_radius',
					'box_shadow',
				],
				'css_selector'      => null,
				'css_var_prefix'    => null,
				'css_vars'          => null,

				// Background, border and box shadow props automatically add has-padding class
				'add_padding_class' => false,
			]
		);

		// Generate vars based on prop type
		$generate_vars = function ( $control_type, $setting_id, $css_selector, $css_var_name ) {
			$vars = [];

			if ( $setting_value = kalium_get_theme_option( $setting_id ) ) {
				switch ( $control_type ) {
					// Background and text
					case 'bg':
					case 'text':
					case 'headings':
						$vars[ $css_var_name ] = kalium_replace_color_references( $setting_value );
						break;

					// Link colors
					case 'link':
						if ( is_array( $setting_value ) ) {
							foreach ( $setting_value as $state => $value ) {
								$vars[ $css_var_name[ $state ] ?? $state ] = kalium_replace_color_references( $value );
							}
						}
						break;

					// Padding and border radius
					case 'padding':
					case 'vpadding':
					case 'border_radius':
						$vars[ $css_var_name ] = kalium_map_responsive_value(
							$setting_value,
							function ( $value ) use ( &$vars ) {
								return kalium_to_length( $value );
							}
						);
						break;

					// Border
					case 'border':
						kalium_map_responsive_value(
							$setting_value,
							function ( $value, $viewport ) use ( &$vars, $css_var_name ) {
								$border_props = kalium_to_border_box( $value );

								foreach ( $border_props as $prop => $value ) {
									$vars[ $css_var_name . '-' . $prop ][ $viewport ] = $value;
								}
							}
						);
						break;

					// Box shadow
					case 'box_shadow':
						$vars[ $css_var_name ] = kalium_map_responsive_value(
							$setting_value,
							function ( $value ) use ( &$vars ) {
								return kalium_to_box_shadow( $value );
							}
						);
						break;
				}
			}

			return $vars;
		};

		// Map settings
		$settings = array_map(
			function ( $prop ) use ( $namespace, $args ) {
				$setting_id   = $namespace . '_' . $prop;
				$css_selector = $args['css_selector'] ?? ':root';
				$css_var_name = $args['css_vars'][ $prop ] ?? str_replace( '_', '-', ( $args['css_var_prefix'] ?? $namespace ) . '-' . $prop );

				return [
					'id'           => $setting_id,
					'type'         => $prop,
					'css_selector' => $css_selector,
					'css_var_name' => $css_var_name,
				];
			},
			$args['settings']
		);

		$settings = array_combine( $args['settings'], $settings );

		// CSS vars
		$vars = [];

		// Register cars
		foreach ( $settings as $prop => $setting ) {
			$setting_id   = $setting['id'];
			$css_selector = $setting['css_selector'];
			$css_var_name = $setting['css_var_name'];

			$vars = array_merge( $vars, $generate_vars( $prop, $setting_id, $css_selector, $css_var_name ) );
		}

		// Add padding class
		if ( $args['add_padding_class'] ) {
			$add_padding_vars = [
				'bg'         => 'kalium-color',
				'border'     => 'kalium-border-box',
				'box_shadow' => 'kalium-box-shadow',
			];

			foreach ( $add_padding_vars as $prop => $control_type ) {
				if ( ! isset( $settings[ $prop ] ) ) {
					continue;
				}

				$setting = $settings[ $prop ];

				if ( ! kalium_is_theme_option_empty( $setting['id'], $control_type ) ) {
					$has_padding = true;
				}
			}

			if ( ! empty( $has_padding ) ) {
				$classes[] = 'has-padding';
			}
		}

		// Print styles
		kalium_print_inline_style(
			[
				'id'       => str_replace( '_', '-', $namespace ) . '-styles',
				'selector' => $css_selector,
				'vars'     => $vars,
			]
		);
	}

	/**
	 * Register pagination settings.
	 *
	 * @param string $namespace
	 */
	public function register_pagination_settings( $namespace ) {
		$settings = $this->get_prefixed_settings(
			$namespace,
			[
				'type',
				'numbers_display',
				'infinite_scroll_auto_reveal',
				'infinite_scroll_loading_effect',
				'align',
			]
		);

		// Pagination type
		$this->add_control(
			$settings['type'],
			[
				'type'         => 'kalium-radio-button',
				'label'        => 'Type',
				'inline_label' => false,
				'choices'      => [
					'numbers'   => 'Numbers',
					'load-more' => 'Load More',
				],
			]
		);

		// Display type
		$this->add_control(
			$settings['numbers_display'],
			[
				'type'       => 'kalium-select',
				'label'      => 'Display Type',
				'choices'    => [
					'numbers'           => 'Numbers',
					'numbers-prev-next' => 'Numbers + Previous/Next',
					'prev-next'         => 'Previous/Next',
				],
				'default'    => 'numbers-prev-next',
				'dependency' => [
					$settings['type'] => 'numbers',
				],
			]
		);

		// Infinite scroll
		$this->add_control(
			$settings['infinite_scroll_auto_reveal'],
			[
				'type'       => 'kalium-toggle',
				'label'      => 'Infinite Scroll',
				'help'       => 'Infinite scroll loads more content as you scroll down the page, without the need for clicking a button or pagination.',
				'dependency' => [
					$settings['type'] => 'load-more',
				],
			]
		);

		// Infinite scroll loading effect
		$this->add_control(
			$settings['infinite_scroll_loading_effect'],
			[
				'type'       => 'kalium-radio-button',
				'label'      => 'Loading Effect',
				'choices'    => [
					'spinner' => 'Spinner',
					'pulsate' => 'Pulsate',
				],
				'dependency' => [
					$settings['type'] => 'load-more',
				],
			]
		);

		// Alignment
		$this->add_control(
			$settings['align'],
			[
				'type'           => 'kalium-radio-button',
				'label'          => 'Alignment',
				'separator'      => true,
				'separator_type' => 3,
				'inline_label'   => false,
				'choices'        => [
					'left'          => [
						'label' => 'Left',
						'icon'  => 'justifyLeft',
						'size'  => 16,
					],
					'center'        => [
						'label' => 'Center',
						'icon'  => 'justifyCenter',
						'size'  => 16,
					],
					'right'         => [
						'label' => 'Right',
						'icon'  => 'justifyRight',
						'size'  => 16,
					],
					'space-between' => [
						'label'      => 'Space Between',
						'icon'       => 'justifySpaceBetween',
						'size'       => 16,
						'dependency' => [
							$settings['type']            => 'numbers',
							$settings['numbers_display'] => [ 'numbers-prev-next', 'prev-next' ],
						],
					],
				],
				'default'        => 'center',
			]
		);
	}
}
