<?php
/**
 * Kalium WordPress Theme
 *
 * Blog template functions.
 *
 * @author Laborator
 * @link   https://kaliumtheme.com
 */
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Direct access not allowed.
}

if ( ! function_exists( 'kalium_blog_page_header' ) ) {

	/**
	 * Blog page heading title and description.
	 */
	function kalium_blog_page_header() {
		if ( ! kalium()->is->blog_page() ) {
			return;
		}

		if ( kalium_get_blog_option( 'loop/header/show' ) ) {
			$queried_object = get_queried_object();
			$title          = kalium_get_blog_option( 'loop/header/title' );
			$description    = kalium_get_blog_option( 'loop/header/description' );

			// Description from post content
			if ( is_home() && ! empty( $queried_object->post_content ) ) {
				$description = apply_filters( 'the_content', $queried_object->post_content );
			}

			// Category, tag and author pages show custom title
			if ( apply_filters( 'kalium_blog_page_heading_replace_for_taxonomies', true ) ) {
				$separator = apply_filters( 'kalium_blog_page_heading_replace_for_taxonomies_separator', ' / ' );

				// Category
				if ( is_category() ) {
					if ( apply_filters( 'kalium_blog_page_header_last_category_title_only', false ) ) {
						$categories = single_cat_title( '', false );
					} else {
						$categories = get_the_category_list( $separator );
					}

					if ( apply_filters( 'kalium_blog_page_header_show_category_prefix', true ) ) {
						$title = sprintf( '%s %s <span>%s</span>', esc_html__( 'Category', 'kalium' ), $separator, $categories );
					} else {
						$title = sprintf( '<span>%s</span>', $categories );
					}

					$description = category_description( $queried_object->object_id );
				} // Tag
				elseif ( is_tag() ) {
					$tag         = single_term_title( '', false );
					$title       = sprintf( '%s %s <span>%s</span>', esc_html__( 'Tag', 'kalium' ), $separator, $tag );
					$description = tag_description( $queried_object->object_id );
				} // Author
				elseif ( is_author() ) {
					$author       = get_user_by( 'id', get_queried_object_id() );
					$display_name = $author instanceof WP_User ? $author->display_name : get_the_author();
					$title        = sprintf( '%s %s <span>%s</span>', esc_html__( 'Author', 'kalium' ), $separator, $display_name );
					$description  = get_the_author_meta( 'description' );
				} // Year
				elseif ( is_year() ) {
					$title       = sprintf( '%s %s <span>%s</span>', esc_html__( 'Year', 'kalium' ), $separator, get_the_date( 'Y' ) );
					$description = '';
				} // Month
				elseif ( is_month() ) {
					$title       = sprintf( '%s %s <span>%s</span>', esc_html__( 'Month', 'kalium' ), $separator, get_the_date( 'F Y' ) );
					$description = '';
				} // Day
				elseif ( is_day() ) {
					$title       = sprintf( '%s %s <span>%s</span>', esc_html__( 'Month', 'kalium' ), $separator, get_the_date( 'F j, Y' ) );
					$description = '';
				}
			}

			// Show post content instead of heading
			if ( is_home() && ! is_front_page() ) {
				$current_post      = get_queried_object();
				$post_content      = $current_post->post_content;
				$is_elementor_page = kalium()->elementor->is_built_with_elementor( $current_post->ID );

				if ( has_shortcode( $post_content, 'vc_row' ) || $is_elementor_page ) {
					add_action(
						'kalium_wrapper_start',
						function () use ( $current_post ) {
							setup_postdata( $current_post );

							// Wrapper start
							kalium_content_wrapper_start( 'blog_archive_heading' );

							// Show the content
							kalium_the_content();

							// Wrapper end
							kalium_content_wrapper_end( 'blog_archive_heading' );

							// Reset post data
							wp_reset_postdata();
						},
						30
					);

					return;
				}
			}

			// Page heading visibility
			kalium()->page->show_page_heading( true );

			// Page heading data
			kalium()->page->set_page_heading(
				[
					'title'       => $title,
					'description' => $description,
				]
			);
		}
	}
}

if ( ! function_exists( 'kalium_blog_no_posts_found_message' ) ) {

	/**
	 * No posts to show message.
	 */
	function kalium_blog_no_posts_found_message() {

		?>
		<h3 class="no-posts-found"><?php esc_html_e( 'There are no posts to show', 'kalium' ); ?></h3>
		<?php
	}
}

if ( ! function_exists( 'kalium_blog_loop_post_template' ) ) {

	/**
	 * Blog posts loop template.
	 */
	function kalium_blog_loop_post_template() {
		kalium_get_template(
			'blog/loop/post-item.php',
			[
				'classes' => [
					'post-item',
				],
			]
		);
	}
}

if ( ! function_exists( 'kalium_blog_posts_loop' ) ) {

	/**
	 * Blog posts loop.
	 *
	 * @param array $args
	 */
	function kalium_blog_posts_loop( $args = [] ) {
		$args = wp_parse_args(
			[
				'id'      => kalium_get_blog_instance_id(),
				'classes' => kalium_get_blog_option( 'loop/container_classes' ),
			],
			$args
		);

		// Vars
		$css_vars = [];

		// Layout class
		$args['classes'][] = 'blog-posts--' . kalium_get_blog_layout();

		// List type
		if ( 'list' === kalium_get_blog_layout() ) {
			$css_vars['post-item-image-width']   = kalium_get_theme_option( 'blog_thumbnail_width' );
			$css_vars['post-item-image-spacing'] = kalium_get_theme_option( 'blog_thumbnail_spacing' );

			if ( 'center' === kalium_get_blog_option( 'loop/post_thumbnail/content_valign' ) ) {
				$args['classes'][] = 'blog-posts--valign-center';
			}
		}

		// Boundless
		if ( kalium_get_blog_option( 'loop/post_thumbnail/boundless' ) ) {
			$args['classes'][] = 'blog-posts--boundless-image';
		}

		// Gap
		$columns_gap = kalium_get_blog_option( 'loop/gap' );

		if ( is_numeric( $columns_gap ) ) {
			$css_vars['grid-gap'] = $columns_gap . 'px';
		} else {
			$css_vars['grid-gap'] = kalium_map_responsive_value(
				$columns_gap,
				function ( $value ) {
					if ( is_numeric( $value ) ) {
						return $value . 'px';
					}
				}
			);
		}

		// Placeholder background
		if ( kalium_get_blog_option( 'loop/post_thumbnail/placeholder/enabled' ) && ! empty( $placeholder_bg = kalium_get_blog_option( 'loop/post_thumbnail/placeholder/bg_color' ) ) ) {
			$css_vars['post-item-image-placeholder-bg'] = $placeholder_bg;
		}

		// Aspect ratio
		$css_vars['custom-aspect-ratio'] = kalium_get_blog_option( 'loop/post_thumbnail/aspect_ratio' );

		// Post image border radius
		$css_vars['post-item-image-border-radius'] = kalium_map_responsive_value( kalium_get_blog_option( 'loop/post_thumbnail/border_radius' ), 'kalium_to_length' );

		// Hover layer style
		if ( kalium_get_blog_option( 'loop/post_thumbnail/hover/visible' ) ) {
			$background = kalium_get_blog_option( 'loop/post_thumbnail/hover/background' );
			$padding    = kalium_get_blog_option( 'loop/post_thumbnail/hover/padding' );

			// Background
			if ( $background ) {
				$css_vars['post-item-hover-layer-bg'] = $background;
			}

			// Spacing/padding
			$css_vars['post-item-hover-layer-offset'] = kalium_map_responsive_value(
				$padding,
				function ( $value ) {
					if ( is_numeric( $value ) ) {
						return $value . 'px';
					}
				}
			);
		}

		// Print styles
		kalium_print_inline_style(
			[
				'selector' => kalium_get_blog_option( 'selector' ),
				'vars'     => $css_vars,
			]
		);

		// Styles tab
		kalium()->customize->print_style_settings(
			'blog_post_style',
			[
				'css_selector'      => '.blog-posts',
				'css_var_prefix'    => 'post-item',
				'css_vars'          => [
					'headings' => 'post-item-title',
				],
				'add_padding_class' => true,
			],
			$args['classes']
		);

		// Masonry container
		kalium_make_masonry_posts_container();

		// Posts template
		kalium_get_template( 'blog/posts.php', $args );
	}
}

if ( ! function_exists( 'kalium_blog_sidebar_loop' ) ) {

	/**
	 * Blog archive sidebar.
	 */
	function kalium_blog_sidebar_loop() {

		if ( kalium_get_blog_option( 'loop/sidebar/visible' ) ) :

			?>
			<aside <?php kalium_class_attr( kalium_get_sidebar_classes() ); ?>>
				<?php
				// Show widgets
				kalium_dynamic_sidebar( 'blog_sidebar', 'blog-archive--widgets' );
				?>
			</aside>
			<?php

		endif;
	}
}

if ( ! function_exists( 'kalium_blog_archive_posts_pagination' ) ) {

	/**
	 * Blog archive pagination.
	 */
	function kalium_blog_archive_posts_pagination() {
		global $wp_query;

		$total_pages     = $wp_query->max_num_pages;
		$pagination_type = kalium_get_blog_option( 'loop/pagination/type' );

		// More link
		if ( 'more-link' === $pagination_type ) {
			kalium_load_more_link(
				[
					'url'    => kalium_get_blog_option( 'loop/pagination/more_link/url' ),
					'target' => kalium_get_blog_option( 'loop/pagination/more_link/target' ),
					'text'   => kalium_get_blog_option( 'loop/pagination/more_link/text' ),
				]
			);
			return;
		}

		if ( ! kalium_get_blog_option( 'loop/pagination/enabled' ) || $total_pages <= 1 ) {
			return;
		}

		// Load more pagination
		if ( 'load-more' === $pagination_type ) {
			kalium_load_more_pagination(
				[
					'align'         => kalium_get_blog_option( 'loop/pagination/align' ),
					'loading_style' => kalium_get_blog_option( 'loop/pagination/load_more_loading_effect' ),
					'options'       => [
						'container'       => '.blog-posts',
						'item'            => '.post',
						'infinite_scroll' => kalium_get_blog_option( 'loop/pagination/load_more_infinite_scroll' ),
					],
				]
			);
		}

		// Pagination
		kalium_pagination(
			[
				'align'           => kalium_get_blog_option( 'loop/pagination/align' ),
				'display'         => kalium_get_blog_option( 'loop/pagination/numbers_display' ),
				'pagination_args' => [
					'total' => $total_pages,
				],
			]
		);
	}
}

if ( ! function_exists( 'kalium_blog_post_thumbnail' ) ) {

	/**
	 * Loop post thumbnail.
	 */
	function kalium_blog_post_thumbnail() {
		global $post;

		// Args
		$args = [];

		// Thumbnail size
		$args['thumbnail_size'] = kalium_get_blog_option( 'loop/post_thumbnail/size' );

		// Supported post formats
		if ( kalium_get_blog_option( 'loop/post_formats' ) ) {
			$args['post_format_content'] = kalium_extract_post_format_content( $post );
		}

		// Show post thumbnails only if they are set to be visible
		if ( kalium_get_blog_option( 'loop/post_thumbnail/visible' ) ) {

			if ( kalium_get_blog_option( 'loop/post_thumbnail/placeholder/enabled' ) || has_post_thumbnail( $post ) || ! empty( $args['post_format_content'] ) ) {

				// Post thumbnail template
				kalium_get_template( 'blog/loop/post-thumbnail.php', $args );
			}
		}
	}
}

if ( ! function_exists( 'kalium_blog_post_hover_layer' ) ) {

	/**
	 * Loop post thumbnail hover layer.
	 */
	function kalium_blog_post_hover_layer() {
		global $post;

		$post_format = get_post_format( $post );

		// Args
		$args = [];

		// Show hover layer or not
		$hover_layer = kalium_get_blog_option( 'loop/post_thumbnail/hover/visible' );

		if ( kalium_get_blog_option( 'loop/post_formats' ) && in_array(
			$post_format,
			[
				'quote',
				'gallery',
				'video',
				'audio',
			]
		) ) {
			$hover_layer = false;
		}

		// Hover layer is shown
		if ( $hover_layer ) {
			$args['hover_icon'] = '';

			// Icon
			switch ( kalium_get_blog_option( 'loop/post_thumbnail/hover/icon' ) ) {

				// Eye icon
				case 'eye':
					$args['hover_icon'] = 'animated' === kalium_get_blog_option( 'loop/post_thumbnail/hover/eye_icon' ) ? 'animated-eye' : 'static-eye';
					break;

				// Custom icon
				case 'custom':
					$atts = [];

					$attachment_id = kalium_get_blog_option( 'loop/post_thumbnail/hover/custom_icon/image_id' );

					// Icon width
					$hover_icon_custom_width = kalium_get_blog_option( 'loop/post_thumbnail/hover/custom_icon/width' );

					if ( $attachment_id && is_numeric( $hover_icon_custom_width ) ) {
						$attachment = wp_get_attachment_image_src( $attachment_id, 'full' );

						if ( $attachment ) {
							$hover_icon_custom_height = absint( $attachment[2] * ( $hover_icon_custom_width / $attachment[1] ) );
							$atts['style']            = "width: {$hover_icon_custom_width}px; height: {$hover_icon_custom_height}px;";
						}
					}

					// Custom hover icon
					$hover_icon_custom = wp_get_attachment_image( $attachment_id, 'full', null, $atts );

					$args['hover_icon'] = 'custom';

					$args['hover_icon_custom'] = $hover_icon_custom;
					break;
			}

			// Hover layer classes
			$classes = [ 'post-hover' ];

			$args['classes'] = $classes;

			// Post thumbnail hover template
			kalium_get_template( 'blog/loop/post-thumbnail-hover.php', $args );
		}
	}
}

if ( ! function_exists( 'kalium_blog_post_format_icon' ) ) {

	/**
	 * Blog post format icon.
	 */
	function kalium_blog_post_format_icon() {
		global $post;

		if ( kalium_get_blog_option( 'loop/post_format_icon' ) ) {
			$post_format = get_post_format( $post );

			// Args
			$args = [
				'classes'             => [
					'post-format-icon',
				],
				'position_vertical'   => null,
				'position_horizontal' => null,
			];

			// Position
			if ( $position = kalium_get_blog_option( 'loop/post_format_icon_position' ) ) {
				$position = explode( ' ', $position );

				$args['position_vertical']   = $position[0];
				$args['position_horizontal'] = $position[1];
			}

			// Default post icon
			$icon = 'icon kalium-icon-blog-post-default';

			// Available icons
			$post_format_icons = [
				'quote'   => 'icon kalium-icon-blog-post-blockquote',
				'video'   => 'icon kalium-icon-blog-post-video',
				'audio'   => 'icon kalium-icon-blog-post-audio',
				'link'    => 'icon kalium-icon-blog-post-link',
				'image'   => 'icon kalium-icon-blog-post-photo',
				'gallery' => 'icon kalium-icon-blog-post-gallery',
			];

			if ( $post_format && isset( $post_format_icons[ $post_format ] ) ) {
				$icon = $post_format_icons[ $post_format ];
			}

			$args['icon'] = $icon;

			// Post icon template
			kalium_get_template( 'blog/loop/post-icon.php', $args );
		}
	}
}

if ( ! function_exists( 'kalium_blog_post_excerpt' ) ) {

	/**
	 * Blog post excerpt.
	 */
	function kalium_blog_post_excerpt() {

		if ( kalium_get_blog_option( 'loop/post_excerpt' ) ) :

			?>
			<div class="post-excerpt entry-summary">
				<?php the_excerpt(); ?>
			</div>
			<?php

		endif;
	}
}

if ( ! function_exists( 'kalium_blog_post_content' ) ) {

	/**
	 * Blog post content.
	 */
	function kalium_blog_post_content() {
		$show_post_content = kalium_get_blog_option( 'loop/post_excerpt' );

		if ( kalium_is_single_post() ) {
			$show_post_content = kalium_get_blog_option( 'single/post_content' );
		}

		if ( $show_post_content ) :
			$classes = [
				'entry-content',
				'post-content',
				'container-child',
				kalium_get_link_style_class(),
			];
			?>
			<section <?php kalium_class_attr( $classes ); ?>>
				<?php
				// Post content
				echo apply_filters( 'the_content', apply_filters( 'kalium_blog_post_content', get_the_content() ) );

				// Post content pagination
				if ( is_single() ) {
					wp_link_pages(
						[
							'before'           => '<div class="pagination pagination--post-pagination">',
							'after'            => '</div>',
							'next_or_number'   => 'next',
							'previouspagelink' => sprintf( '%2$s %1$s', esc_html__( 'Previous page', 'kalium' ), '&laquo;' ),
							'nextpagelink'     => sprintf( '%1$s %2$s', esc_html__( 'Next page', 'kalium' ), '&raquo;' ),
						]
					);
				}
				?>
			</section>
			<?php

		endif;
	}
}

if ( ! function_exists( 'kalium_blog_post_date' ) ) {

	/**
	 * Blog post date.
	 */
	function kalium_blog_post_date() {
		$show_post_date = kalium_get_blog_option( 'loop/post_meta/post_date' );

		if ( kalium_is_single_post() ) {
			$show_post_date = kalium_get_blog_option( 'single/post_meta/post_date' );
		}

		if ( $show_post_date ) :

			?>
			<div class="post-meta date updated published">
				<?php if ( kalium_is_post_meta_icon_visible() ) : ?>
					<i class="kalium-icon-blog-post-date"></i>
				<?php endif; ?>

				<span><?php the_time( apply_filters( 'kalium_post_date_format', get_option( 'date_format' ) ) ); ?></span>
			</div>
			<?php

		endif;
	}
}

if ( ! function_exists( 'kalium_blog_post_category' ) ) {

	/**
	 * Blog post category.
	 */
	function kalium_blog_post_category() {
		$show_post_category = kalium_get_blog_option( 'loop/post_meta/post_category' );

		if ( kalium_is_single_post() ) {
			$show_post_category = kalium_get_blog_option( 'single/post_meta/post_category' );
		}

		if ( $show_post_category && has_category() ) :
			$classes = [
				'post-meta',
				'category',
				kalium_get_link_text_class(),
				kalium_get_link_block_class(),
				kalium_get_link_style_class_by_context( 'blog-loop-category' ),
			];

			?>
			<div <?php kalium_class_attr( $classes ); ?>>
				<?php if ( kalium_is_post_meta_icon_visible() ) : ?>
					<i class="kalium-icon-blog-post-category"></i>
				<?php endif; ?>
				<span class="links"><?php the_category( ', ' ); ?></span>
			</div>
			<?php
		endif;
	}
}

if ( ! function_exists( 'kalium_blog_post_tags' ) ) {

	/**
	 * Blog post tags.
	 */
	function kalium_blog_post_tags() {
		$show_post_tags = kalium_get_blog_option( 'loop/post_meta/post_tags' );

		if ( $show_post_tags ) :

			?>
			<div class="post-meta tags">
				<?php if ( kalium_is_post_meta_icon_visible() ) : ?>
					<i class="kalium-icon-blog-post-category"></i>
				<?php endif; ?>
				<?php the_tags( ', ' ); ?>
			</div>
			<?php

		endif;
	}
}

if ( ! function_exists( 'kalium_blog_single_post_tags_list' ) ) {

	/**
	 * Single post tags list.
	 */
	function kalium_blog_single_post_tags_list() {
		if ( kalium_get_blog_option( 'single/post_meta/post_tags' ) && has_tag() ) :
			$classes = [
				'post-tags',
				'wp-block-tag-cloud',
				'is-style-outline',
				kalium_get_link_text_class(),
			];

			?>
			<section <?php kalium_class_attr( $classes ); ?>>
				<?php the_tags( '', ' ', '' ); ?>
			</section>
			<?php

		endif;
	}
}

if ( ! function_exists( 'kalium_blog_single_post_share_networks' ) ) {

	/**
	 * Single post share networks.
	 */
	function kalium_blog_single_post_share_networks() {

		if ( kalium_get_blog_option( 'single/share/visible' ) ) :
			$share_networks = kalium_get_blog_option( 'single/share/networks' );
			$share_style    = kalium_get_blog_option( 'single/share/style' );

			if ( empty( $share_networks['elements'] ) ) {
				return;
			}
			?>
			<section
					class="<?php printf( 'post-share-networks post-share-networks--style-%s', esc_attr( $share_style ) ); ?>">

				<div class="share-title">
					<?php esc_html_e( 'Share:', 'kalium' ); ?>
				</div>

				<div class="networks-list">
					<?php
					kalium_social_networks_share(
						[
							'networks' => $share_networks,
							'style'    => $share_style,
						]
					);
					?>
				</div>

			</section>
			<?php

		endif;
	}
}

if ( ! function_exists( 'kalium_blog_single_post_author_info' ) ) {

	/**
	 * Post author info.
	 */
	function kalium_blog_single_post_author_info() {
		global $wp_roles;

		if ( ! kalium_get_blog_option( 'single/author/visible' ) ) {
			return;
		}

		$classes = [
			'entry-author',
		];

		$author_id = get_the_author_meta( 'ID' );
		$userdata  = get_userdata( $author_id );
		$role_name = $wp_roles->roles[ current( $userdata->roles ) ]['name'];

		// Author URL
		$link_target = '_self';

		if ( $author_url = get_the_author_meta( 'url' ) ) {
			$link_target = '_blank';
		} else {
			$author_url = get_author_posts_url( $author_id );
		}

		$author_url = apply_filters( 'kalium_blog_single_author_url', $author_url );

		// Author details classes
		$author_details_classes = [
			'entry-author__details-name',
			kalium_get_link_title_class(),
			kalium_get_link_block_class(),
		];

		// Print style settings
		kalium()->customize->print_style_settings(
			'blog_author_info',
			[
				'css_selector'      => '.entry-author',
				'css_var_prefix'    => 'author',
				'add_padding_class' => true,

				// Replaced CSS vars
				'css_vars'          => [
					'headings' => 'heading-color',
					'link'     => [
						'normal' => 'link-color',
						'hover'  => 'link-hover-color',
					],
				],
			],
			$classes
		);

		?>
		<div <?php kalium_class_attr( $classes ); ?>>

			<?php if ( kalium_get_theme_option( 'blog_author_info_avatar' ) ) : ?>
				<div class="entry-author__image">
					<?php
					echo kalium_get_avatar( $author_id );
					?>
				</div>
			<?php endif; ?>

			<div class="entry-author__details">

				<div <?php kalium_class_attr( $author_details_classes ); ?>>
					<a href="<?php echo esc_url( $author_url ); ?>" class="vcard author author-name" target="<?php echo $link_target; ?>"<?php echo '_blank' == $link_target ? ' rel="author"' : ''; ?>>
						<span class="fn"><?php the_author(); ?></span>
					</a>
				</div>

				<?php if ( apply_filters( 'kalium_blog_single_post_author_info_show_subtitle', true ) ) : ?>
					<span><?php echo apply_filters( 'kalium_blog_single_post_author_info_subtitle', $role_name ); ?></span>
				<?php endif; ?>

				<?php
				/**
				 * Other author info details
				 *
				 * @hooked kalium_blog_single_post_author_info_description - 10
				 */
				do_action( 'kalium_blog_single_post_author_info_details', $author_id, $userdata );
				?>
			</div>

		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_blog_single_post_sidebar' ) ) {

	/**
	 * Single post sidebar.
	 */
	function kalium_blog_single_post_sidebar() {

		if ( kalium_get_blog_option( 'single/sidebar/visible' ) ) :

			?>
			<aside <?php kalium_class_attr( kalium_get_sidebar_classes() ); ?>>

				<?php
				// Post sidebar
				$sidebar_id = 'blog_sidebar_single';

				if ( ! is_active_sidebar( $sidebar_id ) ) {
					$sidebar_id = 'blog_sidebar';
				}

				kalium_dynamic_sidebar( $sidebar_id, 'single-post--widgets' );
				?>

			</aside>
			<?php

		endif;
	}
}

if ( ! function_exists( 'kalium_blog_single_post_author_info_description' ) ) {

	/**
	 * Single post author description when its shown below.
	 */
	function kalium_blog_single_post_author_info_description( $author_id, $userdata ) {
		if ( kalium_get_theme_option( 'blog_author_info_description' ) && ( $description = get_the_author_meta( 'description', $author_id ) ) ) :
			?>
			<div class="entry-author__description">
				<?php
				echo wpautop( wp_unslash( wp_rel_nofollow( $description ) ) );
				?>
			</div>
			<?php

		endif;
	}
}

if ( ! function_exists( 'kalium_blog_post_comment' ) ) {

	/**
	 * Blog post comment.
	 *
	 * @param WP_Comment $comment
	 * @param array      $args
	 * @param int        $depth
	 *
	 * @since 4.0
	 */
	function kalium_blog_post_comment( $comment, $args, $depth ) {
		$comment_avatar = kalium_get_avatar( $comment );

		// Time and Date format
		$date_format = get_option( 'date_format', 'F d, Y' );
		$time_format = get_option( 'time_format', 'h:m A' );

		$comment_date = apply_filters( 'kalium_blog_post_comment_date', sprintf( _x( '%1$s at %2$s', 'comment submit date', 'kalium' ), get_comment_date( $date_format ), get_comment_date( $time_format ) ), $comment );

		// Parent comment
		$parent_comment_id = $comment->comment_parent;

		// In reply to
		$parent_comment = $parent_comment_id ? get_comment( $parent_comment_id ) : null;

		// Commenter image
		$commenter_image = $comment_avatar;

		if ( $parent_comment ) {
			$commenter_image .= '<div class="comment-connector"></div>';
		}

		?>
		<li class="comments-list__comment">
		<div <?php comment_class(); ?> id="comment-<?php comment_id(); ?>">
			<?php if ( $comment_avatar ) : ?>
				<div class="comment-image">
					<?php
					// Comment avatar
					echo $commenter_image;
					?>
				</div>
			<?php endif; ?>

			<div class="comment-details">

				<div 
				<?php
				kalium_class_attr(
					[
						'name',
						kalium_get_link_plain_class(),
						kalium_get_link_title_class(),
					]
				);
				?>
				>

					<?php
					// Comment Author
					comment_author_link();

					// Reply Link
					comment_reply_link(
						array_merge(
							$args,
							[
								'reply_text' => esc_html__( 'reply', 'kalium' ),
								'depth'      => $depth,
								'max_depth'  => $args['max_depth'],
								'before'     => '',
							]
						)
					);
					?>

				</div>

				<div class="comment-meta">
					<div class="date">
						<?php
						// Comment date
						echo $comment_date;
						?>
					</div>

					<?php
					// Parent comment (in reply to)
					if ( $parent_comment ) :

						?>
						<div class="separator">&ndash;</div>
						<div class="in-reply-to">
							<?php printf( esc_html__( 'In reply to: %s', 'kalium' ), '<strong class="replied-to">' . get_comment_author( $parent_comment_id ) . '</strong>' ); ?>
						</div>
						<?php
					endif;
					?>
				</div>

				<div <?php kalium_class_attr( [ 'comment-text', 'entry-content', kalium_get_link_style_class() ] ); ?>>

					<?php
					// Comment text
					comment_text();

					// Moderation note
					if ( ! kalium_validate_boolean( $comment->comment_approved ) ) {
						printf( '<em class="comment-moderation-note">%s</em>', kalium_xtranslate( 'esc_html__', 'Your comment is awaiting moderation.' ) );
					}
					?>

				</div>

			</div>
		</div>
		<?php
	}
}

if ( ! function_exists( 'kalium_blog_post_comment_end' ) ) {

	/**
	 * Blog post comment end.
	 *
	 * @since 4.0
	 */
	function kalium_blog_post_comment_end( $comment, $args, $depth ) {
		echo '</li>';
	}
}

if ( ! function_exists( 'kalium_blog_single_post_related_posts' ) ) {

	/**
	 * Blog related posts.
	 *
	 * @since 3.4.4
	 */
	function kalium_blog_single_post_related_posts() {
		$show_related_posts     = kalium_get_theme_option( 'blog_related_posts' );
		$related_posts_for_post = kalium_get_field( 'related_posts_visibility' );

		if ( in_array( $related_posts_for_post, [ 'visible', 'hidden' ] ) ) {
			$show_related_posts = 'visible' === $related_posts_for_post;
		}

		if ( $show_related_posts ) {
			$criteria           = kalium_get_theme_option( 'blog_related_posts_criteria' );
			$columns            = kalium_get_theme_option( 'blog_related_posts_columns' );
			$posts_per_page     = kalium_get_theme_option( 'blog_related_posts_per_page' );
			$post_details       = kalium_get_theme_option( 'blog_related_posts_details' );
			$section_title      = kalium_get_theme_option( 'blog_related_posts_title' );
			$section_visibility = kalium_get_theme_option( 'blog_related_posts_visibility' );
			$inherit_container  = kalium_get_theme_option( 'blog_related_posts_inherit_container' );

			// Get related posts
			$related_posts = kalium_get_related_posts(
				[
					'posts_per_page' => absint( $posts_per_page ),
					'taxonomies'     => array_filter(
						[
							kalium_conditional( 'category' === $criteria, 'category' ),
							kalium_conditional( 'tag' === $criteria, 'tag' ),
						]
					),
				]
			);

			// Only when there are related posts
			if ( ! empty( $related_posts ) ) {
				global $blog_options;

				$current_blog_options = $blog_options;

				// Related posts card
				kalium_init_blog_options(
					'related-posts',
					[
						'layout' => 'grid',
						'loop'   => [
							'columns'      => $columns,
							'masonry_mode' => false,
							'post_excerpt' => false,
							'post_formats' => false,
						],
					]
				);

				if ( ! $post_details ) {
					$blog_options['loop']['post_date'] = $blog_options['loop']['post_category'] = false;
				}

				// Classes
				$classes = [
					'related-posts',
				];

				// Inherit container width
				if ( $inherit_container && kalium_is_narrow_blog_layout() ) {
					$classes[] = 'container-narrow';
				}

				if ( ! empty( $section_visibility ) ) {
					$classes = array_merge( $classes, kalium_get_device_visibility_classes( $section_visibility ) );
				}

				// Style
				kalium()->customize->print_style_settings(
					'blog_related_posts',
					[
						'settings'          => [
							'bg',
							'vpadding',
						],
						'css_var_prefix'    => 'related-posts',
						'css_selector'      => '.related-posts',
						'add_padding_class' => true,
					],
					$classes
				);

				// Related posts query
				query_posts(
					[
						'post_type'           => 'post',
						'post__in'            => $related_posts,
						'ignore_sticky_posts' => true,
					]
				);

				// Related posts
				kalium_get_template(
					'blog/single/related.php',
					[
						'title'   => $section_title,
						'classes' => $classes,
					]
				);

				// Reset data
				$blog_options = $current_blog_options;
				wp_reset_query();
				wp_reset_postdata();
			}
		}
	}
}

if ( ! function_exists( 'kalium_blog_archive_wrapper_start' ) ) {

	/**
	 * Blog wrapper start.
	 *
	 * @since 4.0
	 */
	function kalium_blog_archive_wrapper_start() {
		kalium_content_wrapper_start(
			'blog_archive',
			[
				'classes'       => [
					'blog',
					'blog-' . kalium_get_blog_layout(),
				],

				// Sidebar
				'sidebar'       => kalium_get_blog_option( 'loop/sidebar/visible' ),
				'sidebar_align' => kalium_get_blog_option( 'loop/sidebar/alignment' ),
			]
		);
	}
}

if ( ! function_exists( 'kalium_blog_archive_wrapper_end' ) ) {

	/**
	 * Blog wrapper end.
	 *
	 * @since 4.0
	 */
	function kalium_blog_archive_wrapper_end() {
		kalium_content_wrapper_end( 'blog_archive' );
	}
}

if ( ! function_exists( 'kalium_blog_single_wrapper_start' ) ) {

	/**
	 * Single post wrapper start.
	 *
	 * @since 4.0
	 */
	function kalium_blog_single_wrapper_start() {
		$classes = [
			'single-post',
		];

		// Featured image missing
		if ( ! has_post_thumbnail() || ! kalium_get_blog_option( 'single/post_image/visible' ) ) {
			$classes[] = 'single-post--no-featured-image';
		}

		kalium_content_wrapper_start(
			'blog_single',
			[
				'classes'             => $classes,
				'container_fullwidth' => false === kalium_get_blog_option( 'single/sidebar/visible' ),

				// Sidebar
				'sidebar'             => kalium_get_blog_option( 'single/sidebar/visible' ),
				'sidebar_align'       => kalium_get_blog_option( 'single/sidebar/alignment' ),
			]
		);
	}
}

if ( ! function_exists( 'kalium_blog_single_wrapper_end' ) ) {

	/**
	 * Single post wrapper end.
	 *
	 * @since 4.0
	 */
	function kalium_blog_single_wrapper_end() {
		kalium_content_wrapper_end( 'blog_single' );
	}
}

if ( ! function_exists( 'kalium_blog_single_post_comments' ) ) {

	/**
	 * Single post comments.
	 */
	function kalium_blog_single_post_comments() {
		if ( apply_filters( 'kalium_blog_comments', true ) && ! post_password_required() ) {
			comments_template();
		}
	}
}

if ( ! function_exists( 'kalium_blog_post_title' ) ) {

	/**
	 * Blog post title.
	 */
	function kalium_blog_post_title() {
		$is_single     = kalium_is_single_post();
		$heading_tag   = $is_single ? 'h1' : 'h3';
		$title_classes = [
			'entry-title',
			'post-title',
			kalium_get_link_plain_class(),
		];

		// Title wrap
		$heading_tag_open  = sprintf( '<%s %s>', $heading_tag, kalium_class_attr( $title_classes, false ) );
		$heading_tag_close = sprintf( '</%s>', $heading_tag );

		// Post link wrap (in the archive page)
		if ( ! $is_single ) {
			$heading_tag_open .= sprintf( '<a href="%s" target="%s" rel="bookmark">', get_permalink(), kalium_get_post_link_target() );
			$heading_tag_close = '</a>' . $heading_tag_close;
		}

		if ( $is_single ) {
			$show_post_title = kalium_get_blog_option( 'single/post_title' );
		} else {
			$show_post_title = kalium_get_blog_option( 'loop/post_title' );
		}

		// Show title
		if ( $show_post_title ) {
			the_title( $heading_tag_open, $heading_tag_close );
		}
	}
}

if ( ! function_exists( 'kalium_blog_post_meta' ) ) {

	/**
	 * Blog post entry meta.
	 *
	 * @since 4.0
	 */
	function kalium_blog_post_meta() {
		$show_post_meta = kalium_get_blog_option( 'loop/post_meta/visible' );

		if ( kalium_is_single_post() ) {
			$show_post_meta = kalium_get_blog_option( 'single/post_meta/visible' );
		}

		if ( $show_post_meta ) {
			echo '<div class="entry-meta">';

			/**
			 * Entry meta.
			 *
			 * @hooked kalium_blog_post_date - 10
			 * @hooked kalium_blog_post_category - 20
			 */
			do_action( 'kalium_blog_post_meta' );

			echo '</div>';
		}
	}
}

if ( ! function_exists( 'kalium_blog_single_content' ) ) {

	/**
	 * Single post layout
	 */
	function kalium_blog_single_content() {
		kalium_get_template( 'blog/single.php' );
	}
}

if ( ! function_exists( 'kalium_blog_single_post_featured_image' ) ) {

	/**
	 * Single post featured image or post format content.
	 */
	function kalium_blog_single_post_featured_image() {
		global $post;

		$show_post_image = kalium_get_blog_option( 'single/post_image/visible' );

		if ( $show_post_image ) {
			$args = [
				'classes'        => [
					'post-image',
					'theme-lightbox',
				],
				'post'           => get_post(),
				'thumbnail_size' => kalium_get_blog_option( 'single/post_image/size' ),
			];

			// Featured image placement
			switch ( kalium_get_blog_option( 'single/post_image/placement' ) ) {
				case 'full-width':
					$args['classes'][] = 'post-image--full-width';
					break;

				case 'container-wide':
					$args['classes'][] = 'container-wide';
					break;

				default:
					$args['classes'][] = 'container-child';
			}

			// Supported post formats
			if ( kalium_get_blog_option( 'single/post_formats' ) ) {
				$args['post_format_content'] = kalium_extract_post_format_content( $post );
			}

			// Show only if there is post image or post format content
			if ( has_post_thumbnail() || ! empty( $args['post_format_content'] ) ) {

				// Aspect ratio
				kalium_print_inline_style(
					[
						'selector' => '.single-post .post-image',
						'vars'     => [
							'custom-aspect-ratio' => kalium_get_blog_option( 'single/post_image/aspect_ratio' ),
						],
					]
				);

				// Border radius
				kalium()->customize->print_style_settings(
					'blog_featured_image',
					[
						'settings'       => [
							'border_radius',
						],
						'css_selector'   => '.single-post .post-image',
						'css_var_prefix' => 'pi',
					]
				);

				// Post image template
				kalium_get_template( 'blog/single/post-image.php', $args );
			}
		}
	}
}

if ( ! function_exists( 'kalium_blog_single_post_navigation' ) ) {

	/**
	 * Single post prev and next navigation.
	 *
	 * @param array $args
	 *
	 * @since 4.1.2 Param $args added
	 */
	function kalium_blog_single_post_navigation( $args = [] ) {
		if ( ! kalium_get_blog_option( 'single/post_navigation/enabled' ) ) {
			return;
		}

		kalium_post_navigation(
			wp_parse_args(
				$args,
				kalium_get_blog_option( 'single/post_navigation/options' )
			)
		);
	}
}
